#-*- coding: utf-8 -*-
"""

Utilitaire de mise à jour des variables
pour les versions >= 2.4.1

"""
try:
    from .upgrade import migration_23_to_tiramisu
except:
    migration_23_to_tiramisu = None
from .var_loader import convert_value
from .eosfunc import is_ip
from tiramisu.setting import owners
from tiramisu.setting import undefined
from packaging.version import Version
from pyeole.encode import normalize
from pyeole.i18n import i18n
from IPy import IP
import logging
log = logging.getLogger('creole.upgrade')
_ = i18n('creole')


class Upgrade():
    """
    Méthodes pour la mise à niveau des variables
    """

    old_release = None
    current_release = None

    def __init__(self, config):
        owner = 'upgrade'
        if owner not in dir(owners):
            owners.addowner(owner)
        self.config = config
        self.owner = getattr(owners, owner)
        self.unknown_options = config.impl_get_information('unknown_options')

    def run(self):
        log.info(_("Starting {0} to {1} upgrade").format(self.old_release, self.current_release))
        self._run()
        # rule for all versions
        ## change PNER repository source
        values = self.get_value('additional_repository_source')
        if values:
            for idx, source in enumerate(values):
                if source.startswith('deb http://debmiroir-02.eole.e2.rie.gouv.fr/mirror/debpner/ eole-'):
                    values[idx] = source.replace('eole-' + self.old_release, 'eole-' + self.current_release)
            self.set_value('additional_repository_source', values)
        return self.current_release

    def get_old_value(self, variable, old_variable, default=None):
        """
        Retourne la valeur d'une variable "disparue"
        """
        try:
            old_obj = self.unknown_options[old_variable]
            if old_obj.get('old_format', False):
                if not migration_23_to_tiramisu:
                    raise Exception(_('old format no more supported in migration'))
                try:
                    path = self.get_path(variable)
                except AttributeError:
                    return default
                opt = self.config.unwrap_from_path(path)
                val = migration_23_to_tiramisu(opt, old_obj['val'])
            else:
                val = old_obj['val']
            return val
        except KeyError:
            return default

    def get_value(self, variable, default=None):
        """
        Retourne la valeur d'une variable "connue"
        """
        try:
            path = self.get_path(variable)
        except AttributeError:
            return default
        return self.config.getattr(path,
                                   force_permissive=True)

    def get_unvalid_value(self, variable, default=None):
        try:
            path = self.get_path(variable)
        except AttributeError:
            return default
        try:
            return self.config.impl_get_information('orig_value_{}'.format(path))
        except ValueError:
            return default

    def get_noncalculated_value_for_auto(self, variable):
        """
        Retourne la valeur contenue dans le fichier config.eol dans le cas où la variable
        est calculée (auto), forcé à la valeur par défaut, ...
        """
        try:
            path = self.get_path(variable)
        except AttributeError:
            log.error(_('get_noncalculated_value_for_auto: unknown variable {}').format(variable))
            return None
        values = self.config.cfgimpl_get_values()
        if values._contains(path):
            option = self.config.unwrap_from_path(path)
            if not option.impl_is_multi():
                return values._p_.getvalue(path, values._p_.getsession())
            idx = 0
            vals = []
            while True:
                val = values._p_.getvalue(path, values._p_.getsession(), idx)
                if val is undefined:
                    break
                vals.append(val)
                idx += 1
            if len(vals) > 0:
                return vals
            else:
                return None
        return None

    def var_exists(self, variable):
        try:
            self.get_path(variable)
            return True
        except AttributeError:
            return False

    def get_path(self, variable):
        """
        Retourne le chemin complet d'une variable
        """
        return self.config.find_first(byname=variable, type_='path')

    def modify_owner(self, path, value=None, index=None):
        """
        Modifie le propriétaire d'une variable
        """
        option = self.config.unwrap_from_path(path)
        if option.impl_is_master_slaves('slave'):
            if index is not None:
                self.config.cfgimpl_get_values().setowner(option,
                                                          self.owner,
                                                          index=index)
            elif value is not None:
                for idx in range(len(value)):
                    self.config.cfgimpl_get_values().setowner(option,
                                                              self.owner,
                                                              index=idx)
            else:
                raise Exception('must have value or index for slave')

        else:
            self.config.cfgimpl_get_values().setowner(option,
                                                      self.owner)

    def is_default(self, variable, default=True):
        """
        Retourne True si la valeur n'a pas été personnalisée par l'utilisateur
        """
        try:
            path = self.get_path(variable)
        except AttributeError:
            return default
        option = self.config.unwrap_from_path(path)
        return self.config.cfgimpl_get_values().is_default_owner(option)

    def set_value(self, variable, value, index=None):
        """
        Modifie la valeur d'une variable
        """
        try:
            path = self.get_path(variable)
        except AttributeError:
            log.error(_("Try to set value to unknown option: {0} = {1}").format(variable, value))
        else:
            if index is not None:
                varmsg = "{}[{}]".format(variable, index)
            else:
                varmsg = variable
            try:
                self.config.setattr(path, value,
                                    force_permissive=True,
                                    index=index)
                self.modify_owner(path, value, index=index)
                log.info(_("Variable updated: {0} = {1}").format(varmsg, value))
                self.config.impl_del_information('error_msg_{}'.format(path), raises=False)
                self.config.impl_del_information('orig_value_{}'.format(path), raises=False)
                option = self.config.unwrap_from_path(path)
                self.config.cfgimpl_get_settings()[option].remove('load_error')
            except ValueError:
                option = self.config.unwrap_from_path(path)
                try:
                    # the value could be in Creole 2.3 format #13957
                    if not option.impl_is_multi() and isinstance(value, list) and len(value) == 1:
                        value = value[0]
                    if value in ['', ['']]:
                        err_msg = _("empty value")
                        log.error(_("{0} for {1}").format(err_msg, varmsg))
                        return
                    self.config.setattr(path, convert_value(option, value),
                                        force_permissive=True, index=index)
                    self.modify_owner(path, value, index=index)
                    log.info(_("Variable updated: {0} = {1}").format(varmsg, value))
                except Exception as err:
                    log.error(_("{0} for {1}").format(err, varmsg))
                    self.config.cfgimpl_get_settings()[option].append('load_error')
            except Exception as err:
                option = self.config.unwrap_from_path(path)
                log.error(_("{0} for {1}").format(normalize(str(err)), varmsg))
                self.config.cfgimpl_get_settings()[option].append('load_error')

    def del_value(self, variable):
        try:
            path = self.get_path(variable)
        except AttributeError:
            log.error(_('Try to delete an unknown option: {0}').format(variable))
        else:
            option = self.config.unwrap_from_path(path)
            self.config.cfgimpl_get_values().__delitem__(option)
            log.info(_("Variable {0} reinitialized").format(variable))

    def append_value(self, variable, value):
        """
        Ajoute une valeur à une variable multi
        """
        try:
            path = self.get_path(variable)
        except AttributeError:
            log.error(_('Try to append a value to an unknown option: {0} += {1}').format(variable, value))
        else:
            multi = self.config.getattr(path,
                                    force_permissive=True)
            multi.append(value)
            self.modify_owner(path, index=len(multi) - 1)

    def modify_last_value(self, variable, value):
        """
        Modifie la dernière valeur d'une variable multi
        """
        try:
            path = self.get_path(variable)
        except AttributeError:
            log.error(_('Try to modify last value of an unknown option: {0}[-1] = {1}').format(variable, value))
        else:
            multi = self.config.getattr(path,
                                force_permissive=True)
            multi[-1] = value
            self.modify_owner(path, index=len(multi) - 1)

    def move(self, old_variable, new_variable):
        """
        Déplace les données d'une variable "disparue"
        vers une nouvelle variable
        """
        if old_variable in self.unknown_options:
            value = self.unknown_options[old_variable]['val']
            try:
                path = self.get_path(new_variable)
            except AttributeError:
                log.error(_('Try to rename {0} to an unknown option {1}').format(old_variable, new_variable))
                return
            option = self.config.unwrap_from_path(path)
            if value in ['', ['']]:
                err_msg = _("empty value")
                log.error(_("{0} for {1}").format(err_msg, old_variable))
                return
            if option.impl_is_multi() and isinstance(value, list):
                for val in value:
                    self.append_value(new_variable, val)
            else:
                self.set_value(new_variable, value)
            del self.unknown_options[old_variable]
            log.info(_("Variable {0} has been renamed to {1}").format(old_variable, new_variable))

    def copy(self, old_variable, new_variable, only_if_modified=True):
        """
        Copie la valeur d'une variable existante vers une autre
        Si la valeur "old" est une multi et pas la "new" => copie la 1er valeur de la liste
        Si la valeur "old" n'est pas  une multi et la "new" ne l'est pas => transforme la valeur en liste
        only_if_modified: si True ne copie que les valeurs qui sont modifiées
        """
        try:
            # si les deux variables existe => migration
            old_path = self.get_path(old_variable)
            new_path = self.get_path(new_variable)
        except AttributeError:
            pass
        else:
            old_option = self.config.unwrap_from_path(old_path)
            new_option = self.config.unwrap_from_path(new_path)
            # si la nouvelle option n'est pas modifié et si la valeur est modifié ou only_if_modified est False
            if self.config.cfgimpl_get_values().is_default_owner(new_option) and \
                    (not only_if_modified or
                     not self.config.cfgimpl_get_values().is_default_owner(old_option)):
                old_value = self.config.getattr(old_path,
                                                force_permissive=True)
                if old_option.impl_is_multi() and not new_option.impl_is_multi():
                    if len(old_value) != 0:
                        old_value = old_value[0]
                    else:
                        old_value = None
                if not old_option.impl_is_multi() and new_option.impl_is_multi():
                    if old_value is None:
                        old_value = []
                    else:
                        old_value = [old_value]
                self.set_value(new_variable, old_value)

    def get_slaves(self, variable):
        """
        Retourne la liste (generator) des esclaves d'une variable maître
        """
        optiondescription, option = self.get_path(variable).rsplit('.', 1)
        for varname, opt in self.config.getattr(optiondescription).iter_all():
            if varname != variable:
                yield varname


class Upgrade_2_4_1(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.4.0 vers 2.4.1
    """

    current_release = '2.4.1'
    old_release = '2.4.0'

    def _run(self):
        """
        Lancement des traitements
        """
        # renommage des variables "era_proxy_bypass"
        for i in range(1, 5):
            self.move('era_proxy_bypass_eth{0}'.format(i), 'proxy_bypass_network_eth{0}'.format(i))

        # fusion des variables "proxy_bypass" et "wpad_exclude"
        if 'adresse_ip_wpad_exclude' in self.unknown_options:
            #le 1er argument sert a récupérer les propriétés des option (choiceoption, multi, ...)
            #on lui passe la variable de la 1er interface
            old_interfaces = self.get_old_value('proxy_bypass_network_eth1', 'interface_wpad_exclude')
            netmasks = self.get_old_value('proxy_bypass_netmask_eth1', 'adresse_netmask_wpad_exclude')
            for idx, value in enumerate(self.get_old_value('proxy_bypass_network_eth1', 'adresse_ip_wpad_exclude')):
                interface = old_interfaces[idx]
                if interface == 'Toutes':
                    interfaces = range(1, 5)
                elif int(interface) in range(1, 5):
                    interfaces = [interface]
                else:
                    log.error(_("Invalid value : {0} in old variable {1}").format(interface, 'interface_wpad_exclude'))
                    continue
                for i in interfaces:
                    self.append_value('proxy_bypass_network_eth{0}'.format(i), value)
                    self.modify_last_value('proxy_bypass_netmask_eth{0}'.format(i), netmasks[idx])
            del self.unknown_options['adresse_ip_wpad_exclude']
            del self.unknown_options['adresse_netmask_wpad_exclude']
            del self.unknown_options['interface_wpad_exclude']

        # passage à oui des variables "proxy_bypass_ethX" si nécessaire
        for i in range(1, 5):
            if len(self.get_value('proxy_bypass_network_eth{0}'.format(i), [])) > 0:
                self.set_value('proxy_bypass_eth{0}'.format(i), 'oui')

        # transfert des variables nom_domaine_wpad_exclude
        if 'nom_domaine_wpad_exclude' in self.unknown_options:
            old_interfaces = self.get_old_value('proxy_bypass_domain_eth1', 'nom_interface_wpad_exclude')
            for idx, value in enumerate(self.get_old_value('proxy_bypass_domain_eth1', 'nom_domaine_wpad_exclude')):
                interface = old_interfaces[idx]
                if interface == 'Toutes':
                    interfaces = range(1, 5)
                elif int(interface) in range(1, 5):
                    interfaces = [interface]
                else:
                    log.error(_("Invalid value : {0} in old variable {1}").format(interface, 'nom_interface_wpad_exclude'))
                    continue
                for i in interfaces:
                    self.append_value('proxy_bypass_domain_eth{0}'.format(i), value)
            del self.unknown_options['nom_domaine_wpad_exclude']
            del self.unknown_options['nom_interface_wpad_exclude']

        # nom_serveur_scribe_dmz/ip_serveur_scribe_dmz => mandatory (#11713)
        if self.get_value('install_scribe_dmz') == 'oui':
            if self.get_value('nom_serveur_scribe_dmz') == None or self.get_value('ip_serveur_scribe_dmz') == None:
                self.set_value('install_scribe_dmz', 'non')


class Upgrade_2_4_2(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.4.1 vers 2.4.2
    """

    current_release = '2.4.2'
    old_release = '2.4.1'

    def _run(self):
        """
        Lancement des traitements
        """
        # migration des variables eolesso vers client LDAP #10821
        self.copy('eolesso_port_ldap', 'ldap_port')
        self.copy('eolesso_ldap_reader', 'ldap_reader')
        self.copy('eolesso_ldap_reader_passfile', 'ldap_reader_passfile')
        self.copy('eolesso_ldap_match_attribute', 'ldap_match_attribute')
        self.copy('eolesso_ldap_filter_user', 'ldap_filter_user')
        self.copy('eolesso_ldap_filter_group', 'ldap_filter_group')
        self.copy('eolesso_ldap_dntree_user', 'ldap_dntree_user')
        self.copy('eolesso_ldap_dntree_group', 'ldap_dntree_group')
        self.copy('eolesso_ldap_fill_displayname', 'ldap_fill_displayname')
        self.copy('eolesso_ldap_fill_mail', 'ldap_fill_mail')
        self.copy('eolesso_ldap_fill_fonction', 'ldap_fill_fonction')
        self.copy('eolesso_ldap_fill_categorie', 'ldap_fill_categorie')
        self.copy('eolesso_ldap_fill_rne', 'ldap_fill_rne')
        self.copy('eolesso_ldap_fill_fredurne', 'ldap_fill_fredurne')
        self.copy('eolesso_ldap_fill_displaygroup', 'ldap_fill_displaygroup')

        # migration des variables courier #10987
        courier_val = self.get_old_value('activer_recuperation_courriel', 'activer_courier')
        if courier_val is not None:
            if courier_val == 'non':
                self.set_value('activer_recuperation_courriel', 'non')
            elif not 'imap' in courier_val:
                self.set_value('activer_courier_imap', 'non')
            if 'pop' in courier_val:
                self.set_value('activer_courier_pop', 'oui')


class Upgrade_2_5_0(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.4.X vers 2.5.0
    """

    current_release = '2.5.0'
    old_release = '2.4.2'

    def _run(self):
        """
        Lancement des traitements
        """
        # migration des variables nut #11608
        monitor = self.get_value('nut_monitor_user')
        if monitor != []:
            self.set_value('nut_monitor', 'oui')

        # migration des variables postgresql pour Zéphir #11974
        old_pg_shared_buffers = self.get_value('pg_shared_buffers')
        if old_pg_shared_buffers is not None:
            if int(old_pg_shared_buffers) == 3072:
                self.del_value('pg_shared_buffers')
            else:
                self.set_value('pg_shared_buffers_unit', 'kB')
            self.del_value('pg_effective_cache_size')


class Upgrade_2_5_1(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.5.0 vers 2.5.1
    """
    current_release = '2.5.1'
    old_release = '2.5.0'

    def _run(self):
        """
        Lancement des traitements
        """
        # migration des variables zone_forward (#11922)
        zone_forward = self.get_value('nom_zone_forward', [])
        if zone_forward != []:
            self.set_value('activer_zone_forward', 'oui')

        # passage de bacula à bareos (#12425)
        for var in ['activer_bareos_dir', 'activer_bareos_sd',
                    'bareos_dir_name', 'bareos_full_retention',
                    'bareos_full_retention_unit', 'bareos_diff_retention',
                    'bareos_diff_retention_unit', 'bareos_inc_retention',
                    'bareos_inc_retention_unit', 'bareos_max_run_time',
                    'bareos_compression', 'bareos_dir_password',
                    'bareos_fd_password', 'bareos_sd_local',
                    'bareos_sd_adresse', 'bareos_sd_password',
                    'bareos_sd_name', 'bareos_sd_remote_dir_name',
                    'bareos_sd_remote_ip', 'bareos_sd_remote_password']:
            self.move(var.replace('bareos', 'bacula'), var)

        if self.get_value('activer_bareos_dir') == 'oui':
            #sauvegarde déjà programmé en sqlite3, ne gère pas la migration vers mysql
            self.set_value('bareos_db_type', 'sqlite3')

        if self.get_value('ldap_ca_cert') == '/etc/certs/CA2008.pem':
            self.set_value('ldap_ca_cert', '/etc/certs/certificat.pem')


class Upgrade_2_5_2(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.5.1 vers 2.5.2
    """

    current_release = '2.5.2'
    old_release = '2.5.1'

    def _run(self):
        """
        Lancement des traitements
        """
        # haute dispo présente
        if self.var_exists('activer_haute_dispo'):
            # migration HD sphynx #14881
            if self.var_exists('activer_resource_arv'):
                activer_haute_dispo = self.get_value('activer_haute_dispo')
                if activer_haute_dispo == 'maitre':
                    service_resource_name = self.get_noncalculated_value_for_auto('service_resource_name')
                    service_resource_startdelay = self.get_noncalculated_value_for_auto('service_resource_startdelay')
                    need_update = False
                    startdelay_index = 1
                    need_disabled_arv = False
                    if service_resource_startdelay is not None:
                        if service_resource_name is not None:
                            need_update = 'arv_rsc' in service_resource_name
                            if need_update:
                                startdelay_index = service_resource_name.index('arv_rsc')
                            need_disabled_arv = not need_update
                        else:
                            need_update = True
                    self.del_value('service_resource_name')
                    self.del_value('service_resource_script')
                    self.del_value('service_resource_interval')
                    self.del_value('service_resource_timeout')
                    self.del_value('service_resource_startdelay')
                    if need_update and service_resource_startdelay[startdelay_index] != 15:
                        self.set_value('service_resource_arv_startdelay', service_resource_startdelay[startdelay_index])
                    if need_disabled_arv:
                        self.set_value('activer_resource_arv', 'non')
                    #
                    vip_resource_adresseip = self.get_noncalculated_value_for_auto('vip_resource_adresseip')
                    self.del_value('vip_resource_name')
                    self.del_value('vip_resource_if')
                    self.del_value('vip_resource_adresseip')
                    self.del_value('vip_resource_location')
                    if vip_resource_adresseip is not None:
                        if len(vip_resource_adresseip) > 0:
                            self.set_value('vip_externe', vip_resource_adresseip[0])
                        if len(vip_resource_adresseip) > 1:
                            self.set_value('vip_interne', vip_resource_adresseip[1])
            # migration HD non Sphynx #14951
            else:
                vip_resource_if = self.get_noncalculated_value_for_auto('vip_resource_if')
                vip_netmask = []
                for vip_if in vip_resource_if:
                    netmask_var = 'adresse_netmask_{0}'.format(vip_if.lower())
                    vip_netmask.append(self.get_value(netmask_var))
                if len(vip_netmask) > 0:
                    self.set_value('vip_resource_netmask', vip_netmask)
                service_resource_name = self.get_noncalculated_value_for_auto('service_resource_name')
                if len(service_resource_name) > 0:
                    self.set_value('activer_service_resource', 'oui')


class Upgrade_2_6_0(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.5.X vers 2.6.0
    """
    current_release = '2.6.0'
    old_release = '2.5.2'

    def get_eth_no(self, eth):
        """
        Retourne le numéro X du nom de l'interface ethX
        """
        try:
            return eth.split("eth")[1]
        except:
            log.error(_("Interface {0} name has not an 'ethX' format").format(eth))

    def _run(self):
        """
        Lancement des traitements
        """
        # migration des variables faisant référence au nom des interfaces ethX
        eth_vars = ['route_int', 'fw_rule_int', 'dhcrelay_server_interface', 'freerad_listen_int',
                    'sw_force_ip_src', 'corosync_dial_if', 'dhcrelay_interfaces']
        for eth_var in eth_vars:
            eth_name = self.get_unvalid_value(eth_var)
            if isinstance(eth_name, list):
                eth_no = []
                for eth in eth_name:
                    if eth == 'all':
                        eth_no.append(eth)
                    else:
                        eth_no.append(self.get_eth_no(eth))
                if eth_no != [] and eth_no != eth_name:
                    self.set_value(eth_var, eth_no)
            elif isinstance(eth_name, dict):
                eth_no = []
                for eth_key, eth_value in eth_name.items():
                    if eth_value == 'all':
                        eth_no.append(eth_value)
                    else:
                        eth_no.append(self.get_eth_no(eth_value))
                if eth_no != [] and eth_no != eth_name:
                    self.set_value(eth_var, eth_no)
            elif eth_name is not None:
                eth_no = self.get_eth_no(eth_name)
                self.set_value(eth_var, eth_no)
            elif eth_var == 'dhcrelay_server_interface' and self.get_value('adresse_ip_dhcp_dhcrelay') is not None:
                # migration de l'ancienne valeur par défaut de dhcrelay_server_interface #18329
                self.set_value(eth_var, '3')
        # haute dispo présente
        if self.var_exists('activer_haute_dispo'):
            # migration HD non sphynx
            if not self.var_exists('activer_resource_arv'):
                eth_name = self.get_noncalculated_value_for_auto('vip_resource_if')
                eth_no = []
                for eth in eth_name:
                    eth_no.append(self.get_eth_no(eth))
                self.set_value('vip_resource_if', eth_no)

        self.migrate_hapy()


    def migrate_hapy(self):
        vnets = self.get_value('vnets')
        start_ips = self.get_value('vnet_range_start')
        end_ips = self.get_old_value('vnet_range_end', 'vnet_range_end')
        range_sizes = []

        if not end_ips:
            # No value for vnet_range_end
            return True

        for index, vnet in enumerate(self.get_value('vnets')):
            start = IP(start_ips[index])
            end = IP(end_ips[index])
            size = int(end.ip - start.ip + 1)
            range_sizes.append(size)

        self.set_value('vnet_range_size', range_sizes)


class Upgrade_2_6_1(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.6.0 vers 2.6.1
    """

    current_release = '2.6.1'
    old_release = '2.6.0'

    def _run(self):
        """
        Lancement des traitements
        """
        # migration des variables NTLM/SMB : multi -> non multi (#18277)
        if self.var_exists('nom_serveur_smb'):
            for varname in ('nom_serveur_smb', 'nom_domaine_smb', 'ip_serveur_smb'):
                value = self.get_unvalid_value(varname)
                if isinstance(value, list) and len(value) > 1:
                    self.set_value(varname, value[0])

        # nom_carte_ethX => multi-valuées (#18609)
        for numint in range(0, 5):
            varname = 'nom_carte_eth{}'.format(numint)
            value = self.get_unvalid_value(varname)
            if value != None:
                self.set_value(varname, [value])

        # migration variable 'module_type' pour le module esbl ('ESBL') -> ('eSBL') (#21677)
        if self.get_value('eole_module') == 'esbl':
            self.set_value('module_type', 'eSBL')


class Upgrade_2_6_2(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.6.1 vers 2.6.2
    """

    current_release = '2.6.2'
    old_release = '2.6.1'

    def _run(self):
        """
        Lancement des traitements
        """
        adresse_network_dhcp = self.get_value('adresse_network_dhcp')
        if adresse_network_dhcp:
            plages = []
            for idx in range(len(adresse_network_dhcp)):
                plages.append('plage{}'.format(idx))
            self.set_value('nom_plage_dhcp', plages)
        if self.var_exists('acces_distant_backend_ead'):
            self.set_value('acces_distant_backend_ead', 'oui')
        for interface in [str(n) for n in range(5)]:
            variable = 'frontend_ead_distant_eth' + interface
            if self.var_exists(variable):
                self.set_value(variable, 'oui')
            variable = 'ip_frontend_ead_distant_eth' + interface
            if self.var_exists(variable):
                self.set_value(variable, ['0.0.0.0'])
            variable = 'netmask_frontend_ead_distant_eth' + interface
            if self.var_exists(variable):
                self.set_value(variable, ['0.0.0.0'])
        # Upgrade Seth
        # AD firewall - mix old multi variables ad_clients_ip and
        # ad_servers_ip in ad_peer_ip
        ad_servers_ip = self.get_old_value('ad_peer_ip', 'ad_servers_ip')
        ad_clients_ip = self.get_old_value('ad_peer_ip', 'ad_clients_ip')
        if ad_servers_ip or ad_clients_ip:
            self.set_value('ad_filter_network', 'oui')
        if ad_servers_ip:
            ad_servers_netmask = self.get_old_value('ad_peer_netmask', 'ad_servers_netmask')
            for ip, netmask in zip(ad_servers_ip, [nm[1] for nm in sorted(ad_servers_netmask.items())]):
                self.append_value('ad_peer_ip', ip)
                self.modify_last_value('ad_peer_netmask', netmask)
            del self.unknown_options['ad_servers_ip']
            del self.unknown_options['ad_servers_netmask']
        if ad_clients_ip:
            ad_clients_netmask = self.get_old_value('ad_peer_netmask', 'ad_clients_netmask')
            for ip, netmask in zip(ad_clients_ip, [nm[1] for nm in sorted(ad_clients_netmask.items())]):
                self.append_value('ad_peer_ip', ip)
                self.modify_last_value('ad_peer_netmask', netmask)
            del self.unknown_options['ad_clients_ip']
            del self.unknown_options['ad_clients_netmask']
        # Force SID
        force_sid = self.get_value('ad_domain_sid')
        if force_sid:
            self.set_value('ad_force_domain_sid', 'oui')
        # Squid modified variables : minutes -> seconds
        for squidvar in ['forward_timeout', 'connect_timeout', 'read_timeout', 'request_timeout', 'persistent_request_timeout']:
            squidval = self.get_value(squidvar)
            if squidval is not None and not self.is_default(squidvar):
                self.set_value(squidvar, squidval*60)
        # Exim relay : force to "activate" when upgrade from Scribe 2.6.1 only
        if self.var_exists('synchro_aaf'):
            self.set_value('exim_relay', 'oui')
            if self.get_value('activer_dhcp') == 'oui' and self.is_default('exim_relay_dhcp'):
                self.set_value('exim_relay_dhcp', 'oui')
        # Autosign certificat modified by user must be manual
        if self.get_value('cert_type') == 'autosigné':
            cert_is_modified = False
            # set manuel to access to variable
            self.set_value('cert_type', 'manuel')
            for cert in ['server_cert', 'server_key', 'server_pem']:
                if not self.is_default(cert):
                    cert_is_modified = True
                    break
            if not cert_is_modified:
                self.set_value('cert_type', 'autosigné')
        # Store autosign certificat in manual type
        if self.get_value('cert_type') == 'manuel':
            for cert, filename in [('server_cert', '/etc/ssl/certs/eole.crt'), ('server_pem', '/etc/ssl/certs/eole.pem')]:
                if self.is_default(cert):
                    self.set_value(cert, filename)
        # gaspacho agent needs to pass by port 8080 has in 2.6.1 and ealier
        if self.var_exists('gaspacho_https'):
            self.set_value('gaspacho_https', 'non')


class Upgrade_2_7_0(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.6.X vers 2.7.0
    """

    current_release = '2.7.0'
    old_release = '2.6.2'

    def _run(self):
        """
        Lancement des traitements
        """
        # Nouvelle variable slave de route_adresse : route_in_vpn (#23164)
        # On conserve le comportement défini dans le template ip_xfrm_policy de la version 2.6
        if self.get_value('install_rvp') == 'oui':
            route_int = self.get_value('route_int')
            if route_int:
                route_in_vpn = []
                for idx in range(len(route_int)):
                    if route_int[idx] == "0":
                        route_in_vpn.append('oui')
                    else:
                        route_in_vpn.append('non')
                if 'oui' in route_in_vpn:
                    self.set_value('route_in_vpn', route_in_vpn)

        # Nouveau chemin par défaut de la clé privée
        if self.get_value('cert_type') == 'manuel' and self.get_value('server_key') == '/etc/ssl/certs/eole.key':
            self.del_value('server_key')
        if self.get_value('eolesso_key') == '/etc/ssl/certs/eole.key':
            self.del_value('eolesso_key')

        # Migration des variables Bareos
        ## Migration du directeur
        bareos_sd_local = self.get_old_value('bareos_dir_use_local_sd', 'bareos_sd_local')
        #self.set_value('bareos_dir_use_local_sd', bareos_sd_local)
        if bareos_sd_local is not None:
            if bareos_sd_local == 'non':
                self.move('bareos_sd_adresse', 'bareos_dir_remote_sd_address')
                self.set_value('bareos_dir_remote_sd_password', self.get_value('bareos_sd_password'))
                self.set_value('bareos_dir_remote_sd_name', '-'.join([self.get_value('nom_machine'), 'sd']))
            else:
                self.set_value('bareos_sd_name', '-'.join([self.get_value('nom_machine'), 'sd']))
        ## Migration du stockage
        remote_dir_names = self.get_value('bareos_sd_remote_dir_name')
        if remote_dir_names is not None:
            if remote_dir_names:
                self.set_value('bareos_sd_set_remote_dirs', 'oui')
                self.move('bareos_sd_remote_ip', 'bareos_sd_remote_dir_ip')
                self.move('bareos_sd_remote_password', 'bareos_sd_remote_dir_password')
            else:
                self.set_value('bareos_sd_set_remote_dirs', 'non')
        ## Migration du client - rien

        # Les variables vlan_id_ethX sont des entiers #25084
        for idx in range(1, 5):
            varname = 'vlan_id_eth{}'.format(idx)
            vlan_ids = self.get_unvalid_value(varname)
            if vlan_ids is not None:
                self.set_value(varname, [int(x) for x in vlan_ids])
                # we must restore invalidated slaves !
                for slave in self.get_slaves(varname):
                    values = self.get_unvalid_value(slave)
                    if values:
                        for idx, value in values.items():
                            self.set_value(slave, value, index=int(idx))

        # Suppression des IP dans les noms alternatifs de certificats (#25010)
        altnames = self.get_unvalid_value('ssl_subjectaltname')
        if altnames is not None:
            new_altnames = []
            for alt in altnames:
                if not is_ip(alt):
                    new_altnames.append(alt)
            if altnames != new_altnames:
                self.set_value('ssl_subjectaltname', new_altnames)


class Upgrade_2_7_1(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.7.1 vers 2.7.2
    """

    current_release = '2.7.1'
    old_release = '2.7.0'

    def _run(self):
        pass

class Upgrade_2_7_2(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.7.1 vers 2.7.2
    """

    current_release = '2.7.2'
    old_release = '2.7.1'

    def _run(self):
        """
        Lancement des traitements
        """
        # Hâpy int -> unicode (#28783)
        if self.var_exists('vnet_vlan_trunk'):
            for slave in ('vnet_vlan_trunk', 'l2_vnet_vlan_trunk'):
                values = self.get_unvalid_value(slave)
                if values:
                    for idx, value in values.items():
                        self.set_value(slave, str(value).decode("utf-8"), index=int(idx))

        # Hâpy rename variable vnc_proxy_port_sunstone in one_sunstone_vnc_proxy_port
        if self.var_exists('one_sunstone_vnc_proxy_port'):
            self.move('vnc_proxy_port_sunstone', 'one_sunstone_vnc_proxy_port')

        # Hâpy rename activer_onesinglenode variable into activer_oned
        if self.var_exists('activer_oned'):
            self.move('activer_onesinglenode', 'activer_oned')

        # NUT none multi variable to master/slaves variables
        master = self.get_unvalid_value('nut_monitor_foreign_name')
        if master is not None:
            self.set_value('nut_monitor_foreign_name', [master])
            for name in ['nut_monitor_foreign_host',
                         'nut_monitor_foreign_user',
                         'nut_monitor_foreign_password']:
                slave = self.get_unvalid_value(name)
                if slave is not None:
                    self.set_value(name, slave, index=0)

        # '' is no more a valid debit_carte_ethX
        for i in range(0, 5):
            variable_name = 'debit_carte_eth{}'.format(i)
            old_invalid_value = self.get_unvalid_value(variable_name)
            if old_invalid_value == '':
                self.set_value(variable_name, 'autoneg on')

        # if eolesso_port is 443, set eolesso_cas_folder to old value
        if self.var_exists('eolesso_port') and not self.is_default('eolesso_port') and self.get_value('eolesso_port') == '443' and self.is_default('eolesso_cas_folder'):
            self.set_value('eolesso_cas_folder', None)

        # if ad_public_address is not set, AD is not local
        if self.get_value('ad_public_address', 'pasNone') is None and self.get_noncalculated_value_for_auto('ad_address') != None:
            self.set_value('ad_local', 'non')

        # force old default bareos_compression value
        if self.var_exists('bareos_compression') and self.is_default('bareos_compression'):
            self.set_value('bareos_compression', 'GZIP6')


class Upgrade_2_8_0(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.7.2 vers 2.8.0
    """

    current_release = '2.8.0'
    old_release = '2.7.2'

    def _run(self):
        """
        Lancement des traitements
        """
        if self.var_exists('eolesso_port'):
            if self.is_default('eolesso_port'):
                self.set_value('eolesso_port', 8443)


class Upgrade_2_8_1(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.8.0 vers 2.8.1
    """

    current_release = '2.8.1'
    old_release = '2.8.0'

    def _run(self):
        """
        Lancement des traitements
        """
        self.move('bareos_db_mysql_password', 'bareos_db_password')
        if 'bareos_db_type' in self.unknown_options:
            del self.unknown_options['bareos_db_type']

        if self.get_value('activer_filtrage_proxy', 'non') == 'oui':
            # désactiver la modération youtube si elle est désactivée sur toutes les zones
            if (self.get_value('dans_instance_1_active') == 'non' or self.unknown_options.get('youtube_moderate_1', 'oui') == 'non') and \
                    (self.get_value('dans_instance_2_active') == 'non' or self.unknown_options.get('youtube_moderate_2', 'oui') == 'non') and \
                    (self.get_value('dans_instance_3_active') == 'non' or self.unknown_options.get('youtube_moderate_3', 'oui') == 'non'):
                self.set_value('youtube_moderate', 'non')

        if self.var_exists('activer_sympa'):
            if self.is_default('activer_sympa'):
                self.set_value('activer_sympa', 'oui')

        if self.is_default('activer_ead3'):
            self.set_value('activer_ead3', 'non')


class Upgrade_2_9_0(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.8.1 vers 2.9.0
    """

    current_release = '2.9.0'
    old_release = '2.8.1'

    def _run(self):
        """
        Lancement des traitements
        """
        if self.var_exists('type_squid_auth') and self.is_default('type_squid_auth') and not self.var_exists('ad_realm'):
            if self.var_exists('activer_winbind'):
                # target 2.10+
                self.set_value('type_squid_auth', 'NTLM')
                self.set_value('winbind_auth_mode', 'SMB')
            else:
                # target 2.9
                self.set_value('type_squid_auth', 'NTLM/SMB')

        for index in range(int(self.get_value('nombre_interfaces'))):
            if f'debit_carte_eth{index}' not in self.unknown_options:
                continue
            value = self.unknown_options[f'debit_carte_eth{index}']['val']
            del self.unknown_options[f'debit_carte_eth{index}']
            if value == 'autoneg on' or not self.is_default(f'debit_carte_eth{index}_autoneg') or not self.is_default(f'debit_carte_eth{index}_speed') or not self.is_default(f'debit_carte_eth{index}_duplex'):
                continue
            try:
                debit_carte_eth_autoneg = 'non'
                speed, debit_carte_eth_speed, duplex, debit_carte_eth_duplex, other = value.split(' ', 4)
                self.set_value(f'debit_carte_eth{index}_autoneg', debit_carte_eth_autoneg)
                self.set_value(f'debit_carte_eth{index}_speed', debit_carte_eth_speed)
                self.set_value(f'debit_carte_eth{index}_duplex', debit_carte_eth_duplex)
            except Exception as err:
                log.error(_('cannot convert debit_carte_eth{} {}').format(index, err))
        if self.var_exists('clam_max_file_size') and self.is_default('clam_max_file_size'):
            self.set_value('clam_max_file_size', 10)
        if self.var_exists('clam_max_scan_size') and self.is_default('clam_max_scan_size'):
            self.set_value('clam_max_scan_size', 50)
        if self.var_exists('clam_max_files') and self.is_default('clam_max_files'):
            self.set_value('clam_max_files', 5000)
        if self.var_exists('ad_trace_connect_disconnect') and self.get_value('ad_trace_connect_disconnect') == 'oui':
            self.set_value('ad_log_vfs', 'oui')
            self.set_value('full_audit_on_success', ['connect'])


class Upgrade_2_10_0(Upgrade):
    """
    Mise à jour d'une configuration
    de 2.9.0 vers 2.10.0
    """

    current_release = '2.10.0'
    old_release = '2.9.0'

    def _run(self):
        """
        Lancement des traitements
        """
        if self.var_exists('type_squid_auth'):
            if self.get_unvalid_value('type_squid_auth') == 'NTLM/SMB':
                self.set_value('winbind_auth_mode', 'SMB')
                self.set_value('type_squid_auth', 'NTLM')
            elif self.get_unvalid_value('type_squid_auth') == 'NTLM/KERBEROS':
                self.set_value('winbind_auth_mode', 'KERBEROS')
                self.set_value('type_squid_auth', 'NTLM')


class UpgradeGraph:
    def __init__(self):
        self.graph = {}
        upgrade_classes = [globals()[f]
                           for f in globals()
                           if f.startswith('Upgrade_')]
        for uc in upgrade_classes:
            self.add_vertex(uc.old_release)
            self.add_vertex(uc.current_release)
            self.add_edge(uc.old_release, uc.current_release, uc)


    def add_vertex(self, vertex):
        if vertex not in self.graph:
            self.graph[vertex] = {}

    def add_edge(self, from_vertex, to_vertex, func):
        if from_vertex in self.graph and to_vertex in self.graph:
            self.graph[from_vertex][to_vertex] = func
        else:
            raise ValueError("One or both vertices not found in the graph.")

    def display(self):
        for vertex in self.graph:
            print(f"{vertex} -> {', '.join(self.graph[vertex])}")

    def find_shortest_path(self, start, end, path=None):
        if not path:
            path = []
        path = path + [start]
        if start == end:
            return path
        if start not in self.graph:
            return None
        shortest = None
        for node in self.graph[start]:
            if node not in path:
                newpath = self.find_shortest_path(node, end, path)
                if newpath:
                    if not shortest or len(newpath) < len(shortest):
                        shortest = newpath
        return shortest

    def get_upgrade_path(self, start, end):
        versions = self.find_shortest_path(start, end)
        steps = [versions[i-2:i] for i in range(len(versions), 1, -1)]
        steps.reverse()
        upgrade_path = []
        for step in steps:
            upgrade_path.append(self.graph[step[0]][step[1]])
        return upgrade_path


def upgrade2(old_release, current_release, config):
    """
    old_release: version du config.eol à migrer (ex : 2.4.0)
    current_release: version du serveur (ex : 2.5.1)
    config: objet de configuration Tiramisu
    """

    upgrade_graph = UpgradeGraph()
    for uc in upgrade_graph.get_upgrade_path(old_release, current_release):
        upgrade = uc(config)
        last_migration = upgrade.run()

    return last_migration
