# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agents zephir pour Information Système
"""

from zephir.monitor.agentmanager.agent import MultiRRDAgent
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.data import TableData, HTMLData

from twisted.internet.utils import getProcessOutput

SECONDS_PER_DAY = 3600*24


def _mega(val):
    """ transfo de la valeur passée (string en octets) en Mo
    """
    return int(val)/1024/1024

def _pourcent(use, tot):
    """ calcul du pourcentage de mémoire utilisée
    """
    #return (str(int(use)*100/int(tot))+"%")
    if int(tot) != 0:
        return (int(int(use)*100/int(tot)))
    return 0

def _stats(x):
    return "%.1f" % x

class SysInfo(MultiRRDAgent):
    """
    Bilan de l'etat de la memoire
    présentation en tableau
    """

    def __init__(self, name, **params):
        MultiRRDAgent.__init__(self, name, **params)
        self.fich = "/proc/meminfo"
        self.swap_pourcent = 0
        self.kernel_version = ""
        self.table = TableData([
            ('name', 'Type', {'align':'right'}, None),
            ('perc', 'utilisation', {'align':'right'}, None),
            ('free', 'libre', {'align':'right'}, None),
            ('used', 'utilisé', {'align':'right'}, None),
            ('size', 'taille', {'align':'right'}, None) ])
        self.table2 = TableData([
            ('name', ' ', {'align':'left'}, None),
            ('value', 'valeur', {'align':'right'}, None)])

    def init_data(self, archive_dir):
        """on initialise les archives rrd, et on définit
        la liste des données"""
        title1 = HTMLData("<h3>Information processeur<h3>")
        title2 = HTMLData("<h3>Informations mémoire<h3>")
        title3 = HTMLData("<h3>Historique CPU/mémoire<h3>")
        self.data.extend([title1, self.table2, title2, self.table, title3])
        MultiRRDAgent.init_data(self,archive_dir)
        #self.data.extend([title3, self.table2])

    def measure(self):
        self.dico = {}
        self.measure_mem()
        self.measure_load_rrd()
        data = self.measure_proc()
        return data

    def measure_mem(self):
        # mémoire
        out = open(self.fich)
        res = out.read().strip()
        out.close()
        res=res.split("\n")
        statistics = []
        self.mem_stats = {}
        for line in res:
            data=line.split()
            self.mem_stats[data[0][:-1]]=data[1]

        mem_used = int(self.mem_stats['MemTotal']) - int(self.mem_stats['MemFree'])
        mem_phys = {'name' : 'physique',
                 'perc' : str(_pourcent(mem_used,self.mem_stats['MemTotal']))+" %",
                 'free' : "%.2f Mo" % _mega(int(self.mem_stats['MemFree'])*1024),
                 'used' : "%.2f Mo" % _mega((mem_used)*1024),
                 'size' : "%.2f Mo" % _mega(int(self.mem_stats['MemTotal'])*1024),
                }
        self.measure_data['physique'] = mem_phys['perc']
        self.measure_data['physique_total'] = mem_phys['size']
        self.measure_data['physique_used'] = mem_phys['used']
        self.measure_data['physique_free'] = mem_phys['free']
        statistics.append(mem_phys)
        swap_used = int(self.mem_stats['SwapTotal']) - int(self.mem_stats['SwapFree'])
        mem_swap = {'name' : 'swap',
                 'perc' : str(_pourcent(swap_used,self.mem_stats['SwapTotal']))+" %",
                 'free' : "%.2f Mo" % _mega(int(self.mem_stats['SwapFree'])*1024),
                 'used' : "%.2f Mo" % _mega((swap_used)*1024),
                 'size' : "%.2f Mo" % _mega(int(self.mem_stats['SwapTotal'])*1024),
                }
        # sentinelle : pourcentage de swap
        self.measure_data['swap'] = str(_pourcent(swap_used,self.mem_stats['SwapTotal']))
        self.measure_data['swap_total'] = mem_swap['size']
        self.measure_data['swap_used'] = mem_swap['used']
        self.measure_data['swap_free'] = mem_swap['free']
        self.swap_pourcent = int(self.measure_data['swap'])
        statistics.append(mem_swap)
        self.dico['statistics'] = statistics
        # stats pour graphes RRD
        # pourcentage de mémoire pour le cache
        self.dico['cached'] = _pourcent(self.mem_stats['Cached'],self.mem_stats['MemTotal'])
        # swapping
        self.dico['sperc'] = _pourcent((int(self.mem_stats['SwapTotal']) - int(self.mem_stats['SwapFree'])), self.mem_stats['SwapTotal'])

    def measure_proc(self):
        version = getProcessOutput('/bin/uname',
                                   args = ['-r'],
                                   env = {'LC_ALL': 'C'})
        version.addCallback(self.measure_uptime)
        version.addErrback(self.error)
        return version

    def measure_uptime(self, version):
        self.kernel_version = version.strip()
        uptime = getProcessOutput('/usr/bin/uptime',
                                   env = {'LC_ALL': 'C'})
        uptime.addCallback(self.measure_process)
        uptime.addErrback(self.error)
        return uptime

    def error(self, err):
        self.status = status.Error()

    def measure_process(self,result):
        result = result.decode()
        lignes = result.splitlines()
        # On parse la sortie standard de la commande
        ligne = lignes[0]
        # nombre d'utilisateurs
        champs=ligne[:ligne.index('user')]
        users=str(int(champs[champs.rindex(',')+1:]))
        # on récupère le temps d'uptime
        uptime_str = champs[:champs.rindex(',')]
        uptime_str = uptime_str[champs.index('up')+2:].strip()
        statistics = []
        statistics.append( { 'name' : "Serveur en marche depuis",
                             'value': uptime_str.replace('day','jour') }
                         )
        statistics.append( { 'name' : "Nombre d'utilisateurs système connectés",
                             'value': users }
                         )
        # version du noyau
        statistics.append( { 'name' : "Version du noyau",
                             'value': self.kernel_version.decode() }
                         )
        self.dico['statistics2'] = statistics
        self.measure_data['kernel'] = self.kernel_version
        # sentinelle : uptime et version kernel
        f_uptime = open("/proc/uptime")
        uptime = f_uptime.read().strip().split()[0]
        f_uptime.close()
        self.measure_data['uptime'] = int(uptime.split('.')[0])
        return self.dico

    def measure_load_rrd(self):
        # cpu load average
        f_load = open("/proc/loadavg")
        # on garde la moyenne sur 5 minutes
        loadavg = f_load.read().strip().split()
        f_load.close()
        self.measure_data['load1'] = loadavg[0]
        self.measure_data['load5'] = loadavg[1]
        self.measure_data['load15'] = loadavg[2]
        self.dico['load'] = loadavg[0]

    def write_data(self):
        MultiRRDAgent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['statistics']
            self.table2.table_data = self.last_measure.value['statistics2']

    def check_status(self):
        """
        Warning en cas de Swap important
        Erreur en cas de Swap très important
        """
        warn_step = 60
        error_step = 90
        if self.swap_pourcent >= error_step:
            return status.Error("Swap utilisé à plus de %d%%" % error_step)
        if self.swap_pourcent >= warn_step:
            return status.Warn("Swap utilisé à plus de %d%%" % warn_step)
        return status.OK()
