# Copyright (C) 2010 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

"""Various tools to compare files and images, take screenshot, run commands,
manage packages, ...
"""

from filecmp import cmp
import subprocess
import gconf

def a11y_enabled():
    """Return True is a11y is enabled"""
    client = gconf.client_get_default()
    return client.get_bool('/desktop/gnome/interface/accessibility')


def file_compare(file1, file2):
    """Compare 2 files and returns True if files are equals"""
    try:
        return cmp(file1, file2)
    except OSError:
        return False

# http://www.chiark.greenend.org.uk/ucgi/~cjwatson/blosxom/2009-07-02-python-sigpipe.html
# This is needed so that the subprocesses that produce endless output
# actually quit when the reader goes away.
import signal
def subprocess_setup():
    # Python installs a SIGPIPE handler by default. This is usually not what
    # non-Python subprocesses expect.
    signal.signal(signal.SIGPIPE, signal.SIG_DFL)


def cmd(command, input = None, stderr = subprocess.STDOUT,
        stdout = subprocess.PIPE, stdin = None, env = None):
    """Try to execute given command (array) and return its stdout, or return
    a textual error if it failed.

    :param command:
    :param input: Default: None
    :param stderr: Default: STDOUT
    :param stdout:
    :param stdin:
    :param env:
    """

    try:
        sp = subprocess.Popen(command, stdin=stdin, stdout=stdout, stderr=stderr, close_fds=True, preexec_fn=subprocess_setup, env = env)
    except OSError, e:
        return [127, str(e)]

    out, outerr = sp.communicate(input)
    # Handle redirection of stdout
    if out == None:
        out = ''
    # Handle redirection of stderr
    if outerr == None:
        outerr = ''
    return [sp.returncode,out+outerr]

def dpkg(package, action = 'status', queryformat = "${Package} ${Version}\t${Status}"):
    """Execute a dpkg action on a package. The user needs sudo privileges in
    order to execute the actions install, purge and remove.

    :param package: A package name or a package file for the action 'install'
    :param action: Valid actions are:
        - install: Install a package. the argument 'package' needs to be a
                   valid deb archive
        - remove: Remove a package
        - purge: Remove a package and its configuration files
        - query: Show information about the package.
                 Default format "${Package} ${Version}\t${Status}"
                 man dpkg-query for additional information
        - status: Report the status of a specified package. This is the default
                  action

    Returns the return code and output of dpkg
    """
    action = action.lower()
    if action == 'install':
        dpkg_cmd = ['sudo', 'dpkg', '-i', package]
    elif action == 'purge':
        dpkg_cmd = ['sudo', 'dpkg', '-p', package]
    elif action == 'remove':
        dpkg_cmd = ['sudo', 'dpkg', '-r', package]
    elif action == 'query':
        dpkg_cmd = ['dpkg-query', '-W', '-f', queryformat, package]
    elif action == 'status':
        dpkg_cmd = ['dpkg', '-s', package]
    else:
        raise NotImplementedError, "dpkg action '%s' is not supported" % action

    return cmd(dpkg_cmd)

def is_package_installed(package):
    """Query the dpkg status database and returns True is the specified
    package is installed

    :param package: Package name
    """
    (rc, output) = dpkg(package, action='query', queryformat = '${Status}')
    return output.startswith('install')

