# ----------------------------------------------------------------------
#
# This file is part of Rheolef.
#
# Copyright (C) 2000-2009 Pierre Saramito 
#
# Rheolef is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Rheolef is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Rheolef; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------
#  Inspirated from:
#  Effective Tcl/Tk Programming
#    Mark Harrison, DSC Communications Corp.
#    Michael McLennan, Bell Labs Innovations for Lucent Technologies
#    Addison-Wesley Professional Computing Series
# ======================================================================

set animate_data(counter) 0

# ----------------------------------------------------------------------
#  USAGE:  animate_start <delay> <values> <command>
#
#  This starts an animation sequence.  The given <command> is
#  evaluated at regular intervals of <delay> ms.  Each time, a
#  different value from the <values> list is substituted into
#  the "%v" field in the <command>.  When the <values> list is
#  exhausted, animation starts again with the first element.
#  Returns an identifier which can be passed to animate_stop
#  to terminate the animation.
# ----------------------------------------------------------------------
proc animate_start {delay vlist command} {
    set id [animate_create $delay $vlist $command]
    animate_play $id
    return $id
}
# ----------------------------------------------------------------------
#  USAGE:  animate_stop <id>
#
#  Terminates an animation previously started by animate_start.
#  If the <id> is not recognized, this command does nothing.
# ----------------------------------------------------------------------
proc animate_stop {id} {
    global animate_data

    if {[info exists animate_data($id-pending)]} {
        after cancel $animate_data($id-pending)
        if {[info exists animate_data($id-callback-at-end)]} {
            set pos $animate_data($id-pos)
            set val [lindex $animate_data($id-vlist) $pos]
            set cmd [percent_subst %v $animate_data($id-callback-at-end) $val]
            uplevel #0 $cmd
            unset animate_data($id-callback-at-end)
        }
        unset animate_data($id-delay)
        unset animate_data($id-vlist)
        unset animate_data($id-command)
        unset animate_data($id-pos)
        unset animate_data($id-pending)
        unset animate_data($id-do-infinite-loop)
    }
}
# ----------------------------------------------------------------------
#  USAGE:  animate_create <delay> <values> <command>
#
#  This defines an animation sequence but do not play it imediatly.
# ----------------------------------------------------------------------
proc animate_create {delay vlist command} {
    global animate_data

    set id "animate[incr animate_data(counter)]"

    set animate_data($id-delay)            $delay
    set animate_data($id-vlist)            $vlist
    set animate_data($id-command)          $command
    set animate_data($id-pos)              0
    set animate_data($id-do-infinite-loop) 0
    return $id
}
# ----------------------------------------------------------------------
#  USAGE:  animate_play <id>
# ----------------------------------------------------------------------
proc animate_play {id} {
    global animate_data

    if {[info exists animate_data($id-delay)]} {
        set delay                      $animate_data($id-delay)
        set animate_data($id-pending)  [after $delay "animate_handle $id"]
    }
    return $id
}
# ---------------------------------------------------------------------------
# USAGE: animate_configure <id> {option value}*
#  options recognized:
#	-callback-at-end  <cmd>
#	-delay            <delay>
#	-do-infinite-loop [0|1]
# ---------------------------------------------------------------------------
proc animate_configure {id args} {
    global animate_data
    set std_args ""
    while {[llength $args] > 0} {
        set arg [lindex $args 0]
        set args [lrange $args 1 end]
        switch x"$arg" {
            x"-callback-at-end" {
               set cmd  [lindex $args 0]
               set args [lrange $args 1 end]
               set animate_data($id-callback-at-end) $cmd
               continue
	    }
            default {
                if {[info exists animate_data(${id}-vlist)]} {
        	    set value [lindex $args 0]
        	    set args  [lrange $args 1 end]
        	    set animate_data(${id}${arg}) $value
	        }
            }
	}
    }
}
# ======================================================================
# private members
# ======================================================================
# ----------------------------------------------------------------------
#  USAGE:  animate_handle <id>
#
#  Used internally to handle the various frames of an animation.
#  Executes the animation command with the current value substituted
#  into the "%v" field, then bumps the position to the next value
#  and schedules another call to this procedure.
# ----------------------------------------------------------------------
proc animate_handle {id} {
    global animate_data

    if {[info exists animate_data($id-pending)]} {
        set pos $animate_data($id-pos)
        set val [lindex $animate_data($id-vlist) $pos]
        set cmd [percent_subst %v $animate_data($id-command) $val]

        uplevel #0 $cmd
    } else {
	return
    }
    # $cmd can take a while and stop occurs here

    if {[info exists animate_data($id-vlist)]} {
        if {[incr pos] >= [llength $animate_data($id-vlist)]} {
            set pos 0
	}
        set animate_data($id-pos) $pos
    }
    if {[info exists animate_data($id-do-infinite-loop)]} {
        if {$pos != 0 || $animate_data($id-do-infinite-loop)} {
          if {[info exists animate_data($id-delay)]} {
            set animate_data($id-pending) [after $animate_data($id-delay) "animate_handle $id"]
	  }
	} else {
	  animate_stop $id
        }
    }
}
