-- ------------------------------------------------------------------------- --
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/contrib/objects/persistent.adb,v $
--  Description     : Base class for all persistent objects                  --
--  Author          : Michael Erdmann <Michael.Erdmann@snafu.de>             --
--  Created On      : 30-April-2005                                          --
--  Last Modified By: $Author: merdmann $                                    --
--  Last Modified On: $Date: 2007/02/03 14:59:52 $                           --
--  Status          : $State: Exp $                                          --
--                                                                           --
--  Copyright (C) 2006-2007 Michael Erdmann                                  --
--                                                                           --
--  GNADE is copyrighted by the persons and institutions enumerated in the   --
--  AUTHORS file. This file is located in the root directory of the          --
--  GNADE distribution.                                                      --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNADE;  see file COPYING. If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from      --
--  GNADE Ada units, or you link GNADE Ada units or libraries with other     --
--  files  to produce an executable, these  units or libraries do not by     --
--  itself cause the resulting  executable  to  be covered  by the  GNU      --
--  General  Public  License.  This exception does not however invalidate    --
--  any other reasons why  the executable file  might be covered by the      --
--  GNU Public License.                                                      --
--                                                                           --
-- ------------------------------------------------------------------------- --
with Ada.Streams;               	use Ada.Streams;
with Ada.Streams.Stream_IO;     	use Ada.Streams.Stream_IO;
with Ada.Text_IO;			use Ada.Text_IO;
with Ada.Calendar;			use Ada.Calendar;
with Ada.Tags;				use Ada.Tags;
with Ada.Unchecked_Conversion;
use  Ada;

with GNU.DB.SQLCLI;			use GNU.DB.SQLCLI;
with SQL_Standard;      		use SQL_Standard;

with Util.Trace_Helper;
use  Util;

with Cache;				use Cache;
with Objects;				use Objects;
with BLOB_Stream;			use BLOB_Stream;
with MD5;				use MD5;

package body Persistent is

   Version : constant String :=
       "$Id: persistent.adb,v 1.16 2007/02/03 14:59:52 merdmann Exp $";

   -- Trace facitlity package
   package Tracer is new Util.Trace_Helper( Module => "Persistent");
   use Tracer;

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize(
      This : in out Object ) is
   begin
      Enter("Initialize");

      Leave("Initialize");
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize(
      This : in out Object ) is
   begin
      Enter("Finalize( " &
            "ID=" & Integer'Image(OID(This)) & ", " &
            "Version=" & Integer'Image(This.Version) & ")" );

      Cache.Invalidate( OID(This), This.Version );

      -- Finalize( Persistent.Object(This) );

      Leave("Finalize");
   end Finalize;

   -----------------
   -- Get_Object --
   -----------------
   procedure Get_Object(
      This    : in out Object'Class;
      Create  : in Boolean := False;
      Version : in Integer := Highest_Version;
      ID      : Integer := 0) is
      -- open the persitent object for reading
      S       : Stream_Access := BLOB_Stream.Stream( Blob_Size );
      BLOB    : BLOB_Access := null;
      Length  : Integer := 0;
      Vers    : Integer := Version;
   begin
      Enter("Get_Object(" &
            Integer'Image(OID(This)) & ", " &
            Boolean'Image(Create) & ", " &
            Integer'Image(Version) & ", " &
            Integer'Image(ID) & ")" );

      if ID /= 0 then
         OID( This, ID );
      end if;

      begin
         Cache.Get_Object( OID(This), Vers, BLOB, Length );

      exception
         when Object_Not_Existing =>
            if Create then
               Put_Object( This, Create => True );
            else
               raise ;
            end if;
      end ;

      This.Version := Vers;
      Set_Contents( S, BLOB(1..Length) );

      This := Object'Class'Input(S);

      Self( This );

      Destroy(S);
      Leave("Get_Object");
   end Get_Object;

   ----------------
   -- Get_Object --
   ----------------
   procedure Get_Object(
      This    : in out Object'Class;
      Key     : Key_Type.Object'Class;
      Create  : in Boolean := False;
      Version : in Integer := Highest_Version ) is
      -- open the persitent object for reading
      S       : Stream_Access := BLOB_Stream.Stream( Blob_Size );
      BLOB    : BLOB_Access := null;
      Length  : Integer := 0;
      Vers    : Integer := Version;
   begin
      Enter("Get_Object( <key>, " &
            Boolean'Image(Create) & ", " &
            Integer'Image(Version) & ")" );
      begin
         Cache.Get_Object( Hash(Key), Vers, BLOB, Length );

      exception
         when Object_Not_Existing =>
            if Create then
               Put_Object( This, Key, Create => True );
            else
               raise ;
            end if;
      end ;

      This.Version := Vers;
      Set_Contents( S, BLOB(1..Length) );

      This := Object'Class'Input(S);

      Self( This );

      Destroy(S);

      Leave("Get_Object( <key>," &
            "Version = " & Integer'Image(Version) & "," &
            "Id =" & Integer'Image(OID(This)) & ")" );
   end Get_Object;

   ---------------
   -- To_Binary --
   ---------------
   function To_BINARY(
      F : in MD5.Fingerprint ) return GNADE.BINARY is

      function To_SQLCHAR is new Ada.Unchecked_Conversion
        (Target => SQL.SQLCHAR, Source  => Stream_Element );

      Result : GNADE.BINARY( 1..16 );
   begin
      for i in F'Range loop
         Result(Integer(i)) := To_SQLCHAR( F(i) );
      end loop;

      return Result;
   end To_BINARY;

   ----------------
   -- Put_Object --
   ----------------
   procedure Put_Object(
      This    : in out Object'Class;
      Create  : in Boolean := True ) is
      -- store an object and create a new version if requested.
      Vers    : Integer := New_Version;
      S       : Stream_Access := BLOB_Stream.Stream( Blob_Size );
      BLOB    : GNADE.BINARY( 1..Blob_Size );
      Length  : Integer := 0;

   begin
      Enter("Put_Object( " &
            "This.ID =" & Integer'Image(OID(This)) & "," &
            "Create =" & Boolean'Image(Create) & ")" );

      if not Create then
         Vers := This.Version;
      end if;

      Object'Class'Output(S, This );
      Get_Contents( S, BLOB, Length );

      Cache.Put_Object(
         OID(This),
         Vers,
         Hash(This),
         BLOB(1..Length),
         External_Tag(This'Tag)
      );

      This.Version := Vers;

      Destroy(S);

      Leave("Put_Object( This.Version =" & Integer'Image(This.Version) & ")" );
   end Put_Object;

   ----------------
   -- Put_Object --
   ----------------
   procedure Put_Object(
      This    : in out Object'Class;
      Key     : in Key_Type.Object'Class;
      Create  : in Boolean := True ) is
      -- create an object with key
      Vers    : Integer := New_Version;
      S       : Stream_Access := BLOB_Stream.Stream( Blob_Size );
      BLOB    : GNADE.BINARY( 1..Blob_Size );
      Length  : Integer := 0;
   begin
      Enter("Put_Object( " &
            "This.ID =" & Integer'Image(OID(This)) & ", <key>, " &
            "Create =" & Boolean'Image(Create) & ")" );

      if not Create then
         Vers := This.Version;
      end if;

      Object'Class'Output(S, This );
      Get_Contents( S, BLOB, Length );

      Cache.Put_Object(
         OID(This),
         Vers,
         Key_Type.Hash(Key),
         BLOB(1..Length),
         External_Tag( This'Tag )
      );

      This.Version := Vers;

      Destroy(S);

      Leave("Put_Object( This.Version =" & Integer'Image(This.Version) & ")" );
   end Put_Object;

   ------------
   -- Relate --
   ------------
   procedure Relate(
      This     : in Object'Class;
      Other    : in Object'Class;
      Relation : in String ) is
   begin
      Relate( OID(This), OID(Other), Relation );
   end Relate;

   ----------
   -- Hash --
   ----------
   function Hash(
      This   : in Object'Class ) return GNADE.BINARY is
      S      : Stream_Access := BLOB_Stream.Stream( Blob_Size );
   begin
      Object'Class'Output(S, This );

      return To_Binary(Hash(S));
   end Hash;

end Persistent;
