/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace@gnome.org>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n.h>
#include <gnome-scan-module.h>
#include <gnome-scan-param-specs.h>
#include "gsfile-pspec.h"
#include "gsfile-scanner.h"

#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE ((o), GSFILE_TYPE_SCANNER, GSFileScannerPrivate))

typedef struct _GSFileScannerPrivate GSFileScannerPrivate;

struct _GSFileScannerPrivate
{
  GeglNode		*load;
};

static GnomeScannerClass* parent_class = NULL;

static void	gsf_configure			(GnomeScanPlugin *plugin,
									 GnomeScanSettings *settings);
static GList*	gsf_get_child_nodes	(GnomeScanPlugin *plugin,
										 GeglNode *root);
static gboolean	gsf_start_frame		(GnomeScanPlugin *plugin);
static gboolean	gsf_work			(GnomeScanPlugin *plugin,
										gdouble *progress);

GS_DEFINE_MODULE_TYPE (GSFileScanner, gsfile_scanner, GNOME_TYPE_SCANNER);

static void
gsfile_scanner_init (GSFileScanner *object)
{
	GParamSpec *pspec;
	GSList *formats = NULL;
	
	static gchar *png_mime[] = {
		"image/png",
		NULL
	};
	static gchar *png_exts[] = {
		"png",
		NULL
	};
	
	formats = g_slist_append (formats,
							  gnome_scan_format_new ("png",
													 GETTEXT_PACKAGE,
													 N_("PNG picture"),
													 png_mime,
													 png_exts));
	
	static gchar *jpeg_mime[] = {
		"image/jpeg",
		NULL
	};
	static gchar *jpeg_exts[] = {
		"jpeg",
		"jpe",
		"jpg",
		NULL
	};
	
	formats = g_slist_append (formats,
							  gnome_scan_format_new ("jpeg",
													 GETTEXT_PACKAGE,
													 N_("JPEG picture"),
													 jpeg_mime,
													 jpeg_exts));
	
	pspec = gsfile_param_spec_filenames ("input-filenames",
										 N_("Files"),
										 N_("Files to import."),
										 GS_PARAM_GROUP_SCANNER_FRONT,
										 formats, G_PARAM_WRITABLE);
	gs_param_spec_set_domain (pspec, GETTEXT_PACKAGE);
	gs_param_spec_set_unit (pspec, GS_UNIT_NONE);
	gnome_scan_plugin_params_add (GNOME_SCAN_PLUGIN (object), pspec);
	GNOME_SCANNER (object)->status = GNOME_SCANNER_READY;
}

static void
gsfile_scanner_finalize (GObject *object)
{
	/* TODO: Add deinitalization code here */
	
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gsfile_scanner_class_init (GSFileScannerClass *klass)
{
	GObjectClass* object_class = G_OBJECT_CLASS (klass);
	parent_class = GNOME_SCANNER_CLASS (g_type_class_peek_parent (klass));
	GnomeScanPluginClass* plugin_class = GNOME_SCAN_PLUGIN_CLASS (klass);
	
	g_type_class_add_private (object_class, sizeof (GSFileScannerPrivate));
	
	plugin_class->configure			= gsf_configure;
	plugin_class->start_frame		= gsf_start_frame;
	plugin_class->work				= gsf_work;
	plugin_class->get_child_nodes   = gsf_get_child_nodes;
	
	object_class->finalize	= gsfile_scanner_finalize;
}

GnomeScanner*
gsfile_scanner_new ()
{
	GObject *object = g_object_new (GSFILE_TYPE_SCANNER,
									/* translator: this is the name of the file
									 loader backend which allow user to scan 
									 a list a images */
									"name", _("Files"),
									"blurb", _("Import from files."),
									"icon-name", "gnome-mime-image",
									NULL);
	return GNOME_SCANNER (object);
}

/* INTERNALS */

static void
gsf_configure (GnomeScanPlugin *plugin, GnomeScanSettings *settings)
{
	GSFileScanner *scanner = GSFILE_SCANNER (plugin);
	GValue *value;
	
	value = gnome_scan_settings_get (settings, "input-filenames");
	scanner->filenames = g_value_get_pointer (value);
	scanner->current = scanner->filenames;
}

static GList*
gsf_get_child_nodes (GnomeScanPlugin *plugin,
					 GeglNode *root)
{
	GSFileScanner *scanner = GSFILE_SCANNER (plugin);
	GSFileScannerPrivate *priv = GET_PRIVATE (scanner);
	GList *list = NULL;
	
	priv->load = gegl_node_new_child (root,
									  "operation", "gegl:load",
									  NULL);
	
	list = g_list_append (list, priv->load);
	return list;
}


static gboolean
gsf_start_frame (GnomeScanPlugin *plugin)
{
	GSFileScanner *scanner = GSFILE_SCANNER (plugin);
	GSFileScannerPrivate *priv = GET_PRIVATE (scanner);
	
	if (scanner->current == NULL) {
		return FALSE;
	}
	
	gegl_node_set (priv->load,
				   "path", scanner->current->data,
				   NULL);
	
	scanner->current = scanner->current->next;
	return TRUE;
}

static gboolean
gsf_work (GnomeScanPlugin *plugin, gdouble *progress)
{
  *progress = 1.;
  return FALSE;
}
