# Generate Go types and signatures for the LSP protocol

## Setup

1. Make sure `node` and `tsc` are installed. There are detailed instructions below.
2. Get the typescript code for the jsonrpc protocol with `git clone git@github.com:microsoft vscode-languageserver-node.git`
    1. If you want to reproduce the existing files you need to be on a branch with the same git hash, for instance, `git checkout 635ab1f`

## Usage

To generate the protocol types (x/tools/internal/lsp/protocol/tsprotocol.go)
```tsc go.ts && node go.js [-d dir] [-o out.go]```

and for simple checking

```gofmt -w out.go && golint out.go && go build out.go```

`-d dir` names the directory into which the `vscode-languageserver-node` repository was cloned.
It defaults to `$(HOME)`.

`-o out.go` says where the generated go code goes.
It defaults to `tsprotocol.go`.

To generate the client and server boilerplate (tsclient.go and tsserver.go)
```tsc requests.ts && node requests.js [-d dir] && gofmt -w tsclient.go tsserver.go```

-d dir is the same as above. The output files are written into the current directory.

## Notes

1. `go.ts` and `requests.ts` use the Typescript compiler's API, which is [introduced](https://github.com/Microsoft/TypeScript/wiki/Architectural-Overview) in their wiki.
2. Because the Typescript and Go type systems are incompatible, `go.ts` and `request.ts` are filled with heuristics and special cases. Therefore they are tied to a specific commit of `vscode-languageserver-node`. The hash code of the commit is included in the header of `tsprotocol.go` and stored in the variable `gitHash` in `go.ts`. It is checked (see `git()` in `go.ts`) on every execution of `go.ts`.
3. Generating the `ts*.go` files is only semi-automated. Please file an issue if the released version is too far behind.
4. For the impatient, first change `gitHash` by hand (`git()` shows how to find the hash).
    1. Then try to run `go.ts` and  `requests.ts`. This will likely fail because the heuristics don't cover some new case. For instance, some simple type like `string` might have changed to a union type `string | [number,number]`. (Look at the `UnionTypeNode` code near line 588 of `go.ts`.) Another example is that some formal parameter generated by `requests.ts` will have anonymous structure type, which is essentially unusable. (See the code related to `ourTypes`.)
    1. Next step is to try to move the generated code to `internal/lsp/protocol` and try to build `gopls` and its tests. This will likely fail because types have changed. Generally the fixes are fairly easy. (The code for `ourTypes` was a case where changes had to be made to `requests.ts`.)
    1. Since there are not adequate integration tests, the next step is to run `gopls`. A common failure will be a nil dereference, because some previously simple default is now in an optional structure.

## Detailed instructions for installing node and typescript

(The instructions are somewhat different for  Linux and MacOS. They install some things locally, so `$PATH` needs to be changed.)

1. For Linux, it is possible to build node from scratch, but if there's a package manager, that's simpler.
    1. To use the Ubuntu package manager
        1. `sudo apt update` (if you can't `sudo` these instructions are not helpful)
        2. `sudo apt install nodejs` (this may install `/usr/bin/nodejs` rather than `/usr/bin/node`. For me, `/usr/bin/nodejs` pointed to an actual executable `/etc/alternatives/nodejs`, which should be copied to `/usr/bin/node`)
        3. `sudo apt intall npm`
    1. To build from scratch
        1. Go to the [node site](https://nodejs.org), and download the one recommended for most users, and then you're on your own. (It's got binaries in it. Untar the file somewhere and put its `bin` directory in your path, perhaps?)
2. The Mac is easier. Download the macOS installer from [nodejs](https://nodejs.org), click on it, and let it install.
3. (There's a good chance that soon you will be asked to upgrade your new npm. `sudo npm install -g npm` is the command.)
4. For either system, node and nvm should now be available. Running `node -v` and `npm -v` should produce version numbers.
5. `npm install typescript`
    1. This will likely give warning messages that indicate you've failed to set up a project. Ignore them.
    2. Your home directory will now have new directories `.npm` and `node_modules` (and a `package_lock.json` file)
    3. The typescript executable `tsc` will be in `node_modules/.bin`, so put that directory in your path.
    4. `tsc -v` should print "Version 3.7.2" (or later). If not you may (as I did) have an obsolete tsc earlier in your path.
6. `npm install @types/node` (Without this there will be many incomprehensible typescript error messages.)
