use nettle_sys::{
    nettle_hash, nettle_sha3_384, nettle_sha3_384_digest, nettle_sha3_384_init,
    nettle_sha3_384_update, sha3_384_ctx,
};
use std::default::Default;
use std::mem::zeroed;

use crate::hash::NettleHash;
use crate::hash::Hash;

/// 384 bit variant of the Secure Hash Algorithm 3 (SHA-3) defined in FIPS 202.
#[allow(non_camel_case_types)]
pub struct Sha3_384 {
    context: sha3_384_ctx,
}

impl_write_for_hash!(Sha3_384);

impl Clone for Sha3_384 {
    fn clone(&self) -> Self {
        use std::intrinsics::copy_nonoverlapping;

        unsafe {
            let mut ctx: sha3_384_ctx = zeroed();
            copy_nonoverlapping(&self.context, &mut ctx, 1);

            Sha3_384 { context: ctx }
        }
    }
}

impl Default for Sha3_384 {
    fn default() -> Self {
        let mut ctx = unsafe { zeroed() };

        unsafe {
            nettle_sha3_384_init(&mut ctx as *mut _);
        }

        Sha3_384 { context: ctx }
    }
}

impl Hash for Sha3_384 {
    fn digest_size(&self) -> usize {
        ::nettle_sys::SHA3_384_DIGEST_SIZE as usize
    }

    fn update(&mut self, data: &[u8]) {
        unsafe {
            nettle_sha3_384_update(
                &mut self.context as *mut _,
                data.len(),
                data.as_ptr(),
            );
        }
    }

    fn digest(&mut self, digest: &mut [u8]) {
        unsafe {
            nettle_sha3_384_digest(
                &mut self.context as *mut _,
                digest.len(),
                digest.as_mut_ptr(),
            );
        }
    }

    fn box_clone(&self) -> Box<dyn Hash> {
        Box::new(self.clone())
    }
}

impl NettleHash for Sha3_384 {
    type Context = sha3_384_ctx;

    unsafe fn nettle_hash() -> &'static nettle_hash {
        &nettle_sha3_384
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn do_nothing() {
        let _ = Sha3_384::default();
    }

    //  CAVS 19.0
    //  "SHA3-384 ShortMsg" information for "SHA3AllBytes1-28-16"
    //  Length values represented in bits
    //  Generated on Thu Jan 28 13:32:46 2016
    #[test]
    fn nist_cavs_short_msg() {
        let mut ctx = Sha3_384::default();
        let mut digest = vec![0u8; ctx.digest_size()];

        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0c\x63\xa7\x5b\x84\x5e\x4f\x7d\x01\x10\x7d\x85\x2e\x4c\x24\x85\xc5\x1a\x50\xaa\xaa\x94\xfc\x61\x99\x5e\x71\xbb\xee\x98\x3a\x2a\xc3\x71\x38\x31\x26\x4a\xdb\x47\xfb\x6b\xd1\xe0\x58\xd5\xf0\x04"[..]);

        ctx.update(b"\x80");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x75\x41\x38\x48\x52\xe1\x0f\xf1\x0d\x5f\xb6\xa7\x21\x3a\x4a\x6c\x15\xcc\xc8\x6d\x8b\xc1\x06\x8a\xc0\x4f\x69\x27\x71\x42\x94\x4f\x4e\xe5\x0d\x91\xfd\xc5\x65\x53\xdb\x06\xb2\xf5\x03\x9c\x8a\xb7"[..]);

        ctx.update(b"\xfb\x52");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd7\x3a\x9d\x0e\x7f\x18\x02\x35\x2e\xa5\x4f\x3e\x06\x2d\x39\x10\x57\x7b\xf8\x7e\xdd\xa4\x81\x01\xde\x92\xa3\xde\x95\x7e\x69\x8b\x83\x60\x85\xf5\xf1\x0c\xab\x1d\xe1\x9f\xd0\xc9\x06\xe4\x83\x85"[..]);

        ctx.update(b"\x6a\xb7\xd6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xea\x12\xd6\xd3\x2d\x69\xad\x21\x54\xa5\x7e\x0e\x1b\xe4\x81\xa4\x5a\xdd\x73\x9e\xe7\xdd\x6e\x2a\x27\xe5\x44\xb6\xc8\xb5\xad\x12\x26\x54\xbb\xf9\x51\x34\xd5\x67\x98\x71\x56\x29\x5d\x5e\x57\xdb"[..]);

        ctx.update(b"\x11\x58\x7d\xcb");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xcb\x6e\x6c\xe4\xa2\x66\xd4\x38\xdd\xd5\x28\x67\xf2\xe1\x83\x02\x1b\xe5\x02\x23\xc7\xd5\x7f\x8f\xdc\xaa\x18\x09\x3a\x9d\x01\x26\x60\x7d\xf0\x26\xc0\x25\xbf\xf4\x0b\xc3\x14\xaf\x43\xfd\x8a\x08"[..]);

        ctx.update(b"\x4d\x7f\xc6\xca\xe6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe5\x70\xd4\x63\xa0\x10\xc7\x1b\x78\xac\xd7\xf9\x79\x0c\x78\xce\x94\x6e\x00\xcc\x54\xda\xe8\x2b\xfc\x38\x33\xa1\x0f\x0d\x8d\x35\xb0\x3c\xbb\x4a\xa2\xf9\xba\x4b\x27\x49\x88\x07\xa3\x97\xcd\x47"[..]);

        ctx.update(b"\x5a\x66\x59\xe9\xf0\xe7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x21\xb1\xf3\xf6\x3b\x90\x7f\x96\x88\x21\x18\x5a\x7f\xe3\x0b\x16\xd4\x7e\x1d\x6e\xe5\xb9\xc8\x0b\xe6\x89\x47\x85\x4d\xe7\xa8\xef\x4a\x03\xa6\xb2\xe4\xec\x96\xab\xdd\x4f\xa2\x9a\xb9\x79\x6f\x28"[..]);

        ctx.update(b"\x17\x51\x0e\xca\x2f\xe1\x1b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x35\xfb\xa6\x95\x8b\x6c\x68\xea\xe8\xf2\xb5\xf5\xbd\xf5\xeb\xcc\x56\x52\x52\xbc\x70\xf9\x83\x54\x8c\x2d\xfd\x54\x06\xf1\x11\xa0\xa9\x5b\x1b\xb9\xa6\x39\x98\x8c\x8d\x65\xda\x91\x2d\x2c\x3e\xa2"[..]);

        ctx.update(b"\xc4\x4a\x2c\x58\xc8\x4c\x39\x3a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x60\xad\x40\xf9\x64\xd0\xed\xcf\x19\x28\x1e\x41\x5f\x73\x89\x96\x82\x75\xff\x61\x31\x99\xa0\x69\xc9\x16\xa0\xff\x7e\xf6\x55\x03\xb7\x40\x68\x31\x62\xa6\x22\xb9\x13\xd4\x3a\x46\x55\x9e\x91\x3c"[..]);

        ctx.update(b"\xa3\x6e\x5a\x59\x04\x3b\x63\x33\xd7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xbd\x04\x56\x61\x66\x34\x36\xd0\x77\x20\xff\x3c\x8b\x6f\x92\x20\x66\xdf\xe2\x44\x45\x6a\x56\xca\x46\xdf\xb3\xf7\xe2\x71\x11\x6d\x93\x21\x07\xc7\xb0\x4c\xc7\xc6\x01\x73\xe0\x8d\x0c\x2e\x10\x7c"[..]);

        ctx.update(b"\xc0\x92\x0f\x2b\xd1\xe2\xd3\x02\x25\x9b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x3d\x15\x84\x22\x04\x09\xf8\x8d\x38\x40\x9a\x29\xec\xae\xbb\x49\x0e\xf8\x84\xb5\xac\xba\x2c\x7e\xaf\x23\x91\x4b\xab\x7f\x5f\x0f\xc9\x7e\xe1\xe6\x33\x6f\x88\xdf\xd4\xd0\xa0\x6e\x90\x2c\xcd\x25"[..]);

        ctx.update(b"\x70\xae\x73\x1a\xf5\xe0\xd9\x2d\x26\x4e\xc9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x56\x33\x59\xfd\x93\xfe\x09\xf3\xfe\x49\xfc\xf5\xf1\x7e\x7f\x92\xaa\xb5\x89\xcd\xec\x3e\x55\xe4\xc3\x71\x5e\x77\x75\x81\x4b\xbb\xfb\x8c\x4c\x73\x2e\x28\xd3\xb6\xe6\x40\x48\x60\x81\x2d\xc6\xe9"[..]);

        ctx.update(b"\x69\xc7\x4a\x9b\x0d\xb5\x38\xee\xff\x64\xd9\x3d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x88\xc6\x63\x89\xca\x2c\x32\x0a\x39\x02\x2a\xa4\x41\xfa\x88\x4f\xbc\x6e\xd2\xd3\xcc\x9a\xc4\x75\x37\x2d\x94\x7d\x49\x60\x57\x9a\x64\xe0\x61\xa2\x97\xd1\x83\x1d\x35\x24\xf9\x8d\x80\x94\x40\x4b"[..]);

        ctx.update(b"\xa4\xa9\x32\x7b\xe2\x1b\x92\x77\xe0\x8c\x40\xab\xc7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x75\x1f\x5d\xa5\xff\x9e\x24\x60\xc9\x93\x48\x07\x0d\x50\x68\xd8\xa3\xd7\xff\xce\xc7\xfd\x0e\x6f\x68\xf6\xcd\x4a\x2e\xf4\x22\x6d\xf8\xd9\xb4\x61\x3c\x3b\x0d\x10\xa1\x68\xea\xf5\x4e\xab\xe0\x1a"[..]);

        ctx.update(b"\xcc\x47\x64\xd3\xe2\x95\x09\x72\x98\xf2\xaf\x88\x82\xf6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x10\xf2\x87\xf2\x56\x64\x3a\xd0\xdf\xb5\x95\x5d\xd3\x45\x87\x88\x2e\x44\x5c\xd5\xae\x8d\xa3\x37\xe7\xc1\x70\xfc\x0c\x1e\x48\xa0\x3f\xb7\xa5\x4e\xc7\x13\x35\x11\x3d\xbd\xcc\xcc\xc9\x44\xda\x41"[..]);

        ctx.update(
            b"\x5a\x23\xad\x0c\xe8\x9e\x0f\xb1\xdf\x4a\x95\xbb\x24\x88\xf0",
        );
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x23\x84\x06\x71\xe7\x57\x0a\x24\x8c\xf3\x57\x9c\x7c\x88\x10\xb5\xfc\xc3\x5b\x97\x5a\x3a\x43\xb5\x06\xcc\x67\xfa\xef\xa6\xdb\xe1\xc9\x45\xab\xc0\x9a\x90\x3e\x19\x9f\x75\x9d\xcb\xc7\xf2\xc4\xd0"[..]);

        ctx.update(
            b"\x65\xb2\x7f\x6c\x55\x78\xa4\xd5\xd9\xf6\x51\x9c\x55\x4c\x30\x97",
        );
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdd\x73\x4f\x49\x87\xfe\x1a\x71\x45\x5c\xf9\xfb\x1e\xe8\x98\x68\x82\xc8\x24\x48\x82\x7a\x78\x80\xfc\x90\xd2\x04\x3c\x33\xb5\xcb\xc0\xed\x58\xb8\x52\x9e\x4c\x6b\xc3\xa7\x28\x88\x29\xe0\xa4\x0d"[..]);

        ctx.update(b"\xa7\x48\x47\x93\x0a\x03\xab\xee\xa4\x73\xe1\xf3\xdc\x30\xb8\x88\x15");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdb\xa6\xf9\x29\xfe\x55\xf9\xd6\x6c\x5f\x67\xc0\xaf\x3b\x82\xf1\x7b\xcf\x58\xb3\x67\x52\xf3\x16\x5c\x16\x08\x3f\xea\x8f\xd4\x78\xee\x69\x03\xf2\x7f\x82\x0a\xd2\xdd\x99\x50\xaf\xb4\x8c\x67\x00"[..]);

        ctx.update(b"\x6e\xfa\xf7\x8e\xd4\xd2\x93\x92\x7e\xef\x2c\x3a\x71\x93\x0e\x6e\x88\x7a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x82\x18\x49\x8a\xb0\x1b\x63\x04\x1c\x2b\xa0\x70\x9e\x33\x09\x49\x61\x24\xdd\xf0\x90\x45\x43\xa9\xe0\xd9\xd0\x96\xa7\x50\xdd\xa9\x7f\x7a\x02\x20\x8a\xf3\xd8\xc6\x18\xd4\xbe\x7c\x2b\xb2\xa2\x88"[..]);

        ctx.update(b"\xfd\x03\x9e\xb6\xe4\x65\x73\x88\xb9\x47\xec\x01\xe7\x37\xef\xbb\xad\x47\xda");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xc5\xb3\x13\x0e\xf8\xdb\xc5\x80\xe1\x10\x3f\xec\xae\x69\xc9\xa8\x82\xd9\xeb\xf5\xa3\xde\xf5\x93\x8b\x07\xf8\x43\x45\x2a\x09\xc9\xf7\x2f\x0d\xbc\xa9\x1d\x33\xb0\x21\xcf\x6a\xa6\xfe\x60\xd2\xed"[..]);

        ctx.update(b"\x9c\x69\x49\x43\x38\x9b\xdc\x4e\x05\xad\x7c\x2f\x63\xce\xac\x28\x20\xe1\xd2\xd7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf6\x92\xc0\x25\xc5\xc5\xf3\xd1\x27\x52\x13\xc1\xdf\x9b\xf9\xeb\x6d\x21\x88\xed\xa9\x0a\xb5\xbf\xfe\x63\x1f\x1d\xbf\x70\xeb\xd6\x28\xca\xee\x88\xb7\xd1\x49\xe1\xac\x4e\x26\x28\x73\x97\x9a\xfe"[..]);

        ctx.update(b"\x0f\xb1\x83\x57\xb0\x18\xb9\xbb\xb2\xcb\xb4\xca\xc5\x0b\xc8\x56\x09\xc9\x2b\x8e\x7f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd1\x64\x30\x6c\x99\xe3\x79\x87\x90\xf0\x92\x3f\xe9\x2d\xbf\x2f\x96\xc3\x90\x71\x27\xda\xca\xa4\x67\xc7\x66\xac\x75\x78\x80\x62\x58\x92\x72\xcb\x76\x90\xb8\xaf\x20\x30\xdd\x8b\xd6\x1a\x3d\xf2"[..]);

        ctx.update(b"\x26\xcb\x40\xa4\x60\xe2\xe7\x27\xae\xb8\x67\xe0\x14\x0d\x0f\x34\x79\x01\x10\xde\xb5\xd7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xaf\x2a\x42\xa4\xc6\x7c\x32\x26\xc5\x5b\x89\x60\x5b\x0d\xee\x27\xe7\x96\xc2\x79\x21\x15\xf6\x09\x72\x03\xdb\x5a\xed\x89\xe3\x5f\x56\x3a\x82\x46\xd3\x99\xfd\xe0\x0c\x2a\x5b\x97\xed\x5a\x5e\x17"[..]);

        ctx.update(b"\x66\x90\xa3\xa0\x37\x3c\x82\x9f\xac\xc5\x6f\x82\x43\x82\xf4\xfe\xed\x6e\xb1\x84\x64\x2b\x4f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x84\xe1\xb6\x8b\xc9\xe2\xda\xef\xc1\x9b\x56\x7d\xec\x91\x1e\xf4\x6f\x5f\x37\xa7\x4f\xdb\xbb\x61\x55\xe7\xe6\x46\xf2\x73\x5d\xf2\xac\x44\xe2\x39\x68\x9e\xb5\xb5\x36\x46\x5d\xc5\x71\xe5\x5c\xb2"[..]);

        ctx.update(b"\x7d\x80\xb1\x60\xc4\xb5\x36\xa3\xbe\xb7\x99\x80\x59\x93\x44\x04\x7c\x5f\x82\xa1\xdf\xc3\xee\xd4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x04\x1c\xc5\x86\x1b\xa3\x34\x56\x3c\x61\xd4\xef\x97\x10\xd4\x89\x6c\x31\x1c\x92\xed\xbe\x0d\x7c\xd5\x3e\x80\x3b\xf2\xf4\xeb\x60\x57\x23\x55\x70\x77\x0c\xe8\x7c\x55\x20\xd7\xec\x14\x19\x87\x22"[..]);

        ctx.update(b"\x02\x12\x82\x83\xff\xc0\xcf\xe2\x54\xac\x8f\x54\x2b\xe3\xf0\x5f\xbe\x4e\x85\x5d\xd2\x2a\xe9\x8a\x81");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x38\x40\x98\x1a\x76\x6d\x72\x5f\x83\xd3\x34\xe8\x98\x29\x65\x03\x3a\x5f\xbb\x51\x07\xd9\x4f\xfe\xf3\x3b\x1f\x70\x0c\xd4\x63\x48\x09\x1a\x49\xf6\x62\x0c\x37\xae\x3e\xf5\xb2\x05\x13\x49\x48\x26"[..]);

        ctx.update(b"\x27\x91\x1d\xd0\xa6\x84\x3c\xca\xe9\x65\xd8\x76\xaa\x19\x16\xf1\xdc\xd7\x1e\x51\x8f\x7f\x21\x97\x15\x2e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf5\x9f\x84\x28\x55\x59\x84\xd1\x52\x6c\xde\xd8\x12\x9c\x64\x9f\xb1\xb6\x83\xd3\x5c\xec\x7c\x5e\x12\x09\x44\x1a\x6a\x9e\x7c\x17\xf0\x78\x41\x51\xb5\xab\x8a\x8c\x49\x2b\x40\x2a\x3a\xcb\x98\xc4"[..]);

        ctx.update(b"\xd9\x37\x8b\xb6\x6e\x8c\x8d\xee\x55\x6d\x69\x1c\xbc\x9f\xdd\xdd\x63\x33\xca\x5d\x50\x66\x88\x62\xc3\xc5\x7d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x99\x45\x32\xd1\xa5\x57\xe9\x90\xb1\xcc\x9e\x03\x95\xa2\xad\x8b\x05\x61\x9c\xa3\x22\xdb\x9d\xa3\xc4\xed\x2e\xe1\x94\xc0\x51\xd0\x45\x82\xfd\xe7\x2d\xd2\xb8\xf6\x74\xcf\x6e\xc9\x58\xdb\x75\xda"[..]);

        ctx.update(b"\xae\x18\x28\x04\x7c\x5f\x82\xa7\xb9\x71\x2f\x33\x99\x83\x21\x24\xb8\x92\xf2\xf7\xae\xa5\x1c\x8f\xe3\x53\x6c\xd6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd5\x11\x11\xf8\xbf\xfb\x44\xd8\x1a\xd1\x96\x83\x19\x8f\x29\xd2\x03\x31\x44\xd3\xcd\x85\x6c\x74\x9c\xac\x5b\x9c\xae\x0e\x71\x2f\x50\x0f\x8d\x0e\xf8\x13\xf3\x8e\x30\x5c\xe1\x75\xa7\xd6\x16\x2c"[..]);

        ctx.update(b"\x7d\xd2\xd7\x6f\xa0\x54\xcf\x46\x1e\x13\x2e\x9e\xf9\x14\xac\xdc\x53\x08\x0a\x50\x8c\xdc\x53\x68\xab\x8c\x62\x24\xff");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6c\x0b\x33\x95\xe4\xc8\x65\x18\xab\x0a\x06\x26\x73\x20\xee\x9e\xc9\x5e\x50\x38\x5b\x7a\x25\x27\xdd\xaa\x1b\xd0\xea\xd2\x62\xc5\x61\x22\xd4\xf4\xeb\x08\xb0\xae\x22\xb3\xee\x7e\x6f\x44\xdd\x18"[..]);

        ctx.update(b"\x6f\xd7\x28\x88\xa0\x21\xf3\x6e\x55\x09\x67\xcb\x56\x05\xb5\x5b\x78\x65\x7c\x92\x72\xd9\x3c\x3d\xed\x34\x0d\x67\xda\x6f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x05\x51\x58\x3a\x5b\x40\x07\x40\x1c\x77\xef\x43\x82\xfd\x8e\x24\x5c\x9c\xf1\x2e\x97\x6c\x97\x66\xaf\x6b\x7a\xe3\xc7\xe0\x7a\x82\xb3\x07\x9f\x90\x3b\x08\x3d\x5e\xc8\x5c\xb9\x4e\x46\xa8\x5a\xc0"[..]);

        ctx.update(b"\xd5\x00\xeb\x95\x46\x55\x36\x19\xcd\xc3\x1e\x08\x48\xc5\x02\xdb\x92\xd5\x47\xef\xef\x3a\xe5\xee\xaa\x22\x25\x8a\xfc\xf0\xa9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5e\xdd\xe2\xf9\x4f\x86\x95\xf2\x77\xec\x05\xef\xcc\x00\x76\x1f\xaf\xd2\x72\x20\x0a\xed\x0e\x63\xd2\x21\xc2\xb6\xc6\x5b\x49\x72\xa6\x52\x6f\x9a\x1f\x2e\x6a\xce\x0e\x81\x93\x8f\x04\x3f\xe8\x77"[..]);

        ctx.update(b"\x61\x89\x59\x7e\x01\x98\xa1\x8c\x65\xfa\x0b\xdd\x07\x97\xf1\x30\x37\xc7\x5c\x40\x58\xb7\xd3\x45\x4c\x0f\x71\xbd\x2d\xd1\x3b\x6c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x11\x06\x30\xca\x76\x31\xb7\x62\x0e\x6b\xee\x6e\xd6\xe9\x29\x09\x89\x65\x57\x19\x36\xc3\x48\x29\x48\x49\x83\xeb\xa9\x53\x2b\x81\x75\x52\x8c\x22\x8c\x57\x43\x94\x53\xf0\x27\xa4\xf7\xc8\x3c\xa3"[..]);

        ctx.update(b"\x24\x3b\x94\x1d\x74\x85\x41\xaf\x30\x3f\x8e\x9d\x2c\x37\x1c\xd0\x3e\x43\x7d\x62\xa9\xdf\x48\x5d\xdc\x17\x6d\xc6\x5d\xa8\xc7\xda\x00");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x58\x84\x20\x1f\x7a\x55\x5e\xa3\xc5\xde\xeb\x01\x9f\xd9\xe8\xc1\x61\xe1\xb8\x97\x56\x04\x5e\x47\x5b\x14\x1e\xc5\x13\x5c\xe5\xa4\x1c\x93\xe5\xe1\xf7\x95\x34\xd3\x6f\xd8\x34\x5b\xa4\x34\xda\x43"[..]);

        ctx.update(b"\x2d\xc3\xd7\x89\x58\x2c\x1a\x80\x6c\x3b\x49\x1d\x59\x72\xef\x8f\x17\x33\xf1\xf5\xe0\x28\x66\xdc\x9d\xe2\xa8\x02\x9e\xc0\xab\x60\x8d\x13");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x05\xa3\x90\x3b\x51\x9c\xdf\x67\x91\x20\xc7\xcc\xb4\xef\x17\x8b\x58\xe4\x50\x2f\xcd\x46\x13\x60\x98\x8f\xa0\x66\x69\x29\x48\x51\xe6\x29\xd9\xdd\x3e\x77\xff\xb7\x3d\x24\x59\x9d\x5d\x3e\xdd\x36"[..]);

        ctx.update(b"\xe5\xb3\xf6\x96\x2f\xe5\x72\x30\x78\x0b\x3d\x55\xb2\x9e\xff\xe0\xdf\xeb\xde\x2c\x81\xba\x97\xd4\x51\x2e\xcd\xbd\x33\xec\xa1\x57\x6a\x7f\x82");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7a\xc2\x77\x6a\xfb\x74\xf5\x5b\xbc\x4f\x6e\xcc\xf8\x25\xee\x13\xac\x74\x45\xfb\x54\x97\x4e\x6c\x24\xeb\xc0\xf0\x3f\xdc\xd8\x53\x01\x99\xa6\x11\x06\xa3\x1b\x42\x79\xe0\x22\x01\xee\x0f\x54\xfd"[..]);

        ctx.update(b"\xda\x03\x48\x6a\xa3\xce\xbb\xd6\x50\x2e\x9f\x5a\x6f\x0f\x83\x5e\x97\x3a\x58\x1b\xef\xcc\x1a\xad\xef\xe7\xb3\x69\x6b\xa7\x1c\x70\xcd\x58\xc5\x84");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x02\xc4\x4c\xee\xc0\xbb\x7d\xc0\xf6\x64\xeb\xe4\x42\x30\x19\x2b\x5b\x0b\xb6\x46\xbb\x94\x4d\x23\xfa\x1f\xf3\x58\x6d\xc0\x52\x3f\xa9\xd7\xf0\xdd\x6d\xf5\x44\x9a\xb9\xed\xd9\xa1\x09\x6b\x07\xdc"[..]);

        ctx.update(b"\x3c\x68\x6d\x32\x1b\xa6\x61\x85\xcd\xca\x83\xba\x9f\x41\x98\x4f\xa6\x1b\x82\x6e\xf5\x6b\x13\x6e\x13\xf1\x23\x9d\xad\xf6\xe0\x3d\x87\x78\x66\xcc\xb8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xad\x62\x4e\xdd\x9f\x2c\x3a\x32\xb5\x6c\x53\xd9\xe8\x13\xc0\x1d\x66\xbc\xfe\x42\x4c\x4a\x96\x90\x7d\x52\xac\x1d\xdd\x68\x37\x0e\xc8\x6d\xac\x67\x50\x4a\x90\xe8\xa8\xe7\x55\x02\xe0\x10\x81\xd2"[..]);

        ctx.update(b"\x4d\xcf\xf9\x9f\xac\x33\x84\x0f\x65\x32\x54\x7f\xb6\x9b\x45\x69\x02\xd6\x71\x8f\xd5\xd4\x53\x8e\x23\x46\x2d\xb6\xd0\x0d\xa6\x19\x75\xf2\xb8\xe2\x62\x98");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xcf\x37\xdd\x27\x99\x7c\x1b\xb7\xe6\xdc\x40\x51\x70\x06\x6e\x74\xc6\xce\x51\x7c\x02\x9e\xd8\xdc\xe1\x26\xd0\x25\xda\x74\xe0\xb8\xe8\x6d\xa5\x67\xe8\xd7\xd8\xd5\xb5\xd3\xe2\xa5\x46\xdf\x74\x89"[..]);

        ctx.update(b"\x27\x99\xf6\x72\x32\x88\x34\xd7\xea\xef\x94\x39\x79\x5d\x35\xce\x93\xc9\x09\x4f\x58\xde\xd9\xf1\x7c\x96\x8a\x97\xa5\x0a\x9e\x46\x14\x89\xfe\xd9\x88\xe7\xf6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x85\xcf\xc2\x3c\x97\xcb\x13\x91\x0b\x80\x8e\x70\x33\x80\x9a\x45\xaa\x0b\x7f\x71\x38\xde\x61\x8c\x2c\xa6\x22\xc8\xb8\x13\xc9\x88\xe2\x64\xaf\x3b\x96\xc7\x92\x5d\xcb\xd1\xd2\x76\x17\x57\xd8\x00"[..]);

        ctx.update(b"\xc7\xe9\x47\x50\x78\x22\xf2\x8a\x56\x27\x45\xa8\xfe\x6f\xed\x6c\xb4\x7d\x73\x14\x58\x04\xc8\x94\x95\x4e\x21\x24\x5c\xde\x04\xfa\x91\x55\xa3\x59\x04\x92\x6a\xca");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8b\xdd\xf3\xba\xeb\xbc\x5b\x04\xfe\x0b\x0a\x9c\x3c\x2b\x73\x0a\xbe\x91\x8c\xe4\x89\x2d\x28\x43\xc6\x13\xee\x96\xda\x02\x28\x51\x2f\x0d\x13\x07\xc7\xd1\xa8\x92\x2e\x79\xa9\x2e\x95\x7d\xd1\x8e"[..]);

        ctx.update(b"\x6c\x49\x7b\xf6\xff\x69\xcb\x39\xe3\xfa\xa3\x49\x21\x2b\x8b\x66\x91\xca\x23\x79\x05\xac\x00\x99\xc4\x50\xb6\xd3\x3a\xbf\x36\x2b\xed\xb6\x5b\xde\xb3\x07\xbf\xea\x23");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x36\x39\xfa\xb6\x19\x1b\x35\x24\x62\x78\x52\x2c\xfa\xce\xe0\xcd\x5b\x15\x58\x0a\x26\xc5\x05\xae\x3c\x46\xb4\xb1\xc2\x57\x20\x16\xb4\x8f\x1b\x01\x2b\xbb\xed\xec\x47\x91\x69\x50\xfb\xb3\x3a\x1d"[..]);

        ctx.update(b"\xd1\x59\x36\xf3\xb0\xc9\x01\x82\x71\x81\x2b\x4c\x81\x45\x3c\x44\x57\xc7\xed\xd1\x10\xbc\xea\x7f\x57\x35\xd6\xf5\x88\x2d\x8f\x27\x15\x5e\xb4\xcc\x28\x5a\x65\x13\x8a\xd6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x02\x93\xee\xef\x0a\xa3\x39\x2c\x93\xd9\xc6\xca\x89\xc0\x8b\x31\x76\x22\x57\x2d\x4d\xe2\x28\x6a\x4b\x9a\xe6\xc2\xf9\xc9\xe0\xe6\x4e\xe6\xc4\x83\xd4\xf1\x08\x59\x07\x7e\x3c\x68\x68\x43\x02\x14"[..]);

        ctx.update(b"\xdf\x18\x13\x9f\x34\xb8\x90\x4e\xf0\x68\x1c\x1b\x7a\x3c\x86\x65\x3e\x44\xb2\x53\x5d\x6c\xec\xd1\xa2\xa1\x7c\xd5\xb9\x35\x7b\xe7\x9b\x85\xe5\xe0\x4d\xd9\xef\xf2\xca\x8b\x9a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdb\x9e\x17\x1d\x6e\x33\x36\x63\x1c\x9c\xee\xc6\xb4\xd7\x32\xce\x62\xb0\x15\x93\x92\x69\xfb\x69\xfa\xe7\xd2\x27\x25\x50\x0e\x8a\x2f\xc9\xf1\x45\x9c\xf0\xa3\x1f\xb9\xd1\x6d\x7c\x44\x58\x3f\x52"[..]);

        ctx.update(b"\x04\x59\xdc\xbc\x14\x93\x33\xea\x2f\x93\x7b\x77\x9a\x5f\x37\x28\x14\x84\x49\xa9\xae\xa3\x66\x2c\xdd\x2c\xc6\x53\xce\x6a\x20\x50\xf9\xc0\xd5\x4b\xf9\x32\x6c\x03\x9b\x26\x3e\xb9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x46\x4b\xa4\x09\xfb\xb4\x5e\x98\x5f\x84\xee\x24\x66\x2e\xb7\xc0\x42\xc3\xc2\xad\x96\x49\xf1\xac\x4a\x8b\x2b\xe9\xc0\x7d\x37\xed\x2e\x42\x84\x36\x20\x57\x49\x3f\x6a\x7e\x52\xc3\x56\xb0\x5b\xc5"[..]);

        ctx.update(b"\xeb\x3f\x70\x02\xc8\x35\x22\x70\x34\x0b\x8d\xa8\x64\x36\x22\xe5\xf7\xe3\x2c\xdb\x20\x8a\x0d\xec\x06\xc6\xcb\x9e\x6b\x64\xcc\x4d\x8c\xb9\xde\x1d\x49\x39\x7b\x33\x86\x46\x4a\x25\xd1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa2\x6b\xd7\x6c\xe4\x2d\x81\x8d\xbe\xc4\x62\xd8\xfe\x7c\xdd\x95\x7e\x6b\x84\xae\x87\x50\xfb\x5e\x1c\x9c\x76\xbc\x60\x00\xe2\x37\x37\xe0\x73\xa5\x9b\x46\x00\xe5\x05\x65\x24\xed\xc6\x67\x90\x9d"[..]);

        ctx.update(b"\x47\xe3\xe3\xd8\xc6\x8a\xc9\xd9\xf4\xb3\x75\x9d\x8c\x7d\x9d\xd9\x01\xe3\x5b\x09\x6e\xe4\xc8\xb6\xcb\xe0\xcd\xf4\x67\x46\x36\x30\x92\x6c\x08\x28\x9a\xbe\x15\x3b\xfa\x1b\xcd\xe3\xcd\x7c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb5\x04\xef\x47\x5a\x56\x8f\x9c\xab\xa8\x35\x2a\x0b\x2d\x24\x3a\xcd\xf3\xd2\xb4\x1d\x88\x90\xa6\xfb\x3a\xbb\x8a\xa2\x8a\x29\xe0\xc7\x52\x7d\x20\xe2\xd7\x9b\x25\xb4\x00\xec\x27\xc3\x14\xdb\x72"[..]);

        ctx.update(b"\x83\x8d\x9c\x18\x1c\x5a\xb5\x95\x92\x72\x3b\xd6\x93\x60\xe0\xd7\xfd\x15\x23\x2b\xea\xda\x75\x91\xea\x89\x9a\xc7\x8f\xfd\x53\xa3\x2f\xc7\x3a\x5f\xe5\x22\xed\x35\xd9\x2a\x6e\x2b\xc1\x48\xca");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x53\xe9\x9e\x11\x58\xd5\x90\x32\xff\xe4\xb5\xea\x30\x4c\x7d\x2f\x7a\x61\xb6\xb2\xa9\x6a\xc9\x78\x32\xca\x26\x01\x35\x49\xfe\x3f\x7d\xcd\xf9\x26\xbd\x74\xce\xab\xe4\xf1\xff\x17\x2d\xae\xd6\xe6"[..]);

        ctx.update(b"\xa9\x0d\x2a\xa5\xb2\x41\xe1\xca\x9d\xab\x5b\x6d\xc0\x5c\x3e\x2c\x93\xfc\x5a\x22\x10\xa6\x31\x5d\x60\xf9\xb7\x91\xb3\x6b\x56\x0d\x70\xe1\x35\xef\x8e\x7d\xba\x94\x41\xb7\x4e\x53\xda\xb0\x60\x6b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x4a\x16\x88\x1c\xe1\x56\xf4\x5f\xdf\xdb\x45\x08\x8e\x3f\x23\xbe\x1b\x4c\x5a\x7a\x6a\x35\x31\x5d\x36\xc5\x1c\x75\xf2\x75\x73\x33\x19\xac\xa1\x85\xd4\xab\x33\x13\x0f\xfe\x45\xf7\x51\xf1\xbb\xc5"[..]);

        ctx.update(b"\x8c\x29\x34\x5d\x3a\x09\x1a\x5d\x5d\x71\xab\x8f\x5a\x06\x8a\x57\x11\xf7\xba\x00\xb1\x83\x0d\x5e\xd0\xbc\xdf\xb1\xbb\x8b\x03\xcd\x0a\xf5\xfe\x78\x78\x9c\x73\x14\xf2\x89\xdf\x7e\xee\x28\x87\x35\xfe");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe2\x7b\x39\xa9\x62\x55\xff\x69\xc4\x52\x85\xfc\xa6\xed\xaa\xa3\x95\x4c\xe3\x2c\x1e\x3d\x9b\x1f\x60\xc1\xb6\x67\x65\x94\xbb\x45\xca\xf0\x88\x9f\xc1\x1d\xaf\x93\xa1\xb6\x07\x46\x22\x96\x89\xdd"[..]);

        ctx.update(b"\x32\x87\x6f\xee\xfe\x99\x15\xa3\x23\x99\x08\x34\x72\xe3\xc3\x80\x5e\xf2\x61\x80\x0b\x25\x58\x2a\xa7\xc3\x63\x95\xfd\x3e\xc0\x5d\x47\xb4\x9c\x49\x44\xbb\xcc\x2b\x8b\x5e\xbd\x08\x1f\x63\xae\x79\x43\xd0");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xf9\x64\x33\xcd\xb6\x9a\x60\x74\x33\xea\x2e\xb7\x7d\x87\xd3\x32\x88\x67\xdc\x40\x76\xb6\x7c\xcf\x17\xf5\x0f\x9e\x08\xe8\x9a\x86\x62\x4b\x60\xf2\xec\xdb\x8a\xff\xcd\x43\x1f\xc1\x31\x73\xfe\x75"[..]);

        ctx.update(b"\xe2\xe7\x7e\xb5\x4f\x32\x1f\x86\xf5\x2e\xa3\xd3\xc8\xcd\xc3\xbc\x74\xd8\xb4\xf2\xf3\x34\x59\x1e\x5e\x63\xb7\x81\x03\x4d\xa9\xd7\xb9\x41\xd5\x82\x70\x37\xde\xe4\x0c\x58\xdc\x0d\x74\xc0\x09\x96\xe5\x82\xbc");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa3\x52\xab\x33\xca\x73\x04\x82\xc3\x76\xbd\xc5\x73\xc9\xd1\xdc\x6d\x35\x97\xf9\xbe\x9f\x79\x8b\x74\xa5\x7b\xea\xa8\xe9\xc5\x7b\x78\xee\x67\x61\x05\x6e\xb6\x73\x63\xe8\x82\xfe\xfc\xad\x4f\xb9"[..]);

        ctx.update(b"\xda\x14\xb6\xd0\xb2\xec\x4c\xf1\xe7\xc7\x90\xe7\xf8\xf4\x21\x2b\x8f\x4d\x05\xf5\x0e\x75\xe2\xa5\x6a\x5d\x70\x62\x3c\x0d\x2e\x01\x15\xa1\x54\x28\x12\x91\x09\xb3\xb1\x36\xd7\x56\xe3\x8a\x5c\x84\x63\x30\x42\x90");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xaa\xe7\xad\x97\x7e\x17\xac\x0e\x56\x0c\x0e\x01\x86\x43\x34\x20\xf9\xfd\xdc\xd1\x91\xb9\xe9\x15\x67\xce\xe0\x5d\xf8\x8f\x1e\x1a\xee\x50\x42\x4a\x31\x39\x98\xa8\x73\xf7\xa9\xc2\x89\xa0\x22\x17"[..]);

        ctx.update(b"\x2d\xb0\x6f\x09\xab\xaa\x6a\x9e\x94\x2d\x62\x74\x1e\xac\xd0\xaa\x3b\x60\xd8\x68\xbd\xdf\x87\x17\xbe\xf0\x59\xd2\x3f\x9e\xfe\x17\x0f\x8b\x5d\xc3\xef\x87\xda\x3d\xf3\x61\xd4\xf1\x2b\xfd\x72\x00\x83\xa7\xa0\x35\xe8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x85\xd4\xe3\xe5\xab\xcb\x1b\x59\xca\x6f\x55\x1e\xb4\x3b\x43\xff\x64\x89\x05\x11\xf7\x3a\x90\x83\xa2\xce\x6e\x9c\x28\x61\xc6\xe9\x66\x4c\x76\x56\x29\x02\x4f\x4b\x01\xb0\xcd\x15\x94\xa5\x98\x1b"[..]);

        ctx.update(b"\x26\xba\xd2\x3e\x51\xc4\x56\x0c\x17\x20\x76\x53\x8b\x28\x71\x67\x82\xee\x63\x04\x96\x2f\x68\xe2\x71\x82\x04\x89\x48\xd5\xc3\x67\xa5\x1a\x1c\x20\x6a\x3e\x9b\x25\x13\x5b\x40\x88\x3b\x2e\x22\x0f\x61\xcb\x57\x87\xed\x8f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa4\x4c\x7f\x84\xab\x96\x2f\x68\x28\x34\x04\xf8\xc5\xc4\x02\x9d\xbc\x35\xd2\x13\x8e\x07\x5c\x93\x27\x58\x0b\xaf\x89\xf2\x92\x93\x7b\xf9\x94\x22\xe4\x57\x56\xb3\xf9\x42\xbf\x0a\x5a\xe4\xac\xb6"[..]);

        ctx.update(b"\x77\xa9\xf6\x52\xa0\x03\xa8\x3d\x22\xfb\x84\x9b\x73\xfe\xd7\xd3\x78\x30\xc0\xdc\x53\xf8\x9c\xea\x7d\xbe\xc2\x4e\x14\xf3\x71\x97\x76\x52\x06\xfe\x0e\x66\x72\x01\x6e\x4d\xec\x4d\x9e\xbb\xe3\xe1\xb4\x42\x37\x71\xa5\xd0\xa8");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x29\xc8\xbb\x39\xbb\x2a\xad\x41\x9a\x00\xa8\x02\x16\xec\x71\xec\x5e\xc9\xab\x54\xc4\x19\x27\xe3\xe3\xf2\xf4\x8f\x07\x9a\x58\x86\xd7\xfe\x89\xdb\x98\xc8\x07\xab\x68\x6d\x23\x39\x00\x1d\x62\x52"[..]);

        ctx.update(b"\x26\x8c\x7b\x3a\x84\x84\x9f\xec\x5c\x76\x9b\xc4\xad\x37\x7d\xea\x10\xc9\xd2\x0c\x91\xdd\x17\xfd\xbd\x96\x70\xa2\xfc\x90\x9d\x0e\x21\x21\x29\xec\x40\xde\xe4\x1d\xbf\x61\x94\xa3\xb0\x4a\xe8\xbe\x5e\x84\xad\x54\x26\xca\x44\x96");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x0d\xfc\x6f\xfc\xf4\xa3\x87\xec\x09\xff\x86\x2c\x61\x39\xa6\xf7\xac\x77\xab\xb2\xb5\xe1\xf6\xdc\x81\x4e\xb7\x15\x25\xf8\x65\x7a\xc7\x4a\x76\x97\xc2\x97\x5c\x70\xa5\x43\xaf\x0e\x22\x7d\x03\xca"[..]);

        ctx.update(b"\xb8\x32\x43\x41\xa6\x89\x1a\x6b\x5e\x00\x1a\x7d\x2e\xbb\xa6\xe0\x2e\x83\x35\xc1\x24\x18\x53\x09\xa4\xc9\xe9\x90\x7c\x43\xbd\x8d\x4f\xa7\x3c\x52\x7f\xdf\x78\x36\x50\x31\x6d\xd2\x4b\x14\x88\x70\xe1\x43\x6a\xc0\x51\x11\xe9\xcd\xcc");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x62\x78\xd1\xcc\x17\xfb\x6d\x54\x12\x9d\x04\x98\x7d\x47\x74\xfa\x84\x6d\xca\xc4\xba\x8b\x6b\x72\xf4\x1e\x63\xdc\x38\x7c\xe0\x08\x1b\xa2\x9f\xb2\xc1\x7c\x67\x44\xed\xae\x24\xe6\x69\xcc\x9e\x75"[..]);

        ctx.update(b"\x5e\xf8\xb3\xd7\x9d\x29\x9b\xee\x2c\x41\x45\x60\xc7\xde\x62\x6c\xc0\xd9\xfb\x42\x98\x84\xaa\x69\xcc\x30\x09\x5e\xf1\xf3\x6b\x7e\x03\xa8\xca\x25\xfb\x36\x01\x18\x9f\x16\x3b\x20\x9e\x0f\xac\xf8\xdc\x44\x7f\x69\x0b\x71\x0f\xb4\x7b\x72");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x7e\xc9\x50\x5f\x33\xf4\xa5\x49\x35\x74\x42\x2d\xe0\x78\xe0\x49\x0b\x61\xbe\x8e\x8d\x6f\x15\x81\x92\xbb\x7d\x2b\xdc\x2d\xc3\x35\x59\x8d\xc8\x8d\x9b\x44\x3c\xd1\xc1\x4b\x88\x3a\x77\x11\x9d\xf1"[..]);

        ctx.update(b"\xad\x73\x21\xc9\xa8\xb8\xf0\xbf\xe1\x00\x81\x11\x14\x27\x0d\xaa\xd5\x7f\x6e\x88\x77\x23\x26\xb6\x2d\x88\xa3\x7a\x6f\x55\xc2\xcf\x9f\x75\x91\x15\xed\x6a\x59\x08\x78\xe4\xdc\xef\xb5\x92\xdb\x15\x15\x38\xdb\x7d\xe2\x02\x29\xd2\x6a\x18\x1c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x37\x82\xd2\xca\xa5\x37\x29\x4e\x80\x9e\x9d\xf8\x37\xb1\xb0\x7e\x2f\x1d\xf0\x7d\x0f\x4c\x12\xe1\x24\x59\xf5\x6e\xea\xa4\x78\xd5\xb3\xa4\x1e\x51\x9d\x94\x14\xea\xfa\x5d\xdd\x56\x61\xc8\x31\xba"[..]);

        ctx.update(b"\x07\x19\xd9\x66\x45\x41\xf0\xa8\x24\xf7\x1c\x83\xb8\x09\xbb\x6a\xfc\x97\x3c\x9f\x74\x28\xe1\xed\x11\xf7\xc2\x9a\x55\x8e\x16\x98\xb7\x96\xae\xfb\x49\xee\xc2\xb0\x98\xfa\xf0\x6b\xd4\x3e\x82\xe1\x31\x2b\xf0\x38\x8c\x38\xa5\xbb\x52\x35\x06\xd3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x36\x2c\x05\xf6\x78\xdf\x92\x88\x3d\x56\xe1\x92\x21\x39\x1f\xb0\x0d\x0f\x0a\xfc\xec\x51\xd3\xe0\xfe\xb1\x5b\xa2\xfb\x60\x69\x3b\x09\xd6\x91\x18\xaf\x64\x96\x48\x93\x32\x59\xd7\xb1\xe2\x40\xab"[..]);

        ctx.update(b"\x54\x15\xc2\x59\x6a\xa7\xd2\x1e\x85\x5b\xe9\x84\x91\xbd\x70\x23\x57\xc1\x9f\x21\xf4\x62\x94\xf9\x8a\x8a\xa3\x7b\x35\x32\xee\x15\x41\xca\x35\x50\x9a\xdb\xef\x9d\x83\xeb\x99\x52\x8b\xa1\x4e\xf0\xbd\x29\x98\xa7\x18\xda\x86\x1c\x3f\x16\xfe\x69\x71");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x8f\x9f\xd7\xd8\x79\xd6\xb5\x1e\xe8\x43\xe1\xfb\xcd\x40\xbb\x67\x44\x9a\xe7\x44\xdb\x9f\x67\x3e\x34\x52\xf0\x28\xcb\x01\x89\xd9\xcb\x0f\xef\x7b\xdb\x5c\x76\x0d\x63\xfe\xa0\xe3\xba\x3d\xd8\xd1"[..]);

        ctx.update(b"\xb9\x79\xa2\x5a\x42\x4b\x1e\x4c\x7e\xa7\x1b\x66\x45\x54\x52\x48\x49\x8a\x2b\x8c\x4b\x56\x8e\x4c\x8f\x3f\xf6\xe5\x8d\x2a\xc8\xfb\xe9\x7b\xe4\xbe\xa5\x7d\x79\x6b\x96\x04\x1d\x15\x14\x51\x1d\xa5\xf6\x35\x11\x20\xbe\x7a\xb4\x28\x10\x7e\xf3\xc6\x69\x21");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xe2\x48\xa6\x4b\x6e\xf1\x12\xbf\x3d\x29\x94\x8b\x1c\x99\x58\x08\xe5\x06\xc0\x49\xf3\x90\x6d\x74\xc3\xee\x1e\x4d\x9f\x35\x16\x58\x68\x19\x01\xfe\x42\xc8\xe2\x80\x24\xfe\x31\x01\x4e\x2d\x34\x2b"[..]);

        ctx.update(b"\xe6\x4c\x7b\xb9\xcd\x99\xce\x54\x7d\x43\xde\x3c\xc3\xb6\xf7\xd8\x7a\x2d\xf9\xd8\xa4\x76\x0c\x18\xba\xf5\x90\xc7\x40\xec\x53\xc8\x9b\xfa\x07\x58\x27\xe1\xf3\xf2\x85\x8c\xe8\x6f\x32\x50\x77\x72\x5e\x72\x61\x03\xfb\xe9\x4f\x7a\x14\x66\xc3\x9f\x60\x92\x4f");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd1\xe5\xa7\x2d\x25\x95\xf3\x87\x14\xc6\x19\x8a\xc1\x4f\x8a\x5c\xdd\x89\x4d\xcf\x9b\x4b\x8e\x97\x51\x74\xb1\x00\xdf\x7b\xbf\x4f\x7c\xe2\x91\xb4\x86\x4f\x27\xc0\xb6\x4e\x63\x30\xf6\xc1\xc8\x2c"[..]);

        ctx.update(b"\x91\xb7\xa1\xfd\x0e\x20\x07\x2d\x9c\x5b\xe7\x19\x6e\x5e\xaf\x8d\xf3\x6f\xdf\x14\x58\x95\xb3\x0d\x4e\x4c\x02\x01\x0d\x7c\x66\x34\x99\xac\x9d\x7a\x44\x73\x2f\x4c\x74\x30\x51\x1b\xa6\xfb\x0a\xe4\xb3\xdc\x94\x05\x52\x3a\x05\x4f\xdf\x96\x2f\x5c\x5b\x79\xc4\x23");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x07\xc2\xe0\xae\xae\x30\xda\x83\xb5\xa6\xb3\x20\xaa\x1c\xf7\x27\xb1\x0c\x20\x34\x58\x3d\x7a\xcd\xa5\x56\x48\xfa\x3d\xaa\x01\x7a\xa1\x55\x88\xb6\xe2\x14\x91\x01\xc5\x6e\x3d\x7d\xf7\xc7\x6d\xf1"[..]);

        ctx.update(b"\x5b\xbc\x2d\x4e\xfe\x63\xcb\xfc\x9f\xc2\x21\xdd\x8d\x83\x84\x07\x5a\x79\xc8\x0a\x27\xd6\xa8\xc5\x21\x9e\x67\x7f\x4c\x5b\xb8\x33\x80\x13\xdc\x2a\xb1\x77\x0a\xcf\x73\x5d\x13\xc0\xbc\x70\x46\x21\xec\x26\x91\x35\x0c\xf3\xea\x2f\x53\xbd\xed\x45\xef\x8f\xc7\x07\x02");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xdd\x0b\xbf\xe4\xb7\x99\x64\x21\x91\xab\xe3\x16\xdf\x9d\x59\xa3\x74\x35\x66\x77\x8b\x44\x59\xc5\x1c\x3b\xe3\xf6\x58\xbd\xce\x45\x51\x6a\xd1\x88\xfb\xe1\xa8\xca\xd8\xa1\xfa\x78\xf8\xeb\xb6\x45"[..]);

        ctx.update(b"\x12\x95\x49\x27\x8e\x89\x76\xc3\x8b\x55\x05\x81\x57\x25\x40\x0c\x3d\x20\x81\xed\xf1\x41\xad\x00\x2e\x62\xff\x29\x9d\x9a\x07\x43\xf9\xc9\xf2\x59\x71\x71\x0b\x19\x4d\xc8\x82\x85\xd5\x0b\x6c\xec\x6e\x14\x0c\x19\x07\x2f\x51\xca\xb3\x2a\x9f\x64\x97\xab\xd3\xe4\x07\xc6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xca\x26\xae\xc5\x27\xfa\xdc\xd5\xeb\xeb\x4e\xaf\xa7\xc1\x02\xf7\x9a\x3c\x2e\xdb\x45\x2a\xfd\x04\xf6\x16\x2d\xd7\xa1\x7b\xdd\x1a\xad\x7d\x61\x65\x08\xa8\x9a\x3e\xc6\xa4\x07\x91\xd9\x15\xac\xc8"[..]);

        ctx.update(b"\xb9\xa9\xf3\x78\xad\xef\xf4\x33\x7b\xc7\xec\x10\xd5\x26\xc6\xdd\xa0\x70\x28\x37\x55\x49\xf7\xfd\xa7\xa8\x1d\x05\x66\x2c\x8a\x0d\xa3\xb4\x78\xf4\x15\x2a\xf4\x2a\xbb\x9f\x9a\x65\xc3\x9d\xa0\x95\xab\xb8\x16\x1b\xa6\x67\x6b\x35\x41\x12\x34\xbd\x46\x6c\x29\x14\xe0\x03\x70");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x99\x91\x4f\x68\x4e\x0b\x31\x7f\x93\x38\xaf\x0c\x71\xe9\x65\x5a\x3a\xf7\x15\x3e\xb9\xfa\xba\xae\x61\x45\x4b\xf8\xde\x9e\x0b\xfd\x27\x4c\x1e\xff\x6c\x4b\x55\x0e\x47\xaf\xcb\x3b\x20\xfa\x7d\x9e"[..]);

        ctx.update(b"\x10\x1d\xa5\xb0\x97\x00\xdc\xad\xf8\x0e\x5b\x79\x00\xf4\xe9\x4c\x54\xd5\xf1\x75\x56\x9a\x85\x4e\x48\x8a\xa3\x6f\xb4\x1a\xb7\x22\x0b\x06\x62\x17\x8c\xa0\x7a\x59\x67\x68\x52\x81\x23\xde\x3b\x2a\x3d\x94\x4a\xa4\x12\x87\x5c\xed\xfe\xaf\x58\xdc\xc6\xd5\xb4\xa0\x33\xa5\x3b\x69");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xd3\xe3\x2c\x9b\x27\x1e\x11\xe4\x96\x83\x97\xd8\x5d\x76\x93\x8b\x97\x4a\xc1\xba\x55\xbc\xbe\x8d\x7b\x7d\xa0\x2d\xbd\x7e\x3b\x9c\x9a\xf0\xd9\x8b\xbd\x7e\x50\xc4\x36\xfc\xf9\xe3\x55\x1e\x34\x32"[..]);

        ctx.update(b"\x14\x76\x1b\xbc\x56\x85\xb5\xde\x69\x29\x73\xe2\xdf\x7c\x9c\x47\x50\x88\x9c\x19\xa9\x52\xf9\x12\xc8\x17\x89\x05\x46\xd5\xe3\x7d\x94\x0d\x13\xa1\x4a\xc7\x92\x5a\xbb\xd8\x75\xb8\xcd\x60\xe4\x92\x08\x96\xce\x6d\xec\xc8\xdb\x9f\x88\x9d\xa2\xb5\x48\x9e\x1d\x11\x0f\xf4\x59\xd8\x85");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x27\x22\x22\xed\x50\x63\x1a\xff\x46\x5c\x0e\x6f\xe4\x9e\xcd\xfd\xca\x98\x3b\xcb\x72\x31\xe5\x09\x03\xe2\x00\xb3\x35\xb8\x45\x10\x82\x02\xc2\x83\x15\x91\x2c\x9c\x4f\xd5\x0e\x2c\x6f\x13\xa9\xea"[..]);

        ctx.update(b"\xed\x53\x80\x09\xae\xae\xd3\x28\x4c\x29\xa6\x25\x37\x02\x90\x49\x67\xe0\xea\x97\x9f\x0a\x34\xa5\xf3\xd7\xb5\xab\x88\x66\x62\xda\x9b\x8e\x01\xef\xc4\x18\x8e\x07\x7c\x2c\xde\xb5\xde\x0a\x82\x52\xaa\xfb\xee\x94\x8f\x86\xdb\x62\xaa\xe6\xe9\xe7\x4a\xbc\x89\xe6\xf6\x02\x1a\x4d\xb1\x40");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x83\x61\xb6\x80\x24\x3b\x16\x61\xd6\xf1\xdf\x53\xdb\x36\x3c\xae\x41\xc2\xeb\xb7\x43\x8c\x00\x60\x6d\x76\xb9\xc2\xa2\x53\xfa\xa1\xf0\x9d\x6f\x52\x0d\x69\xd6\x92\xec\x1d\xca\x0c\x78\x85\x11\x9c"[..]);

        ctx.update(b"\xc4\x34\xd8\x84\x68\xf1\xed\xa2\x38\x48\xd0\x80\x4b\x47\x69\x33\xf2\x4b\xae\xad\xec\x69\x74\x3d\xd9\x0d\x84\x55\xf1\xe1\xf2\x90\xf6\xf1\xaa\xf3\x67\x0c\x4c\x74\xf7\x6d\x3a\xb8\x3e\x9b\xef\x21\xad\x8d\x92\x08\xc7\x12\xca\x47\x8e\x70\xd5\xfb\x3c\x4b\xd4\x88\x34\xc9\x69\xdd\x38\xf4\x84");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x9c\x26\xe9\x6f\xcc\x09\xa7\x6c\xc1\x3d\x24\xad\x25\xc9\xce\xf4\x30\x0e\x96\xe9\x7e\x4f\xb5\x9b\x44\x1b\xaf\xfe\xd0\x7f\x6a\x70\xb1\x46\x4f\x25\x48\xc7\xfd\x78\x39\x81\x0d\xbb\x9e\x9c\x1e\x18"[..]);

        ctx.update(b"\x30\x64\xe5\xba\x1e\x77\x51\xbf\x71\x98\xe0\x81\x1f\xf4\xd4\xca\x17\xd1\x31\x1c\x25\xd9\xc3\xa3\x16\xb5\x62\x69\x1c\xde\x75\xc9\x74\xb0\xb5\x26\x45\xc1\x34\xdd\xcc\x70\x9d\x77\xb6\xc1\xbd\x24\xcd\x68\x42\x65\xd7\x23\xc3\x08\xbb\x4d\x01\x59\xe6\xb1\x6d\x97\xed\x9c\xea\xa5\x74\x36\xd3\x02");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x1e\xa7\x79\x73\x9b\x20\x4a\xbe\x91\x1b\x49\x23\xe6\xf6\x0f\xec\xe2\x71\xee\xdf\xc7\xf0\x74\xfe\x19\x19\xf0\xcb\xc6\xce\x2a\x99\x23\x4b\x00\x33\x89\x52\x08\x84\xb6\x60\x16\x5f\x5a\x1e\x80\xf8"[..]);

        ctx.update(b"\x89\xd9\x52\x1a\xd8\x4b\x1c\x9a\xfc\x2f\xbd\x0e\xdc\x22\x71\x93\xac\xd3\x33\x07\x64\xb0\xd2\xcb\x71\xbf\x47\xc7\xaa\xc9\x46\xaf\x85\xbe\x13\x85\x8b\x55\x97\x60\x09\xf3\xb3\x6b\x09\xce\xd4\x30\x80\x52\xc8\x17\xc9\xc4\xd0\x29\x52\x25\xf6\x1a\x96\x59\xa0\x87\x4b\x88\x66\x7c\xdc\xc5\x21\x39\x19");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x42\x09\xbb\x8f\x86\x9f\x6f\x17\xc8\xd5\xc3\x68\xc4\x89\xac\x51\xa7\x5e\x24\xa8\x5a\x12\xde\x1b\x16\xfe\xfc\x29\x2c\xe6\x36\xff\x8f\xa3\x60\xe8\x2f\x05\x68\x4f\x6b\x0b\x07\x4b\xa3\x70\xa9\x33"[..]);

        ctx.update(b"\x32\x16\x66\x2d\xa0\x22\x79\x93\xd8\x82\x88\x18\x71\x77\xa0\x28\x7d\xe4\xec\xcf\x24\x5d\x7c\x71\x8b\x80\x45\xbb\xfb\x88\x69\xd9\x3f\x1f\xb9\xe9\x4d\x74\x78\xb0\x29\x8e\x62\x8c\x07\xe0\xed\xaa\xb0\x1d\xcf\x79\x26\x4d\xc0\x5f\x8b\x21\x81\xaa\x3f\x83\x1d\xc9\x49\x72\x6f\xbc\xf8\x0d\xe4\xc9\xc9\xed");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x64\xc4\x5e\x01\x8c\xfb\xc8\x8f\x8f\x4f\xfe\x3c\xef\x0d\xf3\xa9\x4a\xab\x30\x49\xfa\xfa\xe2\x8e\x28\xef\xbb\x2a\x4b\x94\x80\x9e\xb3\x02\xca\xf9\x01\x01\x0a\xbf\xa1\x94\xf7\x29\x65\x66\x3d\x35"[..]);

        ctx.update(b"\xe7\x76\xe6\x74\x9c\x5b\x6c\x7d\xef\x59\xcb\x98\x34\x09\x84\x53\x92\x80\xa9\x87\x4f\x80\x41\x2d\x4d\xf0\xee\x73\xd5\x8a\xcd\x10\x94\xd4\x9e\xd4\xe3\x51\x25\x83\x4c\xf8\xcf\xe3\x49\xe5\x99\x14\x4e\x4f\x2e\x20\x0a\xba\x4f\xd3\xeb\x6d\x78\xcd\xe0\x27\xc1\xd5\x62\x0e\x02\x70\xb5\xe8\x3a\xb2\x6b\x8d\x32");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x94\xbd\x67\xb7\xf2\x58\x7b\x0b\xda\x54\x87\xcc\x45\xd0\x0e\x43\x65\xf1\xee\x40\x07\x3c\xdf\x0d\x23\xa5\xea\x3f\xba\x01\xee\xf4\x2a\x46\xbf\xba\xc5\x30\x6d\x67\xbe\x02\xd8\xd9\x18\xae\x5c\x9a"[..]);

        ctx.update(b"\x5d\x8f\x84\xb2\xf2\x08\xb5\x8a\x68\xe8\x8c\xe8\xef\xb5\x43\xa8\x40\x4f\x0e\xc0\xc9\x80\x5c\x76\x0a\xd3\x59\xd1\x3f\xaa\xb8\x4d\x3f\x8b\xb1\xd2\xa4\xbb\x45\xe7\x2c\x0e\xc9\x24\x5f\xfd\xa2\xe5\x72\xf9\x4e\x46\x6c\xff\xa4\x4b\x87\x6d\x5c\x5e\xd9\x14\xd1\xff\x33\x8e\x06\xb7\x4a\xd1\xe7\x4d\x14\x05\xd2\x3d");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x94\x73\x50\x30\x77\x48\xc2\x94\x67\xf0\x01\x03\xd0\xa0\x7c\x3c\x22\x8c\x5f\x49\x4f\xc8\x8f\xe2\x35\x2c\xa5\xd1\x04\x49\xd0\xdd\xa7\x07\x67\x80\xc0\x54\x39\xa0\x96\x94\xeb\x52\x8d\x1f\x47\x7a"[..]);

        ctx.update(b"\x35\x7d\x57\x65\x59\x50\x65\xef\xe2\x81\xaf\xb8\xd0\x21\xd4\x76\x4f\xba\x09\x1a\xdd\xe0\x5e\x02\xaf\x0a\x43\x70\x51\xa0\x4a\x3b\x8e\x55\x2e\xc4\x8f\xb7\x15\x2c\x47\x04\x12\xc4\x0e\x40\xee\xc5\x8b\x84\x28\x42\xd8\x99\x3a\x5a\xe1\xc6\x1e\xb2\x0d\xe5\x11\x23\x21\xbc\x97\xaf\x61\x8b\xbf\xba\xf8\xe2\xa8\x76\x99");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x32\x28\x69\x70\x20\x4c\x34\x51\x95\x8f\x51\x55\xf0\x90\x44\x8f\x06\x1d\xd8\x1b\x13\x6a\x14\x59\x2a\x32\x04\xc6\xb0\x8e\x92\x2e\xe5\xbb\x6d\x65\x34\xdb\xf8\xef\xb4\xbb\x73\x87\x09\x2c\x84\x00"[..]);

        ctx.update(b"\xa8\xcb\x78\xe1\x48\x5c\xbb\x7a\x94\x74\xc1\xc1\xf8\xe0\xf3\x07\xcd\xa5\x13\x9a\x7e\x94\x7d\xf5\xea\x20\xac\x33\x0a\x6d\xff\xca\xd4\xa9\xbd\x75\x5f\x9f\x58\x72\x47\x89\xee\xee\x53\x26\x15\xbe\x55\x0d\xd8\x4f\x52\x41\xfd\xe0\xe3\x05\x8a\xee\xdb\xf2\x87\xf0\x2a\x46\x04\x45\x02\x7f\x5e\x6b\x38\x29\xbf\x71\xec\xf4");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x51\x16\x8b\xfe\xef\x8a\x98\x1c\x0d\xef\x0c\x4c\xb0\x67\xba\xf1\x5c\xe5\xfe\xb8\xd5\xf7\xe9\xd6\x07\x6b\x28\x36\x26\x73\x91\xae\xe1\xfd\x3a\x0b\x5d\x34\x34\xce\xb5\xcf\x2d\x6f\xa0\x6f\xa0\x63"[..]);

        ctx.update(b"\x81\xac\xca\x82\x54\x5e\x76\x7a\xb5\x9d\xcc\x75\x0a\x09\x84\x9c\xeb\xad\x08\xff\x31\xc9\x29\x7f\x4f\xd5\x10\xeb\xe6\xc2\x77\x69\x93\x83\x19\x18\x0c\xcc\x66\xf3\x6b\x1a\x7c\xf9\xc9\xf3\x53\x8b\x0f\x6f\x37\x15\x09\xf7\x7c\xf0\xbc\x4d\x6d\x87\xfa\xcc\x85\xb9\x33\xf2\xe2\x7f\x8e\x1b\xf6\xcf\x38\x8f\x80\xc0\xfc\xbf\xba");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x4a\xe4\x4d\x65\x09\x98\x68\x93\xa8\x41\x47\x53\xb5\x7d\x11\xf9\xc5\x54\xd8\x9c\x15\xad\x6d\x70\x68\x7c\x56\xc6\xc2\xac\x73\x53\x7a\xcb\xb0\xd5\x1f\x48\xe6\xbe\xa6\xcf\x76\x2d\x58\x89\x0d\x7a"[..]);

        ctx.update(b"\x94\x98\x74\x98\xb1\xca\x87\xa6\xf3\xfa\x4b\x99\x9d\xb7\x26\x11\x5c\x45\x5d\x0e\xc2\x40\x29\xb2\xf5\x81\x0e\x49\xa9\x46\x68\x86\x4b\x8c\x47\x0f\x7f\xc0\x7c\x3d\xcd\x97\xf4\x1c\x97\x3b\x45\xba\x4f\xa7\x87\x9e\xe7\x54\x65\x96\x88\x15\x73\xb6\x86\x3f\xc3\x9d\x94\x0e\xb3\xfa\x34\x44\x08\x4f\x72\x13\x41\xf5\xd2\x3d\x25\x61");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xa7\x33\xb1\x18\xbe\x72\xa1\x87\xdd\xcb\xe5\xba\x67\xe0\x4b\x58\x9f\x9c\xd9\xf8\x48\x2c\x4b\xd9\xd6\x4c\x58\x0a\xba\x7d\x19\xd2\xd1\xf9\xc1\xdd\xf9\x5f\xe6\xef\xde\xff\xd4\x4f\x67\xfc\xab\xb5"[..]);

        ctx.update(b"\xde\x6b\x32\xc2\xd4\x0d\x06\x59\x16\x6d\xb2\x35\x25\x9b\x53\x0e\xa4\x3f\x44\xe7\x5d\x8b\x3e\x9e\x85\x6e\xc4\xc1\x41\x0b\xbe\xa3\x69\x69\x64\xaf\x8b\x6c\x5d\xfd\x33\x04\x28\x23\x69\xa4\xbc\x4e\x7c\xf6\x6b\x91\xfe\xcd\x0c\x7c\x10\x5b\x59\xf1\xe0\xa4\x96\x33\x6f\x32\x74\x40\x98\x0a\x34\x61\x4e\xe0\x0f\xff\x25\x87\xd6\xb8\x13");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x17\xba\x30\xc0\xb5\xfc\x18\x5b\x32\x45\x31\x3b\x83\xdd\x04\x81\x14\x59\x53\x10\x11\x28\x91\x47\x65\x78\x4a\xf7\x51\x74\x5b\x8a\x2b\x6a\x90\xa4\x34\x54\x8f\x3a\xda\xf1\xf0\x7f\x18\x64\x98\x90"[..]);

        ctx.update(b"\x85\x42\x11\xbe\xda\xcc\x19\xf7\x7b\x46\xcf\xa4\x47\xa4\xad\x67\x2e\xa9\xb6\x43\xf0\x9f\x5c\xf5\x27\x4b\xa2\x88\x88\x20\x7e\x24\x66\xb3\x81\x27\x77\x6f\xb9\x76\xdb\x8a\xd7\x16\x5a\x37\x8d\xf6\xee\x1e\x3a\x0f\x81\x09\xc9\xaf\xf7\xe0\xd6\x12\x6f\xd7\x13\x33\xc6\xe6\xeb\xe1\x5d\x7a\x65\x15\x1d\x6a\x4a\x83\xb8\x2c\x8a\x6f\x31\x49");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xca\x85\x63\x2a\x9f\x7c\x32\xac\x47\x05\xc6\x45\x87\x70\x02\x5d\xda\x4f\xd0\x7a\x8d\x5d\x69\x21\xb8\x97\xb0\xda\x49\x0d\x64\x40\x05\x87\x64\x9f\x2d\x20\xbf\x60\x8b\x9a\x18\xd0\x71\xb6\x3b\x48"[..]);

        ctx.update(b"\x82\x23\x73\xd9\xd3\xd5\xb0\x6a\x8d\xa4\x8a\x43\x09\x57\x40\xfb\x98\xc9\xca\xf7\x17\x35\x0f\xd2\xc3\xb0\x58\x02\x4f\xf7\x05\xb9\x34\x6b\x7f\x0a\x49\x5a\x6d\x4d\x93\x80\x2b\xc4\x5e\xce\x77\x7f\x8c\x6a\x6e\x7c\x2e\xf6\xb8\x13\x51\x15\xff\x91\x1a\x2b\xa5\x24\x16\x65\xb6\xf7\xcb\xfa\x1b\x9d\x93\xb0\x11\xb3\xaa\xa1\xda\xc1\x85\x3f\xb2");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x6e\x84\x58\x7c\x8c\x6e\x54\x35\x3a\x60\x32\xe7\x50\x59\x02\xef\x7f\x0f\x05\x38\xdd\x1b\xb3\x29\x22\xe1\x3a\x7d\x4d\x98\xc4\x7a\x54\x10\x15\x38\x1e\xab\x27\xe9\x18\x63\x98\x12\x0d\xa7\xfb\x32"[..]);

        ctx.update(b"\xc0\x4b\x70\x1f\x68\x80\x92\xbb\xd1\xcf\x42\x17\xbc\x4b\x58\x77\xf2\xe6\x0c\x08\x7b\xda\xc4\x66\x11\x48\x2a\x61\xd5\x1f\x82\x01\x40\x40\x3b\xc8\x5b\xe0\xc3\x36\x33\x2d\xa0\x93\x87\x34\xbd\xe8\xc5\x02\x01\x4f\x35\x09\x26\x6c\x73\xc6\xc9\x3c\x22\xa1\xbd\x0d\xdf\x15\xa5\xce\x74\x10\xc2\x89\x4e\x9d\x09\x2e\x32\xc0\x79\x92\x2b\xa1\xab\xb7");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x75\xc5\x85\x50\x3f\x15\xa5\x26\x11\x36\x08\xbc\x18\x31\x80\xb1\xcb\x80\xf4\xd1\xb4\x66\xc5\x76\xbf\x02\x1b\x1c\xe7\xa1\x52\x83\x91\xf7\x0e\x10\x44\x66\x81\x84\x9f\xa8\xa6\x43\xcb\x2b\x68\x28"[..]);

        ctx.update(b"\x00\x9d\xd8\x21\xcb\xed\x12\x35\x88\x0f\xe6\x47\xe1\x91\xfe\x6f\x65\x55\xfd\xc9\x8b\x8a\xad\x0f\xf3\xda\x5a\x6d\xf0\xe5\x79\x90\x44\xef\x8e\x01\x2a\xd5\x4c\xb1\x9a\x46\xfd\xd5\xc8\x2f\x24\xf3\xee\x77\x61\x3d\x4b\xed\x96\x1f\x6b\x7f\x48\x14\xaa\xac\x48\xbd\xf4\x3c\x92\x34\xce\x2e\x75\x9e\x9a\xf2\xf4\xff\x16\xd8\x6d\x53\x27\xc9\x78\xda\xd5");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x02\xa0\x9d\x37\xd3\x1e\x43\x65\xc2\x6b\xec\x0e\xaa\xce\xcf\x29\xee\xa4\xe8\xd2\x1a\xb9\x15\xdd\x60\x52\x48\x76\x4d\x96\x4f\x10\xeb\xb8\xfa\xfd\xb5\x91\x98\x2d\x33\x86\x9a\x1d\x08\xa7\xe3\x13"[..]);

        ctx.update(b"\x0b\x7d\xd6\x70\x9d\x55\xe0\xd5\x26\xd6\x4c\x0c\x5a\xf4\x0a\xcf\x59\x5b\xe3\x53\xd7\x05\xbe\x7b\x7a\x0b\x1c\x4c\x83\xbb\xe6\xa1\xb1\xec\x68\x1f\x62\x8e\x9d\x6c\xfc\x85\xad\x9c\x8b\xb8\xb4\xec\xac\x64\xc5\xb3\xa9\xb7\x2f\x95\xe5\x9a\xfe\xfa\x7b\xce\xc5\xbe\x22\x3a\x9b\x2b\x54\x83\x64\x24\xaf\xde\x52\xa2\x9b\x22\xab\x65\x2d\x22\xcc\xe3\x4b\x39");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5c\x84\xae\x39\xd9\x59\xb7\x95\x55\x23\x17\x46\xad\x5b\x33\x68\x9a\x31\x72\x0e\xd0\x07\x0f\x67\x72\x14\x79\x77\xed\xd0\xae\xad\x07\xfb\x8b\x7b\x71\xb0\xbd\x58\x7e\xbc\x5c\x1a\x80\xd5\x64\xc7"[..]);

        ctx.update(b"\x3e\x9b\x65\xd7\xbf\x42\x39\x42\x0a\xfa\x86\x39\xc8\x19\x5b\x63\x90\x2b\x24\x49\x5b\x95\xc4\x14\x39\x78\xe4\x98\x43\xd8\x8a\x92\xd1\xfe\xed\x2e\xed\x1a\x88\xcd\x07\x2d\x6d\x04\xea\x26\xdc\xe8\xee\x4b\x14\x89\x6f\xdb\x69\xbc\x7f\xf2\x97\x1e\xd8\xac\x56\x55\x14\x8d\x2e\x99\x21\x21\x8d\x74\xef\xdf\x17\xc5\x6b\x53\x3d\x0b\xb1\x7d\x11\xe0\x7d\x74\x58");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xab\x78\x90\xd1\xb5\x1a\xf1\x02\x85\x75\x2b\xf9\xda\x5e\xee\x5c\x3e\x87\xa2\x85\xdc\x33\x26\x2d\x02\x61\xaa\x9a\x57\x5f\x30\x3e\x94\x84\x5d\x7a\xb2\x1b\x48\xf4\xe6\x88\x45\x68\xcd\x78\xb5\x50"[..]);

        ctx.update(b"\x94\x36\xda\x43\x3d\x1e\xbd\x10\xb9\x46\xb1\x29\xcb\x34\xbc\xce\xc9\xb8\xf7\x05\xaa\xba\x3f\x85\x61\x35\x2e\xd3\x6a\x84\x49\xab\xa2\xdd\x7b\xa1\x5b\x1b\xc3\x08\xb0\xc0\x29\x13\x16\x3a\xf6\x3a\x34\x65\x24\xdf\xf5\x52\x14\x32\xdb\x47\x7f\x52\x96\x06\xaf\xb5\xd5\x52\xef\xc9\x5c\xb0\x40\xdb\x56\x6b\x4d\x39\xed\xda\xa1\x93\x19\xe5\x18\xa7\xb5\xc6\x93\x1e");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x96\x8a\xe9\x10\x4f\x9c\x90\x7c\x5a\x72\x93\x62\x50\xdf\xed\xd6\x2c\xd0\x4f\x6e\x5d\xdd\x2c\x11\x34\x90\x80\x8a\x11\x88\x44\x49\xaa\xef\x5d\x01\x3e\xa3\x99\x3a\x6c\xb6\xfc\x5c\x08\x75\x44\x08"[..]);

        ctx.update(b"\x37\x25\x4b\xf9\xbc\x7c\xd4\xed\x72\xe7\x2b\x6b\xb6\x23\xa0\xcc\x8e\xeb\x96\x3d\x82\x7a\xef\x65\xad\x4b\xc5\x49\x13\x23\x5b\x6d\x35\x51\x53\x3c\xe3\x34\x21\xaa\x52\xff\xbf\x18\x6e\xb9\xa2\x78\x71\x88\xee\xb1\xb5\x2e\xe6\x45\xc6\xd4\xa6\x31\xbc\x07\x14\x15\xc8\x00\x14\x94\x0c\x28\xfb\xfe\xb0\xdb\x47\x2c\x32\x6c\x8d\xac\xfd\x6a\xb2\x1f\x3e\x22\x5e\xde\xf3");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x97\x5e\x10\xfa\xc9\xaa\x77\xb7\x80\xe5\xf6\xc2\x15\x1e\xc4\xa3\xc7\x2f\xf2\x6e\x41\x23\x3c\xc7\x74\xc0\x74\xdf\x1b\x78\xcc\xe5\xaf\x11\x91\xba\x95\x5a\x0b\xce\x15\x92\x6a\xe6\x91\xb0\xff\xe7"[..]);

        ctx.update(b"\x79\xe7\x7c\xd0\x8a\x6e\xf7\x70\xbb\xe4\xbe\xdf\x61\x55\x7e\xa6\x32\xb4\x2d\x78\x63\x71\x49\x67\x0d\x4d\x61\x57\xd5\x6e\xd7\xb2\xcc\xae\xe4\x5d\x94\x39\xdc\xeb\xc5\x57\xb4\x11\x8e\x86\xc1\x5a\xa0\xcc\xc2\x1c\x47\x4b\x21\xab\xda\x16\x76\xcc\x56\x43\x4d\x6d\x46\x42\x29\x93\xe6\x6d\xc9\x93\x87\xdf\xa9\x85\x35\x8a\xcc\xf6\x98\x84\xb9\xdd\x18\xa2\xc4\xd0\x44\x48");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x94\x72\x9f\x5f\x99\xa5\x4f\x5a\x3e\xa6\x92\x33\xff\x9d\x52\x23\x92\xd4\x59\x6e\xb6\xac\x2b\xbb\x07\x49\x2e\xce\x3c\x67\x31\x74\x12\xbb\x47\xae\x31\x7d\xdd\x20\x53\x6c\x3a\xdc\x00\x38\x62\xf1"[..]);

        ctx.update(b"\x64\xb7\x6c\xb5\x54\xf6\xbe\xcc\x23\x8a\x3f\xcf\xc3\xeb\x97\x99\x36\x67\xec\x82\xfd\xc3\xfb\x28\xd4\x25\x67\x70\x9c\x32\x50\xc7\x99\x73\x28\xae\xdd\xfd\xc2\x75\x04\x51\xac\x46\x22\x81\xbf\x66\xfa\x94\xf4\xb8\x71\x2c\x7a\x83\x42\x66\x05\x74\xf2\x02\x68\xe7\x07\xc4\x66\x62\x75\x19\xc5\x62\x59\xfe\xa5\x5b\xe9\x1e\x10\xfa\xab\x3a\xd2\xad\xe6\xce\x8b\x65\x57\xf2\x02");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x26\xd4\x8e\xf5\x06\x7d\x70\x4e\xe9\xe2\xa6\x4e\x39\x9d\xe2\x30\x68\x90\x8b\x3c\x91\x1f\xfc\x40\x56\xc1\x68\x36\x2c\x37\x38\x5c\x92\xd3\x7d\x51\x35\x4b\x65\x05\xa8\x2c\x4d\x22\xfe\xc3\x7e\xaa"[..]);

        ctx.update(b"\x3d\xf2\x78\x29\xbf\xb1\xab\x7d\x38\x1f\x14\x6b\x30\x37\x0e\xf5\x6b\x39\x2b\x73\xb3\x5b\x1b\xe5\xd8\xbb\xcf\x88\xf4\x99\xdd\xa7\xf3\xc3\x27\xb4\x53\x50\xb8\x97\x29\x91\xee\x46\x65\x45\xde\x96\x56\x0c\xf4\x51\x71\x1f\xda\x88\x4e\x3d\x9b\x2a\xf3\xe9\x09\xd6\x55\xd2\x5c\xee\x1c\x93\x1b\xed\xa7\x9c\x40\xfa\x50\x70\x97\xbd\xf1\x12\x67\x71\xa7\xb9\x54\x3a\xd5\xcb\x84\xb9");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x5f\xa4\xeb\xfa\x24\x15\x02\x36\xc0\x34\x09\xf0\x85\x7b\x31\xcb\x95\xb0\x15\x0f\x38\x1c\x88\x58\xb0\x15\x59\x95\x7b\x12\x68\xf7\x3c\x69\x87\x09\x23\x3e\x6b\x15\x46\x86\x75\xa1\x02\xd0\xc5\xe5"[..]);

        ctx.update(b"\xb0\x0f\x4e\x67\xca\x08\xcc\xfa\x32\xb2\x69\x8f\x70\x41\x1d\x8f\x57\x0f\x69\xc8\x96\xe1\x8e\xc8\x89\x6c\xfe\x89\x55\x18\x10\x54\x33\x03\xf7\xdf\x0c\x49\xf5\xb9\x47\x83\xcc\xe7\xdf\x8d\x76\xd0\xb8\x8d\x15\x56\x33\x30\x2d\x46\x00\x37\x11\xf2\x33\x33\x9b\x1c\x9a\x8c\x20\x16\x4e\xc8\xa3\x28\x89\x0a\x49\x32\xb7\xd9\x0d\x92\xd0\x23\xb5\x48\xe4\x82\x05\x58\xf8\xbd\x32\x70\x10");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xea\xa7\x56\xb5\x89\x2f\xdf\xc7\x93\xd7\x4e\x3f\x9f\x4d\x6c\x7a\x5a\x6a\x22\x41\xdd\x11\xe0\xc3\x8c\xed\x59\xc8\xec\x7b\xe3\x77\xa4\x1d\x1d\x06\x77\x4a\x59\x70\xce\x97\x22\xd8\xe1\x19\xd0\xad"[..]);

        ctx.update(b"\xa4\xf9\x5f\x6a\x46\xa9\xcb\xf3\x84\xa7\xe9\x8e\x10\x2d\x1f\xdc\x96\x83\x9d\x1b\xf2\x6b\x35\xa5\xa0\xbd\x6c\xb9\x73\x4f\xd1\x7e\x8a\x17\x8d\x45\x81\x94\x3c\x0f\xe4\x69\xfb\x4f\xe9\x4c\xc2\xf1\x5e\x1e\xf5\x9a\xe0\x5b\x35\x32\x4e\xb5\x7c\xa0\x7d\xfc\x69\xd4\x2d\x41\xd8\x0b\x3c\x3b\xb6\x4e\x1a\xea\x14\x3c\x7d\x79\x79\x0a\x56\x69\x7d\xc8\x03\xec\x93\xe6\xc6\x8f\x27\xf6\x76\x1c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x1a\xff\x8d\x9c\x64\xf0\xc1\x62\xed\x01\x95\xd1\xf3\xa3\x42\xa0\x10\xd1\x4b\xe0\x63\x69\x03\xc4\x80\x20\xba\x42\xde\x1c\xfa\x8b\x98\xae\x21\x42\xd8\x9a\xf3\xe6\x9e\x9e\xb4\xc7\x35\x85\x7d\xd1"[..]);

        ctx.update(b"\x02\x71\x30\x84\xbf\x93\xfd\xc3\x51\x35\x51\x52\x43\xc3\xbc\x0f\x4b\x2b\x44\x7f\x2d\x34\x61\xc0\xdc\x10\x4c\xbf\xe2\x34\x79\xab\x03\x67\x62\xa9\x1d\x19\x87\xc9\x53\xf7\xb3\x38\x6a\xbc\x80\xb8\x73\x4a\x1d\x4e\xab\xf9\x4f\x3a\x9f\x2f\xb6\x2c\x94\x31\x52\xb5\x25\x38\x46\xfc\x2e\xc8\xdb\xb2\xe9\x3d\xc7\x48\x57\xa7\xb0\x5f\xe2\xd7\xec\x80\x40\xba\x8b\x0d\x9a\xe6\x97\x77\xee\x73\x9a");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x84\xda\x02\x11\x4e\x34\x1a\x36\x36\xf0\x08\x22\xb3\x2b\xd2\x1a\x8a\x1f\x7b\x39\xf2\x95\x6b\xd9\x7f\x39\x34\x6f\xed\xf9\xaa\xe6\x3b\x30\x4c\x65\xc9\x3a\x54\x1e\x8b\xcd\xa5\x49\x57\x6d\x5f\x27"[..]);

        ctx.update(b"\x00\xce\x22\x5e\xae\xa2\x48\x43\x40\x6f\xa4\x2c\xc8\x45\x0e\x66\xf7\x6a\xc9\xf5\x49\xb8\x59\x1f\x7d\x40\x94\x2f\x48\x33\xfc\x73\x4a\x03\x4c\x87\x41\xc5\x51\xd5\x7d\xda\xfb\x5d\x94\xce\xb4\xb2\x56\x80\xf0\x45\x03\x83\x06\xe6\xbc\xc5\x3e\x88\x38\x6e\x2b\x45\xb8\x0b\x3b\xa2\x3d\xec\x8c\x13\xf8\xca\x01\xc2\x02\xae\x96\x8c\x4d\x0d\xf0\x4c\xdb\x38\x39\x5d\x2d\xf4\x2a\x5a\xff\x64\x69\x28");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x81\xd6\xe0\xd9\x65\x75\xa9\xb8\xca\x08\x3e\xe9\xec\x2e\xad\x57\xdd\xf7\x2b\x97\xd7\x70\x90\x86\xa2\xf4\xa7\x49\xd3\xf6\x1d\x16\x42\x34\x63\x48\x75\x62\xc7\xf0\x9a\xba\x1b\x26\xe8\xca\xe4\x7b"[..]);

        ctx.update(b"\x7a\xf3\xfe\xed\x9b\x0f\x6e\x94\x08\xe8\xc0\x39\x7c\x9b\xb6\x71\xd0\xf3\xf8\x09\x26\xd2\xf4\x8f\x68\xd2\xe8\x14\xf1\x2b\x3d\x31\x89\xd8\x17\x48\x97\xf5\x2a\x0c\x92\x6c\xcf\x44\xb9\xd0\x57\xcc\x04\x89\x9f\xdc\x5a\x32\xe4\x8c\x04\x3f\xd9\x98\x62\xe3\xf7\x61\xdc\x31\x15\x35\x1c\x81\x38\xd0\x7a\x15\xac\x23\xb8\xfc\x54\x54\xf0\x37\x3e\x05\xca\x1b\x7a\xd9\xf2\xf6\x2d\x34\xca\xf5\xe1\x43\x5c");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x00\xe9\x5f\x4e\x8a\x32\xa0\x3e\x0a\x3a\xfb\xa0\xfd\x62\xc7\xc3\xc7\x12\x0b\x41\xe2\x97\xa7\xff\x14\x95\x8c\x0b\xdf\x01\x5a\x47\x8f\x7b\xab\x9a\x22\x08\x2b\xfb\x0d\x20\x6e\x88\xf4\x68\x51\x17"[..]);

        ctx.update(b"\x2e\xae\x76\xf4\xe7\xf4\x8d\x36\xcd\x83\x60\x78\x13\xce\x6b\xd9\xab\x0e\xcf\x84\x6a\xd9\x99\xdf\x67\xf6\x47\x06\xa4\x70\x89\x77\xf0\xe9\x44\x0f\x0b\x31\xdc\x35\x0c\x17\xb3\x55\x00\x7f\xed\x90\xd4\xb5\x77\xb1\x75\x01\x47\x63\x35\x7c\xe5\xa2\x71\x21\x2a\x70\x70\x27\x47\xc9\x8f\x8f\x0a\xd8\x9b\xf9\x5d\x6b\x7f\xbb\x10\xa5\x1f\x34\xd8\xf2\x83\x5e\x97\x40\x38\xa3\xdd\x6d\xf3\xf2\xaf\xfb\x78\x11");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xeb\x39\x6c\xfa\xf2\x6e\xe2\x77\x5a\xf3\xc9\xa3\xa3\x04\x76\x64\xca\x34\xcb\xc2\x28\xcc\xbb\x96\x6d\xf1\x87\xd5\x18\x71\x7d\xf6\xa3\x28\xec\xc3\x16\xed\x0e\xd0\x9b\x17\x00\x80\xec\xcc\x48\x6f"[..]);

        ctx.update(b"\x09\x3e\x56\xd3\x3b\xd9\x33\x7a\xd2\xad\x26\x8d\x14\xba\xc6\x9a\x64\xa8\xa7\x36\x13\x50\xcf\x9f\x78\x7e\x69\xa0\x43\xf5\xbe\xb5\x0e\xb4\x60\x70\x35\x78\xa8\x1b\xe8\x82\x63\x9f\x7e\x9a\xc9\xa5\x0c\x54\xaf\xfa\x37\x92\xfd\x38\x46\x4a\x61\xa3\x7c\x8a\x45\x51\xa4\xb9\xff\x8e\xed\x1f\x48\x7e\xf8\xa8\xf0\x04\x30\xe4\xd0\xe3\x5a\x53\xff\x23\x6c\xe0\x49\xb7\xa3\xab\xdc\x5c\xd0\x0b\x45\xc4\xf3\xd4\x9b");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x4a\x33\x91\x28\x48\x6e\x5b\x27\x4f\xc4\xed\x53\x8c\x0e\xc9\xe5\x7f\x78\x0e\x9c\x50\x0c\x5f\x92\xb0\x4a\xe8\x1a\x22\xfb\xee\xbf\x37\x85\x25\x9a\x0b\xb3\xb6\xd9\xb4\x7f\x31\x87\x3c\xd8\xdf\xfa"[..]);

        ctx.update(b"\x05\x93\xba\xbe\x7a\x62\x02\x07\x7c\x02\x6e\x25\x3c\xb4\xc6\x0e\xe7\xba\xd7\xb1\xc3\x1a\x20\xda\x7a\xa0\xce\x56\xb6\x22\xeb\x57\xed\x07\xd2\x1a\x7f\x0a\xe6\xc6\xfe\x3c\x83\x98\xcc\x48\x35\x3d\xec\xfb\x28\x7f\x12\x04\xe0\x24\xfc\xf8\x2a\x13\x05\x99\x53\xb9\xf8\x57\x97\xab\x22\x17\xdc\x8d\xab\x34\xa1\x32\x26\xc3\x31\x04\x66\x1c\x1c\xa7\x93\x96\xe7\xd9\x7e\x91\x03\x9d\x32\xba\xfc\x98\xcc\x8a\xf3\xbb");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x59\x81\x81\x5c\x16\x18\xcc\x49\xcd\x5c\xf7\x1a\x4b\x7b\x32\xb8\xcd\x7b\x7e\xf5\x53\xbf\xae\xf2\x14\x9a\xc7\x23\xff\x25\x82\xa2\xd3\x45\xc5\xbd\x05\x94\x3e\x15\x5c\xed\x1e\x5f\x09\x1c\x56\x01"[..]);

        ctx.update(b"\xae\x18\x28\x04\x7c\x5f\x82\xa7\xb9\x71\x2f\x33\x99\x83\x21\x24\xb8\x92\xf2\xf7\xae\xa5\x1c\x8f\xe3\x53\x6c\xd6\xa5\x84\xb4\xa7\x77\x7c\xc1\xec\xac\x15\x8c\x03\x35\x4b\xb4\x67\xb8\xfe\x2c\x8c\xe2\xf4\x31\x0a\xfd\x1e\x80\xfe\xc5\x1c\xc5\xad\x77\x02\x56\x6b\x2c\x5d\x21\xbc\x65\x71\xe4\xb8\xe7\xc5\x9c\xb4\xc9\xe2\x3f\x1e\xcb\x57\xad\xa9\xe9\x00\xe4\xaa\x30\x88\x74\xc2\xd1\x2d\x34\xbe\x74\xc3\x32\xbb\xce");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x72\x57\xf5\xbf\xa7\xd3\x3d\x1c\xf5\xf4\x55\x0d\x0c\xb7\x87\x50\xe8\x4c\x5b\x7d\x25\x02\x7d\xa6\xac\xec\x64\xbd\xf3\x08\x79\xa0\xe5\xc9\x7f\xe7\xc4\x68\xe7\x43\xaa\x5e\xc2\xbd\xdb\x29\xd1\x93"[..]);

        ctx.update(b"\x3b\xce\xed\xf5\xdf\x8f\xe6\x99\x87\x1d\xec\xb7\xdd\x48\x20\x3e\x25\x18\xfb\x0f\xce\x0f\x86\x5f\x46\xad\xce\x5c\x13\x3a\x92\x13\x20\xbf\x40\x91\x54\x56\x20\x48\x69\xa3\xce\xb5\xfc\xa3\xed\x40\xe0\xa4\x1a\x64\xb8\x95\x1f\x0f\xc5\x80\x69\x4c\xfc\x55\xbd\x1f\x5c\xe9\x26\xb0\x7e\x3e\x32\xac\x6e\x05\x5d\xe9\xb9\x61\xce\x49\xc7\xee\x41\xe0\x6b\x02\x45\x59\xb9\x33\xa7\x95\x18\x19\x2e\x96\x98\x55\x88\x9c\x85\xd1");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x60\xd7\xf8\xbd\x85\xfb\x7a\x13\x70\x1d\xb5\xad\xed\x2b\x77\x71\xab\x5e\x47\x6e\xc3\x4f\x1f\xd4\x29\x89\x78\xde\xfb\xd2\xb3\x1b\xb2\x97\x93\x91\x55\x9a\x16\x4b\x3e\xd2\x8f\x6a\x39\x03\x1a\x11"[..]);

        ctx.update(b"\x6c\x36\x14\x76\x52\xe7\x1b\x56\x0b\xec\xbc\xa1\xe7\x65\x6c\x81\xb4\xf7\x0b\xec\xe2\x63\x21\xd5\xe5\x5e\x67\xa3\xdb\x9d\x89\xe2\x6f\x2f\x2a\x38\xfd\x0f\x28\x9b\xf7\xfa\x22\xc2\x87\x7e\x38\xd9\x75\x54\x12\x79\x4c\xef\x24\xd7\xb8\x55\x30\x3c\x33\x2e\x0c\xb5\xe0\x1a\xa5\x0b\xb7\x48\x44\xf5\xe3\x45\x10\x8d\x68\x11\xd5\x01\x09\x78\x03\x8b\x69\x9f\xfa\xa3\x70\xde\x84\x73\xf0\xcd\xa3\x8b\x89\xa2\x8e\xd6\xca\xba\xf6");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\xb1\x31\x91\x92\xdf\x11\xfa\xa0\x0d\x3c\x4b\x06\x8b\xec\xc8\xf1\xba\x3b\x00\xe0\xd1\xff\x1f\x93\xc1\x1a\x36\x63\x52\x2f\xdb\x92\xab\x3c\xca\x38\x96\x34\x68\x7c\x63\x2e\x0a\x4b\x5a\x26\xce\x92"[..]);

        ctx.update(b"\x92\xc4\x1d\x34\xbd\x24\x9c\x18\x2a\xd4\xe1\x8e\x3b\x85\x67\x70\x76\x6f\x17\x57\x20\x96\x75\x02\x0d\x4c\x1c\xf7\xb6\xf7\x68\x6c\x8c\x14\x72\x67\x8c\x7c\x41\x25\x14\xe6\x3e\xb9\xf5\xae\xe9\xf5\xc9\xd5\xcb\x8d\x87\x48\xab\x7a\x54\x65\x05\x9d\x9c\xbb\xb8\xa5\x62\x11\xff\x32\xd4\xaa\xa2\x3a\x23\xc8\x6e\xad\x91\x6f\xe2\x54\xcc\x6b\x2b\xff\x7a\x95\x53\xdf\x15\x51\xb5\x31\xf9\x5b\xb4\x1c\xbb\xc4\xac\xdd\xbd\x37\x29\x21");
        ctx.digest(&mut digest);
        assert_eq!(digest, &b"\x71\x30\x7e\xec\x13\x55\xf7\x3e\x5b\x72\x6e\xd9\xef\xa1\x12\x90\x86\xaf\x81\x36\x4e\x30\xa2\x91\xf6\x84\xdf\xad\xe6\x93\xcc\x4b\xc3\xd6\xff\xcb\x7f\x3b\x40\x12\xa2\x19\x76\xff\x9e\xdc\xab\x61"[..]);
    }
}
