<?php
// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.

/**
 * Merges en_fix strings into the Git working directory.
 *
 * @package     local_amos
 * @copyright   2013 David Mudrak <david@moodle.com>
 * @license     https://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

define('CLI_SCRIPT', true);

require(__DIR__ . '/../../../config.php');
require_once($CFG->dirroot . '/local/amos/mlanglib.php');
require_once($CFG->dirroot . '/local/amos/cli/utilslib.php');

$help =
"Merges en_fix strings into the Git working directory.

Options:

--symlinksdir   Full path to the directory with symbolic links generated
                by the enfix-makesymlinks.sh script
--enfixdir      Full path to the directory containing en_fix string files
                generated by enfix-export.php script
--help, -h      Print out this help

Example:
\$ php enfix-merge.php --symlinksdir=/path/to/moodle --enfixdir=/tmp/export-enfix
";

list($options, $unrecognized) = cli_get_params([
    'symlinksdir' => '',
    'enfixdir' => '',
    'help' => false,
], [
    'h' => 'help',
]);

if ($options['help'] or empty($options['symlinksdir']) or empty($options['enfixdir'])) {
    cli_error($help, 2);
}

// Make sure that all target files exist, are symlinks and are writable.

foreach (new DirectoryIterator($options['enfixdir']) as $enfixfileinfo) {
    if ($enfixfileinfo->isDot()) {
        continue;
    }
    $filename = $enfixfileinfo->getFilename();

    if (!is_link($options['symlinksdir'].'/'.$filename)) {
        cli_error('File not symlink: '.$options['symlinksdir'].'/'.$filename);
    }

    if (!is_writable($options['symlinksdir'].'/'.$filename)) {
        cli_error('File not writable: '.$options['symlinksdir'].'/'.$filename);
    }
}

// Let's rock!

$logger = new amos_cli_logger();
$helper = new amos_merge_string_files($logger);
$total = 0;

foreach (new DirectoryIterator($options['enfixdir']) as $enfixfileinfo) {
    if ($enfixfileinfo->isDot()) {
        continue;
    }
    $filename = $enfixfileinfo->getFilename();

    $logger->log('enfix-merge', 'Processing file '.$filename.' ...');

    if ($filename === 'langconfig.php') {
        $logger->log('enfix-merge', 'Skipping file '.$filename, amos_cli_logger::LEVEL_DEBUG);
        continue;
    }

    $filecontents = file_get_contents($options['symlinksdir'].'/'.$filename);

    $fromstrings = $helper->load_strings_from_file($options['symlinksdir'].'/'.$filename);
    $logger->log('enfix-merge', count($fromstrings) . ' string(s) found in ' . $options['symlinksdir'] . '/' . $filename,
        amos_cli_logger::LEVEL_DEBUG);

    $tostrings = $helper->load_strings_from_file($options['enfixdir'] . '/' . $filename);
    $logger->log('enfix-merge', count($tostrings) . ' string(s) found in ' . $options['enfixdir'] . '/' . $filename,
        amos_cli_logger::LEVEL_DEBUG);

    $changes = $helper->replace_strings_in_file($filecontents, $fromstrings, $tostrings);

    if ($changes) {
        $total += $changes;
        file_put_contents($options['symlinksdir'].'/'.$filename, $filecontents);
        $logger->log('enfix-merge', $changes.' string(s) fixed in '.$filename);
    } else if ($changes === 0) {
        $logger->log('enfix-merge', 'No changes in '.$filename);
    } else {
        $logger->log('enfix-merge', 'Error while processing file '.$filename, amos_cli_logger::LEVEL_ERROR);
    }
}

$logger->log('enfix-merge', 'Finished! Total of ' . $total . ' string(s) merged.');
