<?php

namespace App\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\KernelInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\ORM\Id\AssignedGenerator;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

use App\Service\toolService;
use App\Entity\Scrum as Entity;
use App\Form\ScrumType as Form;
use App\Entity\Scrumcolumn as Scrumcolumn;
use App\Entity\Scrumwidget as Scrumwidget;
use App\Entity\Document as Document;
use App\Entity\Message as Message;

class ScrumController extends AbstractController
{
    private $data   = "scrum";
    private $route  = "app_user_scrum";
    private $render = "Scrum/";
    private $entity = "App:Scrum";
    private $appKernel;
    private $tool;

    public function __construct(KernelInterface $appKernel,toolService $tool)
    {
        $this->appKernel = $appKernel;
        $this->tool = $tool;
    }

    public function list(Request $request)
    {
        $em = $this->getDoctrine()->getManager();
        $datas = $em->getRepository($this->entity)->findAll();

        return $this->render($this->render.'list.html.twig',[
            $this->data."s" => $datas,
            "useheader"     => true,
            "usesidebar"    => true,
        ]);
    }

    public function view($id, Request $request, $access="user")
    {
        $em = $this->getDoctrine()->getManager();
        $data=$em->getRepository($this->entity)->find($id);
        if (!$data) throw $this->createNotFoundException('Unable to find entity.');

        $perm = $this->perm($data);
        if(!$perm) return $this->redirect($this->generateUrl('app_user_noperm',["displayname"=>$data->getUser()->getDisplayname(),"email"=>$data->getUser()->getEmail(),"url"=>$this->generateUrl($this->route."_view",["id"=>$id], UrlGeneratorInterface::ABSOLUTE_URL)]));    

        $form = $this->createForm(Form::class,$data,array("mode"=>"view", "access" => $access,"id" => $id));

        return $this->render($this->render.'view.html.twig',[
            'useheader'     => true,
            'usemenu'       => false,
            'usesidebar'    => ($access=="admin"),   
            'maxwidth'      => false,           
            $this->data     => $data,
            'form'          => $form->createView(),
            'access'        => $access,
            'perm'          => $perm,
        ]);
    }

    public function info($id, Request $request, $access="user")
    {
        // S'assurer que c'est un appel ajax
        if (!$request->isXmlHttpRequest()) {
            return new JsonResponse(array('message' => 'Interdit'), 400);
        }

        $em = $this->getDoctrine()->getManager();
        $data=$em->getRepository($this->entity)->find($id);
        if (!$data) return new JsonResponse(array('message' => 'Interdit'), 400);

        $output=[];
        foreach($data->getScrumcolumns() as $scrumcolumn) {
            $output[$scrumcolumn->getRoworder()] = [
                "id" => $scrumcolumn->getId(),
                "title" => $scrumcolumn->getTitle(),
                "size" => $scrumcolumn->getSize(),
                "widgets" => [],
            ];

            foreach($scrumcolumn->getScrumwidgets() as $scrumwidget) {
                $description=$scrumwidget->getDescription();
                $filetype=$scrumwidget->getFiletype();
                $file=$scrumwidget->getFile();
                $messages=$scrumwidget->getMessages();

                // Modification de l'url youtube
                if($filetype=="youtube"&&stripos($file,"https://www.youtube.com/embed")===false&&stripos($file,"youtube")!==false) {
                    $file=str_replace("http://www.youtube.com","https://www.youtube.com",$file);
                    $file=str_replace("https://www.youtube.com/watch?v=","",$file);   
                    $tmp=explode("&",$file);
                    $file="https://www.youtube.com/embed/".$tmp[0];
                }

                // Modification de l'url peertube
                if($filetype=="youtube"&&stripos($file,"/watch/")!==false) {
                    $file=str_replace("/watch/","/embed/",$file);   
                }

                $output[$scrumcolumn->getRoworder()]["widgets"][$scrumwidget->getRoworder()] = [
                    "id" => $scrumwidget->getId(),
                    "title" => $scrumwidget->getTitle(),
                    "description" => ($description?$description:""),
                    "filetype" => ($filetype?$filetype:""),
                    "file" => ($file?$file:""),
                    "username" => ($scrumwidget->getUser()?$scrumwidget->getUser()->getUsername():""),
                    "cptmessage" => $messages->count(),
                ];
            }
        }

        $response = new Response(json_encode($output));    
        $response->headers->set('Content-Type', 'application/json');      
        return $response;
    }

    public function image($id)
    {
        return $this->render($this->render.'image.html.twig',["id"=>$id]);
    }

    public function message(Request $request, $access="user")
    {
        // S'assurer que c'est un appel ajax
        if (!$request->isXmlHttpRequest()) {
            return new JsonResponse(array('message' => 'Interdit'), 400);
        }

        $id=$request->request->get('id');
        $em = $this->getDoctrine()->getManager();
        $data=$em->getRepository("App:Scrumwidget")->find($id);
        if (!$data) return new JsonResponse(array('message' => 'Interdit'), 400);
        $perm = $this->perm($data->getScrumColumn()->getScrum());
        if(!$perm) return false;

        $messages=$em->getRepository("App:Message")->findBy(["scrumwidget"=>$data],["submitdate"=>"DESC"]);
        $output=[];
        foreach($messages as $message) {
            $msg=$message->getMessage();
            $username=$message->getUser()->getUsername();

            $fgcandel=false;
            if($message->getUser()==$this->getUser()) $fgcandel=true;
            if($data->getUser()==$this->getUser()) $fgcandel=true;
            if($data->getScrumcolumn()->getScrum()->getUser()==$this->getUser()) $fgcandel=true;

            array_push($output, [
                "idwid" => $data->getId(),
                "idmsg" => $message->getId(),
                "message" => $msg,
                "date" => $message->getSubmitdate()->format("d/m/Y H:i"),
                "username" => $username,
                "fgcandel" => $fgcandel,
            ]);
        }

        $response = new Response(json_encode($output));    
        $response->headers->set('Content-Type', 'application/json');      
        return $response;        
        
    }

    public function submit(Request $request, $access="user") {
        // Initialisation de l'enregistrement
        $em = $this->getDoctrine()->getManager();
        $data = new Entity();
        if($access=="user") $data->setUser($this->getUser());

        // Création du formulaire
        $form = $this->createForm(Form::class,$data,array("mode"=>"submit", "access" => $access));

        // Récupération des data du formulaire
        $form->handleRequest($request);
        
        // Sur erreur
        $this->getErrorForm(null,$form,$request,$data,"submit");
        
        // Sur validation
        if ($form->get('submit')->isClicked() && $form->isValid()) {  
            $data = $form->getData();  
            
            $em->persist($data);
            $em->flush();

            // Retour à la liste
            return $this->redirect($this->generateUrl('app_'.$access.'_scrum_view',["id"=>$data->getId()]));    
        }

        // Affichage du formulaire
        return $this->render($this->render.'edit.html.twig', [
            'useheader'         => true,
            'usesidebar'        => ($access=="admin"),     
            'maxwidth'          => ($access=="user"),            
            $this->data         => $data,
            'mode'              => 'submit',
            'form'              => $form->createView(),
            'access'            => $access,
        ]);    
    }

    public function update($id, Request $request, $access="user") {
        $em = $this->getDoctrine()->getManager();
        $data=$em->getRepository($this->entity)->find($id);
        if (!$data) throw $this->createNotFoundException('Unable to find entity.');
        if($data->getUser()!=$this->getUser()&&!$this->getUser()->hasRole("ROLE_ADMIN")&&!$this->getUser()->hasRole("ROLE_MODO")) 
        throw $this->createNotFoundException('Permission denied');

        // Création du formulaire
        $form = $this->createForm(Form::class,$data,array("mode"=>"update", "access" => $access));

        // Récupération des data du formulaire
        $form->handleRequest($request);

        // Sur erreur
        $this->getErrorForm(null,$form,$request,$data,"update");
        
        // Sur validation
        if ($form->get('submit')->isClicked() && $form->isValid()) {  
            $data = $form->getData();  
            $em->persist($data);
            $em->flush();

            // Retour à la liste
            if($access=="user")
                return $this->redirect($this->generateUrl('app_'.$access.'_scrum_view',["id"=>$data->getId()]));          
            else 
                return $this->redirect($this->generateUrl('app_admin_scrum'));     

        }
        
        // Affichage du formulaire
        return $this->render($this->render.'edit.html.twig', [
            'useheader'         => true,
            'usesidebar'        => ($access=="admin"),  
            'maxwidth'          => ($access=="user"),
            $this->data         => $data,
            'mode'              => 'update',
            'form'              => $form->createView(),
            'access'            => $access
        ]);         
    }

    public function delete($id, Request $request, $access="user") {
        $em = $this->getDoctrine()->getManager();
        $data=$em->getRepository($this->entity)->find($id);
        if (!$data) throw $this->createNotFoundException('Unable to find entity.');
        if($data->getUser()!=$this->getUser()&&!$this->getUser()->hasRole("ROLE_ADMIN")&&!$this->getUser()->hasRole("ROLE_MODO")) 
        throw $this->createNotFoundException('Permission denied');

        $em->remove($data);
        $em->flush();

        if($access=="user")
            return $this->redirect($this->generateUrl('app_home'));          
        else
            return $this->redirect($this->generateUrl('app_admin_scrum'));          
    }    


    public function export($id, Request $request, $access="user") {
        $em = $this->getDoctrine()->getManager();
        $scrum=$em->getRepository($this->entity)->find($id);
        if (!$scrum) throw $this->createNotFoundException('Unable to find entity.');
        if($scrum->getUser()!=$this->getUser()&&!$this->getUser()->hasRole("ROLE_ADMIN")&&!$this->getUser()->hasRole("ROLE_MODO")) 
        throw $this->createNotFoundException('Permission denied');

        $fs = new Filesystem();
        $rootdir = $this->appKernel->getProjectDir();
        $destdir = $rootdir."/uploads/export/scrum/$id";

        // Regénération du répertoire d'export
        $fs->remove($destdir);
        $fs->mkdir($rootdir."/uploads");
        $fs->mkdir($rootdir."/uploads/export");
        $fs->mkdir($rootdir."/uploads/export/scrum");
        $fs->mkdir($rootdir."/uploads/export/scrum/$id");
        $fs->mkdir($rootdir."/uploads/export/scrum/$id/public");
        $fs->mkdir($rootdir."/uploads/export/scrum/$id/public/scrum");
        $fs->mkdir($rootdir."/uploads/export/scrum/$id/public/document");
        $fs->mkdir($rootdir."/uploads/export/scrum/$id/private");
        $fs->mkdir($rootdir."/uploads/export/scrum/$id/private/scrum");
        $fs->mkdir($rootdir."/uploads/export/scrum/$id/private/document");

        // Création du json d'export
        $export = new \stdClass();
        $export->type = "scrum";
        $export->id = $scrum->getId();
        $export->title = $scrum->getTitle();
        $export->fgcomment = $scrum->getFgcomment();
        $export->user = $scrum->getUser()->getId();

        $export->groups = [];
        $groups=$scrum->getGroups();
        foreach($groups as $group) {
            $idgrp=$group->getId();
            $export->groups[$idgrp] = new \stdClass();
            $export->groups[$idgrp]->id=$idgrp;
        }
        
        $export->groupwriters = [];
        $groups=$scrum->getGroupwriters();
        foreach($groups as $group) {
            $idgrp=$group->getId();
            $export->groupwriters[$idgrp] = new \stdClass();
            $export->groupwriters[$idgrp]->id=$idgrp;
        }            
        
        $export->groupreaders = [];
        $groups=$scrum->getGroupreaders();
        foreach($groups as $group) {
            $idgrp=$group->getId();
            $export->groupreaders[$idgrp] = new \stdClass();
            $export->groupreaders[$idgrp]->id=$idgrp;
        } 

        $export->users = [];
        $users=$scrum->getUsers();
        foreach($users as $user) {
            $idusr=$user->getId();
            $export->users[$idusr] = new \stdClass();
            $export->users[$idusr]->id=$idusr;
        }
        
        $export->userwriters = [];
        $users=$scrum->getUserwriters();
        foreach($users as $user) {
            $idusr=$user->getId();
            $export->userwriters[$idusr] = new \stdClass();
            $export->userwriters[$idusr]->id=$idusr;
        }            
        
        $export->userreaders = [];
        $users=$scrum->getUserreaders();
        foreach($users as $user) {
            $idusr=$user->getId();
            $export->userreaders[$idusr] = new \stdClass();
            $export->userreaders[$idusr]->id=$idusr;
        } 

        $export->columns = [];
        $scrumcolumns = $scrum->getScrumcolumns();
        foreach($scrumcolumns as $scrumcolumn) {
            $idcol=$scrumcolumn->getId();
            $export->columns[$idcol] = new \stdClass();
            $export->columns[$idcol]->id=$idcol;
            $export->columns[$idcol]->title=$scrumcolumn->getTitle();
            $export->columns[$idcol]->roworder=$scrumcolumn->getRoworder();
            $export->columns[$idcol]->size=$scrumcolumn->getSize();

            $export->columns[$idcol]->widgets = [];
            $scrumwidgets = $scrumcolumn->getScrumwidgets();
            foreach($scrumwidgets as $scrumwidget) {
                $idwid=$scrumwidget->getId();
                $export->columns[$idcol]->widgets[$idwid] = new \stdClass();
                $export->columns[$idcol]->widgets[$idwid]->id=$idwid;
                $export->columns[$idcol]->widgets[$idwid]->title=$scrumwidget->getTitle();
                $export->columns[$idcol]->widgets[$idwid]->description=$scrumwidget->getDescription();
                $export->columns[$idcol]->widgets[$idwid]->filetype=$scrumwidget->getFiletype();
                $export->columns[$idcol]->widgets[$idwid]->file=$scrumwidget->getFile();
                $export->columns[$idcol]->widgets[$idwid]->roworder=$scrumwidget->getRoworder();
                $export->columns[$idcol]->widgets[$idwid]->user=($scrumwidget->getUser()?$scrumwidget->getUser()->getId():$scrum->getUser()->getId());
                
                $export->columns[$idcol]->widgets[$idwid]->documents = [];
                $documents=$scrumwidget->getScrumwidgetdocuments();
                foreach($documents as $document) {
                    $iddoc=$document->getId();
                    $export->columns[$idcol]->widgets[$idwid]->documents[$iddoc] = new \stdClass();
                    $export->columns[$idcol]->widgets[$idwid]->documents[$iddoc]->id=$iddoc;
                    $export->columns[$idcol]->widgets[$idwid]->documents[$iddoc]->name=$document->getName();
                    $export->columns[$idcol]->widgets[$idwid]->documents[$iddoc]->filename=$document->getFilename();
                    $export->columns[$idcol]->widgets[$idwid]->documents[$iddoc]->description=$document->getDescription();
                    $export->columns[$idcol]->widgets[$idwid]->documents[$iddoc]->entity=$document->getEntity();
                    $export->columns[$idcol]->widgets[$idwid]->documents[$iddoc]->roworder=$document->getRoworder();
                    $export->columns[$idcol]->widgets[$idwid]->documents[$iddoc]->extention=$document->getExtention();
                    $export->columns[$idcol]->widgets[$idwid]->documents[$iddoc]->minetype=$document->getMinetype();
                    $export->columns[$idcol]->widgets[$idwid]->documents[$iddoc]->havethumb=$document->getHavethumb();
                }

                $export->columns[$idcol]->widgets[$idwid]->messages = [];
                $messages=$scrumwidget->getMessages();
                foreach($messages as $message) {
                    $idmsg=$message->getId();
                    $export->columns[$idcol]->widgets[$idwid]->messages[$idmsg] = new \stdClass();
                    $export->columns[$idcol]->widgets[$idwid]->messages[$idmsg]->id=$idmsg;
                    $export->columns[$idcol]->widgets[$idwid]->messages[$idmsg]->message=$message->getMessage();
                    $export->columns[$idcol]->widgets[$idwid]->messages[$idmsg]->submitdate=$message->getSubmitdate()->format("Y-m-d H:i");
                    $export->columns[$idcol]->widgets[$idwid]->messages[$idmsg]->user=$message->getUser()->getId();
                }

                // Copier les documents
                if($fs->exists($rootdir."/uploads/document/scrumwidget/$idwid")) $fs->mirror($rootdir."/uploads/document/scrumwidget/$idwid", $destdir."/private/document/$idwid");
            }
        }

        // Copier les fichiers  public/privé
        if($fs->exists($rootdir."/public/uploads/scrum/$id")) $fs->mirror($rootdir."/public/uploads/scrum/$id", $destdir."/public/scrum/$id");
        if($fs->exists($rootdir."/uploads/scrum/$id")) $fs->mirror($rootdir."/uploads/scrum/$id", $destdir."/private/scrum/$id");

        
        file_put_contents($destdir."/info.json",json_encode($export));
        $now=new \Datetime();
        $zipName=$rootdir."/uploads/export/Scrum-".$now->format("Ymd")."-$id.zip";
        $this->tool->zip($destdir,$zipName);

        $response = new BinaryFileResponse($zipName);
        $response->setContentDisposition(ResponseHeaderBag::DISPOSITION_INLINE);
        return $response;
    }


    public function import(Request $request, $access="user") {
        return $this->render($this->render.'import.html.twig',[
            'useheader'     => true,
            'usemenu'       => false,
            'usesidebar'    => ($access=="admin"),
            'access'        => $access
        ]);
    }

    public function importzip($access="user")
    {
        return $this->render($this->render.'importzip.html.twig',[
            'useheader'     => false,
            'usemenu'       => false,
            'usesidebar'    => false,
            'access'        => $access
        ]);
    }
    
    public function importexec(Request $request,$access="user")
    {
        if (!$request->isXmlHttpRequest()) return new JsonResponse(array('message' => 'Interdit'), 400);

        $em = $this->getDoctrine()->getManager();
        $metadata =  $em->getClassMetaData($this->entity);
        $metadata->setIdGeneratorType(ClassMetadata::GENERATOR_TYPE_NONE);
        $metadata->setIdGenerator(new AssignedGenerator());

        $metadata =  $em->getClassMetaData("App:Scrumcolumn");
        $metadata->setIdGeneratorType(ClassMetadata::GENERATOR_TYPE_NONE);
        $metadata->setIdGenerator(new AssignedGenerator());
        
        $metadata =  $em->getClassMetaData("App:Scrumwidget");
        $metadata->setIdGeneratorType(ClassMetadata::GENERATOR_TYPE_NONE);
        $metadata->setIdGenerator(new AssignedGenerator());
        
        $metadata =  $em->getClassMetaData("App:Document");
        $metadata->setIdGeneratorType(ClassMetadata::GENERATOR_TYPE_NONE);
        $metadata->setIdGenerator(new AssignedGenerator());

        $output=array();

        // Décompresser le fichier zip et controler son format
        $rootdir = $this->appKernel->getProjectDir();
        $zip = $rootdir."/".$request->request->get('file');
        $tmp=explode("/",$zip);
        $tmp=explode(".",end($tmp));
        $id=$tmp[0];
        $destdir = $rootdir."/uploads/import/$id";
        $unzipped = $this->tool->unzip($zip,$destdir);
        $error="";
        if(!$unzipped) 
            return $this->importreturnerror("<p>Votre fichier ZIP n'a pas pu être décompressé<p>");
        else {
            // Chargement du fichier json associé
            $json=file_get_contents($destdir."/info.json");
            if(!$json) return $this->importreturnerror("<p>Contenu du fichier ZIP invalide<p>");;
            $json= json_decode($json);
            if(!$json) return $this->importreturnerror("<p>Contenu du fichier ZIP invalide<p>");;
        }

        // Structure attendu
        $frmboards=["type","id","title","fgcomment","user","groups","groupwriters","groupreaders","users","userwriters","userreaders","columns"];
        $frmcolumns=["id","title","roworder","size","widgets"];
        $frmwidgets=["id","title","description","filetype","file","roworder","user","documents","messages"];
        $frmdocuments=["id","name","filename","description","entity","roworder","extention","minetype","havethumb"];
        $frmmessages=["id","message","submitdate","user"];

        // On vérifie la structure du json
        foreach($frmboards as $frm) {
            if (!property_exists($json,$frm)) $error.="<p>Contenu du fichier ZIP invalide = propriété $frm manquante<p>";
        }
        if($error) return $this->importreturnerror($error);
        if($json->type!="scrum") return $this->importreturnerror("<p>Contenu du fichier ZIP invalide = type invalide<p>");

        // On s'assure que les columns sont correctement formée
        foreach($json->columns as $column) {
            foreach($frmcolumns as $frm) {
                if (!property_exists($column,$frm)) $error.="<p>Contenu du fichier ZIP invalide = propriété colonne $frm manquante<p>";
            }
            if($error) return $this->importreturnerror($error);

            foreach($column->widgets as $widget) {
                foreach($frmwidgets as $frm) {
                    if (!property_exists($widget,$frm)) $error.="<p>Contenu du fichier ZIP invalide = propriété billet $frm manquante<p>";
                }

                foreach($widget->documents as $document) {
                    foreach($frmdocuments as $frm) {
                        if (!property_exists($document,$frm)) $error.="<p>Contenu du fichier ZIP invalide = propriété document $frm manquante<p>";
                    }
                }

                foreach($widget->messages as $message) {
                    foreach($frmmessages as $frm) {
                        if (!property_exists($message,$frm)) $error.="<p>Contenu du fichier ZIP invalide = propriété message $frm manquante<p>";
                    }
                }            
            }
        }
        if($error) return $this->importreturnerror($error);      

        // On s'assure que le board à restaurer existe
        $board=$em->getRepository($this->entity)->find($json->id);
        if(!$board) return $this->importreturnerror("<p>Impossible de restaurer le board n'existe pas</p>");      

        // On s'assure que l'utilisateur à la permission de restaurer ce board
        if($access=="user") {
            if($board->getUser()!=$this->getUser()) return $this->importreturnerror("<p>N'étant pas propriétaire du board vous ne pouvait pas le restaurer</p>");
        }
        $owner=$board->getUser();

        if($error!="") {
            $output["status"]="KO";
            $output["error"]=$error;
        }
        else {
            // On supprime le board pour le regénrer
            $em->remove($board);
            $em->flush();
               
            // Génération de la sauvegarde
            $board = new Entity();
            $board->setId($json->id);
            $board->setTitle($json->title);
            $board->setFgComment($json->title);
            $board->setUser($owner);

            foreach($json->groups as $widgrp) {
                $group=$em->getRepository("App:Group")->find($widgrp->id);
                if($group) $board->addGroup($group);
            }
            foreach($json->groupwriters as $widgrp) {
                $group=$em->getRepository("App:Group")->find($widgrp->id);
                if($group) $board->addGroupwriter($group);
            }
            foreach($json->groupreaders as $widgrp) {
                $group=$em->getRepository("App:Group")->find($widgrp->id);
                if($group) $board->addGroupreader($group);
            }
            
            foreach($json->users as $widusr) {
                $user=$em->getRepository("App:User")->find($widusr->id);
                if($user) $board->addUser($user);
            }
            foreach($json->userwriters as $widusr) {
                $user=$em->getRepository("App:User")->find($widusr->id);
                if($user) $board->addUserwriter($user);
            }
            foreach($json->userreaders as $widusr) {
                $user=$em->getRepository("App:User")->find($widusr->id);
                if($user) $board->addUserreader($user);
            }

            $em->persist($board);
            $em->flush();

            foreach($json->columns as $coljson) {
                $column = new Scrumcolumn();
                $column->setId($coljson->id);
                $column->setTitle($coljson->title);
                $column->setRoworder($coljson->roworder);
                $column->setSize($coljson->size);
                $column->setScrum($board);

                $em->persist($column);
                $em->flush();

                foreach($coljson->widgets as $widjson) {
                    $user=$em->getRepository("App:User")->find($widjson->user);
                    if(!$user) $user=$owner;

                    $widget = new Scrumwidget();
                    $widget->setId($widjson->id);
                    $widget->setTitle($widjson->title);
                    $widget->setDescription($widjson->description);
                    $widget->setFiletype($widjson->filetype);
                    $widget->setFile($widjson->file);
                    $widget->setRoworder($widjson->roworder);
                    $widget->setUser($user);
                    $widget->setScrumcolumn($column);

                    $em->persist($widget);
                    $em->flush();

                    foreach($widjson->documents as $docjson) {                   
                        $document = new Document();
                        $document->setId($docjson->id);
                        $document->setName($docjson->name);
                        $document->setFilename($docjson->filename);
                        $document->setDescription($docjson->description);
                        $document->setEntity($docjson->entity);
                        $document->setRoworder($docjson->roworder);
                        $document->setExtention($docjson->extention);
                        $document->setMinetype($docjson->minetype);
                        $document->setHavethumb($docjson->havethumb);
                        $document->setScrumwidget($widget);
        
                        $em->persist($document);
                        $em->flush();                    
                    }

                    foreach($widjson->messages as $msgjson) {
                        $user=$em->getRepository("App:User")->find($msgjson->user);
                        if(!$user) $user=$owner;

                        $message = new Message();
                        $message->setId($msgjson->id);
                        $message->setMessage($msgjson->message);
                        $message->setSubmitdate(new \DateTime($msgjson->submitdate));
                        $message->setUser($user);
                        $message->setScrumwidget($widget);
        
                        $em->persist($message);
                        $em->flush();                    
                    }                
                }
            }
            
            // On clone les fichiers
            $fs = new Filesystem();
            $finder = new Finder();
            if($fs->exists($destdir."/private/document")) {
                $finder->in($destdir."/private/document")->directories()->depth('== 0');
                foreach ($finder as $dir) {
                    $dest=str_replace("/private/document/","public/uploads/document/scrumwidget/",$dir);
                    $fs->mirror($dir->getPathname(), $rootdir."/uploads/document/scrumwidget/".$dir->getBasename());
                }
            }
            if($fs->exists($destdir."/public/scrum/".$json->id)) $fs->mirror($destdir."/public/scrum/".$json->id, $rootdir."/public/uploads/scrum/".$json->id);        
        }

        $output["status"]="OK";
        $output["id"]=$json->id;
        $response = new Response(json_encode($output));    
        $response->headers->set('Content-Type', 'application/json');        
        return $response;   
    }

    private function importreturnerror($error) {
        $output["status"]="KO";
        $output["error"]=$error;

        $response = new Response(json_encode($output));    
        $response->headers->set('Content-Type', 'application/json');        
        return $response;        
    }

    private function perm($entity) {
        $perm=false;

        if($entity->getUser()==$this->getUser()) $perm="write";
        if($this->getUser()->hasRole("ROLE_ADMIN")) $perm="write";
        if($this->getUser()->hasRole("ROLE_MODO")) $perm="write";

        // Récupération des groupes de l'utilisateur
        $groups=$this->getUser()->getGroups();

        // Ses groupes ont-ils la permission 
        foreach($groups as $group) {
            if($entity->getGroups()->contains($group)) $perm="write";
        }

        // Son compte a-til la permission
        if($entity->getUsers()->contains($this->getUser())) $perm="write";

        // Si pas de permission on regarde s'il a les permissions en écriture limitée
        if(!$perm) {
            // Ses groupes ont-ils la permission 
            foreach($groups as $group) {
                if($entity->getGroupwriters()->contains($group)) $perm="writeuser";
            }

            // Son compte a-til la permission
            if($entity->getUserwriters()->contains($this->getUser())) $perm="writeuser";
        }

        // Si pas de permission on regarde s'il a les permissions en lecture
        if(!$perm) {
            // Ses groupes ont-ils la permission 
            foreach($groups as $group) {
                if($entity->getGroupreaders()->contains($group)) $perm="read";
            }

            // Son compte a-til la permission
            if($entity->getUserreaders()->contains($this->getUser())) $perm="read";
        }

        // Erreur si non permis
        if(!$perm) return false;

        return $perm;
    }

    protected function getErrorForm($id,$form,$request,$data,$mode) {
        if ($form->get('submit')->isClicked()&&$mode=="delete") {
        }

        if ($form->get('submit')->isClicked() && $mode=="submit") {
        }

        if ($form->get('submit')->isClicked() && ($mode=="submit" || $mode=="update")) {
        }

        if ($form->get('submit')->isClicked() && !$form->isValid()) {
            $this->get('session')->getFlashBag()->clear();

            $errors = $form->getErrors();
            foreach( $errors as $error ) {
                $request->getSession()->getFlashBag()->add("error", $error->getMessage());
            }
        }
    }      
}
