<?php
/**
 * PHP 5
 *
 * Gère les actions relatives à un sondage
 */
class SondagesController extends Controller {

	public $modelName = "Sondage";

	/**
	 * Page d'accueil
	 * Se charge de récupérer tous les sondages de l'utilisateur connecté
	 * Action cassifiée
	 * @return void
	 */
	public function index() {
		$user = $this->Session->read('User');
		$sondages = $this->Sondage->get_sondage_user($user['email']);
		$this->loadModel('Destinataire');
		$participes = $this->Destinataire->find(array(
			'fields' => array("Destinataire.*", "sondages.*"),
			'join' => array('sondages' => 'sondages.token = Destinataire.sondage_id'),
			'conditions' => array('mail' => $user['email']),
			'order' => 'sondages.date_end ASC'
			));

		foreach($participes as $k => $sond) {
			if ($sond->mail_admin == $user['email'])
				unset($participes[$k]);
		}

		$this->set(compact('sondages', 'participes', 'user'));
	}

	/**
	 * Page de création d'un sondage (etape 1)
	 * Action cassifiée
	 * @return void
	 */
	public function create() {
		$user = $this->Session->read('User');

		// Si des données sont postées
		//
		if ($this->request->data) {
			$postData = $this->request->data;

			if ($postData->titre != "") {
				$postData->mail_admin = $user['email'];
				$postData->nom_admin  = $user['fullname'];
				$postData->token = $this->Sondage->genere_ID();
				$date_end = new DateTime(date('Y-m-d'));
				$date_end->add(new DateInterval('P60D'));
				$postData->date_end = $date_end->format('Y-m-d');
				$postData->created = $postData->modified = date('Y-m-d H:i:s');
				$postData->published = 0;
				$postData->step = 1;

				$this->Sondage->save($postData);

				$sondage = array('sondages' => $postData);
				$save = array_merge($s_sondage, $sondage);
				$this->Session->write('sondages', $sondage);

				$this->redirect('/options/'.$postData->token);
				return;
			} else {
				$this->Session->setFlash("Veuillez préciser un titre pour votre sondage. Exemple: Quel jour pour la réunion?", "error");
			}
		}

		// Préparation de la session à l'enregistrement du nouveau sondage
		//
		if (!$this->Session->read('sondages')) {
			$this->Session->write('sondages', array());
		}

		$s_sondages = $this->Session->read('sondages');
		$title_for_layout = "Création d'un sondage";
		$page = "create";

		$this->set(compact(
			'title_for_layout',
			'page',
			's_sondages'
			));
	}

	/**
	 * Permet de définir les options d'un sondage (heures, questions,... - Etape 2)
	 * Action cassifiée
	 *
	 * @param  string $token Token du sondage
	 * @return void   Affiche la vue correspondant au type de sondage
	 */
	public function options( $token ) {
		//$sondage = $this->Session->read('sondages');
		$sondage = $this->Sondage->findFirst(array('conditions' => array('token' => $token)));

		if (isset($_SESSION['sondages']['options'])) {
			$_SESSION['sondages']['options'] = array();
		}

		if (empty($sondage)) {
			$this->error('404', 404, "Impossible de trouver cette page.");
		}

		if ($sondage->mail_admin != $_SESSION['User']['email']) {
			$this->error('403', 403, "Vous n'êtes pas autorisé à accéder à cette page.");
		}

		// Si des données sont envoyées
		//
		if ($this->request->data) {
			$postData = $this->request->data;
			if ($sondage['sondages']->type == "D+" AND !$this->Sondage->cleanHours($postData)){
				$this->Session->setFlash("Une des heures renseignées ne respecte pas au moins un des formats autorisé.", 'error');
				$this->set(compact(
					'postData'
					));
			}
			else {
				// Les données sont traitées, on les enregistres
				// dans la session
				//
				if ($sondage['sondages']->type != "D+") {
					$this->Sondage->cleanText($postData);
				}
				unset($sondage['sujets']);
				$sondage['sujets'] = array();
				foreach ($postData as $k => $v) {
					foreach ($v as $i => $sujet) {
						if ($this->Sondage->cleanText($sujet)) {
							$data = array(
								'sondage_id' => $token,
								'value' => $sujet
								);
							array_push($sondage['sujets'], $data);
						}
					}
				}
				$this->Session->write('sondages', $sondage);
				$this->redirect('/destinataires/'.$token);
				return;
			}
		}

		if ($this->Session->read('sondages')) {
			$s_sondage = $this->Session->read('sondages');
			if ($s_sondage['sondages']->token == $token) {
				$title_for_layout = "Options | Création d'un sondage";
				$page = "create";
				$type = $s_sondage['sondages']->type;
				$options = array();
				$this->loadModel('Sujet');


				if ($type == 'D+') {
					$this->loadLib('Calendar');
					$calendar = new Calendar();
					$this->set(compact('calendar'));
				} else {
					$options = $this->Sujet->find(array('conditions' => array('sondage_id' => $token)));
				}

				$this->set(compact(
					'title_for_layout',
					'page',
					's_sondage',
					'sondage',
					'options',
					'type',
					'token'
					));
			} else {
				return $this->redirect('/options/'.$s_sondages['sondages']->token);
			}
		} else {
			$this->error("403", 403, "Impossible d'ajouter des destinataires pour ce sondage.");
		}
	}

	/**
	 * Permet de définir les destinataires du sondage
	 * La limite de destinataire est déifnie dans le fichier de configuration (config/conf.php)
	 * Action cassifiée
	 * @return void
	 */
	public function destinataires( $id_sondage ) {
		$user = $this->Session->read('User');
		if (!isset($s_sondage)) {
			$sondage = $this->Sondage->findFirst(array('conditions' => array('token' => $id_sondage)));
			$sondage->step = 3;
			$this->Sondage->save($sondage);
			$this->Session->write('sondages', $sondage);
			$s_sondage['sondages'] = $this->Session->read('sondages');
			$this->loadModel('Destinataire');
			$s_sondage['sondages']->dests = $this->Destinataire->find(array('conditions' => array('sondage_id' => $id_sondage)));
		}


		if (empty($sondage)) {
			$this->error("404", 404, "Impossible de trouver cette page.");
		}

		if ($sondage->mail_admin != $_SESSION['User']['email']) {
			$this->error("403", 403, "Vous ne disposez pas des droits nécessaires pour accéder à cette page.");
		}

		$this->layout = "default";
		$page = "create";
		$this->set(compact('sondage', 'page', 's_sondage'));
	}

	/**
	 * Permet de publier un sondage
	 * @param  string $id_sondage Token du sondage
	 * @return void
	 */
	public function publication($id_sondage){
		$this->autoRender = false;

		$sondage = $this->Sondage->findFirst(array('conditions' => array('token' => $id_sondage)));

		if (empty($sondage)) {
			$this->error("404", 404, "Impossible de trouver cette page.");
		}

		if ($sondage->mail_admin != $_SESSION['User']['email']) {
			$this->error("403", 403, "Vous ne disposez pas des droits nécessaires pour accéder à cette page.");
		}

		$sondage->published = 1;

		$this->Sondage->save($sondage);

		// TODO envoi de mail
		// Envoi du mail d'invitation
		$this->loadModel('Destinataire');
		$dests = $this->Destinataire->find(array('conditions' => array('sondage_id' => $sondage->token)));

		if (!empty($dests)) {
			foreach ($dests as $k => $dest) {

				list(, $extension) = explode('@', $dest->mail);

				if (in_array($extension, Conf::$mail_to_CAS)) {
					$link_to_polls = HOST.URL.'/sondage/resp/'.$id_sondage.'/'.md5($extension.$sondage->token);
				} else {
					$link_to_polls = HOST.URL.'/sondage/resp/'.$id_sondage;
				}

				// bundary
				$boundary = "-----=".md5(rand());

				// headers
				$headers  = 'MIME-Version: 1.0' . "\r\n";
				$headers .= 'Content-type: text/html; charset=utf8' . "\r\n";
				$headers .= 'From: '.Conf::$nameApp.' <'.Conf::$noreply_mail.'>' . "\r\n";

				// sujet
				$sujet = "[".Conf::$nameApp."] ".$sondage->titre;

				// Récupération du template pour l'envoi de mail
				ob_start();
				require ROOT . DS . 'view' . DS . 'mails' . DS . 'invitation.php';
				$body = ob_get_clean();

				mail($dest->mail,$sujet, $body, $headers);
				$this->Session->setFlash("Votre sondage a bien été publié.");
			}
		} else {

		}




		return $this->redirect('/');
	}

	/**
	 * Permet d'enregistrer un sondage en base de donnée
	 * @return void
	 */
	protected function _save() {
		$this->autoRender = false;
		$user = $this->Session->read('User');
		$sondage = $this->Session->read('sondages');

		// Sauvegarde des informations du sondage
		//
		if (isset($sondage['sondages'])) {
			$this->Sondage->save($sondage['sondages']);
		}

		// Sauvegarde du sujet du sondage
		//
		if (isset($sondage['sujets'])) {
			$this->loadModel('Sujet');
			foreach ($sondage['sujets'] as $k => $v) {
				$this->Sujet->save($v);
			}
		}

		// Sauvegarde des destinataires & envoi de mail
		//
		if (isset($sondage['destinataires'])) {
			$this->loadModel('Destinataire');

			foreach ($sondage['destinataires'] as $k => $v) {
				list(, $extension) = explode('@', $v);

				if (in_array($extension, Conf::$mail_to_CAS)) {
					$link_to_polls = HOST.URL.'/sondage/resp/'.$sondage['sondages']->token.'/'.md5($extension.$sondage['sondages']->token);
				} else {
					$link_to_polls = HOST.URL.'/sondage/resp/'.$sondage['sondages']->token;
				}
				$data = array(
					'sondage_id' => $sondage['sondages']->token,
					'mail' => $v
					);
				$this->Destinataire->save($data);

				// bundary
				$boundary = "-----=".md5(rand());

				// headers
				$headers  = 'MIME-Version: 1.0' . "\r\n";
				$headers .= 'Content-type: text/html; charset=utf8' . "\r\n";
				$headers .= 'From: '.Conf::$nameAppMail.' <'.Conf::$noreply_mail.'>' . "\r\n";

				// sujet
				$sujet = "[".Conf::$nameApp."] ".$sondage['sondages']->titre;

				// Récupération du template pour l'envoi de mail
				ob_start();
				require ROOT . DS . 'view' . DS . 'mails' . DS . 'invitation.php';
				$body = ob_get_clean();

				mail($v,$sujet, $body, $headers);
			}

			// On ajoute le créateur du sondage à la liste des destinataires, sans lui envoyer de mail.
			//
			$data = array(
				'sondage_id' => $sondage['sondages']->token,
				'mail' => $user['email']
				);
			$this->Destinataire->save($data);
		}
	}

	/**
	 * Permet de voir un sondage et d'y répondre
	 * Action NON cassifiée mais demande d'authentification par adresse email, afin de
	 * vérifier que l'utilisateur demandant l'accès au sondage fait bien partie des invités
	 * @param  string $id_sondage ID du sondage à afficher
	 * @return void
	 */
	public function view( $id_sondage, $cas = false ) {
		$user = $this->Session->read('User');
		$page = "reponse";
		$isAuthor = false;
		$user = array();
		
		if (isset($_SESSION['phpCAS']['attributes']['utilisateur'])) {
			if(!isset($_SESSION['phpCAS']['attributes']['utilisateur'])&&isset($_SESSION['phpCAS']['attributes']['attributes']))
			$_SESSION['phpCAS']['attributes']['utilisateur']=$_SESSION['phpCAS']['attributes']['attributes'];
			
			if (isset($_SESSION['phpCAS']['attributes']['utilisateur']['email'][0])) {
				$user['email'] = $_SESSION['phpCAS']['attributes']['utilisateur']['email'][0];
			}
			if (isset($_SESSION['phpCAS']['attributes']['utilisateur']['cn'][0])) {
				$user['fullname'] = $_SESSION['phpCAS']['attributes']['utilisateur']['cn'][0];
			}
			if (empty($user)) {
				die('error');
			}
			
			$this->Session->write('User', $user);
		}
		
		if ($cas !== false) {
			$token_match = md5(unserialize(MAIL_TO_CAS).$id_sondage);
			if ($cas != $token_match) {
				return $this->redirect('/sondage/resp/'.$id_sondage.'/'.$token_match);
			}

			require_once ROOT. DS . 'lib' . DS . 'CAS' . DS . 'auth.inc.php';
			cas_auth();
			if(!isset($_SESSION['phpCAS']['attributes']['utilisateur'])&&isset($_SESSION['phpCAS']['attributes']['attributes']))
			$_SESSION['phpCAS']['attributes']['utilisateur']=$_SESSION['phpCAS']['attributes']['attributes'];
			
			if (isset($_SESSION['phpCAS']['attributes']['utilisateur']['email'][0])) {
				$user['email'] = $_SESSION['phpCAS']['attributes']['utilisateur']['email'][0];
			}
			if (isset($_SESSION['phpCAS']['attributes']['utilisateur']['cn'][0])) {
				$user['fullname'] = $_SESSION['phpCAS']['attributes']['utilisateur']['cn'][0];
			}
			if (empty($user)) {
				die('error');
			}
			
			$this->Session->write('User', $user);
		}
		
		$sondage = $this->Sondage->findFirst(array(
			'conditions' => array(
				'token' => $id_sondage
				)
			));

		if (empty($sondage)) {
			$this->e404("Ce sondage est inexistant ou a été supprimé [VIEW].");
		}

		// Détermine si l'utilisateur qui affiche la page est l'auteur du sondage ou non.
		//
		if (!empty($user['email']))
			if ($sondage->mail_admin == $user['email'])
				$isAuthor = true;

		if (!$sondage->published AND !$isAuthor) {
			$this->error("403", 403,"Vous ne possedez pas les droits nécessaires pour accéder à ce sondage.");
		}

		$this->loadModel('Sujet');
		$sujets = $this->Sujet->find(array(
			'conditions' => array(
				'sondage_id' => $sondage->token
				),
			'order' => 'id ASC	'
			));

		$this->loadModel('Reponse');
		$reponses = $this->Reponse->find(array(
			'fields' => array('Reponse.*', 'destinataires.mail', 'sujets.value as sujet'),
			'join'   => array(
				'destinataires' => 'destinataires.id = Reponse.destinataires_id',
				'sujets' => 'sujets.id = Reponse.value'
			),
			'conditions' => array(
					'Reponse.sondage_id' => $sondage->token
				),
			'order' => 'Reponse.id ASC'
			));
		// Tri des réponses
		//
		if (!empty($reponses)) {
			$respByAuthors = array();
			foreach ($reponses as $k => $v) {
				if ($v->mail != $user['email']) {
					if (!isset($respByAuthors[$v->mail]))
						$respByAuthors[$v->mail] = array();

					array_push($respByAuthors[$v->mail], array('value'=>$v->value, 'resp' => $v->resp, 'name' => $v->name));

				}
			}
		}
		$user = $this->Session->read('User');

		if (empty($user) || $user['anonymous_failed_auth']) {
			$this->set(compact('page', 'sondage'));
			if ($user['anonymous_failed_auth']) {
				$this->Session->setFlash("Vous ne pouvez pas accéder à ce sondage.", "danger");
			}
			$this->render('basic_auth');
		} else {
			// Récupération des données de l'utilisateur
			//
			$this->loadModel('Destinataire');

			$dest = $this->Destinataire->findFirst(array(
					'conditions' => array(
						'mail' => $user['email'],
						'sondage_id' => $sondage->token
					)
				));

			if (empty($dest)) {
				if ((!isset($_SESSION['phpCAS']) AND isset($_SESSION['User']))) {
					$this->set(compact('page', 'sondage'));
					$this->Session->setFlash("Vous ne pouvez pas accéder à ce sondage.", "danger");
					return $this->render('basic_auth');
				}
			}

			if (empty($dest) AND $isAuthor) {
				$data = array(
					'mail' => $user['email'],
					'sondage_id' => $sondage->token
					);
				$this->Destinataire->save($data);

				$dest = $this->Destinataire->findFirst(array(
					'conditions' => array(
						'mail' => $user['email'],
						'sondage_id' => $sondage->token
					)
				));
			}

			if (empty($dest)) {
				if($sondage->public) {
					$data = array(
						'mail' => $user['email'],
						'sondage_id' => $sondage->token
						);
					$this->Destinataire->save($data);
				}
				else {
					$this->set(compact('page', 'sondage'));
					$this->Session->setFlash("Vous ne pouvez pas accéder à ce sondage.", "danger");
					return $this->render('basic_auth');
					//$this->error("403", 403,"Vous ne possedez pas les droits nécessaires pour accéder à ce sondage.");
				}
			}
			
			$dests = $this->Destinataire->find(array(
				'fields' => array('Destinataire.* '),
				'conditions' => array(
					'Destinataire.sondage_id' => $sondage->token
				),
			));


			// Récupération des réponses de l'utilisateur
			//
			if (!empty($dest)) {
				$resps = $this->Reponse->find(array(
				'fields' => array('value', 'resp'),
				'conditions' => array(
						'destinataires_id' => $dest->id,
						'Reponse.sondage_id' => $sondage->token
					),
				'order' => 'Reponse.id ASC'
				));
			}

			//echo Debug::debug($resps); die();
			// Si des données sont postées
			//
			if ($this->request->data) {
				$postData = $this->request->data;
				//echo Debug::debug($postData); die();


					strip_tags($postData->name);
					// On supprime les anciennes réponse
					//
					$this->Reponse->query('DELETE FROM reponses WHERE (sondage_id = "'.$sondage->token.'" AND destinataires_id = "'.$dest->id.'")', false);

					foreach ($postData as $k=>$v) {
							if ($k !== "name") {
								if ($sondage->type == "D+") {
									$data = array(
										'sondage_id' => $sondage->token,
										'destinataires_id' => $dest->id,
										'value' => $k,
										'resp' => $v,
										'name' => $postData->name
			 						);
								} else {
									$data = array(
										'sondage_id' => $sondage->token,
										'destinataires_id' => $dest->id,
										'value' => $k,
										'resp' => $v,
										'name' => $postData->name
			 						);
								}
								$this->Reponse->save($data);
							}
					}

					// On enregistre le fait que l'utlisateur a répondu
					// Si ce n'est pas déjà fait
					if (!$dest->answered){
						$dest->answered = 1;
						$dest->date_answered = date('Y-m-d H:i:s');
						$this->Destinataire->save($dest);
					}

					$register = array(
						$sondage->token => $postData
						);
					//$this->Session->write('reponse', $register);
					$this->Session->setFlash("Réponse enregistrée avec succès!", 'success');

					// Si l'auteur du sondage souhaite recevoir un mail de notification
					//
					if ($sondage->notif_mail) {
						$username = $dest->mail;
						list(, $extension) = explode('@', $sondage->mail_admin);
						if (in_array($extension,Conf::$mail_to_CAS)) {
							$link_to_polls = HOST.URL.'/sondage/resp/'.$sondage->token.'/'.md5($extension.$sondage->token);
						} else {
							$link_to_polls = HOST.URL.'/sondage/resp/'.$sondage->token;
						}
						// Récupération du template pour l'envoi de mail
						ob_start();
						require ROOT . DS . 'view' . DS . 'mails' . DS . 'notification.php';
						$body = ob_get_clean();

						// bundary
						$boundary = "-----=".md5(rand());

						// headers
						$headers  = 'MIME-Version: 1.0' . "\r\n";
						$headers .= 'Content-type: text/html; charset=utf8' . "\r\n";
						$headers .= 'From: '.Conf::$nameApp.' <'.Conf::$noreply_mail.'>' . "\r\n";

						// sujet
						$sujet = "[".Conf::$nameApp." - Nouvelle réponse] ".$sondage->titre;

						mail($sondage->mail_admin,$sujet, $body, $headers);
					}
					$this->redirect('/sondage/resp/'.$sondage->token);
			}

			$this->set(compact('page', 'sondage', 'sujets', 'reponses', 'dest', 'dests', 'resps', 'user', 'isAuthor', 'respByAuthors'));
		}
	}

	/**
	 * Permet de mettre à jour les informations générales d'un sondage.
	 * @param  [type] $sondage_id [description]
	 * @return [type]             [description]
	 */
	public function update( $sondage_id ) {
		$this->autoRender = false;

		if (!$this->request->data) {
			$this->error("400", 400, "Bad Request");
		}

		$sondage = $this->Sondage->findFirst(array(
			'conditions' => array('token' => $sondage_id)
			));

		if (empty($sondage)) {
			$this->error('404', 404, "Impossible de trouver ce sondage.");
		}

		$user = $this->Session->read('User');

		if ($sondage->mail_admin != $user['email']) {
			$this->error('403', 403, "Seul l'auteur du sondage peut accéder à cette page.");
		}

		$postData = $this->request->data;

		if (!empty($postData->titre) AND is_string($postData->titre)) {
			$postData->id = $sondage->id;

			$postData->notif_mail = (!empty($postData->notif_mail) && isset($postData->notif_mail)) ? $postData->notif_mail : '0';
			
			$this->Sondage->save($postData);
			$this->Session->setFlash("Les informations de votre sondage ont bien été mises à jours.", "success");
		}

		unset($_SESSION['active_pan']);
		return $this->redirect('/sondage/edit/' . $sondage_id);
	}

	/**
	 * Permet d'identifier un utilisateur lorsqu'il tente d'accéder à un sondage
	 * @return void
	 */
	public function basicAuth() {
		$this->autoRender = false;
		$user = $this->Session->read('User');
		$postData = $this->request->data;

		
		if (isset($_SESSION['sondageID'])) {
			require_once ROOT. DS . 'lib' . DS . 'CAS' . DS . 'auth.inc.php';
			cas_auth();
			if(!isset($_SESSION['phpCAS']['attributes']['utilisateur']) && isset($_SESSION['phpCAS']['attributes']['attributes'])) {
				$_SESSION['phpCAS']['attributes']['utilisateur']=$_SESSION['phpCAS']['attributes']['attributes'];
			}
			if (isset($_SESSION['phpCAS']['attributes']['utilisateur']['email'][0])) {
				$user['email'] = $_SESSION['phpCAS']['attributes']['utilisateur']['email'][0];
			}
			if (isset($_SESSION['phpCAS']['attributes']['utilisateur']['cn'][0])) {
				$user['fullname'] = $_SESSION['phpCAS']['attributes']['utilisateur']['cn'][0];
			}
			if (empty($user)) {
				die('error');
			}
			$this->Session->write('User', $user);
		}
		
		$user = $this->Session->read('User');
		
		if ($postData || !empty($user)) {
			// Evolution TRANSUM
			// https://forge.cadoles.com/Cadoles/ninegate/issues/97
			// Si verification via SSO, alors on envoi l'utilisateur sur la mire d'authentification
			if (isset($postData->ssoCheck)) {
				if ($postData) {
					$_SESSION['sondageID'] = $postData->sondage;
				}
				require_once ROOT. DS . 'lib' . DS . 'CAS' . DS . 'auth.inc.php';
				cas_auth();
			}
			
			$sondageData = [
				'id' => ($postData ? $postData->sondage : $_SESSION['sondageID']),
				'email' => ($postData ? ($postData->verifmail ? $postData->verifmail : $user['email']) : $user['email']), 
			];

			var_dump($sondageData);
			

			$this->loadModel('Destinataire');
			$exist = $this->Destinataire->findFirst(array(
				'conditions' => array(
					'sondage_id' => $sondageData['id'],
					'mail' => $sondageData['email']
					)
				));
			$sondage = $this->Sondage->findFirst(array('conditions' => array('token' => $sondageData['id'])));
			
			if (!$sondage) {
				var_dump("sondage inconnu");
				$this->error('404', 404, "Impossible de trouver ce sondage.");
				exit();
			}
			
			if ($sondage->public AND empty($exist)) {
				$dest = array('sondage_id' => $sondageData['id'], 'mail' => $sondageData['email']);
				$this->loadModel('Destinataire');
				$this->Destinataire->save($dest);
			}

			if (empty($exist) AND !$sondage->public) {
				$this->Session->setFlash("Cette adresse mail n'est pas autorisée à répondre à ce sondage.", "error");
				$this->Session->write('User', ['anonymous_failed_auth' => true]);
				$this->redirect('/sondage/resp/'.$sondageData['id']);
			} else {
				list(, $extension) = explode('@', $sondageData['email']);
				if (in_array($extension,Conf::$mail_to_CAS) && empty($user)) {
					if ($postData) {
						$_SESSION['sondageID'] = $postData->sondage;
					}
					require_once ROOT. DS . 'lib' . DS . 'CAS' . DS . 'auth.inc.php';
					cas_auth();
					if(!isset($_SESSION['phpCAS']['attributes']['utilisateur']) && isset($_SESSION['phpCAS']['attributes']['attributes'])) {
						$_SESSION['phpCAS']['attributes']['utilisateur']=$_SESSION['phpCAS']['attributes']['attributes'];
					}
					if (isset($_SESSION['phpCAS']['attributes']['utilisateur']['email'][0])) {
						$user['email'] = $_SESSION['phpCAS']['attributes']['utilisateur']['email'][0];
					}
					if (isset($_SESSION['phpCAS']['attributes']['utilisateur']['cn'][0])) {
						$user['fullname'] = $_SESSION['phpCAS']['attributes']['utilisateur']['cn'][0];
					}
					if (empty($user)) {
						die('error');
					}
					$this->Session->write('User', $user);
				} else {
					$this->Session->write('User', array('email' => $sondageData['email']));
				}
				$this->redirect('/sondage/resp/'.$sondageData['id']);
			}
		}
	}


	/**
	 * Permet de supprimer un sondage
	 * Un utilisateur peut supprimer les sondages dont il est l'auteur.
	 * Action cassifiée
	 * @param  string $id_sondage ID du sondage
	 * @return void
	 */
	public function delete( $id_sondage ) {
		$user = $this->Session->read('User');
		$this->autoRender = false;
		// Vérification que le sondage qui doit etre supprime a bien été
		// créer par l'utilisateur qui souhaite le supprimer
		$sondage = $this->Sondage->findFirst(array('conditions' => array('token' => $id_sondage)));
		if ($sondage->mail_admin == $user['email']) {
			$this->loadModel('Destinataire');
			$this->Destinataire->primaryKey = "sondage_id";
			$this->Destinataire->delete($sondage->token);

			$this->loadModel('Sujet');
			$this->Sujet->primaryKey = "sondage_id";
			$this->Sujet->delete($sondage->token);
			$this->loadModel('Reponse');
			$this->Reponse->primaryKey = "sondage_id";
			$this->Reponse->delete($sondage->token);

			$this->Sondage->delete($sondage->id);
			$this->Session->setFlash("Sondage supprimé avec succès.", "success");
			$this->redirect('/');
		} else {
			$this->error("403", 403, "Seul l'auteur d'un sondage peut le supprimer. Veuillez contacter l'auteur de ce sondage.");
		}
	}

	/**
	 * Permet d'enregistrer une date en Session
	 * @param  string $id_sondage ID du sondage concerné
	 * @param  int $timestamp  Timestamp de la date à sauvegarder
	 * @return JSON
	 */
	public function saveDate( $id_sondage, $timestamp, $action, $edit = false ) {
		$allowed_actions = array('add', 'remove');
		$this->autoRender = false;
		$this->layout = "ajax";
		//echo Debug::debug(func_get_args()); die();
		$user = $this->Session->read('User');

		$resp = array();
		$resp['error'] = true;

		if ($this->Session->read('sondages')) {
			$s_sondages = $this->Session->read('sondages');
			if ($s_sondages['sondages']->token == $id_sondage) {

				if ($action == "add"){
					if (!isset($s_sondages['options'])) {
						$options = array();
						if ($timestamp >= strtotime(date('Y-m-d'))) {
							if (!in_array($timestamp, $options))
								array_push($options, $timestamp);

							$s_sondages['options'] = $options;
							$resp['error'] = false;

							if (!$edit) {
								ob_start();
								require ROOT . DS . 'view' . DS . 'elements' . DS . 'optionsdate.php';
								$template = ob_get_clean();
							}

							if ($edit == "edit") {
								ob_start();
								require ROOT . DS . 'view' . DS . 'elements' . DS . 'edit_optionsdate.php';
								$template = ob_get_clean();
							}

							$resp['template'] = $template;
						} else {
							$resp['msg'] = "Veuillez sélectionner une date supérieur ou égale à aujourd'hui.";
						}
					} else {
						if ($timestamp >= strtotime(date('Y-m-d'))) {
							if (!in_array($timestamp, $s_sondages['options']))
								array_push($s_sondages['options'], $timestamp);
							$resp['error'] = false;
							if (!$edit) {
								ob_start();
								require ROOT . DS . 'view' . DS . 'elements' . DS . 'optionsdate.php';
								$template = ob_get_clean();
							}

							if ($edit == "edit") {
								ob_start();
								require ROOT . DS . 'view' . DS . 'elements' . DS . 'edit_optionsdate.php';
								$template = ob_get_clean();
							}
							$resp['template'] = $template;
						} else {
							$resp['msg'] = "Veuillez sélectionner une date supérieur ou égale à aujourd'hui.";
						}
					}
				} else {
					if ($action == "remove") {
						if (array_search($timestamp, $s_sondages['options']) !== false) {
							$key = array_search($timestamp, $s_sondages['options']);
							//unset($s_sondages['options'][$key]);
							$this->loadModel('Reponse');
							$reponses = $this->Reponse->find(array(
								'conditions' => array(
									'sondage_id' => $id_sondage
									)
								));
							if (!empty($reponses)) {
								foreach($reponses as $k => $reponse) {
									list($time,$hours) = explode('@', $reponse->value);
									if ($timestamp == $time) {
									  $this->Reponse->delete($reponse->id);
									}
								}
							}
						}
						$resp['error'] = false;
						$resp['timestamp']  = $timestamp;
					}
				}
				$this->Session->write('sondages', $s_sondages);
			} else {
				$resp['msg'] = "Erreur lors de la sélection. Impossible d'ajouter cette option à ce sondage";
			}
		} else {
			die('sondage introuvable');
		}

		echo json_encode($resp);

	}

	/**
	 * Permet d'annuler la création d'un sondage
	 * @param  string $id_sondage Token du sondage
	 * @return void
	 */
	public function erase($id_sondage) {
		$this->autoRender = false;
		$user = $this->Session->read('User');

		if ($this->Session->read('sondages')) {
			$this->Session->delete('sondages');
		}

		// redirection vers la page d'accueil
		$this->redirect('/');
	}

	/**
	 * Permet d'éditer un sondage
	 * @param  string $id_sondage ID du sondage
	 * @return void
	 */
	public function edit( $id_sondage ) {
		$user = $this->Session->read('User');
		// Vérification du droit d'édition
		//
		$sondage = $this->Sondage->findFirst(array('conditions' => array('token' => $id_sondage)));
		if (empty($sondage) OR $sondage->mail_admin !== $user['email']) {
			$this->error("403", 403, "Vous ne disposez pas des droits nécessaires pour effectuer cette action");
		}

		$s_sondage = array();
		$s_sondage['sondages'] = $sondage;


		// Récupération des options du sondage
		//
		$this->loadModel('Sujet');
		$options = $this->Sujet->find(array(
			'conditions' => array(
				'sondage_id' => $sondage->token
				),
			'order' => 'id ASC'
			));

		// Récupération des destinataires déjà enregistrés
		//
		$this->loadModel('Destinataire');
		$destinataires = $this->Destinataire->find(array(
			'conditions' => array(
				'sondage_id' => $sondage->token
				),
			'order' => 'id DESC'
			));

		$s_sondage['sondages']->dests  = $destinataires;

		$s_sondage['options'] = array();
		if ($sondage->type == "D+") {
			foreach ($options as $k => $opt) {
				list($time, $hour) = explode('@', $opt->value);
				if (!in_array($time, $s_sondage['options']))
					array_push($s_sondage['options'], $time);
			}
		} else {
			foreach ($options as $k => $opt) {
				array_push($s_sondage['options'], $opt);
			}
		}

		$this->Session->write('sondages', $s_sondage);

		// Si des données sont postées
		//
		if ($this->request->data) {
			$postData = $this->request->data;

			$postData->dest = $this->Sondage->cleanDest( $postData->dest );

			if ($sondage->type == "D+") {
				unset($postData->dates);
				unset($postData->month);
				unset($postData->year);

				$error = false;

				foreach ($postData->date as $k => $date) {
					if (!$this->Sondage->cleanHours( $date )) {
						$this->Session->setFlash("Veuillez vérifier le format de vos horaires.", "error");
						$error = true;
					}
				}
			}

			if (!$error) {
				// Préparation à la sauvegarde des informations
				//
				$sond = new stdClass();
				$sond->id = $sondage->id;
				$sond->titre = $postData->titre;
				$sond->commentaire = $postData->commentaire;
				$sond->notif_mail = $postData->notif_mail;

				// Suppression de tous les sujets du sondage déjà enregistré ainsi que les réponses associées
				// Et préparation à l'enregistrement des nouvelles valeurs
				$this->Sondage->query('DELETE FROM sujets WHERE sondage_id = "'.$id_sondage.'"', false);

				$values = array();
				if ($sondage->type == "D+") {
					foreach ($postData->date as $k => $sujet) {
						foreach ($sujet as $l => $hours) {
							foreach( $hours as $a => $v) {
								if (!empty($v)) {
									$s = $l.'@'.$v;
									$data = array('sondage_id' => $id_sondage, 'value' => $s);
									array_push($values, $data);
								}
							}
						}
					}
				} else {

					foreach ($postData->choice as $k => $sujet) {
						array_push($values, $sujet);
					}

				}


				// Envoi d'un mail de notification de modification aux destinataires du sondage
				// déjà enregistré
				$this->loadModel('Destinataire');
				$dests = $this->Destinataire->find(array('conditions' => array('sondage_id' => $id_sondage)));

				// bundary
				$boundary = "-----=".md5(rand());

				// headers
				$headers  = 'MIME-Version: 1.0' . "\r\n";
				$headers .= 'Content-type: text/html; charset=utf8' . "\r\n";
				$headers .= 'From: '.Conf::$nameAppMail.' <'.Conf::$noreply_mail.'>' . "\r\n";

				// sujet modification
				$sujetModif = "[".Conf::$nameApp." - Modification] ".$sondage->titre;

				// sujet invitation
				$sujet = "[".Conf::$nameApp."] ".$sondage->titre;

				foreach ($dests as $k => $v) {
					list(, $extension) = explode('@', $v->mail);
					if (in_array($extension,Conf::$mail_to_CAS)) {
						$link_to_polls = HOST.URL.'/sondage/resp/'.$sondage->token.'/'.md5($extension.$sondage->token);
					} else {
						$link_to_polls = HOST.URL.'/sondage/resp/'.$sondage->token;
					}

					// Récupération du template pour l'envoi de mail (notif de modification)
					ob_start();
					require ROOT . DS . 'view' . DS . 'mails' . DS . 'modification.php';
					$modif = ob_get_clean();

					mail($v->mail,$sujetModif, $modif, $headers);
				}

				// Envoi d'un mail d'invitation à répondre au sondage pour les nouveaux destinataires
				//
				if (!empty($postData->dest)) {
					foreach ($postData->dest as $k => $v) {
						$data = array(
							'sondage_id' => $sondage->token,
							'mail' => $v
							);
						$this->Destinataire->save($data);

						list(, $extension) = explode('@', $v);
						if (in_array($extension,Conf::$mail_to_CAS)) {
							$link_to_polls = HOST.URL.'/sondage/resp/'.$sondage->token.'/'.md5($extension.$sondage->token);
						} else {
							$link_to_polls = HOST.URL.'/sondage/resp/'.$sondage->token;
						}

						// Récupération du template pour l'envoi de mail (notif de modification)
						ob_start();
						require ROOT . DS . 'view' . DS . 'mails' . DS . 'new_invitation.php';
						$body = ob_get_clean();

						mail($v,$sujet, $body, $headers);
					}
				}

				// Enregistrement du sondage en BDD
				$this->Sondage->save($sond);

				// Enregistrement des sujets
				$this->loadModel('Sujet');
				$this->Sujet->primaryKey = 'id';

				if ($sondage->type != "D+") {
					foreach ($values as $k => $val) {
						$data  = array(
							'sondage_id' => $sondage->token,
							'value' => $val
							);
						$this->Sujet->save($data);
					}
				} else {
					foreach ($values as $k => $val) {
						$this->Sujet->save($val);
					}
				}

				// Redirection vers la page d'accueil
				$this->redirect('/');
				return;
			}
		}

		if ($sondage->type == "D+") {
			$this->loadLib('Calendar');
			$calendar = new Calendar();
			$this->set(compact('calendar'));
		}

		$page = "edition";
		$this->set(compact('page', 'sondage', 'options', 's_sondage', 'destinataires'));
	}

	/**
	 * Permet de cloturer un sondage.
	 * La cloture empeche les invités au sondage d'y accéder mais pas à son
	 * auteur
	 * @param  string $id_sondage Token du sondage
	 * @return void
	 */
	public function close( $id_sondage ){
		$this->autoRender = false;

		$sondage = $this->Sondage->findFirst(array('conditions' => array('token' => $id_sondage)));

		if (empty($sondage))
			$this->error('404', 404, "Impossible de trouver ce sondage.");

		if ($sondage->mail_admin != $_SESSION['User']['email']) {
			$this->error('403', 403, "Vous ne disposez pas des droits nécessaires pour effectuer cette action.");
		}

		$sondage->published = -1;

		$this->Sondage->save($sondage);

		$this->Session->setFlash($sondage->titre . " a bien été clôturé.", 'success');

		return $this->redirect('/sondage/edit/' . $id_sondage);
	}

	/**
	 * Permet de validé un format horaire
	 * Requete faite par AJAX lors de la saisie d'un horaire
	 * @return json
	 */
	public function checkhoraire(){
		$this->autoRender = false;
		$resp = array();
		$resp['error'] = true;

		$pattern = "/^((?(?=2)2[0-3]?|[0-1]?\d)[:hH][0-5]\d)(?(?=-)-(?1))$/i";
		if ($this->request->data) {

			$string = $this->request->data->format;

			if (preg_match($pattern, $string)) {
				$resp['error'] = false;
			}
		}

		echo json_encode($resp);
	}

	/**
	 * Permet de déconnecter un utilisateur
	 * @return void
	 */
	public function logout() {
		if ($this->Session->isLogged()) {
			require_once 'CAS-1.3.1/eoleCAS.php';
			require_once 'configCAS/cas.inc.php';
			eolephpCAS::client(CAS_VERSION_2_0, __CAS_SERVER, __CAS_PORT, __CAS_FOLDER, true);
			// Maj pour phpCAS 1.3.1
		    if (__CAS_VALIDER_CA) {
		        EolephpCAS::setCasServerCACert(__CAS_CA_LOCATION); // vérification par rapport à la CA
		    } else {
		        if (method_exists("EolephpCAS", "setNoCasServerValidation")){
		            EolephpCAS::setNoCasServerValidation();
		        }
		    }
		    $orig_url = $_SERVER['SCRIPT_URI'];
		    $_url = explode('/', $orig_url);
		    array_pop($_url);
		    $url  = implode('/', $_url);
		    eolephpCAS::logout(array("service"=> $url));
		}
		return $this->redirect('/');
	}
}
