#-*-coding:utf-8-*-
"""
    gestion des postes et destinations à interdire
    édite les fichiers de configuration en fonction des zones
"""
from datetime import datetime, time, timedelta
from os.path import isfile
from amon.config import VIRTDISABLED, INTERFACE_GW, WEB_PORTS, CONTAINER_IP_PROXY
from amon.backend import get_filter_zones, get_interface, get_dst_offset
from amon.ipset.time_manager import test_time_range
from amon.era.config import f_poste_name
from creole.config import VIRTMASTER
from pyeole.process import system_out
from pyeole.ansiprint import print_red

#IPTABLES_POSTES = """/sbin/iptables -t nat -i %(interface)s -I PREROUTING \
#-s %(poste)s -j REJECT"""
IPTABLES_POSTES = """/sbin/iptables -I FORWARD -i %(interface)s -s %(poste)s \
-j REJECT"""
IPTABLES_POSTES2 = """/sbin/iptables -i %(interface)s -I INPUT -s %(poste)s \
-j REJECT"""

IPTABLES_DEST = """/sbin/iptables -I FORWARD -i %(interface)s -d %(poste)s \
-j REJECT"""
IPTABLES_DEST2 = """/sbin/iptables -I INPUT -i %(interface)s -s %(poste)s -j REJECT"""

if VIRTDISABLED:
    IPTABLES_HORAIRE = """/sbin/iptables -t nat -I PREROUTING -i %(interface)s -m multiport -p tcp --dports %(web_ports)s -s %(poste)s -m time --timestart %(hdeb)s --timestop %(hfin)s %(contiguous)s --weekdays %%s -j REDIRECT --to-ports 83"""
    IPTABLES_HORAIRE2 = """/sbin/iptables -I INPUT -i %(interface)s -s %(poste)s -p tcp --dport 83  -m time --timestart %(hdeb)s --timestop %(hfin)s %(contiguous)s --weekdays %%s -j ACCEPT"""
else:
    IPTABLES_HORAIRE = """/sbin/iptables -t nat -I PREROUTING -i %(interface)s -m multiport -p tcp --dports %(web_ports)s -s %(poste)s -m time --timestart %(hdeb)s --timestop %(hfin)s %(contiguous)s --weekdays %%s -j DNAT  --to-destination 192.0.2.1:83"""
    IPTABLES_HORAIRE2 = """/sbin/iptables -I FORWARD -i %(interface)s -o containers -s %(poste)s -p tcp --dport 83  -m time --timestart %(hdeb)s --timestop %(hfin)s %(contiguous)s --weekdays %%s -j ACCEPT"""
    IPTABLES_HORAIRE3 = """/sbin/iptables -t nat -A POSTROUTING -s %(poste)s -o containers -j SNAT --to-source %(container_ip_proxy)s"""
    IPTABLES_HORAIRE4 = """/sbin/iptables -I INPUT -i br0 -s %(poste)s -p tcp --dport 83  -m time --timestart %(hdeb)s --timestop %(hfin)s %(contiguous)s --weekdays %%s -j ACCEPT"""

CONTAINER_PROXY = 'proxy'
CONTAINER_DOMAINE = 'domaine'

semaine = ['Mon','Tue','Wed','Thu','Fri','Sat','Sun']
semaine_fr = ['lundi','mardi','mercredi','jeudi','vendredi','samedi','dimanche']
day_translate = {'lundi':'Mon', 'mardi':'Tue', 'mercredi':'Wed',
                'jeudi':'Thu', 'vendredi':'Fri', 'samedi':'Sat',
                'dimanche':'Sun'}

def get_jliste(jdeb, jfin):
    """
        renvoie les jours compris entre jdeb et jfin
    """
    if jdeb in day_translate:
        jdeb  = day_translate[jdeb]
        jfin = day_translate[jfin]
    istart, iend = semaine.index(jdeb), semaine.index(jfin)
    if istart < iend:
        jliste = ','.join(semaine[istart: iend+1])
    elif istart == iend:
        jliste = semaine[istart]
    else:
        jliste = ','.join(semaine[istart:])
        jliste += ','
        jliste += ','.join(semaine[:iend+1])
    return jliste

############################
# GESTION DES FICHIERS TEXTE
############################
#PARSER:
def parse_all(line):
    """ parse une ligne du fichier d'interdiction totale """
    line = line.strip()
    if len(line.split('#')) == 2:
        poste, interface = line.split('#')
        return {'interface':interface,
                'poste':poste}
    return None

def parse_web(line):
    """ parse une ligne du fichier d'interdiction web """
    line = line.strip()
    if len(line.split('#')) == 6:
        poste,  hdeb, hfin, jdeb, jfin, interface = line.split('#')
        return dict(poste=poste,
                    hdeb=hdeb,
                    hfin=hfin,
                    jdeb=jdeb,
                    jfin=jfin,
                    interface=interface)
    elif len(line.split('#')) == 5:
        poste,  hdeb, hfin, jdeb, jfin = line.split('#')
        return dict(poste=poste,
                    hdeb=hdeb,
                    hfin=hfin,
                    jdeb=jdeb,
                    jfin=jfin)
    return None

poste_syntax = {'web': "%(poste)s#%(hdeb)s#%(hfin)s#%(jdeb)s#%(jfin)s#%(interface)s",
                'all': "%(poste)s#%(interface)s",
                'destination':"%(poste)s#%(interface)s",
               }
poste_keys = {'web':('poste',  'hdeb', 'hfin', 'jdeb', 'jfin'),
              'all':('poste',),
              'destination':('poste',)}

poste_inverse_syntax = {'all':parse_all,
                        'web':parse_web,
                        'destination':parse_all}

def recalc_poste_time(poste):
    """
       recalcule les horaires donnés par l'EAD en fonction de la timezone du serveur
    """
    if 'hdeb' in poste:
        poste['contiguous'] = ""
        deb = poste['hdeb']
        fin = poste['hfin']
        jdeb = poste['jdeb']
        jfin = poste['jfin']
        tz_offset = timedelta(seconds=get_dst_offset())

        # on applique l'offset en simulant les horaires sur la date courante
        hdeb, mindeb = deb.split(':')
        timedeb = datetime.combine(datetime.now().date(),
                                            time(int(hdeb),int(mindeb)))
        hfin, minfin = fin.split(':')
        timefin = datetime.combine(datetime.now().date(),
                                            time(int(hfin),int(minfin)))
        # utilisé pour vérifier si le décalage entraîne un changement de jour
        # jour suivant et précédent par rapport à la 'fausse' date de début
        next_day = timedeb + timedelta(days=1)
        prev_day = timedeb + timedelta(days=-1)
        cur_day = timedeb.day
        jour_next_deb = semaine_fr[(semaine_fr.index(jdeb)+1) % 7]
        jour_prev_deb = semaine_fr[(semaine_fr.index(jdeb)-1) % 7]
        jour_next_fin = semaine_fr[(semaine_fr.index(jfin)+1) % 7]
        jour_prev_fin = semaine_fr[(semaine_fr.index(jfin)-1) % 7]
        jours_deb = {prev_day.day:jour_prev_deb, next_day.day:jour_next_deb, cur_day:jdeb}
        # applique l'offset
        utc_deb = timedeb + tz_offset
        utc_fin = timefin + tz_offset
        # vérifie les jours de début/fin et détermine les plages à retourner
        poste["jdeb"] = jours_deb[utc_deb.day]
        # applique le même changement de jour sur jdeb et jfin
        if utc_deb.day == prev_day.day:
            poste["jfin"] = jour_prev_fin
        elif utc_deb.day == next_day.day:
            poste["jfin"] = jour_next_fin
        if utc_deb.day != timedeb.day:
            # ajout de l'option --contiguous à la règle iptable
            poste['contiguous'] = "--contiguous "
        poste["hdeb"] = "%d:%d" % (utc_deb.hour, utc_deb.minute)
        poste["hfin"] = "%d:%d" % (utc_fin.hour, utc_fin.minute)
    return poste


def load_forbidden_poste(zone, mode):
    """
        récupère la  liste des poste interdits
        @zone: indice de la configuration
        @mode: mode d'interdiction
    """
    f_name = f_poste_name[mode]%zone
    if isfile(f_name):
        with open(f_name, 'r') as f_stream:
            f_postes = f_stream.read().splitlines()
        return [poste_inverse_syntax[mode](line)
                for line in f_postes]
    else:
        return []

def save_forbidden_poste(zone, mode, postes):
    """
        sauvegarde une liste de postes interdits
        @zone: indice de la configuration
        @mode: mode d'interdiction
        @postes: liste des postes à interdire
            si l'interface n'est pas précisée on tente
            de la récupérer depuis l'ip du poste
            sinon on renvoie le premier eth de la zone
    """
    if type(postes) != list:
        postes = [postes]
    f_name = f_poste_name[mode]%zone
    to_w = ''
    for poste in postes:
        if not all([i in poste for i in poste_keys[mode]]):
            raise Exception("Il manque des données pour les interdictions de poste.")
        if 'interface' not in poste:
            poste['interface'] = get_interface(zone, poste['poste'])
        to_w += '%s\n'%poste_syntax[mode]%poste

    if mode == 'web':
        for poste in postes:
            if not test_time_range(poste['hdeb'], poste['hfin']):
                raise Exception("Erreur: l'heure de début doit précéder l'heure de fin.")

    with open(f_name, 'w') as f_postes:
        f_postes.write(to_w)
    return True


#######################
# GENERATEURS DE REGLES
#######################

def interdit_all_postes(zone, poste, interface=None):
    """
        interdit le poste 'poste' de réseau sur la zone (politique de filtrage)
        d'indice 'zone', on écrit la règle pour l'interface où le poste
        est connecté et on détecte automatiquement l'interface
        dans le cas où elle n'est pas renseignée
    """
    if interface is None and 'interface' not in poste:
        poste['interface'] = get_interface(zone, poste['poste'])
    ret = [(IPTABLES_POSTES%(poste), VIRTMASTER), (IPTABLES_POSTES2%(poste), VIRTMASTER)]
    if not VIRTDISABLED:
        poste['interface'] = 'eth1'
        ret.extend([(IPTABLES_POSTES%(poste), CONTAINER_PROXY), (IPTABLES_POSTES2%(poste), CONTAINER_PROXY), (IPTABLES_POSTES%(poste), CONTAINER_DOMAINE), (IPTABLES_POSTES2%(poste), CONTAINER_DOMAINE)])
    return ret

def interdit_destination(zone, poste):
    """
        interdit le poste 'poste' en tant que destination
        @zone: indice de la configuration à éditer
        @poste: identifiant du poste (sous-réseau)
        on écrit la règle pour chaque interface de la zone
    """
    regles = []
    if 'interface' not in poste:
        for interface in get_interface(zone):
            poste['interface'] = interface
            regles.append((IPTABLES_DEST % (poste), VIRTMASTER))
            regles.append((IPTABLES_DEST2 % (poste), VIRTMASTER))
    else:
        regles.append((IPTABLES_DEST % (poste), VIRTMASTER))
        regles.append((IPTABLES_DEST2 % (poste), VIRTMASTER))
    return regles

def interdit_postes(zone, poste, interface=None):
    """
        interdit un poste de navigation internet selon des horaires*
        @zone: indice de la configuration à éditer
        @poste: de la forme {'poste':'ip du poste',
                             'hdeb': debut de plage,
                             'hfin': fin de plage,
                             'gw': gateway (interface de sortie vers le web),
                             'jdeb': jour de debut
                             'jfin': jour de fin
                             }
    """
    if interface is None and 'interface' not in poste:
        poste['interface'] = get_interface(zone, poste['poste'])
    if not VIRTDISABLED:
        # FIXME : hack pour AmonEcole #19804
        poste['interface'] = 'eth1'
    poste['gw'] = INTERFACE_GW
    poste['web_ports'] = WEB_PORTS
    poste['container_ip_proxy'] = CONTAINER_IP_PROXY
    poste = recalc_poste_time(poste)
    jliste = get_jliste(poste['jdeb'], poste['jfin'])
    ret = [(IPTABLES_HORAIRE%poste%jliste, CONTAINER_PROXY),
            (IPTABLES_HORAIRE2%poste%jliste, CONTAINER_PROXY),
          ]
    if not VIRTDISABLED:
        ret.extend([(IPTABLES_HORAIRE3%poste, CONTAINER_PROXY),
                   (IPTABLES_HORAIRE4%poste%jliste, None),
                   ])
    return ret

def generate_postes_rules():
    """
        génère les règles pour les interdictions de poste
        web
    """
    regles = []
    for zone in get_filter_zones():
        if isfile(f_poste_name['web']%zone):
            for poste in load_forbidden_poste(zone, 'web'):
                regles.extend(interdit_postes(zone, poste))
    return regles

def generate_all_postes_rules():
    """
        génère les règles pour l'interdiction totale de poste
    """
    regles = []
    for zone in get_filter_zones():
        if isfile(f_poste_name['all']%zone):
            for poste in load_forbidden_poste(zone, 'all'):
                regles.extend(interdit_all_postes(zone, poste))
    return regles

def generate_destinations_rules():
    """
        génère les règles pour les interdictions de destinations
    """
    regles = []
    for zone in get_filter_zones():
        if isfile(f_poste_name['destination']%zone):
            for poste in load_forbidden_poste(zone, 'destination'):
                regles.extend(interdit_destination(zone, poste))
    return regles

def generate_rules():
    """
       génère les règles iptables pour l'interdiction de postes
    """
    regles = generate_postes_rules()
    regles.extend(generate_all_postes_rules())
    regles.extend(generate_destinations_rules())
    for rule, container in regles:
        rule = rule.strip().split()
        ret, out, err = system_out(rule, container=container)
        if ret != 0:
            print_red("Gestion poste : erreur à la mise en place des restrictions de postes/destinations ({0}): {1}, {2}".format(' '.join(rule), out, err))
    return True

