# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# Génère les règles iptables pour les groupes de machine
#
###########################################################################
import datetime
from amon.ipset import group_manager
from amon.ipset import time_manager
from amon.backend import get_filter_zones, get_dst_offset
from amon.config import VIRTDISABLED, INTERFACE_GW, WEB_PORTS
from pyeole.process import system_out
from creole.config import VIRTMASTER
#Pour avoir des prints
debug = False

#### template pour les règles iptables
# partie ipset de la règle avec gestion de ports (portset webservices)
GROUP_RULE = """-i %(interface)s -m set --match-set group-%(name)s src"""
# sans la gestion de ports
TOTAL_GROUP_RULE = """-i %(interface)s -m set --match-set group-%(name)s src"""

# règles Web :
WEB_TIME_INPUT_RULE = """/sbin/iptables -I INPUT -p tcp --dport 83 %s %s -j ACCEPT"""
WEB_TIME_FORWARD_RULE = """/sbin/iptables -t nat -I PREROUTING -m multiport -p tcp --dports {} {} {} -j REDIRECT --to-ports 83"""

WEB_INPUT_RULE = """/sbin/iptables -I INPUT -p tcp --dport 83 %s -j ACCEPT"""
WEB_FORWARD_RULE = """/sbin/iptables -t nat -I PREROUTING -m multiport -p tcp --dports {} {} -j REDIRECT --to-ports 83"""

# règles horaires :
TIME_RULE = """-m time --timestart %(hdeb)s --timestop %(hfin)s --weekdays %(jour)s"""

# règle totale
TOTAL_INPUT_RULE = """/sbin/iptables -I INPUT %s -j REJECT"""
TOTAL_PREROUTING_RULE = """/sbin/iptables -I FORWARD %s -j REJECT"""

# Traduction des jours
JOUR = dict(lundi='Mon', mardi='Tue', mercredi='Wed', jeudi='Thu',
            vendredi='Fri', samedi='Sat', dimanche='Sun')

# Liste des jours
semaine = ['lundi','mardi','mercredi','jeudi','vendredi','samedi','dimanche']
#semaine = ['Mon','Tue','Wed','Thu','Fri','Sat','Sun']

if not VIRTDISABLED:
    containers = ['proxy', 'domaine']
else:
    containers = ['proxy']


class IptablesGenerator:
    """
        génère les règles iptables pour les set d'ips en fonction des interdictions
        1 - creation des regles pour le module ipset restreint au port 3128 et sans restriction
        2 - séparation des règles web ou total réseau
        3 - traitement horaires ou pas horaire
    """

    def launch_rule(self, rule, container=VIRTMASTER):
        """
            lance l'éxécution d'une règle iptable
        """
        # FIXME !!!
        #desactive les règles dans le conteneur si désactivé
        #if container != VIRTMASTER and VIRTDISABLED:
        #    return
        code, out, err = system_out(rule.split(), container=container)
        if code != 0:
            print("erreur à l'exécution de la commande {0} dans le conteneur {1} : {2}, {3}".format(rule, container, out, err))
        if debug:
            print("Execution de %s depuis iptables_generator.py"%(rule))
            print(code, out, err)


    def create_rules(self):
        """
        Génération des règles iptables pour les groupes de machines
        """
        I_GW = INTERFACE_GW
        for zone, active_groups in get_active_groups().items():
            for group in active_groups:
                group_datas = get_group(zone, group)

                ## si web
                ##      creer regles generales
                ##      si horaires:
                ##          generation des regles horaires
                ##      sinon
                ##          generation des regles sans horaires
                ## sinon
                ##      generation des regles globales

                if group_datas['active'] != '2':                    # interdiction web

                    if not VIRTDISABLED:
                        # FIXME : hack pour AmonEcole #19804
                        group_datas['interface'] = 'eth1'
                    total_group_rule = TOTAL_GROUP_RULE % group_datas
                    group_rule = GROUP_RULE % group_datas

                    if 'schedules' in group_datas:            # avec horaire
                        for schedule in group_datas['schedules']:
                            time_rule = TIME_RULE % schedule

                            rule = WEB_TIME_INPUT_RULE % (group_rule, time_rule)
                            for container in containers:
                                self.launch_rule(rule, container)

                            rule = WEB_TIME_FORWARD_RULE.format(WEB_PORTS, total_group_rule, time_rule)
                            for container in containers:
                                self.launch_rule(rule, container)
                    else:                                           # sans horaire (tout le temps)
                        rule = WEB_INPUT_RULE % (group_rule)
                        for container in containers:
                            self.launch_rule(rule, container)

                        rule = WEB_FORWARD_RULE.format(WEB_PORTS, total_group_rule)
                        for container in containers:
                            self.launch_rule(rule, container)

                else: ## interdiction totale
                    total_group_rule = TOTAL_GROUP_RULE % group_datas
                    rule = TOTAL_INPUT_RULE % total_group_rule
                    self.launch_rule(rule)
                    rule = TOTAL_PREROUTING_RULE % total_group_rule
                    self.launch_rule(rule)
                    if not VIRTDISABLED:
                        # FIXME : hack pour AmonEcole #19804
                        group_datas['interface'] = 'eth1'
                        total_group_rule = TOTAL_GROUP_RULE % group_datas
                        rule = TOTAL_INPUT_RULE % total_group_rule
                        for container in containers:
                            self.launch_rule(rule, container)
                        rule = TOTAL_PREROUTING_RULE % total_group_rule
                        self.launch_rule(rule)


# fonctions utiles
def get_active_groups():
    """ charger la liste des groupes actifs """
    return dict([(zone,
                 [group for group in group_manager.get_all_groups(zone) if group['active'] != '0'])
                 for zone in get_filter_zones()])

def get_group(zone, group):
    """ renvoie le groupe group formaté pour la mise en place des règles iptables
        @group: dico de description du groupe
    """
    group_schedules = time_manager.get_schedules(zone, group['name'])
    if group_schedules != {}:
        if group['time'] == '1':
            group['schedules'] = format_schedules(group_schedules)
    return group

def format_min(horaire):
    if horaire.minute == 0:
        return "00"
    else:
        return str(horaire.minute)


def gen_horaire(deb=None, fin=None):
    if deb:
        hdeb = '{0}:{1}'.format(deb.hour, format_min(deb))
    else:
        hdeb = '0:00'
    if fin:
        hfin = '{0}:{1}'.format(fin.hour, format_min(fin))
    else:
        hfin = '24:00'
    return {'hdeb':hdeb, 'hfin':hfin}

def to_utc(jour, horaires):
    """recalcule les plages par jour après application du décalage de la timezone (DST)
    pour permettre une application via ipset (heure au format UTC)
    Retourne une liste de plages (jour, hdeb, hfin)
    """
    utc_horaires = {}
    for horaire in horaires:
        deb = horaire['hdeb']
        fin = horaire['hfin']
        tz_offset = datetime.timedelta(seconds=get_dst_offset())
        # on applique l'offset en simulant les horaires sur la date courante
        hdeb, mindeb = deb.split(':')
        timedeb = datetime.datetime.combine(datetime.datetime.now().date(),
                                            datetime.time(int(hdeb),int(mindeb)))
        hfin, minfin = fin.split(':')
        timefin = datetime.datetime.combine(datetime.datetime.now().date(),
                                            datetime.time(int(hfin),int(minfin)))
        # utilisé pour vérifier si le décalage entraîne un changement de jour
        # jour suivant et précédent par rapport à la 'fausse' date de début
        next_day = timedeb + datetime.timedelta(days=1)
        prev_day = timedeb + datetime.timedelta(days=-1)
        jour_next = semaine[(semaine.index(jour)+1) % 7]
        jour_prev = semaine[(semaine.index(jour)-1) % 7]
        cur_day = timedeb.day
        jours = {prev_day.day:jour_prev, next_day.day:jour_next, cur_day:jour}
        # applique l'offset
        utc_deb = timedeb + tz_offset
        utc_fin = timefin + tz_offset
        # vérifie les jours de début/fin et détermine les plages à retourner
        if utc_deb.day == utc_fin.day:
            # plage simple
            plages = utc_horaires.get(jours[utc_deb.day], [])
            plages.append(gen_horaire(utc_deb, utc_fin))
            utc_horaires[jours[utc_deb.day]] = plages
        else:
            # plage sur 2 jours
            plages = utc_horaires.get(jours[utc_deb.day], [])
            plages.append(gen_horaire(deb=utc_deb))
            utc_horaires[jours[utc_deb.day]] = plages
            plages = utc_horaires.get(jours[utc_fin.day], [])
            plages.append(gen_horaire(fin=utc_fin))
            utc_horaires[jours[utc_fin.day]] = plages
    return utc_horaires


def format_schedules(schedules):
    """ formatte les plages horaires afin de minimiser le nombre de règle iptables à implémenter
        on interdit tout sauf les horaires indiqués
    """
    def compare_schedules(a):
        """ compare deux plages horaires """
        a = a['hdeb']
        return int(a.split(':')[0])

    formatted_schedules = []
    # repérer les plages continues sur deux jours et les rassembler ? (contiguous=True)
    utc_schedules = {}
    for jour in semaine:
        # conversion au format UTC (utilisé par ipset)
        horaires_local = schedules.get(jour,[])
        if horaires_local:
            sorted(horaires_local, key=compare_schedules)
            for utc_day, horaires in to_utc(jour, horaires_local).items():
                utc_horaires = utc_schedules.get(utc_day, [])
                utc_horaires.extend(horaires)
                utc_schedules[utc_day] = utc_horaires

    for jour in semaine:
        horaires = utc_schedules.get(jour, [])
        sorted(horaires, key=compare_schedules)
        deb = '0:00'
        for horaire in horaires:
            # si le debut de plage n'est pas égal à la fin (pour le début de journée de 0:00 à 0:00)
            if horaire['hdeb'] != deb:
                # correction de hfin pour minimiser la coupure
                if horaire['hfin'] in ['23:59', '24:00']:
                    horaire['hfin'] = "23:59:59"
                formatted_schedules.append({'jour':JOUR[jour], 'hdeb':deb, 'hfin': horaire['hdeb']})
            deb = horaire['hfin']
        # si le début de la nouvelle plage ne correspond pas à la fin de la journée
        if deb not in ['23:59:59']:
            formatted_schedules.append({'jour':JOUR[jour], 'hdeb':deb, 'hfin':'23:59:59'})
        # regrouper les plages consécutives ? (23:59:59, 0:00 le jour suivant)
    return formatted_schedules
