# -*- coding: utf-8 -*-

"""Apply configuration of EOLE servers.

"""


import os
import argparse
import time
import shutil
import json

from glob import glob

import pwd
import PAM
import getpass
from itertools import count
from configparser import ConfigParser

from pyeole.log import getLogger
from pyeole.log import init_logging
from pyeole.log import set_formatter
from pyeole.log import set_filters

from pyeole import scriptargs
from pyeole.lock import acquire, release
from pyeole import process
from pyeole.schedule import display_schedules, apply_schedules
from pyeole import ihm
from pyeole.ansiprint import print_red, print_orange
from pyeole.pkg import report, EolePkg, _configure_sources_mirror, _MIRROR_DIST
from pyeole.pkg import PackageNotFoundError, RepositoryError, AptProxyError, AptCacherError
from pyeole.service import unmanaged_service, manage_services, \
    ServiceError
from pyeole.encode import normalize
from pyeole.diagnose.diagnose import MAJ_SUCCES_LOCK

from .error import FileNotFound, LockError, UnlockError
from .error import UserExit, UserExitError
from .error import VirtError
from .client import CreoleClient, CreoleClientError, NotFoundError
from . import fonctionseole, template, cert
from .eosfunc import is_instanciate, get_devices
from .config import configeol, INSTANCE_LOCKFILE, UPGRADE_LOCKFILE, NEED_MAJ_AUTO_LOCKFILE, \
    container_instance_lockfile, gen_conteneurs_needed, VIRTROOT
from .containers import is_lxc_enabled, is_lxc_running, \
    generate_lxc_container, create_mount_point, lxc_need_restart
from .error import NetworkConfigError

from pyeole.i18n import i18n
_ = i18n('creole')

try:
    from zephir.lib_zephir import lock, unlock
    zephir_libs = True
except Exception:
    zephir_libs = False

client = CreoleClient()

global PKGMGR
PKGMGR = None

SHELL_EOLE = '/usr/bin/manage-eole'

error_msg_documentation = _("""For more informations, read section
'Mise en œuvre des modules EOLE' in module documentation or
common documentation.""")
def load_pkgmgr():
    global PKGMGR
    if PKGMGR is None:
        cache()
        PKGMGR = EolePkg('apt', container_mode=CACHE['is_lxc_enabled'])
        PKGMGR.pkgmgr.groups = CACHE
        PKGMGR.pkgmgr._load_apt_cache()
        eoles = []
        for eole in client.get_creole('serveur_maj'):
            eoles.append('http://{0}/eole/'.format(eole))
        ubuntus = []
        for ubuntu in client.get_creole('ubuntu_update_mirrors'):
            ubuntus.append('http://{0}/ubuntu/'.format(ubuntu))
        envoles = []
        try:
            for envole in client.get_creole('envole_update_mirrors'):
                envoles.append('http://{0}/envole/'.format(envole))
        except NotFoundError:
            pass
        for cache_ in PKGMGR.pkgmgr.cache._list.list:
            if cache_.uri in eoles:
                PKGMGR.pkgmgr._test_mirror(cache_.uri, _MIRROR_DIST['EOLE'])
                eoles = []
            if cache_.uri in ubuntus:
                PKGMGR.pkgmgr._test_mirror(cache_.uri, _MIRROR_DIST['Ubuntu'])
                ubuntus = []
            if cache_.uri in envoles:
                PKGMGR.pkgmgr._test_mirror(cache_.uri, _MIRROR_DIST['Envole'])
                envoles = []
        fonctionseole.PkgManager = PKGMGR

_LOGFILENAME = '/var/log/reconfigure.log'

# Command line options
class Option:
    """Manage commande line options with defaults

    """
    def __init__(self):
        self.parser = argparse.ArgumentParser(
                description=_("Applying EOLE configuration."),
                parents=[scriptargs.container(),
                scriptargs.logging(level='info')])
        self.parser.add_argument('-i', '--interactive', action='store_true',
                help=_("leave process in interactive mode"))
        self.parser.add_argument('-f', '--force', action='store_true',
                help=_("override Zéphir lock"))
        self.parser.add_argument('-a', '--auto', action='store_true',
                help=_("automatic reboot if necessary"))
        self.__opts = self.parser.parse_args([])

    def update_from_cmdline(self, force_args=None, force_options=None):
        """Parse command line
        """
        self.__opts = self.parser.parse_args(force_args)
        if self.__opts.verbose:
            self.__opts.log_level = 'info'
        if self.__opts.debug:
            self.__opts.log_level = 'debug'
        if force_options is not None:
            for key, value in list(force_options.items()):
                setattr(self.__opts, key, value)
        self.__dict__.update(self.__opts.__dict__)

    def __getattr__(self, name):
        if name in ['__opts', 'update_from_cmdline']:
            return self.__dict__[name]
        return getattr(self.__opts, name)

options = Option()

# To use log from every functions
log = getLogger(__name__)

# Same name for instance and reconfigure
LOCK_NAME = 'reconfigure'

# Run scripts in directories
RUNPARTS_PATH = '/usr/share/eole'
RUNPARTS_CMD = '/bin/run-parts --exit-on-error -v {directory} --arg {compat} 2>&1'

# Compatibility
COMPAT_NAME = 'reconfigure'

#def parse_cmdline():
#    """Parse command line
#    """
#    descr = u"Application de la configuration EOLE"
#    parser = argparse.ArgumentParser(description=descr,
#                                     parents=[scriptargs.container(),
#                                              scriptargs.logging(level='info')])
#    parser.add_argument('-i', '--interactive', action='store_true',
#                        help=u"lancer le processus en mode interactif")
#    parser.add_argument('-f', '--force', action='store_true',
#                        help=u"force l'action même s'il existe des verrous")
#    parser.add_argument('-a', '--auto', action='store_true',
#                        help=u"redémarrage automatique si nécessaire")
#
#    opts = parser.parse_args()
#    if opts.verbose:
#        opts.log_level = 'info'
#    if opts.debug:
#        opts.log_level = 'debug'
#    return opts

def copyDirectoryContent(src, dst):
    for fic in os.listdir(src):
        fullsrc = os.path.join(src, fic)
        # Skip links or we ovewrite existing certificates
        if os.path.islink(fullsrc):
            continue
        if os.path.isdir(fullsrc):
            log.debug('Skipping {}: is a directory'.format(fullsrc))
            continue
        try:
            shutil.copy2(fullsrc, dst)
        except shutil.Error as err:
            # ignore if files already exists
            pass

def user_exit(*args, **kwargs):
    """
    sortie utilisateur "propre"
    """
    log.warning(_('! Abandoning configuration !'))
    log.warning(_('System may be in an incoherent state.\n\n'))
    raise UserExitError()

def unlock_actions(need_lock=True):
    if zephir_libs:
        #FIXME: lock de Zephir !
        unlock('actions')
    try:
        release(LOCK_NAME, level='system')
    except Exception as err:
        # FIXME: move lock exception to pyeole.lock #7400
        if need_lock:
            raise UnlockError(str(err))

def lock_actions():
    try:
        acquire(LOCK_NAME, level="system")
    except Exception as err:
        # FIXME: move lock exception to pyeole.lock #7400
        raise LockError(str(err))
    if zephir_libs:
        #FIXME: lock de Zephir !
        lock('actions')

def reset_compat_name():
    """
    Réinitialise le nom de la procédure en cours
    en fonction de l'environnement
    """
    global COMPAT_NAME
    if options.interactive:
        COMPAT_NAME = 'instance'
    else:
        COMPAT_NAME = 'reconfigure'

def run_parts(directory):
    """Run scripts in a directory

    @param directory: name of a directory
    @type directory: C{str}
    """
    dirpath = os.path.join(RUNPARTS_PATH, directory)
    if os.path.isdir(dirpath):
        ihm.print_title(_('Running scripts {0}').format(directory))
        code = os.system(RUNPARTS_CMD.format(directory=dirpath, compat=COMPAT_NAME))
        if code != 0:
            raise Exception(_('Error {0}').format(directory))

def restart_creoled():
    """
    Restart creoled service and verify if the client is OK
    """
    unmanaged_service('restart', 'creoled', 'service', display='console')
    try:
        client.get_creole('eole_version')
    except CreoleClientError:
        msg = _("Please check creoled's log (/var/log/rsyslog/local/creoled/creoled.info.log)\nand restart service with command 'service creoled start'")
        raise CreoleClientError(msg)

def prepare(need_lock=True):
    """Sanity checks.

    """
    global RUNPARTS_CMD
    # Clean exit
    if need_lock:
        ihm.catch_signal(user_exit)
        lock_actions()

    if options.container != None:
        RUNPARTS_CMD += " --regex '^[09][09]-{0}$'".format(options.container)

    ihm.print_title(_("Preparation for {0}").format(COMPAT_NAME))

    if not os.path.isfile(configeol):
        print(_("Server is not configured."))
        print('')
        print(error_msg_documentation)
        print('')
        raise FileNotFound(_('Missing file {0}.').format(configeol))

    display_info = False

    if os.path.isfile(NEED_MAJ_AUTO_LOCKFILE):
        raise Exception(_(u'Server is not updated, please use Maj-Auto before launch instance or reconfigure'))

    if not options.interactive and (is_instanciate() == 'non' or os.path.isfile(UPGRADE_LOCKFILE)):
        print_red(_("Server must be instantiated before any reconfiguration can occur."))
        display_info = True

    if options.interactive and is_instanciate() == 'oui' and \
            not os.path.isfile(UPGRADE_LOCKFILE) and \
            not os.path.isfile(container_instance_lockfile):
        print_red(_("Server already instantiated."))
        print('')
        print(_("To modify configuration parameter (e.g. IP address), use:"))
        print(_("'gen_config'"))
        print(_("then 'reconfigure' to apply changes."))
        display_info = True

    if os.path.isfile(container_instance_lockfile) and not options.interactive:
        raise Exception(_('you have run gen_conteneurs, please use instance instead of reconfigure'))

    if os.path.isfile(gen_conteneurs_needed):
        raise Exception(_('You have to run gen_conteneurs before instance'))

    if display_info:
        print('')
        print(error_msg_documentation)
        print('')
        if not options.interactive:
            raise Exception(_("First instantiate server."))
        else:
            if ihm.prompt_boolean(_("Proceeding with instantiation ?"),
                                   interactive=options.interactive,
                                   default=False) is False:
                raise UserExit()
            else:
                fonctionseole.zephir("MSG", "Instance forcée par l'utilisateur",
                                     COMPAT_NAME.upper())

    # redémarrage du service creoled
    restart_creoled()

    if fonctionseole.init_proc(COMPAT_NAME.upper()) == False and not options.force:
        log.warning(_("This process is blocked, contact Zéphir administrator."))
        if ihm.prompt_boolean(_("Force execution?"),
                               interactive=options.interactive,
                               default=False) is False:
            if not options.interactive:
                log.warning(_("Use -f option if you want to force execution"))
            raise UserExitError()
        else:
            fonctionseole.zephir("MSG",
                                 "Instance forcée par l'utilisateur",
                                 COMPAT_NAME.upper())


def valid_mandatory(need_lock):
    try:
        client.valid_mandatory()
    except Exception as err:
        log.warning(_('Configuration validation problem, please check server configuration.'))
        print('')
        print(error_msg_documentation)
        print('')
        unlock_actions(need_lock)
        raise ValueError(str(err))

def _start_containers():
    """ Try to start containers and make sure they are started
    """
    cache()
    for group_name in CACHE['groups_container']:
        group = CACHE['group_infos'][group_name]
        create_mount_point(group)

    if os.access('/usr/share/eole/preservice/00-lxc-net', os.X_OK):
        log.debug("Override lxc-net systemd script")
        process.system_code(['/usr/share/eole/preservice/00-lxc-net'])

    unmanaged_service('start', 'lxc-net', 'systemd', display='console', ctx=CACHE['group_infos']['root'])
    try:
        unmanaged_service('status', 'lxc', 'systemd')
    except ServiceError:
        unmanaged_service('start', 'lxc', 'systemd', display='console', ctx=CACHE['group_infos']['root'])
        #if lxc not started, do not wait for it
        #(we already waiting for it in systemd service)
        #if started, waiting for ssh access

    # addc container start very slowly
    max_try = 1200
    for count in range(max_try):
        s_code, s_out, s_err = process.system_out(['lxc-ls', '--stopped'])
        stopped = s_out.split()
        f_code, f_out, f_err = process.system_out(['lxc-ls', '--frozen'])
        frozen = f_out.split()

        if stopped or frozen:
            not_running = stopped + frozen
        else:
            # Everything is started by LXC
            # Are they reachable by SSH?
            not_running = []
            for group_name in CACHE['groups_container']:
                group_infos = CACHE['group_infos'][group_name]
                if not is_lxc_running(group_infos, CACHE['is_lxc_enabled']):
                    not_running.append(group_name)

            log.debug('Waiting 1 second for SSH access')
            time.sleep(1)

            if not not_running:
                break

        if stopped:
            for cont in stopped:
                log.debug('Manual start of stopped container “{0}”'.format(cont))
                process.system_out(['lxc-start', '--name', cont, '--daemon',
                                    '-o', '/var/log/lxc-{0}.log'.format(cont)])

        if frozen:
            for cont in frozen:
                log.debug('Manual unfreeze of frozen container “{0}”'.format(cont))
                process.system_out(['lxc-unfreeze', '--name', cont,
                                    '-o', '/var/log/lxc-{0}.log'.format(cont)])

    if not_running:
        waiting_for = ', '.join(not_running)
        msg = _('Unable to start LXC container : {0}',
                'Unable to start LXC containers : {0}', len(not_running))
        raise VirtError(msg.format(waiting_for))


def containers(minimal=False, log_=None):
    """Generate containers
    """
    if log_ is None:
        log_ = log
    VAR_LXC = '/var/lib/lxc'
    OPT_LXC = '/opt/lxc'

    cache()
    if not CACHE['is_lxc_enabled']:
        log.debug(_('Container mode is disabled.'))
        return True
    if not options.interactive:
        for group in CACHE['groups_container']:
            if not os.path.isdir(os.path.join(VIRTROOT, group)):
                raise Exception(_('container {0} does not already exist, please use gen_conteneurs to create this container').format(group))
    else:
        # make /var/lib/lxc -> /opt/lxc
        if os.path.isdir(VAR_LXC) and not os.path.exists(OPT_LXC):
            ihm.print_title(_("Setting up {0}").format(OPT_LXC))
            unmanaged_service('stop', 'lxc', 'systemd', display='console')
            unmanaged_service('stop', 'lxc-net', 'systemd', display='console')
            shutil.move(VAR_LXC, OPT_LXC)
            os.symlink(OPT_LXC, VAR_LXC)
            #first instance should be in minimal mode
            minimal = True

        ihm.print_title(_('Generating containers'))

        engine = template.CreoleTemplateEngine()
        rootctx = CACHE['group_infos']['root']
        if minimal:
            # inject var _minimal_mode in creole's vars that can be used in template
            engine.creole_variables_dict['_minimal_mode'] = True
            engine.instance_file('/etc/ssh/ssh_config', ctx=rootctx)
            engine.instance_file('/etc/lxc/default.conf', ctx=rootctx)
            engine.instance_file('/etc/dnsmasq.d/lxc', ctx=rootctx)
            engine.instance_file('/etc/default/lxc-net', ctx=rootctx)
            engine.instance_file('/etc/apt/apt.conf.d/02eoleproxy', ctx=rootctx)
        if CACHE['module_instancie'] == 'oui':
            engine.instance_file('/etc/resolv.conf', ctx=rootctx)

        load_pkgmgr()
        PKGMGR.pkgmgr._prepare_cache()
        for group in CACHE['groups_container']:
            generate_lxc_container(group)
            groupctx = CACHE['group_infos'][group]
            if minimal:
                engine.instance_file('../fstab', container=group, ctx=groupctx)
                engine.instance_file('../config', container=group, ctx=groupctx)
                engine.instance_file('../devices.hook', container=group, ctx=groupctx)
                engine.instance_file('/etc/apt/apt.conf.d/02eoleproxy', container=group, ctx=groupctx)
                engine.instance_file('/etc/ssh/sshd_config', container=group, ctx=groupctx)
                engine.instance_file('/etc/sysctl.d/50_disabled_ipv6.conf', container=group, ctx=groupctx)
                engine.instance_file('/etc/systemd/system/systemd-logind.service.d/override.conf', container=group, ctx=groupctx)
                engine.instance_file('/etc/systemd/system/systemd-hostnamed.service.d/override.conf', container=group, ctx=groupctx)
                engine.instance_file('/etc/systemd/system/systemd-networkd.service.d/override.conf', container=group, ctx=groupctx)
            if CACHE['module_instancie'] == 'oui':
                container_path = os.path.join(groupctx['path'], 'etc/resolv.conf')
                if os.path.islink(container_path):
                    os.remove(container_path)
                engine.instance_file('/etc/resolv.conf', container=group, ctx=groupctx)

    ihm.print_title(_('Starting containers'))
    _start_containers()

def remove_packages():
    """ Remove packages listed in /usr/share/eole/remove.d/ files
        param: repo: EoleApt Object
    """
    torm_conf = glob('/usr/share/eole/remove.d/*.conf')
    pkg_list = []
    for config in torm_conf:
        try:
            f_h = open(config, 'r')
            for line in f_h.readlines():
                pkg_list.append(line.strip('\n'))
            f_h.close()
        except IOError as err:
            log.error(_('Can not read file {0}: {1}').format(config, err))

    try:
        load_pkgmgr()
    except (RepositoryError, AptProxyError, AptCacherError) as err:
        pass

    kernels = fonctionseole.get_kernel_to_remove()

    if kernels:
        ihm.print_line(_("Removing old linux kernels and associate headers."))
        pkg_list.extend(kernels)

    if pkg_list != []:
        try:
            PKGMGR.remove(packages=pkg_list)
        except (PackageNotFoundError, SystemError) as err:
            msg = _('Unable to remove some packages: {0}')
            log.warning(msg.format(err))
            log.warning(_("These packages will be removed next 'reconfigure'"))


CACHE = {}
def cache():
    global CACHE
    if not 'groups' in CACHE:
        CACHE['groups'] = client.get_groups()
        CACHE['groups_container'] = []
        for group in CACHE['groups']:
            if group not in ['root', 'all']:
                CACHE['groups_container'].append(group)
        CACHE['group_infos'] = {}
        for group_name in CACHE['groups']:
            group_infos = client.get_group_infos(group_name)
            CACHE['group_infos'][group_name] = group_infos
        CACHE['is_lxc_enabled'] = is_lxc_enabled()
        CACHE['module_instancie'] = client.get_creole('module_instancie')



def install_packages(silent=False):
    """Install package for each container group
    """
    load_pkgmgr()

    cache()
    header = _('Checking Packages for container')
    for group_name, group_infos in list(CACHE['group_infos'].items()):
        package_names = [pkg['name'] for pkg in group_infos['packages']]
        if package_names != []:
            msg = header + ' {0}: {1}'.format(group_name, ' '.join(package_names))
            ihm.print_line(msg)
            PKGMGR.install(packages=package_names,
                           silent=silent,
                           container=group_infos['name'])


def packages():
    """Manage packages
    """
    ihm.print_title(_('Managing packages'))
    log.info(_('   Removing packages'))
    ihm.print_line(_('Removing packages'))
    remove_packages()
    log.info(_('   Installing packages'))
    ihm.print_line(_('Installing packages'))
    install_packages()


def templates():
    """Run pretemplate scripts and manage templates
    """
    ihm.print_title(_('Generating configuration files'))
    log.info(_('Generating configuration files'))
    cache()
    try:
        tmpl = template.CreoleTemplateEngine()
        tmpl.instance_files(container=options.container, containers_ctx=list(CACHE['group_infos'].values()))
    except Exception as err:
        if options.debug:
            log.debug(err, exc_info=True)
        else:
            log.error(err)
        raise err


def services(action, display_title=True, try_restart_lxc=True):
    """Manage services
    """
    cache()
    exclude = None
    if action == 'stop':
        if display_title:
            ihm.print_title(_("Stopping services"))
    elif action == 'start':
        if display_title:
            ihm.print_title(_("Starting services"))
        # ne pas demarrer le service certbot, c'est un service oneshot
        # et pyeole.service n'a pas l'air d'aimer ... #22092
        exclude = (('root', 'certbot'))
        ctx = CACHE['group_infos']['root']
        if try_restart_lxc and CACHE['is_lxc_enabled']:
            if lxc_need_restart():
                unmanaged_service('stop', 'lxc', 'systemd', display='console', ctx=ctx)
                unmanaged_service('stop', 'lxc-net', 'systemd', display='console', ctx=ctx)
            _start_containers()
    elif action == 'configure':
        if display_title:
            ihm.print_title(_("Configuring services"))
    else:
        raise ValueError(_("Unknown service action: {0}").format(action))
    if options.container is not None:
        containers_ctx = [CACHE['group_infos'][options.containers]]
    else:
        containers_ctx = list(CACHE['group_infos'].values())
    manage_services(action, container=options.container, display='console', exclude=exclude, containers_ctx=containers_ctx)


def _gen_user_list():
    """Generate list of users for password modification

    Start with basic one and ask for supplementary users.
    """
    yield 'root'

    node = client.get_creole('activer_onenode', 'non')
    master = client.get_creole('activer_onesinglenode', 'non')
    if node == 'oui' and master == 'non':
        yield 'oneadmin'

    for number in count(1):
        if number == 1:
            yield 'eole'
        else:
            yield 'eole{0}'.format(number)


def users():
    """Manage users
    """
    ihm.print_title(_('Managing system user accounts'))
    default_pass = {'root': '$eole&123456$',
                    'eole': '$fpmf&123456$',
                    'oneadmin': '$eole&123456$'}

    def pam_test_password(user, password):
        """Test validity of password authencating account.
        """
        def dummy_conv(*args):
            return [(password, 0)]

        auth = PAM.pam()
        auth.start('auth')
        auth.set_item(PAM.PAM_USER, user)
        auth.set_item(PAM.PAM_CONV, dummy_conv)
        try:
            auth.authenticate()
            return True
        except PAM.error:
            return False

    def user_has_password(user):
        """Test wether password is set for user.
        """
        with open('/etc/shadow', 'r') as shadow_fh:
            for entry in shadow_fh.readlines():
                account, pwd = entry.split(':')[:2]
                if account == user and pwd not in ['!', '*']:
                    return True
        return False

    if not options.interactive:
        log.debug(_('No system user account management in non-interactive mode.'))
        return

    for user in _gen_user_list():
        try:
            user_infos = pwd.getpwnam(user)
        except KeyError:
            if user == 'root':
                msg = _("'root' user unknown. This is abnormal.")
                raise Exception(msg)

            # no new administrator with NFS (#16321)
            if user != 'eole' and client.get_creole('adresse_serveur_nfs', None) is not None:
                log.warning(_('No new EOLE account with /home on NFS'))
                break

            prompt = _('Create new administrator user account {0}?')
            if user != 'eole' and ihm.prompt_boolean(prompt.format(user)) is False:
                break

            msg = _("Creating unexistent user {0}")
            log.info(msg.format(user))

            cmd = ['adduser', '--quiet', '--shell', SHELL_EOLE,
                   '--gecos', '{0} user'.format(user.upper()),
                   '--disabled-password', user]
            code = process.system_code(cmd)
            if code != 0:
                msg = _("Unable to create user {0}")
                raise Exception(msg.format(user))

            cmd = ['usermod', '--append', '--groups', 'adm,mail', user]
            code, out, err = process.system_out(cmd)
            if code != 0:
                msg = _("Unable to add '{0}' to group 'adm'.")
                raise Exception(msg.format(user))

            # Update informations
            user_infos = pwd.getpwnam(user)

        if user not in default_pass and user_has_password(user):
            msg = _("No modification of password of administrator user account {0}.")
            log.warning(msg.format(user))
            continue

        # Change password:
        # - on first instance
        # - if user is not an EOLE default user
        # - if user password match default ones
        if (not os.path.isfile(INSTANCE_LOCKFILE)
            or (user not in default_pass or not user_has_password(user)
                or pam_test_password(user, default_pass[user]))):

            msg = _("# Modificating password for user account {0} #")
            msg = msg.format(user)
            log.warning('#' * len(msg))
            log.warning(msg)
            log.warning('#' * len(msg))
            max_try = 5
            prompt = '{0}{1}: '
            first_prompt = _("New password")
            second_prompt = _("Confirming new password")
            loop_counter = ''
            for attempt in range(1, max_try+2):
                if attempt == max_try+1:
                    msg = _("Password input errors for {0}. Abandon.")
                    raise Exception(msg.format(user))

                loop_counter = loop_counter.format(attempt, max_try)
                passwd = getpass.getpass(prompt.format(first_prompt,
                                                       loop_counter))
                confirm_pass = getpass.getpass(prompt.format(second_prompt,
                                                             loop_counter))
                if passwd == confirm_pass:
                    if user in default_pass and default_pass[user] == passwd:
                        log.error(_("Can not use default password."))
                    else:
                        # Now we have the password
                        stdin = '{0}:{1}'.format(user, passwd)
                        code, stdout, stderr = process.system_out(['chpasswd'],
                                                                  stdin=stdin)
                        if code == 0:
                            msg = _('User {0} password updated.')
                            log.info(msg.format(user))
                            # Success
                            break
                        msg = _("Error changing password for {0}.")
                        try_again_pos = stdout.find('Try again.')
                        chpassmsg = stdout[0:try_again_pos]
                        log.error(msg.format(user))
                        print(chpassmsg)
                else:
                    log.error(_("Passwords mismatch."))

                # Display counter
                loop_counter = ' ({0}/{1})'
            # vérification du shell pour l'utilisateur eole #34541
            if user == 'eole' and user_infos.pw_shell != SHELL_EOLE:
                log.debug("Update shell for user {}".format(user))
                cmd = ['chsh', '--shell', SHELL_EOLE, user]
                code = process.system_code(cmd)
                if code != 0:
                    msg = _("Unable to update shell for user {}".format(user))
                    raise Exception(msg.format(user))


def certificates():
    """Manage certificates

    """
    ihm.print_title(_('Managing certificates'))
    try:
        # regénération des hashes des certificats SSL après avec créé les nouveaux certificats
        # porté de 2.3 #8488
        cert.rehash_if_needed()
        # si instance et certificat autosigné et certificats présents, proposer le remplacement
        if (os.path.isfile(UPGRADE_LOCKFILE) and
            os.path.isfile(client.get_creole('server_cert')) and
            client.get_creole('cert_type') == 'autosigné'):
            prompt = _("Do you want to replace old self-signed certificate?")
            regen_needed = ihm.prompt_boolean(prompt, default=True)
        else:
            regen_needed = False
        cert.gen_certs(regen=regen_needed)
    except Exception as err:
        if options.debug:
            log.debug(err, exc_info=True)
        else:
            log.error(err)
        raise Exception(_("Error while generating certificates: {0}").format(err))
    cache()
    if CACHE['is_lxc_enabled']:
        srcs = [os.path.join(cert.ssl_dir, "certs"), os.path.join(cert.ssl_dir, "private")]
        for group_name in CACHE['groups_container']:
            group = CACHE['group_infos'][group_name]
            ihm.print_line(_("Copying certificates in {0}").format(group['name']))
            for src in srcs:
                dst = os.path.join('/', group['path'].lstrip('/'), src.lstrip('/'))
                copyDirectoryContent(src, dst)
            process.system_out(['/usr/bin/c_rehash'], container=group_name)


def param_kernel():
    """Manage kernel parameters
    """
    ihm.print_title(_('Applying kernel parameters'))
    os.system('/sbin/sysctl -p >/dev/null')

def kill_dhclient():
    """Kill dhclient for static IP configuration.

    """
    if client.get_creole('eth0_method') == 'statique':
        os.system('killall dhclient dhclient3 2>/dev/null')

def finalize(need_lock=True):
    """Clean up
    """
    ihm.print_title(_('Finalizing configuration'))
    # enregistrement
    try:
        process.system_out("/usr/share/creole/diag.py")
    except Exception:
        pass
    fonctionseole.zephir("FIN", "Configuration terminée", COMPAT_NAME.upper())
    if not os.path.isfile(INSTANCE_LOCKFILE):
        # l'instance est allée à son terme (#7051)
        open(INSTANCE_LOCKFILE, 'w').close()

    if os.path.isfile(UPGRADE_LOCKFILE):
        os.unlink(UPGRADE_LOCKFILE)

    if os.path.isfile(container_instance_lockfile):
        os.unlink(container_instance_lockfile)

    # sauvegarde des 2 dernières versions de la configuration (#8455)
    old = '{0}.bak'.format(configeol)
    old1 = '{0}.bak.1'.format(configeol)
    if not os.path.isfile(old):
        log.debug(_('Backup {0} in {1}'.format(configeol, old)))
        shutil.copy(configeol, old)
    elif process.system_out(['diff', '-q', configeol, old])[0] == 0:
        log.debug(_("{0} was not modified".format(configeol)))
    else:
        log.debug(_('Backup {0} in {1}'.format(old, old1)))
        shutil.copy(old, old1)
        log.debug(_('Backup {0} in {1}'.format(configeol, old)))
        shutil.copy(configeol, old)
    if need_lock:
        unlock_actions()

def update_server():
    """Manage server update
    """
    if os.path.isfile(MAJ_SUCCES_LOCK):
        os.remove(MAJ_SUCCES_LOCK)
    if options.interactive:
        log.info(_('Managing update'))

        ihm.print_title(_('Updating server'))
        if ihm.prompt_boolean(_("""An update is recommended.
Do you want to proceed with network update now ?"""),
                               default=True, level='warn',
                               default_uninteractive=False) is True:
            report(2)
            try:
                load_pkgmgr()
                _configure_sources_mirror(PKGMGR.pkgmgr)
                PKGMGR.update(silent=True)
                upgrades = PKGMGR.get_upgradable_list(silent=True)
                require_dist_upgrade = False
                for container, upgrades in list(upgrades.items()):
                    if upgrades:
                        require_dist_upgrade = True
                        break
                if require_dist_upgrade:
                    # At least one container require upgrade
                    PKGMGR.dist_upgrade()
                    # Update lock => OK, will be deleted at next reconfigure
                    report(0)
                    # Purge cache
                    global CACHE
                    CACHE = {}
                    # recall reconfigure
                    main(force_options={'interactive': False})
                    # back to instance
                    options.interactive = True
                    reset_compat_name()
                else:
                    log.warning(_("No updates available."))
                    report(3)
            except Exception as err:
                report(1, normalize(err))
                raise err


def schedule():
    """Manage task scheduling
    """
    ihm.print_title(_('Task scheduling'))
    apply_schedules()
    display_schedules()
    # 1er lancement de instance
    #if not os.path.isfile(schedule.SCHEDULE_FILE):
    #    schedule.add_post_schedule('majauto', 'weekly')
    #schedule.prog_schedule()


def reboot_server():
    """Reboot the server if required
    """
    if fonctionseole.controle_kernel():
        if options.interactive:
            print()
            if ihm.prompt_boolean(_("""Reboot is necessary.
Do you want to reboot now?"""),
                                   default=True, level='warn') is True:
                fonctionseole.zephir("MSG",
                                     "Demande de redémarrage acceptée par l'utilisateur",
                                     COMPAT_NAME.upper())
                process.system_code(['reboot'])
            else:
                fonctionseole.zephir("MSG",
                                     "Demande de redémarrage refusée par l'utilisateur",
                                     COMPAT_NAME.upper())
        else:
            print()
            print_orange(_('Reboot necessary'))
            time.sleep(1)
            print()
            if options.auto:
                fonctionseole.zephir("MSG", "Redémarrage automatique",
                                     COMPAT_NAME.upper())
                process.system_code(['reboot'])
            else:
                try:
                    smtpconfig = ConfigParser()
                    smtpconfig.read('/etc/eole/server.cfg')
                    if smtpconfig.get('update', 'notification') in ('tous', 'kernel'):
                        wanted_kernel = fonctionseole.get_wanted_kernel()
                        if wanted_kernel != fonctionseole.get_current_kernel():
                            content = _('Need reboot to use expected kernel {0}').format(wanted_kernel)
                        else:
                            content = _('Need reboot')
                        fromaddr = smtpconfig.get('smtp', 'system_mail_from')
                        toaddrs = smtpconfig.get('smtp', 'system_mail_to')
                        mailhost = smtpconfig.get('smtp', 'mailhost')
                        hostname = smtpconfig.get('global', 'hostname')
                        hostip = smtpconfig.get('global', 'ip')

                        subject = _('Server {0} ({1}) reboot must be planned.').format(hostname, hostip)
                        smtp_params = {'subject': subject,
                                       'fromaddr': fromaddr,
                                       'toaddrs': [toaddrs],
                                       'secure': (),
                                       'mailhost': 'localhost',
                                      }
                        reboot_logger = init_logging(level='warning', name='reconfigure', smtp=smtp_params, console=False)
                        reboot_logger.warning(content)
                except Exception as err:
                    log.warning(_('Error while attempting notifying by mail: {0}').format(err))
                fonctionseole.zephir("MSG", "Redémarrage du serveur à planifier",
                                     COMPAT_NAME.upper())

class NetworkctlOutput:
    def __init__(self):
        self.previous = None
        self.current = None
        self.get()

    def get(self):
        try:
            code, output, err = process.system_out(['networkctl', '--json=short'])
            if code:
                raise Exception(_(f"networkctl exited with non-zero code: {err}"))
            managed_interfaces = {interface['Name']: (interface['AdministrativeState'], interface['OperationalState'])
                                 for interface in json.loads(output)['Interfaces']
                                 if interface.get('AdministrativeState') != 'unmanaged'}
            if self.current is not None:
                self.previous = self.current
            self.current = managed_interfaces
            self._display()
        except BaseException as err:
            print(_("Unable to get network interfaces status."))
            log.warning(_(f"Unable to get network interfaces status: {err}"))

    def _display(self):
        if self.previous is None and self.current is None:
            print(_("No available information on network interfaces."))
        else:
            reset = "\033[0m"
            def bold(content):
                return f"\033[1m{content}{reset}"
            def color_status(status):
                color = "\033[32m" if status in ["configured", "routable"] else "\033[33m"
                return f"{color}{status}{reset}"
            if self.previous is not None:
                print(f"\033[{len(self.previous) + 1}A\033[0J")
            formatted_output = f"{reset}\n". join([f"{bold(name)}\t{color_status(status[0])}\t{color_status(status[1])}" for name, status in self.current.items()])
            print(formatted_output)

    def has_changed(self):
        return self.previous != self.current

def wait_for_ip(is_dhcp, nom_carte, ip_eth0):
    done = False
    idx = 0
    while True:
        unmanaged_service('restart', 'systemd-networkd', 'systemd', display='console')
        try:
            unmanaged_service('restart', 'systemd-networkd-wait-online', 'systemd')
        except:
            return False
        if _is_valid_ip_eth0(is_dhcp, nom_carte, ip_eth0, timeout=30):
            # waiting for the netplan-wait-online service to be started
            max_loop = client.get_creole("network_min_wait", 20)
            loop = 0
            managed_interfaces = NetworkctlOutput()
            while True:
                try:
                    unmanaged_service('status', 'systemd-networkd-wait-online', 'systemd')
                except ServiceError as err:
                    managed_interfaces.get()
                    if managed_interfaces.has_changed():
                        loop = 0
                    else:
                        loop += 1
                    if max_loop < loop:
                        raise err from err
                    time.sleep(1)
                    continue
                break
            return True
        idx += 1
        if idx == 2:
            break
    return False


def _is_valid_ip_eth0(is_dhcp, nom_carte, ip_eth0, timeout):
    import re, time
    wanted = (ip_eth0 or '').split('/')[0]

    for _ in range(int(timeout) + 1):
        # demande l’état réel au noyau via ip -o -4 addr show dev (que les ipv4)
        code, out, err = process.system_out(['ip', '-o', '-4', 'addr', 'show', 'dev', nom_carte])
        if code == 0:
            # on coupe pour n'avoir que les ipv4
            have = [m.split('/')[0] for m in re.findall(r'\binet\s+(\S+)', out)]
            if is_dhcp:
                # OK dès qu'il y a au moins une IPv4 non lien-local 169.254/16
                if any(not ip.startswith('169.254.') for ip in have):
                    return True
            else:
                if wanted and wanted in have:
                    return True
        time.sleep(1)
    return False

def main(force_options=None, force_args=None, need_lock=True):
    """Entry point
    """
    global log
    options.update_from_cmdline(force_args=force_args,
                                force_options=force_options)

    try:
        # module level logger
        log = init_logging(name='reconfigure', level=options.log_level,
                           console=['stderr', 'stddebug'],
                           filename=_LOGFILENAME)

        # Remove module name prefix from Warn/error messages emitted
        # from here
        set_formatter(log, 'stderr', 'brief')

        # Define handlers for additional loggers
        # Thoses logger are not for use
        # Log pyeole.service
        pyeole_service_log = init_logging(name='pyeole.service',
                                          level=options.log_level,
                                          filename=_LOGFILENAME,
                                          console=['stderr'])
        # Log pyeole.pkg
        pyeole_pkg_log = init_logging(name='pyeole.pkg',
                                          level=options.log_level,
                                          filename=_LOGFILENAME)
        passlib_log = init_logging(name='passlib.registry',
                                   level='error',
                                   filename=_LOGFILENAME)

        # Disable warnings from pyeole.service
        set_filters(pyeole_service_log, 'stderr',
                    ['error', 'critical'])

        if options.verbose or options.debug:
            # Enable creole logs
            creole_log = init_logging(name='creole', level=options.log_level,
                                      filename=_LOGFILENAME)
            # Define a root logger when verbose or debug is activated
            root_log = init_logging(level=options.log_level)
        else:
            # Enable creole logs
            creole_log = init_logging(name='creole', level=options.log_level,
                                      filename=_LOGFILENAME,
                                      console=['stderr'])

        creolemajauto_log = init_logging(name='creole.majauto', level=options.log_level,
                                  filename=_LOGFILENAME, console=['stderr', 'stdout'])

        ihm.print_title(_('Beginning of configuration'))
        # instance or reconfigure ?
        reset_compat_name()
        fonctionseole.zephir("INIT", "Début de configuration",
                             COMPAT_NAME.upper())
        prepare(need_lock)
        valid_mandatory(need_lock)
        cache()
        containers()
        packages()
        run_parts('preservice')
        services(action='stop')
        rootctx = CACHE['group_infos']['root']
        engine = template.CreoleTemplateEngine()
        test_activer_resolvconf_template = client.get_creole('test_activer_resolvconf_template') == 'oui'
        if test_activer_resolvconf_template:
            engine.instance_file('/etc/resolv.conf.full', ctx=rootctx)
            engine.instance_file('/etc/resolv.conf.minimal', ctx=rootctx)
            if CACHE['is_lxc_enabled']:
                for group in CACHE['groups_container']:
                    groupctx = CACHE['group_infos'][group]
                    engine.instance_file('/etc/resolv.conf.full', container=group, ctx=groupctx)
                    engine.instance_file('/etc/resolv.conf.minimal', container=group, ctx=groupctx)
            elif os.path.isdir('/var/lib/lxc/addc/rootfs'):
                engine.instance_file('/var/lib/lxc/addc/rootfs/etc/resolv.conf.full', ctx=rootctx)
                engine.instance_file('/var/lib/lxc/addc/rootfs/etc/resolv.conf.minimal', ctx=rootctx)
            process.system_code(['/usr/bin/CreoleSwitchResolv', 'minimal'])
        run_parts('pretemplate')
        ihm.print_title(_('Network restart'))
        engine.instance_file('/etc/eole/network.json', ctx=rootctx)
        for f in rootctx['files']:
            fnames = f['full_name']
            if not isinstance(fnames, list):
                fnames = [fnames]
            if f.get('activate', False):
                for fname in fnames:
                    if fname.startswith('/etc/netplan') or fname.startswith('/etc/systemd/network/'):
                        engine.instance_file(fname, ctx=rootctx)
            elif f.get('rm', False):
                for fname in fnames:
                    if fname.startswith('/etc/netplan') or fname.startswith('/etc/systemd/network/'):
                        engine.remove_destfile(fname)
        engine.instance_file('/usr/lib/eole/set_linkspeed', ctx=rootctx)
        if CACHE['is_lxc_enabled']:
            engine.instance_file('/etc/eole/ignore-interfaces.csv', ctx=rootctx)
        process.system_code('/usr/sbin/eole-purge-interfaces')
        time.sleep(2)
        if process.system_code(['/usr/sbin/netplan', 'generate']):
            msg = _("Unable to obtain IP address.")
            raise NetworkConfigError(msg)
        is_dhcp = client.get_creole('eth0_method') == 'dhcp'
        nom_zone_eth0 = client.get_creole('nom_zone_eth0')
        if is_dhcp:
            adresse_ip_eth0 = None
        else:
            adresse_ip_eth0 = client.get_creole('adresse_ip_eth0')
        if not wait_for_ip(is_dhcp, nom_zone_eth0, adresse_ip_eth0):
            msg = _("Unable to obtain IP address.")
            raise NetworkConfigError(msg)
        templates()
        if test_activer_resolvconf_template:
            process.system_code(['/usr/bin/CreoleSwitchResolv', 'minimal'])
        unmanaged_service('restart', 'systemd-journald', 'systemd')
        services(action='configure')
        # posttemplate/00-annuaire needs the certificates
        certificates()
        run_parts('posttemplate')
        #close all connexion before param kernel #17408
        client.close()
        param_kernel()
        kill_dhclient()
        if test_activer_resolvconf_template:
            process.system_code(['/usr/bin/CreoleSwitchResolv', 'full'])
        services(action='start')
        users()
        run_parts('postservice')
        schedule()
        finalize(need_lock)
        ihm.print_title(_('Reconfiguration OK'))
        update_server()
        # IMPORTANT : Ne rien faire après ces lignes
        # car le serveur est susceptible d'être redémarré
        reboot_server()

    except (UserExit, UserExitError) as err:
        unlock_actions(need_lock)
        fonctionseole.zephir("FIN", "Abandon par l'utilisateur",
                             COMPAT_NAME.upper())
        raise err

    except Exception as err:
        if options.debug:
            log.debug(err, exc_info=True)
        else:
            log.error(err)
        fonctionseole.zephir('ERR', str(err),
                             COMPAT_NAME.upper(),
                             console=False)
        if need_lock:
            release(LOCK_NAME, valid=False, level='system')
        raise err

if __name__ == '__main__':
    main()
