# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# dataproxy.py
#
# classes permettant de vérifier l'authentification sur différents supports
# (actuellement seulement annuaire LDAP)
#
###########################################################################

# utile ?
class IAuthProxy:
    """Interface pour un système d'authentification et de récupération des infos utilisateurs.
    Mettre les éventuels commandes de connextion, ... dans __init__.py"""

    def get_data(self, identifier, identifier_field):
        """Récupère les données correspondant à identifier

        @param identifier: à priori l'identifiant d'un utilisateur dans le système de données
        """
        pass

    def authenticate(self, login, password):
        """Vérifie la validité d'un couple login/password

        @param login: identifiant de l'utilisateur dans ce système
        @param password: mot de passe

        retourne True ou False
        """
        pass

from eoleldaptor import eoleldapproxy
from twisted.python import log, failure
from util import get_replication_branches
import os, socket

class LDAPProxy:
    """récupère les infos d'un utilisateur dans un annuaire LDAP"""
    __implements__ = (IAuthProxy,)

    ignored_attrs = ['userPassword', 'sambaLMPassword', 'sambaNTPassword']
    service_name = 'ldap'
    group_attrs = ['sambaGroupType', 'displayName', 'cn', 'objectClass', 'gidNumber', 'mail', 'description', 'niveau']

    def __init__(self, hosts, ports, bases, ldap_labels, ldap_infos=[], readers=[], pass_files=[], login_otp=[], static_data={}, match_attributes=[], strict_base=True, use_tls=[]):
        """
        hosts : liste de serveurs ldap (ou un seul)
        ports : liste des ports correspondants
        """
        # on attend des listes de valeur (de même taille) pour hosts/ports/bases/readers/pass_files
        self.ldap_servers = []
        self.otp_config = {}
        if type(hosts) is not list:
            hosts = [hosts]
        if type(ports) is not list:
            ports = [ports]
        if type(bases) is not list:
            bases = [bases]
        if type(ldap_infos) is not list:
            ldap_infos = [ldap_infos]
        if type(ldap_labels) is not list:
            ldap_labels = [ldap_labels]
        if type(readers) is not list:
            readers = [readers]
        if type(pass_files) is not list:
            pass_files = [pass_files]
        if type(login_otp) is not list:
            login_otp = [login_otp]
        if type(match_attributes) is not list:
            match_attributes = [match_attributes]
        if type(use_tls) is not list:
            use_tls = [use_tls]
        # si ancienne configuration, on met les valeurs uniques dans une liste
        self.static_data = static_data
        self.strict_base = strict_base
        # initialisation des proxies ldap
        self.use_branches = False
        self.search_branches = {}
        self.ldap_infos = {}
        nb_branches = 0
        for host, port, base, ldap_label, ldap_infos, reader, pass_file, match_attribute, login_otp, use_tls \
                in zip(hosts, ports, bases, ldap_labels, ldap_infos, readers, pass_files, match_attributes, login_otp, use_tls):
            search_branches = {}
            # recherche des branches de recherche des utilisateurs dans les annuaires disponibles
            # la branche par défaut est la base de recherche renseignée dans la configuration
            if host in ['localhost', '127.0.0.1']:
                # branches de recherche dans le cas d'un annuaire local répliqué
                repl_branches = get_replication_branches()
                if repl_branches:
                    for dn, label in repl_branches.items():
                        search_branches[dn] = label
                        nb_branches +=1
                else:
                    search_branches[base] = ldap_label
                    nb_branches +=1
            else:
                # XXX FIXME: conserver la branche de base même en cas d'annuaires répliqués ?
                # si libellé non rempli, on en crée un automatiquement
                if ldap_label == "":
                    try:
                        ldap_label = "Annuaire de %s" % socket.gethostbyaddr(host)[0]
                    except:
                        ldap_label = "Annuaire de %s" % host
                #search_branches['default'] = ldap_label
                search_branches[base] = ldap_label
                nb_branches +=1
            if reader != "" and os.path.isfile(pass_file):
                reader_dn = reader
                reader_pass = passwd = open(pass_file).read().strip()
            else:
                reader_dn = reader_pass = None
            eole_proxy = eoleldapproxy.EoleLdapProxy(base, host, port, reader_dn, reader_pass, match_attribute, use_tls=use_tls)
            host_port = "%s_%s" % (host, port)
            self.ldap_servers.append((host_port, eole_proxy, (reader, pass_file), match_attribute))
            self.search_branches[host_port] = search_branches
            self.ldap_infos[host_port] = ldap_infos
            self.otp_config[host_port] = login_otp
        if nb_branches > 1:
            self.use_branches = True

    def check_otp_config(self, search_branch):
        """retourne le mode gestion de l'identifiant OTP pour un annuaire particulier
        inactif : OTP non géré pour cet annuaire
        identique : login OTP identique au login LDAP
        configurable : login OTP déclaré par l'utilisateur
        """
        if search_branch == "default":
            host_port = self.ldap_servers[0][0]
        else:
            host_port = search_branch.split(':', 1)[0]
        return self.otp_config.get(host_port, 'inactifs')

    def get_search_branch(self, attributes, attr_set):
        # on regarde si les attributs permettent de limiter la recherche à une branche
        # (peut résoudre les problèmes de doublons dans le cas d'un annuaire multi établissements)
        # lancement de l'authentification sur le premier proxy de la liste
        search_exprs = []
        search_base_attrs = attr_set['branch_attrs']
        # search_expr correspond aux expressions à rechercher dans le dn des branches connues
        for attr_name, attr_value in attributes.items():
            if attr_name in search_base_attrs:
                search_exprs.append('%s=%s' % (search_base_attrs[attr_name], attr_value[0]))
        if search_exprs:
            for proxy_data in self.ldap_servers:
                s_branches = self.search_branches[proxy_data[0]]
                for s_branch in s_branches.keys():
                    branch_ok = True
                    for expr in search_exprs:
                        if expr not in s_branch:
                            branch_ok = False
                            break
                    if branch_ok:
                        # toutes les valeurs demandées correspondent
                        return '%s:%s' % (proxy_data[0], s_branch)
            if self.strict_base:
                # mode strict : si au moins un attribut de recherche de
                # branche est donné, on n'utilise pas la branche de
                # recherche par défaut si aucune ne correspond
                return None
        return 'default'
        #        del(attributes[attr_name])
        #        return s_branch, attributes
        #return 'default', attributes

    def get_user_branches(self, user_id, branches=None, servers=None):
        """
            retourne les branches et libellés d'établissement dans lesquels un login donné est présent
        """
        # on fait une recherche anonyme sur la racine pour récupérer
        # tous les dn correspondants à user_id
        if servers is None:
            servers = self.ldap_servers
        if branches is None:
            branches = []
        host_port = servers[0][0]
        proxy = servers[0][1]
        defer_branches = proxy.get_user_dn(user_id, unicity=False)
        return defer_branches.addBoth(self.callb_branches, user_id, branches, host_port, servers[1:])

    def callb_branches(self, result_dn, user_id, branches, host_port, servers):
        if isinstance(result_dn, failure.Failure):
            log.msg(result_dn.getErrorMessage())
        else:
            # recherche des libellé etab correspondant à ces dn dans l'annuaire
            for user_dn in result_dn:
                # on recherche la branche d'annuaire à laquelle ce dn appartient
                for branche in self.search_branches[host_port]:
                    # on force la casse en minuscule pour la recherche
                    # des branches correspondant au dn
                    if branche.lower() in str(user_dn).lower():
                        # pour chaque branche, on renvoie : index du serveur ldap, dn de la branche, libellé
                        branches.append(('%s:%s' % (host_port, branche), self.search_branches[host_port][branche]))
                        break
        if len(servers) > 0:
            return self.get_user_branches(user_id, branches, servers)
        return branches

    def authenticate(self, user_id, passwd, search_branch='default', servers=None):
        """
        lancement de l'authentification sur le premier proxy de la liste
        branch_attrs : attributs pour déterminer une branche de recherche dans l'annuaire (ex: rne)
        servers : liste des serveurs LDAP sur lesquels essayer l'authentification
        """
        if search_branch == 'default':
            branch = None
            if servers is None:
                servers = self.ldap_servers
            proxy = servers[0][1]
        else:
            # on utilise directement le serveur spécifié avec la branche
            host_port, branch = search_branch.split(':', 1)
            for server in self.ldap_servers:
                if server[0] == host_port:
                    proxy = server[1]
                    servers = []
                    break
        defer_auth = proxy.authenticate(user_id, passwd, branch)
        return defer_auth.addCallbacks(self.callb_auth, self.errb_auth, callbackArgs=[user_id, passwd, search_branch, servers[1:]])

    def callb_auth(self, result_auth, user_id, passwd, search_branch, servers):
        success, user_data = result_auth
        # vérification du résultat de l'authentification sur le serveur précédent
        if success == False:
            # echec de l'authentification  sur ce serveur
            if len(servers) > 0:
                # test sur le serveur suivant
                return self.authenticate(user_id, passwd, search_branch, servers)
            else:
                # on a essayé sur tous les serveurs
                return False, {}
        # on supprime certains champs (mot de passe)
        for user_attr in self.ignored_attrs:
            if user_attr in user_data:
                del(user_data[user_attr])
        return success, user_data

    def errb_auth(self, failure):
        log.msg(failure.getErrorMessage())
        return False, {}

    def get_user_data(self, filter_attrs, search_branch='default', servers=None):
        """recherche des données utilisateur locales depuis un ensemble d'attributs/valeurs à matcher.
        utilisé dans le cadre de la fédération pour retrouver l'utilisateur local
        """
        if search_branch == 'default':
            branch = None
            if servers is None:
                servers = self.ldap_servers
            proxy = servers[0][1]
        else:
            # on utilise directement le serveur spécifié avec la branche
            host_port, branch = search_branch.split(':', 1)
            for server in self.ldap_servers:
                if server[0] == host_port:
                    proxy = server[1]
                    servers = []
                    break
        defer_auth = proxy.get_user_data(filter_attrs, branch)
        return defer_auth.addCallbacks(self.callb_get_data, self.errb_auth, callbackArgs=[filter_attrs, search_branch, servers[1:]])

    def callb_get_data(self, result_auth, filter_attrs, search_branch, servers):
        success, user_data = result_auth
        # vérification du résultat de l'authentification sur le serveur précédent
        if success == False:
            # echec de l'authentification  sur ce serveur
            if len(servers) > 0:
                # test sur le serveur suivant
                return self.get_user_data(filter_attrs, search_branch, servers)
            else:
                # on a essayé sur tous les serveurs
                return False, {}
        # on supprime certains champs (mot de passe)
        for user_attr in self.ignored_attrs:
            if user_attr in user_data:
                del(user_data[user_attr])
        return success, user_data
