#! /usr/bin/env python
# -*- coding: utf-8 -*-

###########################################################################
#
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# oidc_utils.py
#
# Fonctions utilitaires pour gérer l'authentification via OpenID Connect
#
###########################################################################

from eolesso.util import gen_random_id

from config import (CERTFILE, KEYFILE, PROXY_PORT, PROXY_SERVER, OPENID_PROVIDERS, \
                    OPENID_CONFFILE, CA_LOCATION, DEBUG_LOG, AUTH_FORM_URL)
from page import trace, log

# imports pour la gestion d'OpenID Connect
from oic.oauth2.message import ErrorResponse
from oic.oic.message import AccessTokenResponse
from oic.oic.message import ProviderConfigurationResponse
from oic.oic.message import AuthorizationResponse
from oic.oic.message import RegistrationResponse
from oic.oic import Client, IdToken
from oic.utils.authn.client import CLIENT_AUTHN_METHOD
from configobj import ConfigObj
import hashlib
import urllib
import cjson
import time
from os.path import join, isfile

class EoleClient(Client):

    # specifies if client implements a logout method
    can_logout = False

    def parse_response(self, response, info="", sformat="json", state="", **kwargs):
        """redefines oic.Client parse_response in order to store id_token.
        """
        self.id_token_hint = None
        if response == AccessTokenResponse:
            if info:
                params = cjson.decode(info)
                if 'id_token' in params:
                    self.id_token_hint = params['id_token']
        return super(EoleClient, self).parse_response(response, info, sformat, state, **kwargs)

class AccessDeniedError(Exception):
    pass

# providers for which a specific logout procedure is implemented
SPECIFIC_PROVIDERS = ['google.com','live.com','facebook.com']

def provider_can_logout(client):
    # checks if OpenID provider has a specific logout procédure
    if client.logout_endpoint is not None:
        return True
    for prov in SPECIFIC_PROVIDERS:
        if prov in client.provider_info['issuer']:
            return True
    return False

@trace
def init_oidclient(op_ref):
    """Initializes an OpenID Connect client using pyoidc library
    https://github.com/rohe/pyoidc.git
    """
    op_infos = ProviderConfigurationResponse(version = "1.0",
                    issuer = OPENID_PROVIDERS[op_ref]['issuer'],
                    authorization_endpoint = OPENID_PROVIDERS[op_ref]['auth_endp'],
                    token_endpoint = OPENID_PROVIDERS[op_ref]['token_endp'],
                    registration_endpoint = OPENID_PROVIDERS[op_ref]['reg_endp'],
                    logout_endpoint = OPENID_PROVIDERS[op_ref]['logout_endp'],
                    userinfo_endpoint = OPENID_PROVIDERS[op_ref]['userinfo_endp'],
                    jwks_uri = OPENID_PROVIDERS[op_ref]['jwks_uri'])
    uris = [AUTH_FORM_URL + "/oidcallback"]
    # FIXME : use system default CA (/etc/ssl/certs/ca-certificates.crt)
    # or configure specific ca_certs per provider ?
    # ca_certs = "/etc/ssl/certs/ca-certificates.crt"
    ca_certs = CA_LOCATION
    # load client ID/secret from config file
    if isfile(OPENID_CONFFILE):
        cnf_secret = ConfigObj(infile=OPENID_CONFFILE).get(op_ref, "")
        if cnf_secret:
            client_id, client_secret = cnf_secret.split(":")
        else:
            client_id = client_secret = ""
    setup_libraries()
    client = EoleClient(client_authn_method=CLIENT_AUTHN_METHOD, ca_certs=ca_certs)
    register_ok = False
    if client_id and client_secret:
        info = {"client_id": client_id, "client_secret": client_secret}
        client_reg = RegistrationResponse(**info)
        client.store_registration_info(client_reg)
        client.handle_provider_config(op_infos, op_infos['issuer'])
        register_ok = True
    else:
        log.msg('Client ID/Secret not found in {0} for {1}, trying automatic registration'.format(op_ref, OPENID_CONFFILE))
        infos = {"redirect_uris": uris}
        # id and secret are not available, try to register automatically
        try:
            registration_response = client.register(op_infos["registration_endpoint"], **infos)
            register_ok = True
        except:
            log.msg('auto-registration failed for {}, please specify client ID/Secret in {}'.format(op_ref, OPENID_CONFFILE))
    if register_ok:
        client.can_logout = provider_can_logout(client)
    client.redirect_uris = uris
    # initialize certificates and proxies
    client.request_args['cert'] = (CERTFILE, KEYFILE)
    if PROXY_SERVER:
        # XXX FIXME : may not be needed if http_proxy is set at shell level
        proxies = {'https':'http://{0}:{1}'.format(PROXY_SERVER, PROXY_PORT)}
        client.request_args['proxies'] = proxies
    return client

def setup_libraries():
    """disables some warnings in python requests library

    /usr/local/lib/python2.7/dist-packages/requests-2.8.0-py2.7.egg/requests/packages/urllib3/util/ssl_.py:100: requests.packages.urllib3.exceptions.InsecurePlatformWarning: A true SSLContext object is not available. This prevents urllib3 from configuring SSL appropriately and may cause certain SSL connections to fail. For more information, see https://urllib3.readthedocs.org/en/latest/security.html#insecureplatformwarning.

    other solutions may be possible

    * see this thread : http://stackoverflow.com/questions/29099404/ssl-insecureplatform-error-when-using-requests-package
    * another possibility is to refactor http_request function in oauth2/base.py and use another library (pyeole/httprequest).
      This may require more refactoring where http_request is used

    """
    try:
        import requests.packages.urllib3
        requests.packages.urllib3.disable_warnings()
    except:
        # older requests version : no warnings
        pass

def check_state(temp_cookie, state):
    """checks specified state validity against temp_cookie value
    """
    if temp_cookie:
        hash_val = hashlib.sha256(temp_cookie.value).hexdigest()
    return temp_cookie and state == hash_val

@trace
def request_auth(provider, manager, orig_args):
    """returns an URL allowing to perform an authorization request
    provider : openid provider to redirect to
    manager : EoleSSO authserver
    orig_args : request args received on login page
    """
    client = manager.external_providers[provider][0]
    user_store = manager.external_providers[provider][1]
    # generate a state value from temp_cookie value
    temp_cookie = gen_random_id('OIDC-COOKIE-')
    state = hashlib.sha256(temp_cookie).hexdigest()
    nonce = gen_random_id('OIDC-NONCE-')
    # store request persistent data in saml relay_state cache
    manager.relay_state_cache.add(state, {'provider':provider, 'orig_args':orig_args, 'state':state, 'nonce':nonce})
    args = {
        "client_id": client.client_id,
        "response_type": "code",
        "scope": ["openid"],
        "nonce": nonce,
        "redirect_uri": client.redirect_uris[0],
        "state": state
    }
    auth_req = client.construct_AuthorizationRequest(request_args=args)
    auth_url = client.provider_info['authorization_endpoint']  + "?" + auth_req.to_urlencoded()
    return auth_url, temp_cookie

@trace
def openid_logout_url(provider_data, manager, sso_session, return_url):
    prov_label = manager.external_providers[provider_data['provider']][2]
    # calculate logout url
    client = manager.external_providers[provider_data['provider']][0]
    state=provider_data['state']
    post_logout_uri = AUTH_FORM_URL + "/logout"
    needs_confirm = True
    # no global standard for single logout, use known solutions for most used providers
    # http://stackoverflow.com/questions/16946798/logout-from-external-login-service-gmail-facebook-using-oauth/17127549#17127549
    if 'google.com' in client.provider_info['issuer']:
        logout_url = "https://www.google.com/accounts/Logout?continue=https://appengine.google.com/_ah/logout"
        args = {"continue" : post_logout_uri}
    # XXX FIXME : facebook and microsoft (live) authentications have not been tested yet
    # not sure they are OpenID Connect compliant
    elif 'facebook.com' in client.provider_info['issuer']:
        logout_url = "https://www.facebook.com/logout.php"
        args = {"next": post_logout_uri,
                "access_token":provider_data['access_token']}
    # facebook needs to store access_token ?
    elif 'live.com' in client.provider_info['issuer']:
        logout_url = "https://login.live.com/oauth20_logout.srf"
        args = {"client_id": client.client_id,
                "redirect_url": post_logout_uri}
    elif client.provider_info['logout_endpoint']:
        # Use Openid Connect front channel logout for other Providers (France Connect, ...)
        # http://openid.net/specs/openid-connect-frontchannel-1_0.html#RPLogout
        needs_confirm = False
        # logout confirmation will be asked by Provider
        logout_url = client.provider_info['logout_endpoint']
        args = {
            "id_token_hint": provider_data['id_token_hint'],
            "post_logout_redirect_uri": post_logout_uri,
            "state":state
        }
    logout_url =  logout_url + '?' + urllib.urlencode(args)
    log.msg("%s -- %s" % (sso_session, _("Sending OPENID logout request to {0} ({1}) : {2}").format(prov_label, client.provider_info['logout_endpoint'], state)))
    return state, logout_url, needs_confirm

@trace
def get_user_identifier(response, client, state, nonce):
    """checks the response sent to callback URL and gets user infos when possible
    """
    # get and parse query_string
    aresp = client.parse_response(AuthorizationResponse, info=response,
                                          sformat="urlencoded")
    if isinstance(aresp, AuthorizationResponse):
        err_msg = None
    elif isinstance(aresp, ErrorResponse):
        # Get Error detail ?
        err_msg = aresp["error"]
        if str(err_msg) == "access_denied":
            # User has denied access to account information
            raise AccessDeniedError, _('User denied access to account information')
    # for other errors (bad configuration, invalid requests, ...), log error and send
    # user back to login page with original parameters
    else:
        err_msg = _("Unknown response type received")
    # unknown response type or user consent not given
    assert err_msg is None, err_msg
    code = aresp["code"]
    assert aresp["state"] == state, _("Invalid response received (parameter {} does not match original request)").format("state")
    args = {
        "code": aresp["code"],
        "redirect_uri": client.redirect_uris[0],
        "client_id": client.client_id,
        "client_secret": client.client_secret,
    }
    if client.provider_info['jwks_uri']:
        if DEBUG_LOG:
            log.msg("USING JWKS URI FOR CERTIFICATE VALIDATION : {}".format(client.provider_info['jwks_uri']))
        args['jwks_uri'] = client.provider_info['jwks_uri']
    # XXX FIXME : add option to disable ssl cert validation ?
    # else:
    #     client.verify_ssl=False
    resp = client.do_access_token_request(scope = "openid",
                                  state = aresp["state"],
                                  request_args=args,
                                  )
    # checks Response validity and content
    if isinstance(resp, AccessTokenResponse):
        err_msg = None
    elif isinstance(resp, ErrorResponse):
        # Get Error detail ?
        err_msg = resp["error"]
    else:
        err_msg = _("Unknown response type received")
    if DEBUG_LOG:
        log.msg(_('Error returned from access token endpoint : {0}').format(str(err_msg)))
    if 'nonce' in resp['id_token'] and resp['id_token']['nonce'] != nonce:
        # if parameter nonce appears in token, check that it's equal to nonce sent in original request
        err_msg = _("Invalid response received (parameter {} does not match original request)").format("nonce")
    # check if id_token has expired
    if "exp" in resp['id_token'] and time.time() > resp['id_token']['exp']:
        err_msg = _('ID Token has expired')
    assert err_msg is None, err_msg
    # check if user sub is included in access token, otherwise request user info
    if "sub" in resp['id_token']:
        if DEBUG_LOG:
            log.msg(_('Got user sub from {0} endpoint').format('token'))
        user_sub = resp['id_token']['sub']
    else:
        # use received token to retrieve user identifier (sub: unique and opaque identifier)
        userinfo = client.do_user_info_request(state=aresp["state"],
                                               userinfo_endpoint=client.provider_info['userinfo_endpoint'],
                                               token=resp["access_token"])
        assert "sub" in userinfo, _('Identity provider did not return user identifier')
        if DEBUG_LOG:
            log.msg(_('Got user sub from {0} endpoint').format('userinfo'))
        user_sub = userinfo['sub']
    return client.id_token_hint, resp['access_token'], user_sub
