#! /usr/bin/env python3
# -*- coding: utf-8 -*-

###########################################################################
#
# Eole NG - 2011
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# update_conf.py
#
# procédure de mise à jour de la configuration d'eole-sso
#
###########################################################################
import glob, os, sys, copy, shutil

try:
    from creole.loader import creole_loader, config_save_values
    from creole.utils import print_orange, print_red
except:
    # librairies non disponibles
    print("Configuration Creole non accessible")
    sys.exit(0)
from configparser import ConfigParser

CONF_FILE = "/etc/eole/config.eol"
SSO_DIR = "/usr/share/sso"
BACKUP_DIR = os.path.join(SSO_DIR, "backup_conf")
FIC_RAPPORT = "/root/rapport_maj_sso.txt"
RAPPORT_MAJ = """Informations sur la mise à niveau de la configuration du service EoleSSO

Les anciens fichiers de configuration sont sauvegardés dans le répertoire %s

""" % BACKUP_DIR

def update_report(msg, start_line="\n - "):
    global RAPPORT_MAJ
    RAPPORT_MAJ += "%s%s" % (start_line, msg)

def save_report():
    # écriture du rapport
    global RAPPORT_MAJ, FIC_RAPPORT
    f_rapport = open(FIC_RAPPORT, 'w')
    f_rapport.write(RAPPORT_MAJ)
    f_rapport.close()
    print_orange("Le rapport de la procédure est disponible dans %s" % FIC_RAPPORT)

def exit_err(msg_err, tb=None):
    global RAPPORT_MAJ
    print_red("** Erreur lors de la mise à niveau de la configuration **")
    RAPPORT_MAJ += """\n\n !! Erreur : %s

Vous pouvez restaurer les fichiers d'origine situés dans le dossier %s

Pour relancer la procédure de mise à niveau :
- Supprimez le répertoire de sauvegarde après restauration des fichiers
- Relancez le service eole-sso (service eole-sso restart)\n""" % (msg_err, BACKUP_DIR)
    if tb:
        RAPPORT_MAJ += "\nErreur rencontrée :\n\n%s\n" % tb
    save_report()
    sys.exit(1)

def update_entity_id(config):
    # on regarde si des metadata ont été échangés
    metadata = glob.glob(os.path.join(config.METADATA_DIR, '*.xml'))
    if metadata and os.path.isfile(CONF_FILE):
        try:
            if os.path.isfile(os.path.join(SSO_DIR, '.old_entity_id')):
                # sauvegarde des fichiers d'origine
                os.system('/bin/cp -rf %s %s' % (CONF_FILE, BACKUP_DIR))
                # mise à jour du nom d'entité en mode SAML
                # on ne prend que la dernière ligne au cas où l'import de config.py ait affiché des messages
                OLD_IDP_IDENTITY = open(os.path.join(SSO_DIR, '.old_entity_id')).read().strip().split('\n')[-1]
                dico = creole_loader(load_extra=True, reload_config=False)
                dico.read_write()
                dico.creole.eole_sso.eolesso_entity_name = OLD_IDP_IDENTITY
                assert config_save_values(dico, 'creole', False) == True, " Erreur de sauvegarde du fichier de configuration"
                update_report("Mise à jour de config.eol pour conserver l'ancien nom d'entité locale: %s" % OLD_IDP_IDENTITY)
        except Exception as e:
            exit_err("EoleSSO: Erreur de mise à jour du nom d'entité SAML", str(e))


class EoleParser(ConfigParser):
    """Sous classe de ConfigParser pour préserver la casse des options"""

    def optionxform(self, option_str):
        return option_str


def udpate_attr_sets(config):
    from saml_utils import get_metadata
    # construction du fichier d'association par défaut
    associations = EoleParser()
    try:
        update_report('sauvegarde des anciens fichiers de configuration')
        # sauvegarde des fichiers d'origine
        if len(glob.glob('%s/*.ini' % config.ATTR_SET_DIR)) > 0:
            os.system('/bin/mv %s/*.ini %s/attribute_sets/' % (config.ATTR_SET_DIR, BACKUP_DIR))
            # on conserve default.ini au cas où il aurait été modifié
            # os.system('/bin/cp -f %s/attribute_sets/default.ini %s' % (BACKUP_DIR, config.ATTR_SET_DIR))
        old_ini = os.path.join(config.METADATA_DIR, 'attributes.ini')
        # sauvegarde fichier obsolète attributes.ini
        if os.path.isfile(old_ini):
            os.system('/bin/mv -f %s %s' % (old_ini, BACKUP_DIR))
        ATTR_BACKUP_DIR = os.path.join(BACKUP_DIR, os.path.basename(config.ATTR_SET_DIR))
        # création des sections de configuration pour les entités partenaires
        partner_files = glob.glob(os.path.join(config.METADATA_DIR, '*.xml'))
        if partner_files:
            update_report("""
Détection des fournisseurs d'identité disponibles.
Pour interdire la fédération depuis une de ces entités,
modifiez l'option "allow_idp" dans la section du fournisseur
concerné dans le fichier %s
""" % os.path.join(config.ATTR_SET_DIR, 'associations.ini'), "\n")
        for partner_file in partner_files:
            # récupération du nom d'entité dans les metadata
            meta_info = get_metadata(os.path.splitext(os.path.basename(partner_file))[0])
            entity_id = meta_info.get('entityID', None)
            # on regarde si l'entité est fournisseur d'identité
            if entity_id and meta_info.get('IDPSSODescriptor', {}) != {}:
                associations.add_section(entity_id)
                # par défaut, on accepte les assertions provenant des entités déjà configurées
                associations.set(entity_id, 'allow_idp', 'true')
                update_report("Ajout d'un fournisseur d'identité : %s" % entity_id)
    except Exception as e:
        exit_err("EoleSSO: Erreur de conversion des jeux d'attributs existants", str(e))
    try:
        # prise en compte des jeux personnalisés à l'ancien format
        attr_sets = glob.glob(os.path.join(ATTR_BACKUP_DIR, '*.ini'))
        sets = {}
        # indice utilisé pour gérer le nom de fichier des jeux d'attributs convertis
        file_index = 1
        update_report("""
Détection des éventuels jeux d'attributs personnalisés définis
""", "\n")
        for attr_file in attr_sets:
            basename = os.path.splitext(os.path.basename(attr_file))[0]
            if not basename.startswith('associations'):
                # Lecture d'un fichier attribute_set
                attr_set = EoleParser()
                attr_set.read(attr_file)
                for entity_name in attr_set.sections():
                    # construction des jeux d'attributs au nouveau format
                    new_set = EoleParser()
                    section_ok = False
                    for attr, value in attr_set.items(entity_name):
                        if not section_ok:
                            new_set.add_section('user_attrs')
                            section_ok = True
                        new_set.set('user_attrs', attr, value)
                    set_file = ""
                    # on regarde si ce jeu existe déjà
                    for setname, attrs in list(sets.items()):
                        if attrs == new_set._sections:
                            set_file = setname
                    if not set_file:
                        # écriture d'un nouveau fichier de jeux d'attributs
                        set_file = 'jeu_perso_%s' % file_index
                        file_index += 1
                        if entity_name == 'default':
                            # cas particulier du jeu par défaut, on regarde si il a été personnalisé
                            if not (attr_set.options('default') == ['FederationKey'] and
                                attr_set.get('default', 'FederationKey') == 'FederationKey'):
                                set_file = 'default_set'
                                file_index -= 1
                                if not 'default' in associations.sections():
                                    associations.add_section('default')
                                associations.set('default', 'attribute_set', 'default_set')
                            else:
                                # jeu d'attribut fourni par eole, on ne le prend pas en compte
                                continue
                        f_perso = open(os.path.join(config.ATTR_SET_DIR, '%s.ini' % set_file), 'w')
                        new_set.write(f_perso)
                        f_perso.close()
                        sets[set_file] = copy.copy(new_set._sections)
                        if set_file != 'default_set':
                            update_report("Jeu d'attributs personnalisé créé (%s.ini) :" % set_file)
                        else:
                            update_report("redéfinition du jeu d'attribut par défaut (%s.ini) :" % set_file)
                        for attr_local, attr_sent in new_set.items('user_attrs'):
                            update_report("attribut local %s : envoyé comme %s" % (attr_local, attr_sent), start_line="\n     ")
                    # ajout de l'association pour activer l'entité partenaire comme fournisseur d'identité
                    # si on a des données concernant le fonctionnement dans ce mode
                    if entity_name in associations.sections():
                        # ajout / mise à jour du jeu d'attributs pour cette entité
                        associations.set(entity_name, 'attribute_set', set_file)
    except Exception as e:
        import traceback
        traceback.print_exc()
        exit_err("EoleSSO: Erreur de mise à niveau des fichiers de jeux d'attributs", str(e))
    # stockage des associations
    if associations.sections() != []:
        assoc_file = os.path.join(config.ATTR_SET_DIR, 'associations.ini')
        try:
            f_assoc = open(assoc_file, 'w')
            associations.write(f_assoc)
            f_assoc.close()
            update_report("\nsauvegarde des associations avec les fournisseur d'identité dans %s" % assoc_file, "\n")
        except Exception as e:
            exit_err("Erreur lors de l'écriture du fichier %s" % assoc_file)

def run_update():
    # vérification d'éventuels fichiers backup
    for bak_file in ("default_ini.bak", "multi_rne_ini.bak"):
        if os.path.isfile("/usr/share/sso/attribute_sets/%s" % bak_file):
            shutil.move("/usr/share/sso/attribute_sets/%s" % bak_file,
                        "/usr/share/sso/attribute_sets/%s.ini" % bak_file[:bak_file.rindex('_')])
    # migration de la configuration si besoin
    assoc_files = glob.glob(os.path.join(SSO_DIR, 'attribute_sets/associations*.ini'))
    if not os.path.isdir(BACKUP_DIR) and assoc_files == []:
        print_orange("Mise à niveau des fichiers de configuration d'EoleSSO ...")
        os.makedirs(os.path.join(BACKUP_DIR, 'attribute_sets'))
        try:
            os.chdir(SSO_DIR)
            import config
            update_entity_id(config)
            udpate_attr_sets(config)
        except Exception as e:
            exit_err("Erreur recontrée lors de la mise à niveau de la configuration", str(e))
        save_report()

if __name__ == '__main__':
    run_update()
