#! /usr/bin/env python3
# -*- coding: utf-8 -*-

###########################################################################
#
# Eole NG - 2011
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# get_domains.py
#
# procédure de récupération des adresses de portails
# établissements via Zéphir
#
###########################################################################
import xmlrpc.client, sys, os, getpass
from configparser import ConfigParser
import config


class EoleParser(ConfigParser):
    """Sous classe de ConfigParser pour préserver la casse des options"""

    def optionxform(self, option_str):
        return option_str


def exit(msg):
    sys.exit('\n%s\n' % msg)

try:
    from zephir.zephir_conf.zephir_conf import adresse_zephir
except:
    exit("Ce serveur n'est pas enregistré sur Zéphir")

login = input('Entrez votre login Zéphir :')

con_ok = False
while not con_ok:
    passwd = getpass.getpass('Mot de passe (rien pour sortir):')
    if passwd == "":
        exit('Abandon')
    z_prox = xmlrpc.client.ServerProxy('https://%s:%s@%s:7080' % (login, passwd, adresse_zephir))
    try:
        req_ok, mods = z_prox.modules.get_module()
        con_ok = True
    except:
        print("! Erreur d'authentification ou droits insuffisants")


# modules existants
modules = {}
for mod in mods:
    modules[mod['id']] = mod['libelle']

# recherche des rne des serveurs répliqués
liste_rne = []
if os.path.isfile('/etc/ldap/replication.conf'):
    for line in file('/etc/ldap/replication.conf'):
        if line.strip().startswith('searchbase='):
            try:
                rne = line[line.index('ou=')+3:]
                rne = rne[:rne.index(',')]
                liste_rne.append(rne)
            except:
                continue
else:
    exit("Configuration de réplication non retrouvée")

data = {}
for rne in liste_rne:
    print("\nRecherche du domaine pour l'établissement %s ..." % rne)
    req_ok, serveurs = z_prox.serveurs.groupe_serveur({'rne':rne})
    if req_ok:
        for serv in serveurs:
            posh_url = cas_domainname = libelle_etab = None
            # on cherche seulement dans la configuration de scribe et amon(/amonecole)
            if ('amon' in modules[serv['module_actuel']] or \
                'scribe' in modules[serv['module_actuel']]):
                req_ok, conf_serv = z_prox.serveurs.get_config(serv['id'])
                if req_ok:
                    # on recherche les variables susceptibles d'indiquer l'adresse du portail
                    for var, value in list(conf_serv.items()):
                        if var == 'posh_url' and value:
                            posh_url = value
                        if var == 'cas_domainname' and value:
                            cas_domainname = value
                        if var == 'libelle_etab' and value:
                            libelle_etab = value
                    # essai de détermination de la forme du nom d'entité (si c'est l'adresse fqdn
                    # qui est utilisé, on ne pourra pas le retrouver)
                    if libelle_etab or cas_domainname:
                        if posh_url or cas_domainname:
                            store = False
                            entity_end = ":et-%s:1.0" % libelle_etab or cas_domainname
                            entity_end = entity_end.encode(config.encoding)
                            # correspondance entité/nom de domaine présumé
                            redirect_url = "https://%s" % posh_url or cas_domainname
                            print(" - serveur %s : %s" % (serv['id'], redirect_url))
                            if entity_end in data:
                                print("\n Un nom de domaine a déjà été trouvé pour cet entité (%s)" % data[entity_end])
                                response = input("Voulez vous le remplacer par %s ? [o/n]" % redirect_url)
                                if response.lower().startswith('o'):
                                    store = True
                            else:
                                store = True
                            if store:
                                data[entity_end] = redirect_url
                else:
                    print ("Récupération de la configuration du serveur %s impossible (Zéphir)")
                    continue
                if rne in data:
                    # adresse trouvée, on passe à l'établissement suivant
                    break
    else:
        print ("Récupération de la liste des seveurs de l'établissement impossible")

# sauvegarde dans le fichier d'associations
# on essaye de retrouver les correspondances entre
# les serveurs et les entités des metadata (nom de domaine)
assoc = EoleParser()
assoc.read(os.path.join(config.ATTR_SET_DIR, 'associations.ini'))
print("\n")

for entity_end in data:
    for entity_id in assoc.sections():
        if entity_id.endswith(entity_end):
            # entité retrouvée, on stocke l'url d'accès au portail
            assoc.set(entity_id, 'default_logout_url', data[entity_end])
            assoc.set(entity_id, 'force_logout_url', 'true')
            print(("Entité retrouvée : %s (redirection sur %s)" % (entity_id, data[entity_end])))

# sauvegarde du fichier mis à jour sous un autre_nom
assoc_temp = os.path.join(config.ATTR_SET_DIR, 'associations.domains')
try:
    f_assoc = open(assoc_temp, 'w')
    assoc.write(f_assoc)
    f_assoc.close()
except:
    exit("\nErreur lors de l'écriture du fichier d'associations temporaire :\n\t- %s\n" % assoc_temp)
print("\nnouveau fichier d'association créé : %s" % assoc_temp)
print("Vérifiez la validité des adresses ajoutées et remplacez le fichier")
print("associations.ini (du même répertoire) par celui-ci\n")
