#!/usr/bin/env python
#-*- coding:utf-8 -*-

###########################################################################
#
# Eole - 2015
# Copyright Pole de Competences Logiciels Libres
# (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# userstore.py
#
# Classe de gestion d'associations entre des utilisateurs et des
# identifiants externes au référentiel
#
###########################################################################

import os, stat
from eolesso.util import EoleParser
from page import log, trace

def normalize_section(section):
    if section == "default":
        # ConfigParser n'accepte pas qu'un section s'appelle 'default'
        section = "_default"
    return section

def normalize_branch(branch):
    if branch == '_default':
        branch = 'default'
    return branch

class UserStore():
    """Ressource de vérification de l'association d'un utilisateur dans la base locale
    """
    def __init__(self, store_dir, store_file):
        self.store_dir = store_dir
        self.store_file = os.path.join(store_dir, store_file)
        # structure pour recherche des utilisateurs locaux ayant enregistré leur identifiant
        self.user_store = EoleParser()
        self.check_store_perms()
        self.load_users()
        assert os.path.isfile(self.store_file), _("UserStore File could not be created ({0})").format(self.store_file)
        # structure pour recherche inversée (login local depuis login external)
        self.external_logins = {}
        for section in self.user_store.sections():
            branch = normalize_branch(section)
            for local_user, external_user in self.user_store.items(section):
                self.external_logins[external_user] = (branch, local_user)

    def check_store_perms(self):
        if not os.path.isdir(self.store_dir):
            os.makedirs(self.store_dir)
        if not os.path.isfile(self.store_file):
            f_users = open(self.store_file, 'w')
            f_users.close()
        # vérifie que seul root peut accéder au fichier des correspondances de login externes
        dir_perms = stat.S_IRWXU #(drwx------)
        os.chmod(self.store_dir, dir_perms)
        file_perms = stat.S_IRUSR|stat.S_IWUSR #(rw-------)
        os.chmod(self.store_file, file_perms)

    def save_users(self):
        f_users = open(self.store_file, 'w')
        self.user_store.write(f_users)
        f_users.close()

    def load_users(self):
        self.user_store.read(self.store_file)

    @trace
    def check_registered(self, username, search_branch):
        section = normalize_section(search_branch)
        if self.user_store.has_option(section, username):
            return "true"
        return "false"

    @trace
    def get_local_user(self, external_login):
        return self.external_logins.get(external_login, (None, None))

    @trace
    def get_external_user(self, username, search_branch):
        section = normalize_section(search_branch)
        if self.user_store.has_option(section, username):
            return self.user_store.get(section, username)
        else:
            return ''

    @trace
    def register_user(self, username, external_name, search_branch):
        section = normalize_section(search_branch)
        if external_name in self.external_logins:
            if self.external_logins[external_name] != (username, search_branch):
                # on ne permet pas de doublons dans les identifiants externes
                log.msg("register_user : %s (%s)" % (_("Error: this Id is already associated with a different user"), external_name))
                return False
            else:
                # déjà enregistré, on ne fait rien
                return True
        if not self.user_store.has_section(section):
            self.user_store.add_section(section)
        self.user_store.set(section, username, external_name)
        self.external_logins[external_name] = (search_branch, username)
        self.save_users()
        return True

