# -*- coding: UTF-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################

"""
Module chargé de la gestion des fenêtres pour les zones.
"""

import gtk, gtk.glade

from era.tool.IPy import IP
from era.noyau.erreurs import EmptyNameError, ExistingZoneError, TrigrammeError
from era.noyau.dpatterns import Singleton
from era.noyau.fwobjects import Zone, Extremite
from era.noyau.pool import library_store

from ihm_utils import (create_error_dialog, create_yes_no_dialog, 
                                                           make_template_string)
from gtk_models import set_empty_combo_model, set_combo_model


class ZoneDialogManager(Singleton):
    """ Classe qui va gérer l'édition de zones
    """

    def __init__(self, glade_file, matrix_model, zone = None):
        """
        glade : l'objet XML glade
        matrix_model : le modèle associé à la matrice de flux
        """
        self.matrix_model = matrix_model
        self.glade_file = glade_file

        # ATTENTION : il ne faut pas recréer une nouvelle référence pour la
        # fenêtre de dialog. On s'assure donc que self.dlg n'existe pas déjà
        # et seulement dans ce cas là, on récupère la widget.
        # RAPPEL : ZoneDialogManager est un Singleton !
        if 'dlg' not in self.__dict__:
            self.glade = gtk.glade.XML(self.glade_file, "zone_dialog_editor", "editeur")
            self.dlg = self.glade.get_widget('zone_dialog_editor')

            self.name_entry = self.glade.get_widget('zone_edit_name_entry')
            self.level_scale = self.glade.get_widget('zone_edit_level_scale')
            self.interface_entry = self.glade.get_widget('interface_entry')
            self.ip_var_entry = self.glade.get_widget('ip_variable_entry')
            self.ip_box = self.glade.get_widget('zone_dialog_ipbox')
            self.ip_radiobutton = self.glade.get_widget('ip_radiobutton')
            self.ip_var_radiobutton = self.glade.get_widget('ip_var_radiobutton')
            self.network_var_entry = self.glade.get_widget('network_variable_entry')
            self.network_radiobutton = self.glade.get_widget('network_radiobutton')
            self.network_var_radiobutton = self.glade.get_widget('network_var_radiobutton')
            self.network_box = self.glade.get_widget('zone_dialog_networkbox')
            self.netmask_var_entry = self.glade.get_widget('netmask_variable_entry')
            self.netmask_radiobutton = self.glade.get_widget('netmask_radiobutton')
            self.netmask_var_radiobutton = self.glade.get_widget('netmask_var_radiobutton')
            self.netmask_box = self.glade.get_widget('zone_dialog_netmaskbox')
#             self.bandwidth = self.glade.get_widget('zone_bandwidth_entry')

            #fonctions de remplissage automatique de network et netmask
            self.ip_entry = []
            self.netmask_entry = []
            self.network_entry = []
            for i in range(1,5):
                ip_entry = self.glade.get_widget('zone_edit_ip_entry%d'%i)
#                ip_entry.connect('focus_out_event',self.set_mask, i)
                ip_entry.connect('focus_out_event',self.set_network, i)
                self.ip_entry.append(ip_entry)
                netmask_entry = self.glade.get_widget('zone_edit_netmask_entry%d'%i)
                netmask_entry.connect('focus_out_event',self.set_network, i)
                self.netmask_entry.append(netmask_entry)
                self.network_entry.append(self.glade.get_widget('zone_edit_network_entry%d'%i))

            handlers = {"on_zone_dialog_editor_response" : self.validate_edit,
                        "on_zone_dialog_editor_close" : self.hide_and_flush,
                        "on_ip_variable_toggled" : self.ip_type_changed,
                        "on_ip_toggled" : self.ip_type_changed,
                        "on_network_radiobutton_toggled" : self.network_type_changed,
                        "on_network_var_radiobutton_toggled" : self.network_type_changed,
                        "on_netmask_radiobutton_toggled" : self.netmask_type_changed,
                        "on_netmask_var_radiobutton_toggled" : self.netmask_type_changed }

            self.glade.signal_autoconnect(handlers)

            self.dlg.connect('delete-event', self.hide_and_flush)

        self.template_ip = False
        self.template_network = False
        self.template_netmask = False
        self.ip_radiobutton.set_active(True)
        self.network_radiobutton.set_active(True)
        self.netmask_radiobutton.set_active(True)
        self.flush()
        if zone is not None:
            self.editing = True
            self.edited_zone = zone
            self.fill_dialog(zone)
        else:
            self.editing = False


    def ip_type_changed(self, *args):
        """Callback appelée pour déterminer quel type d'ip est entré
        """
        template_ip = not self.ip_radiobutton.get_active()
        self.ip_var_entry.set_sensitive(template_ip)
        self.ip_box.set_sensitive(not template_ip)
        self.template_ip = template_ip

    def network_type_changed(self, *args):
        """Callback appelée pour déterminer quel type de network est entré
        """
        template_network = not self.network_radiobutton.get_active()
        self.network_var_entry.set_sensitive(template_network)
        self.network_box.set_sensitive(not template_network)
        self.template_network = template_network

    def netmask_type_changed(self, *args):
        """Callback appelée pour déterminer quel type de netmask est entré
        """
        template_netmask = not self.netmask_radiobutton.get_active()
        self.netmask_var_entry.set_sensitive(template_netmask)
        self.netmask_box.set_sensitive(not template_netmask)
        self.template_netmask = template_netmask


    def set_network(self,entry,event,num):
        entry_ip = self.ip_entry[num-1]
        entry_mask = self.netmask_entry[num-1]
        entry_network = self.network_entry[num-1]
        ip = ".".join([entry.get_text() for entry in self.ip_entry])
        netmask = ".".join([entry.get_text() for entry in self.netmask_entry])
        try:
            network = IP(ip).make_net(netmask).strNormal(0).split('.')
            if len(network) != 4: raise Exception, "un network a quatre valeurs"
            for i in range(1,5):
                self.network_entry[i-1].set_text(network[i-1])
        except:
            if num == 4:
                result='0'
            elif entry_mask.get_text() == '255':
                result = entry_ip.get_text()
            else:
                result='0'
            entry_network.set_text(result)

    def set_mask(self,entry_ip,event,num):
        entry_netmask=self.netmask_entry[num-1]
        if num == 4:
        #if entry_network.get_text()=="0":
            entry_netmask.set_text("0")
        else:
            entry_netmask.set_text("255")
        self.set_network(entry_netmask, event, num)

    def fill_dialog(self, zone):
        """Remplit la fenêtre d'édition avec les données d'une zone précise
        """
        self.name_entry.set_text(zone.name)
        # On interdit de changer le nom de la zone
        self.name_entry.set_sensitive(0)
        self.interface_entry.set_text(zone.interface)
        self.level_scale.set_value(zone.level)

        if zone.network.startswith('%'):
            self.network_var_radiobutton.set_active(True)
            self.network_var_entry.set_text(zone.network)
        else :
             self.fill_ip(zone, "network")

        if zone.netmask.startswith('%'):
            self.netmask_var_radiobutton.set_active(True)
            self.netmask_var_entry.set_text(zone.netmask)
        else :
            self.fill_ip(zone,"netmask")

        # XXX FIXME : rajouter la propriété is_template() dans FwObject
        if zone.ip.startswith('%'):
            self.ip_var_radiobutton.set_active(True)
            self.ip_var_entry.set_text(zone.ip)
        else:
            self.fill_ip(zone, 'ip')


    def fill_ip(self, zone, type = 'ip'):
        """Remplit les gtk.Entry pour l'IP concernée avec celle de la zone
        """
        if type == 'ip':
            address = zone.ip
        elif type == 'network':
            address = zone.network
        else:
            address = zone.netmask

        widget_names = ['zone_edit_%s_entry%d'%(type, i) for i in range(1,5)]

        ip_parts = address.split('.')
        it = iter(ip_parts)
        for w_name in widget_names:
            entry = self.glade.get_widget(w_name)
            entry.set_text(it.next())


    def get_ip(self, type = 'ip'):
        """ Returns a string representing the IP contained in the text entries
        """
        assert type in ('ip', 'network', 'netmask')

        entry_var = self.glade.get_widget(type+"_variable_entry")

        if type == 'ip':
            template = self.template_ip
        elif type == 'network':
            template = self.template_network
        else :
            template = self.template_netmask

        if template:
            result = make_template_string(entry_var.get_text())

        else:
            widget_names = ['zone_edit_%s_entry%d'%(type, i) for i in range(1,5)]

            ip_parts = []
            for w_name in widget_names:
                entry = self.glade.get_widget(w_name)
                # XXX FIXME : faire assert 0 <= int(entry.get_text()) <= 255
                ip_parts.append(entry.get_text())

                result = '.'.join(ip_parts)
        return result


    def flush(self):
        """ Flushes all entries in the dialog
        """
        self.name_entry.set_sensitive(True)
        self.name_entry.set_text('')
        self.interface_entry.set_text('')
        self.flush_ip_entries()
        self.level_scale.set_value(0)
        self.ip_var_entry.set_text('')
        self.ip_radiobutton.set_active(True)
        self.network_var_entry.set_text('')
        self.network_radiobutton.set_active(True)
        self.netmask_var_entry.set_text('')
        self.netmask_radiobutton.set_active(True)
#         self.bandwidth.set_text('')

    def flush_ip_entries(self):
        """ Flushes all ip entries
        """
        for i in range(1,5):
#            entry_ip = self.glade.get_widget('zone_edit_ip_entry%d' % i)
            self.ip_entry[i-1].set_text('255')
            self.set_mask(None, None, i)
#        types = ('ip', 'network', 'netmask')
#        for t in types:
#            widget_names = ['zone_edit_%s_entry%d'%(t, i) for i in range(1,5)]
#            for w_name in widget_names:
#                entry = self.glade.get_widget(w_name)
#                entry.set_text('255')

    def hide_and_flush(self, *args):
        """ Cache et vide la fenêtre d'édition
        """
        self.dlg.hide()
        self.flush()

        return True


    def refresh_zonelevel(self, zone, default_policy, oldlevel):
        """
            - supprime les directives identiques à la nouvelle politique
            par défaut
            - indique la nouvelle politique par défaut dans le directive_store
            :param oldlevel: le level de la zone si rien n'est changé
            :param default_policy: la politique par défaut appliquée dans le
                                   directive_store
            :return: - True si on valide les changements
                     - False si on annule
        """
        # récupérer le flux correspondant
        # puis les directive_store de fwobjects
        # DirectiveStore.default_policy passe à False

        for flux in self.matrix_model.flux:
            # verifions que la zone ``edited_zone`` est bien
            # dans le flux, qu'il soit montant ou descendant
            if ((self.edited_zone == flux.zoneA and zone == flux.zoneB)
            or (self.edited_zone == flux.zoneB and zone == flux.zoneA)):
                dialog = create_yes_no_dialog(
                  _("Attention, toutes les directives standard entre %s et %s vont être supprimées, êtes-vous sûr de vouloir continuer ?" % \
                  (self.edited_zone.name, zone.name)))
                dialog.show_all()
                dialog.set_default_response(gtk.RESPONSE_NO)
                ret = dialog.run()
                dialog.destroy()
                # on teste le retour du dialogue
                if ret == gtk.RESPONSE_YES:

                    down_directive_store = flux.down_directives_store
                    for directive in down_directive_store.get_non_standard_directives():
                            down_directive_store.del_directive(down_directive_store.get_directive_index(directive))
                    # on applique set_default_policy_to_[true, false]
                    getattr(down_directive_store,
                            "set_default_policy_to_" + default_policy)()
                    return True
                else:
                    self.level_scale.set_value(oldlevel)
                    self.edited_zone.level = oldlevel
                    return False

    def validate_edit(self, dialog, response_code, *args):
        """ Callback called when validating the edit form
        """

        # On ne tient compte que si l'on clique sur OK
        if response_code != gtk.RESPONSE_OK:
            return self.hide_and_flush()


        name = self.name_entry.get_text()
        ip, network, netmask = [self.get_ip(t) for t in ('ip', 'network', 'netmask')]
        interface = self.interface_entry.get_text()
        level = int(self.level_scale.get_value())
#         bandwidth = self.bandwidth.get_text()
        # flag de validation de fermeture de la boite de dialogue de validation
        return_code = True
        if self.editing:
            self.edited_zone.ip = ip
            self.edited_zone.network = network
            self.edited_zone.netmask = netmask
            self.edited_zone.interface = interface
            if self.edited_zone.level != level:
                oldlevel = self.edited_zone.level
                self.edited_zone.level = level

                zones = self.matrix_model.real_zones()
                for zone in zones:
                    if zone == self.edited_zone:
                        continue
                    # si les zones n'étaient pas égales, et qu'elles le deviennent
                    if zone.strict_eq(self.edited_zone):
                        return_code = self.refresh_zonelevel(zone, "false", oldlevel)
                    # si les zones étaient égales, et qu'elles ne le sont plus
                    if zone.level == oldlevel:
                        return_code = self.refresh_zonelevel(zone, "true", oldlevel)

            self.edited_zone.notify_listeners()
#            self.edited_zone.bandwidth = bandwidth

            if self.edited_zone.name + "_restreint" in library_store.extremites:
                zone_rest = self.edited_zone.name + "_restreint"
                library_store.extremites[zone_rest].ip_list = [network]
                library_store.extremites[zone_rest].netmask = netmask

        else:
            zone = Zone(name, level, ip, network, netmask, interface)
            zone_rest = name + "_restreint"
            zone_restreinte = Extremite(zone, zone_rest, "zone restreinte",
                                        [zone.network], zone.netmask, subnet=1)
            try:
                self.matrix_model.add_zone(zone)
                library_store.add_extremite(zone_restreinte)
            except (EmptyNameError,
                    ExistingZoneError,
                    TrigrammeError), e:
                dlg = create_error_dialog(str(e),self.dlg)
                dlg.show_all()
                return

        if return_code:
            self.hide_and_flush()



    def show_dialog(self):
        """ Shows the handled dialog
        """
        self.dlg.show_all()


