# -*- coding: utf-8 -*-

"""Generic command line arguments parsers

Functions in this module provide standard ArgumentParser objects
usable as parents in a script parser.

If you need to override one of the options provided by a ArgumentParser object, you must use "conflict_handler='resolve'"

>>> import argparse
>>> from pyeole.scriptargs import container, logging
>>> parser = argparse.ArgumentParser(description="A command that make bidule", parents=[container(), logging()])
>>> parser.parse_args('-d'.split())
Namespace(container=None, debug=True, log_level='error', verbose=False)
>>> parser.parse_args('-c root'.split())
Namespace(container='root', debug=False, log_level='error', verbose=False)
>>> parser.parse_args('-c root --container mail'.split())
Namespace(container='mail', debug=False, log_level='error', verbose=False)
>>> parser.parse_args('-l info'.split())
Namespace(container=None, debug=False, log_level='info', verbose=False)
>>> parser.parse_args('-l critical -v'.split())
Namespace(container=None, debug=False, log_level='critical', verbose=True)
>>> parser.parse_args('-l critical -v -d'.split())
Namespace(container=None, debug=True, log_level='critical', verbose=True)
>>> parser.add_argument('-d', action='store_true', help='Run as daemon')
Traceback (most recent call last):
...
ArgumentError: argument -d/--debug: conflicting option string(s): -d
>>> parser = argparse.ArgumentParser(description="A command that make bidule", parents=[container(), logging()], conflict_handler='resolve')
>>> parser.add_argument('-d', action='store_true', help='Run as daemon')
_StoreTrueAction(option_strings=['-d'], dest='d', nargs=0, const=True, default=False, type=None, choices=None, help='Run as daemon', metavar=None)
>>> parser.parse_args('-d'.split())
Namespace(container=None, d=True, debug=False, log_level='error', verbose=False)
"""

from pyeole.i18n import i18n
import argparse

_ = i18n('scriptargs')

def container():
    """Command line arguments parser for containers

    @return: Parser for container options
    @rtype: L{argparse.ArgumentParser}
    """
    parser = argparse.ArgumentParser(description=_(u'Containers options'), add_help=False)
    group = parser.add_argument_group('container')
    group.add_argument('-c', '--container', help=_(u'Name of LXC container'))
    return parser

def logging(level='error'):
    """Command line arguments parser for logging

    @param level: default log level
    @type level: C{str}
    @return: Parser for logging options
    @rtype: L{argparse.ArgumentParser}
    """
    parser = argparse.ArgumentParser(description=_(u'Logging options'), add_help=False)
    group = parser.add_argument_group('logging')
    group.add_argument('-l', '--log-level', choices=['debug', 'info', 'warning', 'error', 'critical'], default=level, help=_(u'Log level'))
    group.add_argument('-v', '--verbose', action='store_true', help=_(u'Verbose mode, equivalent to -l info'))
    group.add_argument('-d', '--debug', action='store_true', help=_(u'Debug mode, equivalent to -l debug'))
    return parser

if __name__ == "__main__":
    import doctest
    doctest.testmod()
