#!/bin/bash
# CUPS print queue plugin for Nagios
# Written by John E. Vincent (nagios-plugs@lusis.org)
# Last Modified: 06-27-2006
#
# Description:
#
# This plugin will check the status of a remote CUPS
# print queue. It will provide the size of the queue
# and optionally the age of the queue
#

# Location of the lpstat command (if not in path)
LPSTAT="/usr/bin/lpstat"




# Don't change anything below here

# Nagios return codes
STATE_OK=0
STATE_WARNING=1
STATE_CRITICAL=2
STATE_UNKNOWN=3
STATE_DEPENDENT=4

if [ ! -x "$LPSTAT" ]
then
	echo "UNKNOWN: $LPSTAT not found or is not executable by the nagios user"
	exitstatus=$STATE_UNKNOWN
	exit $exitstatus
fi

PROGNAME=`basename $0`

print_usage() {
	echo "Usage: $PROGNAME -H <hostname> -T <s|b> -w <size warning level> -c <size critical level> -a <max age>"
	echo ""
	echo "Notes:"
	echo "-H: Hostname - Can be a hostname or IP address"
	echo "-T: Type of check - Can be queue size (s) or both queu size and queue age (b)"
	echo "-w: WARNING level for queue size"
	echo "-c: CRITICAL level for queue size"
	echo "-a: Max age of queue. Returns CRITICAL if jobs exists older than <max age> days"
	echo ""
}

print_help() {
	print_usage
	echo ""
	echo "This plugin will check the CUPS print queue on a remote (or local with -H localhost) CUPS server."
	echo "It can check both the size of the queue and the age of the oldest print job in the queue."
	echo "-w and -c are for reporting warning and critical levels of the queue size."
	echo "-a is optional for specifying the max age of a job in the print queue. Anything older thatn <max age>"
	echo "will return a CRITICAL"
	echo ""
	exit 0
}

check_queue_size()
{
                        if [ "$JOBCOUNT" -ge "$critlevel" ]
                        then
                                MESSAGE="CRITICAL: CUPS queue size : $JOBCOUNT| $PERFDATA"
                                exitstatus=$STATE_CRITICAL
                        elif [ "$JOBCOUNT" -ge "$warnlevel" ]
                        then
                                MESSAGE="WARNING: CUPS queue size : $JOBCOUNT| $PERFDATA"
                                exitstatus=$STATE_WARNING
                        else
                                MESSAGE="OK: CUPS queue size : $JOBCOUNT| $PERFDATA"
                                exitstatus=$STATE_OK
                        fi 
}

if [ $# -lt 4 ]; then
	print_usage
	exit $STATE_UNKNOWN
fi

exitstatus=$STATE_UNKNOWN #default

while test -n "$1"; do
	case "$1" in
		--help)
			print_help
			exit $STATE_OK
			;;
		-h)
			print_help
			exit $STATE_OK
			;;
		-H)
			hostname=$2
			shift
			;;
		-T)
			testtype=$2
			shift
			;;
		-w)
			warnlevel=$2
			shift
			;;
		-c)	
			critlevel=$2
			shift
			;;
		-a)	
			maxage=$2
			shift
			;;
	esac
	shift
done

# Check arguments for validity
if [ -z $hostname ]
then
	echo "You must specify a hostname (or localhost to test the local system)"
	print_usage
	exitstatus=$STATE_UNKNOWN
	exit $exitstatus
fi

if [[ -z $critlevel || -z $warnlevel ]] # Did we get warn and crit values?
then    
        echo "You must specify a warning and critical level"
	print_usage
        exitstatus=$STATE_UNKNOWN
        exit $exitstatus 
elif [ $critlevel -lt $warnlevel ] # Do the warn/crit values make sense?
then
	echo "CRITICAL value of $critlevel is less than WARNING level of $warnlevel"
	print_usage
	exitstatus=$STATE_UNKNOWN
	exit $exitstatus
fi

if [ -z $testtype ] # We require a test type
then
	echo "You must specify a test type"
	print_usage
	exitstatus=$STATE_UNKNOWN
	exit $exitstatus
elif [[ "$testtype" = [b]* && -z $maxage ]]
then
	echo "You must specify <max age> when using a test type of 'b'"
	print_usage
	exitstatus=$STATE_UNKNOWN
	exit $exitstatus
else
	
	JOBTMP=`mktemp -t lpstat.XXXXXX` # Create a tmpfile to store the lpstat results
	STALEJOBCOUNT=0 # default number of old jobs
	CURDATETS=`date +%s` # Get the current date as unixtime
	$LPSTAT -h $hostname -o > $JOBTMP # run the lpstat command against the host.
	if [ $? -ne 0 ]
	then
	    rm -rf $JOBTMP
		echo "UNKNOWN: lpstat command returned an error. Please test this script manually."
		exitstatus=$STATE_UNKNOWN
		exit $exitstatus
	fi
	JOBCOUNT=`wc -l < $JOBTMP` # populate the jobcount
	PERFDATA="print_jobs=$JOBCOUNT;$warnlevel;$critlevel;0"
	if [[ "$JOBCOUNT" -gt 0 && $maxage ]]
	then
		MAXAGETS=`echo "86400 * $maxage" | bc` # 86400 seconds in a day * maxage
		exec<$JOBTMP # read the file to determine job age
		while read PRINTJOB
		do
			JOBDATE=`echo $PRINTJOB | awk '{ print $4, $5, $6, $7, $8 }'` # Grab the job date from the job listing
			#JOBDATETS=`date --date="$JOBDATE" +%s` # Convert the job date to unixtime
			#DATEDIFF=`echo "($CURDATETS - $JOBDATETS)" | bc`
			#if [ $DATEDIFF -gt $MAXAGETS ]
			#then
			#	MESSAGE="CRITICAL: Some CUPS jobs are older than $maxage days| $PERFDATA"
			#	exitstatus=$STATE_CRITICAL
			#else
				check_queue_size
			#fi
		done
	else
		check_queue_size
	fi
	rm -rf $JOBTMP
fi

echo $MESSAGE
exit $exitstatus

