#!/usr/bin/python3.10
# -*- coding: utf-8 -*-

import getpass # pour cacher le mdp
import subprocess
import argparse # pour les options
import sys
from xmlrpc.client import Server
from xmlrpc import client as xmlrpclib # ligne à rajouter


def getserveur(user_zephir: str, mdp_user: str, serveur_address: str):
    class SpecialTransport(xmlrpclib.SafeTransport):
        def send_content(self, connection, request_body):
            connection.putheader("Referer", f"https://{serveur_address}:7080")  # URL dynamique du zephir
            connection.putheader("Origin", f"https://{serveur_address}:7080")  # URL dynamique du zephir
            connection.putheader("X-Requested-With", "XMLHttpRequest")
            connection.putheader("Content-Type", "text/xml")
            connection.putheader("Content-Length", str(len(request_body)))
            connection.endheaders()
            if request_body:
                connection.send(request_body)
    z = Server(
        f"https://{user_zephir}:{mdp_user}@{serveur_address}:7080",  # URL dynamique pour la connexion
        transport=SpecialTransport(),  # Appel de la classe de surcharge transport
        verbose=False,
        allow_none=True,
    )
    return z

def add(args):
    print(f"Ajout de l'archive pour ID Zephir {args.id_zephir} et UAI {args.uai}")
    uai_file = f"{args.uai}.tar.gz"
    z = getserveur(args.user_zephir,args.mdp_user,args.serveur_address)
    res = z.aaf.add_file(args.id_zephir, uai_file)
    print(res)

def delete(args):
    uai_file = f"{args.uai}.tar.gz"
    print(f"Suppression de l'archive aaf : {uai_file}")
    z = getserveur(args.user_zephir, args.mdp_user, args.serveur_address)
    res = z.aaf.del_file(uai_file)
    print(res)


def getlist(args):
    print(f"Listing des archives pour l'ID Zephir : {args.id_zephir}")
    z = getserveur(args.user_zephir, args.mdp_user, args.serveur_address)
    res = z.aaf.get_list(args.id_zephir)
    print(res)


def notify(args):
    print(f"Synchronisation avec {args.user_zephir}@{args.serveur_address}")
    z = getserveur(args.user_zephir, args.mdp_user, args.serveur_address)
    uai_file = f"{args.uai}.tar.gz"
    if args.annual:
        res = z.aaf.notify_upload_annual(uai_file)
    else:
        res = z.aaf.notify_upload(uai_file)
    print(res)

# Fonction pour gérer les options avec argparse
if __name__ == "__main__":

    parser = argparse.ArgumentParser(description="Outil de gestion archive AAF Zephir")
    # Option associate
    subparsers = parser.add_subparsers(dest="command", help="sub-command")

    add_subparser = subparsers.add_parser("add", help="Ajoute une archive aaf (.tar.gz) dans Zephir, sans prise en compte pour action.")
    add_subparser.set_defaults(func=add)
    add_subparser.add_argument("--serveur_address", type=str, required=True, help="Adresse du serveur Zephir (FQDN)")
    add_subparser.add_argument("--user_zephir", type=str, required=True, help="Utilisateur Zephir (droits requis)")
    add_subparser.add_argument("--mdp_user", type=str, required=True, help="Mot de passe de l'utilisateur Zephir")
    add_subparser.add_argument("--id_zephir", type=int, required=True)
    add_subparser.add_argument("--uai", type=str, required=True, help="Numéro UAI (nom de l'archive)")

    delete_subparser = subparsers.add_parser("delete", help="Suppression d'une archive de la base zephir")
    delete_subparser.set_defaults(func=delete)
    delete_subparser.add_argument("--serveur_address", type=str, required=True, help="Adresse du serveur Zephir (FQDN)")
    delete_subparser.add_argument("--user_zephir", type=str, required=True, help="Utilisateur Zephir (droits requis)")
    delete_subparser.add_argument("--mdp_user", type=str, required=True, help="Mot de passe de l'utilisateur Zephir")
    delete_subparser.add_argument("--uai", type=str, required=True, help="Numéro UAI (nom de l'archive)")

    getlist_subparser = subparsers.add_parser("getlist", help="Liste les archives associées à l'ID serveur indiqué")
    getlist_subparser.set_defaults(func=getlist)
    getlist_subparser.add_argument("--serveur_address", type=str, required=True, help="Adresse du serveur Zephir (FQDN)")
    getlist_subparser.add_argument("--user_zephir", type=str, required=True, help="Utilisateur Zephir (droits requis)")
    getlist_subparser.add_argument("--mdp_user", type=str, required=True, help="Mot de passe de l'utilisateur Zephir")
    getlist_subparser.add_argument("--id_zephir", type=int, required=True, help="Identifiant du serveur cible sur Zephir")

    notify_subparser = subparsers.add_parser("notify", help="Notifie Zephir d'un upload (.tar.gz) pour prise en compte (option --annual pour l’annuel).")
    notify_subparser.set_defaults(func=notify)
    notify_subparser.add_argument("--serveur_address", type=str, required=True, help="Adresse du serveur Zephir (FQDN)")
    notify_subparser.add_argument("--user_zephir", type=str, required=True, help="Utilisateur Zephir (droits requis)")
    notify_subparser.add_argument("--mdp_user", type=str, required=True, help="Mot de passe de l'utilisateur Zephir")
    notify_subparser.add_argument("--uai", type=str, required=True, help="Numéro UAI (nom de l'archive)")
    notify_subparser.add_argument("--annual", action="store_true")

    # Analyser les arguments, si rien envoyer l'aide.
    args = parser.parse_args()
    if not hasattr(args, "func"):
        parser.print_help()
    else:
        args.func(args)
