# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agent zephir pour la surveillance de la crl agriates
"""

from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager.data import HTMLData, TableData
from zephir.monitor.agentmanager import status, util
from twisted.internet.utils import getProcessOutput
import datetime

class Crl(Agent):
    def __init__(self, name, **params):
        Agent.__init__(self, name, **params)
        self.status = status.OK()
        self.table = TableData([
            ('ca_crl', "Autorité de certification de la CRL", {'align':'left'}, None),
            ('expire', "Date d'expiration", {'align':'left'}, None),
            ])
        title = HTMLData("<h3>Surveillance de la CRL</h3>")
        self.data = [title, self.table]

    def measure(self):
        self.status = status.OK()
        cmd = "/usr/sbin/ipsec"
        args = ["listcrls"]
        update = getProcessOutput(cmd, args)
        return update.addCallbacks(self.callback_expire, self.errback_expire)

    def callback_expire(self, response):
        # récupération de la date
        expire = {}
        if isinstance(response, bytes):
            response = response.decode()
        for line in response.split('\n'):
            line = line.strip()
            if line.startswith('issuer'):
                issuer_key = line.split("issuer:")[1].strip()
            if line.startswith('next'):
                exp_crl = line.split()[1:5]
                expire[issuer_key] = " ".join(exp_crl).split(',')[0]
        if expire == {}:
            self.status = status.Warn("date d'invalidation non trouvée")
            return [{'ca_crl': "",
                     'expire': "ipsec listcrls ne renvoie pas de date d'invalidation de la crl" }]
        # calcul de la date avant laquelle on est en warning (expiration - 20 minutes)
        res = []
        warn_delta = datetime.timedelta(seconds=60*20)
        date = datetime.datetime.now()
        status_err = False
        status_warn = False
        for issuer_key, exp_date in expire.items():
            expdate = util.parse_date(exp_date)
            warndate = expdate - warn_delta
            if date > expdate:
                status_err = True
                msg = "Au moins une CRL a expiré"
            elif date > warndate and not status_err:
                status_warn = True
                msg = "Au moins une CRL expire dans les 20 minutes"
            res.append({'ca_crl': issuer_key,
                        'expire': expdate.strftime('%d %b %Y %H:%M:%S')})
        if status_err:
            self.status = status.Error(msg)
        elif status_warn:
            self.status = status.Warn(msg)
        return res

    def errback_expire(self, err):
        msg = u"erreur de lecture de la date d'expiration"
        self.status = status.Warn(msg)
        return None

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value

    def check_status(self):
        return self.status

