package Color::Library::Dictionary::NBS_ISCC::M;

use strict;
use warnings;

use base qw/Color::Library::Dictionary/;

__PACKAGE__->_register_dictionary;

package Color::Library::Dictionary::NBS_ISCC::M;

=pod

=head1 NAME

Color::Library::Dictionary::NBS_ISCC::M - (NBS/ISCC M) Dictionary of Color

=head1 DESCRIPTION

    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


Color Name Dictionary: (M) Dictionary of Color

Maerz and Paul,
Dictionary of Color, 1st ed. 

This idiosyncratic dictionary's 3064 names map to 4401 colors, using 223 of the 267 centroids.

L<http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#M>

=head1 COLORS

	abbey                                   abbey                           #604e97

	absinthe [green]                        absinthegreen                   #8a9a5b

	absinthe yellow                         absintheyellow                  #b9b57d

	acacia                                  acacia                          #eae679

	acacia                                  acacia                          #b9b459

	academy blue                            academyblue                     #367588

	acajou                                  acajou                          #79443b

	acanthe                                 acanthe                         #6c541e

	acier                                   acier                           #8c8767

	ackermann's green                       ackermannsgreen                 #355e3b

	aconite violet                          aconiteviolet                   #86608e

	acorn                                   acorn                           #7e6d5a

	adamia                                  adamia                          #563c5c

	adelaide                                adelaide                        #32174d

	aden                                    aden                            #6aab8e

	admiral                                 admiral                         #291e29

	admiral                                 admiral                         #242124

	adobe                                   adobe                           #826644

	adrianople red                          adrianoplered                   #ab4e52

	adriatic                                adriatic                        #317873

	adust                                   adust                           #ae6938

	aero                                    aero                            #536878

	afghan                                  afghan                          #3e322c

	afghan red                              afghanred                       #be0032

	african                                 african                         #222222

	african brown                           africanbrown                    #3e322c

	afterglow                               afterglow                       #9690ab

	agate                                   agate                           #882d17

	agate                                   agate                           #80461b

	agate grey                              agategrey                       #b9b8b5

	ageratum blue                           ageratumblue                    #aa98a9

	ageratum violet                         ageratumviolet                  #796878

	air blue                                airblue                         #436b95

	airedale                                airedale                        #826644

	airedale                                airedale                        #7e6d5a

	airway                                  airway                          #70a3cc

	akbar                                   akbar                           #56070c

	akbar                                   akbar                           #79443b

	alabaster                               alabaster                       #ecd5c5

	alabaster                               alabaster                       #bfb8a5

	alamo                                   alamo                           #80461b

	alcanna                                 alcanna                         #882d17

	alcazar                                 alcazar                         #722f37

	alcazar                                 alcazar                         #882d17

	alderney                                alderney                        #826644

	alesan                                  alesan                          #977f73

	alesan                                  alesan                          #a67b5b

	alesan                                  alesan                          #958070

	alexandria blue                         alexandriablue                  #00a1c2

	alexandria blue                         alexandriablue                  #0067a5

	alfalfa                                 alfalfa                         #355e3b

	algerian                                algerian                        #6f4e37

	algerian red                            algerianred                     #56070c

	algonquin                               algonquin                       #be6516

	algonquin                               algonquin                       #ae6938

	alhambra                                alhambra                        #3a4b47

	alhambra                                alhambra                        #004b49

	alice blue                              aliceblue                       #91a3b0

	alkermes                                alkermes                        #be0032

	almond                                  almond                          #a67b5b

	almond                                  almond                          #958070

	almond                                  almond                          #c19a6b

	almond                                  almond                          #ae9b82

	almond blossom                          almondblossom                   #f9ccca

	almond blossom                          almondblossom                   #dea5a4

	almond blossom                          almondblossom                   #ead8d7

	almond blossom                          almondblossom                   #c4aead

	almond brown                            almondbrown                     #a67b5b

	almond brown                            almondbrown                     #958070

	almond brown                            almondbrown                     #c19a6b

	almond brown                            almondbrown                     #ae9b82

	almond green                            almondgreen                     #5e716a

	almond [pink]                           almondpink                      #f9ccca

	almond [pink]                           almondpink                      #dea5a4

	almond [pink]                           almondpink                      #ead8d7

	almond [pink]                           almondpink                      #c4aead

	aloes [green]                           aloesgreen                      #8da399

	aloma                                   aloma                           #a67b5b

	aloma                                   aloma                           #c19a6b

	aloma                                   aloma                           #826644

	alpine                                  alpine                          #536878

	alpine green                            alpinegreen                     #3a4b47

	aluminum                                aluminum                        #b9b8b5

	aluminum                                aluminum                        #848482

	amaranth                                amaranth                        #702963

	amaranthine                             amaranthine                     #673147

	amaranth pink                           amaranthpink                    #de6fa1

	amaranth purple                         amaranthpurple                  #78184a

	amarna                                  amarna                          #00543d

	amarna                                  amarna                          #1b4d3e

	amarylis                                amarylis                        #ab4e52

	amarylis                                amarylis                        #905d5d

	amazon                                  amazon                          #3b7861

	amber brown                             amberbrown                      #ae6938

	amberglow                               amberglow                       #ae6938

	amberglow                               amberglow                       #a67b5b

	amber white                             amberwhite                      #f3e5ab

	amber white                             amberwhite                      #c2b280

	amber white                             amberwhite                      #ebe8a4

	amber white                             amberwhite                      #b9b57d

	amber white                             amberwhite                      #dadfb7

	amber [yellow]                          amberyellow                     #f8de7e

	amber [yellow]                          amberyellow                     #c9ae5d

	ambrosia                                ambrosia                        #79443b

	ambulance                               ambulance                       #43302e

	amelie                                  amelie                          #3e322c

	american beauty                         americanbeauty                  #722f37

	american green                          americangreen                   #679267

	american green                          americangreen                   #355e3b

	american green                          americangreen                   #5e716a

	amethyst [violet]                       amethystviolet                  #86608e

	amethyst [violet]                       amethystviolet                  #563c5c

	amethyst [violet]                       amethystviolet                  #796878

	amparo blue                             amparoblue                      #0067a5

	amparo blue                             amparoblue                      #6c79b8

	amparo purple                           amparopurple                    #86608e

	amulet                                  amulet                          #004b49

	anamite                                 anamite                         #8c8767

	anatolia                                anatolia                        #56070c

	anatta                                  anatta                          #d9a6a9

	andorra                                 andorra                         #3e1d1e

	andorra                                 andorra                         #43302e

	andorre                                 andorre                         #679267

	andorre                                 andorre                         #5e716a

	andover green                           andovergreen                    #515744

	andrinople berries                      andrinopleberries               #eaa221

	andrinople berries                      andrinopleberries               #e3a857

	anemone                                 anemone                         #aa8a9e

	angel red                               angelred                        #79443b

	animal rouge                            animalrouge                     #be0032

	animal rouge                            animalrouge                     #bc3f4a

	annapolis                               annapolis                       #555555

	annatto                                 annatto                         #d9a6a9

	annotto                                 annotto                         #d9a6a9

	antelope                                antelope                        #967117

	antimony yellow                         antimonyyellow                  #e3a857

	antique                                 antique                         #967117

	antique                                 antique                         #6c541e

	antique brass                           antiquebrass                    #867e36

	antique brass                           antiquebrass                    #8c8767

	antique bronze                          antiquebronze                   #826644

	antique brown                           antiquebrown                    #6f4e37

	antique drab                            antiquedrab                     #826644

	antique drab                            antiquedrab                     #967117

	antique drab                            antiquedrab                     #6c541e

	antique fuchsia                         antiquefuchsia                  #86608e

	antique fuchsia                         antiquefuchsia                  #915c83

	antique gold                            antiquegold                     #be8a3d

	antique gold                            antiquegold                     #996515

	antique green                           antiquegreen                    #3a4b47

	antique red                             antiquered                      #cb6d51

	antique red                             antiquered                      #9e4732

	antique red                             antiquered                      #b4745e

	antique red                             antiquered                      #79443b

	antique ruby                            antiqueruby                     #841b2d

	antwerp blue                            antwerpblue                     #2e8495

	antwerp blue                            antwerpblue                     #436b95

	antwerp brown                           antwerpbrown                    #483c32

	antwerp red                             antwerpred                      #79443b

	apache                                  apache                          #6f4e37

	aphrodite                               aphrodite                       #66ada4

	appleblossom [pink]                     appleblossompink                #905d5d

	apple-fallow                            applefallow                     #841b2d

	apple-fallow                            applefallow                     #ab4e52

	apple green                             applegreen                      #8a9a5b

	apricot                                 apricot                         #d99058

	apricot yellow                          apricotyellow                   #f8de7e

	apricot yellow                          apricotyellow                   #c9ae5d

	aquagreen                               aquagreen                       #93c592

	aquamarine                              aquamarine                      #66aabc

	aquamarine                              aquamarine                      #91a3b0

	arab [brown]                            arabbrown                       #80461b

	arabesque                               arabesque                       #cb6d51

	arabian brown                           arabianbrown                    #80461b

	arabian brown                           arabianbrown                    #6f4e37

	arabian red                             arabianred                      #56070c

	araby                                   araby                           #b4745e

	araby                                   araby                           #a87c6d

	aragon                                  aragon                          #79443b

	arbutus                                 arbutus                         #ea9399

	arbutus                                 arbutus                         #e4717a

	arbutus                                 arbutus                         #e68fac

	arbutus                                 arbutus                         #de6fa1

	arcadian green                          arcadiangreen                   #93c592

	archel                                  archel                          #a8516e

	archil                                  archil                          #a8516e

	arctic                                  arctic                          #222222

	arctic blue                             arcticblue                      #536878

	ardoise                                 ardoise                         #536878

	ardoise                                 ardoise                         #796878

	argali                                  argali                          #5b504f

	argali                                  argali                          #555555

	argent                                  argent                          #848482

	argentina                               argentina                       #422518

	argus brown                             argusbrown                      #6f4e37

	argyle purple                           argylepurple                    #aa8a9e

	argyle purple                           argylepurple                    #836479

	arizona                                 arizona                         #c19a6b

	armada                                  armada                          #43302e

	armada                                  armada                          #3e322c

	armenian blue                           armenianblue                    #00a1c2

	armenian blue                           armenianblue                    #0067a5

	armenian bole                           armenianbole                    #79443b

	armenian red                            armenianred                     #80461b

	armenian stone                          armenianstone                   #436b95

	army brown                              armybrown                       #a67b5b

	arno blue                               arnoblue                        #367588

	arno blue                               arnoblue                        #004958

	arnotto                                 arnotto                         #d9a6a9

	arona                                   arona                           #36454f

	arrowwood                               arrowwood                       #967117

	arsenate                                arsenate                        #007959

	art brown                               artbrown                        #422518

	artemesia green                         artemesiagreen                  #7d8984

	art gray, art grey                      artgrayartgrey                  #8b8589

	art gray, art grey                      artgrayartgrey                  #848482

	art green                               artgreen                        #8a9a5b

	art green                               artgreen                        #4a5d23

	artichoke green                         artichokegreen                  #8f9779

	artificial vermilion                    artificialvermilion             #915f6d

	artillery                               artillery                       #be0032

	ascot tan                               ascottan                        #635147

	ashes of rose                           ashesofrose                     #915f6d

	ash [grey]                              ashgrey                         #b2beb5

	asmalte                                 asmalte                         #00416a

	asmalte                                 asmalte                         #436b95

	asparagus green                         asparagusgreen                  #dadfb7

	aspen green                             aspengreen                      #5e716a

	asphalt                                 asphalt                         #555555

	asphaltum                               asphaltum                       #483c32

	asphodel green                          asphodelgreen                   #8a9a5b

	aster                                   aster                           #796878

	aster purple                            asterpurple                     #78184a

	athenia                                 athenia                         #905d5d

	atlantic                                atlantic                        #36454f

	atlantis                                atlantis                        #004958

	atlantis                                atlantis                        #36454f

	atmosphere                              atmosphere                      #c2ac99

	atmosphere                              atmosphere                      #ae9b82

	atmosphere                              atmosphere                      #bfb8a5

	atramentous                             atramentous                     #51585e

	attar of roses                          attarofroses                    #b4745e

	attar of roses                          attarofroses                    #a87c6d

	aubergine                               aubergine                       #291e29

	aubergine                               aubergine                       #242124

	auburn                                  auburn                          #6f4e37

	aubusson                                aubusson                        #722f37

	aucuba                                  aucuba                          #867e36

	aureolin                                aureolin                        #fada5e

	aureolin                                aureolin                        #d4af37

	auricula purple                         auriculapurple                  #702963

	auripigmentum                           auripigmentum                   #fada5e

	auripigmentum                           auripigmentum                   #d4af37

	auripigmentum                           auripigmentum                   #f8de7e

	auripigmentum                           auripigmentum                   #c9ae5d

	aurora [orange]                         auroraorange                    #f88379

	aurora [orange]                         auroraorange                    #e66761

	aurora [orange]                         auroraorange                    #cb6d51

	aurora red                              aurorared                       #ab4e52

	aurora red                              aurorared                       #e66761

	aurora red                              aurorared                       #cb6d51

	aurora yellow                           aurorayellow                    #ed872d

	aurora yellow                           aurorayellow                    #d99058

	aurora yellow                           aurorayellow                    #eaa221

	aurora yellow                           aurorayellow                    #e3a857

	aurore                                  aurore                          #dea5a4

	aurum                                   aurum                           #967117

	australian pine                         australianpine                  #355e3b

	auteuil                                 auteuil                         #e3a857

	autumn                                  autumn                          #5b5842

	autumn                                  autumn                          #363527

	autumn blonde                           autumnblonde                    #977f73

	autumn blonde                           autumnblonde                    #6f4e37

	autumn blonde                           autumnblonde                    #958070

	autumn blonde                           autumnblonde                    #635147

	autumn brown                            autumnbrown                     #483c32

	autumn glory                            autumnglory                     #722f37

	autumn glory                            autumnglory                     #9e4732

	autumn green                            autumngreen                     #8a9a5b

	autumn green                            autumngreen                     #4a5d23

	autumn leaf                             autumnleaf                      #be6516

	autumn leaf                             autumnleaf                      #ae6938

	autumn leaf                             autumnleaf                      #80461b

	autumn oak                              autumnoak                       #674c47

	avellaneous                             avellaneous                     #a67b5b

	avellaneous                             avellaneous                     #996515

	avignon berries                         avignonberries                  #eaa221

	avignon berries                         avignonberries                  #e3a857

	azalea                                  azalea                          #905d5d

	aztec                                   aztec                           #996515

	aztec                                   aztec                           #c19a6b

	aztec                                   aztec                           #826644

	aztec maroon                            aztecmaroon                     #915f6d

	azulin                                  azulin                          #536878

	baby blue                               babyblue                        #91a3b0

	baby blue                               babyblue                        #b4bcc0

	baby blue                               babyblue                        #81878b

	baby pink                               babypink                        #f4c2c2

	baby pink                               babypink                        #d9a6a9

	baby rose                               babyrose                        #f88379

	bacchus                                 bacchus                         #a8516e

	bachelor button                         bachelorbutton                  #e4717a

	bagdad                                  bagdad                          #00304e

	bagdad                                  bagdad                          #36454f

	bagdad                                  bagdad                          #202830

	bakst green                             bakstgreen                      #1b4d3e

	balge yellow                            balgeyellow                     #fada5e

	ball lake                               balllake                        #be0032

	ball lake                               balllake                        #bc3f4a

	balsam                                  balsam                          #36454f

	balsam green                            balsamgreen                     #222222

	baltic                                  baltic                          #3a4b47

	baltic                                  baltic                          #004b49

	bambino                                 bambino                         #91a3b0

	bamboo                                  bamboo                          #c19a6b

	bamboo                                  bamboo                          #826644

	banana                                  banana                          #c2b280

	banner                                  banner                          #4c516d

	banshee                                 banshee                         #3a4b47

	barberry                                barberry                        #841b2d

	barberry                                barberry                        #ab4e52

	bark                                    bark                            #3b3121

	barwood                                 barwood                         #ab4e52

	basketball                              basketball                      #9065ca

	bastard saffron                         bastardsaffron                  #be0032

	bat                                     bat                             #555555

	bath stone                              bathstone                       #fad6a5

	battleship grey                         battleshipgrey                  #848482

	bay                                     bay                             #6f4e37

	bayou                                   bayou                           #3a4b47

	beach                                   beach                           #8c8767

	beach                                   beach                           #8a8776

	beach tan                               beachtan                        #635147

	bear                                    bear                            #555555

	bear's hair                             bearshair                       #555555

	bearskin [grey]                         bearskingrey                    #555555

	beaucaire                               beaucaire                       #007791

	beaucaire                               beaucaire                       #367588

	beaver                                  beaver                          #5b504f

	beaverpelt                              beaverpelt                      #57554c

	beech                                   beech                           #5b5842

	beech                                   beech                           #57554c

	beechnut                                beechnut                        #635147

	beef's blood                            beefsblood                      #79443b

	beeswax                                 beeswax                         #826644

	beetle                                  beetle                          #515744

	begonia                                 begonia                         #e4717a

	begonia rose                            begoniarose                     #ab4e52

	beige                                   beige                           #f3e5ab

	beige                                   beige                           #c2b280

	beige soire                            beigesoire                      #c19a6b

	belladonna                              belladonna                      #563c5c

	belleek                                 belleek                         #fad6a5

	belleek                                 belleek                         #ae9b82

	belleek                                 belleek                         #c2b280

	belleek                                 belleek                         #bfb8a5

	bellemonte                              bellemonte                      #ab4e52

	bellflower                              bellflower                      #272458

	berlin brown                            berlinbrown                     #905d5d

	berlin brown                            berlinbrown                     #674c47

	berlin red                              berlinred                       #79443b

	bermuda                                 bermuda                         #e4717a

	beryl                                   beryl                           #66aabc

	beryl blue                              berylblue                       #66aabc

	beryl green                             berylgreen                      #66ada4

	biarritz                                biarritz                        #ae9b82

	biarritz                                biarritz                        #967117

	big four yellow, big 4 yellow           bigfouryellowbig4yellow         #f38400

	big four yellow, big 4 yellow           bigfouryellowbig4yellow         #ed872d

	billiard                                billiard                        #1b4d3e

	biscay green                            biscaygreen                     #8a9a5b

	biscuit                                 biscuit                         #a67b5b

	biscuit                                 biscuit                         #958070

	biscuit                                 biscuit                         #c19a6b

	biscuit                                 biscuit                         #ae9b82

	bishop                                  bishop                          #604e81

	bishop's purple                         bishopspurple                   #915c83

	bishop's violet                         bishopsviolet                   #915c83

	biskra                                  biskra                          #483c32

	bismarck                                bismarck                        #79443b

	bismarck brown                          bismarckbrown                   #826644

	bison                                   bison                           #483c32

	bisque                                  bisque                          #fad6a5

	bisque                                  bisque                          #bfb8a5

	bister                                  bister                          #635147

	bister                                  bister                          #7e6d5a

	bistre                                  bistre                          #635147

	bistre                                  bistre                          #7e6d5a

	bistre green                            bistregreen                     #ab9144

	bistre green                            bistregreen                     #a18f60

	bittersweet                             bittersweet                     #aa381e

	bittersweet                             bittersweet                     #9e4732

	bittersweet orange                      bittersweetorange               #be6516

	bittersweet pink                        bittersweetpink                 #f88379

	bitumen                                 bitumen                         #483c32

	bladder green                           bladdergreen                    #867e36

	blaze                                   blaze                           #ab4e52

	blaze                                   blaze                           #9e4732

	bleu de lyons                           bleudelyons                     #00416a

	bleu de lyons                           bleudelyons                     #436b95

	bleu d'orient                           bleudorient                     #0067a5

	bleu louise                             bleulouise                      #317873

	bleu louise                             bleulouise                      #367588

	bleu pass                              bleupass                        #91a3b0

	bleu pass                              bleupass                        #536878

	bleu pass                              bleupass                        #81878b

	blond                                   blond                           #c19a6b

	blond                                   blond                           #a18f60

	blondine                                blondine                        #7e6d5a

	blonket                                 blonket                         #b4bcc0

	blood red                               bloodred                        #bc3f4a

	blood red                               bloodred                        #ab4e52

	blossom                                 blossom                         #dea5a4

	blue ashes                              blueashes                       #436b95

	blue aster                              blueaster                       #436b95

	bluebell                                bluebell                        #436b95

	bluebell                                bluebell                        #00304e

	blue bice                               bluebice                        #436b95

	blue bird, bluebird                     bluebirdbluebird                #436b95

	blue devil                              bluedevil                       #4c516d

	blue fox                                bluefox                         #5c504f

	blue grass                              bluegrass                       #004b49

	blue haze                               bluehaze                        #555555

	bluejay                                 bluejay                         #00416a

	bluejay                                 bluejay                         #00304e

	blue jewel                              bluejewel                       #007791

	blue lavender                           bluelavender                    #8c92ac

	blue lotus                              bluelotus                       #70a3cc

	blue ochre                              blueochre                       #5e716a

	blue ochre                              blueochre                       #536878

	blue orchid                             blueorchid                      #536878

	blue sapphire                           bluesapphire                    #367588

	blue spruce                             bluespruce                      #3b7861

	blue spruce                             bluespruce                      #5e716a

	bluet                                   bluet                           #436b95

	blue turquoise                          blueturquoise                   #66ada4

	blue turquoise                          blueturquoise                   #66aabc

	blue untramarine ash                    blueuntramarineash              #436b95

	blue verditer                           blueverditer                    #436b95

	blunket                                 blunket                         #b4bcc0

	blush                                   blush                           #d9a6a9

	blush                                   blush                           #a67b5b

	blush rose                              blushrose                       #905d5d

	boa                                     boa                             #8a9a5b

	boa                                     boa                             #dadfb7

	boa                                     boa                             #93c592

	bobolink                                bobolink                        #7e6d5a

	bois de rose                            boisderose                      #905d5d

	bokhara                                 bokhara                         #673147

	bole                                    bole                            #79443b

	bole armoniack                          bolearmoniack                   #79443b

	bolus                                   bolus                           #79443b

	bombay                                  bombay                          #80461b

	bombay                                  bombay                          #6f4e37

	bone brown                              bonebrown                       #6c541e

	bone brown                              bonebrown                       #3b3121

	bonfire                                 bonfire                         #bc3f4a

	bonito                                  bonito                          #5b504f

	bonnie blue                             bonnieblue                      #00a1c2

	boreal                                  boreal                          #79443b

	bosphorus                               bosphorus                       #3a4b47

	botticelli                              botticelli                      #702963

	bottle green                            bottlegreen                     #3a4b47

	bottle green                            bottlegreen                     #004b49

	bougainville                            bougainville                    #86608e

	bougainville                            bougainville                    #915c83

	boulevard                               boulevard                       #5b504f

	bouquet green                           bouquetgreen                    #6aab8e

	bourgeon                                bourgeon                        #8a9a5b

	box green                               boxgreen                        #8a9a5b

	bracken                                 bracken                         #6c541e

	bracken                                 bracken                         #3b3121

	bramble                                 bramble                         #2f2140

	bran                                    bran                            #a67b5b

	bran                                    bran                            #c19a6b

	brass                                   brass                           #c9ae5d

	brazen yellow                           brazenyellow                    #c9ae5d

	brazil brown                            brazilbrown                     #43302e

	brazil [red]                            brazilred                       #9e4732

	brazilwood                              brazilwood                      #ab4e52

	brazilwood                              brazilwood                      #905d5d

	bremen blue                             bremenblue                      #317873

	bremen green                            bremengreen                     #679267

	brewster green                          brewstergreen                   #4a5d23

	briar                                   briar                           #80461b

	briar                                   briar                           #6f4e37

	briarwood                               briarwood                       #3e322c

	brickdust                               brickdust                       #841b2d

	brick red                               brickred                        #80461b

	brick red                               brickred                        #6f4e37

	bridal rose                             bridalrose                      #c08081

	brigand                                 brigand                         #bc3f4a

	brigand                                 brigand                         #ab4e52

	brigand                                 brigand                         #d9603b

	brigand                                 brigand                         #cb6d51

	brimstone [yellow]                      brimstoneyellow                 #eae679

	brittany                                brittany                        #367588

	brittany                                brittany                        #436b95

	brittany                                brittany                        #536878

	broccoli brown                          broccolibrown                   #5b504f

	broncho                                 broncho                         #483c32

	broncho                                 broncho                         #6c541e

	broncho                                 broncho                         #3b3121

	bronze                                  bronze                          #826644

	bronze brown                            bronzebrown                     #5b504f

	bronze clair                            bronzeclair                     #ae9b82

	bronze green                            bronzegreen                     #515744

	bronze lustre                           bronzelustre                    #5b5842

	bronze nude                             bronzenude                      #6c541e

	bronze red                              bronzered                       #ab4e52

	bronzesheen                             bronzesheen                     #5b5842

	bronze yellow                           bronzeyellow                    #d99058

	bronze yellow                           bronzeyellow                    #be8a3d

	brown bay                               brownbay                        #635147

	brown bread                             brownbread                      #80461b

	brown bread                             brownbread                      #6f4e37

	brown madder                            brownmadder                     #79443b

	brown ochre                             brownochre                      #d99058

	brown pink                              brownpink                       #be8a3d

	brown pink                              brownpink                       #996515

	brown red                               brownred                        #79443b

	brown stone                             brownstone                      #635147

	brownstone                              brownstone                      #635147

	brown sugar                             brownsugar                      #6f4e37

	brown sugar                             brownsugar                      #635147

	brown sugar                             brownsugar                      #826644

	brown sugar                             brownsugar                      #7e6d5a

	bruges                                  bruges                          #004958

	bruges                                  bruges                          #36454f

	brushwood                               brushwood                       #635147

	brussels brown                          brusselsbrown                   #6f4e37

	brussels brown                          brusselsbrown                   #635147

	brussels brown                          brusselsbrown                   #826644

	brussels brown                          brusselsbrown                   #7e6d5a

	buccaneer                               buccaneer                       #bc3f4a

	buccaneer                               buccaneer                       #ab4e52

	buccaneer                               buccaneer                       #aa381e

	buccaneer                               buccaneer                       #9e4732

	buckskin                                buckskin                        #977f73

	buckskin                                buckskin                        #958070

	buckthorn berries                       buckthornberries                #eaa221

	buckthorn berries                       buckthornberries                #e3a857

	buckthorn brown                         buckthornbrown                  #be8a3d

	buckthorn brown                         buckthornbrown                  #996515

	buckthorn brown                         buckthornbrown                  #c19a6b

	buckthorn brown                         buckthornbrown                  #826644

	buckthorn brown                         buckthornbrown                  #ab9144

	buckthorn brown                         buckthornbrown                  #a18f60

	buckthorn brown                         buckthornbrown                  #967117

	buddha                                  buddha                          #bc3f4a

	buddha                                  buddha                          #ab4e52

	bud green                               budgreen                        #44944a

	buff                                    buff                            #e3a857

	buffalo                                 buffalo                         #826644

	bulgare                                 bulgare                         #905d5d

	bunny                                   bunny                           #826644

	bure                                    bure                            #996515

	bure                                    bure                            #c19a6b

	bure                                    bure                            #826644

	burel                                   burel                           #6f4e37

	burgundy                                burgundy                        #50404d

	burgundy                                burgundy                        #291e29

	burgundy violet                         burgundyviolet                  #875692

	burlwood                                burlwood                        #958070

	burlwood                                burlwood                        #ae9b82

	burma                                   burma                           #ae6938

	burmese gold                            burmesegold                     #cb6d51

	burmese ruby                            burmeseruby                     #722f37

	burn blue                               burnblue                        #bcd4e6

	burn blue                               burnblue                        #91a3b0

	burn blue                               burnblue                        #c0c8e1

	burn blue                               burnblue                        #8c92ac

	burnished gold                          burnishedgold                   #be8a3d

	burnished gold                          burnishedgold                   #996515

	burnished gold                          burnishedgold                   #c19a6b

	burnished gold                          burnishedgold                   #ab9144

	burnished gold                          burnishedgold                   #967117

	burnous                                 burnous                         #fad6a5

	burnt                                   burnt                           #a18f60

	burnt almond                            burntalmond                     #6f4e37

	burnt carmine                           burntcarmine                    #841b2d

	burnt carmine                           burntcarmine                    #ab4e52

	burnt coral                             burntcoral                      #cb6d51

	burnt crimson lake                      burntcrimsonlake                #841b2d

	burnt crimson lake                      burntcrimsonlake                #ab4e52

	burnt italian earth                     burntitalianearth               #9e4732

	burnt italian ochre                     burntitalianochre               #9e4732

	burnt lake                              burntlake                       #841b2d

	burnt lake                              burntlake                       #ab4e52

	burnt ochre                             burntochre                      #cb6d51

	burnt orange                            burntorange                     #cb6d51

	burnt roman ochre                       burntromanochre                 #ed872d

	burnt rose                              burntrose                       #56070c

	burnt rose                              burntrose                       #79443b

	burnt russet                            burntrusset                     #722f37

	burnt russet                            burntrusset                     #79443b

	burnt sienna                            burntsienna                     #9e4732

	burnt terre verte                       burntterreverte                 #6f4e37

	burnt umber                             burntumber                      #6f4e37

	buttercup [yellow]                      buttercupyellow                 #c2b280

	butterfly                               butterfly                       #b9b459

	butternut                               butternut                       #6c541e

	butterscotch                            butterscotch                    #6f4e37

	butterscotch                            butterscotch                    #826644

	byron                                   byron                           #483c32

	byzantine                               byzantine                       #ce4676

	byzantium                               byzantium                       #702963

	byzantium                               byzantium                       #5d3954

	cabaret                                 cabaret                         #6aab8e

	cabbage green                           cabbagegreen                    #7d8984

	cacao                                   cacao                           #826644

	cacao brown                             cacaobrown                      #80461b

	cacha                                   cacha                           #6f4e37

	cachou                                  cachou                          #6f4e37

	cactus                                  cactus                          #355e3b

	cadet                                   cadet                           #536878

	cadet blue                              cadetblue                       #436b95

	cadet grey                              cadetgrey                       #81878b

	cadmium carmine                         cadmiumcarmine                  #bc3f4a

	cadmium green                           cadmiumgreen                    #007959

	cadmium lemon                           cadmiumlemon                    #dcd300

	cadmium lemon                           cadmiumlemon                    #e9e450

	cadmium orange                          cadmiumorange                   #ed872d

	cadmium purple                          cadmiumpurple                   #841b2d

	cadmium purple                          cadmiumpurple                   #ab4e52

	cadmium vermilion                       cadmiumvermilion                #bc3f4a

	cadmium vermilion                       cadmiumvermilion                #ab4e52

	cadmium yellow                          cadmiumyellow                   #ed872d

	cadmium yellow                          cadmiumyellow                   #d99058

	cadmium yellow                          cadmiumyellow                   #eaa221

	cadmium yellow                          cadmiumyellow                   #e3a857

	caen stone                              caenstone                       #fad6a5

	caeruleum                               caeruleum                       #0085a1

	caeruleum                               caeruleum                       #00a1c2

	caf-au-lait                            cafaulait                       #977f73

	caf-au-lait                            cafaulait                       #a67b5b

	caf-au-lait                            cafaulait                       #958070

	caf crme                              cafcrme                         #a67b5b

	caf crme                              cafcrme                         #6f4e37

	caf crme                              cafcrme                         #826644

	caf noir                               cafnoir                         #483c32

	cairo                                   cairo                           #00416a

	calabash                                calabash                        #826644

	calabash                                calabash                        #967117

	calamine blue                           calamineblue                    #66aabc

	caldera                                 caldera                         #79443b

	caledonian brown                        caledonianbrown                 #80461b

	caliaturwood                            caliaturwood                    #ab4e52

	california color                        californiacolor                 #6f4e37

	california green                        californiagreen                 #a18f60

	calla green                             callagreen                      #8a9a5b

	calliste green                          callistegreen                   #7e9f2e

	calliste green                          callistegreen                   #8a9a5b

	cambridge blue                          cambridgeblue                   #66ada4

	cambridge red                           cambridgered                    #ab4e52

	cambridge red                           cambridgered                    #905d5d

	camel                                   camel                           #5b504f

	camellia                                camellia                        #bc3f4a

	camellia                                camellia                        #ab4e52

	camels hair or camel's hair             camelshairorcamelshair          #7e6d5a

	cameo blue                              cameoblue                       #66ada4

	cameo brown                             cameobrown                      #905d5d

	cameo brown                             cameobrown                      #a87c6d

	cameo brown                             cameobrown                      #79443b

	cameo brown                             cameobrown                      #674c47

	cameo green                             cameogreen                      #93c592

	cameo green                             cameogreen                      #6aab8e

	cameo green                             cameogreen                      #8da399

	cameo pink                              cameopink                       #c3a6b1

	cameo yellow                            cameoyellow                     #f3e5ab

	campanula                               campanula                       #604e97

	campanula                               campanula                       #604e81

	campanula blue                          campanulablue                   #8791bf

	campanula blue                          campanulablue                   #6c79b8

	campanula blue                          campanulablue                   #7e73b8

	campanula purple                        campanulapurple                 #836479

	campanula violet                        campanulaviolet                 #836479

	camrier                                camrier                         #563c5c

	camwood                                 camwood                         #ab4e52

	canard                                  canard                          #004b49

	canard                                  canard                          #004958

	canary-bird green                       canarybirdgreen                 #ab9144

	canary green                            canarygreen                     #ab9144

	canary [yellow]                         canaryyellow                    #c9ae5d

	candida                                 candida                         #af868e

	candy pink                              candypink                       #e4717a

	candy pink                              candypink                       #f88379

	candy pink                              candypink                       #e66761

	canna                                   canna                           #cb6d51

	canna                                   canna                           #9e4732

	canna                                   canna                           #b4745e

	canna                                   canna                           #79443b

	cannon                                  cannon                          #1a2421

	canterbury                              canterbury                      #604e97

	canton [blue]                           cantonblue                      #00304e

	canton jade                             cantonjade                      #317873

	canyon                                  canyon                          #722f37

	canyon                                  canyon                          #673147

	cappagh                                 cappagh                         #6f4e37

	cappah brown                            cappahbrown                     #6f4e37

	capri                                   capri                           #007791

	caprice                                 caprice                         #870074

	caprice                                 caprice                         #d6cadd

	capucine                                capucine                        #c9ae5d

	capucine buff                           capucinebuff                    #fad6a5

	capucine lake                           capucinelake                    #f88379

	capucine lake                           capucinelake                    #cb6d51

	capucine lake                           capucinelake                    #ed872d

	capucine lake                           capucinelake                    #d99058

	capucine madder                         capucinemadder                  #d99058

	capucine orange                         capucineorange                  #d99058

	capucine red                            capucinered                     #ffb7a5

	capucine red                            capucinered                     #d9603b

	capucine yellow                         capucineyellow                  #d99058

	caput mortuum                           caputmortuum                    #79443b

	caraibe                                 caraibe                         #6f4e37

	caraibe                                 caraibe                         #635147

	caraibe                                 caraibe                         #826644

	caraibe                                 caraibe                         #7e6d5a

	caramel                                 caramel                         #ae6938

	caramel                                 caramel                         #a67b5b

	carbuncle                               carbuncle                       #43302e

	cardinal [red]                          cardinalred                     #be0032

	carmine                                 carmine                         #be0032

	carmine                                 carmine                         #bc3f4a

	carmine lake                            carminelake                     #be0032

	carmine lake                            carminelake                     #bc3f4a

	carnation                               carnation                       #f3e5ab

	carnation                               carnation                       #c2b280

	carnation [red]                         carnationred                    #ab4e52

	carnation rose                          carnationrose                   #ea9399

	carnelian                               carnelian                       #b4745e

	carnelian red                           carnelianred                    #cb6d51

	carnival red                            carnivalred                     #bc3f4a

	carob brown                             carobbrown                      #6f4e37

	caroubier                               caroubier                       #6f4e37

	carrara green                           carraragreen                    #1b4d3e

	carrot red                              carrotred                       #ed872d

	carrot red                              carrotred                       #d99058

	carthamus red                           carthamusred                    #be0032

	carthamus rose                          carthamusrose                   #be0032

	cartouche                               cartouche                       #6f4e37

	cartridge buff                          cartridgebuff                   #f3e5ab

	cartridge buff                          cartridgebuff                   #c2b280

	cascade                                 cascade                         #66ada4

	cashew                                  cashew                          #6f4e37

	cashew lake                             cashewlake                      #6f4e37

	cashew nut                              cashewnut                       #635147

	cashoo                                  cashoo                          #6f4e37

	casino pink                             casinopink                      #b3446c

	cassel brown                            casselbrown                     #6f4e37

	cassel earth                            casselearth                     #6f4e37

	cassel yellow                           casselyellow                    #fada5e

	cassel yellow                           casselyellow                    #d4af37

	cassel yellow                           casselyellow                    #f8de7e

	cassel yellow                           casselyellow                    #c9ae5d

	casserole                               casserole                       #79443b

	castaneous                              castaneous                      #635147

	castellon                               castellon                       #79443b

	castile earth                           castileearth                    #6f4e37

	castilian brown                         castilianbrown                  #79443b

	castilian red                           castilianred                    #be0032

	castor                                  castor                          #5b504f

	castor grey                             castorgrey                      #4e5755

	catalpa                                 catalpa                         #796878

	catawba                                 catawba                         #3f1728

	catawba                                 catawba                         #2e1d21

	cate                                    cate                            #6f4e37

	catechu                                 catechu                         #6f4e37

	cathay                                  cathay                          #ae6938

	cathay                                  cathay                          #996515

	cathedral                               cathedral                       #2f2140

	cathedral blue                          cathedralblue                   #436b95

	cathedral blue                          cathedralblue                   #4e5180

	cattail                                 cattail                         #483c32

	cattleya                                cattleya                        #86608e

	caucasia                                caucasia                        #7e6d5a

	cauldron                                cauldron                        #722f37

	cauldron                                cauldron                        #882d17

	cauldron                                cauldron                        #79443b

	cavalry                                 cavalry                         #e3a857

	cedar                                   cedar                           #674c47

	cedarbark                               cedarbark                       #905d5d

	cedarbark                               cedarbark                       #79443b

	cedar green                             cedargreen                      #4a5d23

	cedar wood                              cedarwood                       #79443b

	cedre                                   cedre                           #4a5d23

	celadon green                           celadongreen                    #8f9779

	celandine green                         celandinegreen                  #5e716a

	celest                                  celest                          #536878

	celest                                  celest                          #8c92ac

	celestial                               celestial                       #436b95

	celestial blue                          celestialblue                   #91a3b0

	cellini                                 cellini                         #870074

	cellini                                 cellini                         #ce4676

	cement                                  cement                          #848482

	cendrillon                              cendrillon                      #3a4b47

	cendrillon                              cendrillon                      #51585e

	centennial brown                        centennialbrown                 #996515

	centennial brown                        centennialbrown                 #c19a6b

	centennial brown                        centennialbrown                 #826644

	centre blue                             centreblue                      #436b95

	ceramic                                 ceramic                         #436b95

	ceres                                   ceres                           #d99058

	cerise                                  cerise                          #ab4e52

	cerro green                             cerrogreen                      #8a9a5b

	cerro green                             cerrogreen                      #4a5d23

	certosa                                 certosa                         #98943e

	certosa                                 certosa                         #867e36

	certosa                                 certosa                         #8a9a5b

	cerulean                                cerulean                        #70a3cc

	cerulean                                cerulean                        #8791bf

	cerulean blue                           ceruleanblue                    #0085a1

	cerulean blue                           ceruleanblue                    #00a1c2

	ceylon blue                             ceylonblue                      #004958

	chaetura black                          chaeturablack                   #222222

	chaetura drab                           chaeturadrab                    #555555

	chalcedony yellow                       chalcedonyyellow                #eae679

	chalet red                              chaletred                       #905d5d

	chamois                                 chamois                         #c2b280

	chamois skin                            chamoisskin                     #c2b280

	chamoline                               chamoline                       #be8a3d

	chamoline                               chamoline                       #996515

	chamoline                               chamoline                       #c19a6b

	chamoline                               chamoline                       #826644

	chamoline                               chamoline                       #ab9144

	chamoline                               chamoline                       #a18f60

	chamoline                               chamoline                       #967117

	champagne                               champagne                       #fad6a5

	champagne                               champagne                       #ae9b82

	champagne                               champagne                       #c2b280

	champagne                               champagne                       #bfb8a5

	chantilly                               chantilly                       #6aab8e

	charcoal grey                           charcoalgrey                    #555555

	chardon                                 chardon                         #673147

	charles x                               charlesx                        #673147

	chartreuse                              chartreuse                      #e9e450

	chartreuse                              chartreuse                      #beb72e

	chartreuse                              chartreuse                      #eae679

	chartreuse                              chartreuse                      #b9b459

	chartreuse green                        chartreusegreen                 #98943e

	chartreuse yellow                       chartreuseyellow                #eae679

	chartreuse yellow                       chartreuseyellow                #b9b459

	chasseur                                chasseur                        #173620

	chatemuc                                chatemuc                        #ab4e52

	chatenay pink                           chatenaypink                    #f4c2c2

	chatenay pink                           chatenaypink                    #d9a6a9

	chaudron                                chaudron                        #cb6d51

	chaudron                                chaudron                        #9e4732

	chaudron                                chaudron                        #b4745e

	chaudron                                chaudron                        #79443b

	checkerberry                            checkerberry                    #be0032

	chemic blue                             chemicblue                      #004958

	chemic green                            chemicgreen                     #317873

	cherokee                                cherokee                        #cb6d51

	cherry                                  cherry                          #e25822

	cherry bloom                            cherrybloom                     #f88379

	cherry blossom                          cherryblossom                   #ab4e52

	cherry red                              cherryred                       #e25822

	cherub                                  cherub                          #d9a6a9

	chessylite blue                         chessyliteblue                  #436b95

	chestnut                                chestnut                        #635147

	chevreuse                               chevreuse                       #905d5d

	chevreuse                               chevreuse                       #79443b

	chianti                                 chianti                         #841b2d

	chickadee gray                          chickadeegray                   #57554c

	chicle                                  chicle                          #635147

	chicory                                 chicory                         #8c92ac

	chicory blue                            chicoryblue                     #dadfb7

	chicory blue                            chicoryblue                     #8c92ac

	chimney red                             chimneyred                      #841b2d

	chimney red                             chimneyred                      #ab4e52

	china blue                              chinablue                       #436b95

	china blue                              chinablue                       #536878

	china rose                              chinarose                       #a8516e

	chinchilla                              chinchilla                      #848482

	chin-chin blue                          chinchinblue                    #536878

	chinese gold                            chinesegold                     #ae6938

	chinese green                           chinesegreen                    #93c592

	chinese green                           chinesegreen                    #6aab8e

	chinese green                           chinesegreen                    #1b4d3e

	chinese green                           chinesegreen                    #004b49

	chinese lake                            chineselake                     #be0032

	chinese lake                            chineselake                     #bc3f4a

	chinese orange                          chineseorange                   #ffb7a5

	chinese orange                          chineseorange                   #d9603b

	chinese red                             chinesered                      #aa381e

	chinese red                             chinesered                      #9e4732

	chinese red                             chinesered                      #882d17

	chinese rouge                           chineserouge                    #be0032

	chinese vermilion                       chinesevermilion                #be0032

	chinese violet                          chineseviolet                   #86608e

	chinese yellow                          chineseyellow                   #fada5e

	chinese yellow                          chineseyellow                   #d4af37

	chinese yellow                          chineseyellow                   #f8de7e

	chinese yellow                          chineseyellow                   #c9ae5d

	chinese yellow                          chineseyellow                   #e3a857

	ch'ing                                  ching                           #00a1c2

	chinook                                 chinook                         #9e4732

	chip                                    chip                            #8c8767

	chip                                    chip                            #8a8776

	chipmunk                                chipmunk                        #996515

	chippendale                             chippendale                     #3e322c

	chiswick                                chiswick                        #00304e

	chocolate                               chocolate                       #3e322c

	chocolate brown                         chocolatebrown                  #3e322c

	chocolate maroon                        chocolatemaroon                 #722f37

	chrome citron                           chromecitron                    #f8de7e

	chrome green                            chromegreen                     #007959

	chrome green                            chromegreen                     #3b7861

	chrome lemon                            chromelemon                     #f8de7e

	chrome orange                           chromeorange                    #e25822

	chrome orange                           chromeorange                    #f38400

	chrome primrose                         chromeprimrose                  #e9e450

	chrome primrose                         chromeprimrose                  #eae679

	chrome scarlet                          chromescarlet                   #d9603b

	chromium green                          chromiumgreen                   #8a9a5b

	chromium oxide                          chromiumoxide                   #679267

	chrysanthemum                           chrysanthemum                   #841b2d

	chrysanthemum                           chrysanthemum                   #722f37

	chrysocollo                             chrysocollo                     #679267

	chrysolite green                        chrysolitegreen                 #b9b459

	chrysolite green                        chrysolitegreen                 #b9b57d

	chrysolite green                        chrysolitegreen                 #8a9a5b

	chrysopraise [green]                    chrysopraisegreen               #3eb489

	chrysopraise [green]                    chrysopraisegreen               #6aab8e

	chukker brown                           chukkerbrown                    #635147

	chukker brown                           chukkerbrown                    #7e6d5a

	chutney                                 chutney                         #79443b

	cigarette                               cigarette                       #6f4e37

	cinder [grey]                           cindergrey                      #8b8589

	cineraria                               cineraria                       #ce4676

	cinereous                               cinereous                       #b9b8b5

	cinnamon                                cinnamon                        #a67b5b

	cinnamon                                cinnamon                        #c19a6b

	cinnamon brown                          cinnamonbrown                   #6f4e37

	cinnamon brown                          cinnamonbrown                   #826644

	cinnamon pink                           cinnamonpink                    #a87c6d

	circassian                              circassian                      #6f4e37

	citrine                                 citrine                         #867e36

	citronelle                              citronelle                      #e9e450

	citronelle                              citronelle                      #beb72e

	citronelle                              citronelle                      #eae679

	citronelle                              citronelle                      #b9b459

	citron green                            citrongreen                     #b9b459

	citron green                            citrongreen                     #b9b57d

	citron [yellow]                         citronyellow                    #f8de7e

	citrus                                  citrus                          #eae679

	civette green                           civettegreen                    #355e3b

	clair de lune                           clairdelune                     #8da399

	claret cup                              claretcup                       #915f6d

	claret lees                             claretlees                      #673147

	claret [red]                            claretred                       #ab4e52

	claver                                  claver                          #915c83

	clay                                    clay                            #996515

	clay                                    clay                            #c19a6b

	clay                                    clay                            #826644

	clay bank                               claybank                        #ae6938

	clay drab                               claydrab                        #826644

	clay drab                               claydrab                        #967117

	clematis                                clematis                        #604e97

	cleopatra                               cleopatra                       #00a1c2

	clochette                               clochette                       #4e5180

	clochette                               clochette                       #604e81

	cloisonn                               cloisonn                        #00a1c2

	cloisonn                               cloisonn                        #30267a

	cloud                                   cloud                           #536878

	cloud grey                              cloudgrey                       #8b8589

	cloud grey                              cloudgrey                       #848482

	cloudy amber                            cloudyamber                     #c2b280

	cloudy amber                            cloudyamber                     #a18f60

	clove                                   clove                           #6f4e37

	clove brown                             clovebrown                      #5b504f

	clove pink                              clovepink                       #722f37

	clover                                  clover                          #915c83

	cobalt blue                             cobaltblue                      #007791

	cobalt glass                            cobaltglass                     #00416a

	cobalt glass                            cobaltglass                     #436b95

	cobalt green                            cobaltgreen                     #679267

	cobalt red                              cobaltred                       #ce4676

	cobalt red                              cobaltred                       #b3446c

	cobalt ultramarine                      cobaltultramarine               #007791

	cobalt violet                           cobaltviolet                    #86608e

	cobalt yellow                           cobaltyellow                    #fada5e

	cobalt yellow                           cobaltyellow                    #d4af37

	cobblestone                             cobblestone                     #555555

	cobweb                                  cobweb                          #ae9b82

	coccineous                              coccineous                      #be0032

	cocher                                  cocher                          #6c541e

	cochin                                  cochin                          #6f4e37

	cochineal                               cochineal                       #be0032

	cockatoo                                cockatoo                        #f8de7e

	cocoa                                   cocoa                           #6f4e37

	cocoa brown                             cocoabrown                      #6f4e37

	cocobala                                cocobala                        #905d5d

	cocobala                                cocobala                        #674c47

	coconut                                 coconut                         #635147

	coconut brown                           coconutbrown                    #6f4e37

	cod grey                                codgrey                         #5b5842

	cod grey                                codgrey                         #57554c

	coeruleum                               coeruleum                       #0085a1

	coeruleum                               coeruleum                       #00a1c2

	coffee                                  coffee                          #6f4e37

	cognac                                  cognac                          #6f4e37

	colcothar                               colcothar                       #79443b

	colewort green                          colewortgreen                   #7d8984

	colibri                                 colibri                         #317873

	collie                                  collie                          #7e6d5a

	cologne brown                           colognebrown                    #6f4e37

	cologne earth                           cologneearth                    #6f4e37

	cologne yellow                          cologneyellow                   #d4af37

	colonial                                colonial                        #00416a

	colonial                                colonial                        #00304e

	colonial                                colonial                        #272458

	colonial                                colonial                        #4e5180

	colonial buff                           colonialbuff                    #f8de7e

	colonial buff                           colonialbuff                    #f3e5ab

	colonial yellow                         colonialyellow                  #c9ae5d

	columbia                                columbia                        #0067a5

	columbia blue                           columbiablue                    #91a3b0

	columbian red                           columbianred                    #79443b

	columbine blue                          columbineblue                   #4e5180

	columbine blue                          columbineblue                   #604e81

	columbine red                           columbinered                    #673147

	comet                                   comet                           #00304e

	commelina blue                          commelinablue                   #436b95

	commodore                               commodore                       #30267a

	como                                    como                            #367588

	conch shell                             conchshell                      #f88379

	concord                                 concord                         #2f2140

	condor                                  condor                          #826644

	confetti                                confetti                        #e4717a

	confetti                                confetti                        #ab4e52

	congo [brown]                           congobrown                      #483c32

	congo pink                              congopink                       #b4745e

	continental blue                        continentalblue                 #50404d

	continental blue                        continentalblue                 #242124

	cookie                                  cookie                          #6f4e37

	coolie                                  coolie                          #4e5180

	coolie                                  coolie                          #252440

	copenhagen [blue]                       copenhagenblue                  #536878

	copenhagen [blue]                       copenhagenblue                  #4c516d

	copper                                  copper                          #b4745e

	copper blue                             copperblue                      #436b95

	copper brown                            copperbrown                     #80461b

	copper green                            coppergreen                     #679267

	copperleaf                              copperleaf                      #79443b

	copper lustre                           copperlustre                    #80461b

	copper red                              copperred                       #b4745e

	copper rose                             copperrose                      #ab4e52

	copper yellow                           copperyellow                    #c9ae5d

	copra                                   copra                           #5c504f

	coptic                                  coptic                          #79443b

	coquelicot                              coquelicot                      #be0032

	coquelicot                              coquelicot                      #e25822

	coquette                                coquette                        #f88379

	coquette                                coquette                        #d9a6a9

	coralbell                               coralbell                       #ab4e52

	coralbell                               coralbell                       #cb6d51

	coral blush                             coralblush                      #f88379

	coral blush                             coralblush                      #e66761

	coral blush                             coralblush                      #cb6d51

	coral pink                              coralpink                       #d9a6a9

	coral [red]                             coralred                        #e4717a

	coral sands                             coralsands                      #d9a6a9

	coral sands                             coralsands                      #c7ada3

	corcir                                  corcir                          #a8516e

	cordova                                 cordova                         #79443b

	cordovan                                cordovan                        #483c32

	cordovan                                cordovan                        #222222

	corial                                  corial                          #d9a6a9

	corial                                  corial                          #a87c6d

	corinth                                 corinth                         #915f6d

	corinthian pink                         corinthianpink                  #dea5a4

	corinthian pink                         corinthianpink                  #ad8884

	corinthian purple                       corinthianpurple                #673147

	corinthian purple                       corinthianpurple                #915f6d

	corinthian red                          corinthianred                   #905d5d

	cork                                    cork                            #a67b5b

	cork                                    cork                            #a8516e

	corker                                  corker                          #a8516e

	corkur                                  corkur                          #a8516e

	corn                                    corn                            #c9ae5d

	cornelian red                           cornelianred                    #b4745e

	cornflower [blue]                       cornflowerblue                  #4e5180

	cornhusk                                cornhusk                        #fad6a5

	cornhusk                                cornhusk                        #c19a6b

	cornsilk                                cornsilk                        #f8de7e

	coromandel                              coromandel                      #79443b

	coronation                              coronation                      #5d3954

	coronation                              coronation                      #673147

	corsage green                           corsagegreen                    #6aab8e

	corsage green                           corsagegreen                    #66ada4

	corsair                                 corsair                         #00304e

	corsican blue                           corsicanblue                    #4e5180

	corsican blue                           corsicanblue                    #252440

	corsican blue                           corsicanblue                    #2f2140

	corydalis green                         corydalisgreen                  #8f9779

	cosmos                                  cosmos                          #c17e91

	cossack                                 cossack                         #5e716a

	cossack green                           cossackgreen                    #355e3b

	cosse green                             cossegreen                      #8a9a5b

	cotch                                   cotch                           #6f4e37

	cotinga purple                          cotingapurple                   #602f6b

	cotinga purple                          cotingapurple                   #563c5c

	cotinga purple                          cotingapurple                   #702963

	cotinga purple                          cotingapurple                   #5d3954

	cotrine                                 cotrine                         #be8a3d

	cotrine                                 cotrine                         #c19a6b

	courge green                            courgegreen                     #98943e

	courge green                            courgegreen                     #867e36

	courge green                            courgegreen                     #8a9a5b

	cowboy                                  cowboy                          #6f4e37

	cowboy                                  cowboy                          #635147

	cowboy                                  cowboy                          #826644

	cowboy                                  cowboy                          #7e6d5a

	cowslip                                 cowslip                         #ed872d

	cowslip                                 cowslip                         #d99058

	crabapple                               crabapple                       #cb6d51

	cracker                                 cracker                         #c19a6b

	crag                                    crag                            #57554c

	crane                                   crane                           #8b8589

	crash                                   crash                           #c2b280

	crash                                   crash                           #a18f60

	crash                                   crash                           #967117

	crash                                   crash                           #8c8767

	crawshay red                            crawshayred                     #9e4732

	crawshay red                            crawshayred                     #80461b

	crayon blue                             crayonblue                      #536878

	cream                                   cream                           #f3e5ab

	cream beige                             creambeige                      #a87c6d

	cream beige                             creambeige                      #a67b5b

	cream beige                             creambeige                      #958070

	cream buff                              creambuff                       #f3e5ab

	cream buff                              creambuff                       #c2b280

	creole                                  creole                          #ae6938

	creole                                  creole                          #a67b5b

	cress green                             cressgreen                      #8a9a5b

	cresson                                 cresson                         #8a9a5b

	crevette                                crevette                        #f88379

	crimson lake                            crimsonlake                     #ab4e52

	crimson madder                          crimsonmadder                   #be0032

	crimson maple                           crimsonmaple                    #56070c

	croceus                                 croceus                         #d99058

	croceus                                 croceus                         #e3a857

	crocreal                                crocreal                        #d99058

	crocreal                                crocreal                        #e3a857

	crocus                                  crocus                          #d99058

	crocus                                  crocus                          #e3a857

	crocus                                  crocus                          #aa8a9e

	crocus                                  crocus                          #836479

	crocus martis                           crocusmartis                    #79443b

	crostil                                 crostil                         #a8516e

	crotal                                  crotal                          #a8516e

	crottle                                 crottle                         #a8516e

	crowshay red                            crowshayred                     #9e4732

	crowshay red                            crowshayred                     #80461b

	cruiser                                 cruiser                         #555555

	crushed berry                           crushedberry                    #905d5d

	crushed strawberry                      crushedstrawberry               #e4717a

	crushed violets                         crushedviolets                  #796878

	crystal grey                            crystalgrey                     #8b8589

	crystal palace blue                     crystalpalaceblue               #00304e

	crystal palace green                    crystalpalacegreen              #57554c

	crystals of venus                       crystalsofvenus                 #679267

	cub                                     cub                             #57554c

	cuba                                    cuba                            #722f37

	cuba                                    cuba                            #79443b

	cuban sand                              cubansand                       #ae9b82

	cudbear                                 cudbear                         #a8516e

	cuir                                    cuir                            #c19a6b

	cuisse de nymphe                        cuissedenymphe                  #f3e5ab

	cullen earth                            cullenearth                     #6f4e37

	cupid pink                              cupidpink                       #d9a6a9

	cupreous                                cupreous                        #f8de7e

	curcuma                                 curcuma                         #f8de7e

	currant [red]                           currantred                      #bc3f4a

	cutch                                   cutch                           #6f4e37

	cuyahoga red                            cuyahogared                     #722f37

	cyan blue                               cyanblue                        #317873

	cyan blue                               cyanblue                        #367588

	cyanine blue                            cyanineblue                     #00a1c2

	cyanine blue                            cyanineblue                     #4e5180

	cyanine blue                            cyanineblue                     #252440

	cyclamen                                cyclamen                        #341731

	cygnet                                  cygnet                          #673147

	cypress [green]                         cypressgreen                    #4a5d23

	cypress green                           cypressgreen                    #44944a

	cyprus earth                            cyprusearth                     #826644

	cyprus earth                            cyprusearth                     #4b3621

	cyprus umber                            cyprusumber                     #826644

	cyprus umber                            cyprusumber                     #4b3621

	daffodile [yellow]                      daffodileyellow                 #e3a857

	daffodil yellow                         daffodilyellow                  #ed872d

	daffodil yellow                         daffodilyellow                  #d99058

	daffodil yellow                         daffodilyellow                  #eaa221

	daffodil yellow                         daffodilyellow                  #e3a857

	dagestan                                dagestan                        #00304e

	dahlia                                  dahlia                          #86608e

	dahlia carmine                          dahliacarmine                   #673147

	dahlia mauve                            dahliamauve                     #915c83

	dahlia purple                           dahliapurple                    #673147

	dahlia purple                           dahliapurple                    #38152c

	damascen                                damascen                        #604e81

	damascen                                damascen                        #2f2140

	damask                                  damask                          #905d5d

	damonico                                damonico                        #f88379

	damonico                                damonico                        #cb6d51

	damonico                                damonico                        #ed872d

	damonico                                damonico                        #d99058

	damson                                  damson                          #604e81

	damson                                  damson                          #2f2140

	dana                                   dana                            #ed872d

	dana                                   dana                            #d99058

	dana                                   dana                            #536878

	dandelion                               dandelion                       #fada5e

	dante                                   dante                           #50404d

	dante                                   dante                           #5d3954

	dante                                   dante                           #673147

	danube green                            danubegreen                     #355e3b

	danube green                            danubegreen                     #3a4b47

	daphne                                  daphne                          #436b95

	daphne pink                             daphnepink                      #915f6d

	daphne red                              daphnered                       #a8516e

	daphne red                              daphnered                       #915f6d

	dark beaver                             darkbeaver                      #6f4e37

	dark beaver                             darkbeaver                      #635147

	dark cardinal                           darkcardinal                    #722f37

	dark gobelin blue                       darkgobelinblue                 #367588

	dark gobelin blue                       darkgobelinblue                 #536878

	dark wedgwood [blue]                    darkwedgwoodblue                #4e5180

	dark wedgwood [blue]                    darkwedgwoodblue                #4c516d

	date                                    date                            #483c32

	datura                                  datura                          #86608e

	dauphine                                dauphine                        #ad8884

	davy's grey                             davysgrey                       #555555

	dawn                                    dawn                            #dea5a4

	dawn grey                               dawngrey                        #848482

	daybreak                                daybreak                        #8c82b6

	daybreak                                daybreak                        #b695c0

	daytona                                 daytona                         #8e8279

	dbutante pink                          dbutantepink                    #dea5a4

	dead carnations                         deadcarnations                  #a8516e

	dead leaf                               deadleaf                        #be6516

	dead leaf                               deadleaf                        #ae6938

	dead leaf                               deadleaf                        #80461b

	deauville sand                          deauvillesand                   #977f73

	deauville sand                          deauvillesand                   #958070

	deep brunswick green                    deepbrunswickgreen              #1b4d3e

	deep chrome green                       deepchromegreen                 #355e3b

	deep chrome yellow                      deepchromeyellow                #e3a857

	deep stone                              deepstone                       #ab9144

	deep stone                              deepstone                       #a18f60

	deep stone                              deepstone                       #967117

	deer                                    deer                            #7e6d5a

	delft blue                              delftblue                       #4c516d

	della robbia                            dellarobbia                     #545aa7

	del monte                               delmonte                        #796878

	delphinium                              delphinium                      #4997d0

	delphinium                              delphinium                      #0067a5

	delphinium                              delphinium                      #70a3cc

	de'medici                               demedici                        #673147

	denmark                                 denmark                         #536878

	denver                                  denver                          #848482

	derby blue                              derbyblue                       #2f2140

	desert                                  desert                          #c19a6b

	devil's red                             devilsred                       #bc3f4a

	devil's red                             devilsred                       #ab4e52

	devon [brown]                           devonbrown                      #905d5d

	dewey red                               deweyred                        #bc3f4a

	dewey red                               deweyred                        #ab4e52

	dewkiss                                 dewkiss                         #5e716a

	diadem                                  diadem                          #aa98a9

	diana                                   diana                           #70a3cc

	dianthus                                dianthus                        #ab4e52

	dianthus                                dianthus                        #905d5d

	diavolo                                 diavolo                         #905d5d

	digitalis                               digitalis                       #796878

	di palito                               dipalito                        #e3a857

	directoire blue                         directoireblue                  #00a1c2

	directoire blue                         directoireblue                  #0067a5

	distilled green                         distilledgreen                  #679267

	diva blue                               divablue                        #0067a5

	diva blue                               divablue                        #436b95

	doe                                     doe                             #a67b5b

	doe                                     doe                             #958070

	doe                                     doe                             #c19a6b

	doe                                     doe                             #ae9b82

	doe-skin brown                          doeskinbrown                    #a87c6d

	doge                                    doge                            #ab4e52

	dogwood                                 dogwood                         #a67b5b

	dogwood                                 dogwood                         #6f4e37

	dogwood                                 dogwood                         #826644

	dolly pink                              dollypink                       #ab4e52

	domingo                                 domingo                         #79443b

	domingo brown                           domingobrown                    #422518

	dorado                                  dorado                          #c19a6b

	doubloon                                doubloon                        #d99058

	dove                                    dove                            #555555

	dove                                    dove                            #673147

	dove grey                               dovegrey                        #8b8589

	dover grey                              dovergrey                       #555555

	dozer                                   dozer                           #00304e

	drab                                    drab                            #967117

	drabolive                               drabolive                       #5b5842

	dragonfly                               dragonfly                       #004958

	dragon's blood                          dragonsblood                    #905d5d

	dragon's blood                          dragonsblood                    #b4745e

	dragon's blood                          dragonsblood                    #a87c6d

	drake                                   drake                           #004958

	drake's-neck green                      drakesneckgreen                 #004958

	dregs of wine                           dregsofwine                     #722f37

	dregs of wine                           dregsofwine                     #56070c

	dregs of wine                           dregsofwine                     #3e1d1e

	dresden blue                            dresdenblue                     #436b95

	dresden brown                           dresdenbrown                    #826644

	dresden brown                           dresdenbrown                    #967117

	dresden brown                           dresdenbrown                    #6c541e

	dryad                                   dryad                           #5e716a

	dry rose                                dryrose                         #905d5d

	dry rose                                dryrose                         #79443b

	du barry                                dubarry                         #f88379

	duck blue                               duckblue                        #007791

	duck green                              duckgreen                       #004b49

	duckling                                duckling                        #004958

	duckling                                duckling                        #36454f

	duck-wing                               duckwing                        #4c516d

	du gueslin                              dugueslin                       #00304e

	du gueslin                              dugueslin                       #252440

	du gueslin                              dugueslin                       #4c516d

	dumont's blue                           dumontsblue                     #00416a

	dumont's blue                           dumontsblue                     #436b95

	dune                                    dune                            #8c8767

	durango                                 durango                         #6f4e37

	dusk                                    dusk                            #5d555b

	dust                                    dust                            #967117

	dusty green                             dustygreen                      #8c8767

	dutch azure                             dutchazure                      #00416a

	dutch azure                             dutchazure                      #436b95

	dutch blue                              dutchblue                       #536878

	dutch orange                            dutchorange                     #eaa221

	dutch orange                            dutchorange                     #e3a857

	dutch pink                              dutchpink                       #fada5e

	dutch pink                              dutchpink                       #d4af37

	dutch pink                              dutchpink                       #f8de7e

	dutch pink                              dutchpink                       #c9ae5d

	dutch scarlet                           dutchscarlet                    #be0032

	dutch vermilion                         dutchvermilion                  #e25822

	dutch ware blue                         dutchwareblue                   #4c516d

	dyer's broom                            dyersbroom                      #eae679

	dyer's broom                            dyersbroom                      #b9b459

	dyer's greenwood                        dyersgreenwood                  #eae679

	dyer's greenwood                        dyersgreenwood                  #b9b459

	dyer's saffron                          dyerssaffron                    #be0032

	eagle                                   eagle                           #5b504f

	eau-de-javel green                      eaudejavelgreen                 #b9b459

	eau-de-nile                             eaudenile                       #8a9a5b

	eau-de-nile                             eaudenile                       #dadfb7

	eau-de-nile                             eaudenile                       #93c592

	eburnean                                eburnean                        #f3e5ab

	ecclesiastic                            ecclesiastic                    #870074

	ecclesiastic                            ecclesiastic                    #9e4f88

	ecru                                    ecru                            #f3e5ab

	ecru                                    ecru                            #c2b280

	eden green                              edengreen                       #44944a

	eden green                              edengreen                       #679267

	eggplant                                eggplant                        #291e29

	eggplant                                eggplant                        #242124

	eggshell blue                           eggshellblue                    #8da399

	eggshell blue                           eggshellblue                    #66ada4

	eggshell green                          eggshellgreen                   #8da399

	eggshell green                          eggshellgreen                   #66ada4

	eglantine                               eglantine                       #ce4676

	egypt                                   egypt                           #0067a5

	egyptian                                egyptian                        #905d5d

	egyptian blue                           egyptianblue                    #00a1c2

	egyptian blue                           egyptianblue                    #0067a5

	egyptian brown                          egyptianbrown                   #483c32

	egyptian green                          egyptiangreen                   #3b7861

	egyptian red                            egyptianred                     #722f37

	egyptian red                            egyptianred                     #882d17

	eifel                                   eifel                           #905d5d

	eifel                                   eifel                           #79443b

	elderberry                              elderberry                      #2f2140

	eldorado                                eldorado                        #ae6938

	electric [blue]                         electricblue                    #536878

	electric green                          electricgreen                   #536878

	elephant green                          elephantgreen                   #173620

	elephant skin                           elephantskin                    #5b504f

	elf                                     elf                             #6aab8e

	elf                                     elf                             #8da399

	elfin green                             elfingreen                      #6aab8e

	elk                                     elk                             #483c32

	elm green                               elmgreen                        #4a5d23

	elmwood                                 elmwood                         #ae9b82

	elmwood                                 elmwood                         #967117

	email                                   email                           #317873

	email                                   email                           #367588

	ember                                   ember                           #ab4e52

	emberglow                               emberglow                       #ab4e52

	emberglow                               emberglow                       #e66761

	emberglow                               emberglow                       #cb6d51

	emerald green                           emeraldgreen                    #679267

	emerald green                           emeraldgreen                    #3eb489

	emeraude                                emeraude                        #679267

	emeraude                                emeraude                        #007959

	emeraude                                emeraude                        #3b7861

	eminence                                eminence                        #563c5c

	emperor green                           emperorgreen                    #3eb489

	empire                                  empire                          #007959

	empire blue                             empireblue                      #436b95

	empire green                            empiregreen                     #355e3b

	empire green                            empiregreen                     #3a4b47

	empire yellow                           empireyellow                    #fada5e

	empire yellow                           empireyellow                    #f8de7e

	enamel blue                             enamelblue                      #00416a

	enamel blue                             enamelblue                      #436b95

	endive                                  endive                          #e9e450

	endive                                  endive                          #eae679

	endive blue                             endiveblue                      #91a3b0

	endive blue                             endiveblue                      #aa98a9

	english blue                            englishblue                     #436b95

	english green                           englishgreen                    #7e9f2e

	english green                           englishgreen                    #44944a

	english green                           englishgreen                    #1b4d3e

	english grey                            englishgrey                     #57554c

	english inde                            englishinde                     #50404d

	english ivy                             englishivy                      #355e3b

	english oak                             englishoak                      #6f4e37

	english ochre                           englishochre                    #e3a857

	english pink                            englishpink                     #fada5e

	english pink                            englishpink                     #d4af37

	english pink                            englishpink                     #f8de7e

	english pink                            englishpink                     #c9ae5d

	english red                             englishred                      #bc3f4a

	english red                             englishred                      #ab4e52

	english red                             englishred                      #d9603b

	english red                             englishred                      #cb6d51

	english red                             englishred                      #9e4732

	english red                             englishred                      #80461b

	english red                             englishred                      #79443b

	english vermilion                       englishvermilion                #bc3f4a

	english violet                          englishviolet                   #563c5c

	ensign                                  ensign                          #00304e

	epinauche                               epinauche                       #3a4b47

	epsom                                   epsom                           #8a9a5b

	erin                                    erin                            #3b7861

	erlau green                             erlaugreen                      #679267

	escadre                                 escadre                         #536878

	eschel                                  eschel                          #00416a

	eschel                                  eschel                          #436b95

	eskimo                                  eskimo                          #79443b

	esthetic grey                           estheticgrey                    #7d8984

	etain blue                              etainblue                       #c7e6d7

	etain blue                              etainblue                       #bcd4e6

	etain blue                              etainblue                       #c7e6d7

	etain blue                              etainblue                       #96ded1

	etain blue                              etainblue                       #9cd1dc

	etain blue                              etainblue                       #bcd4e6

	etang                                   etang                           #8da399

	etang                                   etang                           #b2beb5

	ether                                   ether                           #66aabc

	ethereal blue                           etherealblue                    #70a3cc

	ethereal blue                           etherealblue                    #8791bf

	eton blue                               etonblue                        #66ada4

	etruscan                                etruscan                        #79443b

	etruscan                                etruscan                        #6f4e37

	etruscan red                            etruscanred                     #b4745e

	eucalyptus [green]                      eucalyptusgreen                 #c2b280

	eucalyptus [green]                      eucalyptusgreen                 #8f9779

	eugenia red                             eugeniared                      #ab4e52

	eugenie                                 eugenie                         #796878

	eupatorium purple                       eupatoriumpurple                #915c83

	eureka red                              eurekared                       #722f37

	eve green                               evegreen                        #e9e450

	eve green                               evegreen                        #83d37d

	eve green                               evegreen                        #44944a

	evenglow                                evenglow                        #8f817f

	eventide                                eventide                        #9690ab

	everglade                               everglade                       #222222

	evergreen                               evergreen                       #355e3b

	evque                                  evque                           #563c5c

	faded rose                              fadedrose                       #905d5d

	faience                                 faience                         #007791

	faience                                 faience                         #367588

	fairway                                 fairway                         #1a2421

	fairy green                             fairygreen                      #679267

	fakir                                   fakir                           #fad6a5

	falcon                                  falcon                          #3e322c

	fallow                                  fallow                          #c19a6b

	fandango                                fandango                        #ce4676

	faon                                    faon                            #977f73

	faon                                    faon                            #6f4e37

	faon                                    faon                            #958070

	faon                                    faon                            #635147

	fashion gray                            fashiongray                     #555555

	fawn [brown]                            fawnbrown                       #977f73

	fawn [brown]                            fawnbrown                       #6f4e37

	fawn [brown]                            fawnbrown                       #958070

	fawn [brown]                            fawnbrown                       #635147

	feldspar                                feldspar                        #6aab8e

	fern                                    fern                            #8a9a5b

	fernambucowood                          fernambucowood                  #ab4e52

	fernambucowood                          fernambucowood                  #905d5d

	fern green                              ferngreen                       #8a9a5b

	fern green                              ferngreen                       #8f9779

	ferruginous                             ferruginous                     #9e4732

	ferruginous                             ferruginous                     #80461b

	feuille                                 feuille                         #6f4e37

	feuille morte                           feuillemorte                    #be6516

	feuille morte                           feuillemorte                    #ae6938

	feuille morte                           feuillemorte                    #80461b

	feulamort                               feulamort                       #be6516

	feulamort                               feulamort                       #ae6938

	feulamort                               feulamort                       #80461b

	fez                                     fez                             #be0032

	field's orange vermilion                fieldsorangevermilion           #d9603b

	fiery red                               fieryred                        #e25822

	fiesta                                  fiesta                          #e4717a

	fieulamort                              fieulamort                      #be6516

	fieulamort                              fieulamort                      #ae6938

	fieulamort                              fieulamort                      #80461b

	filbert [brown]                         filbertbrown                    #a67b5b

	filbert [brown]                         filbertbrown                    #996515

	filemot                                 filemot                         #be6516

	filemot                                 filemot                         #ae6938

	filemot                                 filemot                         #80461b

	fillemot                                fillemot                        #be6516

	fillemot                                fillemot                        #ae6938

	fillemot                                fillemot                        #80461b

	firecracker                             firecracker                     #e25822

	firefly                                 firefly                         #bc3f4a

	firefly                                 firefly                         #ab4e52

	fire red                                firered                         #e25822

	fire scarlet                            firescarlet                     #be0032

	fireweed                                fireweed                        #602f6b

	fir [green]                             firgreen                        #355e3b

	fir [green]                             firgreen                        #173620

	fir [green]                             firgreen                        #3a4b47

	firmament                               firmament                       #536878

	firmament blue                          firmamentblue                   #66aabc

	fish grey                               fishgrey                        #5b5842

	fish grey                               fishgrey                        #57554c

	flag                                    flag                            #2f2140

	flame blue                              flameblue                       #4e5180

	flame blue                              flameblue                       #4c516d

	flame orange                            flameorange                     #e25822

	flame [scarlet]                         flamescarlet                    #e25822

	flaming maple                           flamingmaple                    #be0032

	flaming maple                           flamingmaple                    #bc3f4a

	flaming maple                           flamingmaple                    #841b2d

	flaming maple                           flamingmaple                    #ab4e52

	flamingo                                flamingo                        #cb6d51

	flammeous                               flammeous                       #d9603b

	flash                                   flash                           #bc3f4a

	flash                                   flash                           #ab4e52

	flash                                   flash                           #aa381e

	flash                                   flash                           #9e4732

	flax                                    flax                            #c2b280

	flaxen                                  flaxen                          #c2b280

	flaxflower blossom                      flaxflowerblossom               #536878

	flaxflower blue                         flaxflowerblue                  #436b95

	flaxflower blue                         flaxflowerblue                  #8791bf

	flea                                    flea                            #722f37

	flemish blue                            flemishblue                     #36454f

	flesh                                   flesh                           #fad6a5

	flesh                                   flesh                           #bfb8a5

	flesh blond                             fleshblond                      #d9a6a9

	flesh blond                             fleshblond                      #c7ada3

	flesh blond                             fleshblond                      #c2ac99

	flesh ochre                             fleshochre                      #9e4732

	fleur-de-lys                            fleurdelys                      #86608e

	flint                                   flint                           #222222

	flint grey                              flintgrey                       #848482

	flirt                                   flirt                           #ab4e52

	florence brown                          florencebrown                   #905d5d

	florence earth                          florenceearth                   #79443b

	florentine                              florentine                      #e25822

	florentine lake                         florentinelake                  #be0032

	florentine lake                         florentinelake                  #bc3f4a

	florida gold                            floridagold                     #eaa221

	florida gold                            floridagold                     #e3a857

	floss flower blue                       flossflowerblue                 #aa98a9

	flower de luce green                    flowerdelucegreen               #355e3b

	fluorite green                          fluoritegreen                   #679267

	fluorite violet                         fluoriteviolet                  #563c5c

	fluorite violet                         fluoriteviolet                  #301934

	fog                                     fog                             #b9b8b5

	foliage brown                           foliagebrown                    #be6516

	foliage brown                           foliagebrown                    #ae6938

	foliage brown                           foliagebrown                    #80461b

	foliage green                           foliagegreen                    #8a9a5b

	folimort                                folimort                        #be6516

	folimort                                folimort                        #ae6938

	folimort                                folimort                        #80461b

	folkstone                               folkstone                       #ae9b82

	folly                                   folly                           #be0032

	folly                                   folly                           #bc3f4a

	fontainebleau                           fontainebleau                   #905d5d

	forest                                  forest                          #3a4b47

	forest green                            forestgreen                     #4a5d23

	forest of dean red                      forestofdeanred                 #9e4732

	forest of dean red                      forestofdeanred                 #80461b

	forget-me-not [blue]                    forgetmenotblue                 #91a3b0

	forget-me-not [blue]                    forgetmenotblue                 #536878

	formosa                                 formosa                         #d99058

	formosa                                 formosa                         #a67b5b

	forsythia                               forsythia                       #e3a857

	fox                                     fox                             #826644

	fox                                     fox                             #967117

	fox                                     fox                             #6c541e

	foxglove                                foxglove                        #604e97

	foxglove                                foxglove                        #604e81

	fox trot                                foxtrot                         #66ada4

	fragonard                               fragonard                       #ab4e52

	france                                  france                          #00304e

	france rose                             francerose                      #e4717a

	france rose                             francerose                      #282022

	freedom                                 freedom                         #00304e

	freestone                               freestone                       #fad6a5

	french beige                            frenchbeige                     #a67b5b

	french berries                          frenchberries                   #eaa221

	french berries                          frenchberries                   #e3a857

	french blue                             frenchblue                      #0067a5

	french blue                             frenchblue                      #545aa7

	french green                            frenchgreen                     #679267

	french green                            frenchgreen                     #3b7861

	french grey                             frenchgrey                      #b2beb5

	french grey                             frenchgrey                      #b9b8b5

	french lilac                            frenchlilac                     #86608e

	french lilac                            frenchlilac                     #796878

	french maroon                           frenchmaroon                    #905d5d

	french nude                             frenchnude                      #977f73

	french nude                             frenchnude                      #a67b5b

	french nude                             frenchnude                      #958070

	french ochre                            frenchochre                     #e3a857

	french pink                             frenchpink                      #fad6a5

	french pink                             frenchpink                      #c9ae5d

	french purple                           frenchpurple                    #a8516e

	french scarlet                          frenchscarlet                   #be0032

	french ultramarine                      frenchultramarine               #0067a5

	french ultramarine                      frenchultramarine               #545aa7

	french vermilion                        frenchvermilion                 #e25822

	[french] veronese green                 frenchveronesegreen             #007959

	[french] veronese green                 frenchveronesegreen             #3b7861

	french white                            frenchwhite                     #bfb9bd

	french white                            frenchwhite                     #e8ccd7

	french yellow                           frenchyellow                    #ae6938

	french yellow                           frenchyellow                    #996515

	french yellow                           frenchyellow                    #a67b5b

	friar                                   friar                           #635147

	friar                                   friar                           #3e322c

	frost grey                              frostgrey                       #848482

	frosty green                            frostygreen                     #3a4b47

	fuchsia                                 fuchsia                         #870074

	fudge                                   fudge                           #6f4e37

	fujiyama                                fujiyama                        #79443b

	fuscous                                 fuscous                         #5b504f

	fustet                                  fustet                          #fada5e

	fustet                                  fustet                          #d4af37

	fustic                                  fustic                          #c9ae5d

	fustic                                  fustic                          #c2b280

	gage green                              gagegreen                       #8f9779

	gaiety                                  gaiety                          #e4717a

	galleon                                 galleon                         #3e322c

	gallstone                               gallstone                       #d4af37

	gambia                                  gambia                          #6f4e37

	gamboge                                 gamboge                         #d4af37

	gardenia green                          gardeniagreen                   #3a4b47

	gargoyle                                gargoyle                        #555555

	garland green                           garlandgreen                    #4a5d23

	garland green                           garlandgreen                    #355e3b

	garnet                                  garnet                          #722f37

	garnet brown                            garnetbrown                     #722f37

	garnet brown                            garnetbrown                     #79443b

	garnet red                              garnetred                       #841b2d

	garnet red                              garnetred                       #722f37

	garter blue                             garterblue                      #4c516d

	gaude                                   gaude                           #eae679

	gaude                                   gaude                           #b9b459

	gaude lake                              gaudelake                       #f8de7e

	gaudy green                             gaudygreen                      #8a9a5b

	gaudy green                             gaudygreen                      #4a5d23

	gay green                               gaygreen                        #8a9a5b

	gazelle [brown]                         gazellebrown                    #635147

	gazelle [brown]                         gazellebrown                    #7e6d5a

	geisha                                  geisha                          #915c83

	geisha                                  geisha                          #a8516e

	geisha                                  geisha                          #915f6d

	gendarme [blue]                         gendarmeblue                    #317873

	gendarme [blue]                         gendarmeblue                    #367588

	generall                                generall                        #fada5e

	generall                                generall                        #d4af37

	generall                                generall                        #f8de7e

	generall                                generall                        #c9ae5d

	genestrole                              genestrole                      #eae679

	genestrole                              genestrole                      #b9b459

	genet                                   genet                           #eae679

	genet                                   genet                           #b9b459

	geneva blue                             genevablue                      #4c516d

	genista                                 genista                         #d99058

	genoa blue                              genoablue                       #536878

	gentian                                 gentian                         #436b95

	gentian                                 gentian                         #8791bf

	gentian blue                            gentianblue                     #4e5180

	genuine ultramarine [blue]              genuineultramarineblue          #00a1c2

	genuine ultramarine [blue]              genuineultramarineblue          #0067a5

	geranium                                geranium                        #bc3f4a

	geranium lake                           geraniumlake                    #be0032

	geranium petal                          geraniumpetal                   #be0032

	geranium pink                           geraniumpink                    #e4717a

	ghent                                   ghent                           #00304e

	giallolini                              giallolini                      #f8de7e

	giallolini                              giallolini                      #c9ae5d

	giallolini                              giallolini                      #f3e5ab

	giallolini                              giallolini                      #c2b280

	giallolino                              giallolino                      #f8de7e

	giallolino                              giallolino                      #c9ae5d

	giallolino                              giallolino                      #f3e5ab

	giallolino                              giallolino                      #c2b280

	gigas                                   gigas                           #86608e

	gild                                    gild                            #eaa221

	gild                                    gild                            #e3a857

	gilded                                  gilded                          #eaa221

	gilded                                  gilded                          #e3a857

	gilt                                    gilt                            #eaa221

	gilt                                    gilt                            #e3a857

	gingeline                               gingeline                       #80461b

	gingeoline                              gingeoline                      #80461b

	ginger                                  ginger                          #80461b

	gingerline                              gingerline                      #80461b

	gingerspice                             gingerspice                     #79443b

	gingioline                              gingioline                      #80461b

	giraffe                                 giraffe                         #ae6938

	giraffe                                 giraffe                         #a67b5b

	glacier                                 glacier                         #36454f

	glacier blue                            glacierblue                     #536878

	gladiolus                               gladiolus                       #bc3f4a

	glaeul                                 glaeul                          #ad8884

	glaeul                                 glaeul                          #c48379

	glass green                             glassgreen                      #c9dc89

	glass grey                              glassgrey                       #b4bcc0

	glaucous                                glaucous                        #dadfb7

	glaucous-blue                           glaucousblue                    #536878

	glaucous-green                          glaucousgreen                   #c7e6d7

	glaucous-grey                           glaucousgrey                    #b4bcc0

	glaucous-grey                           glaucousgrey                    #b9b8b5

	glint o'gold                            glintogold                      #c9ae5d

	gloaming                                gloaming                        #ad8884

	glory                                   glory                           #272458

	glory                                   glory                           #32174d

	gloxinia                                gloxinia                        #602f6b

	gmelin's blue                           gmelinsblue                     #0067a5

	gmelin's blue                           gmelinsblue                     #545aa7

	gnaphalium green                        gnaphaliumgreen                 #8da399

	goat                                    goat                            #5b504f

	gobelin blue                            gobelinblue                     #536878

	gobelin blue                            gobelinblue                     #367588

	gobelin blue                            gobelinblue                     #536878

	gobelin green                           gobelingreen                    #3b7861

	gobelin scarlet                         gobelinscarlet                  #be0032

	gold                                    gold                            #967117

	gold bronze                             goldbronze                      #635147

	gold bronze                             goldbronze                      #7e6d5a

	gold brown                              goldbrown                       #80461b

	gold earth                              goldearth                       #e3a857

	golden                                  golden                          #c19a6b

	golden                                  golden                          #a18f60

	golden                                  golden                          #967117

	golden brown                            goldenbrown                     #80461b

	golden chestnut                         goldenchestnut                  #6f4e37

	golden coral                            goldencoral                     #f88379

	golden corn                             goldencorn                      #e3a857

	golden feather                          goldenfeather                   #fada5e

	golden feather                          goldenfeather                   #d4af37

	golden feather                          goldenfeather                   #f8de7e

	golden feather                          goldenfeather                   #c9ae5d

	golden glow                             goldenglow                      #e3a857

	golden glow                             goldenglow                      #d4af37

	golden green                            goldengreen                     #c2b280

	golden green                            goldengreen                     #a18f60

	golden ochre                            goldenochre                     #d99058

	golden poppy                            goldenpoppy                     #e25822

	golden rod                              goldenrod                       #d4af37

	golden wheat                            goldenwheat                     #e3a857

	golden yellow                           goldenyellow                    #eaa221

	golden yellow                           goldenyellow                    #e3a857

	golden yellow                           goldenyellow                    #be8a3d

	golden yellow                           goldenyellow                    #996515

	gold leaf                               goldleaf                        #d99058

	gold leaf                               goldleaf                        #be8a3d

	gold ochre                              goldochre                       #d99058

	gold yellow                             goldyellow                      #eaa221

	gold yellow                             goldyellow                      #e3a857

	golf green                              golfgreen                       #355e3b

	golf [red]                              golfred                         #bc3f4a

	golf [red]                              golfred                         #ab4e52

	gooseberry                              gooseberry                      #673147

	gooseberry green                        gooseberrygreen                 #679267

	goose grey                              goosegrey                       #483c32

	gorevan                                 gorevan                         #6f4e37

	goura                                   goura                           #50404d

	goya                                    goya                            #bc3f4a

	grain                                   grain                           #fad6a5

	grain                                   grain                           #c19a6b

	grain                                   grain                           #ae9b82

	grain                                   grain                           #be0032

	grain in grain                          grainingrain                    #be0032

	granada                                 granada                         #43302e

	granada                                 granada                         #3e322c

	granat                                  granat                          #be0032

	granat                                  granat                          #e25822

	granatflower                            granatflower                    #be0032

	granatflower                            granatflower                    #e25822

	granite                                 granite                         #8b8589

	granite blue                            graniteblue                     #b9b8b5

	grape                                   grape                           #2f2140

	grape blue                              grapeblue                       #2f2140

	grapefruit                              grapefruit                      #b9b459

	grape green                             grapegreen                      #867e36

	grapejuice                              grapejuice                      #602f6b

	grapenuts                               grapenuts                       #6f4e37

	graphite                                graphite                        #50404d

	graphite                                graphite                        #222222

	grass green                             grassgreen                      #8a9a5b

	grasshopper                             grasshopper                     #8a9a5b

	gravel                                  gravel                          #ae9b82

	grayn                                   grayn                           #be0032

	grebe                                   grebe                           #5d555b

	grebe                                   grebe                           #555555

	grecian rose                            grecianrose                     #ad8884

	grecian rose                            grecianrose                     #c48379

	green ash                               greenash                        #93c592

	green finch                             greenfinch                      #867e36

	green slate                             greenslate                      #5e716a

	green slate                             greenslate                      #4e5755

	green stone                             greenstone                      #8f9779

	grenadine red                           grenadinered                    #cb6d51

	grenat                                  grenat                          #722f37

	gretna green                            gretnagreen                     #679267

	gretna green                            gretnagreen                     #3b7861

	grey 31                                 grey31                          #5b504f

	grey dawn                               greydawn                        #8b8589

	grey dawn                               greydawn                        #848482

	grey drab                               greydrab                        #8c8767

	greyn                                   greyn                           #be0032

	grey stone                              greystone                       #8a8776

	grey ultramarine ash                    greyultramarineash              #b2beb5

	gridelin                                gridelin                        #a8516e

	griffin                                 griffin                         #555555

	gris-de-lin                             grisdelin                       #a8516e

	grotto [blue]                           grottoblue                      #007791

	grouse                                  grouse                          #635147

	grouse                                  grouse                          #7e6d5a

	guignet's green                         guignetsgreen                   #004b49

	guimet's blue                           guimetsblue                     #0067a5

	guimet's blue                           guimetsblue                     #545aa7

	guinea green                            guineagreen                     #007a74

	guinea hen                              guineahen                       #555555

	gull                                    gull                            #848482

	gull grey                               gullgrey                        #8b8589

	gunmetal                                gunmetal                        #555555

	gypsy                                   gypsy                           #80461b

	gypsy red                               gypsyred                        #ab4e52

	haematite red                           haematitered                    #722f37

	haematite red                           haematitered                    #905d5d

	haematite red                           haematitered                    #543d3f

	hair brown                              hairbrown                       #5b504f

	hair brown                              hairbrown                       #555555

	hamadan                                 hamadan                         #967117

	hamburg lake                            hamburglake                     #be0032

	hamburg lake                            hamburglake                     #bc3f4a

	hampstead brown                         hampsteadbrown                  #674c47

	hankow                                  hankow                          #00304e

	hankow                                  hankow                          #252440

	hankow                                  hankow                          #4c516d

	harbor blue                             harborblue                      #51585e

	harlem blue                             harlemblue                      #2e8495

	harlem blue                             harlemblue                      #436b95

	harlequin                               harlequin                       #44944a

	harrison red                            harrisonred                     #be0032

	harvard crimson                         harvardcrimson                  #ab4e52

	harvest                                 harvest                         #ae6938

	harvest                                 harvest                         #a67b5b

	hathi gray                              hathigray                       #7d8984

	hathor                                  hathor                          #00a1c2

	hathor                                  hathor                          #0067a5

	havana rose                             havanarose                      #905d5d

	hay                                     hay                             #b9b57d

	hazel                                   hazel                           #a67b5b

	hazel                                   hazel                           #996515

	hazelnut                                hazelnut                        #a67b5b

	hazelnut                                hazelnut                        #996515

	hazy blue                               hazyblue                        #93c592

	hazy blue                               hazyblue                        #6aab8e

	hazy blue                               hazyblue                        #8da399

	heather                                 heather                         #915f6d

	hebe                                    hebe                            #ab4e52

	heliotrope                              heliotrope                      #5d3954

	heliotrope grey                         heliotropegrey                  #aa98a9

	heliotrope grey                         heliotropegrey                  #796878

	heliotrope grey                         heliotropegrey                  #8b8589

	hellebore green                         helleboregreen                  #4a5d23

	hellebore green                         helleboregreen                  #8a9a5b

	hellebore green                         helleboregreen                  #8f9779

	hellebore green                         helleboregreen                  #4a5d23

	hellebore green                         helleboregreen                  #515744

	hellebore red                           helleborered                    #a8516e

	helvetia blue                           helvetiablue                    #00416a

	hemlock                                 hemlock                         #1a2421

	hemp                                    hemp                            #8c8767

	henna                                   henna                           #882d17

	hepatica                                hepatica                        #86608e

	hermosa pink                            hermosapink                     #ea9399

	hermosa pink                            hermosapink                     #dea5a4

	heron                                   heron                           #536878

	hibernian green                         hiberniangreen                  #355e3b

	highland green                          highlandgreen                   #1e2321

	hindu                                   hindu                           #3e322c

	hispano                                 hispano                         #80461b

	hockey                                  hockey                          #be0032

	holland blue                            hollandblue                     #00304e

	holly berry                             hollyberry                      #bc3f4a

	holly berry                             hollyberry                      #ab4e52

	holly green                             hollygreen                      #4a5d23

	holly green                             hollygreen                      #00622d

	holly green                             hollygreen                      #355e3b

	hollyhock                               hollyhock                       #78184a

	hollywood                               hollywood                       #841b2d

	hollywood                               hollywood                       #ab4e52

	homage blue                             homageblue                      #252440

	honey beige                             honeybeige                      #c19a6b

	honey bird                              honeybird                       #239eba

	honeydew                                honeydew                        #f88379

	honeydew                                honeydew                        #d99058

	honeysuckle                             honeysuckle                     #c19a6b

	honeysweet                              honeysweet                      #c9ae5d

	honeysweet                              honeysweet                      #c2b280

	honey [yellow]                          honeyyellow                     #a18f60

	hopi                                    hopi                            #a67b5b

	horace vernet's blue                    horacevernetsblue               #30267a

	horizon [blue]                          horizonblue                     #dfede8

	horsechestnut                           horsechestnut                   #3e322c

	hortense violet                         hortenseviolet                  #86608e

	hortensia                               hortensia                       #563c5c

	hortensia                               hortensia                       #301934

	hortensia                               hortensia                       #50404d

	hortensia                               hortensia                       #291e29

	huckleberry                             huckleberry                     #2f2140

	hudson seal                             hudsonseal                      #555555

	hudson seal                             hudsonseal                      #222222

	hummingbird                             hummingbird                     #317873

	hungarian blue                          hungarianblue                   #436b95

	hungarian green                         hungariangreen                  #679267

	hunter [green]                          huntergreen                     #173620

	hunter's green                          huntersgreen                    #173620

	huron                                   huron                           #317873

	hussar                                  hussar                          #536878

	hyacinth                                hyacinth                        #8c82b6

	hyacinth                                hyacinth                        #86608e

	hyacinth blue                           hyacinthblue                    #272458

	hyacinth red                            hyacinthred                     #b4745e

	hyacinth violet                         hyacinthviolet                  #602f6b

	hydrangea blue                          hydrangeablue                   #536878

	hydrangea blue                          hydrangeablue                   #8c92ac

	hydrangea pink                          hydrangeapink                   #dea5a4

	hydrangea red                           hydrangeared                    #905d5d

	hydro                                   hydro                           #536878

	hypermic red                            hypermicred                     #be0032

	hyssop violet                           hyssopviolet                    #796878

	ibis pink                               ibispink                        #f88379

	ibis pink                               ibispink                        #d9a6a9

	ibis red                                ibisred                         #43302e

	iceberg                                 iceberg                         #536878

	immenssee                               immenssee                       #5e716a

	imperial                                imperial                        #602f6b

	imperial                                imperial                        #563c5c

	imperial                                imperial                        #702963

	imperial                                imperial                        #5d3954

	imperial blue                           imperialblue                    #00416a

	imperial green                          imperialgreen                   #679267

	imperial green                          imperialgreen                   #3eb489

	imperial jade                           imperialjade                    #355e3b

	imperial purple                         imperialpurple                  #554c69

	imperial stone                          imperialstone                   #967117

	imperial yellow                         imperialyellow                  #e3a857

	inca gold                               incagold                        #e3a857

	indebaudias                             indebaudias                     #50404d

	inde blue                               indeblue                        #50404d

	independence                            independence                    #2f2140

	india ink                               indiaink                        #555555

	indian                                  indian                          #ae6938

	indiana                                 indiana                         #ce4676

	indian blue                             indianblue                      #50404d

	indian brown                            indianbrown                     #483c32

	indian brown                            indianbrown                     #6c541e

	indian brown                            indianbrown                     #3b3121

	indian buff                             indianbuff                      #c19a6b

	indian lake                             indianlake                      #a8516e

	indian orange                           indianorange                    #e25822

	indian pink                             indianpink                      #a87c6d

	indian purple                           indianpurple                    #673147

	indian red                              indianred                       #79443b

	indian red                              indianred                       #882d17

	indian saffron                          indiansaffron                   #f8de7e

	indian tan                              indiantan                       #996515

	indian tan                              indiantan                       #c19a6b

	indian tan                              indiantan                       #826644

	indian turquoise                        indianturquoise                 #317873

	indian yellow                           indianyellow                    #eaa221

	indian yellow                           indianyellow                    #e3a857

	india red                               indiared                        #56070c

	india spice                             indiaspice                      #a67b5b

	india tan                               indiatan                        #6f4e37

	indico                                  indico                          #50404d

	indico carmine                          indicocarmine                   #007791

	indigo                                  indigo                          #50404d

	indigo carmine                          indigocarmine                   #007791

	indigo extract                          indigoextract                   #007791

	indo                                    indo                            #536878

	infanta                                 infanta                         #436b95

	infantry                                infantry                        #536878

	infernal blue                           infernalblue                    #252440

	ingenue                                 ingenue                         #dadfb7

	ingenue                                 ingenue                         #8f9779

	ink black                               inkblack                        #51585e

	ink blue                                inkblue                         #36454f

	intense blue                            intenseblue                     #4e5180

	intense blue                            intenseblue                     #252440

	international                           international                   #536878

	invisible green                         invisiblegreen                  #004b49

	invisible green                         invisiblegreen                  #1a2421

	ionian blue                             ionianblue                      #36454f

	ionian blue                             ionianblue                      #202830

	iris                                    iris                            #91a3b0

	iris                                    iris                            #aa98a9

	irisglow                                irisglow                        #8b8589

	iris green                              irisgreen                       #355e3b

	iris green                              irisgreen                       #679267

	iris mauve                              irismauve                       #ecd5c5

	iris mauve                              irismauve                       #c7ada3

	iris mauve                              irismauve                       #c2ac99

	iron                                    iron                            #483c32

	iron                                    iron                            #222222

	iron blue                               ironblue                        #555555

	iron brown                              ironbrown                       #483c32

	iron brown                              ironbrown                       #222222

	iron buff                               ironbuff                        #c19a6b

	iron buff                               ironbuff                        #c2b280

	iron crocus                             ironcrocus                      #79443b

	iron grey                               irongrey                        #555555

	iron minium                             ironminium                      #905d5d

	iron minium                             ironminium                      #674c47

	iron oxide red                          ironoxidered                    #882d17

	iron oxide red                          ironoxidered                    #80461b

	iron red                                ironred                         #882d17

	iron saffron                            ironsaffron                     #79443b

	iron yellow                             ironyellow                      #ed872d

	iron yellow                             ironyellow                      #d99058

	isabella                                isabella                        #c19a6b

	isabella                                isabella                        #826644

	isabella                                isabella                        #a18f60

	isabella                                isabella                        #967117

	ispahan                                 ispahan                         #36454f

	italian blue                            italianblue                     #0085a1

	italian blue                            italianblue                     #007791

	italian lake                            italianlake                     #e3a857

	italian ochre                           italianochre                    #ae6938

	italian ochre                           italianochre                    #a67b5b

	italian pink                            italianpink                     #fada5e

	italian pink                            italianpink                     #d4af37

	italian pink                            italianpink                     #f8de7e

	italian pink                            italianpink                     #c9ae5d

	italian straw                           italianstraw                    #f3e5ab

	italian straw                           italianstraw                    #c2b280

	ivory brown                             ivorybrown                      #6c541e

	ivory brown                             ivorybrown                      #3b3121

	ivory white                             ivorywhite                      #f3e5ab

	ivory [yellow]                          ivoryyellow                     #f3e5ab

	ivy [green]                             ivygreen                        #665d1e

	ivy [green]                             ivygreen                        #5b5842

	jacaranda brown                         jacarandabrown                  #483c32

	jacinthe                                jacinthe                        #d99058

	jack rabbit                             jackrabbit                      #555555

	jack rose                               jackrose                        #be0032

	jacqueminot                             jacqueminot                     #ab4e52

	jadesheen                               jadesheen                       #679267

	jadesheen                               jadesheen                       #355e3b

	jadesheen                               jadesheen                       #5e716a

	jaffa orange                            jaffaorange                     #cb6d51

	jaffa orange                            jaffaorange                     #d99058

	jaffi                                   jaffi                           #57554c

	jalapa                                  jalapa                          #80461b

	japan blue                              japanblue                       #00304e

	japan earth                             japanearth                      #6f4e37

	japanese blue                           japaneseblue                    #5e716a

	japanese green                          japanesegreen                   #5e716a

	japanese red                            japanesered                     #882d17

	japanese yellow                         japaneseyellow                  #ffb7a5

	japanese yellow                         japaneseyellow                  #d9603b

	japan rose                              japanrose                       #d9a6a9

	japan rose                              japanrose                       #d99058

	japonica                                japonica                        #905d5d

	jasmine                                 jasmine                         #fada5e

	jasmine                                 jasmine                         #f8de7e

	jasper                                  jasper                          #1a2421

	jasper green                            jaspergreen                     #3b7861

	jasper pink                             jasperpink                      #f88379

	jasper red                              jasperred                       #ab4e52

	java                                    java                            #422518

	java brown                              javabrown                       #3e322c

	javel green                             javelgreen                      #b9b459

	jay [blue]                              jayblue                         #436b95

	jean bart                               jeanbart                        #2f2140

	jew's pitch                             jewspitch                       #483c32

	jockey                                  jockey                          #ab4e52

	jonquil [yellow]                        jonquilyellow                   #fbc97f

	jonquil [yellow]                        jonquilyellow                   #e3a857

	jonquil [yellow]                        jonquilyellow                   #fad6a5

	jonquil [yellow]                        jonquilyellow                   #f8de7e

	jonquil [yellow]                        jonquilyellow                   #c9ae5d

	josephine                               josephine                       #a67b5b

	jouvence blue                           jouvenceblue                    #317873

	jouvence blue                           jouvenceblue                    #367588

	judee                                   judee                           #796878

	jungle green                            junglegreen                     #1c352d

	jungle green                            junglegreen                     #1a2421

	juniper                                 juniper                         #3a4b47

	kabistan                                kabistan                        #3a4b47

	kaffa                                   kaffa                           #674c47

	kaiser brown                            kaiserbrown                     #80461b

	kangaroo                                kangaroo                        #57554c

	kara                                    kara                            #317873

	kara dagh                               karadagh                        #222222

	kasha-beige                             kashabeige                      #8a8776

	kasha-beige                             kashabeige                      #848482

	kashan                                  kashan                          #007791

	kashmir                                 kashmir                         #355e3b

	kashmir                                 kashmir                         #3b7861

	kashmir                                 kashmir                         #5e716a

	kazak                                   kazak                           #79443b

	kentucky green                          kentuckygreen                   #665d1e

	kermanshah                              kermanshah                      #635147

	kermes                                  kermes                          #be0032

	kermes berries                          kermesberries                   #be0032

	kesseler yellow                         kesseleryellow                  #fada5e

	kesseler yellow                         kesseleryellow                  #d4af37

	kesseler yellow                         kesseleryellow                  #f8de7e

	kesseler yellow                         kesseleryellow                  #c9ae5d

	kettledrum                              kettledrum                      #79443b

	khaki                                   khaki                           #c19a6b

	khaki                                   khaki                           #826644

	khiva                                   khiva                           #9e4732

	kildare green                           kildaregreen                    #8a9a5b

	killarney [green]                       killarneygreen                  #679267

	killarney [green]                       killarneygreen                  #3b7861

	kinema red                              kinemared                       #bc3f4a

	kingfisher                              kingfisher                      #00304e

	king's blue                             kingsblue                       #70a3cc

	king's blue                             kingsblue                       #00416a

	king's blue                             kingsblue                       #436b95

	king's blue                             kingsblue                       #436b95

	king's yellow                           kingsyellow                     #fada5e

	king's yellow                           kingsyellow                     #d4af37

	king's yellow                           kingsyellow                     #f8de7e

	king's yellow                           kingsyellow                     #c9ae5d

	kirchberger green                       kirchbergergreen                #3eb489

	kis kilim                               kiskilim                        #6f4e37

	kobe                                    kobe                            #882d17

	kolinsky                                kolinsky                        #422518

	korea                                   korea                           #9e4732

	korea                                   korea                           #79443b

	kork                                    kork                            #a8516e

	korkir                                  korkir                          #a8516e

	kremlin                                 kremlin                         #004b49

	kronberg's green                        kronbergsgreen                  #4a5d23

	kurdistan                               kurdistan                       #36454f

	kurdistan                               kurdistan                       #202428

	kurdistan                               kurdistan                       #50404d

	kurdistan                               kurdistan                       #242124

	kyoto                                   kyoto                           #f88379

	kyoto                                   kyoto                           #ed872d

	kyoto                                   kyoto                           #d99058

	labrador                                labrador                        #00304e

	labrador                                labrador                        #36454f

	lac lake                                laclake                         #a8516e

	lacquer red                             lacquerred                      #9e4732

	laelia pink                             laeliapink                      #c17e91

	laelia pink                             laeliapink                      #af868e

	la france pink                          lafrancepink                    #dea5a4

	lagoon                                  lagoon                          #66ada4

	lake                                    lake                            #be0032

	lake                                    lake                            #bc3f4a

	lake blue                               lakeblue                        #317873

	lama                                    lama                            #483c32

	lambert's blue                          lambertsblue                    #436b95

	lapis                                   lapis                           #317873

	lariat                                  lariat                          #a67b5b

	lariat                                  lariat                          #c19a6b

	lark                                    lark                            #c2b280

	lark                                    lark                            #a18f60

	lark                                    lark                            #967117

	larkspur                                larkspur                        #367588

	latericeous                             latericeous                     #80461b

	latericeous                             latericeous                     #6f4e37

	lateritious                             lateritious                     #80461b

	lateritious                             lateritious                     #6f4e37

	latoun                                  latoun                          #c9ae5d

	latten                                  latten                          #c9ae5d

	laundry blue                            laundryblue                     #00416a

	laundry blue                            laundryblue                     #436b95

	laurel green                            laurelgreen                     #867e36

	laurel oak                              laureloak                       #79443b

	laurel pink                             laurelpink                      #e4717a

	laurel pink                             laurelpink                      #ab4e52

	lava                                    lava                            #222222

	la vallire                             lavallire                       #796878

	lavender                                lavender                        #aa98a9

	lawn green                              lawngreen                       #8a9a5b

	lead                                    lead                            #5d555b

	lead                                    lead                            #555555

	lead grey                               leadgrey                        #8c8767

	lead ochre                              leadochre                       #f3e5ab

	leadville                               leadville                       #5d555b

	leaf mold                               leafmold                        #422518

	leaf red                                leafred                         #be0032

	leather                                 leather                         #ae6938

	leather brown                           leatherbrown                    #6f4e37

	leather lake                            leatherlake                     #be6516

	leather lake                            leatherlake                     #ae6938

	leather lake                            leatherlake                     #80461b

	leek green                              leekgreen                       #867e36

	leek green                              leekgreen                       #8a9a5b

	leghorn                                 leghorn                         #f3e5ab

	leghorn                                 leghorn                         #c2b280

	legion blue                             legionblue                      #00304e

	leitch's blue                           leitchsblue                     #00a1c2

	leithner's blue                         leithnersblue                   #007791

	lemmian earth                           lemmianearth                    #79443b

	lemnian ruddle                          lemnianruddle                   #79443b

	lemnos                                  lemnos                          #79443b

	lemon chrome                            lemonchrome                     #fada5e

	lemon yellow                            lemonyellow                     #fada5e

	lemon yellow                            lemonyellow                     #d4af37

	lemon yellow                            lemonyellow                     #f8de7e

	lemon yellow                            lemonyellow                     #c9ae5d

	lettuce green                           lettucegreen                    #8a9a5b

	levant red                              levantred                       #ab4e52

	leyden blue                             leydenblue                      #007791

	liberia                                 liberia                         #3e322c

	liberty                                 liberty                         #0067a5

	liberty                                 liberty                         #545aa7

	liberty blue                            libertyblue                     #4e5180

	liberty green                           libertygreen                    #8a9a5b

	lichen                                  lichen                          #515744

	lichen green                            lichengreen                     #b2beb5

	lido                                    lido                            #967117

	lierre                                  lierre                          #665d1e

	lierre                                  lierre                          #5b5842

	light blue 1                            lightblue1                      #dadfb7

	light blue 2, light blue 3              lightblue2lightblue3            #c7e6d7

	light blue 2, light blue 3              lightblue2lightblue3            #bcd4e6

	light blue 4                            lightblue4                      #bcd4e6

	light blue 6                            lightblue6                      #91a3b0

	light blue,, light blue 5               lightbluelightblue5             #c7e6d7

	light brunswick green                   lightbrunswickgreen             #355e3b

	light chrome green                      lightchromegreen                #355e3b

	light chrome yellow                     lightchromeyellow               #d4af37

	light grge                             lightgrge                       #8a8776

	light gunmetal                          lightgunmetal                   #555555

	light red                               lightred                        #cb6d51

	light stone                             lightstone                      #c2b280

	light stone                             lightstone                      #a18f60

	light wedgwood [blue]                   lightwedgwoodblue               #536878

	lilac                                   lilac                           #dea5a4

	lilac                                   lilac                           #ad8884

	lilac                                   lilac                           #aa98a9

	lilaceous                               lilaceous                       #dea5a4

	lilaceous                               lilaceous                       #ad8884

	lilac gray                              lilacgray                       #c2ac99

	lilac gray                              lilacgray                       #bfb8a5

	lilas                                   lilas                           #dea5a4

	lilas                                   lilas                           #ad8884

	lilium                                  lilium                          #86608e

	lilium                                  lilium                          #915c83

	lily green                              lilygreen                       #8a9a5b

	lily green                              lilygreen                       #4e5755

	limawood                                limawood                        #ab4e52

	limawood                                limawood                        #905d5d

	lime blue                               limeblue                        #436b95

	lime green                              limegreen                       #b9b57d

	lime green                              limegreen                       #867e36

	limestone                               limestone                       #5b504f

	limestone                               limestone                       #555555

	lime [yellow]                           limeyellow                      #c9ae5d

	lime [yellow]                           limeyellow                      #c2b280

	limoges                                 limoges                         #0067a5

	limoges                                 limoges                         #436b95

	lincoln green                           lincolngreen                    #4a5d23

	lincoln red                             lincolnred                      #be0032

	linden green                            lindengreen                     #b9b459

	linden yellow                           lindenyellow                    #b9b459

	linoleum brown                          linoleumbrown                   #826644

	lint                                    lint                            #8c8767

	lint-white                              lintwhite                       #f3e5ab

	lint-white                              lintwhite                       #c2b280

	lion                                    lion                            #c19a6b

	lion                                    lion                            #826644

	lion tawny                              liontawny                       #c19a6b

	lion tawny                              liontawny                       #826644

	liqueur green                           liqueurgreen                    #eae679

	liqueur green                           liqueurgreen                    #b9b459

	liseran purple                          liseranpurple                   #9e4f88

	litho purple                            lithopurple                     #86608e

	litho purple                            lithopurple                     #563c5c

	litho purple                            lithopurple                     #796878

	liver                                   liver                           #674c47

	liver brown                             liverbrown                      #674c47

	liver maroon                            livermaroon                     #674c47

	livid                                   livid                           #5d555b

	livid                                   livid                           #555555

	livid brown                             lividbrown                      #905d5d

	livid pink                              lividpink                       #c4aead

	livid purple                            lividpurple                     #836479

	livid violet                            lividviolet                     #ad8884

	lizard bronze                           lizardbronze                    #867e36

	lizard [green]                          lizardgreen                     #3b7861

	loam                                    loam                            #5b504f

	lobelia [blue]                          lobeliablue                     #604e97

	lobelia violet                          lobeliaviolet                   #796878

	lobster                                 lobster                         #aa381e

	lobster                                 lobster                         #9e4732

	locarno green                           locarnogreen                    #3b7861

	loganberry                              loganberry                      #563c5c

	log cabin                               logcabin                        #5b504f

	logwood                                 logwood                         #291e29

	logwood                                 logwood                         #242124

	logwood blue                            logwoodblue                     #51585e

	lo-kao                                  lokao                           #93c592

	lo-kao                                  lokao                           #6aab8e

	london brown                            londonbrown                     #43302e

	london smoke                            londonsmoke                     #222222

	long beach                              longbeach                       #c19a6b

	longchamps                              longchamps                      #a67b5b

	longchamps                              longchamps                      #958070

	lotus                                   lotus                           #c08081

	lotus                                   lotus                           #ad8884

	lotus                                   lotus                           #c48379

	louis philippe                          louisphilippe                   #8791bf

	loutre                                  loutre                          #6c541e

	loutre                                  loutre                          #3b3121

	louvain                                 louvain                         #536878

	louvain                                 louvain                         #51585e

	love bird                               lovebird                        #7e9f2e

	love-in-a-mist                          loveinamist                     #66aabc

	lucerne blue                            lucerneblue                     #8791bf

	luciole                                 luciole                         #81878b

	lucky stone                             luckystone                      #91a3b0

	lumiere blue                            lumiereblue                     #66ada4

	lumiere green                           lumieregreen                    #c9dc89

	lupine                                  lupine                          #70a3cc

	lupine                                  lupine                          #8791bf

	luteous                                 luteous                         #eae679

	luteous                                 luteous                         #b9b459

	luxor                                   luxor                           #545aa7

	lyons blue                              lyonsblue                       #0067a5

	macaroon                                macaroon                        #c19a6b

	madder                                  madder                          #bc3f4a

	madder                                  madder                          #ab4e52

	madder                                  madder                          #d9603b

	madder                                  madder                          #cb6d51

	madder blue                             madderblue                      #9690ab

	madder blue                             madderblue                      #554c69

	madder brown                            madderbrown                     #79443b

	madder carmine                          maddercarmine                   #ab4e52

	madder carmine                          maddercarmine                   #a8516e

	madder indian red                       madderindianred                 #79443b

	madder lake                             madderlake                      #bc3f4a

	madder lake                             madderlake                      #b3446c

	madder pink                             madderpink                      #b3446c

	madder red                              madderred                       #bc3f4a

	madder red                              madderred                       #ab4e52

	madder red                              madderred                       #d9603b

	madder red                              madderred                       #cb6d51

	madder violet                           madderviolet                    #563c5c

	madeline blue                           madelineblue                    #4c516d

	madonna                                 madonna                         #436b95

	madonna                                 madonna                         #536878

	madrid                                  madrid                          #635147

	madrid                                  madrid                          #7e6d5a

	madura                                  madura                          #826644

	magenta                                 magenta                         #78184a

	magenta rose                            magentarose                     #78184a

	magnolia                                magnolia                        #c08081

	magnolia                                magnolia                        #905d5d

	magnolia                                magnolia                        #c48379

	mahogany                                mahogany                        #79443b

	mahogany brown                          mahoganybrown                   #79443b

	mahogany red                            mahoganyred                     #79443b

	maiden's blush                          maidensblush                    #a87c6d

	maintenon                               maintenon                       #cb6d51

	maise                                   maise                           #e3a857

	maise                                   maise                           #fad6a5

	maise                                   maise                           #c9ae5d

	majolica                                majolica                        #00304e

	majolica blue                           majolicablue                    #00304e

	majolica blue                           majolicablue                    #252440

	majolica blue                           majolicablue                    #4c516d

	majolica earth                          majolicaearth                   #882d17

	malabar                                 malabar                         #6f4e37

	malacca                                 malacca                         #c19a6b

	malacca                                 malacca                         #c2b280

	malachite green                         malachitegreen                  #679267

	malaga                                  malaga                          #722f37

	malaga                                  malaga                          #56070c

	malay                                   malay                           #6f4e37

	mallard                                 mallard                         #004958

	mallow pink                             mallowpink                      #d597ae

	mallow purple                           mallowpurple                    #a8516e

	mallow red                              mallowred                       #a8516e

	malmaison                               malmaison                       #f8de7e

	malmaison                               malmaison                       #c9ae5d

	malmaison rose                          malmaisonrose                   #ce4676

	malmaison rose                          malmaisonrose                   #b3446c

	manchu                                  manchu                          #6f4e37

	mandalay                                mandalay                        #635147

	mandalay                                mandalay                        #3e322c

	mandarin orange                         mandarinorange                  #ed872d

	mandarin red                            mandarinred                     #d9603b

	manganese brown                         manganesebrown                  #6f4e37

	manganese velvet brown                  manganesevelvetbrown            #826644

	manganese velvet brown                  manganesevelvetbrown            #4b3621

	manganese violet                        manganeseviolet                 #875692

	mango                                   mango                           #ae6938

	mango                                   mango                           #a67b5b

	manila                                  manila                          #c19a6b

	manon                                   manon                           #958070

	manzanita                               manzanita                       #79443b

	maple                                   maple                           #c2b280

	maple sugar                             maplesugar                      #826644

	maracaibo                               maracaibo                       #422518

	marathon                                marathon                        #ed872d

	marble green                            marblegreen                     #1e2321

	marguerite yellow                       margueriteyellow                #dadfb7

	marie antoinette                        marieantoinette                 #ad8884

	marie antoinette                        marieantoinette                 #905d5d

	marine [blue]                           marineblue                      #00304e

	marine [blue]                           marineblue                      #4e5180

	marine [blue]                           marineblue                      #252440

	marine corps                            marinecorps                     #4c516d

	marine green                            marinegreen                     #1b4d3e

	marine green                            marinegreen                     #3a4b47

	maris                                   maris                           #3a4b47

	maris                                   maris                           #004b49

	marmora                                 marmora                         #317873

	marocain                                marocain                        #635147

	marone                                  marone                          #635147

	maroon                                  maroon                          #722f37

	maroon                                  maroon                          #56070c

	maroon                                  maroon                          #3e1d1e

	marron                                  marron                          #635147

	marrone                                 marrone                         #635147

	marron glac                            marronglac                      #6f4e37

	marsala                                 marsala                         #79443b

	mars brown                              marsbrown                       #6f4e37

	marsh rose                              marshrose                       #905d5d

	mars orange                             marsorange                      #cb6d51

	mars red                                marsred                         #9e4732

	mars violet                             marsviolet                      #56070c

	mars yellow                             marsyellow                      #ed872d

	mars yellow                             marsyellow                      #d99058

	martinique                              martinique                      #ae6938

	martinique                              martinique                      #a67b5b

	martius yellow                          martiusyellow                   #eae679

	marygold                                marygold                        #ed872d

	mascara                                 mascara                         #79443b

	mascot                                  mascot                          #36454f

	mascot                                  mascot                          #202830

	mascot                                  mascot                          #252440

	mascot                                  mascot                          #4c516d

	massicot [yellow]                       massicotyellow                  #f3e5ab

	mast colour                             mastcolour                      #ae6938

	mastic                                  mastic                          #967117

	matelot                                 matelot                         #00416a

	matrix                                  matrix                          #317873

	mauve blush                             mauveblush                      #c2ac99

	mauve blush                             mauveblush                      #ae9b82

	mauve blush                             mauveblush                      #bfb8a5

	mauve castor                            mauvecastor                     #5c504f

	mauve dust                              mauvedust                       #5c504f

	mauveglow                               mauveglow                       #ad8884

	mauverose                               mauverose                       #722f37

	mauve taupe                             mauvetaupe                      #5c504f

	mauvette                                mauvette                        #aa98a9

	mauvewood                               mauvewood                       #79443b

	mauvewood                               mauvewood                       #674c47

	mavis                                   mavis                           #ae9b82

	maya                                    maya                            #ae6938

	maya                                    maya                            #a67b5b

	mayfair tan                             mayfairtan                      #a67b5b

	mayfair tan                             mayfairtan                      #c19a6b

	mayfair tan                             mayfairtan                      #826644

	mayflower                               mayflower                       #ab4e52

	mazarine blue                           mazarineblue                    #272458

	meadowbrook                             meadowbrook                     #3a4b47

	meadowbrook                             meadowbrook                     #004b49

	meadowgrass                             meadowgrass                     #355e3b

	meadowgrass                             meadowgrass                     #3a4b47

	meadow [green]                          meadowgreen                     #355e3b

	meadowlark                              meadowlark                      #7e6d5a

	meadowpink                              meadowpink                      #a8516e

	meadowpink                              meadowpink                      #915f6d

	meadowsweet                             meadowsweet                     #905d5d

	meadow violet                           meadowviolet                    #875692

	meadow violet                           meadowviolet                    #9e4f88

	mecca                                   mecca                           #79443b

	mecca                                   mecca                           #6f4e37

	medal bronze                            medalbronze                     #826644

	medal bronze                            medalbronze                     #967117

	medici blue                             mediciblue                      #81878b

	mediterranian                           mediterranian                   #317873

	medium chrome green                     mediumchromegreen               #355e3b

	medium chrome green                     mediumchromegreen               #1b4d3e

	meerschaum                              meerschaum                      #ae9b82

	mehal                                   mehal                           #6f4e37

	melilot                                 melilot                         #915c83

	meline                                  meline                          #c9ae5d

	mello-mauve                             mellomauve                      #ad8884

	mellowglow                              mellowglow                      #e3a857

	melon                                   melon                           #f88379

	mephisto                                mephisto                        #ab4e52

	merida                                  merida                          #a67b5b

	merida                                  merida                          #c19a6b

	merida                                  merida                          #826644

	merle                                   merle                           #004958

	mermaid                                 mermaid                         #8f9779

	mesa                                    mesa                            #6f4e37

	metal                                   metal                           #b9b8b5

	metal brown                             metalbrown                      #79443b

	metal brown                             metalbrown                      #674c47

	metallic green                          metallicgreen                   #665d1e

	metallic grey                           metallicgrey                    #8b8589

	mexican                                 mexican                         #ae6938

	mexican                                 mexican                         #996515

	mexican red                             mexicanred                      #ae6938

	mexican red                             mexicanred                      #a67b5b

	mexico                                  mexico                          #79443b

	mexico                                  mexico                          #80461b

	miami sand                              miamisand                       #ae9b82

	miami sand                              miamisand                       #967117

	michigan                                michigan                        #004b49

	microcline green                        microclinegreen                 #c7e6d7

	middle brunswick green                  middlebrunswickgreen            #355e3b

	middle stone                            middlestone                     #a18f60

	middy                                   middy                           #00304e

	midnight                                midnight                        #50404d

	midnight sun                            midnightsun                     #d9603b

	mignon                                  mignon                          #86608e

	mignonette [green]                      mignonettegreen                 #867e36

	mignon green                            mignongreen                     #8a9a5b

	mikado                                  mikado                          #d9603b

	mikado                                  mikado                          #ed872d

	mikado brown                            mikadobrown                     #6f4e37

	mikado orange                           mikadoorange                    #d99058

	milano blue                             milanoblue                      #536878

	milk white                              milkwhite                       #f0ead6

	milwaukee brick                         milwaukeebrick                  #fad6a5

	milwaukee brick                         milwaukeebrick                  #f3e5ab

	mimosa                                  mimosa                          #f8de7e

	mindoro                                 mindoro                         #a67b5b

	mineral bister                          mineralbister                   #6f4e37

	mineral blue                            mineralblue                     #436b95

	mineral blue                            mineralblue                     #2e8495

	mineral blue                            mineralblue                     #436b95

	mineral green                           mineralgreen                    #679267

	mineral grey                            mineralgrey                     #b2beb5

	mineral orange                          mineralorange                   #e25822

	mineral pitch                           mineralpitch                    #483c32

	mineral purple                          mineralpurple                   #56070c

	mineral red                             mineralred                      #905d5d

	mineral rouge                           mineralrouge                    #882d17

	mineral rouge                           mineralrouge                    #80461b

	mineral violet                          mineralviolet                   #875692

	mineral yellow                          mineralyellow                   #fada5e

	mineral yellow                          mineralyellow                   #d4af37

	mineral yellow                          mineralyellow                   #f8de7e

	mineral yellow                          mineralyellow                   #c9ae5d

	mineral yellow                          mineralyellow                   #e3a857

	ming green                              minggreen                       #007959

	miniature pink                          miniaturepink                   #ecd5c5

	miniature pink                          miniaturepink                   #c7ada3

	minium                                  minium                          #e25822

	mint                                    mint                            #1b4d3e

	mint                                    mint                            #004b49

	minuet                                  minuet                          #8c82b6

	mirabelle                               mirabelle                       #d99058

	mirabelle                               mirabelle                       #e3a857

	mirador                                 mirador                         #422518

	mirage                                  mirage                          #c19a6b

	mist blue                               mistblue                        #93c592

	mist blue                               mistblue                        #6aab8e

	mist blue                               mistblue                        #8da399

	mist [grey]                             mistgrey                        #8f817f

	mistletoe                               mistletoe                       #8f9779

	misty blue                              mistyblue                       #93c592

	misty blue                              mistyblue                       #6aab8e

	misty blue                              mistyblue                       #8da399

	misty morn                              mistymorn                       #977f73

	misty morn                              mistymorn                       #958070

	mitis green                             mitisgreen                      #3eb489

	mittler's green                         mittlersgreen                   #004b49

	moccasin                                moccasin                        #6f4e37

	mocha                                   mocha                           #3b3121

	mocha bisque                            mochabisque                     #6f4e37

	mode beige                              modebeige                       #967117

	mohawk                                  mohawk                          #79443b

	mohawk                                  mohawk                          #6f4e37

	monaco                                  monaco                          #317873

	monet blue                              monetblue                       #272458

	monet blue                              monetblue                       #4e5180

	monicon                                 monicon                         #f88379

	monicon                                 monicon                         #cb6d51

	monicon                                 monicon                         #ed872d

	monicon                                 monicon                         #d99058

	monkey skin                             monkeyskin                      #a87c6d

	monkshood                               monkshood                       #554c69

	monsignor                               monsignor                       #875692

	monsignor                               monsignor                       #602f6b

	monsoreau                               monsoreau                       #ab4e52

	monte carlo                             montecarlo                      #007a74

	monte carlo                             montecarlo                      #317873

	montella                                montella                        #3e322c

	montella                                montella                        #483c32

	monterey                                monterey                        #9e4732

	monticello green                        monticellogreen                 #355e3b

	montpellier green                       montpelliergreen                #679267

	montpellier yellow                      montpellieryellow               #fada5e

	montpellier yellow                      montpellieryellow               #d4af37

	montpellier yellow                      montpellieryellow               #f8de7e

	montpellier yellow                      montpellieryellow               #c9ae5d

	moonbeam                                moonbeam                        #b9b8b5

	moonlight                               moonlight                       #fad6a5

	moonlight                               moonlight                       #bfb8a5

	moonlight blue                          moonlightblue                   #536878

	moonmist                                moonmist                        #bfb8a5

	moorish red                             moorishred                      #ed872d

	moose                                   moose                           #363527

	mordor                                 mordor                          #79443b

	mordor                                 mordor                          #6f4e37

	morea berries                           moreaberries                    #eaa221

	morea berries                           moreaberries                    #e3a857

	morello                                 morello                         #242124

	moresco                                 moresco                         #ed872d

	moresco                                 moresco                         #d99058

	morillon                                morillon                        #8a9a5b

	morning blue                            morningblue                     #8da399

	morning blue                            morningblue                     #b2beb5

	morning dawning yellow                  morningdawningyellow            #f88379

	morning dawning yellow                  morningdawningyellow            #e66761

	morning dawning yellow                  morningdawningyellow            #cb6d51

	morning glory                           morningglory                    #aa8a9e

	moroccan                                moroccan                        #722f37

	moroccan                                moroccan                        #9e4732

	morocco                                 morocco                         #3e322c

	morocco red                             moroccored                      #722f37

	morocco red                             moroccored                      #882d17

	morocco red                             moroccored                      #79443b

	morocco sand                            moroccosand                     #c19a6b

	moro red                                morored                         #79443b

	morro                                   morro                           #43302e

	morro                                   morro                           #3e322c

	mort d'ore                              mortdore                        #79443b

	mort d'ore                              mortdore                        #6f4e37

	mosaic blue                             mosaicblue                      #436b95

	mosque                                  mosque                          #80461b

	moss [green]                            mossgreen                       #867e36

	moss [green]                            mossgreen                       #8a9a5b

	moss grey                               mossgrey                        #8c8767

	moss pink                               mosspink                        #915f6d

	moss rose                               mossrose                        #e4717a

	moss rose                               mossrose                        #c08081

	mosul                                   mosul                           #a67b5b

	mosul                                   mosul                           #6f4e37

	mosul                                   mosul                           #826644

	moth [grey]                             mothgrey                        #c2b280

	motmot blue                             motmotblue                      #007791

	motmot green                            motmotgreen                     #679267

	motmot green                            motmotgreen                     #6aab8e

	mountain blue                           mountainblue                    #436b95

	mountain green                          mountaingreen                   #3a4b47

	mountain green                          mountaingreen                   #679267

	mountain yellow                         mountainyellow                  #e3a857

	mount vernon green or mt. vernon green  mountvernongreenormtvernongreen #4a5d23

	mouse-dun                               mousedun                        #5b504f

	mouse [grey]                            mousegrey                       #5b504f

	mousse                                  mousse                          #867e36

	mousse                                  mousse                          #8a9a5b

	msange                                 msange                          #8da399

	mtallique                              mtallique                       #57554c

	muffin                                  muffin                          #a67b5b

	muffin                                  muffin                          #996515

	mulberry                                mulberry                        #242124

	mulberry fruit                          mulberryfruit                   #3f1728

	mulberry fruit                          mulberryfruit                   #2e1d21

	mulberry purple                         mulberrypurple                  #563c5c

	mummy                                   mummy                           #483c32

	mummy                                   mummy                           #826644

	mummy brown                             mummybrown                      #6f4e37

	mummy brown                             mummybrown                      #635147

	mummy brown                             mummybrown                      #7e6d5a

	muraille                                muraille                        #6f4e37

	murillo                                 murillo                         #436b95

	murinus                                 murinus                         #5b504f

	murrey                                  murrey                          #242124

	muscade                                 muscade                         #a67b5b

	muscovite                               muscovite                       #4e5755

	mushroom                                mushroom                        #5b504f

	musk                                    musk                            #483c32

	musketeer                               musketeer                       #00304e

	musketeer                               musketeer                       #252440

	musketeer                               musketeer                       #4c516d

	muskmelon                               muskmelon                       #b4745e

	muskmelon                               muskmelon                       #d99058

	muskrat                                 muskrat                         #3e322c

	mustang                                 mustang                         #7e6d5a

	mustard brown                           mustardbrown                    #6f4e37

	mustard [yellow]                        mustardyellow                   #c9ae5d

	mutrie yellow                           mutrieyellow                    #dcd300

	mutrie yellow                           mutrieyellow                    #e9e450

	myosotis blue                           myosotisblue                    #91a3b0

	myosotis blue                           myosotisblue                    #536878

	myrtle [green]                          myrtlegreen                     #3a4b47

	myrtle [green]                          myrtlegreen                     #004b49

	mytho green                             mythogreen                      #8a9a5b

	mytho green                             mythogreen                      #8f9779

	nacarat                                 nacarat                         #be0032

	nacarine                                nacarine                        #be0032

	naiad                                   naiad                           #6aab8e

	nankeen [yellow]                        nankeenyellow                   #f8de7e

	nankeen [yellow]                        nankeenyellow                   #c9ae5d

	nankeen [yellow]                        nankeenyellow                   #f3e5ab

	nankeen [yellow]                        nankeenyellow                   #c2b280

	naples red                              naplesred                       #79443b

	naples red                              naplesred                       #882d17

	naples yellow                           naplesyellow                    #f8de7e

	naples yellow                           naplesyellow                    #c9ae5d

	naples yellow                           naplesyellow                    #f3e5ab

	naples yellow                           naplesyellow                    #c2b280

	napoleon blue                           napoleonblue                    #00416a

	napoli                                  napoli                          #004958

	narcissus                               narcissus                       #d4af37

	narcissus                               narcissus                       #c9ae5d

	narrawood                               narrawood                       #ab4e52

	narva                                   narva                           #3a4b47

	nasturtium [red]                        nasturtiumred                   #9e4732

	nasturtium [yellow]                     nasturtiumyellow                #ed872d

	nasturtium [yellow]                     nasturtiumyellow                #d99058

	nasturtium [yellow]                     nasturtiumyellow                #eaa221

	nasturtium [yellow]                     nasturtiumyellow                #e3a857

	natal brown                             natalbrown                      #635147

	national [blue]                         nationalblue                    #00416a

	national [blue]                         nationalblue                    #436b95

	national grey                           nationalgrey                    #848482

	nattier                                 nattier                         #8791bf

	natural                                 natural                         #fad6a5

	natural                                 natural                         #bfb8a5

	navaho                                  navaho                          #f38400

	navaho                                  navaho                          #ed872d

	navy 1                                  navy1                           #36454f

	navy 2                                  navy2                           #291e29

	navy 3                                  navy3                           #36454f

	navy 3                                  navy3                           #202830

	navy blue                               navyblue                        #252440

	navy blue                               navyblue                        #4c516d

	neapolitan blue                         neapolitanblue                  #4c516d

	neapolitan yellow                       neapolitanyellow                #f8de7e

	neapolitan yellow                       neapolitanyellow                #c9ae5d

	neapolitan yellow                       neapolitanyellow                #f3e5ab

	neapolitan yellow                       neapolitanyellow                #c2b280

	nectar                                  nectar                          #905d5d

	nectarine                               nectarine                       #f4c2c2

	nectarine                               nectarine                       #d9a6a9

	negro                                   negro                           #483c32

	negro                                   negro                           #222222

	neptune [green]                         neptunegreen                    #6aab8e

	neptune [green]                         neptunegreen                    #3b7861

	neutral orange                          neutralorange                   #be6516

	neutral red                             neutralred                      #722f37

	neutral red                             neutralred                      #673147

	neutral tint                            neutraltint                     #222222

	neuvider green                          neuvidergreen                   #93c592

	neuvider green                          neuvidergreen                   #6aab8e

	neuwied blue                            neuwiedblue                     #317873

	neuwieder blue                          neuwiederblue                   #317873

	neuwieder green                         neuwiedergreen                  #93c592

	neuwieder green                         neuwiedergreen                  #6aab8e

	neva green                              nevagreen                       #7e9f2e

	neva green                              nevagreen                       #83d37d

	new blue                                newblue                         #0067a5

	new blue                                newblue                         #545aa7

	new bronze                              newbronze                       #6f4e37

	new bronze                              newbronze                       #635147

	new bronze                              newbronze                       #826644

	new bronze                              newbronze                       #7e6d5a

	new cocoa                               newcocoa                        #635147

	new hay                                 newhay                          #515744

	newport                                 newport                         #004958

	new silver                              newsilver                       #f3e5ab

	new silver                              newsilver                       #c2b280

	new silver                              newsilver                       #bfb8a5

	new silver                              newsilver                       #dadfb7

	new silver                              newsilver                       #848482

	niagara                                 niagara                         #66aabc

	niagara green                           niagaragreen                    #66ada4

	nicaraguawood                           nicaraguawood                   #ab4e52

	nicaraguawood                           nicaraguawood                   #905d5d

	nice                                    nice                            #536878

	nightshade                              nightshade                      #2f2140

	nightshade                              nightshade                      #50404d

	nikko                                   nikko                           #436b95

	nikko                                   nikko                           #536878

	nile blue                               nileblue                        #66ada4

	nile [green]                            nilegreen                       #8a9a5b

	nile [green]                            nilegreen                       #8f9779

	nile [green]                            nilegreen                       #93c592

	nimbus                                  nimbus                          #848482

	nippon                                  nippon                          #00304e

	noisette                                noisette                        #a67b5b

	noisette                                noisette                        #996515

	nomad brown                             nomadbrown                      #422518

	norfolk                                 norfolk                         #3e322c

	normandy                                normandy                        #00304e

	normandy                                normandy                        #252440

	normandy                                normandy                        #4c516d

	normandy blue                           normandyblue                    #8da399

	nougat                                  nougat                          #958070

	nougat                                  nougat                          #ae9b82

	nubian                                  nubian                          #222222

	nude                                    nude                            #c2ac99

	nude                                    nude                            #fad6a5

	nude                                    nude                            #c19a6b

	nude                                    nude                            #ae9b82

	nugget                                  nugget                          #d99058

	nugget                                  nugget                          #be8a3d

	nuncio                                  nuncio                          #86608e

	nuremberg red                           nurembergred                    #79443b

	nuremberg violet                        nurembergviolet                 #875692

	nutmeg                                  nutmeg                          #6f4e37

	nutmeg                                  nutmeg                          #635147

	nutria                                  nutria                          #57554c

	nymphea                                 nymphea                         #86608e

	nymph green                             nymphgreen                      #ebe8a4

	nymph green                             nymphgreen                      #c9dc89

	nymph [pink]                            nymphpink                       #ea9399

	nymph [pink]                            nymphpink                       #dea5a4

	oad                                     oad                             #536878

	oak [brown]                             oakbrown                        #80461b

	oak [brown]                             oakbrown                        #6f4e37

	oakbuff                                 oakbuff                         #a67b5b

	oakbuff                                 oakbuff                         #c19a6b

	oakbuff                                 oakbuff                         #826644

	oak [green]                             oakgreen                        #867e36

	oakheart                                oakheart                        #905d5d

	oakheart                                oakheart                        #79443b

	oakleaf brown                           oakleafbrown                    #be6516

	oakleaf brown                           oakleafbrown                    #ae6938

	oakleaf brown                           oakleafbrown                    #80461b

	oakwood                                 oakwood                         #6f4e37

	oasis                                   oasis                           #867e36

	oasis                                   oasis                           #8a9a5b

	ocean green                             oceangreen                      #93c592

	ocher red                               ocherred                        #905d5d

	ochraceous                              ochraceous                      #e3a857

	ochre                                   ochre                           #e3a857

	ochre de ru                             ochrederu                       #ae6938

	ochre de ru                             ochrederu                       #996515

	ochre red                               ochrered                        #9e4732

	oil blue                                oilblue                         #30267a

	oil green                               oilgreen                        #679267

	oil yellow                              oilyellow                       #b9b459

	oil yellow                              oilyellow                       #98943e

	oker de luce                            okerdeluce                      #ae6938

	oker de luce                            okerdeluce                      #996515

	oker de luke                            okerdeluke                      #ae6938

	oker de luke                            okerdeluke                      #996515

	oker de rouse                           okerderouse                     #ae6938

	oker de rouse                           okerderouse                     #996515

	old amethyst                            oldamethyst                     #722f37

	old amethyst                            oldamethyst                     #673147

	old blue                                oldblue                         #91a3b0

	old blue                                oldblue                         #536878

	old blue                                oldblue                         #81878b

	old bronze                              oldbronze                       #967117

	old bronze                              oldbronze                       #867e36

	old burgundy                            oldburgundy                     #3e1d1e

	old burgundy                            oldburgundy                     #43302e

	old cedar                               oldcedar                        #79443b

	old china                               oldchina                        #536878

	old china                               oldchina                        #4c516d

	old coral                               oldcoral                        #ab4e52

	olde                                    olde                            #eae679

	olde                                    olde                            #b9b459

	old english brown                       oldenglishbrown                 #483c32

	old english brown                       oldenglishbrown                 #6c541e

	old english brown                       oldenglishbrown                 #3b3121

	old gold                                oldgold                         #867e36

	old helio                               oldhelio                        #563c5c

	old ivory                               oldivory                        #c2b280

	old ivory                               oldivory                        #a18f60

	old ivory                               oldivory                        #967117

	old lavender                            oldlavender                     #242124

	old lilac                               oldlilac                        #796878

	old lilac                               oldlilac                        #836479

	old mauve                               oldmauve                        #50404d

	old mauve                               oldmauve                        #673147

	old moss [green]                        oldmossgreen                    #867e36

	old olive                               oldolive                        #6c541e

	old pink                                oldpink                         #a67b5b

	old red                                 oldred                          #841b2d

	old red                                 oldred                          #ab4e52

	old rose                                oldrose                         #905d5d

	old roseleaf                            oldroseleaf                     #722f37

	old silver                              oldsilver                       #848482

	old wood                                oldwood                         #905d5d

	olivaceous                              olivaceous                      #665d1e

	olive brown                             olivebrown                      #6c541e

	olive drab                              olivedrab                       #5b5842

	olive [green]                           olivegreen                      #665d1e

	olive grey                              olivegrey                       #8f9779

	olive grey                              olivegrey                       #7d8984

	olivesheen                              olivesheen                      #a18f60

	olive terra verte                       oliveterraverte                 #867e36

	olive wood                              olivewood                       #7e6d5a

	olive yellow                            oliveyellow                     #98943e

	olive yellow                            oliveyellow                     #867e36

	olivine                                 olivine                         #93c592

	olympia                                 olympia                         #007a74

	olympia                                 olympia                         #317873

	olympian blue                           olympianblue                    #00416a

	olympian green                          olympiangreen                   #679267

	olympic                                 olympic                         #36454f

	olympic blue                            olympicblue                     #007791

	ondine                                  ondine                          #8da399

	onion-peel                              onionpeel                       #905d5d

	onion red                               onionred                        #905d5d

	onion-skin pink                         onionskinpink                   #d9a6a9

	onion-skin pink                         onionskinpink                   #d99058

	onion-skin pink                         onionskinpink                   #a67b5b

	ontario violet                          ontarioviolet                   #8c92ac

	oold                                    oold                            #eae679

	oold                                    oold                            #b9b459

	opal                                    opal                            #317873

	opal blue                               opalblue                        #536878

	opal grey                               opalgrey                        #8f817f

	opal grey                               opalgrey                        #848482

	opaline green                           opalinegreen                    #8ed1b2

	opal mauve                              opalmauve                       #c1b6b3

	opaq                                    opaq                            #679267

	opera mauve                             operamauve                      #b784a7

	opera pink                              operapink                       #f4c2c2

	opera pink                              operapink                       #d9a6a9

	opera pink                              operapink                       #ecd5c5

	opera pink                              operapink                       #c7ada3

	ophelia                                 ophelia                         #915c83

	ophelia                                 ophelia                         #915f6d

	ophelia                                 ophelia                         #836479

	oporto                                  oporto                          #722f37

	oporto                                  oporto                          #3f1728

	orange aurora                           orangeaurora                    #f88379

	orange aurora                           orangeaurora                    #e66761

	orange aurora                           orangeaurora                    #cb6d51

	orange lead                             orangelead                      #e25822

	orange madder                           orangemadder                    #ed872d

	orange ochre                            orangeochre                     #ed872d

	orange-peel                             orangepeel                      #ed872d

	orange rufous                           orangerufous                    #ed872d

	orange tawny                            orangetawny                     #cb6d51

	orange vermilion                        orangevermilion                 #d9603b

	orchadee                                orchadee                        #b784a7

	orchall                                 orchall                         #a8516e

	orchid                                  orchid                          #b695c0

	orchid                                  orchid                          #aa98a9

	orchid pink                             orchidpink                      #ea9399

	orchil                                  orchil                          #ab4e52

	orchil                                  orchil                          #a8516e

	orchis                                  orchis                          #905d5d

	orient                                  orient                          #dea5a4

	oriental                                oriental                        #536878

	oriental blue                           orientalblue                    #4c516d

	oriental blue                           orientalblue                    #0067a5

	oriental blue                           orientalblue                    #545aa7

	oriental bole                           orientalbole                    #79443b

	oriental fuchsia                        orientalfuchsia                 #702963

	oriental green                          orientalgreen                   #44944a

	oriental green                          orientalgreen                   #679267

	oriental pearl                          orientalpearl                   #8a8776

	oriental pearl                          orientalpearl                   #848482

	oriental red                            orientalred                     #bc3f4a

	orient [blue]                           orientblue                      #0067a5

	orient blue                             orientblue                      #436b95

	orient blue                             orientblue                      #536878

	orient pink                             orientpink                      #d9a6a9

	orient red                              orientred                       #bc3f4a

	orient yellow                           orientyellow                    #ed872d

	orient yellow                           orientyellow                    #d99058

	orient yellow                           orientyellow                    #eaa221

	orient yellow                           orientyellow                    #e3a857

	orient yellow                           orientyellow                    #ab9144

	orient yellow                           orientyellow                    #a18f60

	oriole                                  oriole                          #ae6938

	orion                                   orion                           #00304e

	orlean                                  orlean                          #d9a6a9

	ormond                                  ormond                          #977f73

	ormond                                  ormond                          #958070

	ormond                                  ormond                          #8e8279

	orpiment                                orpiment                        #fada5e

	orpiment                                orpiment                        #d4af37

	orpiment                                orpiment                        #f8de7e

	orpiment                                orpiment                        #c9ae5d

	orpiment orange                         orpimentorange                  #f38400

	orpiment orange                         orpimentorange                  #ed872d

	orpiment red                            orpimentred                     #eaa221

	orpiment red                            orpimentred                     #e3a857

	orpin                                   orpin                           #fada5e

	orpin                                   orpin                           #d4af37

	orpin                                   orpin                           #f8de7e

	orpin                                   orpin                           #c9ae5d

	orseille                                orseille                        #a8516e

	otter [brown]                           otterbrown                      #483c32

	oural green                             ouralgreen                      #93c592

	oural green                             ouralgreen                      #c7e6d7

	owl                                     owl                             #5b504f

	oxblood [red]                           oxbloodred                      #79443b

	oxford blue                             oxfordblue                      #291e29

	oxford chrome                           oxfordchrome                    #e3a857

	oxford ochre                            oxfordochre                     #e3a857

	oxford yellow                           oxfordyellow                    #e3a857

	oxgall                                  oxgall                          #d4af37

	oxheart                                 oxheart                         #841b2d

	oxide blue                              oxideblue                       #0085a1

	oxide brown                             oxidebrown                      #79443b

	oxide purple                            oxidepurple                     #79443b

	oxide yellow                            oxideyellow                     #e3a857

	oyster grey                             oystergrey                      #dadfb7

	oyster [white]                          oysterwhite                     #dadfb7

	pablo                                   pablo                           #a67b5b

	pablo                                   pablo                           #c19a6b

	pacific                                 pacific                         #8da399

	pacific                                 pacific                         #66ada4

	paddock                                 paddock                         #004b49

	pagoda [blue]                           pagodablue                      #004958

	pagoda [blue]                           pagodablue                      #36454f

	palestine                               palestine                       #796878

	palestine                               palestine                       #915f6d

	palm                                    palm                            #8a9a5b

	palmetto                                palmetto                        #8f9779

	palm green                              palmgreen                       #355e3b

	palmleaf                                palmleaf                        #867e36

	paloma                                  paloma                          #ae6938

	paloma                                  paloma                          #a67b5b

	pampas                                  pampas                          #5b504f

	pan                                     pan                             #3a4b47

	pancy green                             pancygreen                      #355e3b

	panse                                  panse                           #604e97

	pansy                                   pansy                           #604e97

	pansy-maroon                            pansymaroon                     #673147

	pansy purple                            pansypurple                     #673147

	pansy violet                            pansyviolet                     #5d3954

	paon                                    paon                            #004958

	paper white                             paperwhite                      #bfb8a5

	paprica                                 paprica                         #e25822

	papyrus                                 papyrus                         #a67b5b

	paradise green                          paradisegreen                   #355e3b

	para red                                parared                         #bc3f4a

	para red                                parared                         #ab4e52

	parchment                               parchment                       #c2b280

	parchment                               parchment                       #a18f60

	parchment                               parchment                       #967117

	paris green                             parisgreen                      #679267

	paris green                             parisgreen                      #3eb489

	paris mud                               parismud                        #543d3f

	paris red                               parisred                        #e25822

	paris yellow                            parisyellow                     #d4af37

	park green                              parkgreen                       #8a9a5b

	parma red                               parmared                        #bc3f4a

	parma red                               parmared                        #ab4e52

	parme                                   parme                           #604e81

	parrakeet                               parrakeet                       #8a9a5b

	parrakeet                               parrakeet                       #3eb489

	parroquit green                         parroquitgreen                  #8a9a5b

	parrot green                            parrotgreen                     #8a9a5b

	partridge                               partridge                       #826644

	partridge                               partridge                       #4b3621

	parula blue                             parulablue                      #536878

	pastel                                  pastel                          #536878

	pastel blue                             pastelblue                      #536878

	pastel grey                             pastelgrey                      #c2b280

	pastel parchment                        pastelparchment                 #d9a6a9

	pastel parchment                        pastelparchment                 #c7ada3

	pastel parchment                        pastelparchment                 #c2ac99

	patent yellow                           patentyellow                    #fada5e

	patent yellow                           patentyellow                    #d4af37

	patent yellow                           patentyellow                    #f8de7e

	patent yellow                           patentyellow                    #c9ae5d

	patina green                            patinagreen                     #93c592

	patina green                            patinagreen                     #679267

	patriarch                               patriarch                       #602f6b

	patricia                                patricia                        #9e4f88

	paul veronese green                     paulveronesegreen               #3eb489

	pauncy green                            pauncygreen                     #355e3b

	pavonine                                pavonine                        #004958

	pawnee                                  pawnee                          #a67b5b

	pawnee                                  pawnee                          #958070

	pawnee                                  pawnee                          #c19a6b

	pawnee                                  pawnee                          #ae9b82

	payne's grey                            paynesgrey                      #536878

	payne's grey                            paynesgrey                      #36454f

	peach                                   peach                           #f4c2c2

	peach                                   peach                           #d9a6a9

	peachbeige                              peachbeige                      #d9a6a9

	peach bisque                            peachbisque                     #a87c6d

	peach bloom                             peachbloom                      #d9a6a9

	peach blossom [pink]                    peachblossompink                #ea9399

	peach blossom [pink]                    peachblossompink                #dea5a4

	peach blossom [red]                     peachblossomred                 #ea9399

	peach blossom [red]                     peachblossomred                 #dea5a4

	peach blow                              peachblow                       #fad6a5

	peach blush                             peachblush                      #80461b

	peach red                               peachred                        #f88379

	peachwood                               peachwood                       #ab4e52

	peachwood                               peachwood                       #905d5d

	peacock [blue]                          peacockblue                     #004958

	peacock green                           peacockgreen                    #679267

	pea green                               peagreen                        #8a9a5b

	peanut                                  peanut                          #c2b280

	pearl                                   pearl                           #b9b8b5

	pearl blue                              pearlblue                       #b9b8b5

	pearlblush                              pearlblush                      #c2ac99

	pearlblush                              pearlblush                      #958070

	pearl gray                              pearlgray                       #bfb8a5

	pearl white                             pearlwhite                      #b9b8b5

	peasant                                 peasant                         #252440

	peasant                                 peasant                         #4c516d

	peasant blue                            peasantblue                     #536878

	peasant blue                            peasantblue                     #00304e

	peasant blue                            peasantblue                     #252440

	peasant blue                            peasantblue                     #4c516d

	pebble                                  pebble                          #c2b280

	pecan                                   pecan                           #ae9b82

	pecan brown                             pecanbrown                      #6f4e37

	pekinese                                pekinese                        #ae6938

	pekinese                                pekinese                        #a67b5b

	peking blue                             pekingblue                      #00304e

	pelican                                 pelican                         #555555

	peligot's blue                          peligotsblue                    #317873

	pelt                                    pelt                            #5b504f

	pencilwood                              pencilwood                      #79443b

	pencilwood                              pencilwood                      #6f4e37

	penguin                                 penguin                         #848482

	peony                                   peony                           #722f37

	peony red                               peonyred                        #841b2d

	peony red                               peonyred                        #722f37

	pepita                                  pepita                          #967117

	peppermint                              peppermint                      #44944a

	pepper red                              pepperred                       #ab4e52

	peridot                                 peridot                         #8a9a5b

	peridot                                 peridot                         #8f9779

	peridot                                 peridot                         #4a5d23

	peridot                                 peridot                         #515744

	perilla                                 perilla                         #9065ca

	perilla                                 perilla                         #604e97

	perilla purple                          perillapurple                   #673147

	perique                                 perique                         #483c32

	periwinkle                              periwinkle                      #8c92ac

	periwinkle                              periwinkle                      #4c516d

	permanent blue                          permanentblue                   #0067a5

	permanent blue                          permanentblue                   #545aa7

	permanent red                           permanentred                    #bc3f4a

	permanent red                           permanentred                    #ab4e52

	permanent violet                        permanentviolet                 #875692

	permanent yellow                        permanentyellow                 #e3a857

	perma red                               permared                        #bc3f4a

	perma red                               permared                        #ab4e52

	pernambucowood                          pernambucowood                  #ab4e52

	pernambucowood                          pernambucowood                  #905d5d

	perruche                                perruche                        #8a9a5b

	persenche                               persenche                       #9690ab

	persian blue                            persianblue                     #91a3b0

	persian blue                            persianblue                     #4c516d

	persian earth                           persianearth                    #882d17

	persian green                           persiangreen                    #3a4b47

	persian lilac                           persianlilac                    #c17e91

	persian melon                           persianmelon                    #b4745e

	persian orange                          persianorange                   #d99058

	persian pink                            persianpink                     #a8516e

	persian red                             persianred                      #882d17

	persian rose                            persianrose                     #ce4676

	persian yellow                          persianyellow                   #e25822

	persian yellow                          persianyellow                   #f38400

	persimmon                               persimmon                       #80461b

	persio                                  persio                          #a8516e

	persis                                  persis                          #a8516e

	peruvian brown                          peruvianbrown                   #ae6938

	peruvian brown                          peruvianbrown                   #a67b5b

	peruvian yellow                         peruvianyellow                  #cb6d51

	pervenche                               pervenche                       #4c516d

	petunia                                 petunia                         #563c5c

	petunia [violet]                        petuniaviolet                   #602f6b

	pewke                                   pewke                           #722f37

	pewter                                  pewter                          #555555

	phantom                                 phantom                         #555555

	phantom red                             phantomred                      #e25822

	pharaoh                                 pharaoh                         #4e5180

	pharaoh                                 pharaoh                         #252440

	pheasant                                pheasant                        #d99058

	pheasant                                pheasant                        #a67b5b

	philamot                                philamot                        #be6516

	philamot                                philamot                        #ae6938

	philamot                                philamot                        #80461b

	philimot                                philimot                        #be6516

	philimot                                philimot                        #ae6938

	philimot                                philimot                        #80461b

	philomot                                philomot                        #be6516

	philomot                                philomot                        #ae6938

	philomot                                philomot                        #80461b

	phlox                                   phlox                           #702963

	phlox                                   phlox                           #673147

	phlox pink                              phloxpink                       #aa98a9

	phlox purple                            phloxpurple                     #9e4f88

	phyliamort                              phyliamort                      #be6516

	phyliamort                              phyliamort                      #ae6938

	phyliamort                              phyliamort                      #80461b

	phyllamort                              phyllamort                      #be6516

	phyllamort                              phyllamort                      #ae6938

	phyllamort                              phyllamort                      #80461b

	piccadilly                              piccadilly                      #79443b

	piccadilly                              piccadilly                      #674c47

	piccaninny                              piccaninny                      #483c32

	piccolpasso red                         piccolpassored                  #79443b

	pigeon                                  pigeon                          #5d555b

	pigeon blood                            pigeonblood                     #722f37

	pigeon neck                             pigeonneck                      #555555

	pigeon's-breast                         pigeonsbreast                   #905d5d

	pigeon's throat                         pigeonsthroat                   #3b7861

	pigskin                                 pigskin                         #a67b5b

	pigskin                                 pigskin                         #6f4e37

	pigskin                                 pigskin                         #826644

	pilgrim                                 pilgrim                         #555555

	pilgrim brown                           pilgrimbrown                    #635147

	pilgrim brown                           pilgrimbrown                    #3e322c

	pilot blue                              pilotblue                       #436b95

	pimento                                 pimento                         #be0032

	pimlico                                 pimlico                         #554c69

	pinard yellow                           pinardyellow                    #f8de7e

	pinchbeck brown                         pinchbeckbrown                  #be8a3d

	pinchbeck brown                         pinchbeckbrown                  #996515

	pinchbeck brown                         pinchbeckbrown                  #c19a6b

	pinchbeck brown                         pinchbeckbrown                  #ab9144

	pinchbeck brown                         pinchbeckbrown                  #967117

	pineapple                               pineapple                       #f8de7e

	pineapple                               pineapple                       #c9ae5d

	pinecone                                pinecone                        #635147

	pinecone                                pinecone                        #7e6d5a

	pine frost                              pinefrost                       #8da399

	pinegrove                               pinegrove                       #3a4b47

	pineneedle                              pineneedle                      #1b4d3e

	pineneedle                              pineneedle                      #3a4b47

	pine tree                               pinetree                        #004b49

	pink 1                                  pink1                           #c1b6b3

	pink 2                                  pink2                           #ead8d7

	pink 3                                  pink3                           #f9ccca

	pink 3                                  pink3                           #ead8d7

	pink 4                                  pink4                           #f9ccca

	pink 4                                  pink4                           #dea5a4

	pink 5                                  pink5                           #dea5a4

	pink coral                              pinkcoral                       #e4717a

	pink coral                              pinkcoral                       #f88379

	pink pearl                              pinkpearl                       #dea5a4

	piping rock                             pipingrock                      #8a8776

	piquant green                           piquantgreen                    #8a9a5b

	pirate                                  pirate                          #bc3f4a

	pirate                                  pirate                          #d9603b

	pistache                                pistache                        #8a9a5b

	pistache                                pistache                        #93c592

	pistachio green                         pistachiogreen                  #8a9a5b

	pistachio green                         pistachiogreen                  #93c592

	pitchpine                               pitchpine                       #3a4b47

	piuree                                  piuree                          #eaa221

	piuree                                  piuree                          #e3a857

	piuri                                   piuri                           #eaa221

	piuri                                   piuri                           #e3a857

	pi yu                                   piyu                            #3b7861

	plantation                              plantation                      #79443b

	platina yellow                          platinayellow                   #ae6938

	platinum                                platinum                        #848482

	plaza grey                              plazagrey                       #bfb8a5

	pleroma violet                          pleromaviolet                   #86608e

	plover                                  plover                          #5b504f

	plumbaceous                             plumbaceous                     #5d555b

	plumbaceous                             plumbaceous                     #555555

	plumbago blue                           plumbagoblue                    #aa98a9

	plumbago blue                           plumbagoblue                    #8b8589

	plumbago grey                           plumbagogrey                    #aa98a9

	plumbago grey                           plumbagogrey                    #8b8589

	plumbago slate                          plumbagoslate                   #5d555b

	plumbet                                 plumbet                         #b4bcc0

	plum [purple]                           plumpurple                      #563c5c

	plum purple                             plumpurple                      #2f2140

	plum violet                             plumviolet                      #722f37

	plum violet                             plumviolet                      #673147

	plunket                                 plunket                         #b4bcc0

	plymouth                                plymouth                        #555555

	poil d'ours                             poildours                       #555555

	poilu                                   poilu                           #81878b

	poinsettia                              poinsettia                      #be0032

	poinsettia                              poinsettia                      #bc3f4a

	pois green                              poisgreen                       #8f9779

	polar bear                              polarbear                       #fad6a5

	polar bear                              polarbear                       #f0ead6

	polignac                                polignac                        #ad8884

	polignac                                polignac                        #a87c6d

	polignac                                polignac                        #977f73

	polo green                              pologreen                       #4a5d23

	polo green                              pologreen                       #355e3b

	polo tan                                polotan                         #c19a6b

	pomegranate                             pomegranate                     #722f37

	pomegranate blossom                     pomegranateblossom              #be0032

	pomegranate blossom                     pomegranateblossom              #e25822

	pomegranate purple                      pomegranatepurple               #ab4e52

	pomegranate purple                      pomegranatepurple               #a8516e

	pomegranate purple                      pomegranatepurple               #a8516e

	pomegranate purple                      pomegranatepurple               #915f6d

	pomona green                            pomonagreen                     #8a9a5b

	pompadour [green]                       pompadourgreen                  #436b95

	pomp and power                          pompandpower                    #86608e

	pompeian blue                           pompeianblue                    #91a3b0

	pompeian red                            pompeianred                     #905d5d

	pompeian red                            pompeianred                     #b4745e

	pompeian red                            pompeianred                     #a87c6d

	pompeian yellow                         pompeianyellow                  #e3a857

	pompeii                                 pompeii                         #56070c

	pompeii                                 pompeii                         #79443b

	ponceau                                 ponceau                         #be0032

	ponceau                                 ponceau                         #e25822

	ponce de leon                           poncedeleon                     #ed872d

	pond lily                               pondlily                        #f8de7e

	pond lily                               pondlily                        #c9ae5d

	pongee                                  pongee                          #c19a6b

	pontiff [purple]                        pontiffpurple                   #602f6b

	pony brown                              ponybrown                       #635147

	popcorn                                 popcorn                         #f8de7e

	popinjay green                          popinjaygreen                   #8a9a5b

	poplar                                  poplar                          #3a4b47

	poppy [red]                             poppyred                        #be0032

	poppy [red]                             poppyred                        #e25822

	porcelain                               porcelain                       #436b95

	porcelain blue                          porcelainblue                   #536878

	porcelain green                         porcelaingreen                  #317873

	porraaceous                             porraaceous                     #8a9a5b

	porret                                  porret                          #8a9a5b

	port                                    port                            #3f1728

	portable red                            portablered                     #bc3f4a

	portable red                            portablered                     #ab4e52

	portable red                            portablered                     #aa381e

	portable red                            portablered                     #9e4732

	portugese red or portuguese red         portugeseredorportuguesered     #be0032

	port wine                               portwine                        #43302e

	post office red                         postofficered                   #aa381e

	post office red                         postofficered                   #9e4732

	posy green                              posygreen                       #5e716a

	poudre                                  poudre                          #91a3b0

	poudre                                  poudre                          #81878b

	poudre                                  poudre                          #b4bcc0

	poudre blue                             poudreblue                      #91a3b0

	poudre blue                             poudreblue                      #81878b

	powder blue                             powderblue                      #91a3b0

	powder blue                             powderblue                      #81878b

	powder blue                             powderblue                      #00416a

	powder blue                             powderblue                      #436b95

	powdered gold                           powderedgold                    #a18f60

	powder pink                             powderpink                      #d9a6a9

	prairie                                 prairie                         #c19a6b

	prairie brown                           prairiebrown                    #ae6938

	praline                                 praline                         #6f4e37

	praline                                 praline                         #635147

	prawn [pink]                            prawnpink                       #f88379

	prelate                                 prelate                         #604e81

	primrose green                          primrosegreen                   #ebe8a4

	primrose yellow                         primroseyellow                  #f8de7e

	primrose yellow                         primroseyellow                  #c9ae5d

	primuline yellow                        primulineyellow                 #d4af37

	prince grey                             princegrey                      #8b8589

	princeton orange                        princetonorange                 #ed872d

	priscilla                               priscilla                       #b9b8b5

	privet                                  privet                          #515744

	prout's brown                           proutsbrown                     #6f4e37

	prout's brown                           proutsbrown                     #635147

	prout's brown                           proutsbrown                     #826644

	prout's brown                           proutsbrown                     #7e6d5a

	prune                                   prune                           #5d3954

	prunella                                prunella                        #32174d

	prunella                                prunella                        #2f2140

	prune purple                            prunepurple                     #563c5c

	prussian brown                          prussianbrown                   #ae6938

	prussian red                            prussianred                     #882d17

	prussian red                            prussianred                     #79443b

	psyche                                  psyche                          #93c592

	pt shell                              ptshell                         #fad6a5

	pt shell                              ptshell                         #c19a6b

	puce                                    puce                            #722f37

	pueblo                                  pueblo                          #6f4e37

	puke                                    puke                            #722f37

	pumpkin                                 pumpkin                         #ed872d

	punjab                                  punjab                          #be6516

	punjab                                  punjab                          #ae6938

	puritan grey                            puritangrey                     #81878b

	puritan grey                            puritangrey                     #848482

	purple aster                            purpleaster                     #86608e

	purple brown                            purplebrown                     #79443b

	purple heather                          purpleheather                   #86608e

	purple lake                             purplelake                      #841b2d

	purple lake                             purplelake                      #ab4e52

	purple madder                           purplemadder                    #ab4e52

	purple madder                           purplemadder                    #722f37

	purple navy                             purplenavy                      #00304e

	purple navy                             purplenavy                      #4e5180

	purple navy                             purplenavy                      #252440

	purple ochre                            purpleochre                     #882d17

	purple oxide                            purpleoxide                     #79443b

	purple rubiate                          purplerubiate                   #ab4e52

	purple rubiate                          purplerubiate                   #722f37

	purrea arabica                          purreaarabica                   #eaa221

	purrea arabica                          purreaarabica                   #e3a857

	purree                                  purree                          #eaa221

	purree                                  purree                          #e3a857

	putty                                   putty                           #f3e5ab

	putty                                   putty                           #c2b280

	pygmalion                               pygmalion                       #996515

	pygmalion                               pygmalion                       #c19a6b

	pygmalion                               pygmalion                       #826644

	pyrethrum yellow                        pyrethrumyellow                 #fada5e

	pyrethrum yellow                        pyrethrumyellow                 #d4af37

	pyrethrum yellow                        pyrethrumyellow                 #f8de7e

	pyrethrum yellow                        pyrethrumyellow                 #c9ae5d

	pyrite yellow                           pyriteyellow                    #ab9144

	quail                                   quail                           #5b504f

	quail                                   quail                           #555555

	quaker                                  quaker                          #8b8589

	quaker                                  quaker                          #848482

	quaker blue                             quakerblue                      #222222

	quaker drab                             quakerdrab                      #848482

	quaker green                            quakergreen                     #4a5d23

	quaker grey                             quakergrey                      #8c8767

	queen anne green                        queenannegreen                  #8f9779

	queen blue                              queenblue                       #436b95

	queen's blue                            queensblue                      #436b95

	queen's yellow                          queensyellow                    #f8de7e

	quercitron                              quercitron                      #f8de7e

	quercitron lake                         quercitronlake                  #e3a857

	quimper                                 quimper                         #536878

	quince yellow                           quinceyellow                    #c9ae5d

	rabbit                                  rabbit                          #5b504f

	racquet                                 racquet                         #635147

	racquet                                 racquet                         #7e6d5a

	raddle                                  raddle                          #79443b

	radiance                                radiance                        #e4717a

	radiant yellow                          radiantyellow                   #ed872d

	radiant yellow                          radiantyellow                   #d99058

	radiant yellow                          radiantyellow                   #eaa221

	radiant yellow                          radiantyellow                   #e3a857

	radio                                   radio                           #00304e

	radio                                   radio                           #536878

	radio blue                              radioblue                       #36454f

	raffia                                  raffia                          #c19a6b

	raffia                                  raffia                          #c2b280

	ragged sailor                           raggedsailor                    #4e5180

	rail                                    rail                            #555555

	rainette green                          rainettegreen                   #8a9a5b

	raisin                                  raisin                          #673147

	raisin black                            raisinblack                     #242124

	raisin purple                           raisinpurple                    #702963

	rambler rose                            ramblerrose                     #e4717a

	rambler rose                            ramblerrose                     #a8516e

	rameau                                  rameau                          #222222

	rameses                                 rameses                         #00416a

	ramier blue                             ramierblue                      #796878

	rangoon                                 rangoon                         #6f4e37

	raphael                                 raphael                         #436b95

	raphael                                 raphael                         #00304e

	rapids                                  rapids                          #436b95

	raspberry                               raspberry                       #722f37

	raspberry glac                         raspberryglac                   #ab4e52

	raspberry glac                         raspberryglac                   #905d5d

	raspberry glac                         raspberryglac                   #a8516e

	raspberry glac                         raspberryglac                   #915f6d

	raspberry red                           raspberryred                    #ab4e52

	rat                                     rat                             #57554c

	rattan                                  rattan                          #c9ae5d

	raw italian earth                       rawitalianearth                 #ae6938

	raw italian earth                       rawitalianearth                 #a67b5b

	raw sienna                              rawsienna                       #ae6938

	raw sienna                              rawsienna                       #a67b5b

	raw umber                               rawumber                        #826644

	raw umber                               rawumber                        #4b3621

	raw umber                               rawumber                        #6f4e37

	rcamier                                rcamier                         #dea5a4

	rcamier                                rcamier                         #f88379

	realgar                                 realgar                         #eaa221

	realgar                                 realgar                         #e3a857

	red banana                              redbanana                       #841b2d

	red banana                              redbanana                       #ab4e52

	red bole                                redbole                         #79443b

	red chalk                               redchalk                        #79443b

	red cross                               redcross                        #be0032

	red cross                               redcross                        #bc3f4a

	redding                                 redding                         #79443b

	reddle                                  reddle                          #79443b

	red earth                               redearth                        #79443b

	redfeather                              redfeather                      #bc3f4a

	redfeather                              redfeather                      #ab4e52

	red in plates                           redinplates                     #be0032

	red lead                                redlead                         #e25822

	red ochre                               redochre                        #79443b

	red oxide                               redoxide                        #882d17

	red oxide                               redoxide                        #80461b

	red robin                               redrobin                        #56070c

	redwood                                 redwood                         #722f37

	redwood                                 redwood                         #882d17

	redwood                                 redwood                         #79443b

	redwood redwood -)                      redwoodredwood                  #ab4e52

	redwood redwood -)                      redwoodredwood                  #ab4e52

	redwood redwood -)                      redwoodredwood                  #905d5d

	reed green                              reedgreen                       #c9dc89

	reed yellow                             reedyellow                      #eae679

	reed yellow                             reedyellow                      #ebe8a4

	regal                                   regal                           #272458

	regal purple                            regalpurple                     #604e97

	regatta                                 regatta                         #0067a5

	regatta                                 regatta                         #545aa7

	regimental                              regimental                      #4c516d

	rejane green                            rejanegreen                     #679267

	rembrandt                               rembrandt                       #3e322c

	rembrandt's madder                      rembrandtsmadder                #cb6d51

	rembrandt's madder                      rembrandtsmadder                #9e4732

	rembrandt's madder                      rembrandtsmadder                #b4745e

	rembrandt's madder                      rembrandtsmadder                #79443b

	renaissance                             renaissance                     #ce4676

	reseda [green]                          resedagreen                     #867e36

	resolute                                resolute                        #436b95

	reveree                                 reveree                         #ecd5c5

	reveree                                 reveree                         #c7ada3

	rhododendron                            rhododendron                    #673147

	rhodonite pink                          rhodonitepink                   #c17e91

	rhone                                   rhone                           #c9dc89

	rhone                                   rhone                           #dadfb7

	rhubarb                                 rhubarb                         #867e36

	rifle [green]                           riflegreen                      #31362b

	rifle [green]                           riflegreen                      #222222

	riga                                    riga                            #317873

	riga blue                               rigablue                        #3a4b47

	riga blue                               rigablue                        #536878

	rinnemann's green                       rinnemannsgreen                 #679267

	ripple green                            ripplegreen                     #5e716a

	ripple green                            ripplegreen                     #317873

	risigal                                 risigal                         #eaa221

	risigal                                 risigal                         #e3a857

	rivage green                            rivagegreen                     #8a9a5b

	rivage green                            rivagegreen                     #93c592

	riviera                                 riviera                         #0067a5

	roan                                    roan                            #79443b

	robinhood green                         robinhoodgreen                  #1b4d3e

	robin's egg blue                        robinseggblue                   #8da399

	robin's egg blue                        robinseggblue                   #66ada4

	rocou                                   rocou                           #d9a6a9

	rodent                                  rodent                          #555555

	roe                                     roe                             #c19a6b

	roe                                     roe                             #a18f60

	roma blue                               romablue                        #4c516d

	roma blue                               romablue                        #554c69

	roman earth                             romanearth                      #d99058

	romanesque                              romanesque                      #5d3954

	roman green                             romangreen                      #665d1e

	roman lake                              romanlake                       #be0032

	roman lake                              romanlake                       #bc3f4a

	roman ochre                             romanochre                      #d99058

	roman purple                            romanpurple                     #604e81

	roman sepia                             romansepia                      #6f4e37

	romantic green                          romanticgreen                   #867e36

	roman umber                             romanumber                      #826644

	roman umber                             romanumber                      #4b3621

	roman violet                            romanviolet                     #5d3954

	romany                                  romany                          #36454f

	romany                                  romany                          #202830

	romany                                  romany                          #252440

	romany                                  romany                          #4c516d

	romarin                                 romarin                         #3a4b47

	rosalgar                                rosalgar                        #eaa221

	rosalgar                                rosalgar                        #e3a857

	rosario                                 rosario                         #a67b5b

	rose amber                              roseamber                       #ae6938

	rose amber                              roseamber                       #a67b5b

	rose ash                                roseash                         #905d5d

	rose beige                              rosebeige                       #7e6d5a

	rose beige 2                            rosebeige2                      #a67b5b

	rosebisque                              rosebisque                      #a87c6d

	rosebloom                               rosebloom                       #ad8884

	rosebloom                               rosebloom                       #c48379

	rosebloom                               rosebloom                       #b4745e

	rosebloom                               rosebloom                       #a87c6d

	rose blush                              roseblush                       #a87c6d

	rose blush 2                            roseblush2                      #a67b5b

	rose breath                             rosebreath                      #f9ccca

	rose breath                             rosebreath                      #ead8d7

	rosebud                                 rosebud                         #de6fa1

	rosebud                                 rosebud                         #c17e91

	rose caroline                           rosecaroline                    #d9a6a9

	rose carthame                           rosecarthame                    #be0032

	rose castor                             rosecastor                      #8f817f

	rose cendre                             rosecendre                      #d9a6a9

	rose d'alma                             rosedalma                       #915f6d

	rose d'althoea                          rosedalthoea                    #ea9399

	rose dawn                               rosedawn                        #d9a6a9

	rose dawn                               rosedawn                        #a87c6d

	rose de nymphe                          rosedenymphe                    #f3e5ab

	rose de provence                        rosedeprovence                  #b3446c

	rose dore                              rosedore                        #e4717a

	rose dore                              rosedore                        #ab4e52

	rosedust                                rosedust                        #905d5d

	rosedust                                rosedust                        #674c47

	rose ebony                              roseebony                       #43302e

	rose france                             rosefrance                      #d9a6a9

	roseglow                                roseglow                        #a87c6d

	rose grey                               rosegrey                        #5b504f

	rose hermosa                            rosehermosa                     #dea5a4

	rose hortensia                          rosehortensia                   #a87c6d

	rose lake                               roselake                        #9e4732

	roseleaf                                roseleaf                        #dea5a4

	roselustre                              roselustre                      #722f37

	rose madder                             rosemadder                      #b3446c

	rose marie                              rosemarie                       #e4717a

	rosemary                                rosemary                        #836479

	rose morn                               rosemorn                        #d9a6a9

	rose neyron                             roseneyron                      #bc3f4a

	rose neyron                             roseneyron                      #b3446c

	rose nilsson                            rosenilsson                     #e4717a

	rose nude                               rosenude                        #d9a6a9

	rose nude                               rosenude                        #c7ada3

	rose nude                               rosenude                        #c2ac99

	rose oak                                roseoak                         #79443b

	rose of picardy                         roseofpicardy                   #ab4e52

	rose of sharon                          roseofsharon                    #ab4e52

	rose petal                              rosepetal                       #905d5d

	rose pink                               rosepink                        #9e4732

	rose-purple                             rosepurple                      #b695c0

	rose-purple                             rosepurple                      #b784a7

	rose quartz                             rosequartz                      #aa98a9

	rose soire                             rosesoire                       #e4717a

	rosestone                               rosestone                       #977f73

	roset                                   roset                           #9e4732

	rosetan                                 rosetan                         #c2ac99

	rosetan                                 rosetan                         #958070

	rose taupe                              rosetaupe                       #555555

	rosetta                                 rosetta                         #5c504f

	rosevale                                rosevale                        #ab4e52

	rosevale                                rosevale                        #905d5d

	rosewood                                rosewood                        #79443b

	roslyn blue                             roslynblue                      #272458

	roucou                                  roucou                          #d9a6a9

	rouge de fer                            rougedefer                      #79443b

	rouge vegetal                           rougevegetal                    #be0032

	royal blue                              royalblue                       #00416a

	royal blue                              royalblue                       #436b95

	royal blue                              royalblue                       #00416a

	royal pink                              royalpink                       #e4717a

	royal purple                            royalpurple                     #604e97

	royal yellow                            royalyellow                     #fada5e

	royal yellow                            royalyellow                     #d4af37

	royal yellow                            royalyellow                     #f8de7e

	royal yellow                            royalyellow                     #c9ae5d

	rubaiyat                                rubaiyat                        #722f37

	rubber                                  rubber                          #6c541e

	rubelite                                rubelite                        #ce4676

	ruben's madder                          rubensmadder                    #9e4732

	rubient                                 rubient                         #722f37

	rubrica                                 rubrica                         #79443b

	ruby                                    ruby                            #722f37

	ruby of arsenic                         rubyofarsenic                   #eaa221

	ruby of arsenic                         rubyofarsenic                   #e3a857

	ruddle                                  ruddle                          #79443b

	rufous                                  rufous                          #f88379

	rufous                                  rufous                          #d99058

	rugby tan                               rugbytan                        #80461b

	runnymede                               runnymede                       #3b7861

	ru ochre                                ruochre                         #ae6938

	ru ochre                                ruochre                         #996515

	russet brown                            russetbrown                     #80461b

	russet brown                            russetbrown                     #6f4e37

	russet green                            russetgreen                     #b9b459

	russet green                            russetgreen                     #98943e

	russet green                            russetgreen                     #b9b57d

	russet green                            russetgreen                     #867e36

	russet orange                           russetorange                    #d9603b

	russian blue                            russianblue                     #91a3b0

	russian blue                            russianblue                     #81878b

	russian calf                            russiancalf                     #6f4e37

	russian green                           russiangreen                    #679267

	russian green                           russiangreen                    #355e3b

	russian green                           russiangreen                    #5e716a

	russian grey                            russiangrey                     #8a8776

	russian grey                            russiangrey                     #848482

	russian violet                          russianviolet                   #32174d

	rust                                    rust                            #80461b

	rustic brown                            rusticbrown                     #79443b

	rustic drab                             rusticdrab                      #967117

	rut ochre                               rutochre                        #ae6938

	rut ochre                               rutochre                        #996515

	sable                                   sable                           #7e6d5a

	saddle                                  saddle                          #635147

	saddle                                  saddle                          #7e6d5a

	safaran                                 safaran                         #d99058

	safaran                                 safaran                         #e3a857

	safflor                                 safflor                         #be0032

	safflower red                           safflowerred                    #be0032

	saffor                                  saffor                          #be0032

	saffron yellow                          saffronyellow                   #d99058

	saffron yellow                          saffronyellow                   #e3a857

	safrano pink                            safranopink                     #d9a6a9

	sagebrush green                         sagebrushgreen                  #4e5755

	sage drab                               sagedrab                        #515744

	sage [green]                            sagegreen                       #8a9a5b

	sage grey                               sagegrey                        #8c8767

	sahara                                  sahara                          #6f4e37

	sailor                                  sailor                          #291e29

	sailor                                  sailor                          #242124

	sailor blue                             sailorblue                      #4e5180

	sakkara                                 sakkara                         #5b504f

	sallow                                  sallow                          #b9b57d

	salmon [pink]                           salmonpink                      #d9a6a9

	salvia                                  salvia                          #be0032

	salvia blue                             salviablue                      #0067a5

	samara                                  samara                          #317873

	samovar                                 samovar                         #c19a6b

	samurai                                 samurai                         #b4745e

	sand                                    sand                            #8c8767

	sand                                    sand                            #8a8776

	sandalwood                              sandalwood                      #ab4e52

	sandalwood                              sandalwood                      #977f73

	sandalwood                              sandalwood                      #6f4e37

	sandalwood                              sandalwood                      #958070

	sandalwood                              sandalwood                      #635147

	sandarach                               sandarach                       #e25822

	sandarach                               sandarach                       #eaa221

	sandarach                               sandarach                       #e3a857

	sandaracha                              sandaracha                      #e25822

	sandaracha                              sandaracha                      #eaa221

	sandaracha                              sandaracha                      #e3a857

	sand dune                               sanddune                        #967117

	sander's blue                           sandersblue                     #436b95

	sanderswood                             sanderswood                     #ab4e52

	sandix                                  sandix                          #e25822

	sandrift                                sandrift                        #8e8279

	sandstone                               sandstone                       #a87c6d

	sandstone                               sandstone                       #a67b5b

	sandstone                               sandstone                       #958070

	sandust                                 sandust                         #d9a6a9

	sandy-beige                             sandybeige                      #8e8279

	sandyx                                  sandyx                          #e25822

	sang de boeuf                           sangdeboeuf                     #79443b

	sanguine                                sanguine                        #bc3f4a

	sanguine                                sanguine                        #ab4e52

	sanguineus                              sanguineus                      #841b2d

	sanguineus                              sanguineus                      #722f37

	sanguineus                              sanguineus                      #3f1728

	sanguineus                              sanguineus                      #3e1d1e

	sanguineus                              sanguineus                      #5d3954

	sanguineus                              sanguineus                      #673147

	santalwood                              santalwood                      #ab4e52

	santos                                  santos                          #6f4e37

	santos                                  santos                          #635147

	saona                                   saona                           #79443b

	sappanwood                              sappanwood                      #ab4e52

	sappanwood                              sappanwood                      #905d5d

	sapphire [blue]                         sapphireblue                    #00304e

	sappho                                  sappho                          #ad8884

	saraband                                saraband                        #905d5d

	saratoga                                saratoga                        #d99058

	saravan                                 saravan                         #80461b

	saravan                                 saravan                         #6f4e37

	sarouk                                  sarouk                          #635147

	satinwood                               satinwood                       #a67b5b

	satsuma                                 satsuma                         #cb6d51

	saturnine red                           saturninered                    #e25822

	saturn red                              saturnred                       #e25822

	saul                                    saul                            #4e5755

	saunder's blue                          saundersblue                    #436b95

	sauterne                                sauterne                        #c9ae5d

	sauterne                                sauterne                        #a67b5b

	saxe blue                               saxeblue                        #536878

	saxon blue                              saxonblue                       #007791

	saxon blue                              saxonblue                       #004958

	saxon green                             saxongreen                      #317873

	saxony blue                             saxonyblue                      #007791

	saxony blue                             saxonyblue                      #004958

	saxony blue                             saxonyblue                      #00416a

	saxony blue                             saxonyblue                      #436b95

	saxony green                            saxonygreen                     #008856

	sayal brown                             sayalbrown                      #ae6938

	sayal brown                             sayalbrown                      #a67b5b

	sayal brown                             sayalbrown                      #996515

	scabiosa                                scabiosa                        #796878

	scarab                                  scarab                          #3b7861

	scarlet                                 scarlet                         #be0032

	scarlet in grain                        scarletingrain                  #be0032

	scarlet lake                            scarletlake                     #bc3f4a

	scarlet lake                            scarletlake                     #ab4e52

	scarlet madder                          scarletmadder                   #f88379

	scarlet ochre                           scarletochre                    #882d17

	scarlet red                             scarletred                      #be0032

	scarlet vermilion                       scarletvermilion                #e25822

	scheele's green                         scheelesgreen                   #7e9f2e

	scheele's green                         scheelesgreen                   #44944a

	schoenfeld's purple                     schoenfeldspurple               #78184a

	schoenfeld's purple                     schoenfeldspurple               #a8516e

	schweinfurt green                       schweinfurtgreen                #3eb489

	scotch blue                             scotchblue                      #252440

	scotch grey                             scotchgrey                      #8f9779

	scotch grey                             scotchgrey                      #7d8984

	seabird                                 seabird                         #555555

	sea blue                                seablue                         #004b49

	seacrest                                seacrest                        #93c592

	seafoam                                 seafoam                         #3eb489

	seafoam                                 seafoam                         #6aab8e

	seafoam green                           seafoamgreen                    #dadfb7

	seafoam yellow                          seafoamyellow                   #dadfb7

	sea green                               seagreen                        #8a9a5b

	sea hawk                                seahawk                         #57554c

	seal [brown]                            sealbrown                       #483c32

	seaman blue                             seamanblue                      #4c516d

	sea mist                                seamist                         #8b8589

	sea mist                                seamist                         #848482

	sea moss                                seamoss                         #867e36

	sea moss                                seamoss                         #3a4b47

	sea pink                                seapink                         #ea9399

	sea pink                                seapink                         #dea5a4

	seasan not seasand                      seasannotseasand                #c2ac99

	seasan not seasand                      seasannotseasand                #fad6a5

	seasan not seasand                      seasannotseasand                #c19a6b

	seasan not seasand                      seasannotseasand                #ae9b82

	sea shell                               seashell                        #d9a6a9

	sea shell                               seashell                        #d99058

	sea shell                               seashell                        #a67b5b

	seashell pink                           seashellpink                    #f4c2c2

	seashell pink                           seashellpink                    #d9a6a9

	seaside                                 seaside                         #967117

	seaside                                 seaside                         #977f73

	seaside                                 seaside                         #958070

	seaside                                 seaside                         #8e8279

	seaspray                                seaspray                        #8f9779

	seaspray                                seaspray                        #679267

	seaspray                                seaspray                        #8da399

	sea-water green                         seawatergreen                   #8a9a5b

	seaweed green                           seaweedgreen                    #8f9779

	sedge                                   sedge                           #635147

	seed pearl                              seedpearl                       #f3e5ab

	seed pearl                              seedpearl                       #c2b280

	seered green                            seeredgreen                     #8a9a5b

	seladon green                           seladongreen                    #8f9779

	seminole                                seminole                        #6f4e37

	sepia                                   sepia                           #3b3121

	serapi                                  serapi                          #536878

	serpent                                 serpent                         #8da399

	serpentine                              serpentine                      #6aab8e

	serpentine green                        serpentinegreen                 #867e36

	service corps                           servicecorps                    #00543d

	seville                                 seville                         #32174d

	seville orange                          sevilleorange                   #d99058

	sevres [blue]                           sevresblue                      #0067a5

	shadow                                  shadow                          #8f817f

	shadow green                            shadowgreen                     #8a9a5b

	shagbark                                shagbark                        #826644

	shagbark                                shagbark                        #7e6d5a

	shale green                             shalegreen                      #679267

	shamrock                                shamrock                        #44944a

	shamrock [green]                        shamrockgreen                   #44944a

	shark                                   shark                           #796878

	shark                                   shark                           #5d555b

	sheepskin                               sheepskin                       #c2b280

	sheik                                   sheik                           #ab4e52

	sheik                                   sheik                           #9e4732

	shell grey                              shellgrey                       #bfb8a5

	shirvan                                 shirvan                         #3a4b47

	shrimp [pink]                           shrimppink                      #f88379

	shrimp [red]                            shrimpred                       #f88379

	siam                                    siam                            #6f4e37

	siam                                    siam                            #635147

	siam                                    siam                            #826644

	siam                                    siam                            #7e6d5a

	siberian brown                          siberianbrown                   #6c541e

	siberien                                siberien                        #3a4b47

	sicilian umber                          sicilianumber                   #826644

	sicilian umber                          sicilianumber                   #4b3621

	siderin yellow                          siderinyellow                   #ed872d

	siderin yellow                          siderinyellow                   #d99058

	siena                                   siena                           #882d17

	sienna brown                            siennabrown                     #6f4e37

	sienna yellow                           siennayellow                    #fad6a5

	sienna yellow                           siennayellow                    #c9ae5d

	siennese drab                           siennesedrab                    #977f73

	siennese drab                           siennesedrab                    #958070

	sierra                                  sierra                          #882d17

	signal red                              signalred                       #be0032

	sil                                     sil                             #e3a857

	silurian gray                           siluriangray                    #dadfb7

	silver fern                             silverfern                      #867e36

	silver fern                             silverfern                      #8c8767

	silver green                            silvergreen                     #8f9779

	silverpine                              silverpine                      #5e716a

	silver-wing                             silverwing                      #8b8589

	sinbad                                  sinbad                          #36454f

	sinbad                                  sinbad                          #202428

	sinbad                                  sinbad                          #50404d

	sinbad                                  sinbad                          #242124

	sirocco                                 sirocco                         #8c8767

	siskin green                            siskingreen                     #b9b57d

	sistine                                 sistine                         #66aabc

	sistine                                 sistine                         #367588

	sistine                                 sistine                         #91a3b0

	sistine                                 sistine                         #536878

	skating                                 skating                         #004958

	ski                                     ski                             #355e3b

	ski                                     ski                             #3b7861

	skimmed-milk white                      skimmedmilkwhite                #dadfb7

	skobeloff green                         skobeloffgreen                  #007a74

	sky                                     sky                             #70a3cc

	sky                                     sky                             #8791bf

	sky blue                                skyblue                         #70a3cc

	sky blue                                skyblue                         #8791bf

	sky colour                              skycolour                       #70a3cc

	sky colour                              skycolour                       #8791bf

	sky green                               skygreen                        #c9dc89

	slag                                    slag                            #555555

	slate                                   slate                           #51585e

	slate-black                             slateblack                      #222222

	slate blue                              slateblue                       #536878

	slate-blue                              slateblue                       #536878

	slate [color]                           slatecolor                      #5d555b

	slate-green                             slategreen                      #5e716a

	slate-green                             slategreen                      #4e5755

	slate-grey                              slategrey                       #8a8776

	slate-grey                              slategrey                       #848482

	slate-olive                             slateolive                      #4e5755

	slate-purple                            slatepurple                     #50404d

	slate-purple                            slatepurple                     #5d3954

	slate-violet                            slateviolet                     #905d5d

	smalt                                   smalt                           #00416a

	smalt                                   smalt                           #436b95

	smalt green                             smaltgreen                      #679267

	smaltino                                smaltino                        #00416a

	smaltino                                smaltino                        #436b95

	smoke blue                              smokeblue                       #51585e

	smoke brown                             smokebrown                      #555555

	smoked pearl                            smokedpearl                     #555555

	smoke grey                              smokegrey                       #dadfb7

	smoke yellow                            smokeyellow                     #8c8767

	smoke yellow                            smokeyellow                     #8a8776

	snapdragon                              snapdragon                      #f8de7e

	snapdragon                              snapdragon                      #c9ae5d

	snowshoe                                snowshoe                        #eaa221

	snowshoe                                snowshoe                        #e3a857

	soapstone                               soapstone                       #635147

	solferino                               solferino                       #a8516e

	solitaire                               solitaire                       #43302e

	solitaire                               solitaire                       #3e322c

	somalis                                 somalis                         #9e4732

	somalis                                 somalis                         #79443b

	sombrero                                sombrero                        #c2b280

	sonora                                  sonora                          #ae6938

	sonora                                  sonora                          #a67b5b

	soot brown                              sootbrown                       #635147

	soot brown                              sootbrown                       #7e6d5a

	sooty black                             sootyblack                      #222222

	sorbier                                 sorbier                         #915f6d

	sorghum brown                           sorghumbrown                    #977f73

	sorghum brown                           sorghumbrown                    #958070

	sorolla brown                           sorollabrown                    #80461b

	sorrel                                  sorrel                          #ae6938

	sorrel                                  sorrel                          #a67b5b

	sorrento green                          sorrentogreen                   #317873

	souci                                   souci                           #ed872d

	source                                  source                          #679267

	source                                  source                          #6aab8e

	source                                  source                          #8da399

	spalte                                  spalte                          #483c32

	spaltum                                 spaltum                         #483c32

	spanish brown                           spanishbrown                    #882d17

	spanish cedar                           spanishcedar                    #905d5d

	spanish cedar                           spanishcedar                    #79443b

	spanish flesh                           spanishflesh                    #f3e5ab

	spanish flesh                           spanishflesh                    #c2b280

	spanish green                           spanishgreen                    #679267

	spanish ochre                           spanishochre                    #ed872d

	spanish raisin                          spanishraisin                   #43302e

	spanish red                             spanishred                      #be0032

	spanish red                             spanishred                      #882d17

	spanish red                             spanishred                      #80461b

	spanish wine                            spanishwine                     #722f37

	spanish yellow                          spanishyellow                   #eaa221

	spanish yellow                          spanishyellow                   #e3a857

	spark                                   spark                           #be0032

	sparrow                                 sparrow                         #5b504f

	spa-tan                                 spatan                          #ae6938

	spearmint                               spearmint                       #008856

	sphinx                                  sphinx                          #7e6d5a

	spice                                   spice                           #ae6938

	spinach green                           spinachgreen                    #8a9a5b

	spinach green                           spinachgreen                    #4a5d23

	spinel pink                             spinelpink                      #a8516e

	spinel red                              spinelred                       #bc3f4a

	spinel red                              spinelred                       #b3446c

	sponge                                  sponge                          #967117

	spray                                   spray                           #96ded1

	spring beauty                           springbeauty                    #ce4676

	spring beauty                           springbeauty                    #b3446c

	spring green                            springgreen                     #8a9a5b

	spring green                            springgreen                     #93c592

	springtime                              springtime                      #e4717a

	sprite                                  sprite                          #00a693

	sprite                                  sprite                          #66ada4

	spruce                                  spruce                          #3a4b47

	spruce                                  spruce                          #1e2321

	spruce ochre                            spruceochre                     #ae6938

	spruce ochre                            spruceochre                     #996515

	spruce yellow                           spruceyellow                    #be8a3d

	spruce yellow                           spruceyellow                    #c19a6b

	squill blue                             squillblue                      #70a3cc

	squirrel                                squirrel                        #5d555b

	squirrel                                squirrel                        #555555

	stag                                    stag                            #826644

	starch blue                             starchblue                      #00416a

	starch blue                             starchblue                      #436b95

	stardew                                 stardew                         #536878

	starflower                              starflower                      #c17e91

	starlight                               starlight                       #91a3b0

	starling                                starling                        #5b504f

	st. benoit or saint benoit              stbenoitorsaintbenoit           #483c32

	st. benoit or saint benoit              stbenoitorsaintbenoit           #222222

	steel                                   steel                           #555555

	steel grey                              steelgrey                       #555555

	steeplechase                            steeplechase                    #545aa7

	stil de grain brown                     stildegrainbrown                #be8a3d

	stil de grain brown                     stildegrainbrown                #996515

	stil de grain yellow                    stildegrainyellow               #fada5e

	stil de grain yellow                    stildegrainyellow               #d4af37

	stil de grain yellow                    stildegrainyellow               #f8de7e

	stil de grain yellow                    stildegrainyellow               #c9ae5d

	stone blue                              stoneblue                       #436b95

	stone grey                              stonegrey                       #57554c

	storm grey                              stormgrey                       #7d8984

	strawberry                              strawberry                      #905d5d

	strawberry pink                         strawberrypink                  #e4717a

	strawberry pink                         strawberrypink                  #f88379

	strawberry pink                         strawberrypink                  #e66761

	straw [yellow]                          strawyellow                     #f3e5ab

	striegau earth                          striegauearth                   #79443b

	stroller tan                            strollertan                     #6f4e37

	strontian yellow                        strontianyellow                 #e9e450

	stucco                                  stucco                          #977f73

	stucco                                  stucco                          #958070

	succory blue                            succoryblue                     #dadfb7

	sudan                                   sudan                           #a18f60

	sudan                                   sudan                           #967117

	sudan                                   sudan                           #8c8767

	sudan brown                             sudanbrown                      #80461b

	sudan brown                             sudanbrown                      #6f4e37

	suede                                   suede                           #a67b5b

	suede                                   suede                           #6f4e37

	suede                                   suede                           #826644

	sugar cane                              sugarcane                       #fad6a5

	sugar cane                              sugarcane                       #c19a6b

	sulphate green                          sulphategreen                   #66ada4

	sulphate green                          sulphategreen                   #317873

	sulphine yellow                         sulphineyellow                  #ab9144

	sulphur                                 sulphur                         #eae679

	sulphur [yellow]                        sulphuryellow                   #eae679

	sultan                                  sultan                          #ab4e52

	sultana                                 sultana                         #722f37

	sultana                                 sultana                         #673147

	sumac                                   sumac                           #be8a3d

	sumac                                   sumac                           #996515

	sumac                                   sumac                           #c19a6b

	sumac                                   sumac                           #826644

	sumac                                   sumac                           #ab9144

	sumac                                   sumac                           #a18f60

	sumac                                   sumac                           #967117

	sunbeam                                 sunbeam                         #c2b280

	sunburn                                 sunburn                         #a67b5b

	sunburst                                sunburst                        #ed872d

	sunburst                                sunburst                        #d99058

	sundown                                 sundown                         #c19a6b

	sunflower [yellow]                      sunfloweryellow                 #fada5e

	sunglow                                 sunglow                         #e4717a

	sunglow                                 sunglow                         #ab4e52

	sungod                                  sungod                          #e25822

	sunkiss                                 sunkiss                         #d99058

	sunlight                                sunlight                        #c9ae5d

	sunni                                   sunni                           #f8de7e

	sunni                                   sunni                           #c9ae5d

	sunray                                  sunray                          #e3a857

	sunrise yellow                          sunriseyellow                   #d9a6a9

	sunrise yellow                          sunriseyellow                   #d99058

	sunset                                  sunset                          #fad6a5

	sunstone                                sunstone                        #ae6938

	suntan                                  suntan                          #a67b5b

	suntan                                  suntan                          #c19a6b

	suntan                                  suntan                          #826644

	superior                                superior                        #004b49

	surf green                              surfgreen                       #679267

	surrey green                            surreygreen                     #8f9779

	surrey green                            surreygreen                     #8da399

	swamp                                   swamp                           #3a4b47

	swedish green                           swedishgreen                    #7e9f2e

	swedish green                           swedishgreen                    #44944a

	sweet briar                             sweetbriar                      #de6fa1

	sweetmeat                               sweetmeat                       #c19a6b

	sweet pea                               sweetpea                        #c7ada3

	sweet william                           sweetwilliam                    #e4717a

	swiss blue                              swissblue                       #536878

	swiss rose                              swissrose                       #ab4e52

	syrup                                   syrup                           #967117

	syrup                                   syrup                           #6c541e

	taffy                                   taffy                           #c19a6b

	talavera                                talavera                        #ae6938

	tamarach                                tamarach                        #635147

	tamarach                                tamarach                        #7e6d5a

	ta-ming                                 taming                          #eaa221

	ta-ming                                 taming                          #e3a857

	tan                                     tan                             #ae6938

	tanagra                                 tanagra                         #79443b

	tanaura                                 tanaura                         #c2b280

	tanbark                                 tanbark                         #6f4e37

	tangerine                               tangerine                       #cb6d51

	tangier                                 tangier                         #ed872d

	tango pink                              tangopink                       #905d5d

	tansan                                  tansan                          #a67b5b

	tansan                                  tansan                          #c19a6b

	tapestry                                tapestry                        #00304e

	tapestry                                tapestry                        #252440

	tapestry                                tapestry                        #4c516d

	tapestry                                tapestry                        #8b8589

	tapestry red                            tapestryred                     #722f37

	tapis vert                              tapisvert                       #8a9a5b

	tarragon                                tarragon                        #8a9a5b

	tarragon                                tarragon                        #8f9779

	tarragon                                tarragon                        #679267

	tarragona                               tarragona                       #882d17

	tarragona                               tarragona                       #80461b

	tartan green                            tartangreen                     #3a4b47

	taupe                                   taupe                           #483c32

	tawny                                   tawny                           #ae6938

	tawny                                   tawny                           #a67b5b

	tawny birch                             tawnybirch                      #a87c6d

	tawny birch                             tawnybirch                      #a67b5b

	tawny birch                             tawnybirch                      #958070

	tea                                     tea                             #8f9779

	tea                                     tea                             #8da399

	tea green                               teagreen                        #8f9779

	teak [brown]                            teakbrown                       #6f4e37

	teakwood                                teakwood                        #635147

	teakwood                                teakwood                        #7e6d5a

	teal blue                               tealblue                        #004958

	teal blue                               tealblue                        #36454f

	teal duck                               tealduck                        #004958

	tea time                                teatime                         #967117

	telegraph blue                          telegraphblue                   #796878

	tennis                                  tennis                          #ab9144

	tennis                                  tennis                          #a18f60

	terra cotta                             terracotta                      #be6516

	terra cotta                             terracotta                      #ae6938

	terra japonica                          terrajaponica                   #6f4e37

	terra lemnia                            terralemnia                     #79443b

	terra merita                            terramerita                     #f8de7e

	terra orellana                          terraorellana                   #d9a6a9

	terra orellano                          terraorellano                   #d9a6a9

	terrapin                                terrapin                        #6f4e37

	terra pozzuoli                          terrapozzuoli                   #79443b

	terra rosa                              terrarosa                       #79443b

	terra siena                             terrasiena                      #ae6938

	terra siena                             terrasiena                      #a67b5b

	terra sigillata                         terrasigillata                  #79443b

	terrasse                                terrasse                        #8a9a5b

	terra umber                             terraumber                      #826644

	terra umber                             terraumber                      #4b3621

	testaceous                              testaceous                      #d99058

	testaceous                              testaceous                      #a67b5b

	thenard's blue                          thenardsblue                    #007791

	thistle                                 thistle                         #86608e

	thistle bloom                           thistlebloom                    #9e4f88

	thistletuft                             thistletuft                     #702963

	thrush                                  thrush                          #826644

	thrush                                  thrush                          #7e6d5a

	thulite pink                            thulitepink                     #de6fa1

	thyme                                   thyme                           #3a4b47

	tiber                                   tiber                           #1b4d3e

	tiber green                             tibergreen                      #93c592

	tiffin                                  tiffin                          #826644

	tigerlily                               tigerlily                       #d9603b

	tigerlily                               tigerlily                       #cb6d51

	tile blue                               tileblue                        #367588

	tile red                                tilered                         #cb6d51

	tilleul buff                            tilleulbuff                     #ecd5c5

	tilleul buff                            tilleulbuff                     #bfb8a5

	tilleul [green]                         tilleulgreen                    #ebe8a4

	tinsel                                  tinsel                          #ab9144

	tinsel                                  tinsel                          #a18f60

	tinsel                                  tinsel                          #967117

	titania                                 titania                         #5d3954

	titania                                 titania                         #673147

	titian                                  titian                          #ae6938

	titian gold                             titiangold                      #ae6938

	titmouse blue                           titmouseblue                    #8da399

	toast                                   toast                           #a67b5b

	toasted almond                          toastedalmond                   #a67b5b

	toboggan                                toboggan                        #e25822

	toboggan                                toboggan                        #f38400

	toga                                    toga                            #2f2140

	tokay                                   tokay                           #79443b

	tokyo                                   tokyo                           #ffb7a5

	tokyo                                   tokyo                           #f88379

	tokyo                                   tokyo                           #ed872d

	toltec                                  toltec                          #e3a857

	tomato [red]                            tomatored                       #9e4732

	tommy red                               tommyred                        #bc3f4a

	topaz                                   topaz                           #a67b5b

	topaz                                   topaz                           #be8a3d

	topaz                                   topaz                           #c19a6b

	toquet                                  toquet                          #c7ada3

	toreador                                toreador                        #e25822

	torino blue                             torinoblue                      #536878

	tortoise                                tortoise                        #80461b

	tortoise shell                          tortoiseshell                   #6f4e37

	totem                                   totem                           #9e4732

	tourmaline                              tourmaline                      #66ada4

	tourmaline pink                         tourmalinepink                  #915f6d

	tourterelle                             tourterelle                     #967117

	transparent chromium oxide              transparentchromiumoxide        #007959

	transparent chromium oxide              transparentchromiumoxide        #3b7861

	transparent gold ochre                  transparentgoldochre            #ae6938

	transparent gold ochre                  transparentgoldochre            #996515

	traprock                                traprock                        #555555

	travertine                              travertine                      #967117

	trentanel                               trentanel                       #eae679

	trentanel                               trentanel                       #b9b459

	trianon                                 trianon                         #d9a6a9

	trianon                                 trianon                         #d99058

	triton                                  triton                          #1b4d3e

	triumph blue                            triumphblue                     #436b95

	triumph blue                            triumphblue                     #00304e

	trooper                                 trooper                         #00304e

	trooper                                 trooper                         #36454f

	trooper                                 trooper                         #202830

	trotteur tan                            trotteurtan                     #6f4e37

	troubador red                           troubadorred                    #bc3f4a

	trublu                                  trublu                          #536878

	tuileries                               tuileries                       #436b95

	tulipwood                               tulipwood                       #6f4e37

	tunis                                   tunis                           #36454f

	tunis                                   tunis                           #202830

	turbith mineral                         turbithmineral                  #f8de7e

	turkey blue                             turkeyblue                      #4c516d

	turkey red                              turkeyred                       #ab4e52

	turkey umber                            turkeyumber                     #826644

	turkey umber                            turkeyumber                     #4b3621

	turkish blue                            turkishblue                     #4c516d

	turkish crescent red                    turkishcrescentred              #841b2d

	turkish crescent red                    turkishcrescentred              #722f37

	turkish red                             turkishred                      #ab4e52

	turmeric                                turmeric                        #f8de7e

	turner's yellow                         turnersyellow                   #fada5e

	turner's yellow                         turnersyellow                   #d4af37

	turner's yellow                         turnersyellow                   #f8de7e

	turner's yellow                         turnersyellow                   #c9ae5d

	turquoise [blue]                        turquoiseblue                   #66ada4

	turquoise green                         turquoisegreen                  #66ada4

	turtle                                  turtle                          #6f4e37

	turtledove                              turtledove                      #5c504f

	turtledove                              turtledove                      #555555

	turtle green                            turtlegreen                     #8a9a5b

	tuscan                                  tuscan                          #fad6a5

	tuscan brown                            tuscanbrown                     #79443b

	tuscan brown                            tuscanbrown                     #6f4e37

	tuscan red                              tuscanred                       #79443b

	tuscan tan                              tuscantan                       #a67b5b

	tuscany                                 tuscany                         #79443b

	tussore                                 tussore                         #f4c2c2

	tussore                                 tussore                         #d9a6a9

	twilight [blue]                         twilightblue                    #536878

	twine                                   twine                           #8c8767

	twinkle blue                            twinkleblue                     #66aabc

	tyrian blue                             tyrianblue                      #536878

	tyrian blue                             tyrianblue                      #4c516d

	tyrian pink                             tyrianpink                      #ce4676

	tyrian violet                           tyrianviolet                    #a8516e

	tyrol                                   tyrol                           #826644

	tyrolese green                          tyrolesegreen                   #679267

	tyrolian                                tyrolian                        #1a2421

	tyrolite green                          tyrolitegreen                   #66ada4

	tzarine                                 tzarine                         #1e2321

	ultramarine green                       ultramarinegreen                #1e2321

	ultramarine yellow                      ultramarineyellow               #d4af37

	urania blue                             uraniablue                      #2f2140

	vagabond green                          vagabondgreen                   #004b49

	valencia                                valencia                        #de6fa1

	vanda                                   vanda                           #796878

	vanda                                   vanda                           #836479

	vandyke brown                           vandykebrown                    #6f4e37

	vandyke madder                          vandykemadder                   #a8516e

	vandyke red                             vandykered                      #905d5d

	vanilla                                 vanilla                         #f3e5ab

	vanilla                                 vanilla                         #c2b280

	vanity                                  vanity                          #d9a6a9

	variscite green                         variscitegreen                  #6aab8e

	varley's grey                           varleysgrey                     #796878

	vassar rose                             vassarrose                      #c08081

	vassar rose                             vassarrose                      #905d5d

	vassar tan                              vassartan                       #80461b

	vassar tan                              vassartan                       #6f4e37

	vatican                                 vatican                         #563c5c

	vatican                                 vatican                         #301934

	veau d'or                               veaudor                         #fad6a5

	veau d'or                               veaudor                         #c19a6b

	vegetable red                           vegetablered                    #be0032

	vegetable rouge                         vegetablerouge                  #be0032

	velasquez                               velasquez                       #673147

	velvet brown                            velvetbrown                     #826644

	velvet brown                            velvetbrown                     #4b3621

	velvet green                            velvetgreen                     #4a5d23

	venet                                   venet                           #66ada4

	venetian blue                           venetianblue                    #007791

	venetian fuchsia                        venetianfuchsia                 #ce4676

	venetian lake                           venetianlake                    #be0032

	venetian lake                           venetianlake                    #bc3f4a

	venetian pink                           venetianpink                    #dea5a4

	venetian red                            venetianred                     #882d17

	venetian rose                           venetianrose                    #841b2d

	venetian rose                           venetianrose                    #ab4e52

	venetian scarlet                        venetianscarlet                 #be0032

	venetian yellow                         venetianyellow                  #f8de7e

	venetian yellow                         venetianyellow                  #c9ae5d

	venezia                                 venezia                         #317873

	venice berries                          veniceberries                   #eaa221

	venice berries                          veniceberries                   #e3a857

	venice [blue]                           veniceblue                      #66ada4

	venice green                            venicegreen                     #66ada4

	venice red                              venicered                       #79443b

	venus                                   venus                           #ab9144

	venus                                   venus                           #a18f60

	verbena                                 verbena                         #915c83

	verbena [violet]                        verbenaviolet                   #9690ab

	verbena [violet]                        verbenaviolet                   #aa98a9

	verdant green                           verdantgreen                    #8a9a5b

	verdet                                  verdet                          #679267

	verde vessie                            verdevessie                     #867e36

	verd gay                                verdgay                         #8a9a5b

	verdigris [green]                       verdigrisgreen                  #679267

	verditer green                          verditergreen                   #679267

	verdure                                 verdure                         #355e3b

	veridine green                          veridinegreen                   #6aab8e

	vernonia purple                         vernoniapurple                  #915f6d

	verona brown                            veronabrown                     #6f4e37

	verona yellow                           veronayellow                    #fada5e

	verona yellow                           veronayellow                    #d4af37

	verona yellow                           veronayellow                    #f8de7e

	verona yellow                           veronayellow                    #c9ae5d

	veronese yellow                         veroneseyellow                  #fada5e

	veronese yellow                         veroneseyellow                  #d4af37

	veronese yellow                         veroneseyellow                  #f8de7e

	veronese yellow                         veroneseyellow                  #c9ae5d

	veronica                                veronica                        #604e81

	versailles                              versailles                      #66aabc

	vert russe                              vertrusse                       #679267

	vert russe                              vertrusse                       #355e3b

	vert russe                              vertrusse                       #5e716a

	vervain                                 vervain                         #9690ab

	vervain                                 vervain                         #aa98a9

	vestal                                  vestal                          #796878

	vestal                                  vestal                          #836479

	veteran                                 veteran                         #36454f

	vetiver green                           vetivergreen                    #8c8767

	victoria                                victoria                        #563c5c

	victoria blue                           victoriablue                    #0067a5

	victoria green                          victoriagreen                   #6aab8e

	victoria green                          victoriagreen                   #8da399

	victoria green                          victoriagreen                   #66ada4

	victoria lake                           victorialake                    #722f37

	vienna blue                             viennablue                      #007791

	vienna brown                            viennabrown                     #635147

	vienna brown                            viennabrown                     #7e6d5a

	vienna green                            viennagreen                     #3eb489

	vienna lake                             viennalake                      #be0032

	vienna lake                             viennalake                      #bc3f4a

	vienna smoke                            viennasmoke                     #555555

	vineyard                                vineyard                        #722f37

	vineyard                                vineyard                        #3f1728

	viola                                   viola                           #86608e

	violet-carmine                          violetcarmine                   #5d3954

	violine                                 violine                         #604e97

	violine                                 violine                         #604e81

	virgin                                  virgin                          #436b95

	viridian                                viridian                        #007959

	viridian                                viridian                        #3b7861

	viridine green                          viridinegreen                   #c9dc89

	viridine yellow                         viridineyellow                  #7e9f2e

	vitelline yellow                        vitellineyellow                 #d4af37

	vitellinous                             vitellinous                     #d4af37

	vitreous                                vitreous                        #c9dc89

	wad                                     wad                             #536878

	wald                                    wald                            #eae679

	wald                                    wald                            #b9b459

	wallflower [brown]                      wallflowerbrown                 #722f37

	wallflower [brown]                      wallflowerbrown                 #79443b

	wall green                              wallgreen                       #007a74

	walnut [brown]                          walnutbrown                     #c19a6b

	warbler green                           warblergreen                    #867e36

	warm sepia                              warmsepia                       #635147

	water blue                              waterblue                       #317873

	water-color                             watercolor                      #66ada4

	water cress                             watercress                      #8a9a5b

	waterfall                               waterfall                       #317873

	water green                             watergreen                      #b9b57d

	water green                             watergreen                      #c9dc89

	water green                             watergreen                      #dadfb7

	water grey                              watergrey                       #8a8776

	waterloo                                waterloo                        #4c516d

	watermelon                              watermelon                      #e4717a

	watermelon                              watermelon                      #ab4e52

	water sprite                            watersprite                     #93c592

	watteau                                 watteau                         #70a3cc

	wau                                     wau                             #eae679

	wau                                     wau                             #b9b459

	wax brown                               waxbrown                        #826644

	wax color                               waxcolor                        #ebe8a4

	wax color                               waxcolor                        #b9b57d

	waxen                                   waxen                           #ebe8a4

	waxen                                   waxen                           #b9b57d

	wax red                                 waxred                          #b4745e

	wax white                               waxwhite                        #ebe8a4

	wax white                               waxwhite                        #b9b57d

	wax yellow                              waxyellow                       #c9ae5d

	weathered oak                           weatheredoak                    #422518

	weigelia                                weigelia                        #ab4e52

	weld                                    weld                            #eae679

	weld                                    weld                            #b9b459

	westminster                             westminster                     #30267a

	westminster                             westminster                     #9065ca

	west point                              westpoint                       #51585e

	wheat                                   wheat                           #e3a857

	wheat                                   wheat                           #fad6a5

	whippet                                 whippet                         #6c541e

	whirlpool                               whirlpool                       #004958

	white blue                              whiteblue                       #b4bcc0

	white jade                              whitejade                       #ecd5c5

	white jade                              whitejade                       #bfb8a5

	wield                                   wield                           #eae679

	wield                                   wield                           #b9b459

	wigwam                                  wigwam                          #6f4e37

	wild aster                              wildaster                       #602f6b

	wild aster                              wildaster                       #702963

	wild cherry                             wildcherry                      #722f37

	wild cherry                             wildcherry                      #673147

	wild dove grey                          wilddovegrey                    #5b504f

	wild honey                              wildhoney                       #ae6938

	wild honey                              wildhoney                       #a67b5b

	wild iris                               wildiris                        #8b8589

	wild orchid                             wildorchid                      #836479

	wild pigeon                             wildpigeon                      #5b504f

	wild raspberry                          wildraspberry                   #722f37

	wild raspberry                          wildraspberry                   #673147

	wild rose                               wildrose                        #c08081

	wild strawberry                         wildstrawberry                  #841b2d

	wild strawberry                         wildstrawberry                  #722f37

	willow                                  willow                          #665d1e

	willow green                            willowgreen                     #867e36

	windflower                              windflower                      #915f6d

	windsor                                 windsor                         #563c5c

	windsor blue                            windsorblue                     #4c516d

	windsor green                           windsorgreen                    #355e3b

	windsor tan                             windsortan                      #ae6938

	wineberry                               wineberry                       #673147

	wine dregs                              winedregs                       #722f37

	wine dregs                              winedregs                       #56070c

	wine dregs                              winedregs                       #3e1d1e

	wine lees                               winelees                        #722f37

	wine lees                               winelees                        #56070c

	wine lees                               winelees                        #3e1d1e

	wine yellow                             wineyellow                      #f3e5ab

	wine yellow                             wineyellow                      #c2b280

	winter green                            wintergreen                     #679267

	wintergreen                             wintergreen                     #355e3b

	winter leaf                             winterleaf                      #635147

	wireless                                wireless                        #436b95

	wistaria [blue]                         wistariablue                    #8c82b6

	wistaria blue                           wistariablue                    #8791bf

	wistaria [violet]                       wistariaviolet                  #8c82b6

	witchwood                               witchwood                       #6f4e37

	withered leaf                           witheredleaf                    #be6516

	withered leaf                           witheredleaf                    #ae6938

	withered leaf                           witheredleaf                    #80461b

	withered rose                           witheredrose                    #905d5d

	withered rose                           witheredrose                    #79443b

	woad                                    woad                            #536878

	woald                                   woald                           #eae679

	woald                                   woald                           #b9b459

	wod                                     wod                             #eae679

	wod                                     wod                             #b9b459

	wode                                    wode                            #536878

	wold                                    wold                            #eae679

	wold                                    wold                            #b9b459

	woodash                                 woodash                         #bfb8a5

	woodbark                                woodbark                        #7e6d5a

	woodbine green                          woodbinegreen                   #8a9a5b

	woodbine green                          woodbinegreen                   #8f9779

	woodbine green                          woodbinegreen                   #4a5d23

	woodbine green                          woodbinegreen                   #515744

	woodland brown                          woodlandbrown                   #483c32

	woodland brown                          woodlandbrown                   #222222

	woodland green                          woodlandgreen                   #4a5d23

	woodland rose                           woodlandrose                    #a67b5b

	wood rose                               woodrose                        #977f73

	wood rose                               woodrose                        #958070

	wood violet                             woodviolet                      #602f6b

	wren                                    wren                            #363527

	wren                                    wren                            #57554c

	wrought iron                            wroughtiron                     #555555

	yacht                                   yacht                           #0067a5

	yale blue                               yaleblue                        #00416a

	yama                                    yama                            #66ada4

	yellow beige                            yellowbeige                     #c19a6b

	yellow beige                            yellowbeige                     #826644

	yellow berries                          yellowberries                   #eaa221

	yellow berries                          yellowberries                   #e3a857

	yellow brazil wood                      yellowbrazilwood                #c9ae5d

	yellow brazil wood                      yellowbrazilwood                #c2b280

	yellow carmine                          yellowcarmine                   #eaa221

	yellow carmine                          yellowcarmine                   #e3a857

	yellow daisy                            yellowdaisy                     #fada5e

	yellow daisy                            yellowdaisy                     #d4af37

	yellow daisy                            yellowdaisy                     #f8de7e

	yellow daisy                            yellowdaisy                     #c9ae5d

	yellow earth                            yellowearth                     #e3a857

	yellow madder                           yellowmadder                    #fada5e

	yellow madder                           yellowmadder                    #d4af37

	yellow madder                           yellowmadder                    #f8de7e

	yellow madder                           yellowmadder                    #c9ae5d

	yellow ochre                            yellowochre                     #e3a857

	yellow realgar                          yellowrealgar                   #fada5e

	yellow realgar                          yellowrealgar                   #d4af37

	yellow realgar                          yellowrealgar                   #f8de7e

	yellow realgar                          yellowrealgar                   #c9ae5d

	yellow sienna                           yellowsienna                    #e3a857

	yellow stone                            yellowstone                     #b9b57d

	yellowstone                             yellowstone                     #a18f60

	yellow wash                             yellowwash                      #e3a857

	yellow wood                             yellowwood                      #c9ae5d

	yellow wood                             yellowwood                      #c2b280

	yew green                               yewgreen                        #4a5d23

	yolk yellow                             yolkyellow                      #d4af37

	yosemite                                yosemite                        #c19a6b

	yucatan                                 yucatan                         #ae6938

	yucatan                                 yucatan                         #996515

	yucca                                   yucca                           #5e716a

	yu chi                                  yuchi                           #1b4d3e

	yvette violet                           yvetteviolet                    #796878

	zaffre blue                             zaffreblue                      #00416a

	zaffre blue                             zaffreblue                      #436b95

	zanzibar                                zanzibar                        #43302e

	zedoary wash                            zedoarywash                     #e3a857

	zenith [blue]                           zenithblue                      #8791bf

	zenith [blue]                           zenithblue                      #8c92ac

	zephyr                                  zephyr                          #ead8d7

	zephyr                                  zephyr                          #c4aead

	zinc                                    zinc                            #8b8589

	zinc                                    zinc                            #848482

	zinc green                              zincgreen                       #679267

	zinc orange                             zincorange                      #ed872d

	zinc orange                             zincorange                      #d99058

	zinc yellow                             zincyellow                      #d4af37

	zinnia                                  zinnia                          #ab4e52

	zulu                                    zulu                            #43302e

	zuni brown                              zunibrown                       #6f4e37


=cut

sub _load_color_list() {
    return [
['nbs-iscc-m:abbey.207','abbey','abbey',[96,78,151],'604e97',6311575],
['nbs-iscc-m:absinthegreen.120','absinthegreen','absinthe [green] ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:absintheyellow.105','absintheyellow','absinthe yellow',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:acacia.101','acacia','acacia',[234,230,121],'eae679',15394425],
['nbs-iscc-m:acacia.102','acacia','acacia',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:academyblue.173','academyblue','academy blue',[54,117,136],'367588',3569032],
['nbs-iscc-m:acajou.43','acajou','acajou',[121,68,59],'79443b',7947323],
['nbs-iscc-m:acanthe.95','acanthe','acanthe',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:acier.109','acier','acier',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:ackermannsgreen.137','ackermannsgreen','ackermann\'s green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:aconiteviolet.223','aconiteviolet','aconite violet',[134,96,142],'86608e',8806542],
['nbs-iscc-m:acorn.80','acorn','acorn',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:adamia.224','adamia','adamia',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:adelaide.208','adelaide','adelaide',[50,23,77],'32174d',3282765],
['nbs-iscc-m:aden.144','aden','aden',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:admiral.230','admiral','admiral',[41,30,41],'291e29',2694697],
['nbs-iscc-m:admiral.235','admiral','admiral',[36,33,36],'242124',2367780],
['nbs-iscc-m:adobe.77','adobe','adobe',[130,102,68],'826644',8545860],
['nbs-iscc-m:adrianoplered.15','adrianoplered','adrianople red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:adriatic.164','adriatic','adriatic',[49,120,115],'317873',3242099],
['nbs-iscc-m:adust.54','adust','adust',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:aero.186','aero','aero',[83,104,120],'536878',5466232],
['nbs-iscc-m:afghan.62','afghan','afghan',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:afghanred.11','afghanred','afghan red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:african.267','african','african',[34,34,34],'222222',2236962],
['nbs-iscc-m:africanbrown.62','africanbrown','african brown',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:afterglow.214','afterglow','afterglow',[150,144,171],'9690ab',9867435],
['nbs-iscc-m:agate.40','agate','agate',[136,45,23],'882d17',8924439],
['nbs-iscc-m:agate.55','agate','agate',[128,70,27],'80461b',8406555],
['nbs-iscc-m:agategrey.264','agategrey','agate grey',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:ageratumblue.227','ageratumblue','ageratum blue',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:ageratumviolet.228','ageratumviolet','ageratum violet',[121,104,120],'796878',7956600],
['nbs-iscc-m:airblue.182','airblue','air blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:airedale.77','airedale','airedale',[130,102,68],'826644',8545860],
['nbs-iscc-m:airedale.80','airedale','airedale',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:airway.181','airway','airway',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:akbar.41','akbar','akbar',[86,7,12],'56070c',5637900],
['nbs-iscc-m:akbar.43','akbar','akbar',[121,68,59],'79443b',7947323],
['nbs-iscc-m:alabaster.31','alabaster','alabaster',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-m:alabaster.93','alabaster','alabaster',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:alamo.55','alamo','alamo',[128,70,27],'80461b',8406555],
['nbs-iscc-m:alcanna.40','alcanna','alcanna',[136,45,23],'882d17',8924439],
['nbs-iscc-m:alcazar.16','alcazar','alcazar',[114,47,55],'722f37',7483191],
['nbs-iscc-m:alcazar.40','alcazar','alcazar',[136,45,23],'882d17',8924439],
['nbs-iscc-m:alderney.77','alderney','alderney',[130,102,68],'826644',8545860],
['nbs-iscc-m:alesan.45','alesan','alesan',[151,127,115],'977f73',9928563],
['nbs-iscc-m:alesan.57','alesan','alesan',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:alesan.60','alesan','alesan',[149,128,112],'958070',9797744],
['nbs-iscc-m:alexandriablue.176','alexandriablue','alexandria blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:alexandriablue.178','alexandriablue','alexandria blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:alfalfa.137','alfalfa','alfalfa',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:algerian.58','algerian','algerian',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:algerianred.41','algerianred','algerian red',[86,7,12],'56070c',5637900],
['nbs-iscc-m:algonquin.51','algonquin','algonquin',[190,101,22],'be6516',12477718],
['nbs-iscc-m:algonquin.54','algonquin','algonquin',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:alhambra.151','alhambra','alhambra',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:alhambra.165','alhambra','alhambra',[0,75,73],'004b49',19273],
['nbs-iscc-m:aliceblue.185','aliceblue','alice blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:alkermes.11','alkermes','alkermes',[190,0,50],'be0032',12451890],
['nbs-iscc-m:almond.57','almond','almond',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:almond.60','almond','almond',[149,128,112],'958070',9797744],
['nbs-iscc-m:almond.76','almond','almond',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:almond.79','almond','almond',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:almondblossom.4','almondblossom','almond blossom',[249,204,202],'f9ccca',16370890],
['nbs-iscc-m:almondblossom.5','almondblossom','almond blossom',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:almondblossom.7','almondblossom','almond blossom',[234,216,215],'ead8d7',15390935],
['nbs-iscc-m:almondblossom.8','almondblossom','almond blossom',[196,174,173],'c4aead',12889773],
['nbs-iscc-m:almondbrown.57','almondbrown','almond brown ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:almondbrown.60','almondbrown','almond brown ',[149,128,112],'958070',9797744],
['nbs-iscc-m:almondbrown.76','almondbrown','almond brown ',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:almondbrown.79','almondbrown','almond brown ',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:almondgreen.150','almondgreen','almond green',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:almondpink.4','almondpink','almond [pink] ',[249,204,202],'f9ccca',16370890],
['nbs-iscc-m:almondpink.5','almondpink','almond [pink] ',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:almondpink.7','almondpink','almond [pink] ',[234,216,215],'ead8d7',15390935],
['nbs-iscc-m:almondpink.8','almondpink','almond [pink] ',[196,174,173],'c4aead',12889773],
['nbs-iscc-m:aloesgreen.149','aloesgreen','aloes [green] ',[141,163,153],'8da399',9282457],
['nbs-iscc-m:aloma.57','aloma','aloma',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:aloma.76','aloma','aloma',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:aloma.77','aloma','aloma',[130,102,68],'826644',8545860],
['nbs-iscc-m:alpine.186','alpine','alpine',[83,104,120],'536878',5466232],
['nbs-iscc-m:alpinegreen.151','alpinegreen','alpine green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:aluminum.264','aluminum','aluminum',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:aluminum.265','aluminum','aluminum',[132,132,130],'848482',8684674],
['nbs-iscc-m:amaranth.238','amaranth','amaranth',[112,41,99],'702963',7350627],
['nbs-iscc-m:amaranthine.259','amaranthine','amaranthine',[103,49,71],'673147',6762823],
['nbs-iscc-m:amaranthpink.248','amaranthpink','amaranth pink',[222,111,161],'de6fa1',14577569],
['nbs-iscc-m:amaranthpurple.256','amaranthpurple','amaranth purple',[120,24,74],'78184a',7870538],
['nbs-iscc-m:amarna.142','amarna','amarna',[0,84,61],'00543d',21565],
['nbs-iscc-m:amarna.146','amarna','amarna',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:amarylis.15','amarylis','amarylis',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:amarylis.19','amarylis','amarylis',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:amazon.145','amazon','amazon',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:amberbrown.54','amberbrown','amber brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:amberglow.54','amberglow','amberglow',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:amberglow.57','amberglow','amberglow',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:amberwhite.89','amberwhite','amber white',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:amberwhite.90','amberwhite','amber white',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:amberwhite.104','amberwhite','amber white',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-m:amberwhite.105','amberwhite','amber white',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:amberwhite.121','amberwhite','amber white',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:amberyellow.86','amberyellow','amber [yellow] ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:amberyellow.87','amberyellow','amber [yellow] ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:ambrosia.43','ambrosia','ambrosia',[121,68,59],'79443b',7947323],
['nbs-iscc-m:ambulance.47','ambulance','ambulance',[67,48,46],'43302e',4403246],
['nbs-iscc-m:amelie.62','amelie','amelie',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:americanbeauty.16','americanbeauty','american beauty',[114,47,55],'722f37',7483191],
['nbs-iscc-m:americangreen.136','americangreen','american green',[103,146,103],'679267',6787687],
['nbs-iscc-m:americangreen.137','americangreen','american green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:americangreen.150','americangreen','american green',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:amethystviolet.223','amethystviolet','amethyst [violet] ',[134,96,142],'86608e',8806542],
['nbs-iscc-m:amethystviolet.224','amethystviolet','amethyst [violet] ',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:amethystviolet.228','amethystviolet','amethyst [violet] ',[121,104,120],'796878',7956600],
['nbs-iscc-m:amparoblue.178','amparoblue','amparo blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:amparoblue.195','amparoblue','amparo blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-m:amparopurple.223','amparopurple','amparo purple',[134,96,142],'86608e',8806542],
['nbs-iscc-m:amulet.165','amulet','amulet',[0,75,73],'004b49',19273],
['nbs-iscc-m:anamite.109','anamite','anamite',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:anatolia.41','anatolia','anatolia',[86,7,12],'56070c',5637900],
['nbs-iscc-m:anatta.29','anatta','anatta',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:andorra.44','andorra','andorra',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-m:andorra.47','andorra','andorra',[67,48,46],'43302e',4403246],
['nbs-iscc-m:andorre.136','andorre','andorre',[103,146,103],'679267',6787687],
['nbs-iscc-m:andorre.150','andorre','andorre',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:andovergreen.127','andovergreen','andover green',[81,87,68],'515744',5330756],
['nbs-iscc-m:andrinopleberries.68','andrinopleberries','andrinople berries',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:andrinopleberries.71','andrinopleberries','andrinople berries',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:anemone.244','anemone','anemone',[170,138,158],'aa8a9e',11176606],
['nbs-iscc-m:angelred.43','angelred','angel red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:animalrouge.11','animalrouge','animal rouge',[190,0,50],'be0032',12451890],
['nbs-iscc-m:animalrouge.12','animalrouge','animal rouge',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:annapolis.266','annapolis','annapolis',[85,85,85],'555555',5592405],
['nbs-iscc-m:annatto.29','annatto','annatto',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:annotto.29','annotto','annotto',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:antelope.94','antelope','antelope',[150,113,23],'967117',9859351],
['nbs-iscc-m:antimonyyellow.71','antimonyyellow','antimony yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:antique.94','antique','antique',[150,113,23],'967117',9859351],
['nbs-iscc-m:antique.95','antique','antique',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:antiquebrass.106','antiquebrass','antique brass',[134,126,54],'867e36',8814134],
['nbs-iscc-m:antiquebrass.109','antiquebrass','antique brass',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:antiquebronze.77','antiquebronze','antique bronze',[130,102,68],'826644',8545860],
['nbs-iscc-m:antiquebrown.58','antiquebrown','antique brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:antiquedrab.77','antiquedrab','antique drab',[130,102,68],'826644',8545860],
['nbs-iscc-m:antiquedrab.94','antiquedrab','antique drab',[150,113,23],'967117',9859351],
['nbs-iscc-m:antiquedrab.95','antiquedrab','antique drab',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:antiquefuchsia.223','antiquefuchsia','antique fuchsia',[134,96,142],'86608e',8806542],
['nbs-iscc-m:antiquefuchsia.241','antiquefuchsia','antique fuchsia',[145,92,131],'915c83',9526403],
['nbs-iscc-m:antiquegold.72','antiquegold','antique gold',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:antiquegold.74','antiquegold','antique gold',[153,101,21],'996515',10052885],
['nbs-iscc-m:antiquegreen.151','antiquegreen','antique green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:antiquered.37','antiquered','antique red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:antiquered.38','antiquered','antique red',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:antiquered.39','antiquered','antique red',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:antiquered.43','antiquered','antique red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:antiqueruby.13','antiqueruby','antique ruby',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:antwerpblue.170','antwerpblue','antwerp blue',[46,132,149],'2e8495',3048597],
['nbs-iscc-m:antwerpblue.182','antwerpblue','antwerp blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:antwerpbrown.81','antwerpbrown','antwerp brown',[72,60,50],'483c32',4734002],
['nbs-iscc-m:antwerpred.43','antwerpred','antwerp red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:apache.58','apache','apache',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:aphrodite.163','aphrodite','aphrodite',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:appleblossompink.19','appleblossompink','appleblossom [pink] ',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:applefallow.13','applefallow','apple-fallow',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:applefallow.15','applefallow','apple-fallow',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:applegreen.120','applegreen','apple green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:apricot.53','apricot','apricot',[217,144,88],'d99058',14258264],
['nbs-iscc-m:apricotyellow.86','apricotyellow','apricot yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:apricotyellow.87','apricotyellow','apricot yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:aquagreen.135','aquagreen','aquagreen',[147,197,146],'93c592',9684370],
['nbs-iscc-m:aquamarine.172','aquamarine','aquamarine',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:aquamarine.185','aquamarine','aquamarine',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:arabbrown.55','arabbrown','arab [brown] ',[128,70,27],'80461b',8406555],
['nbs-iscc-m:arabesque.37','arabesque','arabesque',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:arabianbrown.55','arabianbrown','arabian brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:arabianbrown.58','arabianbrown','arabian brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:arabianred.41','arabianred','arabian red',[86,7,12],'56070c',5637900],
['nbs-iscc-m:araby.39','araby','araby',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:araby.42','araby','araby',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:aragon.43','aragon','aragon',[121,68,59],'79443b',7947323],
['nbs-iscc-m:arbutus.2','arbutus','arbutus',[234,147,153],'ea9399',15373209],
['nbs-iscc-m:arbutus.3','arbutus','arbutus',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:arbutus.247','arbutus','arbutus',[230,143,172],'e68fac',15110060],
['nbs-iscc-m:arbutus.248','arbutus','arbutus',[222,111,161],'de6fa1',14577569],
['nbs-iscc-m:arcadiangreen.135','arcadiangreen','arcadian green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:archel.258','archel','archel',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:archil.258','archil','archil ',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:arctic.267','arctic','arctic',[34,34,34],'222222',2236962],
['nbs-iscc-m:arcticblue.186','arcticblue','arctic blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:ardoise.186','ardoise','ardoise',[83,104,120],'536878',5466232],
['nbs-iscc-m:ardoise.228','ardoise','ardoise',[121,104,120],'796878',7956600],
['nbs-iscc-m:argali.64','argali','argali',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:argali.266','argali','argali',[85,85,85],'555555',5592405],
['nbs-iscc-m:argent.265','argent','argent',[132,132,130],'848482',8684674],
['nbs-iscc-m:argentina.59','argentina','argentina',[66,37,24],'422518',4334872],
['nbs-iscc-m:argusbrown.58','argusbrown','argus brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:argylepurple.244','argylepurple','argyle purple',[170,138,158],'aa8a9e',11176606],
['nbs-iscc-m:argylepurple.245','argylepurple','argyle purple',[131,100,121],'836479',8610937],
['nbs-iscc-m:arizona.76','arizona','arizona',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:armada.47','armada','armada',[67,48,46],'43302e',4403246],
['nbs-iscc-m:armada.62','armada','armada',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:armenianblue.176','armenianblue','armenian blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:armenianblue.178','armenianblue','armenian blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:armenianbole.43','armenianbole','armenian bole',[121,68,59],'79443b',7947323],
['nbs-iscc-m:armenianred.55','armenianred','armenian red',[128,70,27],'80461b',8406555],
['nbs-iscc-m:armenianstone.182','armenianstone','armenian stone',[67,107,149],'436b95',4418453],
['nbs-iscc-m:armybrown.57','armybrown','army brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:arnoblue.173','arnoblue','arno blue',[54,117,136],'367588',3569032],
['nbs-iscc-m:arnoblue.174','arnoblue','arno blue',[0,73,88],'004958',18776],
['nbs-iscc-m:arnotto.29','arnotto','arnotto',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:arona.187','arona','arona',[54,69,79],'36454f',3556687],
['nbs-iscc-m:arrowwood.94','arrowwood','arrowwood',[150,113,23],'967117',9859351],
['nbs-iscc-m:arsenate.141','arsenate','arsenate',[0,121,89],'007959',31065],
['nbs-iscc-m:artbrown.59','artbrown','art brown',[66,37,24],'422518',4334872],
['nbs-iscc-m:artemesiagreen.155','artemesiagreen','artemesia green',[125,137,132],'7d8984',8227204],
['nbs-iscc-m:artgrayartgrey.233','artgrayartgrey','art gray, art grey',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:artgrayartgrey.265','artgrayartgrey','art gray, art grey',[132,132,130],'848482',8684674],
['nbs-iscc-m:artgreen.120','artgreen','art green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:artgreen.125','artgreen','art green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:artichokegreen.122','artichokegreen','artichoke green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:artificialvermilion.262','artificialvermilion','artificial vermilion',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:artillery.11','artillery','artillery',[190,0,50],'be0032',12451890],
['nbs-iscc-m:ascottan.61','ascottan','ascot tan',[99,81,71],'635147',6508871],
['nbs-iscc-m:ashesofrose.262','ashesofrose','ashes of rose',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:ashgrey.154','ashgrey','ash [grey] ',[178,190,181],'b2beb5',11714229],
['nbs-iscc-m:asmalte.179','asmalte','asmalte',[0,65,106],'00416a',16746],
['nbs-iscc-m:asmalte.182','asmalte','asmalte',[67,107,149],'436b95',4418453],
['nbs-iscc-m:asparagusgreen.121','asparagusgreen','asparagus green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:aspengreen.150','aspengreen','aspen green',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:asphalt.266','asphalt','asphalt',[85,85,85],'555555',5592405],
['nbs-iscc-m:asphaltum.81','asphaltum','asphaltum',[72,60,50],'483c32',4734002],
['nbs-iscc-m:asphodelgreen.120','asphodelgreen','asphodel green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:aster.228','aster','aster',[121,104,120],'796878',7956600],
['nbs-iscc-m:asterpurple.256','asterpurple','aster purple',[120,24,74],'78184a',7870538],
['nbs-iscc-m:athenia.19','athenia','athenia',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:atlantic.187','atlantic','atlantic',[54,69,79],'36454f',3556687],
['nbs-iscc-m:atlantis.174','atlantis','atlantis',[0,73,88],'004958',18776],
['nbs-iscc-m:atlantis.187','atlantis','atlantis',[54,69,79],'36454f',3556687],
['nbs-iscc-m:atmosphere.33','atmosphere','atmosphere',[194,172,153],'c2ac99',12758169],
['nbs-iscc-m:atmosphere.79','atmosphere','atmosphere',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:atmosphere.93','atmosphere','atmosphere',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:atramentous.192','atramentous','atramentous',[81,88,94],'51585e',5331038],
['nbs-iscc-m:attarofroses.39','attarofroses','attar of roses',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:attarofroses.42','attarofroses','attar of roses',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:aubergine.230','aubergine','aubergine',[41,30,41],'291e29',2694697],
['nbs-iscc-m:aubergine.235','aubergine','aubergine',[36,33,36],'242124',2367780],
['nbs-iscc-m:auburn.58','auburn','auburn',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:aubusson.16','aubusson','aubusson',[114,47,55],'722f37',7483191],
['nbs-iscc-m:aucuba.106','aucuba','aucuba',[134,126,54],'867e36',8814134],
['nbs-iscc-m:aureolin.83','aureolin','aureolin',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:aureolin.84','aureolin','aureolin',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:auriculapurple.238','auriculapurple','auricula purple',[112,41,99],'702963',7350627],
['nbs-iscc-m:auripigmentum.83','auripigmentum','auripigmentum',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:auripigmentum.84','auripigmentum','auripigmentum',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:auripigmentum.86','auripigmentum','auripigmentum',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:auripigmentum.87','auripigmentum','auripigmentum',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:auroraorange.26','auroraorange','aurora [orange] ',[248,131,121],'f88379',16286585],
['nbs-iscc-m:auroraorange.27','auroraorange','aurora [orange] ',[230,103,97],'e66761',15099745],
['nbs-iscc-m:auroraorange.37','auroraorange','aurora [orange] ',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:aurorared.15','aurorared','aurora red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:aurorared.27','aurorared','aurora red',[230,103,97],'e66761',15099745],
['nbs-iscc-m:aurorared.37','aurorared','aurora red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:aurorayellow.50','aurorayellow','aurora yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:aurorayellow.53','aurorayellow','aurora yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-m:aurorayellow.68','aurorayellow','aurora yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:aurorayellow.71','aurorayellow','aurora yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:aurore.5','aurore','aurore',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:aurum.94','aurum','aurum',[150,113,23],'967117',9859351],
['nbs-iscc-m:australianpine.137','australianpine','australian pine',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:auteuil.71','auteuil','auteuil',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:autumn.110','autumn','autumn',[91,88,66],'5b5842',5986370],
['nbs-iscc-m:autumn.111','autumn','autumn',[54,53,39],'363527',3552551],
['nbs-iscc-m:autumnblonde.45','autumnblonde','autumn blonde',[151,127,115],'977f73',9928563],
['nbs-iscc-m:autumnblonde.58','autumnblonde','autumn blonde',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:autumnblonde.60','autumnblonde','autumn blonde',[149,128,112],'958070',9797744],
['nbs-iscc-m:autumnblonde.61','autumnblonde','autumn blonde',[99,81,71],'635147',6508871],
['nbs-iscc-m:autumnbrown.81','autumnbrown','autumn brown',[72,60,50],'483c32',4734002],
['nbs-iscc-m:autumnglory.16','autumnglory','autumn glory',[114,47,55],'722f37',7483191],
['nbs-iscc-m:autumnglory.38','autumnglory','autumn glory',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:autumngreen.120','autumngreen','autumn green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:autumngreen.125','autumngreen','autumn green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:autumnleaf.51','autumnleaf','autumn leaf',[190,101,22],'be6516',12477718],
['nbs-iscc-m:autumnleaf.54','autumnleaf','autumn leaf',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:autumnleaf.55','autumnleaf','autumn leaf',[128,70,27],'80461b',8406555],
['nbs-iscc-m:autumnoak.46','autumnoak','autumn oak',[103,76,71],'674c47',6769735],
['nbs-iscc-m:avellaneous.57','avellaneous','avellaneous',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:avellaneous.74','avellaneous','avellaneous',[153,101,21],'996515',10052885],
['nbs-iscc-m:avignonberries.68','avignonberries','avignon berries',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:avignonberries.71','avignonberries','avignon berries',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:azalea.19','azalea','azalea',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:aztec.74','aztec','aztec',[153,101,21],'996515',10052885],
['nbs-iscc-m:aztec.76','aztec','aztec',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:aztec.77','aztec','aztec',[130,102,68],'826644',8545860],
['nbs-iscc-m:aztecmaroon.262','aztecmaroon','aztec maroon',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:azulin.186','azulin','azulin',[83,104,120],'536878',5466232],
['nbs-iscc-m:babyblue.185','babyblue','baby blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:babyblue.190','babyblue','baby blue',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-m:babyblue.191','babyblue','baby blue',[129,135,139],'81878b',8488843],
['nbs-iscc-m:babypink.28','babypink','baby pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-m:babypink.29','babypink','baby pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:babyrose.26','babyrose','baby rose',[248,131,121],'f88379',16286585],
['nbs-iscc-m:bacchus.258','bacchus','bacchus',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:bachelorbutton.3','bachelorbutton','bachelor button',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:bagdad.183','bagdad','bagdad',[0,48,78],'00304e',12366],
['nbs-iscc-m:bagdad.187','bagdad','bagdad',[54,69,79],'36454f',3556687],
['nbs-iscc-m:bagdad.188','bagdad','bagdad',[32,40,48],'202830',2107440],
['nbs-iscc-m:bakstgreen.146','bakstgreen','bakst green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:balgeyellow.83','balgeyellow','balge yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:balllake.11','balllake','ball lake',[190,0,50],'be0032',12451890],
['nbs-iscc-m:balllake.12','balllake','ball lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:balsam.187','balsam','balsam',[54,69,79],'36454f',3556687],
['nbs-iscc-m:balsamgreen.267','balsamgreen','balsam green',[34,34,34],'222222',2236962],
['nbs-iscc-m:baltic.151','baltic','baltic',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:baltic.165','baltic','baltic',[0,75,73],'004b49',19273],
['nbs-iscc-m:bambino.185','bambino','bambino',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:bamboo.76','bamboo','bamboo',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:bamboo.77','bamboo','bamboo',[130,102,68],'826644',8545860],
['nbs-iscc-m:banana.90','banana','banana',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:banner.204','banner','banner',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:banshee.151','banshee','banshee',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:barberry.13','barberry','barberry',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:barberry.15','barberry','barberry',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:bark.96','bark','bark',[59,49,33],'3b3121',3879201],
['nbs-iscc-m:barwood.15','barwood','barwood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:basketball.205','basketball','basketball',[144,101,202],'9065ca',9463242],
['nbs-iscc-m:bastardsaffron.11','bastardsaffron','bastard saffron',[190,0,50],'be0032',12451890],
['nbs-iscc-m:bat.266','bat','bat',[85,85,85],'555555',5592405],
['nbs-iscc-m:bathstone.73','bathstone','bath stone',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:battleshipgrey.265','battleshipgrey','battleship grey',[132,132,130],'848482',8684674],
['nbs-iscc-m:bay.58','bay','bay',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:bayou.151','bayou','bayou',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:beach.109','beach','beach',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:beach.112','beach','beach',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:beachtan.61','beachtan','beach tan ',[99,81,71],'635147',6508871],
['nbs-iscc-m:bear.266','bear','bear',[85,85,85],'555555',5592405],
['nbs-iscc-m:bearshair.266','bearshair','bear\'s hair',[85,85,85],'555555',5592405],
['nbs-iscc-m:bearskingrey.266','bearskingrey','bearskin [grey] ',[85,85,85],'555555',5592405],
['nbs-iscc-m:beaucaire.169','beaucaire','beaucaire',[0,119,145],'007791',30609],
['nbs-iscc-m:beaucaire.173','beaucaire','beaucaire',[54,117,136],'367588',3569032],
['nbs-iscc-m:beaver.64','beaver','beaver',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:beaverpelt.113','beaverpelt','beaverpelt',[87,85,76],'57554c',5723468],
['nbs-iscc-m:beech.110','beech','beech',[91,88,66],'5b5842',5986370],
['nbs-iscc-m:beech.113','beech','beech',[87,85,76],'57554c',5723468],
['nbs-iscc-m:beechnut.61','beechnut','beechnut',[99,81,71],'635147',6508871],
['nbs-iscc-m:beefsblood.43','beefsblood','beef\'s blood',[121,68,59],'79443b',7947323],
['nbs-iscc-m:beeswax.77','beeswax','beeswax',[130,102,68],'826644',8545860],
['nbs-iscc-m:beetle.127','beetle','beetle',[81,87,68],'515744',5330756],
['nbs-iscc-m:begonia.3','begonia','begonia',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:begoniarose.15','begoniarose','begonia rose',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:beige.89','beige','beige',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:beige.90','beige','beige',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:beigesoire.76','beigesoire','beige soire',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:belladonna.224','belladonna','belladonna',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:belleek.73','belleek','belleek',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:belleek.79','belleek','belleek',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:belleek.90','belleek','belleek',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:belleek.93','belleek','belleek',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:bellemonte.15','bellemonte','bellemonte',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:bellflower.197','bellflower','bellflower',[39,36,88],'272458',2565208],
['nbs-iscc-m:berlinbrown.19','berlinbrown','berlin brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:berlinbrown.46','berlinbrown','berlin brown',[103,76,71],'674c47',6769735],
['nbs-iscc-m:berlinred.43','berlinred','berlin red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:bermuda.3','bermuda','bermuda',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:beryl.172','beryl','beryl',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:berylblue.172','berylblue','beryl blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:berylgreen.163','berylgreen','beryl green',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:biarritz.79','biarritz','biarritz',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:biarritz.94','biarritz','biarritz',[150,113,23],'967117',9859351],
['nbs-iscc-m:bigfouryellowbig4yellow.48','bigfouryellowbig4yellow','big four yellow, big 4 yellow',[243,132,0],'f38400',15959040],
['nbs-iscc-m:bigfouryellowbig4yellow.50','bigfouryellowbig4yellow','big four yellow, big 4 yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:billiard.146','billiard','billiard',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:biscaygreen.120','biscaygreen','biscay green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:biscuit.57','biscuit','biscuit',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:biscuit.60','biscuit','biscuit',[149,128,112],'958070',9797744],
['nbs-iscc-m:biscuit.76','biscuit','biscuit',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:biscuit.79','biscuit','biscuit',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:bishop.211','bishop','bishop',[96,78,129],'604e81',6311553],
['nbs-iscc-m:bishopspurple.241','bishopspurple','bishop\'s purple',[145,92,131],'915c83',9526403],
['nbs-iscc-m:bishopsviolet.241','bishopsviolet','bishop\'s violet',[145,92,131],'915c83',9526403],
['nbs-iscc-m:biskra.81','biskra','biskra',[72,60,50],'483c32',4734002],
['nbs-iscc-m:bismarck.43','bismarck','bismarck',[121,68,59],'79443b',7947323],
['nbs-iscc-m:bismarckbrown.77','bismarckbrown','bismarck brown',[130,102,68],'826644',8545860],
['nbs-iscc-m:bison.81','bison','bison',[72,60,50],'483c32',4734002],
['nbs-iscc-m:bisque.73','bisque','bisque',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:bisque.93','bisque','bisque',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:bister.61','bister','bister',[99,81,71],'635147',6508871],
['nbs-iscc-m:bister.80','bister','bister',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:bistre.61','bistre','bistre',[99,81,71],'635147',6508871],
['nbs-iscc-m:bistre.80','bistre','bistre',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:bistregreen.88','bistregreen','bistre green',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:bistregreen.91','bistregreen','bistre green',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:bittersweet.36','bittersweet','bittersweet',[170,56,30],'aa381e',11155486],
['nbs-iscc-m:bittersweet.38','bittersweet','bittersweet',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:bittersweetorange.51','bittersweetorange','bittersweet orange',[190,101,22],'be6516',12477718],
['nbs-iscc-m:bittersweetpink.26','bittersweetpink','bittersweet pink',[248,131,121],'f88379',16286585],
['nbs-iscc-m:bitumen.81','bitumen','bitumen',[72,60,50],'483c32',4734002],
['nbs-iscc-m:bladdergreen.106','bladdergreen','bladder green',[134,126,54],'867e36',8814134],
['nbs-iscc-m:blaze.15','blaze','blaze ',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:blaze.38','blaze','blaze ',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:bleudelyons.179','bleudelyons','bleu de lyons',[0,65,106],'00416a',16746],
['nbs-iscc-m:bleudelyons.182','bleudelyons','bleu de lyons',[67,107,149],'436b95',4418453],
['nbs-iscc-m:bleudorient.178','bleudorient','bleu d\'orient',[0,103,165],'0067a5',26533],
['nbs-iscc-m:bleulouise.164','bleulouise','bleu louise',[49,120,115],'317873',3242099],
['nbs-iscc-m:bleulouise.173','bleulouise','bleu louise',[54,117,136],'367588',3569032],
['nbs-iscc-m:bleupass.185','bleupass','bleu pass ',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:bleupass.186','bleupass','bleu pass ',[83,104,120],'536878',5466232],
['nbs-iscc-m:bleupass.191','bleupass','bleu pass ',[129,135,139],'81878b',8488843],
['nbs-iscc-m:blond.76','blond','blond',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:blond.91','blond','blond',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:blondine.80','blondine','blondine',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:blonket.190','blonket','blonket',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-m:bloodred.12','bloodred','blood red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:bloodred.15','bloodred','blood red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:blossom.5','blossom','blossom',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:blueashes.182','blueashes','blue ashes',[67,107,149],'436b95',4418453],
['nbs-iscc-m:blueaster.182','blueaster','blue aster',[67,107,149],'436b95',4418453],
['nbs-iscc-m:bluebell.182','bluebell','bluebell',[67,107,149],'436b95',4418453],
['nbs-iscc-m:bluebell.183','bluebell','bluebell',[0,48,78],'00304e',12366],
['nbs-iscc-m:bluebice.182','bluebice','blue bice',[67,107,149],'436b95',4418453],
['nbs-iscc-m:bluebirdbluebird.182','bluebirdbluebird','blue bird, bluebird',[67,107,149],'436b95',4418453],
['nbs-iscc-m:bluedevil.204','bluedevil','blue devil',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:bluefox.23','bluefox','blue fox',[92,80,79],'5c504f',6049871],
['nbs-iscc-m:bluegrass.165','bluegrass','blue grass',[0,75,73],'004b49',19273],
['nbs-iscc-m:bluehaze.266','bluehaze','blue haze',[85,85,85],'555555',5592405],
['nbs-iscc-m:bluejay.179','bluejay','bluejay',[0,65,106],'00416a',16746],
['nbs-iscc-m:bluejay.183','bluejay','bluejay',[0,48,78],'00304e',12366],
['nbs-iscc-m:bluejewel.169','bluejewel','blue jewel',[0,119,145],'007791',30609],
['nbs-iscc-m:bluelavender.203','bluelavender','blue lavender',[140,146,172],'8c92ac',9212588],
['nbs-iscc-m:bluelotus.181','bluelotus','blue lotus',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:blueochre.150','blueochre','blue ochre',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:blueochre.186','blueochre','blue ochre',[83,104,120],'536878',5466232],
['nbs-iscc-m:blueorchid.186','blueorchid','blue orchid',[83,104,120],'536878',5466232],
['nbs-iscc-m:bluesapphire.173','bluesapphire','blue sapphire',[54,117,136],'367588',3569032],
['nbs-iscc-m:bluespruce.145','bluespruce','blue spruce',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:bluespruce.150','bluespruce','blue spruce',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:bluet.182','bluet','bluet',[67,107,149],'436b95',4418453],
['nbs-iscc-m:blueturquoise.163','blueturquoise','blue turquoise',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:blueturquoise.172','blueturquoise','blue turquoise',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:blueuntramarineash.182','blueuntramarineash','blue untramarine ash ',[67,107,149],'436b95',4418453],
['nbs-iscc-m:blueverditer.182','blueverditer','blue verditer',[67,107,149],'436b95',4418453],
['nbs-iscc-m:blunket.190','blunket','blunket',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-m:blush.29','blush','blush',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:blush.57','blush','blush',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:blushrose.19','blushrose','blush rose',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:boa.120','boa','boa',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:boa.121','boa','boa',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:boa.135','boa','boa',[147,197,146],'93c592',9684370],
['nbs-iscc-m:bobolink.80','bobolink','bobolink',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:boisderose.19','boisderose','bois de rose ',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:bokhara.259','bokhara','bokhara',[103,49,71],'673147',6762823],
['nbs-iscc-m:bole.43','bole','bole',[121,68,59],'79443b',7947323],
['nbs-iscc-m:bolearmoniack.43','bolearmoniack','bole armoniack',[121,68,59],'79443b',7947323],
['nbs-iscc-m:bolus.43','bolus','bolus',[121,68,59],'79443b',7947323],
['nbs-iscc-m:bombay.55','bombay','bombay',[128,70,27],'80461b',8406555],
['nbs-iscc-m:bombay.58','bombay','bombay',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:bonebrown.95','bonebrown','bone brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:bonebrown.96','bonebrown','bone brown',[59,49,33],'3b3121',3879201],
['nbs-iscc-m:bonfire.12','bonfire','bonfire',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:bonito.64','bonito','bonito',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:bonnieblue.176','bonnieblue','bonnie blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:boreal.43','boreal','boreal',[121,68,59],'79443b',7947323],
['nbs-iscc-m:bosphorus.151','bosphorus','bosphorus',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:botticelli.238','botticelli','botticelli',[112,41,99],'702963',7350627],
['nbs-iscc-m:bottlegreen.151','bottlegreen','bottle green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:bottlegreen.165','bottlegreen','bottle green',[0,75,73],'004b49',19273],
['nbs-iscc-m:bougainville.223','bougainville','bougainville',[134,96,142],'86608e',8806542],
['nbs-iscc-m:bougainville.241','bougainville','bougainville',[145,92,131],'915c83',9526403],
['nbs-iscc-m:boulevard.64','boulevard','boulevard',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:bouquetgreen.144','bouquetgreen','bouquet green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:bourgeon.120','bourgeon','bourgeon',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:boxgreen.120','boxgreen','box green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:bracken.95','bracken','bracken',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:bracken.96','bracken','bracken',[59,49,33],'3b3121',3879201],
['nbs-iscc-m:bramble.212','bramble','bramble',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:bran.57','bran','bran',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:bran.76','bran','bran',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:brass.87','brass','brass',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:brazenyellow.87','brazenyellow','brazen yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:brazilbrown.47','brazilbrown','brazil brown',[67,48,46],'43302e',4403246],
['nbs-iscc-m:brazilred.38','brazilred','brazil [red] ',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:brazilwood.15','brazilwood','brazilwood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:brazilwood.19','brazilwood','brazilwood',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:bremenblue.164','bremenblue','bremen blue',[49,120,115],'317873',3242099],
['nbs-iscc-m:bremengreen.136','bremengreen','bremen green',[103,146,103],'679267',6787687],
['nbs-iscc-m:brewstergreen.125','brewstergreen','brewster green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:briar.55','briar','briar',[128,70,27],'80461b',8406555],
['nbs-iscc-m:briar.58','briar','briar',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:briarwood.62','briarwood','briarwood',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:brickdust.13','brickdust','brickdust',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:brickred.55','brickred','brick red',[128,70,27],'80461b',8406555],
['nbs-iscc-m:brickred.58','brickred','brick red',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:bridalrose.6','bridalrose','bridal rose',[192,128,129],'c08081',12615809],
['nbs-iscc-m:brigand.12','brigand','brigand',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:brigand.15','brigand','brigand',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:brigand.35','brigand','brigand',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:brigand.37','brigand','brigand',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:brimstoneyellow.101','brimstoneyellow','brimstone [yellow] ',[234,230,121],'eae679',15394425],
['nbs-iscc-m:brittany.173','brittany','brittany',[54,117,136],'367588',3569032],
['nbs-iscc-m:brittany.182','brittany','brittany',[67,107,149],'436b95',4418453],
['nbs-iscc-m:brittany.186','brittany','brittany',[83,104,120],'536878',5466232],
['nbs-iscc-m:broccolibrown.64','broccolibrown','broccoli brown',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:broncho.81','broncho','broncho',[72,60,50],'483c32',4734002],
['nbs-iscc-m:broncho.95','broncho','broncho',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:broncho.96','broncho','broncho',[59,49,33],'3b3121',3879201],
['nbs-iscc-m:bronze.77','bronze','bronze',[130,102,68],'826644',8545860],
['nbs-iscc-m:bronzebrown.64','bronzebrown','bronze brown',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:bronzeclair.79','bronzeclair','bronze clair',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:bronzegreen.127','bronzegreen','bronze green',[81,87,68],'515744',5330756],
['nbs-iscc-m:bronzelustre.110','bronzelustre','bronze lustre',[91,88,66],'5b5842',5986370],
['nbs-iscc-m:bronzenude.95','bronzenude','bronze nude',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:bronzered.15','bronzered','bronze red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:bronzesheen.110','bronzesheen','bronzesheen',[91,88,66],'5b5842',5986370],
['nbs-iscc-m:bronzeyellow.53','bronzeyellow','bronze yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-m:bronzeyellow.72','bronzeyellow','bronze yellow',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:brownbay.61','brownbay','brown bay',[99,81,71],'635147',6508871],
['nbs-iscc-m:brownbread.55','brownbread','brown bread',[128,70,27],'80461b',8406555],
['nbs-iscc-m:brownbread.58','brownbread','brown bread',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:brownmadder.43','brownmadder','brown madder',[121,68,59],'79443b',7947323],
['nbs-iscc-m:brownochre.53','brownochre','brown ochre',[217,144,88],'d99058',14258264],
['nbs-iscc-m:brownpink.72','brownpink','brown pink',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:brownpink.74','brownpink','brown pink',[153,101,21],'996515',10052885],
['nbs-iscc-m:brownred.43','brownred','brown red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:brownstone.61','brownstone','brown stone',[99,81,71],'635147',6508871],
['nbs-iscc-m:brownstone.61','brownstone','brownstone',[99,81,71],'635147',6508871],
['nbs-iscc-m:brownsugar.58','brownsugar','brown sugar',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:brownsugar.61','brownsugar','brown sugar',[99,81,71],'635147',6508871],
['nbs-iscc-m:brownsugar.77','brownsugar','brown sugar',[130,102,68],'826644',8545860],
['nbs-iscc-m:brownsugar.80','brownsugar','brown sugar',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:bruges.174','bruges','bruges',[0,73,88],'004958',18776],
['nbs-iscc-m:bruges.187','bruges','bruges',[54,69,79],'36454f',3556687],
['nbs-iscc-m:brushwood.61','brushwood','brushwood',[99,81,71],'635147',6508871],
['nbs-iscc-m:brusselsbrown.58','brusselsbrown','brussels brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:brusselsbrown.61','brusselsbrown','brussels brown',[99,81,71],'635147',6508871],
['nbs-iscc-m:brusselsbrown.77','brusselsbrown','brussels brown',[130,102,68],'826644',8545860],
['nbs-iscc-m:brusselsbrown.80','brusselsbrown','brussels brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:buccaneer.12','buccaneer','buccaneer',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:buccaneer.15','buccaneer','buccaneer',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:buccaneer.36','buccaneer','buccaneer',[170,56,30],'aa381e',11155486],
['nbs-iscc-m:buccaneer.38','buccaneer','buccaneer',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:buckskin.45','buckskin','buckskin',[151,127,115],'977f73',9928563],
['nbs-iscc-m:buckskin.60','buckskin','buckskin',[149,128,112],'958070',9797744],
['nbs-iscc-m:buckthornberries.68','buckthornberries','buckthorn berries',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:buckthornberries.71','buckthornberries','buckthorn berries',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:buckthornbrown.72','buckthornbrown','buckthorn brown',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:buckthornbrown.74','buckthornbrown','buckthorn brown',[153,101,21],'996515',10052885],
['nbs-iscc-m:buckthornbrown.76','buckthornbrown','buckthorn brown',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:buckthornbrown.77','buckthornbrown','buckthorn brown',[130,102,68],'826644',8545860],
['nbs-iscc-m:buckthornbrown.88','buckthornbrown','buckthorn brown',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:buckthornbrown.91','buckthornbrown','buckthorn brown',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:buckthornbrown.94','buckthornbrown','buckthorn brown',[150,113,23],'967117',9859351],
['nbs-iscc-m:buddha.12','buddha','buddha',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:buddha.15','buddha','buddha',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:budgreen.131','budgreen','bud green',[68,148,74],'44944a',4494410],
['nbs-iscc-m:buff.71','buff','buff',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:buffalo.77','buffalo','buffalo',[130,102,68],'826644',8545860],
['nbs-iscc-m:bulgare.19','bulgare','bulgare',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:bunny.77','bunny','bunny',[130,102,68],'826644',8545860],
['nbs-iscc-m:bure.74','bure','bure',[153,101,21],'996515',10052885],
['nbs-iscc-m:bure.76','bure','bure',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:bure.77','bure','bure',[130,102,68],'826644',8545860],
['nbs-iscc-m:burel.58','burel','burel',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:burgundy.229','burgundy','burgundy',[80,64,77],'50404d',5259341],
['nbs-iscc-m:burgundy.230','burgundy','burgundy',[41,30,41],'291e29',2694697],
['nbs-iscc-m:burgundyviolet.218','burgundyviolet','burgundy violet',[135,86,146],'875692',8869522],
['nbs-iscc-m:burlwood.60','burlwood','burlwood',[149,128,112],'958070',9797744],
['nbs-iscc-m:burlwood.79','burlwood','burlwood',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:burma.54','burma','burma',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:burmesegold.37','burmesegold','burmese gold',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:burmeseruby.16','burmeseruby','burmese ruby',[114,47,55],'722f37',7483191],
['nbs-iscc-m:burnblue.184','burnblue','burn blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-m:burnblue.185','burnblue','burn blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:burnblue.202','burnblue','burn blue',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-m:burnblue.203','burnblue','burn blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-m:burnishedgold.72','burnishedgold','burnished gold',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:burnishedgold.74','burnishedgold','burnished gold',[153,101,21],'996515',10052885],
['nbs-iscc-m:burnishedgold.76','burnishedgold','burnished gold',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:burnishedgold.88','burnishedgold','burnished gold',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:burnishedgold.94','burnishedgold','burnished gold',[150,113,23],'967117',9859351],
['nbs-iscc-m:burnous.73','burnous','burnous',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:burnt.91','burnt','burnt',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:burntalmond.58','burntalmond','burnt almond',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:burntcarmine.13','burntcarmine','burnt carmine',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:burntcarmine.15','burntcarmine','burnt carmine',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:burntcoral.37','burntcoral','burnt coral',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:burntcrimsonlake.13','burntcrimsonlake','burnt crimson lake',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:burntcrimsonlake.15','burntcrimsonlake','burnt crimson lake',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:burntitalianearth.38','burntitalianearth','burnt italian earth',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:burntitalianochre.38','burntitalianochre','burnt italian ochre',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:burntlake.13','burntlake','burnt lake',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:burntlake.15','burntlake','burnt lake',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:burntochre.37','burntochre','burnt ochre',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:burntorange.37','burntorange','burnt orange',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:burntromanochre.50','burntromanochre','burnt roman ochre',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:burntrose.41','burntrose','burnt rose',[86,7,12],'56070c',5637900],
['nbs-iscc-m:burntrose.43','burntrose','burnt rose',[121,68,59],'79443b',7947323],
['nbs-iscc-m:burntrusset.16','burntrusset','burnt russet',[114,47,55],'722f37',7483191],
['nbs-iscc-m:burntrusset.43','burntrusset','burnt russet',[121,68,59],'79443b',7947323],
['nbs-iscc-m:burntsienna.38','burntsienna','burnt sienna',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:burntterreverte.58','burntterreverte','burnt terre verte',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:burntumber.58','burntumber','burnt umber',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:buttercupyellow.90','buttercupyellow','buttercup [yellow] ',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:butterfly.102','butterfly','butterfly',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:butternut.95','butternut','butternut',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:butterscotch.58','butterscotch','butterscotch',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:butterscotch.77','butterscotch','butterscotch',[130,102,68],'826644',8545860],
['nbs-iscc-m:byron.81','byron','byron',[72,60,50],'483c32',4734002],
['nbs-iscc-m:byzantine.254','byzantine','byzantine',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:byzantium.238','byzantium','byzantium',[112,41,99],'702963',7350627],
['nbs-iscc-m:byzantium.242','byzantium','byzantium',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:cabaret.144','cabaret','cabaret',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:cabbagegreen.155','cabbagegreen','cabbage green',[125,137,132],'7d8984',8227204],
['nbs-iscc-m:cacao.77','cacao','cacao',[130,102,68],'826644',8545860],
['nbs-iscc-m:cacaobrown.55','cacaobrown','cacao brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:cacha.58','cacha','cacha',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cachou.58','cachou','cachou',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cactus.137','cactus','cactus',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:cadet.186','cadet','cadet',[83,104,120],'536878',5466232],
['nbs-iscc-m:cadetblue.182','cadetblue','cadet blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:cadetgrey.191','cadetgrey','cadet grey ',[129,135,139],'81878b',8488843],
['nbs-iscc-m:cadmiumcarmine.12','cadmiumcarmine','cadmium carmine',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:cadmiumgreen.141','cadmiumgreen','cadmium green',[0,121,89],'007959',31065],
['nbs-iscc-m:cadmiumlemon.97','cadmiumlemon','cadmium lemon',[220,211,0],'dcd300',14471936],
['nbs-iscc-m:cadmiumlemon.98','cadmiumlemon','cadmium lemon',[233,228,80],'e9e450',15328336],
['nbs-iscc-m:cadmiumorange.50','cadmiumorange','cadmium orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:cadmiumpurple.13','cadmiumpurple','cadmium purple',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:cadmiumpurple.15','cadmiumpurple','cadmium purple',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:cadmiumvermilion.12','cadmiumvermilion','cadmium vermilion',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:cadmiumvermilion.15','cadmiumvermilion','cadmium vermilion',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:cadmiumyellow.50','cadmiumyellow','cadmium yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:cadmiumyellow.53','cadmiumyellow','cadmium yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-m:cadmiumyellow.68','cadmiumyellow','cadmium yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:cadmiumyellow.71','cadmiumyellow','cadmium yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:caenstone.73','caenstone','caen stone',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:caeruleum.167','caeruleum','caeruleum',[0,133,161],'0085a1',34209],
['nbs-iscc-m:caeruleum.176','caeruleum','caeruleum',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:cafaulait.45','cafaulait','caf-au-lait',[151,127,115],'977f73',9928563],
['nbs-iscc-m:cafaulait.57','cafaulait','caf-au-lait',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:cafaulait.60','cafaulait','caf-au-lait',[149,128,112],'958070',9797744],
['nbs-iscc-m:cafcrme.57','cafcrme','caf crme',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:cafcrme.58','cafcrme','caf crme',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cafcrme.77','cafcrme','caf crme',[130,102,68],'826644',8545860],
['nbs-iscc-m:cafnoir.81','cafnoir','caf noir',[72,60,50],'483c32',4734002],
['nbs-iscc-m:cairo.179','cairo','cairo',[0,65,106],'00416a',16746],
['nbs-iscc-m:calabash.77','calabash','calabash',[130,102,68],'826644',8545860],
['nbs-iscc-m:calabash.94','calabash','calabash',[150,113,23],'967117',9859351],
['nbs-iscc-m:calamineblue.172','calamineblue','calamine blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:caldera.43','caldera','caldera',[121,68,59],'79443b',7947323],
['nbs-iscc-m:caledonianbrown.55','caledonianbrown','caledonian brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:caliaturwood.15','caliaturwood','caliaturwood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:californiacolor.58','californiacolor','california color',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:californiagreen.91','californiagreen','california green',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:callagreen.120','callagreen','calla green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:callistegreen.117','callistegreen','calliste green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-m:callistegreen.120','callistegreen','calliste green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:cambridgeblue.163','cambridgeblue','cambridge blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:cambridgered.15','cambridgered','cambridge red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:cambridgered.19','cambridgered','cambridge red',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:camel.64','camel','camel',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:camellia.12','camellia','camellia ',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:camellia.15','camellia','camellia ',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:camelshairorcamelshair.80','camelshairorcamelshair','camels hair or camel\'s hair',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:cameoblue.163','cameoblue','cameo blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:cameobrown.19','cameobrown','cameo brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:cameobrown.42','cameobrown','cameo brown',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:cameobrown.43','cameobrown','cameo brown',[121,68,59],'79443b',7947323],
['nbs-iscc-m:cameobrown.46','cameobrown','cameo brown',[103,76,71],'674c47',6769735],
['nbs-iscc-m:cameogreen.135','cameogreen','cameo green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:cameogreen.144','cameogreen','cameo green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:cameogreen.149','cameogreen','cameo green',[141,163,153],'8da399',9282457],
['nbs-iscc-m:cameopink.253','cameopink','cameo pink',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-m:cameoyellow.89','cameoyellow','cameo yellow',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:campanula.207','campanula','campanula',[96,78,151],'604e97',6311575],
['nbs-iscc-m:campanula.211','campanula','campanula',[96,78,129],'604e81',6311553],
['nbs-iscc-m:campanulablue.199','campanulablue','campanula blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:campanulablue.195','campanulablue','campanula blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-m:campanulablue.206','campanulablue','campanula blue',[126,115,184],'7e73b8',8287160],
['nbs-iscc-m:campanulapurple.245','campanulapurple','campanula purple',[131,100,121],'836479',8610937],
['nbs-iscc-m:campanulaviolet.245','campanulaviolet','campanula violet',[131,100,121],'836479',8610937],
['nbs-iscc-m:camrier.224','camrier','camrier',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:camwood.15','camwood','camwood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:canard.165','canard','canard',[0,75,73],'004b49',19273],
['nbs-iscc-m:canard.174','canard','canard',[0,73,88],'004958',18776],
['nbs-iscc-m:canarybirdgreen.88','canarybirdgreen','canary-bird green',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:canarygreen.88','canarygreen','canary green',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:canaryyellow.87','canaryyellow','canary [yellow] ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:candida.261','candida','candida',[175,134,142],'af868e',11503246],
['nbs-iscc-m:candypink.3','candypink','candy pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:candypink.26','candypink','candy pink',[248,131,121],'f88379',16286585],
['nbs-iscc-m:candypink.27','candypink','candy pink',[230,103,97],'e66761',15099745],
['nbs-iscc-m:canna.37','canna','canna',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:canna.38','canna','canna',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:canna.39','canna','canna',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:canna.43','canna','canna',[121,68,59],'79443b',7947323],
['nbs-iscc-m:cannon.152','cannon','cannon',[26,36,33],'1a2421',1713185],
['nbs-iscc-m:canterbury.207','canterbury','canterbury',[96,78,151],'604e97',6311575],
['nbs-iscc-m:cantonblue.183','cantonblue','canton [blue] ',[0,48,78],'00304e',12366],
['nbs-iscc-m:cantonjade.164','cantonjade','canton jade',[49,120,115],'317873',3242099],
['nbs-iscc-m:canyon.16','canyon','canyon',[114,47,55],'722f37',7483191],
['nbs-iscc-m:canyon.259','canyon','canyon',[103,49,71],'673147',6762823],
['nbs-iscc-m:cappagh.58','cappagh','cappagh',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cappahbrown.58','cappahbrown','cappah brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:capri.169','capri','capri ',[0,119,145],'007791',30609],
['nbs-iscc-m:caprice.236','caprice','caprice',[135,0,116],'870074',8847476],
['nbs-iscc-m:caprice.226','caprice','caprice',[214,202,221],'d6cadd',14076637],
['nbs-iscc-m:capucine.87','capucine','capucine ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:capucinebuff.73','capucinebuff','capucine buff ',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:capucinelake.26','capucinelake','capucine lake ',[248,131,121],'f88379',16286585],
['nbs-iscc-m:capucinelake.37','capucinelake','capucine lake ',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:capucinelake.50','capucinelake','capucine lake ',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:capucinelake.53','capucinelake','capucine lake ',[217,144,88],'d99058',14258264],
['nbs-iscc-m:capucinemadder.53','capucinemadder','capucine madder ',[217,144,88],'d99058',14258264],
['nbs-iscc-m:capucineorange.53','capucineorange','capucine orange ',[217,144,88],'d99058',14258264],
['nbs-iscc-m:capucinered.25','capucinered','capucine red ',[255,183,165],'ffb7a5',16758693],
['nbs-iscc-m:capucinered.35','capucinered','capucine red ',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:capucineyellow.53','capucineyellow','capucine yellow ',[217,144,88],'d99058',14258264],
['nbs-iscc-m:caputmortuum.43','caputmortuum','caput mortuum',[121,68,59],'79443b',7947323],
['nbs-iscc-m:caraibe.58','caraibe','caraibe',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:caraibe.61','caraibe','caraibe',[99,81,71],'635147',6508871],
['nbs-iscc-m:caraibe.77','caraibe','caraibe',[130,102,68],'826644',8545860],
['nbs-iscc-m:caraibe.80','caraibe','caraibe',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:caramel.54','caramel','caramel',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:caramel.57','caramel','caramel',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:carbuncle.47','carbuncle','carbuncle',[67,48,46],'43302e',4403246],
['nbs-iscc-m:cardinalred.11','cardinalred','cardinal [red] ',[190,0,50],'be0032',12451890],
['nbs-iscc-m:carmine.11','carmine','carmine',[190,0,50],'be0032',12451890],
['nbs-iscc-m:carmine.12','carmine','carmine',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:carminelake.11','carminelake','carmine lake',[190,0,50],'be0032',12451890],
['nbs-iscc-m:carminelake.12','carminelake','carmine lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:carnation.89','carnation','carnation',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:carnation.90','carnation','carnation',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:carnationred.15','carnationred','carnation [red] ',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:carnationrose.2','carnationrose','carnation rose',[234,147,153],'ea9399',15373209],
['nbs-iscc-m:carnelian.39','carnelian','carnelian',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:carnelianred.37','carnelianred','carnelian red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:carnivalred.12','carnivalred','carnival red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:carobbrown.58','carobbrown','carob brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:caroubier.58','caroubier','caroubier',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:carraragreen.146','carraragreen','carrara green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:carrotred.50','carrotred','carrot red',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:carrotred.53','carrotred','carrot red',[217,144,88],'d99058',14258264],
['nbs-iscc-m:carthamusred.11','carthamusred','carthamus red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:carthamusrose.11','carthamusrose','carthamus rose',[190,0,50],'be0032',12451890],
['nbs-iscc-m:cartouche.58','cartouche','cartouche',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cartridgebuff.89','cartridgebuff','cartridge buff',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:cartridgebuff.90','cartridgebuff','cartridge buff',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:cascade.163','cascade','cascade',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:cashew.58','cashew','cashew',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cashewlake.58','cashewlake','cashew lake',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cashewnut.61','cashewnut','cashew nut',[99,81,71],'635147',6508871],
['nbs-iscc-m:cashoo.58','cashoo','cashoo',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:casinopink.255','casinopink','casino pink',[179,68,108],'b3446c',11748460],
['nbs-iscc-m:casselbrown.58','casselbrown','cassel brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:casselearth.58','casselearth','cassel earth',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:casselyellow.83','casselyellow','cassel yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:casselyellow.84','casselyellow','cassel yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:casselyellow.86','casselyellow','cassel yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:casselyellow.87','casselyellow','cassel yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:casserole.43','casserole','casserole',[121,68,59],'79443b',7947323],
['nbs-iscc-m:castaneous.61','castaneous','castaneous',[99,81,71],'635147',6508871],
['nbs-iscc-m:castellon.43','castellon','castellon',[121,68,59],'79443b',7947323],
['nbs-iscc-m:castileearth.58','castileearth','castile earth',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:castilianbrown.43','castilianbrown','castilian brown',[121,68,59],'79443b',7947323],
['nbs-iscc-m:castilianred.11','castilianred','castilian red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:castor.64','castor','castor',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:castorgrey.156','castorgrey','castor grey ',[78,87,85],'4e5755',5134165],
['nbs-iscc-m:catalpa.228','catalpa','catalpa',[121,104,120],'796878',7956600],
['nbs-iscc-m:catawba.17','catawba','catawba',[63,23,40],'3f1728',4134696],
['nbs-iscc-m:catawba.21','catawba','catawba',[46,29,33],'2e1d21',3022113],
['nbs-iscc-m:cate.58','cate','cate',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:catechu.58','catechu','catechu',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cathay.54','cathay','cathay',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:cathay.74','cathay','cathay',[153,101,21],'996515',10052885],
['nbs-iscc-m:cathedral.212','cathedral','cathedral',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:cathedralblue.182','cathedralblue','cathedral blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:cathedralblue.200','cathedralblue','cathedral blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:cattail.81','cattail','cattail',[72,60,50],'483c32',4734002],
['nbs-iscc-m:cattleya.223','cattleya','cattleya',[134,96,142],'86608e',8806542],
['nbs-iscc-m:caucasia.80','caucasia','caucasia',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:cauldron.16','cauldron','cauldron',[114,47,55],'722f37',7483191],
['nbs-iscc-m:cauldron.40','cauldron','cauldron',[136,45,23],'882d17',8924439],
['nbs-iscc-m:cauldron.43','cauldron','cauldron',[121,68,59],'79443b',7947323],
['nbs-iscc-m:cavalry.71','cavalry','cavalry',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:cedar.46','cedar','cedar',[103,76,71],'674c47',6769735],
['nbs-iscc-m:cedarbark.19','cedarbark','cedarbark',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:cedarbark.43','cedarbark','cedarbark',[121,68,59],'79443b',7947323],
['nbs-iscc-m:cedargreen.125','cedargreen','cedar green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:cedarwood.43','cedarwood','cedar wood',[121,68,59],'79443b',7947323],
['nbs-iscc-m:cedre.125','cedre','cedre',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:celadongreen.122','celadongreen','celadon green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:celandinegreen.150','celandinegreen','celandine green',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:celest.186','celest','celest',[83,104,120],'536878',5466232],
['nbs-iscc-m:celest.203','celest','celest',[140,146,172],'8c92ac',9212588],
['nbs-iscc-m:celestial.182','celestial','celestial',[67,107,149],'436b95',4418453],
['nbs-iscc-m:celestialblue.185','celestialblue','celestial blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:cellini.236','cellini','cellini',[135,0,116],'870074',8847476],
['nbs-iscc-m:cellini.254','cellini','cellini',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:cement.265','cement','cement',[132,132,130],'848482',8684674],
['nbs-iscc-m:cendrillon.151','cendrillon','cendrillon',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:cendrillon.192','cendrillon','cendrillon',[81,88,94],'51585e',5331038],
['nbs-iscc-m:centennialbrown.74','centennialbrown','centennial brown',[153,101,21],'996515',10052885],
['nbs-iscc-m:centennialbrown.76','centennialbrown','centennial brown',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:centennialbrown.77','centennialbrown','centennial brown',[130,102,68],'826644',8545860],
['nbs-iscc-m:centreblue.182','centreblue','centre blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:ceramic.182','ceramic','ceramic',[67,107,149],'436b95',4418453],
['nbs-iscc-m:ceres.53','ceres','ceres',[217,144,88],'d99058',14258264],
['nbs-iscc-m:cerise.15','cerise','cerise',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:cerrogreen.120','cerrogreen','cerro green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:cerrogreen.125','cerrogreen','cerro green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:certosa.103','certosa','certosa',[152,148,62],'98943e',9999422],
['nbs-iscc-m:certosa.106','certosa','certosa',[134,126,54],'867e36',8814134],
['nbs-iscc-m:certosa.120','certosa','certosa',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:cerulean.181','cerulean','cerulean',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:cerulean.199','cerulean','cerulean',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:ceruleanblue.167','ceruleanblue','cerulean blue',[0,133,161],'0085a1',34209],
['nbs-iscc-m:ceruleanblue.176','ceruleanblue','cerulean blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:ceylonblue.174','ceylonblue','ceylon blue',[0,73,88],'004958',18776],
['nbs-iscc-m:chaeturablack.267','chaeturablack','chaetura black',[34,34,34],'222222',2236962],
['nbs-iscc-m:chaeturadrab.266','chaeturadrab','chaetura drab',[85,85,85],'555555',5592405],
['nbs-iscc-m:chalcedonyyellow.101','chalcedonyyellow','chalcedony yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-m:chaletred.19','chaletred','chalet red',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:chamois.90','chamois','chamois',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:chamoisskin.90','chamoisskin','chamois skin',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:chamoline.72','chamoline','chamoline',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:chamoline.74','chamoline','chamoline',[153,101,21],'996515',10052885],
['nbs-iscc-m:chamoline.76','chamoline','chamoline',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:chamoline.77','chamoline','chamoline',[130,102,68],'826644',8545860],
['nbs-iscc-m:chamoline.88','chamoline','chamoline',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:chamoline.91','chamoline','chamoline',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:chamoline.94','chamoline','chamoline',[150,113,23],'967117',9859351],
['nbs-iscc-m:champagne.73','champagne','champagne',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:champagne.79','champagne','champagne',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:champagne.90','champagne','champagne',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:champagne.93','champagne','champagne',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:chantilly.144','chantilly','chantilly',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:charcoalgrey.266','charcoalgrey','charcoal grey',[85,85,85],'555555',5592405],
['nbs-iscc-m:chardon.259','chardon','chardon',[103,49,71],'673147',6762823],
['nbs-iscc-m:charlesx.259','charlesx','charles x',[103,49,71],'673147',6762823],
['nbs-iscc-m:chartreuse.98','chartreuse','chartreuse',[233,228,80],'e9e450',15328336],
['nbs-iscc-m:chartreuse.99','chartreuse','chartreuse',[190,183,46],'beb72e',12498734],
['nbs-iscc-m:chartreuse.101','chartreuse','chartreuse',[234,230,121],'eae679',15394425],
['nbs-iscc-m:chartreuse.102','chartreuse','chartreuse',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:chartreusegreen.103','chartreusegreen','chartreuse green',[152,148,62],'98943e',9999422],
['nbs-iscc-m:chartreuseyellow.101','chartreuseyellow','chartreuse yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-m:chartreuseyellow.102','chartreuseyellow','chartreuse yellow',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:chasseur.138','chasseur','chasseur',[23,54,32],'173620',1521184],
['nbs-iscc-m:chatemuc.15','chatemuc','chatemuc',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:chatenaypink.28','chatenaypink','chatenay pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-m:chatenaypink.29','chatenaypink','chatenay pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:chaudron.37','chaudron','chaudron',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:chaudron.38','chaudron','chaudron',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:chaudron.39','chaudron','chaudron',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:chaudron.43','chaudron','chaudron',[121,68,59],'79443b',7947323],
['nbs-iscc-m:checkerberry.11','checkerberry','checkerberry',[190,0,50],'be0032',12451890],
['nbs-iscc-m:chemicblue.174','chemicblue','chemic blue ',[0,73,88],'004958',18776],
['nbs-iscc-m:chemicgreen.164','chemicgreen','chemic green ',[49,120,115],'317873',3242099],
['nbs-iscc-m:cherokee.37','cherokee','cherokee',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:cherry.34','cherry','cherry',[226,88,34],'e25822',14833698],
['nbs-iscc-m:cherrybloom.26','cherrybloom','cherry bloom',[248,131,121],'f88379',16286585],
['nbs-iscc-m:cherryblossom.15','cherryblossom','cherry blossom',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:cherryred.34','cherryred','cherry red',[226,88,34],'e25822',14833698],
['nbs-iscc-m:cherub.29','cherub','cherub',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:chessyliteblue.182','chessyliteblue','chessylite blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:chestnut.61','chestnut','chestnut',[99,81,71],'635147',6508871],
['nbs-iscc-m:chevreuse.19','chevreuse','chevreuse',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:chevreuse.43','chevreuse','chevreuse',[121,68,59],'79443b',7947323],
['nbs-iscc-m:chianti.13','chianti','chianti',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:chickadeegray.113','chickadeegray','chickadee gray',[87,85,76],'57554c',5723468],
['nbs-iscc-m:chicle.61','chicle','chicle',[99,81,71],'635147',6508871],
['nbs-iscc-m:chicory.203','chicory','chicory',[140,146,172],'8c92ac',9212588],
['nbs-iscc-m:chicoryblue.121','chicoryblue','chicory blue',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:chicoryblue.203','chicoryblue','chicory blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-m:chimneyred.13','chimneyred','chimney red',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:chimneyred.15','chimneyred','chimney red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:chinablue.182','chinablue','china blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:chinablue.186','chinablue','china blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:chinarose.258','chinarose','china rose',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:chinchilla.265','chinchilla','chinchilla',[132,132,130],'848482',8684674],
['nbs-iscc-m:chinchinblue.186','chinchinblue','chin-chin blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:chinesegold.54','chinesegold','chinese gold',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:chinesegreen.135','chinesegreen','chinese green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:chinesegreen.144','chinesegreen','chinese green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:chinesegreen.146','chinesegreen','chinese green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:chinesegreen.165','chinesegreen','chinese green',[0,75,73],'004b49',19273],
['nbs-iscc-m:chineselake.11','chineselake','chinese lake',[190,0,50],'be0032',12451890],
['nbs-iscc-m:chineselake.12','chineselake','chinese lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:chineseorange.25','chineseorange','chinese orange',[255,183,165],'ffb7a5',16758693],
['nbs-iscc-m:chineseorange.35','chineseorange','chinese orange',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:chinesered.36','chinesered','chinese red',[170,56,30],'aa381e',11155486],
['nbs-iscc-m:chinesered.38','chinesered','chinese red',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:chinesered.40','chinesered','chinese red',[136,45,23],'882d17',8924439],
['nbs-iscc-m:chineserouge.11','chineserouge','chinese rouge',[190,0,50],'be0032',12451890],
['nbs-iscc-m:chinesevermilion.11','chinesevermilion','chinese vermilion ',[190,0,50],'be0032',12451890],
['nbs-iscc-m:chineseviolet.223','chineseviolet','chinese violet',[134,96,142],'86608e',8806542],
['nbs-iscc-m:chineseyellow.83','chineseyellow','chinese yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:chineseyellow.84','chineseyellow','chinese yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:chineseyellow.86','chineseyellow','chinese yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:chineseyellow.87','chineseyellow','chinese yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:chineseyellow.71','chineseyellow','chinese yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:ching.176','ching','ch\'ing',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:chinook.38','chinook','chinook',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:chip.109','chip','chip',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:chip.112','chip','chip',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:chipmunk.74','chipmunk','chipmunk',[153,101,21],'996515',10052885],
['nbs-iscc-m:chippendale.62','chippendale','chippendale',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:chiswick.183','chiswick','chiswick',[0,48,78],'00304e',12366],
['nbs-iscc-m:chocolate.62','chocolate','chocolate',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:chocolatebrown.62','chocolatebrown','chocolate brown',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:chocolatemaroon.16','chocolatemaroon','chocolate maroon',[114,47,55],'722f37',7483191],
['nbs-iscc-m:chromecitron.86','chromecitron','chrome citron',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:chromegreen.141','chromegreen','chrome green',[0,121,89],'007959',31065],
['nbs-iscc-m:chromegreen.145','chromegreen','chrome green',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:chromelemon.86','chromelemon','chrome lemon',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:chromeorange.34','chromeorange','chrome orange',[226,88,34],'e25822',14833698],
['nbs-iscc-m:chromeorange.48','chromeorange','chrome orange',[243,132,0],'f38400',15959040],
['nbs-iscc-m:chromeprimrose.98','chromeprimrose','chrome primrose',[233,228,80],'e9e450',15328336],
['nbs-iscc-m:chromeprimrose.101','chromeprimrose','chrome primrose',[234,230,121],'eae679',15394425],
['nbs-iscc-m:chromescarlet.35','chromescarlet','chrome scarlet',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:chromiumgreen.120','chromiumgreen','chromium green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:chromiumoxide.136','chromiumoxide','chromium oxide ',[103,146,103],'679267',6787687],
['nbs-iscc-m:chrysanthemum.13','chrysanthemum','chrysanthemum',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:chrysanthemum.16','chrysanthemum','chrysanthemum',[114,47,55],'722f37',7483191],
['nbs-iscc-m:chrysocollo.136','chrysocollo','chrysocollo',[103,146,103],'679267',6787687],
['nbs-iscc-m:chrysolitegreen.102','chrysolitegreen','chrysolite green',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:chrysolitegreen.105','chrysolitegreen','chrysolite green',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:chrysolitegreen.120','chrysolitegreen','chrysolite green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:chrysopraisegreen.140','chrysopraisegreen','chrysopraise [green] ',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:chrysopraisegreen.144','chrysopraisegreen','chrysopraise [green] ',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:chukkerbrown.61','chukkerbrown','chukker brown',[99,81,71],'635147',6508871],
['nbs-iscc-m:chukkerbrown.80','chukkerbrown','chukker brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:chutney.43','chutney','chutney',[121,68,59],'79443b',7947323],
['nbs-iscc-m:cigarette.58','cigarette','cigarette',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cindergrey.233','cindergrey','cinder [grey] ',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:cineraria.254','cineraria','cineraria',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:cinereous.264','cinereous','cinereous',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:cinnamon.57','cinnamon','cinnamon',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:cinnamon.76','cinnamon','cinnamon',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:cinnamonbrown.58','cinnamonbrown','cinnamon brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cinnamonbrown.77','cinnamonbrown','cinnamon brown',[130,102,68],'826644',8545860],
['nbs-iscc-m:cinnamonpink.42','cinnamonpink','cinnamon pink',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:circassian.58','circassian','circassian',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:citrine.106','citrine','citrine',[134,126,54],'867e36',8814134],
['nbs-iscc-m:citronelle.98','citronelle','citronelle',[233,228,80],'e9e450',15328336],
['nbs-iscc-m:citronelle.99','citronelle','citronelle',[190,183,46],'beb72e',12498734],
['nbs-iscc-m:citronelle.101','citronelle','citronelle',[234,230,121],'eae679',15394425],
['nbs-iscc-m:citronelle.102','citronelle','citronelle',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:citrongreen.102','citrongreen','citron green',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:citrongreen.105','citrongreen','citron green',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:citronyellow.86','citronyellow','citron [yellow] ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:citrus.101','citrus','citrus',[234,230,121],'eae679',15394425],
['nbs-iscc-m:civettegreen.137','civettegreen','civette green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:clairdelune.149','clairdelune','clair de lune',[141,163,153],'8da399',9282457],
['nbs-iscc-m:claretcup.262','claretcup','claret cup',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:claretlees.259','claretlees','claret lees',[103,49,71],'673147',6762823],
['nbs-iscc-m:claretred.15','claretred','claret [red] ',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:claver.241','claver','claver',[145,92,131],'915c83',9526403],
['nbs-iscc-m:clay.74','clay','clay',[153,101,21],'996515',10052885],
['nbs-iscc-m:clay.76','clay','clay',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:clay.77','clay','clay',[130,102,68],'826644',8545860],
['nbs-iscc-m:claybank.54','claybank','clay bank',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:claydrab.77','claydrab','clay drab',[130,102,68],'826644',8545860],
['nbs-iscc-m:claydrab.94','claydrab','clay drab',[150,113,23],'967117',9859351],
['nbs-iscc-m:clematis.207','clematis','clematis',[96,78,151],'604e97',6311575],
['nbs-iscc-m:cleopatra.176','cleopatra','cleopatra',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:clochette.200','clochette','clochette',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:clochette.211','clochette','clochette',[96,78,129],'604e81',6311553],
['nbs-iscc-m:cloisonn.176','cloisonn','cloisonn',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:cloisonn.194','cloisonn','cloisonn',[48,38,122],'30267a',3155578],
['nbs-iscc-m:cloud.186','cloud','cloud',[83,104,120],'536878',5466232],
['nbs-iscc-m:cloudgrey.233','cloudgrey','cloud grey ',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:cloudgrey.265','cloudgrey','cloud grey ',[132,132,130],'848482',8684674],
['nbs-iscc-m:cloudyamber.90','cloudyamber','cloudy amber',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:cloudyamber.91','cloudyamber','cloudy amber',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:clove.58','clove','clove',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:clovebrown.64','clovebrown','clove brown',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:clovepink.16','clovepink','clove pink',[114,47,55],'722f37',7483191],
['nbs-iscc-m:clover.241','clover','clover',[145,92,131],'915c83',9526403],
['nbs-iscc-m:cobaltblue.169','cobaltblue','cobalt blue',[0,119,145],'007791',30609],
['nbs-iscc-m:cobaltglass.179','cobaltglass','cobalt glass',[0,65,106],'00416a',16746],
['nbs-iscc-m:cobaltglass.182','cobaltglass','cobalt glass',[67,107,149],'436b95',4418453],
['nbs-iscc-m:cobaltgreen.136','cobaltgreen','cobalt green',[103,146,103],'679267',6787687],
['nbs-iscc-m:cobaltred.254','cobaltred','cobalt red',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:cobaltred.255','cobaltred','cobalt red',[179,68,108],'b3446c',11748460],
['nbs-iscc-m:cobaltultramarine.169','cobaltultramarine','cobalt ultramarine ',[0,119,145],'007791',30609],
['nbs-iscc-m:cobaltviolet.223','cobaltviolet','cobalt violet',[134,96,142],'86608e',8806542],
['nbs-iscc-m:cobaltyellow.83','cobaltyellow','cobalt yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:cobaltyellow.84','cobaltyellow','cobalt yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:cobblestone.266','cobblestone','cobblestone',[85,85,85],'555555',5592405],
['nbs-iscc-m:cobweb.79','cobweb','cobweb',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:coccineous.11','coccineous','coccineous',[190,0,50],'be0032',12451890],
['nbs-iscc-m:cocher.95','cocher','cocher',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:cochin.58','cochin','cochin',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cochineal.11','cochineal','cochineal',[190,0,50],'be0032',12451890],
['nbs-iscc-m:cockatoo.86','cockatoo','cockatoo',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:cocoa.58','cocoa','cocoa',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cocoabrown.58','cocoabrown','cocoa brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cocobala.19','cocobala','cocobala',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:cocobala.46','cocobala','cocobala',[103,76,71],'674c47',6769735],
['nbs-iscc-m:coconut.61','coconut','coconut',[99,81,71],'635147',6508871],
['nbs-iscc-m:coconutbrown.58','coconutbrown','coconut brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:codgrey.110','codgrey','cod grey',[91,88,66],'5b5842',5986370],
['nbs-iscc-m:codgrey.113','codgrey','cod grey',[87,85,76],'57554c',5723468],
['nbs-iscc-m:coeruleum.167','coeruleum','coeruleum',[0,133,161],'0085a1',34209],
['nbs-iscc-m:coeruleum.176','coeruleum','coeruleum',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:coffee.58','coffee','coffee',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cognac.58','cognac','cognac',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:colcothar.43','colcothar','colcothar',[121,68,59],'79443b',7947323],
['nbs-iscc-m:colewortgreen.155','colewortgreen','colewort green',[125,137,132],'7d8984',8227204],
['nbs-iscc-m:colibri.164','colibri','colibri',[49,120,115],'317873',3242099],
['nbs-iscc-m:collie.80','collie','collie',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:colognebrown.58','colognebrown','cologne brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cologneearth.58','cologneearth','cologne earth',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cologneyellow.84','cologneyellow','cologne yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:colonial.179','colonial','colonial',[0,65,106],'00416a',16746],
['nbs-iscc-m:colonial.183','colonial','colonial',[0,48,78],'00304e',12366],
['nbs-iscc-m:colonial.197','colonial','colonial',[39,36,88],'272458',2565208],
['nbs-iscc-m:colonial.200','colonial','colonial',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:colonialbuff.86','colonialbuff','colonial buff',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:colonialbuff.89','colonialbuff','colonial buff',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:colonialyellow.87','colonialyellow','colonial yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:columbia.178','columbia','columbia',[0,103,165],'0067a5',26533],
['nbs-iscc-m:columbiablue.185','columbiablue','columbia blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:columbianred.43','columbianred','columbian red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:columbineblue.200','columbineblue','columbine blue ',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:columbineblue.211','columbineblue','columbine blue ',[96,78,129],'604e81',6311553],
['nbs-iscc-m:columbinered.259','columbinered','columbine red ',[103,49,71],'673147',6762823],
['nbs-iscc-m:comet.183','comet','comet',[0,48,78],'00304e',12366],
['nbs-iscc-m:commelinablue.182','commelinablue','commelina blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:commodore.194','commodore','commodore',[48,38,122],'30267a',3155578],
['nbs-iscc-m:como.173','como','como',[54,117,136],'367588',3569032],
['nbs-iscc-m:conchshell.26','conchshell','conch shell',[248,131,121],'f88379',16286585],
['nbs-iscc-m:concord.212','concord','concord',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:condor.77','condor','condor',[130,102,68],'826644',8545860],
['nbs-iscc-m:confetti.3','confetti','confetti',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:confetti.15','confetti','confetti',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:congobrown.81','congobrown','congo [brown] ',[72,60,50],'483c32',4734002],
['nbs-iscc-m:congopink.39','congopink','congo pink',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:continentalblue.229','continentalblue','continental blue',[80,64,77],'50404d',5259341],
['nbs-iscc-m:continentalblue.235','continentalblue','continental blue',[36,33,36],'242124',2367780],
['nbs-iscc-m:cookie.58','cookie','cookie',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:coolie.200','coolie','coolie',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:coolie.201','coolie','coolie',[37,36,64],'252440',2434112],
['nbs-iscc-m:copenhagenblue.186','copenhagenblue','copenhagen [blue] ',[83,104,120],'536878',5466232],
['nbs-iscc-m:copenhagenblue.204','copenhagenblue','copenhagen [blue] ',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:copper.39','copper','copper',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:copperblue.182','copperblue','copper blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:copperbrown.55','copperbrown','copper brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:coppergreen.136','coppergreen','copper green',[103,146,103],'679267',6787687],
['nbs-iscc-m:copperleaf.43','copperleaf','copperleaf',[121,68,59],'79443b',7947323],
['nbs-iscc-m:copperlustre.55','copperlustre','copper lustre',[128,70,27],'80461b',8406555],
['nbs-iscc-m:copperred.39','copperred','copper red',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:copperrose.15','copperrose','copper rose',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:copperyellow.87','copperyellow','copper yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:copra.23','copra','copra',[92,80,79],'5c504f',6049871],
['nbs-iscc-m:coptic.43','coptic','coptic',[121,68,59],'79443b',7947323],
['nbs-iscc-m:coquelicot.11','coquelicot','coquelicot',[190,0,50],'be0032',12451890],
['nbs-iscc-m:coquelicot.34','coquelicot','coquelicot',[226,88,34],'e25822',14833698],
['nbs-iscc-m:coquette.26','coquette','coquette',[248,131,121],'f88379',16286585],
['nbs-iscc-m:coquette.29','coquette','coquette',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:coralbell.15','coralbell','coralbell',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:coralbell.37','coralbell','coralbell',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:coralblush.26','coralblush','coral blush',[248,131,121],'f88379',16286585],
['nbs-iscc-m:coralblush.27','coralblush','coral blush',[230,103,97],'e66761',15099745],
['nbs-iscc-m:coralblush.37','coralblush','coral blush',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:coralpink.29','coralpink','coral pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:coralred.3','coralred','coral [red] ',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:coralsands.29','coralsands','coral sands',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:coralsands.32','coralsands','coral sands',[199,173,163],'c7ada3',13086115],
['nbs-iscc-m:corcir.258','corcir','corcir',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:cordova.43','cordova','cordova',[121,68,59],'79443b',7947323],
['nbs-iscc-m:cordovan.81','cordovan','cordovan',[72,60,50],'483c32',4734002],
['nbs-iscc-m:cordovan.267','cordovan','cordovan',[34,34,34],'222222',2236962],
['nbs-iscc-m:corial.29','corial','corial',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:corial.42','corial','corial',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:corinth.262','corinth','corinth',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:corinthianpink.5','corinthianpink','corinthian pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:corinthianpink.18','corinthianpink','corinthian pink',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:corinthianpurple.259','corinthianpurple','corinthian purple',[103,49,71],'673147',6762823],
['nbs-iscc-m:corinthianpurple.262','corinthianpurple','corinthian purple',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:corinthianred.19','corinthianred','corinthian red',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:cork.57','cork','cork',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:cork.258','cork','cork',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:corker.258','corker','corker',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:corkur.258','corkur','corkur',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:corn.87','corn','corn',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:cornelianred.39','cornelianred','cornelian red',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:cornflowerblue.200','cornflowerblue','cornflower [blue] ',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:cornhusk.73','cornhusk','cornhusk',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:cornhusk.76','cornhusk','cornhusk',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:cornsilk.86','cornsilk','cornsilk',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:coromandel.43','coromandel','coromandel',[121,68,59],'79443b',7947323],
['nbs-iscc-m:coronation.242','coronation','coronation',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:coronation.259','coronation','coronation',[103,49,71],'673147',6762823],
['nbs-iscc-m:corsagegreen.144','corsagegreen','corsage green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:corsagegreen.163','corsagegreen','corsage green',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:corsair.183','corsair','corsair',[0,48,78],'00304e',12366],
['nbs-iscc-m:corsicanblue.200','corsicanblue','corsican blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:corsicanblue.201','corsicanblue','corsican blue',[37,36,64],'252440',2434112],
['nbs-iscc-m:corsicanblue.212','corsicanblue','corsican blue',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:corydalisgreen.122','corydalisgreen','corydalis green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:cosmos.251','cosmos','cosmos',[193,126,145],'c17e91',12680849],
['nbs-iscc-m:cossack.150','cossack','cossack',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:cossackgreen.137','cossackgreen','cossack green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:cossegreen.120','cossegreen','cosse green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:cotch.58','cotch','cotch',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cotingapurple.219','cotingapurple','cotinga purple',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:cotingapurple.224','cotingapurple','cotinga purple',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:cotingapurple.238','cotingapurple','cotinga purple',[112,41,99],'702963',7350627],
['nbs-iscc-m:cotingapurple.242','cotingapurple','cotinga purple',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:cotrine.72','cotrine','cotrine',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:cotrine.76','cotrine','cotrine',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:courgegreen.103','courgegreen','courge green',[152,148,62],'98943e',9999422],
['nbs-iscc-m:courgegreen.106','courgegreen','courge green',[134,126,54],'867e36',8814134],
['nbs-iscc-m:courgegreen.120','courgegreen','courge green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:cowboy.58','cowboy','cowboy',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cowboy.61','cowboy','cowboy',[99,81,71],'635147',6508871],
['nbs-iscc-m:cowboy.77','cowboy','cowboy',[130,102,68],'826644',8545860],
['nbs-iscc-m:cowboy.80','cowboy','cowboy',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:cowslip.50','cowslip','cowslip',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:cowslip.53','cowslip','cowslip',[217,144,88],'d99058',14258264],
['nbs-iscc-m:crabapple.37','crabapple','crabapple',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:cracker.76','cracker','cracker',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:crag.113','crag','crag',[87,85,76],'57554c',5723468],
['nbs-iscc-m:crane.233','crane','crane',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:crash.90','crash','crash',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:crash.91','crash','crash',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:crash.94','crash','crash',[150,113,23],'967117',9859351],
['nbs-iscc-m:crash.109','crash','crash',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:crawshayred.38','crawshayred','crawshay red',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:crawshayred.55','crawshayred','crawshay red',[128,70,27],'80461b',8406555],
['nbs-iscc-m:crayonblue.186','crayonblue','crayon blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:cream.89','cream','cream',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:creambeige.42','creambeige','cream beige',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:creambeige.57','creambeige','cream beige',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:creambeige.60','creambeige','cream beige',[149,128,112],'958070',9797744],
['nbs-iscc-m:creambuff.89','creambuff','cream buff',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:creambuff.90','creambuff','cream buff',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:creole.54','creole','creole',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:creole.57','creole','creole',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:cressgreen.120','cressgreen','cress green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:cresson.120','cresson','cresson',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:crevette.26','crevette','crevette',[248,131,121],'f88379',16286585],
['nbs-iscc-m:crimsonlake.15','crimsonlake','crimson lake',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:crimsonmadder.11','crimsonmadder','crimson madder',[190,0,50],'be0032',12451890],
['nbs-iscc-m:crimsonmaple.41','crimsonmaple','crimson maple',[86,7,12],'56070c',5637900],
['nbs-iscc-m:croceus.53','croceus','croceus',[217,144,88],'d99058',14258264],
['nbs-iscc-m:croceus.71','croceus','croceus',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:crocreal.53','crocreal','crocreal',[217,144,88],'d99058',14258264],
['nbs-iscc-m:crocreal.71','crocreal','crocreal',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:crocus.53','crocus','crocus',[217,144,88],'d99058',14258264],
['nbs-iscc-m:crocus.71','crocus','crocus',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:crocus.244','crocus','crocus',[170,138,158],'aa8a9e',11176606],
['nbs-iscc-m:crocus.245','crocus','crocus',[131,100,121],'836479',8610937],
['nbs-iscc-m:crocusmartis.43','crocusmartis','crocus martis',[121,68,59],'79443b',7947323],
['nbs-iscc-m:crostil.258','crostil','crostil',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:crotal.258','crotal','crotal',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:crottle.258','crottle','crottle',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:crowshayred.38','crowshayred','crowshay red',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:crowshayred.55','crowshayred','crowshay red',[128,70,27],'80461b',8406555],
['nbs-iscc-m:cruiser.266','cruiser','cruiser',[85,85,85],'555555',5592405],
['nbs-iscc-m:crushedberry.19','crushedberry','crushed berry',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:crushedstrawberry.3','crushedstrawberry','crushed strawberry',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:crushedviolets.228','crushedviolets','crushed violets',[121,104,120],'796878',7956600],
['nbs-iscc-m:crystalgrey.233','crystalgrey','crystal grey',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:crystalpalaceblue.183','crystalpalaceblue','crystal palace blue',[0,48,78],'00304e',12366],
['nbs-iscc-m:crystalpalacegreen.113','crystalpalacegreen','crystal palace green',[87,85,76],'57554c',5723468],
['nbs-iscc-m:crystalsofvenus.136','crystalsofvenus','crystals of venus',[103,146,103],'679267',6787687],
['nbs-iscc-m:cub.113','cub','cub',[87,85,76],'57554c',5723468],
['nbs-iscc-m:cuba.16','cuba','cuba',[114,47,55],'722f37',7483191],
['nbs-iscc-m:cuba.43','cuba','cuba',[121,68,59],'79443b',7947323],
['nbs-iscc-m:cubansand.79','cubansand','cuban sand',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:cudbear.258','cudbear','cudbear',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:cuir.76','cuir','cuir',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:cuissedenymphe.89','cuissedenymphe','cuisse de nymphe',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:cullenearth.58','cullenearth','cullen earth',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cupidpink.29','cupidpink','cupid pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:cupreous.86','cupreous','cupreous',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:curcuma.86','curcuma','curcuma',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:currantred.12','currantred','currant [red] ',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:cutch.58','cutch','cutch',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:cuyahogared.16','cuyahogared','cuyahoga red',[114,47,55],'722f37',7483191],
['nbs-iscc-m:cyanblue.164','cyanblue','cyan blue',[49,120,115],'317873',3242099],
['nbs-iscc-m:cyanblue.173','cyanblue','cyan blue',[54,117,136],'367588',3569032],
['nbs-iscc-m:cyanineblue.176','cyanineblue','cyanine blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:cyanineblue.200','cyanineblue','cyanine blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:cyanineblue.201','cyanineblue','cyanine blue',[37,36,64],'252440',2434112],
['nbs-iscc-m:cyclamen.243','cyclamen','cyclamen',[52,23,49],'341731',3413809],
['nbs-iscc-m:cygnet.259','cygnet','cygnet',[103,49,71],'673147',6762823],
['nbs-iscc-m:cypressgreen.125','cypressgreen','cypress [green] ',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:cypressgreen.131','cypressgreen','cypress green',[68,148,74],'44944a',4494410],
['nbs-iscc-m:cyprusearth.77','cyprusearth','cyprus earth',[130,102,68],'826644',8545860],
['nbs-iscc-m:cyprusearth.78','cyprusearth','cyprus earth',[75,54,33],'4b3621',4929057],
['nbs-iscc-m:cyprusumber.77','cyprusumber','cyprus umber',[130,102,68],'826644',8545860],
['nbs-iscc-m:cyprusumber.78','cyprusumber','cyprus umber',[75,54,33],'4b3621',4929057],
['nbs-iscc-m:daffodileyellow.71','daffodileyellow','daffodile [yellow] ',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:daffodilyellow.50','daffodilyellow','daffodil yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:daffodilyellow.53','daffodilyellow','daffodil yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-m:daffodilyellow.68','daffodilyellow','daffodil yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:daffodilyellow.71','daffodilyellow','daffodil yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:dagestan.183','dagestan','dagestan',[0,48,78],'00304e',12366],
['nbs-iscc-m:dahlia.223','dahlia','dahlia',[134,96,142],'86608e',8806542],
['nbs-iscc-m:dahliacarmine.259','dahliacarmine','dahlia carmine',[103,49,71],'673147',6762823],
['nbs-iscc-m:dahliamauve.241','dahliamauve','dahlia mauve',[145,92,131],'915c83',9526403],
['nbs-iscc-m:dahliapurple.259','dahliapurple','dahlia purple',[103,49,71],'673147',6762823],
['nbs-iscc-m:dahliapurple.260','dahliapurple','dahlia purple',[56,21,44],'38152c',3675436],
['nbs-iscc-m:damascen.211','damascen','damascen',[96,78,129],'604e81',6311553],
['nbs-iscc-m:damascen.212','damascen','damascen',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:damask.19','damask','damask',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:damonico.26','damonico','damonico',[248,131,121],'f88379',16286585],
['nbs-iscc-m:damonico.37','damonico','damonico',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:damonico.50','damonico','damonico',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:damonico.53','damonico','damonico',[217,144,88],'d99058',14258264],
['nbs-iscc-m:damson.211','damson','damson',[96,78,129],'604e81',6311553],
['nbs-iscc-m:damson.212','damson','damson',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:dana.50','dana','dana',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:dana.53','dana','dana',[217,144,88],'d99058',14258264],
['nbs-iscc-m:dana.186','dana','dana',[83,104,120],'536878',5466232],
['nbs-iscc-m:dandelion.83','dandelion','dandelion',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:dante.229','dante','dante',[80,64,77],'50404d',5259341],
['nbs-iscc-m:dante.242','dante','dante',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:dante.259','dante','dante',[103,49,71],'673147',6762823],
['nbs-iscc-m:danubegreen.137','danubegreen','danube green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:danubegreen.151','danubegreen','danube green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:daphne.182','daphne','daphne',[67,107,149],'436b95',4418453],
['nbs-iscc-m:daphnepink.262','daphnepink','daphne pink',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:daphnered.258','daphnered','daphne red',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:daphnered.262','daphnered','daphne red',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:darkbeaver.58','darkbeaver','dark beaver',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:darkbeaver.61','darkbeaver','dark beaver',[99,81,71],'635147',6508871],
['nbs-iscc-m:darkcardinal.16','darkcardinal','dark cardinal',[114,47,55],'722f37',7483191],
['nbs-iscc-m:darkgobelinblue.173','darkgobelinblue','dark gobelin blue ',[54,117,136],'367588',3569032],
['nbs-iscc-m:darkgobelinblue.186','darkgobelinblue','dark gobelin blue ',[83,104,120],'536878',5466232],
['nbs-iscc-m:darkwedgwoodblue.200','darkwedgwoodblue','dark wedgwood [blue] ',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:darkwedgwoodblue.204','darkwedgwoodblue','dark wedgwood [blue] ',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:date.81','date','date',[72,60,50],'483c32',4734002],
['nbs-iscc-m:datura.223','datura','datura',[134,96,142],'86608e',8806542],
['nbs-iscc-m:dauphine.18','dauphine','dauphine',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:davysgrey.266','davysgrey','davy\'s grey',[85,85,85],'555555',5592405],
['nbs-iscc-m:dawn.5','dawn','dawn',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:dawngrey.265','dawngrey','dawn grey ',[132,132,130],'848482',8684674],
['nbs-iscc-m:daybreak.210','daybreak','daybreak',[140,130,182],'8c82b6',9208502],
['nbs-iscc-m:daybreak.222','daybreak','daybreak',[182,149,192],'b695c0',11965888],
['nbs-iscc-m:daytona.63','daytona','daytona',[142,130,121],'8e8279',9339513],
['nbs-iscc-m:dbutantepink.5','dbutantepink','dbutante pink ',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:deadcarnations.258','deadcarnations','dead carnations',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:deadleaf.51','deadleaf','dead leaf',[190,101,22],'be6516',12477718],
['nbs-iscc-m:deadleaf.54','deadleaf','dead leaf',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:deadleaf.55','deadleaf','dead leaf',[128,70,27],'80461b',8406555],
['nbs-iscc-m:deauvillesand.45','deauvillesand','deauville sand',[151,127,115],'977f73',9928563],
['nbs-iscc-m:deauvillesand.60','deauvillesand','deauville sand',[149,128,112],'958070',9797744],
['nbs-iscc-m:deepbrunswickgreen.146','deepbrunswickgreen','deep brunswick green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:deepchromegreen.137','deepchromegreen','deep chrome green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:deepchromeyellow.71','deepchromeyellow','deep chrome yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:deepstone.88','deepstone','deep stone',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:deepstone.91','deepstone','deep stone',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:deepstone.94','deepstone','deep stone',[150,113,23],'967117',9859351],
['nbs-iscc-m:deer.80','deer','deer',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:delftblue.204','delftblue','delft blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:dellarobbia.196','dellarobbia','della robbia',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:delmonte.228','delmonte','del monte',[121,104,120],'796878',7956600],
['nbs-iscc-m:delphinium.177','delphinium','delphinium',[73,151,208],'4997d0',4822992],
['nbs-iscc-m:delphinium.178','delphinium','delphinium',[0,103,165],'0067a5',26533],
['nbs-iscc-m:delphinium.181','delphinium','delphinium',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:demedici.259','demedici','de\'medici ',[103,49,71],'673147',6762823],
['nbs-iscc-m:denmark.186','denmark','denmark',[83,104,120],'536878',5466232],
['nbs-iscc-m:denver.265','denver','denver',[132,132,130],'848482',8684674],
['nbs-iscc-m:derbyblue.212','derbyblue','derby blue',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:desert.76','desert','desert',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:devilsred.12','devilsred','devil\'s red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:devilsred.15','devilsred','devil\'s red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:devonbrown.19','devonbrown','devon [brown] ',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:deweyred.12','deweyred','dewey red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:deweyred.15','deweyred','dewey red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:dewkiss.150','dewkiss','dewkiss',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:diadem.227','diadem','diadem',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:diana.181','diana','diana',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:dianthus.15','dianthus','dianthus',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:dianthus.19','dianthus','dianthus',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:diavolo.19','diavolo','diavolo',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:digitalis.228','digitalis','digitalis',[121,104,120],'796878',7956600],
['nbs-iscc-m:dipalito.71','dipalito','di palito',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:directoireblue.176','directoireblue','directoire blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:directoireblue.178','directoireblue','directoire blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:distilledgreen.136','distilledgreen','distilled green',[103,146,103],'679267',6787687],
['nbs-iscc-m:divablue.178','divablue','diva blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:divablue.182','divablue','diva blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:doe.57','doe','doe',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:doe.60','doe','doe',[149,128,112],'958070',9797744],
['nbs-iscc-m:doe.76','doe','doe',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:doe.79','doe','doe',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:doeskinbrown.42','doeskinbrown','doe-skin brown',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:doge.15','doge','doge',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:dogwood.57','dogwood','dogwood',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:dogwood.58','dogwood','dogwood',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:dogwood.77','dogwood','dogwood',[130,102,68],'826644',8545860],
['nbs-iscc-m:dollypink.15','dollypink','dolly pink',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:domingo.43','domingo','domingo',[121,68,59],'79443b',7947323],
['nbs-iscc-m:domingobrown.59','domingobrown','domingo brown',[66,37,24],'422518',4334872],
['nbs-iscc-m:dorado.76','dorado','dorado',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:doubloon.53','doubloon','doubloon',[217,144,88],'d99058',14258264],
['nbs-iscc-m:dove.266','dove','dove',[85,85,85],'555555',5592405],
['nbs-iscc-m:dove.259','dove','dove',[103,49,71],'673147',6762823],
['nbs-iscc-m:dovegrey.233','dovegrey','dove grey',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:dovergrey.266','dovergrey','dover grey',[85,85,85],'555555',5592405],
['nbs-iscc-m:dozer.183','dozer','dozer',[0,48,78],'00304e',12366],
['nbs-iscc-m:drab.94','drab','drab',[150,113,23],'967117',9859351],
['nbs-iscc-m:drabolive.110','drabolive','drabolive ',[91,88,66],'5b5842',5986370],
['nbs-iscc-m:dragonfly.174','dragonfly','dragonfly',[0,73,88],'004958',18776],
['nbs-iscc-m:dragonsblood.19','dragonsblood','dragon\'s blood',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:dragonsblood.39','dragonsblood','dragon\'s blood',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:dragonsblood.42','dragonsblood','dragon\'s blood',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:drake.174','drake','drake',[0,73,88],'004958',18776],
['nbs-iscc-m:drakesneckgreen.174','drakesneckgreen','drake\'s-neck green',[0,73,88],'004958',18776],
['nbs-iscc-m:dregsofwine.16','dregsofwine','dregs of wine',[114,47,55],'722f37',7483191],
['nbs-iscc-m:dregsofwine.41','dregsofwine','dregs of wine',[86,7,12],'56070c',5637900],
['nbs-iscc-m:dregsofwine.44','dregsofwine','dregs of wine',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-m:dresdenblue.182','dresdenblue','dresden blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:dresdenbrown.77','dresdenbrown','dresden brown',[130,102,68],'826644',8545860],
['nbs-iscc-m:dresdenbrown.94','dresdenbrown','dresden brown',[150,113,23],'967117',9859351],
['nbs-iscc-m:dresdenbrown.95','dresdenbrown','dresden brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:dryad.150','dryad','dryad',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:dryrose.19','dryrose','dry rose',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:dryrose.43','dryrose','dry rose',[121,68,59],'79443b',7947323],
['nbs-iscc-m:dubarry.26','dubarry','du barry',[248,131,121],'f88379',16286585],
['nbs-iscc-m:duckblue.169','duckblue','duck blue',[0,119,145],'007791',30609],
['nbs-iscc-m:duckgreen.165','duckgreen','duck green',[0,75,73],'004b49',19273],
['nbs-iscc-m:duckling.174','duckling','duckling',[0,73,88],'004958',18776],
['nbs-iscc-m:duckling.187','duckling','duckling',[54,69,79],'36454f',3556687],
['nbs-iscc-m:duckwing.204','duckwing','duck-wing',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:dugueslin.183','dugueslin','du gueslin',[0,48,78],'00304e',12366],
['nbs-iscc-m:dugueslin.201','dugueslin','du gueslin',[37,36,64],'252440',2434112],
['nbs-iscc-m:dugueslin.204','dugueslin','du gueslin',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:dumontsblue.179','dumontsblue','dumont\'s blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:dumontsblue.182','dumontsblue','dumont\'s blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:dune.109','dune','dune',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:durango.58','durango','durango',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:dusk.234','dusk','dusk',[93,85,91],'5d555b',6116699],
['nbs-iscc-m:dust.94','dust','dust',[150,113,23],'967117',9859351],
['nbs-iscc-m:dustygreen.109','dustygreen','dusty green',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:dutchazure.179','dutchazure','dutch azure',[0,65,106],'00416a',16746],
['nbs-iscc-m:dutchazure.182','dutchazure','dutch azure',[67,107,149],'436b95',4418453],
['nbs-iscc-m:dutchblue.186','dutchblue','dutch blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:dutchorange.68','dutchorange','dutch orange',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:dutchorange.71','dutchorange','dutch orange',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:dutchpink.83','dutchpink','dutch pink',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:dutchpink.84','dutchpink','dutch pink',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:dutchpink.86','dutchpink','dutch pink',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:dutchpink.87','dutchpink','dutch pink',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:dutchscarlet.11','dutchscarlet','dutch scarlet',[190,0,50],'be0032',12451890],
['nbs-iscc-m:dutchvermilion.34','dutchvermilion','dutch vermilion',[226,88,34],'e25822',14833698],
['nbs-iscc-m:dutchwareblue.204','dutchwareblue','dutch ware blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:dyersbroom.101','dyersbroom','dyer\'s broom',[234,230,121],'eae679',15394425],
['nbs-iscc-m:dyersbroom.102','dyersbroom','dyer\'s broom',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:dyersgreenwood.101','dyersgreenwood','dyer\'s greenwood',[234,230,121],'eae679',15394425],
['nbs-iscc-m:dyersgreenwood.102','dyersgreenwood','dyer\'s greenwood',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:dyerssaffron.11','dyerssaffron','dyer\'s saffron',[190,0,50],'be0032',12451890],
['nbs-iscc-m:eagle.64','eagle','eagle',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:eaudejavelgreen.102','eaudejavelgreen','eau-de-javel green',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:eaudenile.120','eaudenile','eau-de-nile',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:eaudenile.121','eaudenile','eau-de-nile',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:eaudenile.135','eaudenile','eau-de-nile',[147,197,146],'93c592',9684370],
['nbs-iscc-m:eburnean.89','eburnean','eburnean',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:ecclesiastic.236','ecclesiastic','ecclesiastic',[135,0,116],'870074',8847476],
['nbs-iscc-m:ecclesiastic.237','ecclesiastic','ecclesiastic',[158,79,136],'9e4f88',10375048],
['nbs-iscc-m:ecru.89','ecru','ecru',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:ecru.90','ecru','ecru',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:edengreen.131','edengreen','eden green',[68,148,74],'44944a',4494410],
['nbs-iscc-m:edengreen.136','edengreen','eden green',[103,146,103],'679267',6787687],
['nbs-iscc-m:eggplant.230','eggplant','eggplant',[41,30,41],'291e29',2694697],
['nbs-iscc-m:eggplant.235','eggplant','eggplant',[36,33,36],'242124',2367780],
['nbs-iscc-m:eggshellblue.149','eggshellblue','eggshell blue',[141,163,153],'8da399',9282457],
['nbs-iscc-m:eggshellblue.163','eggshellblue','eggshell blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:eggshellgreen.149','eggshellgreen','eggshell green',[141,163,153],'8da399',9282457],
['nbs-iscc-m:eggshellgreen.163','eggshellgreen','eggshell green',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:eglantine.254','eglantine','eglantine',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:egypt.178','egypt','egypt',[0,103,165],'0067a5',26533],
['nbs-iscc-m:egyptian.19','egyptian','egyptian',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:egyptianblue.176','egyptianblue','egyptian blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:egyptianblue.178','egyptianblue','egyptian blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:egyptianbrown.81','egyptianbrown','egyptian brown',[72,60,50],'483c32',4734002],
['nbs-iscc-m:egyptiangreen.145','egyptiangreen','egyptian green',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:egyptianred.16','egyptianred','egyptian red',[114,47,55],'722f37',7483191],
['nbs-iscc-m:egyptianred.40','egyptianred','egyptian red',[136,45,23],'882d17',8924439],
['nbs-iscc-m:eifel.19','eifel','eifel',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:eifel.43','eifel','eifel',[121,68,59],'79443b',7947323],
['nbs-iscc-m:elderberry.212','elderberry','elderberry',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:eldorado.54','eldorado','eldorado',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:electricblue.186','electricblue','electric [blue] ',[83,104,120],'536878',5466232],
['nbs-iscc-m:electricgreen.186','electricgreen','electric green',[83,104,120],'536878',5466232],
['nbs-iscc-m:elephantgreen.138','elephantgreen','elephant green',[23,54,32],'173620',1521184],
['nbs-iscc-m:elephantskin.64','elephantskin','elephant skin',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:elf.144','elf','elf',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:elf.149','elf','elf',[141,163,153],'8da399',9282457],
['nbs-iscc-m:elfingreen.144','elfingreen','elfin green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:elk.81','elk','elk',[72,60,50],'483c32',4734002],
['nbs-iscc-m:elmgreen.125','elmgreen','elm green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:elmwood.79','elmwood','elmwood',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:elmwood.94','elmwood','elmwood',[150,113,23],'967117',9859351],
['nbs-iscc-m:email.164','email','email',[49,120,115],'317873',3242099],
['nbs-iscc-m:email.173','email','email',[54,117,136],'367588',3569032],
['nbs-iscc-m:ember.15','ember','ember',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:emberglow.15','emberglow','emberglow',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:emberglow.27','emberglow','emberglow',[230,103,97],'e66761',15099745],
['nbs-iscc-m:emberglow.37','emberglow','emberglow',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:emeraldgreen.136','emeraldgreen','emerald green',[103,146,103],'679267',6787687],
['nbs-iscc-m:emeraldgreen.140','emeraldgreen','emerald green',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:emeraude.136','emeraude','emeraude',[103,146,103],'679267',6787687],
['nbs-iscc-m:emeraude.141','emeraude','emeraude',[0,121,89],'007959',31065],
['nbs-iscc-m:emeraude.145','emeraude','emeraude',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:eminence.224','eminence','eminence',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:emperorgreen.140','emperorgreen','emperor green',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:empire.141','empire','empire',[0,121,89],'007959',31065],
['nbs-iscc-m:empireblue.182','empireblue','empire blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:empiregreen.137','empiregreen','empire green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:empiregreen.151','empiregreen','empire green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:empireyellow.83','empireyellow','empire yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:empireyellow.86','empireyellow','empire yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:enamelblue.179','enamelblue','enamel blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:enamelblue.182','enamelblue','enamel blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:endive.98','endive','endive',[233,228,80],'e9e450',15328336],
['nbs-iscc-m:endive.101','endive','endive',[234,230,121],'eae679',15394425],
['nbs-iscc-m:endiveblue.185','endiveblue','endive blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:endiveblue.227','endiveblue','endive blue',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:englishblue.182','englishblue','english blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:englishgreen.117','englishgreen','english green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-m:englishgreen.131','englishgreen','english green',[68,148,74],'44944a',4494410],
['nbs-iscc-m:englishgreen.146','englishgreen','english green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:englishgrey.113','englishgrey','english grey ',[87,85,76],'57554c',5723468],
['nbs-iscc-m:englishinde.229','englishinde','english inde',[80,64,77],'50404d',5259341],
['nbs-iscc-m:englishivy.137','englishivy','english ivy',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:englishoak.58','englishoak','english oak',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:englishochre.71','englishochre','english ochre',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:englishpink.83','englishpink','english pink',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:englishpink.84','englishpink','english pink',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:englishpink.86','englishpink','english pink',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:englishpink.87','englishpink','english pink',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:englishred.12','englishred','english red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:englishred.15','englishred','english red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:englishred.35','englishred','english red',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:englishred.37','englishred','english red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:englishred.38','englishred','english red',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:englishred.55','englishred','english red',[128,70,27],'80461b',8406555],
['nbs-iscc-m:englishred.43','englishred','english red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:englishvermilion.12','englishvermilion','english vermilion',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:englishviolet.224','englishviolet','english violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:ensign.183','ensign','ensign',[0,48,78],'00304e',12366],
['nbs-iscc-m:epinauche.151','epinauche','epinauche',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:epsom.120','epsom','epsom',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:erin.145','erin','erin',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:erlaugreen.136','erlaugreen','erlau green',[103,146,103],'679267',6787687],
['nbs-iscc-m:escadre.186','escadre','escadre',[83,104,120],'536878',5466232],
['nbs-iscc-m:eschel.179','eschel','eschel',[0,65,106],'00416a',16746],
['nbs-iscc-m:eschel.182','eschel','eschel',[67,107,149],'436b95',4418453],
['nbs-iscc-m:eskimo.43','eskimo','eskimo',[121,68,59],'79443b',7947323],
['nbs-iscc-m:estheticgrey.155','estheticgrey','esthetic grey',[125,137,132],'7d8984',8227204],
['nbs-iscc-m:etainblue.148','etainblue','etain blue',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-m:etainblue.184','etainblue','etain blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-m:etainblue.148','etainblue','etain blue',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-m:etainblue.162','etainblue','etain blue',[150,222,209],'96ded1',9887441],
['nbs-iscc-m:etainblue.171','etainblue','etain blue',[156,209,220],'9cd1dc',10277340],
['nbs-iscc-m:etainblue.184','etainblue','etain blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-m:etang.149','etang','etang',[141,163,153],'8da399',9282457],
['nbs-iscc-m:etang.154','etang','etang',[178,190,181],'b2beb5',11714229],
['nbs-iscc-m:ether.172','ether','ether',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:etherealblue.181','etherealblue','ethereal blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:etherealblue.199','etherealblue','ethereal blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:etonblue.163','etonblue','eton blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:etruscan.43','etruscan','etruscan',[121,68,59],'79443b',7947323],
['nbs-iscc-m:etruscan.58','etruscan','etruscan',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:etruscanred.39','etruscanred','etruscan red',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:eucalyptusgreen.90','eucalyptusgreen','eucalyptus [green] ',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:eucalyptusgreen.122','eucalyptusgreen','eucalyptus [green] ',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:eugeniared.15','eugeniared','eugenia red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:eugenie.228','eugenie','eugenie',[121,104,120],'796878',7956600],
['nbs-iscc-m:eupatoriumpurple.241','eupatoriumpurple','eupatorium purple',[145,92,131],'915c83',9526403],
['nbs-iscc-m:eurekared.16','eurekared','eureka red',[114,47,55],'722f37',7483191],
['nbs-iscc-m:evegreen.98','evegreen','eve green',[233,228,80],'e9e450',15328336],
['nbs-iscc-m:evegreen.130','evegreen','eve green',[131,211,125],'83d37d',8639357],
['nbs-iscc-m:evegreen.131','evegreen','eve green',[68,148,74],'44944a',4494410],
['nbs-iscc-m:evenglow.22','evenglow','evenglow',[143,129,127],'8f817f',9404799],
['nbs-iscc-m:eventide.214','eventide','eventide',[150,144,171],'9690ab',9867435],
['nbs-iscc-m:everglade.267','everglade','everglade',[34,34,34],'222222',2236962],
['nbs-iscc-m:evergreen.137','evergreen','evergreen',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:evque.224','evque','evque',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:fadedrose.19','fadedrose','faded rose',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:faience.169','faience','faience',[0,119,145],'007791',30609],
['nbs-iscc-m:faience.173','faience','faience',[54,117,136],'367588',3569032],
['nbs-iscc-m:fairway.152','fairway','fairway',[26,36,33],'1a2421',1713185],
['nbs-iscc-m:fairygreen.136','fairygreen','fairy green ',[103,146,103],'679267',6787687],
['nbs-iscc-m:fakir.73','fakir','fakir',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:falcon.62','falcon','falcon',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:fallow.76','fallow','fallow',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:fandango.254','fandango','fandango',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:faon.45','faon','faon',[151,127,115],'977f73',9928563],
['nbs-iscc-m:faon.58','faon','faon',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:faon.60','faon','faon',[149,128,112],'958070',9797744],
['nbs-iscc-m:faon.61','faon','faon',[99,81,71],'635147',6508871],
['nbs-iscc-m:fashiongray.266','fashiongray','fashion gray',[85,85,85],'555555',5592405],
['nbs-iscc-m:fawnbrown.45','fawnbrown','fawn [brown] ',[151,127,115],'977f73',9928563],
['nbs-iscc-m:fawnbrown.58','fawnbrown','fawn [brown] ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:fawnbrown.60','fawnbrown','fawn [brown] ',[149,128,112],'958070',9797744],
['nbs-iscc-m:fawnbrown.61','fawnbrown','fawn [brown] ',[99,81,71],'635147',6508871],
['nbs-iscc-m:feldspar.144','feldspar','feldspar',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:fern.120','fern','fern',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:fernambucowood.15','fernambucowood','fernambucowood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:fernambucowood.19','fernambucowood','fernambucowood',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:ferngreen.120','ferngreen','fern green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:ferngreen.122','ferngreen','fern green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:ferruginous.38','ferruginous','ferruginous',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:ferruginous.55','ferruginous','ferruginous',[128,70,27],'80461b',8406555],
['nbs-iscc-m:feuille.58','feuille','feuille',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:feuillemorte.51','feuillemorte','feuille morte',[190,101,22],'be6516',12477718],
['nbs-iscc-m:feuillemorte.54','feuillemorte','feuille morte',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:feuillemorte.55','feuillemorte','feuille morte',[128,70,27],'80461b',8406555],
['nbs-iscc-m:feulamort.51','feulamort','feulamort',[190,101,22],'be6516',12477718],
['nbs-iscc-m:feulamort.54','feulamort','feulamort',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:feulamort.55','feulamort','feulamort',[128,70,27],'80461b',8406555],
['nbs-iscc-m:fez.11','fez','fez',[190,0,50],'be0032',12451890],
['nbs-iscc-m:fieldsorangevermilion.35','fieldsorangevermilion','field\'s orange vermilion',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:fieryred.34','fieryred','fiery red',[226,88,34],'e25822',14833698],
['nbs-iscc-m:fiesta.3','fiesta','fiesta',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:fieulamort.51','fieulamort','fieulamort',[190,101,22],'be6516',12477718],
['nbs-iscc-m:fieulamort.54','fieulamort','fieulamort',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:fieulamort.55','fieulamort','fieulamort',[128,70,27],'80461b',8406555],
['nbs-iscc-m:filbertbrown.57','filbertbrown','filbert [brown] ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:filbertbrown.74','filbertbrown','filbert [brown] ',[153,101,21],'996515',10052885],
['nbs-iscc-m:filemot.51','filemot','filemot',[190,101,22],'be6516',12477718],
['nbs-iscc-m:filemot.54','filemot','filemot',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:filemot.55','filemot','filemot',[128,70,27],'80461b',8406555],
['nbs-iscc-m:fillemot.51','fillemot','fillemot',[190,101,22],'be6516',12477718],
['nbs-iscc-m:fillemot.54','fillemot','fillemot',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:fillemot.55','fillemot','fillemot',[128,70,27],'80461b',8406555],
['nbs-iscc-m:firecracker.34','firecracker','firecracker',[226,88,34],'e25822',14833698],
['nbs-iscc-m:firefly.12','firefly','firefly',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:firefly.15','firefly','firefly',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:firered.34','firered','fire red',[226,88,34],'e25822',14833698],
['nbs-iscc-m:firescarlet.11','firescarlet','fire scarlet',[190,0,50],'be0032',12451890],
['nbs-iscc-m:fireweed.219','fireweed','fireweed',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:firgreen.137','firgreen','fir [green] ',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:firgreen.138','firgreen','fir [green] ',[23,54,32],'173620',1521184],
['nbs-iscc-m:firgreen.151','firgreen','fir [green] ',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:firmament.186','firmament','firmament',[83,104,120],'536878',5466232],
['nbs-iscc-m:firmamentblue.172','firmamentblue','firmament blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:fishgrey.110','fishgrey','fish grey ',[91,88,66],'5b5842',5986370],
['nbs-iscc-m:fishgrey.113','fishgrey','fish grey ',[87,85,76],'57554c',5723468],
['nbs-iscc-m:flag.212','flag','flag',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:flameblue.200','flameblue','flame blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:flameblue.204','flameblue','flame blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:flameorange.34','flameorange','flame orange',[226,88,34],'e25822',14833698],
['nbs-iscc-m:flamescarlet.34','flamescarlet','flame [scarlet] ',[226,88,34],'e25822',14833698],
['nbs-iscc-m:flamingmaple.11','flamingmaple','flaming maple',[190,0,50],'be0032',12451890],
['nbs-iscc-m:flamingmaple.12','flamingmaple','flaming maple',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:flamingmaple.13','flamingmaple','flaming maple',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:flamingmaple.15','flamingmaple','flaming maple',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:flamingo.37','flamingo','flamingo',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:flammeous.35','flammeous','flammeous',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:flash.12','flash','flash',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:flash.15','flash','flash',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:flash.36','flash','flash',[170,56,30],'aa381e',11155486],
['nbs-iscc-m:flash.38','flash','flash',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:flax.90','flax','flax',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:flaxen.90','flaxen','flaxen',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:flaxflowerblossom.186','flaxflowerblossom','flaxflower blossom',[83,104,120],'536878',5466232],
['nbs-iscc-m:flaxflowerblue.182','flaxflowerblue','flaxflower blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:flaxflowerblue.199','flaxflowerblue','flaxflower blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:flea.16','flea','flea',[114,47,55],'722f37',7483191],
['nbs-iscc-m:flemishblue.187','flemishblue','flemish blue',[54,69,79],'36454f',3556687],
['nbs-iscc-m:flesh.73','flesh','flesh',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:flesh.93','flesh','flesh',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:fleshblond.29','fleshblond','flesh blond',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:fleshblond.32','fleshblond','flesh blond',[199,173,163],'c7ada3',13086115],
['nbs-iscc-m:fleshblond.33','fleshblond','flesh blond',[194,172,153],'c2ac99',12758169],
['nbs-iscc-m:fleshochre.38','fleshochre','flesh ochre',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:fleurdelys.223','fleurdelys','fleur-de-lys',[134,96,142],'86608e',8806542],
['nbs-iscc-m:flint.267','flint','flint',[34,34,34],'222222',2236962],
['nbs-iscc-m:flintgrey.265','flintgrey','flint grey',[132,132,130],'848482',8684674],
['nbs-iscc-m:flirt.15','flirt','flirt',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:florencebrown.19','florencebrown','florence brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:florenceearth.43','florenceearth','florence earth',[121,68,59],'79443b',7947323],
['nbs-iscc-m:florentine.34','florentine','florentine',[226,88,34],'e25822',14833698],
['nbs-iscc-m:florentinelake.11','florentinelake','florentine lake',[190,0,50],'be0032',12451890],
['nbs-iscc-m:florentinelake.12','florentinelake','florentine lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:floridagold.68','floridagold','florida gold',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:floridagold.71','floridagold','florida gold',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:flossflowerblue.227','flossflowerblue','floss flower blue',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:flowerdelucegreen.137','flowerdelucegreen','flower de luce green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:fluoritegreen.136','fluoritegreen','fluorite green',[103,146,103],'679267',6787687],
['nbs-iscc-m:fluoriteviolet.224','fluoriteviolet','fluorite violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:fluoriteviolet.225','fluoriteviolet','fluorite violet',[48,25,52],'301934',3152180],
['nbs-iscc-m:fog.264','fog','fog',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:foliagebrown.51','foliagebrown','foliage brown',[190,101,22],'be6516',12477718],
['nbs-iscc-m:foliagebrown.54','foliagebrown','foliage brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:foliagebrown.55','foliagebrown','foliage brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:foliagegreen.120','foliagegreen','foliage green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:folimort.51','folimort','folimort',[190,101,22],'be6516',12477718],
['nbs-iscc-m:folimort.54','folimort','folimort',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:folimort.55','folimort','folimort',[128,70,27],'80461b',8406555],
['nbs-iscc-m:folkstone.79','folkstone','folkstone',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:folly.11','folly','folly',[190,0,50],'be0032',12451890],
['nbs-iscc-m:folly.12','folly','folly',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:fontainebleau.19','fontainebleau','fontainebleau',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:forest.151','forest','forest',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:forestgreen.125','forestgreen','forest green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:forestofdeanred.38','forestofdeanred','forest of dean red',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:forestofdeanred.55','forestofdeanred','forest of dean red',[128,70,27],'80461b',8406555],
['nbs-iscc-m:forgetmenotblue.185','forgetmenotblue','forget-me-not [blue] ',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:forgetmenotblue.186','forgetmenotblue','forget-me-not [blue] ',[83,104,120],'536878',5466232],
['nbs-iscc-m:formosa.53','formosa','formosa',[217,144,88],'d99058',14258264],
['nbs-iscc-m:formosa.57','formosa','formosa',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:forsythia.71','forsythia','forsythia',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:fox.77','fox','fox',[130,102,68],'826644',8545860],
['nbs-iscc-m:fox.94','fox','fox',[150,113,23],'967117',9859351],
['nbs-iscc-m:fox.95','fox','fox',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:foxglove.207','foxglove','foxglove',[96,78,151],'604e97',6311575],
['nbs-iscc-m:foxglove.211','foxglove','foxglove',[96,78,129],'604e81',6311553],
['nbs-iscc-m:foxtrot.163','foxtrot','fox trot',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:fragonard.15','fragonard','fragonard',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:france.183','france','france',[0,48,78],'00304e',12366],
['nbs-iscc-m:francerose.3','francerose','france rose',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:francerose.24','francerose','france rose',[40,32,34],'282022',2629666],
['nbs-iscc-m:freedom.183','freedom','freedom',[0,48,78],'00304e',12366],
['nbs-iscc-m:freestone.73','freestone','freestone',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:frenchbeige.57','frenchbeige','french beige',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:frenchberries.68','frenchberries','french berries',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:frenchberries.71','frenchberries','french berries',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:frenchblue.178','frenchblue','french blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:frenchblue.196','frenchblue','french blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:frenchgreen.136','frenchgreen','french green',[103,146,103],'679267',6787687],
['nbs-iscc-m:frenchgreen.145','frenchgreen','french green',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:frenchgrey.154','frenchgrey','french grey',[178,190,181],'b2beb5',11714229],
['nbs-iscc-m:frenchgrey.264','frenchgrey','french grey',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:frenchlilac.223','frenchlilac','french lilac',[134,96,142],'86608e',8806542],
['nbs-iscc-m:frenchlilac.228','frenchlilac','french lilac',[121,104,120],'796878',7956600],
['nbs-iscc-m:frenchmaroon.19','frenchmaroon','french maroon',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:frenchnude.45','frenchnude','french nude',[151,127,115],'977f73',9928563],
['nbs-iscc-m:frenchnude.57','frenchnude','french nude',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:frenchnude.60','frenchnude','french nude',[149,128,112],'958070',9797744],
['nbs-iscc-m:frenchochre.71','frenchochre','french ochre',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:frenchpink.73','frenchpink','french pink',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:frenchpink.87','frenchpink','french pink',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:frenchpurple.258','frenchpurple','french purple',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:frenchscarlet.11','frenchscarlet','french scarlet',[190,0,50],'be0032',12451890],
['nbs-iscc-m:frenchultramarine.178','frenchultramarine','french ultramarine',[0,103,165],'0067a5',26533],
['nbs-iscc-m:frenchultramarine.196','frenchultramarine','french ultramarine',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:frenchvermilion.34','frenchvermilion','french vermilion',[226,88,34],'e25822',14833698],
['nbs-iscc-m:frenchveronesegreen.141','frenchveronesegreen','[french] veronese green ',[0,121,89],'007959',31065],
['nbs-iscc-m:frenchveronesegreen.145','frenchveronesegreen','[french] veronese green ',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:frenchwhite.232','frenchwhite','french white',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-m:frenchwhite.252','frenchwhite','french white',[232,204,215],'e8ccd7',15256791],
['nbs-iscc-m:frenchyellow.54','frenchyellow','french yellow',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:frenchyellow.74','frenchyellow','french yellow',[153,101,21],'996515',10052885],
['nbs-iscc-m:frenchyellow.57','frenchyellow','french yellow',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:friar.61','friar','friar',[99,81,71],'635147',6508871],
['nbs-iscc-m:friar.62','friar','friar',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:frostgrey.265','frostgrey','frost grey',[132,132,130],'848482',8684674],
['nbs-iscc-m:frostygreen.151','frostygreen','frosty green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:fuchsia.236','fuchsia','fuchsia',[135,0,116],'870074',8847476],
['nbs-iscc-m:fudge.58','fudge','fudge',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:fujiyama.43','fujiyama','fujiyama',[121,68,59],'79443b',7947323],
['nbs-iscc-m:fuscous.64','fuscous','fuscous',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:fustet.83','fustet','fustet',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:fustet.84','fustet','fustet',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:fustic.87','fustic','fustic',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:fustic.90','fustic','fustic',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:gagegreen.122','gagegreen','gage green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:gaiety.3','gaiety','gaiety',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:galleon.62','galleon','galleon',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:gallstone.84','gallstone','gallstone',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:gambia.58','gambia','gambia',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:gamboge.84','gamboge','gamboge',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:gardeniagreen.151','gardeniagreen','gardenia green ',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:gargoyle.266','gargoyle','gargoyle',[85,85,85],'555555',5592405],
['nbs-iscc-m:garlandgreen.125','garlandgreen','garland green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:garlandgreen.137','garlandgreen','garland green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:garnet.16','garnet','garnet',[114,47,55],'722f37',7483191],
['nbs-iscc-m:garnetbrown.16','garnetbrown','garnet brown',[114,47,55],'722f37',7483191],
['nbs-iscc-m:garnetbrown.43','garnetbrown','garnet brown',[121,68,59],'79443b',7947323],
['nbs-iscc-m:garnetred.13','garnetred','garnet red',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:garnetred.16','garnetred','garnet red',[114,47,55],'722f37',7483191],
['nbs-iscc-m:garterblue.204','garterblue','garter blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:gaude.101','gaude','gaude',[234,230,121],'eae679',15394425],
['nbs-iscc-m:gaude.102','gaude','gaude',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:gaudelake.86','gaudelake','gaude lake',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:gaudygreen.120','gaudygreen','gaudy green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:gaudygreen.125','gaudygreen','gaudy green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:gaygreen.120','gaygreen','gay green ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:gazellebrown.61','gazellebrown','gazelle [brown] ',[99,81,71],'635147',6508871],
['nbs-iscc-m:gazellebrown.80','gazellebrown','gazelle [brown] ',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:geisha.241','geisha','geisha',[145,92,131],'915c83',9526403],
['nbs-iscc-m:geisha.258','geisha','geisha',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:geisha.262','geisha','geisha',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:gendarmeblue.164','gendarmeblue','gendarme [blue] ',[49,120,115],'317873',3242099],
['nbs-iscc-m:gendarmeblue.173','gendarmeblue','gendarme [blue] ',[54,117,136],'367588',3569032],
['nbs-iscc-m:generall.83','generall','generall',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:generall.84','generall','generall',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:generall.86','generall','generall',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:generall.87','generall','generall',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:genestrole.101','genestrole','genestrole',[234,230,121],'eae679',15394425],
['nbs-iscc-m:genestrole.102','genestrole','genestrole',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:genet.101','genet','genet',[234,230,121],'eae679',15394425],
['nbs-iscc-m:genet.102','genet','genet',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:genevablue.204','genevablue','geneva blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:genista.53','genista','genista',[217,144,88],'d99058',14258264],
['nbs-iscc-m:genoablue.186','genoablue','genoa blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:gentian.182','gentian','gentian',[67,107,149],'436b95',4418453],
['nbs-iscc-m:gentian.199','gentian','gentian',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:gentianblue.200','gentianblue','gentian blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:genuineultramarineblue.176','genuineultramarineblue','genuine ultramarine [blue] ',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:genuineultramarineblue.178','genuineultramarineblue','genuine ultramarine [blue] ',[0,103,165],'0067a5',26533],
['nbs-iscc-m:geranium.12','geranium','geranium',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:geraniumlake.11','geraniumlake','geranium lake',[190,0,50],'be0032',12451890],
['nbs-iscc-m:geraniumpetal.11','geraniumpetal','geranium petal',[190,0,50],'be0032',12451890],
['nbs-iscc-m:geraniumpink.3','geraniumpink','geranium pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:ghent.183','ghent','ghent',[0,48,78],'00304e',12366],
['nbs-iscc-m:giallolini.86','giallolini','giallolini',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:giallolini.87','giallolini','giallolini',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:giallolini.89','giallolini','giallolini',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:giallolini.90','giallolini','giallolini',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:giallolino.86','giallolino','giallolino',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:giallolino.87','giallolino','giallolino',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:giallolino.89','giallolino','giallolino',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:giallolino.90','giallolino','giallolino',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:gigas.223','gigas','gigas',[134,96,142],'86608e',8806542],
['nbs-iscc-m:gild.68','gild','gild',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:gild.71','gild','gild',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:gilded.68','gilded','gilded',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:gilded.71','gilded','gilded',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:gilt.68','gilt','gilt',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:gilt.71','gilt','gilt',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:gingeline.55','gingeline','gingeline',[128,70,27],'80461b',8406555],
['nbs-iscc-m:gingeoline.55','gingeoline','gingeoline',[128,70,27],'80461b',8406555],
['nbs-iscc-m:ginger.55','ginger','ginger',[128,70,27],'80461b',8406555],
['nbs-iscc-m:gingerline.55','gingerline','gingerline',[128,70,27],'80461b',8406555],
['nbs-iscc-m:gingerspice.43','gingerspice','gingerspice',[121,68,59],'79443b',7947323],
['nbs-iscc-m:gingioline.55','gingioline','gingioline',[128,70,27],'80461b',8406555],
['nbs-iscc-m:giraffe.54','giraffe','giraffe',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:giraffe.57','giraffe','giraffe',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:glacier.187','glacier','glacier',[54,69,79],'36454f',3556687],
['nbs-iscc-m:glacierblue.186','glacierblue','glacier blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:gladiolus.12','gladiolus','gladiolus',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:glaeul.18','glaeul','glaeul',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:glaeul.30','glaeul','glaeul',[196,131,121],'c48379',12878713],
['nbs-iscc-m:glassgreen.119','glassgreen','glass green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-m:glassgrey.190','glassgrey','glass grey',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-m:glaucous.121','glaucous','glaucous',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:glaucousblue.186','glaucousblue','glaucous-blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:glaucousgreen.148','glaucousgreen','glaucous-green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-m:glaucousgrey.190','glaucousgrey','glaucous-grey',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-m:glaucousgrey.264','glaucousgrey','glaucous-grey',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:glintogold.87','glintogold','glint o\'gold',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:gloaming.18','gloaming','gloaming',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:glory.197','glory','glory',[39,36,88],'272458',2565208],
['nbs-iscc-m:glory.208','glory','glory',[50,23,77],'32174d',3282765],
['nbs-iscc-m:gloxinia.219','gloxinia','gloxinia',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:gmelinsblue.178','gmelinsblue','gmelin\'s blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:gmelinsblue.196','gmelinsblue','gmelin\'s blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:gnaphaliumgreen.149','gnaphaliumgreen','gnaphalium green',[141,163,153],'8da399',9282457],
['nbs-iscc-m:goat.64','goat','goat',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:gobelinblue.186','gobelinblue','gobelin blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:gobelinblue.173','gobelinblue','gobelin blue',[54,117,136],'367588',3569032],
['nbs-iscc-m:gobelinblue.186','gobelinblue','gobelin blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:gobelingreen.145','gobelingreen','gobelin green',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:gobelinscarlet.11','gobelinscarlet','gobelin scarlet',[190,0,50],'be0032',12451890],
['nbs-iscc-m:gold.94','gold','gold',[150,113,23],'967117',9859351],
['nbs-iscc-m:goldbronze.61','goldbronze','gold bronze',[99,81,71],'635147',6508871],
['nbs-iscc-m:goldbronze.80','goldbronze','gold bronze',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:goldbrown.55','goldbrown','gold brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:goldearth.71','goldearth','gold earth',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:golden.76','golden','golden',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:golden.91','golden','golden',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:golden.94','golden','golden',[150,113,23],'967117',9859351],
['nbs-iscc-m:goldenbrown.55','goldenbrown','golden brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:goldenchestnut.58','goldenchestnut','golden chestnut',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:goldencoral.26','goldencoral','golden coral',[248,131,121],'f88379',16286585],
['nbs-iscc-m:goldencorn.71','goldencorn','golden corn',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:goldenfeather.83','goldenfeather','golden feather',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:goldenfeather.84','goldenfeather','golden feather',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:goldenfeather.86','goldenfeather','golden feather',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:goldenfeather.87','goldenfeather','golden feather',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:goldenglow.71','goldenglow','golden glow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:goldenglow.84','goldenglow','golden glow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:goldengreen.90','goldengreen','golden green',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:goldengreen.91','goldengreen','golden green',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:goldenochre.53','goldenochre','golden ochre',[217,144,88],'d99058',14258264],
['nbs-iscc-m:goldenpoppy.34','goldenpoppy','golden poppy',[226,88,34],'e25822',14833698],
['nbs-iscc-m:goldenrod.84','goldenrod','golden rod',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:goldenwheat.71','goldenwheat','golden wheat',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:goldenyellow.68','goldenyellow','golden yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:goldenyellow.71','goldenyellow','golden yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:goldenyellow.72','goldenyellow','golden yellow',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:goldenyellow.74','goldenyellow','golden yellow',[153,101,21],'996515',10052885],
['nbs-iscc-m:goldleaf.53','goldleaf','gold leaf',[217,144,88],'d99058',14258264],
['nbs-iscc-m:goldleaf.72','goldleaf','gold leaf',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:goldochre.53','goldochre','gold ochre',[217,144,88],'d99058',14258264],
['nbs-iscc-m:goldyellow.68','goldyellow','gold yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:goldyellow.71','goldyellow','gold yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:golfgreen.137','golfgreen','golf green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:golfred.12','golfred','golf [red] ',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:golfred.15','golfred','golf [red] ',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:gooseberry.259','gooseberry','gooseberry',[103,49,71],'673147',6762823],
['nbs-iscc-m:gooseberrygreen.136','gooseberrygreen','gooseberry green',[103,146,103],'679267',6787687],
['nbs-iscc-m:goosegrey.81','goosegrey','goose grey',[72,60,50],'483c32',4734002],
['nbs-iscc-m:gorevan.58','gorevan','gorevan',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:goura.229','goura','goura',[80,64,77],'50404d',5259341],
['nbs-iscc-m:goya.12','goya','goya',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:grain.73','grain','grain',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:grain.76','grain','grain',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:grain.79','grain','grain',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:grain.11','grain','grain',[190,0,50],'be0032',12451890],
['nbs-iscc-m:grainingrain.11','grainingrain','grain in grain',[190,0,50],'be0032',12451890],
['nbs-iscc-m:granada.47','granada','granada',[67,48,46],'43302e',4403246],
['nbs-iscc-m:granada.62','granada','granada',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:granat.11','granat','granat',[190,0,50],'be0032',12451890],
['nbs-iscc-m:granat.34','granat','granat',[226,88,34],'e25822',14833698],
['nbs-iscc-m:granatflower.11','granatflower','granatflower',[190,0,50],'be0032',12451890],
['nbs-iscc-m:granatflower.34','granatflower','granatflower',[226,88,34],'e25822',14833698],
['nbs-iscc-m:granite.233','granite','granite',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:graniteblue.264','graniteblue','granite blue',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:grape.212','grape','grape',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:grapeblue.212','grapeblue','grape blue',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:grapefruit.102','grapefruit','grapefruit',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:grapegreen.106','grapegreen','grape green',[134,126,54],'867e36',8814134],
['nbs-iscc-m:grapejuice.219','grapejuice','grapejuice',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:grapenuts.58','grapenuts','grapenuts',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:graphite.229','graphite','graphite',[80,64,77],'50404d',5259341],
['nbs-iscc-m:graphite.267','graphite','graphite',[34,34,34],'222222',2236962],
['nbs-iscc-m:grassgreen.120','grassgreen','grass green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:grasshopper.120','grasshopper','grasshopper',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:gravel.79','gravel','gravel',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:grayn.11','grayn','grayn',[190,0,50],'be0032',12451890],
['nbs-iscc-m:grebe.234','grebe','grebe',[93,85,91],'5d555b',6116699],
['nbs-iscc-m:grebe.266','grebe','grebe',[85,85,85],'555555',5592405],
['nbs-iscc-m:grecianrose.18','grecianrose','grecian rose',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:grecianrose.30','grecianrose','grecian rose',[196,131,121],'c48379',12878713],
['nbs-iscc-m:greenash.135','greenash','green ash',[147,197,146],'93c592',9684370],
['nbs-iscc-m:greenfinch.106','greenfinch','green finch',[134,126,54],'867e36',8814134],
['nbs-iscc-m:greenslate.150','greenslate','green slate',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:greenslate.156','greenslate','green slate',[78,87,85],'4e5755',5134165],
['nbs-iscc-m:greenstone.122','greenstone','green stone',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:grenadinered.37','grenadinered','grenadine red ',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:grenat.16','grenat','grenat',[114,47,55],'722f37',7483191],
['nbs-iscc-m:gretnagreen.136','gretnagreen','gretna green',[103,146,103],'679267',6787687],
['nbs-iscc-m:gretnagreen.145','gretnagreen','gretna green',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:grey31.64','grey31','grey 31 ',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:greydawn.233','greydawn','grey dawn',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:greydawn.265','greydawn','grey dawn',[132,132,130],'848482',8684674],
['nbs-iscc-m:greydrab.109','greydrab','grey drab',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:greyn.11','greyn','greyn',[190,0,50],'be0032',12451890],
['nbs-iscc-m:greystone.112','greystone','grey stone',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:greyultramarineash.154','greyultramarineash','grey ultramarine ash',[178,190,181],'b2beb5',11714229],
['nbs-iscc-m:gridelin.258','gridelin','gridelin',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:griffin.266','griffin','griffin',[85,85,85],'555555',5592405],
['nbs-iscc-m:grisdelin.258','grisdelin','gris-de-lin',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:grottoblue.169','grottoblue','grotto [blue] ',[0,119,145],'007791',30609],
['nbs-iscc-m:grouse.61','grouse','grouse',[99,81,71],'635147',6508871],
['nbs-iscc-m:grouse.80','grouse','grouse',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:guignetsgreen.165','guignetsgreen','guignet\'s green',[0,75,73],'004b49',19273],
['nbs-iscc-m:guimetsblue.178','guimetsblue','guimet\'s blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:guimetsblue.196','guimetsblue','guimet\'s blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:guineagreen.160','guineagreen','guinea green',[0,122,116],'007a74',31348],
['nbs-iscc-m:guineahen.266','guineahen','guinea hen',[85,85,85],'555555',5592405],
['nbs-iscc-m:gull.265','gull','gull',[132,132,130],'848482',8684674],
['nbs-iscc-m:gullgrey.233','gullgrey','gull grey',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:gunmetal.266','gunmetal','gunmetal',[85,85,85],'555555',5592405],
['nbs-iscc-m:gypsy.55','gypsy','gypsy',[128,70,27],'80461b',8406555],
['nbs-iscc-m:gypsyred.15','gypsyred','gypsy red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:haematitered.16','haematitered','haematite red',[114,47,55],'722f37',7483191],
['nbs-iscc-m:haematitered.19','haematitered','haematite red',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:haematitered.20','haematitered','haematite red',[84,61,63],'543d3f',5520703],
['nbs-iscc-m:hairbrown.64','hairbrown','hair brown',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:hairbrown.266','hairbrown','hair brown',[85,85,85],'555555',5592405],
['nbs-iscc-m:hamadan.94','hamadan','hamadan',[150,113,23],'967117',9859351],
['nbs-iscc-m:hamburglake.11','hamburglake','hamburg lake',[190,0,50],'be0032',12451890],
['nbs-iscc-m:hamburglake.12','hamburglake','hamburg lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:hampsteadbrown.46','hampsteadbrown','hampstead brown',[103,76,71],'674c47',6769735],
['nbs-iscc-m:hankow.183','hankow','hankow',[0,48,78],'00304e',12366],
['nbs-iscc-m:hankow.201','hankow','hankow',[37,36,64],'252440',2434112],
['nbs-iscc-m:hankow.204','hankow','hankow',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:harborblue.192','harborblue','harbor blue',[81,88,94],'51585e',5331038],
['nbs-iscc-m:harlemblue.170','harlemblue','harlem blue',[46,132,149],'2e8495',3048597],
['nbs-iscc-m:harlemblue.182','harlemblue','harlem blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:harlequin.131','harlequin','harlequin',[68,148,74],'44944a',4494410],
['nbs-iscc-m:harrisonred.11','harrisonred','harrison red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:harvardcrimson.15','harvardcrimson','harvard crimson',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:harvest.54','harvest','harvest',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:harvest.57','harvest','harvest',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:hathigray.155','hathigray','hathi gray',[125,137,132],'7d8984',8227204],
['nbs-iscc-m:hathor.176','hathor','hathor',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:hathor.178','hathor','hathor',[0,103,165],'0067a5',26533],
['nbs-iscc-m:havanarose.19','havanarose','havana rose',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:hay.105','hay','hay',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:hazel.57','hazel','hazel',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:hazel.74','hazel','hazel',[153,101,21],'996515',10052885],
['nbs-iscc-m:hazelnut.57','hazelnut','hazelnut',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:hazelnut.74','hazelnut','hazelnut',[153,101,21],'996515',10052885],
['nbs-iscc-m:hazyblue.135','hazyblue','hazy blue',[147,197,146],'93c592',9684370],
['nbs-iscc-m:hazyblue.144','hazyblue','hazy blue',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:hazyblue.149','hazyblue','hazy blue',[141,163,153],'8da399',9282457],
['nbs-iscc-m:heather.262','heather','heather',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:hebe.15','hebe','hebe',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:heliotrope.242','heliotrope','heliotrope',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:heliotropegrey.227','heliotropegrey','heliotrope grey ',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:heliotropegrey.228','heliotropegrey','heliotrope grey ',[121,104,120],'796878',7956600],
['nbs-iscc-m:heliotropegrey.233','heliotropegrey','heliotrope grey ',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:helleboregreen.125','helleboregreen','hellebore green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:helleboregreen.120','helleboregreen','hellebore green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:helleboregreen.122','helleboregreen','hellebore green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:helleboregreen.125','helleboregreen','hellebore green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:helleboregreen.127','helleboregreen','hellebore green',[81,87,68],'515744',5330756],
['nbs-iscc-m:helleborered.258','helleborered','hellebore red',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:helvetiablue.179','helvetiablue','helvetia blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:hemlock.152','hemlock','hemlock',[26,36,33],'1a2421',1713185],
['nbs-iscc-m:hemp.109','hemp','hemp',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:henna.40','henna','henna',[136,45,23],'882d17',8924439],
['nbs-iscc-m:hepatica.223','hepatica','hepatica',[134,96,142],'86608e',8806542],
['nbs-iscc-m:hermosapink.2','hermosapink','hermosa pink',[234,147,153],'ea9399',15373209],
['nbs-iscc-m:hermosapink.5','hermosapink','hermosa pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:heron.186','heron','heron',[83,104,120],'536878',5466232],
['nbs-iscc-m:hiberniangreen.137','hiberniangreen','hibernian green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:highlandgreen.157','highlandgreen','highland green',[30,35,33],'1e2321',1975073],
['nbs-iscc-m:hindu.62','hindu','hindu',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:hispano.55','hispano','hispano',[128,70,27],'80461b',8406555],
['nbs-iscc-m:hockey.11','hockey','hockey',[190,0,50],'be0032',12451890],
['nbs-iscc-m:hollandblue.183','hollandblue','holland blue',[0,48,78],'00304e',12366],
['nbs-iscc-m:hollyberry.12','hollyberry','holly berry',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:hollyberry.15','hollyberry','holly berry',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:hollygreen.125','hollygreen','holly green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:hollygreen.132','hollygreen','holly green',[0,98,45],'00622d',25133],
['nbs-iscc-m:hollygreen.137','hollygreen','holly green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:hollyhock.256','hollyhock','hollyhock',[120,24,74],'78184a',7870538],
['nbs-iscc-m:hollywood.13','hollywood','hollywood',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:hollywood.15','hollywood','hollywood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:homageblue.201','homageblue','homage blue',[37,36,64],'252440',2434112],
['nbs-iscc-m:honeybeige.76','honeybeige','honey beige',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:honeybird.168','honeybird','honey bird',[35,158,186],'239eba',2334394],
['nbs-iscc-m:honeydew.26','honeydew','honeydew',[248,131,121],'f88379',16286585],
['nbs-iscc-m:honeydew.53','honeydew','honeydew',[217,144,88],'d99058',14258264],
['nbs-iscc-m:honeysuckle.76','honeysuckle','honeysuckle',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:honeysweet.87','honeysweet','honeysweet',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:honeysweet.90','honeysweet','honeysweet',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:honeyyellow.91','honeyyellow','honey [yellow] ',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:hopi.57','hopi','hopi',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:horacevernetsblue.194','horacevernetsblue','horace vernet\'s blue',[48,38,122],'30267a',3155578],
['nbs-iscc-m:horizonblue.153','horizonblue','horizon [blue] ',[223,237,232],'dfede8',14675432],
['nbs-iscc-m:horsechestnut.62','horsechestnut','horsechestnut',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:hortenseviolet.223','hortenseviolet','hortense violet',[134,96,142],'86608e',8806542],
['nbs-iscc-m:hortensia.224','hortensia','hortensia',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:hortensia.225','hortensia','hortensia',[48,25,52],'301934',3152180],
['nbs-iscc-m:hortensia.229','hortensia','hortensia',[80,64,77],'50404d',5259341],
['nbs-iscc-m:hortensia.230','hortensia','hortensia',[41,30,41],'291e29',2694697],
['nbs-iscc-m:huckleberry.212','huckleberry','huckleberry',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:hudsonseal.266','hudsonseal','hudson seal',[85,85,85],'555555',5592405],
['nbs-iscc-m:hudsonseal.267','hudsonseal','hudson seal',[34,34,34],'222222',2236962],
['nbs-iscc-m:hummingbird.164','hummingbird','hummingbird',[49,120,115],'317873',3242099],
['nbs-iscc-m:hungarianblue.182','hungarianblue','hungarian blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:hungariangreen.136','hungariangreen','hungarian green',[103,146,103],'679267',6787687],
['nbs-iscc-m:huntergreen.138','huntergreen','hunter [green] ',[23,54,32],'173620',1521184],
['nbs-iscc-m:huntersgreen.138','huntersgreen','hunter\'s green',[23,54,32],'173620',1521184],
['nbs-iscc-m:huron.164','huron','huron',[49,120,115],'317873',3242099],
['nbs-iscc-m:hussar.186','hussar','hussar',[83,104,120],'536878',5466232],
['nbs-iscc-m:hyacinth.210','hyacinth','hyacinth',[140,130,182],'8c82b6',9208502],
['nbs-iscc-m:hyacinth.223','hyacinth','hyacinth',[134,96,142],'86608e',8806542],
['nbs-iscc-m:hyacinthblue.197','hyacinthblue','hyacinth blue',[39,36,88],'272458',2565208],
['nbs-iscc-m:hyacinthred.39','hyacinthred','hyacinth red',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:hyacinthviolet.219','hyacinthviolet','hyacinth violet',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:hydrangeablue.186','hydrangeablue','hydrangea blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:hydrangeablue.203','hydrangeablue','hydrangea blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-m:hydrangeapink.5','hydrangeapink','hydrangea pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:hydrangeared.19','hydrangeared','hydrangea red',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:hydro.186','hydro','hydro',[83,104,120],'536878',5466232],
['nbs-iscc-m:hypermicred.11','hypermicred','hypermic red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:hyssopviolet.228','hyssopviolet','hyssop violet',[121,104,120],'796878',7956600],
['nbs-iscc-m:ibispink.26','ibispink','ibis pink',[248,131,121],'f88379',16286585],
['nbs-iscc-m:ibispink.29','ibispink','ibis pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:ibisred.47','ibisred','ibis red',[67,48,46],'43302e',4403246],
['nbs-iscc-m:iceberg.186','iceberg','iceberg',[83,104,120],'536878',5466232],
['nbs-iscc-m:immenssee.150','immenssee','immenssee',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:imperial.219','imperial','imperial',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:imperial.224','imperial','imperial',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:imperial.238','imperial','imperial',[112,41,99],'702963',7350627],
['nbs-iscc-m:imperial.242','imperial','imperial',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:imperialblue.179','imperialblue','imperial blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:imperialgreen.136','imperialgreen','imperial green',[103,146,103],'679267',6787687],
['nbs-iscc-m:imperialgreen.140','imperialgreen','imperial green',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:imperialjade.137','imperialjade','imperial jade',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:imperialpurple.215','imperialpurple','imperial purple',[85,76,105],'554c69',5590121],
['nbs-iscc-m:imperialstone.94','imperialstone','imperial stone ',[150,113,23],'967117',9859351],
['nbs-iscc-m:imperialyellow.71','imperialyellow','imperial yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:incagold.71','incagold','inca gold',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:indebaudias.229','indebaudias','indebaudias',[80,64,77],'50404d',5259341],
['nbs-iscc-m:indeblue.229','indeblue','inde blue',[80,64,77],'50404d',5259341],
['nbs-iscc-m:independence.212','independence','independence',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:indiaink.266','indiaink','india ink',[85,85,85],'555555',5592405],
['nbs-iscc-m:indian.54','indian','indian',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:indiana.254','indiana','indiana',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:indianblue.229','indianblue','indian blue',[80,64,77],'50404d',5259341],
['nbs-iscc-m:indianbrown.81','indianbrown','indian brown',[72,60,50],'483c32',4734002],
['nbs-iscc-m:indianbrown.95','indianbrown','indian brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:indianbrown.96','indianbrown','indian brown',[59,49,33],'3b3121',3879201],
['nbs-iscc-m:indianbuff.76','indianbuff','indian buff',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:indianlake.258','indianlake','indian lake',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:indianorange.34','indianorange','indian orange',[226,88,34],'e25822',14833698],
['nbs-iscc-m:indianpink.42','indianpink','indian pink',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:indianpurple.259','indianpurple','indian purple',[103,49,71],'673147',6762823],
['nbs-iscc-m:indianred.43','indianred','indian red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:indianred.40','indianred','indian red',[136,45,23],'882d17',8924439],
['nbs-iscc-m:indiansaffron.86','indiansaffron','indian saffron',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:indiantan.74','indiantan','indian tan',[153,101,21],'996515',10052885],
['nbs-iscc-m:indiantan.76','indiantan','indian tan',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:indiantan.77','indiantan','indian tan',[130,102,68],'826644',8545860],
['nbs-iscc-m:indianturquoise.164','indianturquoise','indian turquoise',[49,120,115],'317873',3242099],
['nbs-iscc-m:indianyellow.68','indianyellow','indian yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:indianyellow.71','indianyellow','indian yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:indiared.41','indiared','india red',[86,7,12],'56070c',5637900],
['nbs-iscc-m:indiaspice.57','indiaspice','india spice ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:indiatan.58','indiatan','india tan',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:indico.229','indico','indico',[80,64,77],'50404d',5259341],
['nbs-iscc-m:indicocarmine.169','indicocarmine','indico carmine',[0,119,145],'007791',30609],
['nbs-iscc-m:indigo.229','indigo','indigo',[80,64,77],'50404d',5259341],
['nbs-iscc-m:indigocarmine.169','indigocarmine','indigo carmine',[0,119,145],'007791',30609],
['nbs-iscc-m:indigoextract.169','indigoextract','indigo extract',[0,119,145],'007791',30609],
['nbs-iscc-m:indo.186','indo','indo',[83,104,120],'536878',5466232],
['nbs-iscc-m:infanta.182','infanta','infanta',[67,107,149],'436b95',4418453],
['nbs-iscc-m:infantry.186','infantry','infantry ',[83,104,120],'536878',5466232],
['nbs-iscc-m:infernalblue.201','infernalblue','infernal blue',[37,36,64],'252440',2434112],
['nbs-iscc-m:ingenue.121','ingenue','ingenue',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:ingenue.122','ingenue','ingenue',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:inkblack.192','inkblack','ink black',[81,88,94],'51585e',5331038],
['nbs-iscc-m:inkblue.187','inkblue','ink blue',[54,69,79],'36454f',3556687],
['nbs-iscc-m:intenseblue.200','intenseblue','intense blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:intenseblue.201','intenseblue','intense blue',[37,36,64],'252440',2434112],
['nbs-iscc-m:international.186','international','international',[83,104,120],'536878',5466232],
['nbs-iscc-m:invisiblegreen.165','invisiblegreen','invisible green',[0,75,73],'004b49',19273],
['nbs-iscc-m:invisiblegreen.152','invisiblegreen','invisible green',[26,36,33],'1a2421',1713185],
['nbs-iscc-m:ionianblue.187','ionianblue','ionian blue',[54,69,79],'36454f',3556687],
['nbs-iscc-m:ionianblue.188','ionianblue','ionian blue',[32,40,48],'202830',2107440],
['nbs-iscc-m:iris.185','iris','iris',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:iris.227','iris','iris',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:irisglow.233','irisglow','irisglow',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:irisgreen.137','irisgreen','iris green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:irisgreen.136','irisgreen','iris green',[103,146,103],'679267',6787687],
['nbs-iscc-m:irismauve.31','irismauve','iris mauve',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-m:irismauve.32','irismauve','iris mauve',[199,173,163],'c7ada3',13086115],
['nbs-iscc-m:irismauve.33','irismauve','iris mauve',[194,172,153],'c2ac99',12758169],
['nbs-iscc-m:iron.81','iron','iron',[72,60,50],'483c32',4734002],
['nbs-iscc-m:iron.267','iron','iron',[34,34,34],'222222',2236962],
['nbs-iscc-m:ironblue.266','ironblue','iron blue',[85,85,85],'555555',5592405],
['nbs-iscc-m:ironbrown.81','ironbrown','iron brown',[72,60,50],'483c32',4734002],
['nbs-iscc-m:ironbrown.267','ironbrown','iron brown',[34,34,34],'222222',2236962],
['nbs-iscc-m:ironbuff.76','ironbuff','iron buff',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:ironbuff.90','ironbuff','iron buff',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:ironcrocus.43','ironcrocus','iron crocus',[121,68,59],'79443b',7947323],
['nbs-iscc-m:irongrey.266','irongrey','iron grey ',[85,85,85],'555555',5592405],
['nbs-iscc-m:ironminium.19','ironminium','iron minium',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:ironminium.46','ironminium','iron minium',[103,76,71],'674c47',6769735],
['nbs-iscc-m:ironoxidered.40','ironoxidered','iron oxide red',[136,45,23],'882d17',8924439],
['nbs-iscc-m:ironoxidered.55','ironoxidered','iron oxide red',[128,70,27],'80461b',8406555],
['nbs-iscc-m:ironred.40','ironred','iron red',[136,45,23],'882d17',8924439],
['nbs-iscc-m:ironsaffron.43','ironsaffron','iron saffron',[121,68,59],'79443b',7947323],
['nbs-iscc-m:ironyellow.50','ironyellow','iron yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:ironyellow.53','ironyellow','iron yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-m:isabella.76','isabella','isabella',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:isabella.77','isabella','isabella',[130,102,68],'826644',8545860],
['nbs-iscc-m:isabella.91','isabella','isabella',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:isabella.94','isabella','isabella',[150,113,23],'967117',9859351],
['nbs-iscc-m:ispahan.187','ispahan','ispahan',[54,69,79],'36454f',3556687],
['nbs-iscc-m:italianblue.167','italianblue','italian blue',[0,133,161],'0085a1',34209],
['nbs-iscc-m:italianblue.169','italianblue','italian blue',[0,119,145],'007791',30609],
['nbs-iscc-m:italianlake.71','italianlake','italian lake',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:italianochre.54','italianochre','italian ochre',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:italianochre.57','italianochre','italian ochre',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:italianpink.83','italianpink','italian pink',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:italianpink.84','italianpink','italian pink',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:italianpink.86','italianpink','italian pink',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:italianpink.87','italianpink','italian pink',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:italianstraw.89','italianstraw','italian straw',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:italianstraw.90','italianstraw','italian straw',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:ivorybrown.95','ivorybrown','ivory brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:ivorybrown.96','ivorybrown','ivory brown',[59,49,33],'3b3121',3879201],
['nbs-iscc-m:ivorywhite.89','ivorywhite','ivory white',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:ivoryyellow.89','ivoryyellow','ivory [yellow] ',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:ivygreen.107','ivygreen','ivy [green] ',[102,93,30],'665d1e',6708510],
['nbs-iscc-m:ivygreen.110','ivygreen','ivy [green] ',[91,88,66],'5b5842',5986370],
['nbs-iscc-m:jacarandabrown.81','jacarandabrown','jacaranda brown',[72,60,50],'483c32',4734002],
['nbs-iscc-m:jacinthe.53','jacinthe','jacinthe',[217,144,88],'d99058',14258264],
['nbs-iscc-m:jackrabbit.266','jackrabbit','jack rabbit',[85,85,85],'555555',5592405],
['nbs-iscc-m:jackrose.11','jackrose','jack rose',[190,0,50],'be0032',12451890],
['nbs-iscc-m:jacqueminot.15','jacqueminot','jacqueminot',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:jadesheen.136','jadesheen','jadesheen',[103,146,103],'679267',6787687],
['nbs-iscc-m:jadesheen.137','jadesheen','jadesheen',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:jadesheen.150','jadesheen','jadesheen',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:jaffaorange.37','jaffaorange','jaffa orange',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:jaffaorange.53','jaffaorange','jaffa orange',[217,144,88],'d99058',14258264],
['nbs-iscc-m:jaffi.113','jaffi','jaffi',[87,85,76],'57554c',5723468],
['nbs-iscc-m:jalapa.55','jalapa','jalapa',[128,70,27],'80461b',8406555],
['nbs-iscc-m:japanblue.183','japanblue','japan blue',[0,48,78],'00304e',12366],
['nbs-iscc-m:japanearth.58','japanearth','japan earth',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:japaneseblue.150','japaneseblue','japanese blue',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:japanesegreen.150','japanesegreen','japanese green',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:japanesered.40','japanesered','japanese red',[136,45,23],'882d17',8924439],
['nbs-iscc-m:japaneseyellow.25','japaneseyellow','japanese yellow',[255,183,165],'ffb7a5',16758693],
['nbs-iscc-m:japaneseyellow.35','japaneseyellow','japanese yellow',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:japanrose.29','japanrose','japan rose',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:japanrose.53','japanrose','japan rose',[217,144,88],'d99058',14258264],
['nbs-iscc-m:japonica.19','japonica','japonica',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:jasmine.83','jasmine','jasmine',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:jasmine.86','jasmine','jasmine',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:jasper.152','jasper','jasper',[26,36,33],'1a2421',1713185],
['nbs-iscc-m:jaspergreen.145','jaspergreen','jasper green',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:jasperpink.26','jasperpink','jasper pink',[248,131,121],'f88379',16286585],
['nbs-iscc-m:jasperred.15','jasperred','jasper red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:java.59','java','java',[66,37,24],'422518',4334872],
['nbs-iscc-m:javabrown.62','javabrown','java brown',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:javelgreen.102','javelgreen','javel green',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:jayblue.182','jayblue','jay [blue] ',[67,107,149],'436b95',4418453],
['nbs-iscc-m:jeanbart.212','jeanbart','jean bart',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:jewspitch.81','jewspitch','jew\'s pitch ',[72,60,50],'483c32',4734002],
['nbs-iscc-m:jockey.15','jockey','jockey',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:jonquilyellow.70','jonquilyellow','jonquil [yellow] ',[251,201,127],'fbc97f',16501119],
['nbs-iscc-m:jonquilyellow.71','jonquilyellow','jonquil [yellow] ',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:jonquilyellow.73','jonquilyellow','jonquil [yellow] ',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:jonquilyellow.86','jonquilyellow','jonquil [yellow] ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:jonquilyellow.87','jonquilyellow','jonquil [yellow] ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:josephine.57','josephine','josephine',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:jouvenceblue.164','jouvenceblue','jouvence blue',[49,120,115],'317873',3242099],
['nbs-iscc-m:jouvenceblue.173','jouvenceblue','jouvence blue',[54,117,136],'367588',3569032],
['nbs-iscc-m:judee.228','judee','judee',[121,104,120],'796878',7956600],
['nbs-iscc-m:junglegreen.147','junglegreen','jungle green',[28,53,45],'1c352d',1848621],
['nbs-iscc-m:junglegreen.152','junglegreen','jungle green',[26,36,33],'1a2421',1713185],
['nbs-iscc-m:juniper.151','juniper','juniper',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:kabistan.151','kabistan','kabistan',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:kaffa.46','kaffa','kaffa',[103,76,71],'674c47',6769735],
['nbs-iscc-m:kaiserbrown.55','kaiserbrown','kaiser brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:kangaroo.113','kangaroo','kangaroo',[87,85,76],'57554c',5723468],
['nbs-iscc-m:kara.164','kara','kara',[49,120,115],'317873',3242099],
['nbs-iscc-m:karadagh.267','karadagh','kara dagh',[34,34,34],'222222',2236962],
['nbs-iscc-m:kashabeige.112','kashabeige','kasha-beige',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:kashabeige.265','kashabeige','kasha-beige',[132,132,130],'848482',8684674],
['nbs-iscc-m:kashan.169','kashan','kashan',[0,119,145],'007791',30609],
['nbs-iscc-m:kashmir.137','kashmir','kashmir',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:kashmir.145','kashmir','kashmir',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:kashmir.150','kashmir','kashmir',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:kazak.43','kazak','kazak',[121,68,59],'79443b',7947323],
['nbs-iscc-m:kentuckygreen.107','kentuckygreen','kentucky green',[102,93,30],'665d1e',6708510],
['nbs-iscc-m:kermanshah.61','kermanshah','kermanshah',[99,81,71],'635147',6508871],
['nbs-iscc-m:kermes.11','kermes','kermes',[190,0,50],'be0032',12451890],
['nbs-iscc-m:kermesberries.11','kermesberries','kermes berries',[190,0,50],'be0032',12451890],
['nbs-iscc-m:kesseleryellow.83','kesseleryellow','kesseler yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:kesseleryellow.84','kesseleryellow','kesseler yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:kesseleryellow.86','kesseleryellow','kesseler yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:kesseleryellow.87','kesseleryellow','kesseler yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:kettledrum.43','kettledrum','kettledrum',[121,68,59],'79443b',7947323],
['nbs-iscc-m:khaki.76','khaki','khaki',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:khaki.77','khaki','khaki',[130,102,68],'826644',8545860],
['nbs-iscc-m:khiva.38','khiva','khiva',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:kildaregreen.120','kildaregreen','kildare green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:killarneygreen.136','killarneygreen','killarney [green] ',[103,146,103],'679267',6787687],
['nbs-iscc-m:killarneygreen.145','killarneygreen','killarney [green] ',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:kinemared.12','kinemared','kinema red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:kingfisher.183','kingfisher','kingfisher ',[0,48,78],'00304e',12366],
['nbs-iscc-m:kingsblue.181','kingsblue','king\'s blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:kingsblue.179','kingsblue','king\'s blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:kingsblue.182','kingsblue','king\'s blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:kingsblue.182','kingsblue','king\'s blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:kingsyellow.83','kingsyellow','king\'s yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:kingsyellow.84','kingsyellow','king\'s yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:kingsyellow.86','kingsyellow','king\'s yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:kingsyellow.97','kingsyellow','king\'s yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:kirchbergergreen.140','kirchbergergreen','kirchberger green',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:kiskilim.58','kiskilim','kis kilim',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:kobe.40','kobe','kobe',[136,45,23],'882d17',8924439],
['nbs-iscc-m:kolinsky.59','kolinsky','kolinsky',[66,37,24],'422518',4334872],
['nbs-iscc-m:korea.38','korea','korea',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:korea.43','korea','korea',[121,68,59],'79443b',7947323],
['nbs-iscc-m:kork.258','kork','kork',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:korkir.258','korkir','korkir',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:kremlin.165','kremlin','kremlin',[0,75,73],'004b49',19273],
['nbs-iscc-m:kronbergsgreen.125','kronbergsgreen','kronberg\'s green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:kurdistan.187','kurdistan','kurdistan',[54,69,79],'36454f',3556687],
['nbs-iscc-m:kurdistan.193','kurdistan','kurdistan',[32,36,40],'202428',2106408],
['nbs-iscc-m:kurdistan.229','kurdistan','kurdistan',[80,64,77],'50404d',5259341],
['nbs-iscc-m:kurdistan.235','kurdistan','kurdistan',[36,33,36],'242124',2367780],
['nbs-iscc-m:kyoto.26','kyoto','kyoto',[248,131,121],'f88379',16286585],
['nbs-iscc-m:kyoto.50','kyoto','kyoto',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:kyoto.53','kyoto','kyoto',[217,144,88],'d99058',14258264],
['nbs-iscc-m:labrador.183','labrador','labrador',[0,48,78],'00304e',12366],
['nbs-iscc-m:labrador.187','labrador','labrador',[54,69,79],'36454f',3556687],
['nbs-iscc-m:laclake.258','laclake','lac lake',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:lacquerred.38','lacquerred','lacquer red',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:laeliapink.251','laeliapink','laelia pink',[193,126,145],'c17e91',12680849],
['nbs-iscc-m:laeliapink.261','laeliapink','laelia pink',[175,134,142],'af868e',11503246],
['nbs-iscc-m:lafrancepink.5','lafrancepink','la france pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:lagoon.163','lagoon','lagoon',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:lake.11','lake','lake',[190,0,50],'be0032',12451890],
['nbs-iscc-m:lake.12','lake','lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:lakeblue.164','lakeblue','lake blue',[49,120,115],'317873',3242099],
['nbs-iscc-m:lama.81','lama','lama',[72,60,50],'483c32',4734002],
['nbs-iscc-m:lambertsblue.182','lambertsblue','lambert\'s blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:lapis.164','lapis','lapis',[49,120,115],'317873',3242099],
['nbs-iscc-m:lariat.57','lariat','lariat',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:lariat.76','lariat','lariat',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:lark.90','lark','lark',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:lark.91','lark','lark',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:lark.94','lark','lark',[150,113,23],'967117',9859351],
['nbs-iscc-m:larkspur.173','larkspur','larkspur',[54,117,136],'367588',3569032],
['nbs-iscc-m:latericeous.55','latericeous','latericeous',[128,70,27],'80461b',8406555],
['nbs-iscc-m:latericeous.58','latericeous','latericeous',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:lateritious.55','lateritious','lateritious',[128,70,27],'80461b',8406555],
['nbs-iscc-m:lateritious.58','lateritious','lateritious',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:latoun.87','latoun','latoun',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:latten.87','latten','latten',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:laundryblue.179','laundryblue','laundry blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:laundryblue.182','laundryblue','laundry blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:laurelgreen.106','laurelgreen','laurel green',[134,126,54],'867e36',8814134],
['nbs-iscc-m:laureloak.43','laureloak','laurel oak',[121,68,59],'79443b',7947323],
['nbs-iscc-m:laurelpink.3','laurelpink','laurel pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:laurelpink.15','laurelpink','laurel pink',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:lava.267','lava','lava',[34,34,34],'222222',2236962],
['nbs-iscc-m:lavallire.228','lavallire','la vallire ',[121,104,120],'796878',7956600],
['nbs-iscc-m:lavender.227','lavender','lavender',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:lawngreen.120','lawngreen','lawn green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:lead.234','lead','lead',[93,85,91],'5d555b',6116699],
['nbs-iscc-m:lead.266','lead','lead',[85,85,85],'555555',5592405],
['nbs-iscc-m:leadgrey.109','leadgrey','lead grey ',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:leadochre.89','leadochre','lead ochre ',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:leadville.234','leadville','leadville',[93,85,91],'5d555b',6116699],
['nbs-iscc-m:leafmold.59','leafmold','leaf mold',[66,37,24],'422518',4334872],
['nbs-iscc-m:leafred.11','leafred','leaf red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:leather.54','leather','leather',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:leatherbrown.58','leatherbrown','leather brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:leatherlake.51','leatherlake','leather lake',[190,101,22],'be6516',12477718],
['nbs-iscc-m:leatherlake.54','leatherlake','leather lake',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:leatherlake.55','leatherlake','leather lake',[128,70,27],'80461b',8406555],
['nbs-iscc-m:leekgreen.106','leekgreen','leek green',[134,126,54],'867e36',8814134],
['nbs-iscc-m:leekgreen.120','leekgreen','leek green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:leghorn.89','leghorn','leghorn',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:leghorn.90','leghorn','leghorn',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:legionblue.183','legionblue','legion blue',[0,48,78],'00304e',12366],
['nbs-iscc-m:leitchsblue.176','leitchsblue','leitch\'s blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-m:leithnersblue.169','leithnersblue','leithner\'s blue',[0,119,145],'007791',30609],
['nbs-iscc-m:lemmianearth.43','lemmianearth','lemmian earth',[121,68,59],'79443b',7947323],
['nbs-iscc-m:lemnianruddle.43','lemnianruddle','lemnian ruddle',[121,68,59],'79443b',7947323],
['nbs-iscc-m:lemnos.43','lemnos','lemnos',[121,68,59],'79443b',7947323],
['nbs-iscc-m:lemonchrome.83','lemonchrome','lemon chrome',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:lemonyellow.83','lemonyellow','lemon yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:lemonyellow.84','lemonyellow','lemon yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:lemonyellow.86','lemonyellow','lemon yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:lemonyellow.87','lemonyellow','lemon yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:lettucegreen.120','lettucegreen','lettuce green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:levantred.15','levantred','levant red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:leydenblue.169','leydenblue','leyden blue',[0,119,145],'007791',30609],
['nbs-iscc-m:liberia.62','liberia','liberia',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:liberty.178','liberty','liberty',[0,103,165],'0067a5',26533],
['nbs-iscc-m:liberty.196','liberty','liberty',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:libertyblue.200','libertyblue','liberty blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:libertygreen.120','libertygreen','liberty green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:lichen.127','lichen','lichen',[81,87,68],'515744',5330756],
['nbs-iscc-m:lichengreen.154','lichengreen','lichen green',[178,190,181],'b2beb5',11714229],
['nbs-iscc-m:lido.94','lido','lido',[150,113,23],'967117',9859351],
['nbs-iscc-m:lierre.107','lierre','lierre',[102,93,30],'665d1e',6708510],
['nbs-iscc-m:lierre.110','lierre','lierre',[91,88,66],'5b5842',5986370],
['nbs-iscc-m:lightblue1.121','lightblue1','light blue 1',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:lightblue2lightblue3.148','lightblue2lightblue3','light blue 2, light blue 3',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-m:lightblue2lightblue3.184','lightblue2lightblue3','light blue 2, light blue 3',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-m:lightblue4.184','lightblue4','light blue 4',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-m:lightblue6.185','lightblue6','light blue 6',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:lightbluelightblue5.148','lightbluelightblue5','light blue,, light blue 5',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-m:lightbrunswickgreen.137','lightbrunswickgreen','light brunswick green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:lightchromegreen.137','lightchromegreen','light chrome green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:lightchromeyellow.84','lightchromeyellow','light chrome yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:lightgrge.112','lightgrge','light grge',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:lightgunmetal.266','lightgunmetal','light gunmetal',[85,85,85],'555555',5592405],
['nbs-iscc-m:lightred.37','lightred','light red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:lightstone.90','lightstone','light stone',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:lightstone.91','lightstone','light stone',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:lightwedgwoodblue.186','lightwedgwoodblue','light wedgwood [blue] ',[83,104,120],'536878',5466232],
['nbs-iscc-m:lilac.5','lilac','lilac',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:lilac.18','lilac','lilac',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:lilac.227','lilac','lilac',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:lilaceous.5','lilaceous','lilaceous',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:lilaceous.18','lilaceous','lilaceous',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:lilacgray.33','lilacgray','lilac gray',[194,172,153],'c2ac99',12758169],
['nbs-iscc-m:lilacgray.93','lilacgray','lilac gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:lilas.5','lilas','lilas',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:lilas.18','lilas','lilas',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:lilium.223','lilium','lilium',[134,96,142],'86608e',8806542],
['nbs-iscc-m:lilium.241','lilium','lilium',[145,92,131],'915c83',9526403],
['nbs-iscc-m:lilygreen.120','lilygreen','lily green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:lilygreen.156','lilygreen','lily green',[78,87,85],'4e5755',5134165],
['nbs-iscc-m:limawood.15','limawood','limawood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:limawood.19','limawood','limawood',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:limeblue.182','limeblue','lime blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:limegreen.105','limegreen','lime green',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:limegreen.106','limegreen','lime green',[134,126,54],'867e36',8814134],
['nbs-iscc-m:limestone.64','limestone','limestone',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:limestone.266','limestone','limestone',[85,85,85],'555555',5592405],
['nbs-iscc-m:limeyellow.87','limeyellow','lime [yellow] ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:limeyellow.90','limeyellow','lime [yellow] ',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:limoges.178','limoges','limoges',[0,103,165],'0067a5',26533],
['nbs-iscc-m:limoges.182','limoges','limoges',[67,107,149],'436b95',4418453],
['nbs-iscc-m:lincolngreen.125','lincolngreen','lincoln green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:lincolnred.11','lincolnred','lincoln red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:lindengreen.102','lindengreen','linden green',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:lindenyellow.102','lindenyellow','linden yellow',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:linoleumbrown.77','linoleumbrown','linoleum brown',[130,102,68],'826644',8545860],
['nbs-iscc-m:lint.109','lint','lint',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:lintwhite.89','lintwhite','lint-white',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:lintwhite.90','lintwhite','lint-white',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:lion.76','lion','lion',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:lion.77','lion','lion',[130,102,68],'826644',8545860],
['nbs-iscc-m:liontawny.76','liontawny','lion tawny',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:liontawny.77','liontawny','lion tawny',[130,102,68],'826644',8545860],
['nbs-iscc-m:liqueurgreen.101','liqueurgreen','liqueur green',[234,230,121],'eae679',15394425],
['nbs-iscc-m:liqueurgreen.102','liqueurgreen','liqueur green',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:liseranpurple.237','liseranpurple','liseran purple',[158,79,136],'9e4f88',10375048],
['nbs-iscc-m:lithopurple.223','lithopurple','litho purple',[134,96,142],'86608e',8806542],
['nbs-iscc-m:lithopurple.224','lithopurple','litho purple',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:lithopurple.228','lithopurple','litho purple',[121,104,120],'796878',7956600],
['nbs-iscc-m:liver.46','liver','liver',[103,76,71],'674c47',6769735],
['nbs-iscc-m:liverbrown.46','liverbrown','liver brown',[103,76,71],'674c47',6769735],
['nbs-iscc-m:livermaroon.46','livermaroon','liver maroon',[103,76,71],'674c47',6769735],
['nbs-iscc-m:livid.234','livid','livid',[93,85,91],'5d555b',6116699],
['nbs-iscc-m:livid.266','livid','livid',[85,85,85],'555555',5592405],
['nbs-iscc-m:lividbrown.19','lividbrown','livid brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:lividpink.8','lividpink','livid pink',[196,174,173],'c4aead',12889773],
['nbs-iscc-m:lividpurple.245','lividpurple','livid purple',[131,100,121],'836479',8610937],
['nbs-iscc-m:lividviolet.18','lividviolet','livid violet',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:lizardbronze.106','lizardbronze','lizard bronze',[134,126,54],'867e36',8814134],
['nbs-iscc-m:lizardgreen.145','lizardgreen','lizard [green] ',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:loam.64','loam','loam',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:lobeliablue.207','lobeliablue','lobelia [blue] ',[96,78,151],'604e97',6311575],
['nbs-iscc-m:lobeliaviolet.228','lobeliaviolet','lobelia violet',[121,104,120],'796878',7956600],
['nbs-iscc-m:lobster.36','lobster','lobster',[170,56,30],'aa381e',11155486],
['nbs-iscc-m:lobster.38','lobster','lobster',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:locarnogreen.145','locarnogreen','locarno green',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:loganberry.224','loganberry','loganberry',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:logcabin.64','logcabin','log cabin',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:logwood.230','logwood','logwood',[41,30,41],'291e29',2694697],
['nbs-iscc-m:logwood.235','logwood','logwood',[36,33,36],'242124',2367780],
['nbs-iscc-m:logwoodblue.192','logwoodblue','logwood blue',[81,88,94],'51585e',5331038],
['nbs-iscc-m:lokao.135','lokao','lo-kao',[147,197,146],'93c592',9684370],
['nbs-iscc-m:lokao.144','lokao','lo-kao',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:londonbrown.47','londonbrown','london brown',[67,48,46],'43302e',4403246],
['nbs-iscc-m:londonsmoke.267','londonsmoke','london smoke',[34,34,34],'222222',2236962],
['nbs-iscc-m:longbeach.76','longbeach','long beach ',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:longchamps.57','longchamps','longchamps',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:longchamps.60','longchamps','longchamps',[149,128,112],'958070',9797744],
['nbs-iscc-m:lotus.6','lotus','lotus',[192,128,129],'c08081',12615809],
['nbs-iscc-m:lotus.18','lotus','lotus',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:lotus.30','lotus','lotus',[196,131,121],'c48379',12878713],
['nbs-iscc-m:louisphilippe.199','louisphilippe','louis philippe',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:loutre.95','loutre','loutre',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:loutre.96','loutre','loutre',[59,49,33],'3b3121',3879201],
['nbs-iscc-m:louvain.186','louvain','louvain',[83,104,120],'536878',5466232],
['nbs-iscc-m:louvain.192','louvain','louvain',[81,88,94],'51585e',5331038],
['nbs-iscc-m:lovebird.117','lovebird','love bird',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-m:loveinamist.172','loveinamist','love-in-a-mist',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:lucerneblue.199','lucerneblue','lucerne blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:luciole.191','luciole','luciole',[129,135,139],'81878b',8488843],
['nbs-iscc-m:luckystone.185','luckystone','lucky stone',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:lumiereblue.163','lumiereblue','lumiere blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:lumieregreen.119','lumieregreen','lumiere green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-m:lupine.181','lupine','lupine',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:lupine.199','lupine','lupine',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:luteous.101','luteous','luteous',[234,230,121],'eae679',15394425],
['nbs-iscc-m:luteous.102','luteous','luteous',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:luxor.196','luxor','luxor',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:lyonsblue.178','lyonsblue','lyons blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:macaroon.76','macaroon','macaroon',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:madder.12','madder','madder',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:madder.15','madder','madder',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:madder.35','madder','madder',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:madder.37','madder','madder',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:madderblue.214','madderblue','madder blue',[150,144,171],'9690ab',9867435],
['nbs-iscc-m:madderblue.215','madderblue','madder blue',[85,76,105],'554c69',5590121],
['nbs-iscc-m:madderbrown.43','madderbrown','madder brown',[121,68,59],'79443b',7947323],
['nbs-iscc-m:maddercarmine.15','maddercarmine','madder carmine',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:maddercarmine.258','maddercarmine','madder carmine',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:madderindianred.43','madderindianred','madder indian red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:madderlake.12','madderlake','madder lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:madderlake.255','madderlake','madder lake',[179,68,108],'b3446c',11748460],
['nbs-iscc-m:madderpink.255','madderpink','madder pink',[179,68,108],'b3446c',11748460],
['nbs-iscc-m:madderred.12','madderred','madder red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:madderred.15','madderred','madder red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:madderred.35','madderred','madder red',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:madderred.37','madderred','madder red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:madderviolet.224','madderviolet','madder violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:madelineblue.204','madelineblue','madeline blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:madonna.182','madonna','madonna',[67,107,149],'436b95',4418453],
['nbs-iscc-m:madonna.186','madonna','madonna',[83,104,120],'536878',5466232],
['nbs-iscc-m:madrid.61','madrid','madrid ',[99,81,71],'635147',6508871],
['nbs-iscc-m:madrid.80','madrid','madrid ',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:madura.77','madura','madura',[130,102,68],'826644',8545860],
['nbs-iscc-m:magenta.256','magenta','magenta',[120,24,74],'78184a',7870538],
['nbs-iscc-m:magentarose.256','magentarose','magenta rose',[120,24,74],'78184a',7870538],
['nbs-iscc-m:magnolia.6','magnolia','magnolia',[192,128,129],'c08081',12615809],
['nbs-iscc-m:magnolia.19','magnolia','magnolia',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:magnolia.30','magnolia','magnolia',[196,131,121],'c48379',12878713],
['nbs-iscc-m:mahogany.43','mahogany','mahogany',[121,68,59],'79443b',7947323],
['nbs-iscc-m:mahoganybrown.43','mahoganybrown','mahogany brown',[121,68,59],'79443b',7947323],
['nbs-iscc-m:mahoganyred.43','mahoganyred','mahogany red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:maidensblush.42','maidensblush','maiden\'s blush ',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:maintenon.37','maintenon','maintenon',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:maise.71','maise','maise',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:maise.73','maise','maise',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:maise.87','maise','maise',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:majolica.183','majolica','majolica',[0,48,78],'00304e',12366],
['nbs-iscc-m:majolicablue.183','majolicablue','majolica blue',[0,48,78],'00304e',12366],
['nbs-iscc-m:majolicablue.201','majolicablue','majolica blue',[37,36,64],'252440',2434112],
['nbs-iscc-m:majolicablue.204','majolicablue','majolica blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:majolicaearth.40','majolicaearth','majolica earth',[136,45,23],'882d17',8924439],
['nbs-iscc-m:malabar.58','malabar','malabar',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:malacca.76','malacca','malacca',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:malacca.90','malacca','malacca',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:malachitegreen.136','malachitegreen','malachite green',[103,146,103],'679267',6787687],
['nbs-iscc-m:malaga.16','malaga','malaga',[114,47,55],'722f37',7483191],
['nbs-iscc-m:malaga.41','malaga','malaga',[86,7,12],'56070c',5637900],
['nbs-iscc-m:malay.58','malay','malay',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:mallard.174','mallard','mallard',[0,73,88],'004958',18776],
['nbs-iscc-m:mallowpink.250','mallowpink','mallow pink',[213,151,174],'d597ae',13997998],
['nbs-iscc-m:mallowpurple.258','mallowpurple','mallow purple',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:mallowred.258','mallowred','mallow red',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:malmaison.86','malmaison','malmaison',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:malmaison.87','malmaison','malmaison',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:malmaisonrose.254','malmaisonrose','malmaison rose',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:malmaisonrose.255','malmaisonrose','malmaison rose',[179,68,108],'b3446c',11748460],
['nbs-iscc-m:manchu.58','manchu','manchu',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:mandalay.61','mandalay','mandalay',[99,81,71],'635147',6508871],
['nbs-iscc-m:mandalay.62','mandalay','mandalay',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:mandarinorange.50','mandarinorange','mandarin orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:mandarinred.35','mandarinred','mandarin red',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:manganesebrown.58','manganesebrown','manganese brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:manganesevelvetbrown.77','manganesevelvetbrown','manganese velvet brown',[130,102,68],'826644',8545860],
['nbs-iscc-m:manganesevelvetbrown.78','manganesevelvetbrown','manganese velvet brown',[75,54,33],'4b3621',4929057],
['nbs-iscc-m:manganeseviolet.218','manganeseviolet','manganese violet',[135,86,146],'875692',8869522],
['nbs-iscc-m:mango.54','mango','mango',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:mango.57','mango','mango',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:manila.76','manila','manila ',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:manon.60','manon','manon',[149,128,112],'958070',9797744],
['nbs-iscc-m:manzanita.43','manzanita','manzanita ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:maple.90','maple','maple',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:maplesugar.77','maplesugar','maple sugar',[130,102,68],'826644',8545860],
['nbs-iscc-m:maracaibo.59','maracaibo','maracaibo',[66,37,24],'422518',4334872],
['nbs-iscc-m:marathon.50','marathon','marathon',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:marblegreen.157','marblegreen','marble green ',[30,35,33],'1e2321',1975073],
['nbs-iscc-m:margueriteyellow.121','margueriteyellow','marguerite yellow',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:marieantoinette.18','marieantoinette','marie antoinette',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:marieantoinette.19','marieantoinette','marie antoinette',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:marineblue.183','marineblue','marine [blue] ',[0,48,78],'00304e',12366],
['nbs-iscc-m:marineblue.200','marineblue','marine [blue] ',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:marineblue.201','marineblue','marine [blue] ',[37,36,64],'252440',2434112],
['nbs-iscc-m:marinecorps.204','marinecorps','marine corps',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:marinegreen.146','marinegreen','marine green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:marinegreen.151','marinegreen','marine green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:maris.151','maris','maris',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:maris.165','maris','maris',[0,75,73],'004b49',19273],
['nbs-iscc-m:marmora.164','marmora','marmora',[49,120,115],'317873',3242099],
['nbs-iscc-m:marocain.61','marocain','marocain',[99,81,71],'635147',6508871],
['nbs-iscc-m:marone.61','marone','marone',[99,81,71],'635147',6508871],
['nbs-iscc-m:maroon.16','maroon','maroon',[114,47,55],'722f37',7483191],
['nbs-iscc-m:maroon.41','maroon','maroon',[86,7,12],'56070c',5637900],
['nbs-iscc-m:maroon.44','maroon','maroon',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-m:marron.61','marron','marron',[99,81,71],'635147',6508871],
['nbs-iscc-m:marrone.61','marrone','marrone ',[99,81,71],'635147',6508871],
['nbs-iscc-m:marronglac.58','marronglac','marron glac',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:marsala.43','marsala','marsala',[121,68,59],'79443b',7947323],
['nbs-iscc-m:marsbrown.58','marsbrown','mars brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:marshrose.19','marshrose','marsh rose',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:marsorange.37','marsorange','mars orange',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:marsred.38','marsred','mars red',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:marsviolet.41','marsviolet','mars violet',[86,7,12],'56070c',5637900],
['nbs-iscc-m:marsyellow.50','marsyellow','mars yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:marsyellow.53','marsyellow','mars yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-m:martinique.54','martinique','martinique',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:martinique.57','martinique','martinique',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:martiusyellow.101','martiusyellow','martius yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-m:marygold.50','marygold','marygold',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:mascara.43','mascara','mascara',[121,68,59],'79443b',7947323],
['nbs-iscc-m:mascot.187','mascot','mascot',[54,69,79],'36454f',3556687],
['nbs-iscc-m:mascot.188','mascot','mascot',[32,40,48],'202830',2107440],
['nbs-iscc-m:mascot.201','mascot','mascot',[37,36,64],'252440',2434112],
['nbs-iscc-m:mascot.204','mascot','mascot',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:massicotyellow.89','massicotyellow','massicot [yellow] ',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:mastcolour.54','mastcolour','mast colour',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:mastic.94','mastic','mastic',[150,113,23],'967117',9859351],
['nbs-iscc-m:matelot.179','matelot','matelot',[0,65,106],'00416a',16746],
['nbs-iscc-m:matrix.164','matrix','matrix',[49,120,115],'317873',3242099],
['nbs-iscc-m:mauveblush.33','mauveblush','mauve blush ',[194,172,153],'c2ac99',12758169],
['nbs-iscc-m:mauveblush.79','mauveblush','mauve blush ',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:mauveblush.93','mauveblush','mauve blush ',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:mauvecastor.23','mauvecastor','mauve castor',[92,80,79],'5c504f',6049871],
['nbs-iscc-m:mauvedust.23','mauvedust','mauve dust',[92,80,79],'5c504f',6049871],
['nbs-iscc-m:mauveglow.18','mauveglow','mauveglow',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:mauverose.16','mauverose','mauverose',[114,47,55],'722f37',7483191],
['nbs-iscc-m:mauvetaupe.23','mauvetaupe','mauve taupe',[92,80,79],'5c504f',6049871],
['nbs-iscc-m:mauvette.227','mauvette','mauvette',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:mauvewood.43','mauvewood','mauvewood',[121,68,59],'79443b',7947323],
['nbs-iscc-m:mauvewood.46','mauvewood','mauvewood',[103,76,71],'674c47',6769735],
['nbs-iscc-m:mavis.79','mavis','mavis',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:maya.54','maya','maya',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:maya.57','maya','maya',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:mayfairtan.57','mayfairtan','mayfair tan',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:mayfairtan.76','mayfairtan','mayfair tan',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:mayfairtan.77','mayfairtan','mayfair tan',[130,102,68],'826644',8545860],
['nbs-iscc-m:mayflower.15','mayflower','mayflower',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:mazarineblue.197','mazarineblue','mazarine blue',[39,36,88],'272458',2565208],
['nbs-iscc-m:meadowbrook.151','meadowbrook','meadowbrook',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:meadowbrook.165','meadowbrook','meadowbrook',[0,75,73],'004b49',19273],
['nbs-iscc-m:meadowgrass.137','meadowgrass','meadowgrass',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:meadowgrass.151','meadowgrass','meadowgrass',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:meadowgreen.137','meadowgreen','meadow [green] ',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:meadowlark.80','meadowlark','meadowlark',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:meadowpink.258','meadowpink','meadowpink',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:meadowpink.262','meadowpink','meadowpink',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:meadowsweet.19','meadowsweet','meadowsweet',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:meadowviolet.218','meadowviolet','meadow violet',[135,86,146],'875692',8869522],
['nbs-iscc-m:meadowviolet.237','meadowviolet','meadow violet',[158,79,136],'9e4f88',10375048],
['nbs-iscc-m:mecca.43','mecca','mecca',[121,68,59],'79443b',7947323],
['nbs-iscc-m:mecca.58','mecca','mecca',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:medalbronze.77','medalbronze','medal bronze',[130,102,68],'826644',8545860],
['nbs-iscc-m:medalbronze.94','medalbronze','medal bronze',[150,113,23],'967117',9859351],
['nbs-iscc-m:mediciblue.191','mediciblue','medici blue',[129,135,139],'81878b',8488843],
['nbs-iscc-m:mediterranian.164','mediterranian','mediterranian',[49,120,115],'317873',3242099],
['nbs-iscc-m:mediumchromegreen.137','mediumchromegreen','medium chrome green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:mediumchromegreen.146','mediumchromegreen','medium chrome green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:meerschaum.79','meerschaum','meerschaum',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:mehal.58','mehal','mehal',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:melilot.241','melilot','melilot',[145,92,131],'915c83',9526403],
['nbs-iscc-m:meline.87','meline','meline',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:mellomauve.18','mellomauve','mello-mauve',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:mellowglow.71','mellowglow','mellowglow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:melon.26','melon','melon',[248,131,121],'f88379',16286585],
['nbs-iscc-m:mephisto.15','mephisto','mephisto ',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:merida.57','merida','merida ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:merida.76','merida','merida ',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:merida.77','merida','merida ',[130,102,68],'826644',8545860],
['nbs-iscc-m:merle.174','merle','merle ',[0,73,88],'004958',18776],
['nbs-iscc-m:mermaid.122','mermaid','mermaid ',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:mesa.58','mesa','mesa ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:metal.264','metal','metal ',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:metalbrown.43','metalbrown','metal brown ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:metalbrown.46','metalbrown','metal brown ',[103,76,71],'674c47',6769735],
['nbs-iscc-m:metallicgreen.107','metallicgreen','metallic green ',[102,93,30],'665d1e',6708510],
['nbs-iscc-m:metallicgrey.233','metallicgrey','metallic grey ',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:mexican.54','mexican','mexican ',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:mexican.74','mexican','mexican ',[153,101,21],'996515',10052885],
['nbs-iscc-m:mexicanred.54','mexicanred','mexican red ',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:mexicanred.57','mexicanred','mexican red ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:mexico.43','mexico','mexico ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:mexico.55','mexico','mexico ',[128,70,27],'80461b',8406555],
['nbs-iscc-m:miamisand.79','miamisand','miami sand ',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:miamisand.94','miamisand','miami sand ',[150,113,23],'967117',9859351],
['nbs-iscc-m:michigan.165','michigan','michigan ',[0,75,73],'004b49',19273],
['nbs-iscc-m:microclinegreen.148','microclinegreen','microcline green ',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-m:middlebrunswickgreen.137','middlebrunswickgreen','middle brunswick green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:middlestone.91','middlestone','middle stone ',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:middy.183','middy','middy ',[0,48,78],'00304e',12366],
['nbs-iscc-m:midnight.229','midnight','midnight ',[80,64,77],'50404d',5259341],
['nbs-iscc-m:midnightsun.35','midnightsun','midnight sun ',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:mignon.223','mignon','mignon ',[134,96,142],'86608e',8806542],
['nbs-iscc-m:mignonettegreen.106','mignonettegreen','mignonette [green] ',[134,126,54],'867e36',8814134],
['nbs-iscc-m:mignongreen.120','mignongreen','mignon green ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:mikado.35','mikado','mikado ',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:mikado.50','mikado','mikado ',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:mikadobrown.58','mikadobrown','mikado brown ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:mikadoorange.53','mikadoorange','mikado orange ',[217,144,88],'d99058',14258264],
['nbs-iscc-m:milanoblue.186','milanoblue','milano blue ',[83,104,120],'536878',5466232],
['nbs-iscc-m:milkwhite.92','milkwhite','milk white ',[240,234,214],'f0ead6',15788758],
['nbs-iscc-m:milwaukeebrick.73','milwaukeebrick','milwaukee brick ',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:milwaukeebrick.89','milwaukeebrick','milwaukee brick ',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:mimosa.86','mimosa','mimosa ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:mindoro.57','mindoro','mindoro ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:mineralbister.58','mineralbister','mineral bister ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:mineralblue.182','mineralblue','mineral blue ',[67,107,149],'436b95',4418453],
['nbs-iscc-m:mineralblue.170','mineralblue','mineral blue ',[46,132,149],'2e8495',3048597],
['nbs-iscc-m:mineralblue.182','mineralblue','mineral blue ',[67,107,149],'436b95',4418453],
['nbs-iscc-m:mineralgreen.136','mineralgreen','mineral green ',[103,146,103],'679267',6787687],
['nbs-iscc-m:mineralgrey.154','mineralgrey','mineral grey ',[178,190,181],'b2beb5',11714229],
['nbs-iscc-m:mineralorange.34','mineralorange','mineral orange ',[226,88,34],'e25822',14833698],
['nbs-iscc-m:mineralpitch.81','mineralpitch','mineral pitch ',[72,60,50],'483c32',4734002],
['nbs-iscc-m:mineralpurple.41','mineralpurple','mineral purple ',[86,7,12],'56070c',5637900],
['nbs-iscc-m:mineralred.19','mineralred','mineral red ',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:mineralrouge.40','mineralrouge','mineral rouge ',[136,45,23],'882d17',8924439],
['nbs-iscc-m:mineralrouge.55','mineralrouge','mineral rouge ',[128,70,27],'80461b',8406555],
['nbs-iscc-m:mineralviolet.218','mineralviolet','mineral violet ',[135,86,146],'875692',8869522],
['nbs-iscc-m:mineralyellow.83','mineralyellow','mineral yellow ',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:mineralyellow.84','mineralyellow','mineral yellow ',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:mineralyellow.86','mineralyellow','mineral yellow ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:mineralyellow.87','mineralyellow','mineral yellow ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:mineralyellow.71','mineralyellow','mineral yellow ',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:minggreen.141','minggreen','ming green ',[0,121,89],'007959',31065],
['nbs-iscc-m:miniaturepink.31','miniaturepink','miniature pink ',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-m:miniaturepink.32','miniaturepink','miniature pink ',[199,173,163],'c7ada3',13086115],
['nbs-iscc-m:minium.34','minium','minium ',[226,88,34],'e25822',14833698],
['nbs-iscc-m:mint.146','mint','mint ',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:mint.165','mint','mint ',[0,75,73],'004b49',19273],
['nbs-iscc-m:minuet.210','minuet','minuet ',[140,130,182],'8c82b6',9208502],
['nbs-iscc-m:mirabelle.53','mirabelle','mirabelle ',[217,144,88],'d99058',14258264],
['nbs-iscc-m:mirabelle.71','mirabelle','mirabelle ',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:mirador.59','mirador','mirador ',[66,37,24],'422518',4334872],
['nbs-iscc-m:mirage.76','mirage','mirage ',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:mistblue.135','mistblue','mist blue ',[147,197,146],'93c592',9684370],
['nbs-iscc-m:mistblue.144','mistblue','mist blue ',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:mistblue.149','mistblue','mist blue ',[141,163,153],'8da399',9282457],
['nbs-iscc-m:mistgrey.22','mistgrey','mist [grey] ',[143,129,127],'8f817f',9404799],
['nbs-iscc-m:mistletoe.122','mistletoe','mistletoe ',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:mistyblue.135','mistyblue','misty blue ',[147,197,146],'93c592',9684370],
['nbs-iscc-m:mistyblue.144','mistyblue','misty blue ',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:mistyblue.149','mistyblue','misty blue ',[141,163,153],'8da399',9282457],
['nbs-iscc-m:mistymorn.45','mistymorn','misty morn ',[151,127,115],'977f73',9928563],
['nbs-iscc-m:mistymorn.60','mistymorn','misty morn ',[149,128,112],'958070',9797744],
['nbs-iscc-m:mitisgreen.140','mitisgreen','mitis green ',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:mittlersgreen.165','mittlersgreen','mittler\'s green ',[0,75,73],'004b49',19273],
['nbs-iscc-m:moccasin.58','moccasin','moccasin ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:mocha.96','mocha','mocha ',[59,49,33],'3b3121',3879201],
['nbs-iscc-m:mochabisque.58','mochabisque','mocha bisque ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:modebeige.94','modebeige','mode beige ',[150,113,23],'967117',9859351],
['nbs-iscc-m:mohawk.43','mohawk','mohawk ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:mohawk.58','mohawk','mohawk ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:monaco.164','monaco','monaco ',[49,120,115],'317873',3242099],
['nbs-iscc-m:monetblue.197','monetblue','monet blue ',[39,36,88],'272458',2565208],
['nbs-iscc-m:monetblue.200','monetblue','monet blue ',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:monicon.26','monicon','monicon ',[248,131,121],'f88379',16286585],
['nbs-iscc-m:monicon.37','monicon','monicon ',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:monicon.50','monicon','monicon ',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:monicon.53','monicon','monicon ',[217,144,88],'d99058',14258264],
['nbs-iscc-m:monkeyskin.42','monkeyskin','monkey skin ',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:monkshood.215','monkshood','monkshood ',[85,76,105],'554c69',5590121],
['nbs-iscc-m:monsignor.218','monsignor','monsignor ',[135,86,146],'875692',8869522],
['nbs-iscc-m:monsignor.219','monsignor','monsignor ',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:monsoreau.15','monsoreau','monsoreau ',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:montecarlo.160','montecarlo','monte carlo ',[0,122,116],'007a74',31348],
['nbs-iscc-m:montecarlo.164','montecarlo','monte carlo ',[49,120,115],'317873',3242099],
['nbs-iscc-m:montella.62','montella','montella ',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:montella.81','montella','montella ',[72,60,50],'483c32',4734002],
['nbs-iscc-m:monterey.38','monterey','monterey ',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:monticellogreen.137','monticellogreen','monticello green ',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:montpelliergreen.136','montpelliergreen','montpellier green ',[103,146,103],'679267',6787687],
['nbs-iscc-m:montpellieryellow.83','montpellieryellow','montpellier yellow ',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:montpellieryellow.84','montpellieryellow','montpellier yellow ',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:montpellieryellow.86','montpellieryellow','montpellier yellow ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:montpellieryellow.87','montpellieryellow','montpellier yellow ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:moonbeam.264','moonbeam','moonbeam ',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:moonlight.73','moonlight','moonlight ',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:moonlight.93','moonlight','moonlight ',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:moonlightblue.186','moonlightblue','moonlight blue ',[83,104,120],'536878',5466232],
['nbs-iscc-m:moonmist.93','moonmist','moonmist ',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:moorishred.50','moorishred','moorish red ',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:moose.111','moose','moose ',[54,53,39],'363527',3552551],
['nbs-iscc-m:mordor.43','mordor','mordor ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:mordor.58','mordor','mordor ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:moreaberries.68','moreaberries','morea berries ',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:moreaberries.71','moreaberries','morea berries ',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:morello.235','morello','morello ',[36,33,36],'242124',2367780],
['nbs-iscc-m:moresco.50','moresco','moresco ',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:moresco.53','moresco','moresco ',[217,144,88],'d99058',14258264],
['nbs-iscc-m:morillon.120','morillon','morillon ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:morningblue.149','morningblue','morning blue ',[141,163,153],'8da399',9282457],
['nbs-iscc-m:morningblue.154','morningblue','morning blue ',[178,190,181],'b2beb5',11714229],
['nbs-iscc-m:morningdawningyellow.26','morningdawningyellow','morning dawning yellow ',[248,131,121],'f88379',16286585],
['nbs-iscc-m:morningdawningyellow.27','morningdawningyellow','morning dawning yellow ',[230,103,97],'e66761',15099745],
['nbs-iscc-m:morningdawningyellow.37','morningdawningyellow','morning dawning yellow ',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:morningglory.244','morningglory','morning glory ',[170,138,158],'aa8a9e',11176606],
['nbs-iscc-m:moroccan.16','moroccan','moroccan ',[114,47,55],'722f37',7483191],
['nbs-iscc-m:moroccan.38','moroccan','moroccan ',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:morocco.62','morocco','morocco ',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:moroccored.16','moroccored','morocco red ',[114,47,55],'722f37',7483191],
['nbs-iscc-m:moroccored.40','moroccored','morocco red ',[136,45,23],'882d17',8924439],
['nbs-iscc-m:moroccored.43','moroccored','morocco red ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:moroccosand.76','moroccosand','morocco sand ',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:morored.43','morored','moro red ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:morro.47','morro','morro ',[67,48,46],'43302e',4403246],
['nbs-iscc-m:morro.62','morro','morro ',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:mortdore.43','mortdore','mort d\'ore ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:mortdore.58','mortdore','mort d\'ore ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:mosaicblue.182','mosaicblue','mosaic blue ',[67,107,149],'436b95',4418453],
['nbs-iscc-m:mosque.55','mosque','mosque ',[128,70,27],'80461b',8406555],
['nbs-iscc-m:mossgreen.106','mossgreen','moss [green] ',[134,126,54],'867e36',8814134],
['nbs-iscc-m:mossgreen.120','mossgreen','moss [green] ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:mossgrey.109','mossgrey','moss grey ',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:mosspink.262','mosspink','moss pink ',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:mossrose.3','mossrose','moss rose ',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:mossrose.6','mossrose','moss rose ',[192,128,129],'c08081',12615809],
['nbs-iscc-m:mosul.57','mosul','mosul ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:mosul.58','mosul','mosul ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:mosul.77','mosul','mosul ',[130,102,68],'826644',8545860],
['nbs-iscc-m:mothgrey.90','mothgrey','moth [grey] ',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:motmotblue.169','motmotblue','motmot blue ',[0,119,145],'007791',30609],
['nbs-iscc-m:motmotgreen.136','motmotgreen','motmot green ',[103,146,103],'679267',6787687],
['nbs-iscc-m:motmotgreen.144','motmotgreen','motmot green ',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:mountainblue.182','mountainblue','mountain blue ',[67,107,149],'436b95',4418453],
['nbs-iscc-m:mountaingreen.151','mountaingreen','mountain green ',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:mountaingreen.136','mountaingreen','mountain green ',[103,146,103],'679267',6787687],
['nbs-iscc-m:mountainyellow.71','mountainyellow','mountain yellow ',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:mountvernongreenormtvernongreen.125','mountvernongreenormtvernongreen','mount vernon green or mt. vernon green ',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:mousedun.64','mousedun','mouse-dun ',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:mousegrey.64','mousegrey','mouse [grey] ',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:mousse.106','mousse','mousse ',[134,126,54],'867e36',8814134],
['nbs-iscc-m:mousse.120','mousse','mousse ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:msange.149','msange','msange ',[141,163,153],'8da399',9282457],
['nbs-iscc-m:mtallique.113','mtallique','mtallique ',[87,85,76],'57554c',5723468],
['nbs-iscc-m:muffin.57','muffin','muffin ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:muffin.74','muffin','muffin ',[153,101,21],'996515',10052885],
['nbs-iscc-m:mulberry.235','mulberry','mulberry ',[36,33,36],'242124',2367780],
['nbs-iscc-m:mulberryfruit.17','mulberryfruit','mulberry fruit ',[63,23,40],'3f1728',4134696],
['nbs-iscc-m:mulberryfruit.21','mulberryfruit','mulberry fruit ',[46,29,33],'2e1d21',3022113],
['nbs-iscc-m:mulberrypurple.224','mulberrypurple','mulberry purple ',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:mummy.81','mummy','mummy ',[72,60,50],'483c32',4734002],
['nbs-iscc-m:mummy.77','mummy','mummy ',[130,102,68],'826644',8545860],
['nbs-iscc-m:mummybrown.58','mummybrown','mummy brown ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:mummybrown.61','mummybrown','mummy brown ',[99,81,71],'635147',6508871],
['nbs-iscc-m:mummybrown.80','mummybrown','mummy brown ',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:muraille.58','muraille','muraille ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:murillo.182','murillo','murillo ',[67,107,149],'436b95',4418453],
['nbs-iscc-m:murinus.64','murinus','murinus ',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:murrey.235','murrey','murrey ',[36,33,36],'242124',2367780],
['nbs-iscc-m:muscade.57','muscade','muscade ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:muscovite.156','muscovite','muscovite ',[78,87,85],'4e5755',5134165],
['nbs-iscc-m:mushroom.64','mushroom','mushroom ',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:musk.81','musk','musk ',[72,60,50],'483c32',4734002],
['nbs-iscc-m:musketeer.183','musketeer','musketeer ',[0,48,78],'00304e',12366],
['nbs-iscc-m:musketeer.201','musketeer','musketeer ',[37,36,64],'252440',2434112],
['nbs-iscc-m:musketeer.204','musketeer','musketeer ',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:muskmelon.39','muskmelon','muskmelon ',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:muskmelon.53','muskmelon','muskmelon ',[217,144,88],'d99058',14258264],
['nbs-iscc-m:muskrat.62','muskrat','muskrat ',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:mustang.80','mustang','mustang ',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:mustardbrown.58','mustardbrown','mustard brown ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:mustardyellow.87','mustardyellow','mustard [yellow] ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:mutrieyellow.97','mutrieyellow','mutrie yellow ',[220,211,0],'dcd300',14471936],
['nbs-iscc-m:mutrieyellow.98','mutrieyellow','mutrie yellow ',[233,228,80],'e9e450',15328336],
['nbs-iscc-m:myosotisblue.185','myosotisblue','myosotis blue ',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:myosotisblue.186','myosotisblue','myosotis blue ',[83,104,120],'536878',5466232],
['nbs-iscc-m:myrtlegreen.151','myrtlegreen','myrtle [green] ',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:myrtlegreen.165','myrtlegreen','myrtle [green] ',[0,75,73],'004b49',19273],
['nbs-iscc-m:mythogreen.120','mythogreen','mytho green ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:mythogreen.122','mythogreen','mytho green ',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:nacarat.11','nacarat','nacarat ',[190,0,50],'be0032',12451890],
['nbs-iscc-m:nacarine.11','nacarine','nacarine ',[190,0,50],'be0032',12451890],
['nbs-iscc-m:naiad.144','naiad','naiad ',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:nankeenyellow.86','nankeenyellow','nankeen [yellow] ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:nankeenyellow.87','nankeenyellow','nankeen [yellow] ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:nankeenyellow.89','nankeenyellow','nankeen [yellow] ',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:nankeenyellow.90','nankeenyellow','nankeen [yellow] ',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:naplesred.43','naplesred','naples red ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:naplesred.40','naplesred','naples red ',[136,45,23],'882d17',8924439],
['nbs-iscc-m:naplesyellow.86','naplesyellow','naples yellow ',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:naplesyellow.87','naplesyellow','naples yellow ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:naplesyellow.89','naplesyellow','naples yellow ',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:naplesyellow.90','naplesyellow','naples yellow ',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:napoleonblue.179','napoleonblue','napoleon blue ',[0,65,106],'00416a',16746],
['nbs-iscc-m:napoli.174','napoli','napoli ',[0,73,88],'004958',18776],
['nbs-iscc-m:narcissus.84','narcissus','narcissus ',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:narcissus.87','narcissus','narcissus ',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:narrawood.15','narrawood','narrawood ',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:narva.151','narva','narva ',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:nasturtiumred.38','nasturtiumred','nasturtium [red] ',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:nasturtiumyellow.50','nasturtiumyellow','nasturtium [yellow] ',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:nasturtiumyellow.53','nasturtiumyellow','nasturtium [yellow] ',[217,144,88],'d99058',14258264],
['nbs-iscc-m:nasturtiumyellow.68','nasturtiumyellow','nasturtium [yellow] ',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:nasturtiumyellow.71','nasturtiumyellow','nasturtium [yellow] ',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:natalbrown.61','natalbrown','natal brown',[99,81,71],'635147',6508871],
['nbs-iscc-m:nationalblue.179','nationalblue','national [blue] ',[0,65,106],'00416a',16746],
['nbs-iscc-m:nationalblue.182','nationalblue','national [blue] ',[67,107,149],'436b95',4418453],
['nbs-iscc-m:nationalgrey.265','nationalgrey','national grey ',[132,132,130],'848482',8684674],
['nbs-iscc-m:nattier.199','nattier','nattier',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:natural.73','natural','natural',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:natural.93','natural','natural',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:navaho.48','navaho','navaho',[243,132,0],'f38400',15959040],
['nbs-iscc-m:navaho.50','navaho','navaho',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:navy1.187','navy1','navy 1 ',[54,69,79],'36454f',3556687],
['nbs-iscc-m:navy2.230','navy2','navy 2 ',[41,30,41],'291e29',2694697],
['nbs-iscc-m:navy3.187','navy3','navy 3 ',[54,69,79],'36454f',3556687],
['nbs-iscc-m:navy3.188','navy3','navy 3 ',[32,40,48],'202830',2107440],
['nbs-iscc-m:navyblue.201','navyblue','navy blue',[37,36,64],'252440',2434112],
['nbs-iscc-m:navyblue.204','navyblue','navy blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:neapolitanblue.204','neapolitanblue','neapolitan blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:neapolitanyellow.86','neapolitanyellow','neapolitan yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:neapolitanyellow.87','neapolitanyellow','neapolitan yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:neapolitanyellow.89','neapolitanyellow','neapolitan yellow',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:neapolitanyellow.90','neapolitanyellow','neapolitan yellow',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:nectar.19','nectar','nectar',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:nectarine.28','nectarine','nectarine',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-m:nectarine.29','nectarine','nectarine',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:negro.81','negro','negro',[72,60,50],'483c32',4734002],
['nbs-iscc-m:negro.267','negro','negro',[34,34,34],'222222',2236962],
['nbs-iscc-m:neptunegreen.144','neptunegreen','neptune [green] ',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:neptunegreen.145','neptunegreen','neptune [green] ',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:neutralorange.51','neutralorange','neutral orange',[190,101,22],'be6516',12477718],
['nbs-iscc-m:neutralred.16','neutralred','neutral red',[114,47,55],'722f37',7483191],
['nbs-iscc-m:neutralred.259','neutralred','neutral red',[103,49,71],'673147',6762823],
['nbs-iscc-m:neutraltint.267','neutraltint','neutral tint',[34,34,34],'222222',2236962],
['nbs-iscc-m:neuvidergreen.135','neuvidergreen','neuvider green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:neuvidergreen.144','neuvidergreen','neuvider green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:neuwiedblue.164','neuwiedblue','neuwied blue',[49,120,115],'317873',3242099],
['nbs-iscc-m:neuwiederblue.164','neuwiederblue','neuwieder blue',[49,120,115],'317873',3242099],
['nbs-iscc-m:neuwiedergreen.135','neuwiedergreen','neuwieder green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:neuwiedergreen.144','neuwiedergreen','neuwieder green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:nevagreen.117','nevagreen','neva green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-m:nevagreen.130','nevagreen','neva green',[131,211,125],'83d37d',8639357],
['nbs-iscc-m:newblue.178','newblue','new blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:newblue.196','newblue','new blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:newbronze.58','newbronze','new bronze',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:newbronze.61','newbronze','new bronze',[99,81,71],'635147',6508871],
['nbs-iscc-m:newbronze.77','newbronze','new bronze',[130,102,68],'826644',8545860],
['nbs-iscc-m:newbronze.80','newbronze','new bronze',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:newcocoa.61','newcocoa','new cocoa',[99,81,71],'635147',6508871],
['nbs-iscc-m:newhay.127','newhay','new hay',[81,87,68],'515744',5330756],
['nbs-iscc-m:newport.174','newport','newport',[0,73,88],'004958',18776],
['nbs-iscc-m:newsilver.89','newsilver','new silver',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:newsilver.90','newsilver','new silver',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:newsilver.93','newsilver','new silver',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:newsilver.121','newsilver','new silver',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:newsilver.265','newsilver','new silver',[132,132,130],'848482',8684674],
['nbs-iscc-m:niagara.172','niagara','niagara',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:niagaragreen.163','niagaragreen','niagara green',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:nicaraguawood.15','nicaraguawood','nicaraguawood ',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:nicaraguawood.19','nicaraguawood','nicaraguawood ',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:nice.186','nice','nice',[83,104,120],'536878',5466232],
['nbs-iscc-m:nightshade.212','nightshade','nightshade',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:nightshade.229','nightshade','nightshade',[80,64,77],'50404d',5259341],
['nbs-iscc-m:nikko.182','nikko','nikko',[67,107,149],'436b95',4418453],
['nbs-iscc-m:nikko.186','nikko','nikko',[83,104,120],'536878',5466232],
['nbs-iscc-m:nileblue.163','nileblue','nile blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:nilegreen.120','nilegreen','nile [green] ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:nilegreen.122','nilegreen','nile [green] ',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:nilegreen.135','nilegreen','nile [green] ',[147,197,146],'93c592',9684370],
['nbs-iscc-m:nimbus.265','nimbus','nimbus',[132,132,130],'848482',8684674],
['nbs-iscc-m:nippon.183','nippon','nippon',[0,48,78],'00304e',12366],
['nbs-iscc-m:noisette.57','noisette','noisette',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:noisette.74','noisette','noisette',[153,101,21],'996515',10052885],
['nbs-iscc-m:nomadbrown.59','nomadbrown','nomad brown',[66,37,24],'422518',4334872],
['nbs-iscc-m:norfolk.62','norfolk','norfolk',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:normandy.183','normandy','normandy',[0,48,78],'00304e',12366],
['nbs-iscc-m:normandy.201','normandy','normandy',[37,36,64],'252440',2434112],
['nbs-iscc-m:normandy.204','normandy','normandy',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:normandyblue.149','normandyblue','normandy blue',[141,163,153],'8da399',9282457],
['nbs-iscc-m:nougat.60','nougat','nougat',[149,128,112],'958070',9797744],
['nbs-iscc-m:nougat.79','nougat','nougat',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:nubian.267','nubian','nubian',[34,34,34],'222222',2236962],
['nbs-iscc-m:nude.33','nude','nude',[194,172,153],'c2ac99',12758169],
['nbs-iscc-m:nude.73','nude','nude',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:nude.76','nude','nude',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:nude.79','nude','nude',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:nugget.53','nugget','nugget',[217,144,88],'d99058',14258264],
['nbs-iscc-m:nugget.72','nugget','nugget',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:nuncio.223','nuncio','nuncio',[134,96,142],'86608e',8806542],
['nbs-iscc-m:nurembergred.43','nurembergred','nuremberg red ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:nurembergviolet.218','nurembergviolet','nuremberg violet',[135,86,146],'875692',8869522],
['nbs-iscc-m:nutmeg.58','nutmeg','nutmeg',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:nutmeg.61','nutmeg','nutmeg',[99,81,71],'635147',6508871],
['nbs-iscc-m:nutria.113','nutria','nutria',[87,85,76],'57554c',5723468],
['nbs-iscc-m:nymphea.223','nymphea','nymphea',[134,96,142],'86608e',8806542],
['nbs-iscc-m:nymphgreen.104','nymphgreen','nymph green',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-m:nymphgreen.119','nymphgreen','nymph green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-m:nymphpink.2','nymphpink','nymph [pink] ',[234,147,153],'ea9399',15373209],
['nbs-iscc-m:nymphpink.5','nymphpink','nymph [pink] ',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:oad.186','oad','oad ',[83,104,120],'536878',5466232],
['nbs-iscc-m:oakbrown.55','oakbrown','oak [brown] ',[128,70,27],'80461b',8406555],
['nbs-iscc-m:oakbrown.58','oakbrown','oak [brown] ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:oakbuff.57','oakbuff','oakbuff',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:oakbuff.76','oakbuff','oakbuff',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:oakbuff.77','oakbuff','oakbuff',[130,102,68],'826644',8545860],
['nbs-iscc-m:oakgreen.106','oakgreen','oak [green] ',[134,126,54],'867e36',8814134],
['nbs-iscc-m:oakheart.19','oakheart','oakheart',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:oakheart.43','oakheart','oakheart',[121,68,59],'79443b',7947323],
['nbs-iscc-m:oakleafbrown.51','oakleafbrown','oakleaf brown',[190,101,22],'be6516',12477718],
['nbs-iscc-m:oakleafbrown.54','oakleafbrown','oakleaf brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:oakleafbrown.55','oakleafbrown','oakleaf brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:oakwood.58','oakwood','oakwood',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:oasis.106','oasis','oasis',[134,126,54],'867e36',8814134],
['nbs-iscc-m:oasis.120','oasis','oasis',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:oceangreen.135','oceangreen','ocean green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:ocherred.19','ocherred','ocher red',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:ochraceous.71','ochraceous','ochraceous',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:ochre.71','ochre','ochre',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:ochrederu.54','ochrederu','ochre de ru',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:ochrederu.74','ochrederu','ochre de ru',[153,101,21],'996515',10052885],
['nbs-iscc-m:ochrered.38','ochrered','ochre red',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:oilblue.194','oilblue','oil blue',[48,38,122],'30267a',3155578],
['nbs-iscc-m:oilgreen.136','oilgreen','oil green',[103,146,103],'679267',6787687],
['nbs-iscc-m:oilyellow.102','oilyellow','oil yellow',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:oilyellow.103','oilyellow','oil yellow',[152,148,62],'98943e',9999422],
['nbs-iscc-m:okerdeluce.54','okerdeluce','oker de luce',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:okerdeluce.74','okerdeluce','oker de luce',[153,101,21],'996515',10052885],
['nbs-iscc-m:okerdeluke.54','okerdeluke','oker de luke',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:okerdeluke.74','okerdeluke','oker de luke',[153,101,21],'996515',10052885],
['nbs-iscc-m:okerderouse.54','okerderouse','oker de rouse',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:okerderouse.74','okerderouse','oker de rouse',[153,101,21],'996515',10052885],
['nbs-iscc-m:oldamethyst.16','oldamethyst','old amethyst',[114,47,55],'722f37',7483191],
['nbs-iscc-m:oldamethyst.259','oldamethyst','old amethyst',[103,49,71],'673147',6762823],
['nbs-iscc-m:oldblue.185','oldblue','old blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:oldblue.186','oldblue','old blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:oldblue.191','oldblue','old blue',[129,135,139],'81878b',8488843],
['nbs-iscc-m:oldbronze.94','oldbronze','old bronze',[150,113,23],'967117',9859351],
['nbs-iscc-m:oldbronze.106','oldbronze','old bronze',[134,126,54],'867e36',8814134],
['nbs-iscc-m:oldburgundy.44','oldburgundy','old burgundy',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-m:oldburgundy.47','oldburgundy','old burgundy',[67,48,46],'43302e',4403246],
['nbs-iscc-m:oldcedar.43','oldcedar','old cedar',[121,68,59],'79443b',7947323],
['nbs-iscc-m:oldchina.186','oldchina','old china',[83,104,120],'536878',5466232],
['nbs-iscc-m:oldchina.204','oldchina','old china',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:oldcoral.15','oldcoral','old coral',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:olde.101','olde','olde',[234,230,121],'eae679',15394425],
['nbs-iscc-m:olde.102','olde','olde',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:oldenglishbrown.81','oldenglishbrown','old english brown',[72,60,50],'483c32',4734002],
['nbs-iscc-m:oldenglishbrown.95','oldenglishbrown','old english brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:oldenglishbrown.96','oldenglishbrown','old english brown',[59,49,33],'3b3121',3879201],
['nbs-iscc-m:oldgold.106','oldgold','old gold',[134,126,54],'867e36',8814134],
['nbs-iscc-m:oldhelio.224','oldhelio','old helio',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:oldivory.90','oldivory','old ivory',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:oldivory.91','oldivory','old ivory',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:oldivory.94','oldivory','old ivory',[150,113,23],'967117',9859351],
['nbs-iscc-m:oldlavender.235','oldlavender','old lavender',[36,33,36],'242124',2367780],
['nbs-iscc-m:oldlilac.228','oldlilac','old lilac',[121,104,120],'796878',7956600],
['nbs-iscc-m:oldlilac.245','oldlilac','old lilac',[131,100,121],'836479',8610937],
['nbs-iscc-m:oldmauve.229','oldmauve','old mauve',[80,64,77],'50404d',5259341],
['nbs-iscc-m:oldmauve.259','oldmauve','old mauve',[103,49,71],'673147',6762823],
['nbs-iscc-m:oldmossgreen.106','oldmossgreen','old moss [green] ',[134,126,54],'867e36',8814134],
['nbs-iscc-m:oldolive.95','oldolive','old olive ',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:oldpink.57','oldpink','old pink',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:oldred.13','oldred','old red',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:oldred.15','oldred','old red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:oldrose.19','oldrose','old rose',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:oldroseleaf.16','oldroseleaf','old roseleaf',[114,47,55],'722f37',7483191],
['nbs-iscc-m:oldsilver.265','oldsilver','old silver',[132,132,130],'848482',8684674],
['nbs-iscc-m:oldwood.19','oldwood','old wood',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:olivaceous.107','olivaceous','olivaceous',[102,93,30],'665d1e',6708510],
['nbs-iscc-m:olivebrown.95','olivebrown','olive brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:olivedrab.110','olivedrab','olive drab',[91,88,66],'5b5842',5986370],
['nbs-iscc-m:olivegreen.107','olivegreen','olive [green] ',[102,93,30],'665d1e',6708510],
['nbs-iscc-m:olivegrey.122','olivegrey','olive grey ',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:olivegrey.155','olivegrey','olive grey ',[125,137,132],'7d8984',8227204],
['nbs-iscc-m:olivesheen.91','olivesheen','olivesheen',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:oliveterraverte.106','oliveterraverte','olive terra verte',[134,126,54],'867e36',8814134],
['nbs-iscc-m:olivewood.80','olivewood','olive wood',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:oliveyellow.103','oliveyellow','olive yellow',[152,148,62],'98943e',9999422],
['nbs-iscc-m:oliveyellow.106','oliveyellow','olive yellow',[134,126,54],'867e36',8814134],
['nbs-iscc-m:olivine.135','olivine','olivine',[147,197,146],'93c592',9684370],
['nbs-iscc-m:olympia.160','olympia','olympia',[0,122,116],'007a74',31348],
['nbs-iscc-m:olympia.164','olympia','olympia',[49,120,115],'317873',3242099],
['nbs-iscc-m:olympianblue.179','olympianblue','olympian blue ',[0,65,106],'00416a',16746],
['nbs-iscc-m:olympiangreen.136','olympiangreen','olympian green',[103,146,103],'679267',6787687],
['nbs-iscc-m:olympic.187','olympic','olympic',[54,69,79],'36454f',3556687],
['nbs-iscc-m:olympicblue.169','olympicblue','olympic blue',[0,119,145],'007791',30609],
['nbs-iscc-m:ondine.149','ondine','ondine',[141,163,153],'8da399',9282457],
['nbs-iscc-m:onionpeel.19','onionpeel','onion-peel ',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:onionred.19','onionred','onion red',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:onionskinpink.29','onionskinpink','onion-skin pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:onionskinpink.53','onionskinpink','onion-skin pink',[217,144,88],'d99058',14258264],
['nbs-iscc-m:onionskinpink.57','onionskinpink','onion-skin pink',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:ontarioviolet.203','ontarioviolet','ontario violet',[140,146,172],'8c92ac',9212588],
['nbs-iscc-m:oold.101','oold','oold',[234,230,121],'eae679',15394425],
['nbs-iscc-m:oold.102','oold','oold',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:opal.164','opal','opal',[49,120,115],'317873',3242099],
['nbs-iscc-m:opalblue.186','opalblue','opal blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:opalgrey.22','opalgrey','opal grey ',[143,129,127],'8f817f',9404799],
['nbs-iscc-m:opalgrey.265','opalgrey','opal grey ',[132,132,130],'848482',8684674],
['nbs-iscc-m:opalinegreen.143','opalinegreen','opaline green',[142,209,178],'8ed1b2',9359794],
['nbs-iscc-m:opalmauve.10','opalmauve','opal mauve',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-m:opaq.136','opaq','opaq',[103,146,103],'679267',6787687],
['nbs-iscc-m:operamauve.240','operamauve','opera mauve',[183,132,167],'b784a7',12027047],
['nbs-iscc-m:operapink.28','operapink','opera pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-m:operapink.29','operapink','opera pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:operapink.31','operapink','opera pink',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-m:operapink.32','operapink','opera pink',[199,173,163],'c7ada3',13086115],
['nbs-iscc-m:ophelia.241','ophelia','ophelia',[145,92,131],'915c83',9526403],
['nbs-iscc-m:ophelia.262','ophelia','ophelia',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:ophelia.245','ophelia','ophelia',[131,100,121],'836479',8610937],
['nbs-iscc-m:oporto.16','oporto','oporto',[114,47,55],'722f37',7483191],
['nbs-iscc-m:oporto.17','oporto','oporto',[63,23,40],'3f1728',4134696],
['nbs-iscc-m:orangeaurora.26','orangeaurora','orange aurora',[248,131,121],'f88379',16286585],
['nbs-iscc-m:orangeaurora.27','orangeaurora','orange aurora',[230,103,97],'e66761',15099745],
['nbs-iscc-m:orangeaurora.37','orangeaurora','orange aurora',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:orangelead.34','orangelead','orange lead',[226,88,34],'e25822',14833698],
['nbs-iscc-m:orangemadder.50','orangemadder','orange madder',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:orangeochre.50','orangeochre','orange ochre',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:orangepeel.50','orangepeel','orange-peel',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:orangerufous.50','orangerufous','orange rufous',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:orangetawny.37','orangetawny','orange tawny',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:orangevermilion.35','orangevermilion','orange vermilion',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:orchadee.240','orchadee','orchadee',[183,132,167],'b784a7',12027047],
['nbs-iscc-m:orchall.258','orchall','orchall',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:orchid.222','orchid','orchid',[182,149,192],'b695c0',11965888],
['nbs-iscc-m:orchid.227','orchid','orchid',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:orchidpink.2','orchidpink','orchid pink',[234,147,153],'ea9399',15373209],
['nbs-iscc-m:orchil.15','orchil','orchil ',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:orchil.258','orchil','orchil ',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:orchis.19','orchis','orchis',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:orient.5','orient','orient',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:oriental.186','oriental','oriental',[83,104,120],'536878',5466232],
['nbs-iscc-m:orientalblue.204','orientalblue','oriental blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:orientalblue.178','orientalblue','oriental blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:orientalblue.196','orientalblue','oriental blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:orientalbole.43','orientalbole','oriental bole ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:orientalfuchsia.238','orientalfuchsia','oriental fuchsia',[112,41,99],'702963',7350627],
['nbs-iscc-m:orientalgreen.131','orientalgreen','oriental green',[68,148,74],'44944a',4494410],
['nbs-iscc-m:orientalgreen.136','orientalgreen','oriental green',[103,146,103],'679267',6787687],
['nbs-iscc-m:orientalpearl.112','orientalpearl','oriental pearl',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:orientalpearl.265','orientalpearl','oriental pearl',[132,132,130],'848482',8684674],
['nbs-iscc-m:orientalred.12','orientalred','oriental red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:orientblue.178','orientblue','orient [blue] ',[0,103,165],'0067a5',26533],
['nbs-iscc-m:orientblue.182','orientblue','orient blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:orientblue.186','orientblue','orient blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:orientpink.29','orientpink','orient pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:orientred.12','orientred','orient red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:orientyellow.50','orientyellow','orient yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:orientyellow.53','orientyellow','orient yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-m:orientyellow.68','orientyellow','orient yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:orientyellow.71','orientyellow','orient yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:orientyellow.88','orientyellow','orient yellow',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:orientyellow.91','orientyellow','orient yellow',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:oriole.54','oriole','oriole',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:orion.183','orion','orion',[0,48,78],'00304e',12366],
['nbs-iscc-m:orlean.29','orlean','orlean',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:ormond.45','ormond','ormond',[151,127,115],'977f73',9928563],
['nbs-iscc-m:ormond.60','ormond','ormond',[149,128,112],'958070',9797744],
['nbs-iscc-m:ormond.63','ormond','ormond',[142,130,121],'8e8279',9339513],
['nbs-iscc-m:orpiment.83','orpiment','orpiment',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:orpiment.84','orpiment','orpiment',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:orpiment.86','orpiment','orpiment',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:orpiment.87','orpiment','orpiment',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:orpimentorange.48','orpimentorange','orpiment orange',[243,132,0],'f38400',15959040],
['nbs-iscc-m:orpimentorange.50','orpimentorange','orpiment orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:orpimentred.68','orpimentred','orpiment red',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:orpimentred.71','orpimentred','orpiment red',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:orpin.83','orpin','orpin',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:orpin.84','orpin','orpin',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:orpin.86','orpin','orpin',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:orpin.87','orpin','orpin',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:orseille.258','orseille','orseille',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:otterbrown.81','otterbrown','otter [brown] ',[72,60,50],'483c32',4734002],
['nbs-iscc-m:ouralgreen.135','ouralgreen','oural green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:ouralgreen.148','ouralgreen','oural green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-m:owl.64','owl','owl',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:oxbloodred.43','oxbloodred','oxblood [red] ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:oxfordblue.230','oxfordblue','oxford blue',[41,30,41],'291e29',2694697],
['nbs-iscc-m:oxfordchrome.71','oxfordchrome','oxford chrome',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:oxfordochre.71','oxfordochre','oxford ochre',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:oxfordyellow.71','oxfordyellow','oxford yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:oxgall.84','oxgall','oxgall',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:oxheart.13','oxheart','oxheart',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:oxideblue.167','oxideblue','oxide blue',[0,133,161],'0085a1',34209],
['nbs-iscc-m:oxidebrown.43','oxidebrown','oxide brown',[121,68,59],'79443b',7947323],
['nbs-iscc-m:oxidepurple.43','oxidepurple','oxide purple',[121,68,59],'79443b',7947323],
['nbs-iscc-m:oxideyellow.71','oxideyellow','oxide yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:oystergrey.121','oystergrey','oyster grey ',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:oysterwhite.121','oysterwhite','oyster [white] ',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:pablo.57','pablo','pablo',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:pablo.76','pablo','pablo',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:pacific.149','pacific','pacific',[141,163,153],'8da399',9282457],
['nbs-iscc-m:pacific.163','pacific','pacific',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:paddock.165','paddock','paddock',[0,75,73],'004b49',19273],
['nbs-iscc-m:pagodablue.174','pagodablue','pagoda [blue] ',[0,73,88],'004958',18776],
['nbs-iscc-m:pagodablue.187','pagodablue','pagoda [blue] ',[54,69,79],'36454f',3556687],
['nbs-iscc-m:palestine.228','palestine','palestine',[121,104,120],'796878',7956600],
['nbs-iscc-m:palestine.262','palestine','palestine',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:palm.120','palm','palm',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:palmetto.122','palmetto','palmetto',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:palmgreen.137','palmgreen','palm green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:palmleaf.106','palmleaf','palmleaf',[134,126,54],'867e36',8814134],
['nbs-iscc-m:paloma.54','paloma','paloma',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:paloma.57','paloma','paloma',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:pampas.64','pampas','pampas',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:pan.151','pan','pan',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:pancygreen.137','pancygreen','pancy green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:panse.207','panse','panse',[96,78,151],'604e97',6311575],
['nbs-iscc-m:pansy.207','pansy','pansy',[96,78,151],'604e97',6311575],
['nbs-iscc-m:pansymaroon.259','pansymaroon','pansy-maroon',[103,49,71],'673147',6762823],
['nbs-iscc-m:pansypurple.259','pansypurple','pansy purple',[103,49,71],'673147',6762823],
['nbs-iscc-m:pansyviolet.242','pansyviolet','pansy violet',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:paon.174','paon','paon',[0,73,88],'004958',18776],
['nbs-iscc-m:paperwhite.93','paperwhite','paper white',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:paprica.34','paprica','paprica',[226,88,34],'e25822',14833698],
['nbs-iscc-m:papyrus.57','papyrus','papyrus',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:paradisegreen.137','paradisegreen','paradise green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:parared.12','parared','para red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:parared.15','parared','para red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:parchment.90','parchment','parchment',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:parchment.91','parchment','parchment',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:parchment.94','parchment','parchment',[150,113,23],'967117',9859351],
['nbs-iscc-m:parisgreen.136','parisgreen','paris green ',[103,146,103],'679267',6787687],
['nbs-iscc-m:parisgreen.140','parisgreen','paris green ',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:parismud.20','parismud','paris mud',[84,61,63],'543d3f',5520703],
['nbs-iscc-m:parisred.34','parisred','paris red',[226,88,34],'e25822',14833698],
['nbs-iscc-m:parisyellow.84','parisyellow','paris yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:parkgreen.120','parkgreen','park green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:parmared.12','parmared','parma red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:parmared.15','parmared','parma red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:parme.211','parme','parme',[96,78,129],'604e81',6311553],
['nbs-iscc-m:parrakeet.120','parrakeet','parrakeet',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:parrakeet.140','parrakeet','parrakeet',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:parroquitgreen.120','parroquitgreen','parroquit green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:parrotgreen.120','parrotgreen','parrot green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:partridge.77','partridge','partridge',[130,102,68],'826644',8545860],
['nbs-iscc-m:partridge.78','partridge','partridge',[75,54,33],'4b3621',4929057],
['nbs-iscc-m:parulablue.186','parulablue','parula blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:pastel.186','pastel','pastel',[83,104,120],'536878',5466232],
['nbs-iscc-m:pastelblue.186','pastelblue','pastel blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:pastelgrey.90','pastelgrey','pastel grey ',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:pastelparchment.29','pastelparchment','pastel parchment',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:pastelparchment.32','pastelparchment','pastel parchment',[199,173,163],'c7ada3',13086115],
['nbs-iscc-m:pastelparchment.33','pastelparchment','pastel parchment',[194,172,153],'c2ac99',12758169],
['nbs-iscc-m:patentyellow.83','patentyellow','patent yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:patentyellow.84','patentyellow','patent yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:patentyellow.86','patentyellow','patent yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:patentyellow.87','patentyellow','patent yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:patinagreen.135','patinagreen','patina green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:patinagreen.136','patinagreen','patina green',[103,146,103],'679267',6787687],
['nbs-iscc-m:patriarch.219','patriarch','patriarch',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:patricia.237','patricia','patricia',[158,79,136],'9e4f88',10375048],
['nbs-iscc-m:paulveronesegreen.140','paulveronesegreen','paul veronese green ',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:pauncygreen.137','pauncygreen','pauncy green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:pavonine.174','pavonine','pavonine',[0,73,88],'004958',18776],
['nbs-iscc-m:pawnee.57','pawnee','pawnee',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:pawnee.60','pawnee','pawnee',[149,128,112],'958070',9797744],
['nbs-iscc-m:pawnee.76','pawnee','pawnee',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:pawnee.79','pawnee','pawnee',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:paynesgrey.186','paynesgrey','payne\'s grey ',[83,104,120],'536878',5466232],
['nbs-iscc-m:paynesgrey.187','paynesgrey','payne\'s grey ',[54,69,79],'36454f',3556687],
['nbs-iscc-m:peach.28','peach','peach',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-m:peach.29','peach','peach',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:peachbeige.29','peachbeige','peachbeige',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:peachbisque.42','peachbisque','peach bisque',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:peachbloom.29','peachbloom','peach bloom',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:peachblossompink.2','peachblossompink','peach blossom [pink] ',[234,147,153],'ea9399',15373209],
['nbs-iscc-m:peachblossompink.5','peachblossompink','peach blossom [pink] ',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:peachblossomred.2','peachblossomred','peach blossom [red] ',[234,147,153],'ea9399',15373209],
['nbs-iscc-m:peachblossomred.5','peachblossomred','peach blossom [red] ',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:peachblow.73','peachblow','peach blow',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:peachblush.55','peachblush','peach blush',[128,70,27],'80461b',8406555],
['nbs-iscc-m:peachred.26','peachred','peach red',[248,131,121],'f88379',16286585],
['nbs-iscc-m:peachwood.15','peachwood','peachwood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:peachwood.19','peachwood','peachwood',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:peacockblue.174','peacockblue','peacock [blue] ',[0,73,88],'004958',18776],
['nbs-iscc-m:peacockgreen.136','peacockgreen','peacock green',[103,146,103],'679267',6787687],
['nbs-iscc-m:peagreen.120','peagreen','pea green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:peanut.90','peanut','peanut',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:pearl.264','pearl','pearl',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:pearlblue.264','pearlblue','pearl blue',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:pearlblush.33','pearlblush','pearlblush',[194,172,153],'c2ac99',12758169],
['nbs-iscc-m:pearlblush.60','pearlblush','pearlblush',[149,128,112],'958070',9797744],
['nbs-iscc-m:pearlgray.93','pearlgray','pearl gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:pearlwhite.264','pearlwhite','pearl white',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:peasant.201','peasant','peasant',[37,36,64],'252440',2434112],
['nbs-iscc-m:peasant.204','peasant','peasant',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:peasantblue.186','peasantblue','peasant blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:peasantblue.183','peasantblue','peasant blue',[0,48,78],'00304e',12366],
['nbs-iscc-m:peasantblue.201','peasantblue','peasant blue',[37,36,64],'252440',2434112],
['nbs-iscc-m:peasantblue.204','peasantblue','peasant blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:pebble.90','pebble','pebble',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:pecan.79','pecan','pecan',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:pecanbrown.58','pecanbrown','pecan brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:pekinese.54','pekinese','pekinese',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:pekinese.57','pekinese','pekinese',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:pekingblue.183','pekingblue','peking blue',[0,48,78],'00304e',12366],
['nbs-iscc-m:pelican.266','pelican','pelican',[85,85,85],'555555',5592405],
['nbs-iscc-m:peligotsblue.164','peligotsblue','peligot\'s blue',[49,120,115],'317873',3242099],
['nbs-iscc-m:pelt.64','pelt','pelt',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:pencilwood.43','pencilwood','pencilwood',[121,68,59],'79443b',7947323],
['nbs-iscc-m:pencilwood.58','pencilwood','pencilwood',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:penguin.265','penguin','penguin',[132,132,130],'848482',8684674],
['nbs-iscc-m:peony.16','peony','peony',[114,47,55],'722f37',7483191],
['nbs-iscc-m:peonyred.13','peonyred','peony red',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:peonyred.16','peonyred','peony red',[114,47,55],'722f37',7483191],
['nbs-iscc-m:pepita.94','pepita','pepita',[150,113,23],'967117',9859351],
['nbs-iscc-m:peppermint.131','peppermint','peppermint',[68,148,74],'44944a',4494410],
['nbs-iscc-m:pepperred.15','pepperred','pepper red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:peridot.120','peridot','peridot',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:peridot.122','peridot','peridot',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:peridot.125','peridot','peridot',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:peridot.127','peridot','peridot',[81,87,68],'515744',5330756],
['nbs-iscc-m:perilla.205','perilla','perilla',[144,101,202],'9065ca',9463242],
['nbs-iscc-m:perilla.207','perilla','perilla',[96,78,151],'604e97',6311575],
['nbs-iscc-m:perillapurple.259','perillapurple','perilla purple',[103,49,71],'673147',6762823],
['nbs-iscc-m:perique.81','perique','perique',[72,60,50],'483c32',4734002],
['nbs-iscc-m:periwinkle.203','periwinkle','periwinkle',[140,146,172],'8c92ac',9212588],
['nbs-iscc-m:periwinkle.204','periwinkle','periwinkle',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:permanentblue.178','permanentblue','permanent blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:permanentblue.196','permanentblue','permanent blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:permanentred.12','permanentred','permanent red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:permanentred.15','permanentred','permanent red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:permanentviolet.218','permanentviolet','permanent violet',[135,86,146],'875692',8869522],
['nbs-iscc-m:permanentyellow.71','permanentyellow','permanent yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:permared.12','permared','perma red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:permared.15','permared','perma red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:pernambucowood.15','pernambucowood','pernambucowood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:pernambucowood.19','pernambucowood','pernambucowood',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:perruche.120','perruche','perruche',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:persenche.214','persenche','persenche',[150,144,171],'9690ab',9867435],
['nbs-iscc-m:persianblue.185','persianblue','persian blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:persianblue.204','persianblue','persian blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:persianearth.40','persianearth','persian earth',[136,45,23],'882d17',8924439],
['nbs-iscc-m:persiangreen.151','persiangreen','persian green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:persianlilac.251','persianlilac','persian lilac',[193,126,145],'c17e91',12680849],
['nbs-iscc-m:persianmelon.39','persianmelon','persian melon',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:persianorange.53','persianorange','persian orange',[217,144,88],'d99058',14258264],
['nbs-iscc-m:persianpink.258','persianpink','persian pink',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:persianred.40','persianred','persian red',[136,45,23],'882d17',8924439],
['nbs-iscc-m:persianrose.254','persianrose','persian rose',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:persianyellow.34','persianyellow','persian yellow',[226,88,34],'e25822',14833698],
['nbs-iscc-m:persianyellow.48','persianyellow','persian yellow',[243,132,0],'f38400',15959040],
['nbs-iscc-m:persimmon.55','persimmon','persimmon',[128,70,27],'80461b',8406555],
['nbs-iscc-m:persio.258','persio','persio',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:persis.258','persis','persis',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:peruvianbrown.54','peruvianbrown','peruvian brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:peruvianbrown.57','peruvianbrown','peruvian brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:peruvianyellow.37','peruvianyellow','peruvian yellow',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:pervenche.204','pervenche','pervenche',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:petunia.224','petunia','petunia',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:petuniaviolet.219','petuniaviolet','petunia [violet] ',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:pewke.16','pewke','pewke',[114,47,55],'722f37',7483191],
['nbs-iscc-m:pewter.266','pewter','pewter',[85,85,85],'555555',5592405],
['nbs-iscc-m:phantom.266','phantom','phantom',[85,85,85],'555555',5592405],
['nbs-iscc-m:phantomred.34','phantomred','phantom red',[226,88,34],'e25822',14833698],
['nbs-iscc-m:pharaoh.200','pharaoh','pharaoh',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:pharaoh.201','pharaoh','pharaoh',[37,36,64],'252440',2434112],
['nbs-iscc-m:pheasant.53','pheasant','pheasant',[217,144,88],'d99058',14258264],
['nbs-iscc-m:pheasant.57','pheasant','pheasant',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:philamot.51','philamot','philamot',[190,101,22],'be6516',12477718],
['nbs-iscc-m:philamot.54','philamot','philamot',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:philamot.55','philamot','philamot',[128,70,27],'80461b',8406555],
['nbs-iscc-m:philimot.51','philimot','philimot',[190,101,22],'be6516',12477718],
['nbs-iscc-m:philimot.54','philimot','philimot',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:philimot.55','philimot','philimot',[128,70,27],'80461b',8406555],
['nbs-iscc-m:philomot.51','philomot','philomot',[190,101,22],'be6516',12477718],
['nbs-iscc-m:philomot.54','philomot','philomot',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:philomot.55','philomot','philomot',[128,70,27],'80461b',8406555],
['nbs-iscc-m:phlox.238','phlox','phlox',[112,41,99],'702963',7350627],
['nbs-iscc-m:phlox.259','phlox','phlox',[103,49,71],'673147',6762823],
['nbs-iscc-m:phloxpink.227','phloxpink','phlox pink',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:phloxpurple.237','phloxpurple','phlox purple',[158,79,136],'9e4f88',10375048],
['nbs-iscc-m:phyliamort.51','phyliamort','phyliamort ',[190,101,22],'be6516',12477718],
['nbs-iscc-m:phyliamort.54','phyliamort','phyliamort ',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:phyliamort.55','phyliamort','phyliamort ',[128,70,27],'80461b',8406555],
['nbs-iscc-m:phyllamort.51','phyllamort','phyllamort',[190,101,22],'be6516',12477718],
['nbs-iscc-m:phyllamort.54','phyllamort','phyllamort',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:phyllamort.55','phyllamort','phyllamort',[128,70,27],'80461b',8406555],
['nbs-iscc-m:piccadilly.43','piccadilly','piccadilly',[121,68,59],'79443b',7947323],
['nbs-iscc-m:piccadilly.46','piccadilly','piccadilly',[103,76,71],'674c47',6769735],
['nbs-iscc-m:piccaninny.81','piccaninny','piccaninny',[72,60,50],'483c32',4734002],
['nbs-iscc-m:piccolpassored.43','piccolpassored','piccolpasso red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:pigeon.234','pigeon','pigeon',[93,85,91],'5d555b',6116699],
['nbs-iscc-m:pigeonblood.16','pigeonblood','pigeon blood',[114,47,55],'722f37',7483191],
['nbs-iscc-m:pigeonneck.266','pigeonneck','pigeon neck',[85,85,85],'555555',5592405],
['nbs-iscc-m:pigeonsbreast.19','pigeonsbreast','pigeon\'s-breast',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:pigeonsthroat.145','pigeonsthroat','pigeon\'s throat',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:pigskin.57','pigskin','pigskin',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:pigskin.58','pigskin','pigskin',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:pigskin.77','pigskin','pigskin',[130,102,68],'826644',8545860],
['nbs-iscc-m:pilgrim.266','pilgrim','pilgrim',[85,85,85],'555555',5592405],
['nbs-iscc-m:pilgrimbrown.61','pilgrimbrown','pilgrim brown',[99,81,71],'635147',6508871],
['nbs-iscc-m:pilgrimbrown.62','pilgrimbrown','pilgrim brown',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:pilotblue.182','pilotblue','pilot blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:pimento.11','pimento','pimento',[190,0,50],'be0032',12451890],
['nbs-iscc-m:pimlico.215','pimlico','pimlico',[85,76,105],'554c69',5590121],
['nbs-iscc-m:pinardyellow.86','pinardyellow','pinard yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:pinchbeckbrown.72','pinchbeckbrown','pinchbeck brown',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:pinchbeckbrown.74','pinchbeckbrown','pinchbeck brown',[153,101,21],'996515',10052885],
['nbs-iscc-m:pinchbeckbrown.76','pinchbeckbrown','pinchbeck brown',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:pinchbeckbrown.88','pinchbeckbrown','pinchbeck brown',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:pinchbeckbrown.94','pinchbeckbrown','pinchbeck brown',[150,113,23],'967117',9859351],
['nbs-iscc-m:pineapple.86','pineapple','pineapple',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:pineapple.87','pineapple','pineapple',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:pinecone.61','pinecone','pinecone',[99,81,71],'635147',6508871],
['nbs-iscc-m:pinecone.80','pinecone','pinecone',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:pinefrost.149','pinefrost','pine frost',[141,163,153],'8da399',9282457],
['nbs-iscc-m:pinegrove.151','pinegrove','pinegrove',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:pineneedle.146','pineneedle','pineneedle',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:pineneedle.151','pineneedle','pineneedle',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:pinetree.165','pinetree','pine tree',[0,75,73],'004b49',19273],
['nbs-iscc-m:pink1.10','pink1','pink 1 ',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-m:pink2.7','pink2','pink 2 ',[234,216,215],'ead8d7',15390935],
['nbs-iscc-m:pink3.4','pink3','pink 3 ',[249,204,202],'f9ccca',16370890],
['nbs-iscc-m:pink3.7','pink3','pink 3 ',[234,216,215],'ead8d7',15390935],
['nbs-iscc-m:pink4.4','pink4','pink 4 ',[249,204,202],'f9ccca',16370890],
['nbs-iscc-m:pink4.5','pink4','pink 4 ',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:pink5.5','pink5','pink 5 ',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:pinkcoral.3','pinkcoral','pink coral',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:pinkcoral.26','pinkcoral','pink coral',[248,131,121],'f88379',16286585],
['nbs-iscc-m:pinkpearl.5','pinkpearl','pink pearl',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:pipingrock.112','pipingrock','piping rock',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:piquantgreen.120','piquantgreen','piquant green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:pirate.12','pirate','pirate',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:pirate.35','pirate','pirate',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:pistache.120','pistache','pistache',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:pistache.135','pistache','pistache',[147,197,146],'93c592',9684370],
['nbs-iscc-m:pistachiogreen.120','pistachiogreen','pistachio green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:pistachiogreen.135','pistachiogreen','pistachio green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:pitchpine.151','pitchpine','pitchpine',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:piuree.68','piuree','piuree',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:piuree.71','piuree','piuree',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:piuri.68','piuri','piuri',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:piuri.71','piuri','piuri',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:piyu.145','piyu','pi yu',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:plantation.43','plantation','plantation',[121,68,59],'79443b',7947323],
['nbs-iscc-m:platinayellow.54','platinayellow','platina yellow',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:platinum.265','platinum','platinum',[132,132,130],'848482',8684674],
['nbs-iscc-m:plazagrey.93','plazagrey','plaza grey',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:pleromaviolet.223','pleromaviolet','pleroma violet',[134,96,142],'86608e',8806542],
['nbs-iscc-m:plover.64','plover','plover',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:plumbaceous.234','plumbaceous','plumbaceous',[93,85,91],'5d555b',6116699],
['nbs-iscc-m:plumbaceous.266','plumbaceous','plumbaceous',[85,85,85],'555555',5592405],
['nbs-iscc-m:plumbagoblue.227','plumbagoblue','plumbago blue',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:plumbagoblue.233','plumbagoblue','plumbago blue',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:plumbagogrey.227','plumbagogrey','plumbago grey ',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:plumbagogrey.233','plumbagogrey','plumbago grey ',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:plumbagoslate.234','plumbagoslate','plumbago slate',[93,85,91],'5d555b',6116699],
['nbs-iscc-m:plumbet.190','plumbet','plumbet',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-m:plumpurple.224','plumpurple','plum [purple] ',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:plumpurple.212','plumpurple','plum purple',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:plumviolet.16','plumviolet','plum violet',[114,47,55],'722f37',7483191],
['nbs-iscc-m:plumviolet.259','plumviolet','plum violet',[103,49,71],'673147',6762823],
['nbs-iscc-m:plunket.190','plunket','plunket',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-m:plymouth.266','plymouth','plymouth',[85,85,85],'555555',5592405],
['nbs-iscc-m:poildours.266','poildours','poil d\'ours',[85,85,85],'555555',5592405],
['nbs-iscc-m:poilu.191','poilu','poilu',[129,135,139],'81878b',8488843],
['nbs-iscc-m:poinsettia.11','poinsettia','poinsettia',[190,0,50],'be0032',12451890],
['nbs-iscc-m:poinsettia.12','poinsettia','poinsettia',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:poisgreen.122','poisgreen','pois green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:polarbear.73','polarbear','polar bear',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:polarbear.92','polarbear','polar bear',[240,234,214],'f0ead6',15788758],
['nbs-iscc-m:polignac.18','polignac','polignac',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:polignac.42','polignac','polignac',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:polignac.45','polignac','polignac',[151,127,115],'977f73',9928563],
['nbs-iscc-m:pologreen.125','pologreen','polo green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:pologreen.137','pologreen','polo green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:polotan.76','polotan','polo tan',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:pomegranate.16','pomegranate','pomegranate',[114,47,55],'722f37',7483191],
['nbs-iscc-m:pomegranateblossom.11','pomegranateblossom','pomegranate blossom',[190,0,50],'be0032',12451890],
['nbs-iscc-m:pomegranateblossom.34','pomegranateblossom','pomegranate blossom',[226,88,34],'e25822',14833698],
['nbs-iscc-m:pomegranatepurple.15','pomegranatepurple','pomegranate purple',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:pomegranatepurple.258','pomegranatepurple','pomegranate purple',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:pomegranatepurple.258','pomegranatepurple','pomegranate purple',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:pomegranatepurple.262','pomegranatepurple','pomegranate purple',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:pomonagreen.120','pomonagreen','pomona green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:pompadourgreen.182','pompadourgreen','pompadour [green] ',[67,107,149],'436b95',4418453],
['nbs-iscc-m:pompandpower.223','pompandpower','pomp and power',[134,96,142],'86608e',8806542],
['nbs-iscc-m:pompeianblue.185','pompeianblue','pompeian blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:pompeianred.19','pompeianred','pompeian red',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:pompeianred.39','pompeianred','pompeian red',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:pompeianred.42','pompeianred','pompeian red',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:pompeianyellow.71','pompeianyellow','pompeian yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:pompeii.41','pompeii','pompeii',[86,7,12],'56070c',5637900],
['nbs-iscc-m:pompeii.43','pompeii','pompeii',[121,68,59],'79443b',7947323],
['nbs-iscc-m:ponceau.11','ponceau','ponceau',[190,0,50],'be0032',12451890],
['nbs-iscc-m:ponceau.34','ponceau','ponceau',[226,88,34],'e25822',14833698],
['nbs-iscc-m:poncedeleon.50','poncedeleon','ponce de leon',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:pondlily.86','pondlily','pond lily',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:pondlily.87','pondlily','pond lily',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:pongee.76','pongee','pongee',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:pontiffpurple.219','pontiffpurple','pontiff [purple] ',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:ponybrown.61','ponybrown','pony brown',[99,81,71],'635147',6508871],
['nbs-iscc-m:popcorn.86','popcorn','popcorn',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:popinjaygreen.120','popinjaygreen','popinjay green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:poplar.151','poplar','poplar',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:poppyred.11','poppyred','poppy [red] ',[190,0,50],'be0032',12451890],
['nbs-iscc-m:poppyred.34','poppyred','poppy [red] ',[226,88,34],'e25822',14833698],
['nbs-iscc-m:porcelain.182','porcelain','porcelain',[67,107,149],'436b95',4418453],
['nbs-iscc-m:porcelainblue.186','porcelainblue','porcelain blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:porcelaingreen.164','porcelaingreen','porcelain green',[49,120,115],'317873',3242099],
['nbs-iscc-m:porraaceous.120','porraaceous','porraaceous',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:porret.120','porret','porret',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:port.17','port','port',[63,23,40],'3f1728',4134696],
['nbs-iscc-m:portablered.12','portablered','portable red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:portablered.15','portablered','portable red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:portablered.36','portablered','portable red',[170,56,30],'aa381e',11155486],
['nbs-iscc-m:portablered.38','portablered','portable red',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:portugeseredorportuguesered.11','portugeseredorportuguesered','portugese red or portuguese red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:portwine.47','portwine','port wine',[67,48,46],'43302e',4403246],
['nbs-iscc-m:postofficered.36','postofficered','post office red',[170,56,30],'aa381e',11155486],
['nbs-iscc-m:postofficered.38','postofficered','post office red',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:posygreen.150','posygreen','posy green',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:poudre.185','poudre','poudre',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:poudre.191','poudre','poudre',[129,135,139],'81878b',8488843],
['nbs-iscc-m:poudre.190','poudre','poudre',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-m:poudreblue.185','poudreblue','poudre blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:poudreblue.191','poudreblue','poudre blue',[129,135,139],'81878b',8488843],
['nbs-iscc-m:powderblue.185','powderblue','powder blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:powderblue.191','powderblue','powder blue',[129,135,139],'81878b',8488843],
['nbs-iscc-m:powderblue.179','powderblue','powder blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:powderblue.182','powderblue','powder blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:powderedgold.91','powderedgold','powdered gold',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:powderpink.29','powderpink','powder pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:prairie.76','prairie','prairie',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:prairiebrown.54','prairiebrown','prairie brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:praline.58','praline','praline',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:praline.61','praline','praline',[99,81,71],'635147',6508871],
['nbs-iscc-m:prawnpink.26','prawnpink','prawn [pink] ',[248,131,121],'f88379',16286585],
['nbs-iscc-m:prelate.211','prelate','prelate',[96,78,129],'604e81',6311553],
['nbs-iscc-m:primrosegreen.104','primrosegreen','primrose green',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-m:primroseyellow.86','primroseyellow','primrose yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:primroseyellow.87','primroseyellow','primrose yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:primulineyellow.84','primulineyellow','primuline yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:princegrey.233','princegrey','prince grey ',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:princetonorange.50','princetonorange','princeton orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:priscilla.264','priscilla','priscilla',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-m:privet.127','privet','privet',[81,87,68],'515744',5330756],
['nbs-iscc-m:proutsbrown.58','proutsbrown','prout\'s brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:proutsbrown.61','proutsbrown','prout\'s brown',[99,81,71],'635147',6508871],
['nbs-iscc-m:proutsbrown.77','proutsbrown','prout\'s brown',[130,102,68],'826644',8545860],
['nbs-iscc-m:proutsbrown.80','proutsbrown','prout\'s brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:prune.242','prune','prune',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:prunella.208','prunella','prunella',[50,23,77],'32174d',3282765],
['nbs-iscc-m:prunella.212','prunella','prunella',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:prunepurple.224','prunepurple','prune purple',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:prussianbrown.54','prussianbrown','prussian brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:prussianred.40','prussianred','prussian red',[136,45,23],'882d17',8924439],
['nbs-iscc-m:prussianred.43','prussianred','prussian red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:psyche.135','psyche','psyche',[147,197,146],'93c592',9684370],
['nbs-iscc-m:ptshell.73','ptshell','pt shell',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:ptshell.76','ptshell','pt shell',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:puce.16','puce','puce',[114,47,55],'722f37',7483191],
['nbs-iscc-m:pueblo.58','pueblo','pueblo',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:puke.16','puke','puke',[114,47,55],'722f37',7483191],
['nbs-iscc-m:pumpkin.50','pumpkin','pumpkin',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:punjab.51','punjab','punjab',[190,101,22],'be6516',12477718],
['nbs-iscc-m:punjab.54','punjab','punjab',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:puritangrey.191','puritangrey','puritan grey ',[129,135,139],'81878b',8488843],
['nbs-iscc-m:puritangrey.265','puritangrey','puritan grey ',[132,132,130],'848482',8684674],
['nbs-iscc-m:purpleaster.223','purpleaster','purple aster',[134,96,142],'86608e',8806542],
['nbs-iscc-m:purplebrown.43','purplebrown','purple brown',[121,68,59],'79443b',7947323],
['nbs-iscc-m:purpleheather.223','purpleheather','purple heather',[134,96,142],'86608e',8806542],
['nbs-iscc-m:purplelake.13','purplelake','purple lake',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:purplelake.15','purplelake','purple lake',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:purplemadder.15','purplemadder','purple madder',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:purplemadder.16','purplemadder','purple madder',[114,47,55],'722f37',7483191],
['nbs-iscc-m:purplenavy.183','purplenavy','purple navy',[0,48,78],'00304e',12366],
['nbs-iscc-m:purplenavy.200','purplenavy','purple navy',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:purplenavy.201','purplenavy','purple navy',[37,36,64],'252440',2434112],
['nbs-iscc-m:purpleochre.40','purpleochre','purple ochre',[136,45,23],'882d17',8924439],
['nbs-iscc-m:purpleoxide.43','purpleoxide','purple oxide',[121,68,59],'79443b',7947323],
['nbs-iscc-m:purplerubiate.15','purplerubiate','purple rubiate',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:purplerubiate.16','purplerubiate','purple rubiate',[114,47,55],'722f37',7483191],
['nbs-iscc-m:purreaarabica.68','purreaarabica','purrea arabica',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:purreaarabica.71','purreaarabica','purrea arabica',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:purree.68','purree','purree',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:purree.71','purree','purree',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:putty.89','putty','putty',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:putty.90','putty','putty',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:pygmalion.74','pygmalion','pygmalion',[153,101,21],'996515',10052885],
['nbs-iscc-m:pygmalion.76','pygmalion','pygmalion',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:pygmalion.77','pygmalion','pygmalion',[130,102,68],'826644',8545860],
['nbs-iscc-m:pyrethrumyellow.83','pyrethrumyellow','pyrethrum yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:pyrethrumyellow.84','pyrethrumyellow','pyrethrum yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:pyrethrumyellow.86','pyrethrumyellow','pyrethrum yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:pyrethrumyellow.87','pyrethrumyellow','pyrethrum yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:pyriteyellow.88','pyriteyellow','pyrite yellow',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:quail.64','quail','quail',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:quail.266','quail','quail',[85,85,85],'555555',5592405],
['nbs-iscc-m:quaker.233','quaker','quaker',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:quaker.265','quaker','quaker',[132,132,130],'848482',8684674],
['nbs-iscc-m:quakerblue.267','quakerblue','quaker blue',[34,34,34],'222222',2236962],
['nbs-iscc-m:quakerdrab.265','quakerdrab','quaker drab',[132,132,130],'848482',8684674],
['nbs-iscc-m:quakergreen.125','quakergreen','quaker green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:quakergrey.109','quakergrey','quaker grey ',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:queenannegreen.122','queenannegreen','queen anne green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:queenblue.182','queenblue','queen blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:queensblue.182','queensblue','queen\'s blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:queensyellow.86','queensyellow','queen\'s yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:quercitron.86','quercitron','quercitron',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:quercitronlake.71','quercitronlake','quercitron lake',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:quimper.186','quimper','quimper',[83,104,120],'536878',5466232],
['nbs-iscc-m:quinceyellow.87','quinceyellow','quince yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:rabbit.64','rabbit','rabbit',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:racquet.61','racquet','racquet',[99,81,71],'635147',6508871],
['nbs-iscc-m:racquet.80','racquet','racquet',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:raddle.43','raddle','raddle',[121,68,59],'79443b',7947323],
['nbs-iscc-m:radiance.3','radiance','radiance',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:radiantyellow.50','radiantyellow','radiant yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:radiantyellow.53','radiantyellow','radiant yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-m:radiantyellow.68','radiantyellow','radiant yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:radiantyellow.71','radiantyellow','radiant yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:radio.183','radio','radio',[0,48,78],'00304e',12366],
['nbs-iscc-m:radio.186','radio','radio',[83,104,120],'536878',5466232],
['nbs-iscc-m:radioblue.187','radioblue','radio blue',[54,69,79],'36454f',3556687],
['nbs-iscc-m:raffia.76','raffia','raffia',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:raffia.90','raffia','raffia',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:raggedsailor.200','raggedsailor','ragged sailor',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:rail.266','rail','rail',[85,85,85],'555555',5592405],
['nbs-iscc-m:rainettegreen.120','rainettegreen','rainette green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:raisin.259','raisin','raisin',[103,49,71],'673147',6762823],
['nbs-iscc-m:raisinblack.235','raisinblack','raisin black',[36,33,36],'242124',2367780],
['nbs-iscc-m:raisinpurple.238','raisinpurple','raisin purple',[112,41,99],'702963',7350627],
['nbs-iscc-m:ramblerrose.3','ramblerrose','rambler rose',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:ramblerrose.258','ramblerrose','rambler rose',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:rameau.267','rameau','rameau',[34,34,34],'222222',2236962],
['nbs-iscc-m:rameses.179','rameses','rameses',[0,65,106],'00416a',16746],
['nbs-iscc-m:ramierblue.228','ramierblue','ramier blue',[121,104,120],'796878',7956600],
['nbs-iscc-m:rangoon.58','rangoon','rangoon',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:raphael.182','raphael','raphael',[67,107,149],'436b95',4418453],
['nbs-iscc-m:raphael.183','raphael','raphael',[0,48,78],'00304e',12366],
['nbs-iscc-m:rapids.182','rapids','rapids',[67,107,149],'436b95',4418453],
['nbs-iscc-m:raspberry.16','raspberry','raspberry',[114,47,55],'722f37',7483191],
['nbs-iscc-m:raspberryglac.15','raspberryglac','raspberry glac',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:raspberryglac.19','raspberryglac','raspberry glac',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:raspberryglac.258','raspberryglac','raspberry glac',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:raspberryglac.262','raspberryglac','raspberry glac',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:raspberryred.15','raspberryred','raspberry red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:rat.113','rat','rat',[87,85,76],'57554c',5723468],
['nbs-iscc-m:rattan.87','rattan','rattan',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:rawitalianearth.54','rawitalianearth','raw italian earth',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:rawitalianearth.57','rawitalianearth','raw italian earth',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:rawsienna.54','rawsienna','raw sienna',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:rawsienna.57','rawsienna','raw sienna',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:rawumber.77','rawumber','raw umber',[130,102,68],'826644',8545860],
['nbs-iscc-m:rawumber.78','rawumber','raw umber',[75,54,33],'4b3621',4929057],
['nbs-iscc-m:rawumber.58','rawumber','raw umber',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:rcamier.5','rcamier','rcamier',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:rcamier.26','rcamier','rcamier',[248,131,121],'f88379',16286585],
['nbs-iscc-m:realgar.68','realgar','realgar',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:realgar.71','realgar','realgar',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:redbanana.13','redbanana','red banana',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:redbanana.15','redbanana','red banana',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:redbole.43','redbole','red bole',[121,68,59],'79443b',7947323],
['nbs-iscc-m:redchalk.43','redchalk','red chalk',[121,68,59],'79443b',7947323],
['nbs-iscc-m:redcross.11','redcross','red cross',[190,0,50],'be0032',12451890],
['nbs-iscc-m:redcross.12','redcross','red cross',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:redding.43','redding','redding',[121,68,59],'79443b',7947323],
['nbs-iscc-m:reddle.43','reddle','reddle',[121,68,59],'79443b',7947323],
['nbs-iscc-m:redearth.43','redearth','red earth',[121,68,59],'79443b',7947323],
['nbs-iscc-m:redfeather.12','redfeather','redfeather',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:redfeather.15','redfeather','redfeather',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:redinplates.11','redinplates','red in plates',[190,0,50],'be0032',12451890],
['nbs-iscc-m:redlead.34','redlead','red lead',[226,88,34],'e25822',14833698],
['nbs-iscc-m:redochre.43','redochre','red ochre',[121,68,59],'79443b',7947323],
['nbs-iscc-m:redoxide.40','redoxide','red oxide',[136,45,23],'882d17',8924439],
['nbs-iscc-m:redoxide.55','redoxide','red oxide',[128,70,27],'80461b',8406555],
['nbs-iscc-m:redrobin.41','redrobin','red robin',[86,7,12],'56070c',5637900],
['nbs-iscc-m:redwood.16','redwood','redwood',[114,47,55],'722f37',7483191],
['nbs-iscc-m:redwood.40','redwood','redwood',[136,45,23],'882d17',8924439],
['nbs-iscc-m:redwood.43','redwood','redwood',[121,68,59],'79443b',7947323],
['nbs-iscc-m:redwoodredwood.15','redwoodredwood','redwood redwood -)',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:redwoodredwood.15','redwoodredwood','redwood redwood -)',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:redwoodredwood.19','redwoodredwood','redwood redwood -)',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:reedgreen.119','reedgreen','reed green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-m:reedyellow.101','reedyellow','reed yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-m:reedyellow.104','reedyellow','reed yellow',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-m:regal.197','regal','regal',[39,36,88],'272458',2565208],
['nbs-iscc-m:regalpurple.207','regalpurple','regal purple',[96,78,151],'604e97',6311575],
['nbs-iscc-m:regatta.178','regatta','regatta',[0,103,165],'0067a5',26533],
['nbs-iscc-m:regatta.196','regatta','regatta',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:regimental.204','regimental','regimental',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:rejanegreen.136','rejanegreen','rejane green',[103,146,103],'679267',6787687],
['nbs-iscc-m:rembrandt.62','rembrandt','rembrandt',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:rembrandtsmadder.37','rembrandtsmadder','rembrandt\'s madder',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:rembrandtsmadder.38','rembrandtsmadder','rembrandt\'s madder',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:rembrandtsmadder.39','rembrandtsmadder','rembrandt\'s madder',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:rembrandtsmadder.43','rembrandtsmadder','rembrandt\'s madder',[121,68,59],'79443b',7947323],
['nbs-iscc-m:renaissance.254','renaissance','renaissance',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:resedagreen.106','resedagreen','reseda [green] ',[134,126,54],'867e36',8814134],
['nbs-iscc-m:resolute.182','resolute','resolute',[67,107,149],'436b95',4418453],
['nbs-iscc-m:reveree.31','reveree','reveree',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-m:reveree.32','reveree','reveree',[199,173,163],'c7ada3',13086115],
['nbs-iscc-m:rhododendron.259','rhododendron','rhododendron',[103,49,71],'673147',6762823],
['nbs-iscc-m:rhodonitepink.251','rhodonitepink','rhodonite pink',[193,126,145],'c17e91',12680849],
['nbs-iscc-m:rhone.19','rhone','rhone',[201,220,137],'c9dc89',13229193],
['nbs-iscc-m:rhone.121','rhone','rhone',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:rhubarb.106','rhubarb','rhubarb',[134,126,54],'867e36',8814134],
['nbs-iscc-m:riflegreen.128','riflegreen','rifle [green] ',[49,54,43],'31362b',3225131],
['nbs-iscc-m:riflegreen.267','riflegreen','rifle [green] ',[34,34,34],'222222',2236962],
['nbs-iscc-m:riga.164','riga','riga',[49,120,115],'317873',3242099],
['nbs-iscc-m:rigablue.151','rigablue','riga blue',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:rigablue.186','rigablue','riga blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:rinnemannsgreen.136','rinnemannsgreen','rinnemann\'s green',[103,146,103],'679267',6787687],
['nbs-iscc-m:ripplegreen.150','ripplegreen','ripple green',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:ripplegreen.164','ripplegreen','ripple green',[49,120,115],'317873',3242099],
['nbs-iscc-m:risigal.68','risigal','risigal',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:risigal.71','risigal','risigal',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:rivagegreen.120','rivagegreen','rivage green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:rivagegreen.135','rivagegreen','rivage green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:riviera.178','riviera','riviera',[0,103,165],'0067a5',26533],
['nbs-iscc-m:roan.43','roan','roan',[121,68,59],'79443b',7947323],
['nbs-iscc-m:robinhoodgreen.146','robinhoodgreen','robinhood green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:robinseggblue.149','robinseggblue','robin\'s egg blue',[141,163,153],'8da399',9282457],
['nbs-iscc-m:robinseggblue.163','robinseggblue','robin\'s egg blue',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:rocou.29','rocou','rocou',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:rodent.266','rodent','rodent',[85,85,85],'555555',5592405],
['nbs-iscc-m:roe.76','roe','roe',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:roe.91','roe','roe',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:romablue.204','romablue','roma blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:romablue.215','romablue','roma blue',[85,76,105],'554c69',5590121],
['nbs-iscc-m:romanearth.53','romanearth','roman earth',[217,144,88],'d99058',14258264],
['nbs-iscc-m:romanesque.242','romanesque','romanesque',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:romangreen.107','romangreen','roman green',[102,93,30],'665d1e',6708510],
['nbs-iscc-m:romanlake.11','romanlake','roman lake',[190,0,50],'be0032',12451890],
['nbs-iscc-m:romanlake.12','romanlake','roman lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:romanochre.53','romanochre','roman ochre',[217,144,88],'d99058',14258264],
['nbs-iscc-m:romanpurple.211','romanpurple','roman purple',[96,78,129],'604e81',6311553],
['nbs-iscc-m:romansepia.58','romansepia','roman sepia',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:romanticgreen.106','romanticgreen','romantic green',[134,126,54],'867e36',8814134],
['nbs-iscc-m:romanumber.77','romanumber','roman umber',[130,102,68],'826644',8545860],
['nbs-iscc-m:romanumber.78','romanumber','roman umber',[75,54,33],'4b3621',4929057],
['nbs-iscc-m:romanviolet.242','romanviolet','roman violet',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:romany.187','romany','romany',[54,69,79],'36454f',3556687],
['nbs-iscc-m:romany.188','romany','romany',[32,40,48],'202830',2107440],
['nbs-iscc-m:romany.201','romany','romany',[37,36,64],'252440',2434112],
['nbs-iscc-m:romany.204','romany','romany',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:romarin.151','romarin','romarin',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:rosalgar.68','rosalgar','rosalgar',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:rosalgar.71','rosalgar','rosalgar',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:rosario.57','rosario','rosario',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:roseamber.54','roseamber','rose amber',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:roseamber.57','roseamber','rose amber',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:roseash.19','roseash','rose ash',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:rosebeige.80','rosebeige','rose beige',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:rosebeige2.57','rosebeige2','rose beige 2 ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:rosebisque.42','rosebisque','rosebisque',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:rosebloom.18','rosebloom','rosebloom',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:rosebloom.30','rosebloom','rosebloom',[196,131,121],'c48379',12878713],
['nbs-iscc-m:rosebloom.39','rosebloom','rosebloom',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:rosebloom.42','rosebloom','rosebloom',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:roseblush.42','roseblush','rose blush',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:roseblush2.57','roseblush2','rose blush 2 ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:rosebreath.4','rosebreath','rose breath',[249,204,202],'f9ccca',16370890],
['nbs-iscc-m:rosebreath.7','rosebreath','rose breath',[234,216,215],'ead8d7',15390935],
['nbs-iscc-m:rosebud.248','rosebud','rosebud',[222,111,161],'de6fa1',14577569],
['nbs-iscc-m:rosebud.251','rosebud','rosebud',[193,126,145],'c17e91',12680849],
['nbs-iscc-m:rosecaroline.29','rosecaroline','rose caroline',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:rosecarthame.11','rosecarthame','rose carthame',[190,0,50],'be0032',12451890],
['nbs-iscc-m:rosecastor.22','rosecastor','rose castor',[143,129,127],'8f817f',9404799],
['nbs-iscc-m:rosecendre.29','rosecendre','rose cendre',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:rosedalma.262','rosedalma','rose d\'alma',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:rosedalthoea.2','rosedalthoea','rose d\'althoea',[234,147,153],'ea9399',15373209],
['nbs-iscc-m:rosedawn.29','rosedawn','rose dawn',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:rosedawn.42','rosedawn','rose dawn',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:rosedenymphe.89','rosedenymphe','rose de nymphe',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:rosedeprovence.255','rosedeprovence','rose de provence',[179,68,108],'b3446c',11748460],
['nbs-iscc-m:rosedore.3','rosedore','rose dore',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:rosedore.15','rosedore','rose dore',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:rosedust.19','rosedust','rosedust',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:rosedust.46','rosedust','rosedust',[103,76,71],'674c47',6769735],
['nbs-iscc-m:roseebony.47','roseebony','rose ebony ',[67,48,46],'43302e',4403246],
['nbs-iscc-m:rosefrance.29','rosefrance','rose france',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:roseglow.42','roseglow','roseglow',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:rosegrey.64','rosegrey','rose grey ',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:rosehermosa.5','rosehermosa','rose hermosa',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:rosehortensia.42','rosehortensia','rose hortensia',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:roselake.38','roselake','rose lake',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:roseleaf.5','roseleaf','roseleaf',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:roselustre.16','roselustre','roselustre',[114,47,55],'722f37',7483191],
['nbs-iscc-m:rosemadder.255','rosemadder','rose madder',[179,68,108],'b3446c',11748460],
['nbs-iscc-m:rosemarie.3','rosemarie','rose marie',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:rosemary.245','rosemary','rosemary',[131,100,121],'836479',8610937],
['nbs-iscc-m:rosemorn.29','rosemorn','rose morn',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:roseneyron.12','roseneyron','rose neyron',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:roseneyron.255','roseneyron','rose neyron',[179,68,108],'b3446c',11748460],
['nbs-iscc-m:rosenilsson.3','rosenilsson','rose nilsson',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:rosenude.29','rosenude','rose nude',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:rosenude.32','rosenude','rose nude',[199,173,163],'c7ada3',13086115],
['nbs-iscc-m:rosenude.33','rosenude','rose nude',[194,172,153],'c2ac99',12758169],
['nbs-iscc-m:roseoak.43','roseoak','rose oak',[121,68,59],'79443b',7947323],
['nbs-iscc-m:roseofpicardy.15','roseofpicardy','rose of picardy',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:roseofsharon.15','roseofsharon','rose of sharon',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:rosepetal.19','rosepetal','rose petal',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:rosepink.38','rosepink','rose pink',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:rosepurple.222','rosepurple','rose-purple ',[182,149,192],'b695c0',11965888],
['nbs-iscc-m:rosepurple.240','rosepurple','rose-purple ',[183,132,167],'b784a7',12027047],
['nbs-iscc-m:rosequartz.227','rosequartz','rose quartz',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:rosesoire.3','rosesoire','rose soire',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:rosestone.45','rosestone','rosestone',[151,127,115],'977f73',9928563],
['nbs-iscc-m:roset.38','roset','roset',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:rosetan.33','rosetan','rosetan',[194,172,153],'c2ac99',12758169],
['nbs-iscc-m:rosetan.60','rosetan','rosetan',[149,128,112],'958070',9797744],
['nbs-iscc-m:rosetaupe.266','rosetaupe','rose taupe',[85,85,85],'555555',5592405],
['nbs-iscc-m:rosetta.23','rosetta','rosetta',[92,80,79],'5c504f',6049871],
['nbs-iscc-m:rosevale.15','rosevale','rosevale',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:rosevale.19','rosevale','rosevale',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:rosewood.43','rosewood','rosewood',[121,68,59],'79443b',7947323],
['nbs-iscc-m:roslynblue.197','roslynblue','roslyn blue',[39,36,88],'272458',2565208],
['nbs-iscc-m:roucou.29','roucou','roucou',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:rougedefer.43','rougedefer','rouge de fer',[121,68,59],'79443b',7947323],
['nbs-iscc-m:rougevegetal.11','rougevegetal','rouge vegetal',[190,0,50],'be0032',12451890],
['nbs-iscc-m:royalblue.179','royalblue','royal blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:royalblue.182','royalblue','royal blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:royalblue.179','royalblue','royal blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:royalpink.3','royalpink','royal pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:royalpurple.207','royalpurple','royal purple',[96,78,151],'604e97',6311575],
['nbs-iscc-m:royalyellow.83','royalyellow','royal yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:royalyellow.84','royalyellow','royal yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:royalyellow.86','royalyellow','royal yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:royalyellow.87','royalyellow','royal yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:rubaiyat.16','rubaiyat','rubaiyat',[114,47,55],'722f37',7483191],
['nbs-iscc-m:rubber.95','rubber','rubber',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:rubelite.254','rubelite','rubelite',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:rubensmadder.38','rubensmadder','ruben\'s madder',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:rubient.16','rubient','rubient',[114,47,55],'722f37',7483191],
['nbs-iscc-m:rubrica.43','rubrica','rubrica',[121,68,59],'79443b',7947323],
['nbs-iscc-m:ruby.16','ruby','ruby',[114,47,55],'722f37',7483191],
['nbs-iscc-m:rubyofarsenic.68','rubyofarsenic','ruby of arsenic',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:rubyofarsenic.71','rubyofarsenic','ruby of arsenic',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:ruddle.43','ruddle','ruddle',[121,68,59],'79443b',7947323],
['nbs-iscc-m:rufous.26','rufous','rufous',[248,131,121],'f88379',16286585],
['nbs-iscc-m:rufous.53','rufous','rufous',[217,144,88],'d99058',14258264],
['nbs-iscc-m:rugbytan.55','rugbytan','rugby tan',[128,70,27],'80461b',8406555],
['nbs-iscc-m:runnymede.145','runnymede','runnymede',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:ruochre.54','ruochre','ru ochre',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:ruochre.74','ruochre','ru ochre',[153,101,21],'996515',10052885],
['nbs-iscc-m:russetbrown.55','russetbrown','russet brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:russetbrown.58','russetbrown','russet brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:russetgreen.102','russetgreen','russet green',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:russetgreen.103','russetgreen','russet green',[152,148,62],'98943e',9999422],
['nbs-iscc-m:russetgreen.105','russetgreen','russet green',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:russetgreen.106','russetgreen','russet green',[134,126,54],'867e36',8814134],
['nbs-iscc-m:russetorange.35','russetorange','russet orange',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:russianblue.185','russianblue','russian blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:russianblue.191','russianblue','russian blue',[129,135,139],'81878b',8488843],
['nbs-iscc-m:russiancalf.58','russiancalf','russian calf',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:russiangreen.136','russiangreen','russian green',[103,146,103],'679267',6787687],
['nbs-iscc-m:russiangreen.137','russiangreen','russian green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:russiangreen.150','russiangreen','russian green',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:russiangrey.112','russiangrey','russian grey ',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:russiangrey.265','russiangrey','russian grey ',[132,132,130],'848482',8684674],
['nbs-iscc-m:russianviolet.208','russianviolet','russian violet',[50,23,77],'32174d',3282765],
['nbs-iscc-m:rust.55','rust','rust',[128,70,27],'80461b',8406555],
['nbs-iscc-m:rusticbrown.43','rusticbrown','rustic brown',[121,68,59],'79443b',7947323],
['nbs-iscc-m:rusticdrab.94','rusticdrab','rustic drab',[150,113,23],'967117',9859351],
['nbs-iscc-m:rutochre.54','rutochre','rut ochre',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:rutochre.74','rutochre','rut ochre',[153,101,21],'996515',10052885],
['nbs-iscc-m:sable.80','sable','sable',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:saddle.61','saddle','saddle',[99,81,71],'635147',6508871],
['nbs-iscc-m:saddle.80','saddle','saddle',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:safaran.53','safaran','safaran',[217,144,88],'d99058',14258264],
['nbs-iscc-m:safaran.71','safaran','safaran',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:safflor.11','safflor','safflor',[190,0,50],'be0032',12451890],
['nbs-iscc-m:safflowerred.11','safflowerred','safflower red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:saffor.11','saffor','saffor',[190,0,50],'be0032',12451890],
['nbs-iscc-m:saffronyellow.53','saffronyellow','saffron yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-m:saffronyellow.71','saffronyellow','saffron yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:safranopink.29','safranopink','safrano pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:sagebrushgreen.156','sagebrushgreen','sagebrush green',[78,87,85],'4e5755',5134165],
['nbs-iscc-m:sagedrab.127','sagedrab','sage drab',[81,87,68],'515744',5330756],
['nbs-iscc-m:sagegreen.120','sagegreen','sage [green] ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:sagegrey.109','sagegrey','sage grey ',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:sahara.58','sahara','sahara',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:sailor.230','sailor','sailor',[41,30,41],'291e29',2694697],
['nbs-iscc-m:sailor.235','sailor','sailor',[36,33,36],'242124',2367780],
['nbs-iscc-m:sailorblue.200','sailorblue','sailor blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-m:sakkara.64','sakkara','sakkara',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:sallow.105','sallow','sallow',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:salmonpink.29','salmonpink','salmon [pink] ',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:salvia.11','salvia','salvia',[190,0,50],'be0032',12451890],
['nbs-iscc-m:salviablue.178','salviablue','salvia blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:samara.164','samara','samara',[49,120,115],'317873',3242099],
['nbs-iscc-m:samovar.76','samovar','samovar',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:samurai.39','samurai','samurai',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:sand.109','sand','sand',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:sand.112','sand','sand',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:sandalwood.15','sandalwood','sandalwood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:sandalwood.45','sandalwood','sandalwood',[151,127,115],'977f73',9928563],
['nbs-iscc-m:sandalwood.58','sandalwood','sandalwood',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:sandalwood.60','sandalwood','sandalwood',[149,128,112],'958070',9797744],
['nbs-iscc-m:sandalwood.61','sandalwood','sandalwood',[99,81,71],'635147',6508871],
['nbs-iscc-m:sandarach.34','sandarach','sandarach',[226,88,34],'e25822',14833698],
['nbs-iscc-m:sandarach.68','sandarach','sandarach',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:sandarach.71','sandarach','sandarach',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:sandaracha.34','sandaracha','sandaracha',[226,88,34],'e25822',14833698],
['nbs-iscc-m:sandaracha.68','sandaracha','sandaracha',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:sandaracha.71','sandaracha','sandaracha',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:sanddune.94','sanddune','sand dune',[150,113,23],'967117',9859351],
['nbs-iscc-m:sandersblue.182','sandersblue','sander\'s blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:sanderswood.15','sanderswood','sanderswood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:sandix.34','sandix','sandix',[226,88,34],'e25822',14833698],
['nbs-iscc-m:sandrift.63','sandrift','sandrift',[142,130,121],'8e8279',9339513],
['nbs-iscc-m:sandstone.42','sandstone','sandstone',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:sandstone.57','sandstone','sandstone',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:sandstone.60','sandstone','sandstone',[149,128,112],'958070',9797744],
['nbs-iscc-m:sandust.29','sandust','sandust',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:sandybeige.63','sandybeige','sandy-beige',[142,130,121],'8e8279',9339513],
['nbs-iscc-m:sandyx.34','sandyx','sandyx',[226,88,34],'e25822',14833698],
['nbs-iscc-m:sangdeboeuf.43','sangdeboeuf','sang de boeuf',[121,68,59],'79443b',7947323],
['nbs-iscc-m:sanguine.12','sanguine','sanguine',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:sanguine.15','sanguine','sanguine',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:sanguineus.13','sanguineus','sanguineus',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:sanguineus.16','sanguineus','sanguineus',[114,47,55],'722f37',7483191],
['nbs-iscc-m:sanguineus.17','sanguineus','sanguineus',[63,23,40],'3f1728',4134696],
['nbs-iscc-m:sanguineus.44','sanguineus','sanguineus',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-m:sanguineus.242','sanguineus','sanguineus',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:sanguineus.259','sanguineus','sanguineus',[103,49,71],'673147',6762823],
['nbs-iscc-m:santalwood.15','santalwood','santalwood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:santos.58','santos','santos',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:santos.61','santos','santos',[99,81,71],'635147',6508871],
['nbs-iscc-m:saona.43','saona','saona',[121,68,59],'79443b',7947323],
['nbs-iscc-m:sappanwood.15','sappanwood','sappanwood',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:sappanwood.19','sappanwood','sappanwood',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:sapphireblue.183','sapphireblue','sapphire [blue] ',[0,48,78],'00304e',12366],
['nbs-iscc-m:sappho.18','sappho','sappho',[173,136,132],'ad8884',11372676],
['nbs-iscc-m:saraband.19','saraband','saraband',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:saratoga.53','saratoga','saratoga',[217,144,88],'d99058',14258264],
['nbs-iscc-m:saravan.55','saravan','saravan',[128,70,27],'80461b',8406555],
['nbs-iscc-m:saravan.58','saravan','saravan',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:sarouk.61','sarouk','sarouk',[99,81,71],'635147',6508871],
['nbs-iscc-m:satinwood.57','satinwood','satinwood',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:satsuma.37','satsuma','satsuma',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:saturninered.34','saturninered','saturnine red',[226,88,34],'e25822',14833698],
['nbs-iscc-m:saturnred.34','saturnred','saturn red',[226,88,34],'e25822',14833698],
['nbs-iscc-m:saul.156','saul','saul',[78,87,85],'4e5755',5134165],
['nbs-iscc-m:saundersblue.182','saundersblue','saunder\'s blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:sauterne.87','sauterne','sauterne',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:sauterne.57','sauterne','sauterne',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:saxeblue.186','saxeblue','saxe blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:saxonblue.169','saxonblue','saxon blue',[0,119,145],'007791',30609],
['nbs-iscc-m:saxonblue.174','saxonblue','saxon blue',[0,73,88],'004958',18776],
['nbs-iscc-m:saxongreen.164','saxongreen','saxon green',[49,120,115],'317873',3242099],
['nbs-iscc-m:saxonyblue.169','saxonyblue','saxony blue',[0,119,145],'007791',30609],
['nbs-iscc-m:saxonyblue.174','saxonyblue','saxony blue',[0,73,88],'004958',18776],
['nbs-iscc-m:saxonyblue.179','saxonyblue','saxony blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:saxonyblue.182','saxonyblue','saxony blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:saxonygreen.139','saxonygreen','saxony green',[0,136,86],'008856',34902],
['nbs-iscc-m:sayalbrown.54','sayalbrown','sayal brown',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:sayalbrown.57','sayalbrown','sayal brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:sayalbrown.74','sayalbrown','sayal brown',[153,101,21],'996515',10052885],
['nbs-iscc-m:scabiosa.228','scabiosa','scabiosa ',[121,104,120],'796878',7956600],
['nbs-iscc-m:scarab.145','scarab','scarab',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:scarlet.11','scarlet','scarlet',[190,0,50],'be0032',12451890],
['nbs-iscc-m:scarletingrain.11','scarletingrain','scarlet in grain',[190,0,50],'be0032',12451890],
['nbs-iscc-m:scarletlake.12','scarletlake','scarlet lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:scarletlake.15','scarletlake','scarlet lake',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:scarletmadder.26','scarletmadder','scarlet madder',[248,131,121],'f88379',16286585],
['nbs-iscc-m:scarletochre.40','scarletochre','scarlet ochre',[136,45,23],'882d17',8924439],
['nbs-iscc-m:scarletred.11','scarletred','scarlet red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:scarletvermilion.34','scarletvermilion','scarlet vermilion',[226,88,34],'e25822',14833698],
['nbs-iscc-m:scheelesgreen.117','scheelesgreen','scheele\'s green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-m:scheelesgreen.131','scheelesgreen','scheele\'s green',[68,148,74],'44944a',4494410],
['nbs-iscc-m:schoenfeldspurple.256','schoenfeldspurple','schoenfeld\'s purple',[120,24,74],'78184a',7870538],
['nbs-iscc-m:schoenfeldspurple.258','schoenfeldspurple','schoenfeld\'s purple',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:schweinfurtgreen.140','schweinfurtgreen','schweinfurt green',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:scotchblue.201','scotchblue','scotch blue',[37,36,64],'252440',2434112],
['nbs-iscc-m:scotchgrey.122','scotchgrey','scotch grey ',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:scotchgrey.155','scotchgrey','scotch grey ',[125,137,132],'7d8984',8227204],
['nbs-iscc-m:seabird.266','seabird','seabird',[85,85,85],'555555',5592405],
['nbs-iscc-m:seablue.165','seablue','sea blue',[0,75,73],'004b49',19273],
['nbs-iscc-m:seacrest.135','seacrest','seacrest',[147,197,146],'93c592',9684370],
['nbs-iscc-m:seafoam.140','seafoam','seafoam',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:seafoam.144','seafoam','seafoam',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:seafoamgreen.121','seafoamgreen','seafoam green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:seafoamyellow.121','seafoamyellow','seafoam yellow',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:seagreen.120','seagreen','sea green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:seahawk.113','seahawk','sea hawk',[87,85,76],'57554c',5723468],
['nbs-iscc-m:sealbrown.81','sealbrown','seal [brown] ',[72,60,50],'483c32',4734002],
['nbs-iscc-m:seamanblue.204','seamanblue','seaman blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:seamist.233','seamist','sea mist',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:seamist.265','seamist','sea mist',[132,132,130],'848482',8684674],
['nbs-iscc-m:seamoss.106','seamoss','sea moss ',[134,126,54],'867e36',8814134],
['nbs-iscc-m:seamoss.151','seamoss','sea moss ',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:seapink.2','seapink','sea pink',[234,147,153],'ea9399',15373209],
['nbs-iscc-m:seapink.5','seapink','sea pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:seasannotseasand.33','seasannotseasand','seasan not seasand',[194,172,153],'c2ac99',12758169],
['nbs-iscc-m:seasannotseasand.73','seasannotseasand','seasan not seasand',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:seasannotseasand.76','seasannotseasand','seasan not seasand',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:seasannotseasand.79','seasannotseasand','seasan not seasand',[174,155,130],'ae9b82',11443074],
['nbs-iscc-m:seashell.29','seashell','sea shell',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:seashell.53','seashell','sea shell',[217,144,88],'d99058',14258264],
['nbs-iscc-m:seashell.57','seashell','sea shell',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:seashellpink.28','seashellpink','seashell pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-m:seashellpink.29','seashellpink','seashell pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:seaside.94','seaside','seaside',[150,113,23],'967117',9859351],
['nbs-iscc-m:seaside.45','seaside','seaside',[151,127,115],'977f73',9928563],
['nbs-iscc-m:seaside.60','seaside','seaside',[149,128,112],'958070',9797744],
['nbs-iscc-m:seaside.63','seaside','seaside',[142,130,121],'8e8279',9339513],
['nbs-iscc-m:seaspray.122','seaspray','seaspray',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:seaspray.136','seaspray','seaspray',[103,146,103],'679267',6787687],
['nbs-iscc-m:seaspray.149','seaspray','seaspray',[141,163,153],'8da399',9282457],
['nbs-iscc-m:seawatergreen.120','seawatergreen','sea-water green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:seaweedgreen.122','seaweedgreen','seaweed green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:sedge.61','sedge','sedge',[99,81,71],'635147',6508871],
['nbs-iscc-m:seedpearl.89','seedpearl','seed pearl',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:seedpearl.90','seedpearl','seed pearl',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:seeredgreen.120','seeredgreen','seered green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:seladongreen.122','seladongreen','seladon green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:seminole.58','seminole','seminole',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:sepia.96','sepia','sepia',[59,49,33],'3b3121',3879201],
['nbs-iscc-m:serapi.186','serapi','serapi',[83,104,120],'536878',5466232],
['nbs-iscc-m:serpent.149','serpent','serpent',[141,163,153],'8da399',9282457],
['nbs-iscc-m:serpentine.144','serpentine','serpentine',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:serpentinegreen.106','serpentinegreen','serpentine green',[134,126,54],'867e36',8814134],
['nbs-iscc-m:servicecorps.142','servicecorps','service corps',[0,84,61],'00543d',21565],
['nbs-iscc-m:seville.208','seville','seville',[50,23,77],'32174d',3282765],
['nbs-iscc-m:sevilleorange.53','sevilleorange','seville orange',[217,144,88],'d99058',14258264],
['nbs-iscc-m:sevresblue.178','sevresblue','sevres [blue] ',[0,103,165],'0067a5',26533],
['nbs-iscc-m:shadow.22','shadow','shadow',[143,129,127],'8f817f',9404799],
['nbs-iscc-m:shadowgreen.120','shadowgreen','shadow green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:shagbark.77','shagbark','shagbark',[130,102,68],'826644',8545860],
['nbs-iscc-m:shagbark.80','shagbark','shagbark',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:shalegreen.136','shalegreen','shale green',[103,146,103],'679267',6787687],
['nbs-iscc-m:shamrock.131','shamrock','shamrock',[68,148,74],'44944a',4494410],
['nbs-iscc-m:shamrockgreen.131','shamrockgreen','shamrock [green] ',[68,148,74],'44944a',4494410],
['nbs-iscc-m:shark.228','shark','shark',[121,104,120],'796878',7956600],
['nbs-iscc-m:shark.234','shark','shark',[93,85,91],'5d555b',6116699],
['nbs-iscc-m:sheepskin.90','sheepskin','sheepskin',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:sheik.15','sheik','sheik',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:sheik.38','sheik','sheik',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:shellgrey.93','shellgrey','shell grey ',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:shirvan.151','shirvan','shirvan',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:shrimppink.26','shrimppink','shrimp [pink] ',[248,131,121],'f88379',16286585],
['nbs-iscc-m:shrimpred.26','shrimpred','shrimp [red] ',[248,131,121],'f88379',16286585],
['nbs-iscc-m:siam.58','siam','siam',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:siam.61','siam','siam',[99,81,71],'635147',6508871],
['nbs-iscc-m:siam.77','siam','siam',[130,102,68],'826644',8545860],
['nbs-iscc-m:siam.80','siam','siam',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:siberianbrown.95','siberianbrown','siberian brown',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:siberien.151','siberien','siberien',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:sicilianumber.77','sicilianumber','sicilian umber',[130,102,68],'826644',8545860],
['nbs-iscc-m:sicilianumber.78','sicilianumber','sicilian umber',[75,54,33],'4b3621',4929057],
['nbs-iscc-m:siderinyellow.50','siderinyellow','siderin yellow',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:siderinyellow.53','siderinyellow','siderin yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-m:siena.40','siena','siena',[136,45,23],'882d17',8924439],
['nbs-iscc-m:siennabrown.58','siennabrown','sienna brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:siennayellow.73','siennayellow','sienna yellow',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:siennayellow.87','siennayellow','sienna yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:siennesedrab.45','siennesedrab','siennese drab',[151,127,115],'977f73',9928563],
['nbs-iscc-m:siennesedrab.60','siennesedrab','siennese drab',[149,128,112],'958070',9797744],
['nbs-iscc-m:sierra.40','sierra','sierra',[136,45,23],'882d17',8924439],
['nbs-iscc-m:signalred.11','signalred','signal red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:sil.71','sil','sil',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:siluriangray.121','siluriangray','silurian gray',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:silverfern.106','silverfern','silver fern',[134,126,54],'867e36',8814134],
['nbs-iscc-m:silverfern.109','silverfern','silver fern',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:silvergreen.122','silvergreen','silver green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:silverpine.150','silverpine','silverpine',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:silverwing.233','silverwing','silver-wing',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:sinbad.187','sinbad','sinbad',[54,69,79],'36454f',3556687],
['nbs-iscc-m:sinbad.193','sinbad','sinbad',[32,36,40],'202428',2106408],
['nbs-iscc-m:sinbad.229','sinbad','sinbad',[80,64,77],'50404d',5259341],
['nbs-iscc-m:sinbad.235','sinbad','sinbad',[36,33,36],'242124',2367780],
['nbs-iscc-m:sirocco.109','sirocco','sirocco',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:siskingreen.105','siskingreen','siskin green',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:sistine.172','sistine','sistine',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:sistine.173','sistine','sistine',[54,117,136],'367588',3569032],
['nbs-iscc-m:sistine.185','sistine','sistine',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:sistine.186','sistine','sistine',[83,104,120],'536878',5466232],
['nbs-iscc-m:skating.174','skating','skating',[0,73,88],'004958',18776],
['nbs-iscc-m:ski.137','ski','ski',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:ski.145','ski','ski',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:skimmedmilkwhite.121','skimmedmilkwhite','skimmed-milk white',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:skobeloffgreen.160','skobeloffgreen','skobeloff green',[0,122,116],'007a74',31348],
['nbs-iscc-m:sky.181','sky','sky',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:sky.199','sky','sky',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:skyblue.181','skyblue','sky blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:skyblue.199','skyblue','sky blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:skycolour.181','skycolour','sky colour',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:skycolour.199','skycolour','sky colour',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:skygreen.119','skygreen','sky green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-m:slag.266','slag','slag',[85,85,85],'555555',5592405],
['nbs-iscc-m:slate.192','slate','slate',[81,88,94],'51585e',5331038],
['nbs-iscc-m:slateblack.267','slateblack','slate-black',[34,34,34],'222222',2236962],
['nbs-iscc-m:slateblue.186','slateblue','slate blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:slateblue.186','slateblue','slate-blue ',[83,104,120],'536878',5466232],
['nbs-iscc-m:slatecolor.234','slatecolor','slate [color] ',[93,85,91],'5d555b',6116699],
['nbs-iscc-m:slategreen.150','slategreen','slate-green',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:slategreen.156','slategreen','slate-green',[78,87,85],'4e5755',5134165],
['nbs-iscc-m:slategrey.112','slategrey','slate-grey ',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:slategrey.265','slategrey','slate-grey ',[132,132,130],'848482',8684674],
['nbs-iscc-m:slateolive.156','slateolive','slate-olive',[78,87,85],'4e5755',5134165],
['nbs-iscc-m:slatepurple.229','slatepurple','slate-purple',[80,64,77],'50404d',5259341],
['nbs-iscc-m:slatepurple.242','slatepurple','slate-purple',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:slateviolet.19','slateviolet','slate-violet',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:smalt.179','smalt','smalt',[0,65,106],'00416a',16746],
['nbs-iscc-m:smalt.182','smalt','smalt',[67,107,149],'436b95',4418453],
['nbs-iscc-m:smaltgreen.136','smaltgreen','smalt green',[103,146,103],'679267',6787687],
['nbs-iscc-m:smaltino.179','smaltino','smaltino',[0,65,106],'00416a',16746],
['nbs-iscc-m:smaltino.182','smaltino','smaltino',[67,107,149],'436b95',4418453],
['nbs-iscc-m:smokeblue.192','smokeblue','smoke blue',[81,88,94],'51585e',5331038],
['nbs-iscc-m:smokebrown.266','smokebrown','smoke brown',[85,85,85],'555555',5592405],
['nbs-iscc-m:smokedpearl.266','smokedpearl','smoked pearl',[85,85,85],'555555',5592405],
['nbs-iscc-m:smokegrey.121','smokegrey','smoke grey ',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:smokeyellow.109','smokeyellow','smoke yellow',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:smokeyellow.112','smokeyellow','smoke yellow',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:snapdragon.86','snapdragon','snapdragon',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:snapdragon.87','snapdragon','snapdragon',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:snowshoe.68','snowshoe','snowshoe',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:snowshoe.71','snowshoe','snowshoe',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:soapstone.61','soapstone','soapstone',[99,81,71],'635147',6508871],
['nbs-iscc-m:solferino.258','solferino','solferino',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:solitaire.47','solitaire','solitaire',[67,48,46],'43302e',4403246],
['nbs-iscc-m:solitaire.62','solitaire','solitaire',[62,50,44],'3e322c',4076076],
['nbs-iscc-m:somalis.38','somalis','somalis',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:somalis.43','somalis','somalis',[121,68,59],'79443b',7947323],
['nbs-iscc-m:sombrero.90','sombrero','sombrero',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:sonora.54','sonora','sonora',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:sonora.57','sonora','sonora',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:sootbrown.61','sootbrown','soot brown',[99,81,71],'635147',6508871],
['nbs-iscc-m:sootbrown.80','sootbrown','soot brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:sootyblack.267','sootyblack','sooty black',[34,34,34],'222222',2236962],
['nbs-iscc-m:sorbier.262','sorbier','sorbier',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:sorghumbrown.45','sorghumbrown','sorghum brown',[151,127,115],'977f73',9928563],
['nbs-iscc-m:sorghumbrown.60','sorghumbrown','sorghum brown',[149,128,112],'958070',9797744],
['nbs-iscc-m:sorollabrown.55','sorollabrown','sorolla brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:sorrel.54','sorrel','sorrel',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:sorrel.57','sorrel','sorrel',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:sorrentogreen.164','sorrentogreen','sorrento green',[49,120,115],'317873',3242099],
['nbs-iscc-m:souci.50','souci','souci',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:source.136','source','source',[103,146,103],'679267',6787687],
['nbs-iscc-m:source.144','source','source',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:source.149','source','source',[141,163,153],'8da399',9282457],
['nbs-iscc-m:spalte.81','spalte','spalte',[72,60,50],'483c32',4734002],
['nbs-iscc-m:spaltum.81','spaltum','spaltum',[72,60,50],'483c32',4734002],
['nbs-iscc-m:spanishbrown.40','spanishbrown','spanish brown',[136,45,23],'882d17',8924439],
['nbs-iscc-m:spanishcedar.19','spanishcedar','spanish cedar',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:spanishcedar.43','spanishcedar','spanish cedar',[121,68,59],'79443b',7947323],
['nbs-iscc-m:spanishflesh.89','spanishflesh','spanish flesh',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:spanishflesh.90','spanishflesh','spanish flesh',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:spanishgreen.136','spanishgreen','spanish green',[103,146,103],'679267',6787687],
['nbs-iscc-m:spanishochre.50','spanishochre','spanish ochre',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:spanishraisin.47','spanishraisin','spanish raisin',[67,48,46],'43302e',4403246],
['nbs-iscc-m:spanishred.11','spanishred','spanish red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:spanishred.40','spanishred','spanish red',[136,45,23],'882d17',8924439],
['nbs-iscc-m:spanishred.55','spanishred','spanish red',[128,70,27],'80461b',8406555],
['nbs-iscc-m:spanishwine.16','spanishwine','spanish wine',[114,47,55],'722f37',7483191],
['nbs-iscc-m:spanishyellow.68','spanishyellow','spanish yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:spanishyellow.71','spanishyellow','spanish yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:spark.11','spark','spark',[190,0,50],'be0032',12451890],
['nbs-iscc-m:sparrow.64','sparrow','sparrow',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:spatan.54','spatan','spa-tan',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:spearmint.139','spearmint','spearmint',[0,136,86],'008856',34902],
['nbs-iscc-m:sphinx.80','sphinx','sphinx',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:spice.54','spice','spice',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:spinachgreen.120','spinachgreen','spinach green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:spinachgreen.125','spinachgreen','spinach green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:spinelpink.258','spinelpink','spinel pink',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:spinelred.12','spinelred','spinel red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:spinelred.255','spinelred','spinel red',[179,68,108],'b3446c',11748460],
['nbs-iscc-m:sponge.94','sponge','sponge',[150,113,23],'967117',9859351],
['nbs-iscc-m:spray.162','spray','spray',[150,222,209],'96ded1',9887441],
['nbs-iscc-m:springbeauty.254','springbeauty','spring beauty',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:springbeauty.255','springbeauty','spring beauty',[179,68,108],'b3446c',11748460],
['nbs-iscc-m:springgreen.120','springgreen','spring green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:springgreen.135','springgreen','spring green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:springtime.3','springtime','springtime',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:sprite.159','sprite','sprite',[0,166,147],'00a693',42643],
['nbs-iscc-m:sprite.163','sprite','sprite',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:spruce.151','spruce','spruce',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:spruce.157','spruce','spruce',[30,35,33],'1e2321',1975073],
['nbs-iscc-m:spruceochre.54','spruceochre','spruce ochre',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:spruceochre.74','spruceochre','spruce ochre',[153,101,21],'996515',10052885],
['nbs-iscc-m:spruceyellow.72','spruceyellow','spruce yellow',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:spruceyellow.76','spruceyellow','spruce yellow',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:squillblue.181','squillblue','squill blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:squirrel.234','squirrel','squirrel',[93,85,91],'5d555b',6116699],
['nbs-iscc-m:squirrel.266','squirrel','squirrel',[85,85,85],'555555',5592405],
['nbs-iscc-m:stag.77','stag','stag',[130,102,68],'826644',8545860],
['nbs-iscc-m:starchblue.179','starchblue','starch blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:starchblue.182','starchblue','starch blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:stardew.186','stardew','stardew',[83,104,120],'536878',5466232],
['nbs-iscc-m:starflower.251','starflower','starflower',[193,126,145],'c17e91',12680849],
['nbs-iscc-m:starlight.185','starlight','starlight',[145,163,176],'91a3b0',9544624],
['nbs-iscc-m:starling.64','starling','starling',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:stbenoitorsaintbenoit.81','stbenoitorsaintbenoit','st. benoit or saint benoit',[72,60,50],'483c32',4734002],
['nbs-iscc-m:stbenoitorsaintbenoit.267','stbenoitorsaintbenoit','st. benoit or saint benoit',[34,34,34],'222222',2236962],
['nbs-iscc-m:steel.266','steel','steel',[85,85,85],'555555',5592405],
['nbs-iscc-m:steelgrey.266','steelgrey','steel grey ',[85,85,85],'555555',5592405],
['nbs-iscc-m:steeplechase.196','steeplechase','steeplechase',[84,90,167],'545aa7',5528231],
['nbs-iscc-m:stildegrainbrown.72','stildegrainbrown','stil de grain brown',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:stildegrainbrown.74','stildegrainbrown','stil de grain brown',[153,101,21],'996515',10052885],
['nbs-iscc-m:stildegrainyellow.83','stildegrainyellow','stil de grain yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:stildegrainyellow.84','stildegrainyellow','stil de grain yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:stildegrainyellow.86','stildegrainyellow','stil de grain yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:stildegrainyellow.87','stildegrainyellow','stil de grain yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:stoneblue.182','stoneblue','stone blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:stonegrey.113','stonegrey','stone grey ',[87,85,76],'57554c',5723468],
['nbs-iscc-m:stormgrey.155','stormgrey','storm grey ',[125,137,132],'7d8984',8227204],
['nbs-iscc-m:strawberry.19','strawberry','strawberry',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:strawberrypink.3','strawberrypink','strawberry pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:strawberrypink.26','strawberrypink','strawberry pink',[248,131,121],'f88379',16286585],
['nbs-iscc-m:strawberrypink.27','strawberrypink','strawberry pink',[230,103,97],'e66761',15099745],
['nbs-iscc-m:strawyellow.89','strawyellow','straw [yellow] ',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:striegauearth.43','striegauearth','striegau earth',[121,68,59],'79443b',7947323],
['nbs-iscc-m:strollertan.58','strollertan','stroller tan',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:strontianyellow.98','strontianyellow','strontian yellow ',[233,228,80],'e9e450',15328336],
['nbs-iscc-m:stucco.45','stucco','stucco',[151,127,115],'977f73',9928563],
['nbs-iscc-m:stucco.60','stucco','stucco',[149,128,112],'958070',9797744],
['nbs-iscc-m:succoryblue.121','succoryblue','succory blue',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:sudan.91','sudan','sudan',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:sudan.94','sudan','sudan',[150,113,23],'967117',9859351],
['nbs-iscc-m:sudan.109','sudan','sudan',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:sudanbrown.55','sudanbrown','sudan brown',[128,70,27],'80461b',8406555],
['nbs-iscc-m:sudanbrown.58','sudanbrown','sudan brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:suede.57','suede','suede',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:suede.58','suede','suede',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:suede.77','suede','suede',[130,102,68],'826644',8545860],
['nbs-iscc-m:sugarcane.73','sugarcane','sugar cane',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:sugarcane.76','sugarcane','sugar cane',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:sulphategreen.163','sulphategreen','sulphate green',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:sulphategreen.164','sulphategreen','sulphate green',[49,120,115],'317873',3242099],
['nbs-iscc-m:sulphineyellow.88','sulphineyellow','sulphine yellow',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:sulphur.101','sulphur','sulphur',[234,230,121],'eae679',15394425],
['nbs-iscc-m:sulphuryellow.101','sulphuryellow','sulphur [yellow] ',[234,230,121],'eae679',15394425],
['nbs-iscc-m:sultan.15','sultan','sultan',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:sultana.16','sultana','sultana',[114,47,55],'722f37',7483191],
['nbs-iscc-m:sultana.259','sultana','sultana',[103,49,71],'673147',6762823],
['nbs-iscc-m:sumac.72','sumac','sumac',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:sumac.74','sumac','sumac',[153,101,21],'996515',10052885],
['nbs-iscc-m:sumac.76','sumac','sumac',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:sumac.77','sumac','sumac',[130,102,68],'826644',8545860],
['nbs-iscc-m:sumac.88','sumac','sumac',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:sumac.91','sumac','sumac',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:sumac.94','sumac','sumac',[150,113,23],'967117',9859351],
['nbs-iscc-m:sunbeam.90','sunbeam','sunbeam',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:sunburn.57','sunburn','sunburn',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:sunburst.50','sunburst','sunburst',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:sunburst.53','sunburst','sunburst',[217,144,88],'d99058',14258264],
['nbs-iscc-m:sundown.76','sundown','sundown',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:sunfloweryellow.83','sunfloweryellow','sunflower [yellow] ',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:sunglow.3','sunglow','sunglow',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:sunglow.15','sunglow','sunglow',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:sungod.34','sungod','sungod',[226,88,34],'e25822',14833698],
['nbs-iscc-m:sunkiss.53','sunkiss','sunkiss ',[217,144,88],'d99058',14258264],
['nbs-iscc-m:sunlight.87','sunlight','sunlight',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:sunni.86','sunni','sunni',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:sunni.87','sunni','sunni',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:sunray.71','sunray','sunray',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:sunriseyellow.29','sunriseyellow','sunrise yellow',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:sunriseyellow.53','sunriseyellow','sunrise yellow',[217,144,88],'d99058',14258264],
['nbs-iscc-m:sunset.73','sunset','sunset',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:sunstone.54','sunstone','sunstone',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:suntan.57','suntan','suntan',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:suntan.76','suntan','suntan',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:suntan.77','suntan','suntan',[130,102,68],'826644',8545860],
['nbs-iscc-m:superior.165','superior','superior',[0,75,73],'004b49',19273],
['nbs-iscc-m:surfgreen.136','surfgreen','surf green',[103,146,103],'679267',6787687],
['nbs-iscc-m:surreygreen.122','surreygreen','surrey green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:surreygreen.149','surreygreen','surrey green',[141,163,153],'8da399',9282457],
['nbs-iscc-m:swamp.151','swamp','swamp',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:swedishgreen.117','swedishgreen','swedish green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-m:swedishgreen.131','swedishgreen','swedish green',[68,148,74],'44944a',4494410],
['nbs-iscc-m:sweetbriar.248','sweetbriar','sweet briar',[222,111,161],'de6fa1',14577569],
['nbs-iscc-m:sweetmeat.76','sweetmeat','sweetmeat',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:sweetpea.32','sweetpea','sweet pea',[199,173,163],'c7ada3',13086115],
['nbs-iscc-m:sweetwilliam.3','sweetwilliam','sweet william',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:swissblue.186','swissblue','swiss blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:swissrose.15','swissrose','swiss rose',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:syrup.94','syrup','syrup',[150,113,23],'967117',9859351],
['nbs-iscc-m:syrup.95','syrup','syrup',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:taffy.76','taffy','taffy',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:talavera.54','talavera','talavera',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:tamarach.61','tamarach','tamarach',[99,81,71],'635147',6508871],
['nbs-iscc-m:tamarach.80','tamarach','tamarach',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:taming.68','taming','ta-ming',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:taming.71','taming','ta-ming',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:tan.54','tan','tan',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:tanagra.43','tanagra','tanagra',[121,68,59],'79443b',7947323],
['nbs-iscc-m:tanaura.90','tanaura','tanaura ',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:tanbark.58','tanbark','tanbark',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:tangerine.37','tangerine','tangerine',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:tangier.50','tangier','tangier',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:tangopink.19','tangopink','tango pink',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:tansan.57','tansan','tansan',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:tansan.76','tansan','tansan',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:tapestry.183','tapestry','tapestry',[0,48,78],'00304e',12366],
['nbs-iscc-m:tapestry.201','tapestry','tapestry',[37,36,64],'252440',2434112],
['nbs-iscc-m:tapestry.204','tapestry','tapestry',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:tapestry.233','tapestry','tapestry',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:tapestryred.16','tapestryred','tapestry red',[114,47,55],'722f37',7483191],
['nbs-iscc-m:tapisvert.120','tapisvert','tapis vert',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:tarragon.120','tarragon','tarragon',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:tarragon.122','tarragon','tarragon',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:tarragon.136','tarragon','tarragon',[103,146,103],'679267',6787687],
['nbs-iscc-m:tarragona.40','tarragona','tarragona',[136,45,23],'882d17',8924439],
['nbs-iscc-m:tarragona.55','tarragona','tarragona',[128,70,27],'80461b',8406555],
['nbs-iscc-m:tartangreen.151','tartangreen','tartan green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:taupe.81','taupe','taupe',[72,60,50],'483c32',4734002],
['nbs-iscc-m:tawny.54','tawny','tawny',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:tawny.57','tawny','tawny',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:tawnybirch.42','tawnybirch','tawny birch',[168,124,109],'a87c6d',11041901],
['nbs-iscc-m:tawnybirch.57','tawnybirch','tawny birch',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:tawnybirch.60','tawnybirch','tawny birch',[149,128,112],'958070',9797744],
['nbs-iscc-m:tea.122','tea','tea',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:tea.149','tea','tea',[141,163,153],'8da399',9282457],
['nbs-iscc-m:teagreen.122','teagreen','tea green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:teakbrown.58','teakbrown','teak [brown] ',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:teakwood.61','teakwood','teakwood',[99,81,71],'635147',6508871],
['nbs-iscc-m:teakwood.80','teakwood','teakwood',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:tealblue.174','tealblue','teal blue',[0,73,88],'004958',18776],
['nbs-iscc-m:tealblue.187','tealblue','teal blue',[54,69,79],'36454f',3556687],
['nbs-iscc-m:tealduck.174','tealduck','teal duck ',[0,73,88],'004958',18776],
['nbs-iscc-m:teatime.94','teatime','tea time',[150,113,23],'967117',9859351],
['nbs-iscc-m:telegraphblue.228','telegraphblue','telegraph blue',[121,104,120],'796878',7956600],
['nbs-iscc-m:tennis.88','tennis','tennis',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:tennis.91','tennis','tennis',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:terracotta.51','terracotta','terra cotta',[190,101,22],'be6516',12477718],
['nbs-iscc-m:terracotta.54','terracotta','terra cotta',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:terrajaponica.58','terrajaponica','terra japonica',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:terralemnia.43','terralemnia','terra lemnia',[121,68,59],'79443b',7947323],
['nbs-iscc-m:terramerita.86','terramerita','terra merita',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:terraorellana.29','terraorellana','terra orellana',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:terraorellano.29','terraorellano','terra orellano ',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:terrapin.58','terrapin','terrapin',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:terrapozzuoli.43','terrapozzuoli','terra pozzuoli',[121,68,59],'79443b',7947323],
['nbs-iscc-m:terrarosa.43','terrarosa','terra rosa',[121,68,59],'79443b',7947323],
['nbs-iscc-m:terrasiena.54','terrasiena','terra siena ',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:terrasiena.57','terrasiena','terra siena ',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:terrasigillata.43','terrasigillata','terra sigillata',[121,68,59],'79443b',7947323],
['nbs-iscc-m:terrasse.120','terrasse','terrasse',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:terraumber.77','terraumber','terra umber',[130,102,68],'826644',8545860],
['nbs-iscc-m:terraumber.78','terraumber','terra umber',[75,54,33],'4b3621',4929057],
['nbs-iscc-m:testaceous.53','testaceous','testaceous',[217,144,88],'d99058',14258264],
['nbs-iscc-m:testaceous.57','testaceous','testaceous',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:thenardsblue.169','thenardsblue','thenard\'s blue',[0,119,145],'007791',30609],
['nbs-iscc-m:thistle.223','thistle','thistle',[134,96,142],'86608e',8806542],
['nbs-iscc-m:thistlebloom.237','thistlebloom','thistle bloom',[158,79,136],'9e4f88',10375048],
['nbs-iscc-m:thistletuft.238','thistletuft','thistletuft',[112,41,99],'702963',7350627],
['nbs-iscc-m:thrush.77','thrush','thrush',[130,102,68],'826644',8545860],
['nbs-iscc-m:thrush.80','thrush','thrush',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:thulitepink.248','thulitepink','thulite pink',[222,111,161],'de6fa1',14577569],
['nbs-iscc-m:thyme.151','thyme','thyme',[58,75,71],'3a4b47',3820359],
['nbs-iscc-m:tiber.146','tiber','tiber',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:tibergreen.135','tibergreen','tiber green',[147,197,146],'93c592',9684370],
['nbs-iscc-m:tiffin.77','tiffin','tiffin',[130,102,68],'826644',8545860],
['nbs-iscc-m:tigerlily.35','tigerlily','tigerlily',[217,96,59],'d9603b',14245947],
['nbs-iscc-m:tigerlily.37','tigerlily','tigerlily',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:tileblue.173','tileblue','tile blue',[54,117,136],'367588',3569032],
['nbs-iscc-m:tilered.37','tilered','tile red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-m:tilleulbuff.31','tilleulbuff','tilleul buff',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-m:tilleulbuff.93','tilleulbuff','tilleul buff',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:tilleulgreen.104','tilleulgreen','tilleul [green] ',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-m:tinsel.88','tinsel','tinsel',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:tinsel.91','tinsel','tinsel',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:tinsel.94','tinsel','tinsel',[150,113,23],'967117',9859351],
['nbs-iscc-m:titania.242','titania','titania',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:titania.259','titania','titania',[103,49,71],'673147',6762823],
['nbs-iscc-m:titian.54','titian','titian',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:titiangold.54','titiangold','titian gold',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:titmouseblue.149','titmouseblue','titmouse blue ',[141,163,153],'8da399',9282457],
['nbs-iscc-m:toast.57','toast','toast',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:toastedalmond.57','toastedalmond','toasted almond',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:toboggan.34','toboggan','toboggan',[226,88,34],'e25822',14833698],
['nbs-iscc-m:toboggan.48','toboggan','toboggan',[243,132,0],'f38400',15959040],
['nbs-iscc-m:toga.212','toga','toga',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:tokay.43','tokay','tokay',[121,68,59],'79443b',7947323],
['nbs-iscc-m:tokyo.25','tokyo','tokyo',[255,183,165],'ffb7a5',16758693],
['nbs-iscc-m:tokyo.26','tokyo','tokyo',[248,131,121],'f88379',16286585],
['nbs-iscc-m:tokyo.50','tokyo','tokyo',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:toltec.71','toltec','toltec',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:tomatored.38','tomatored','tomato [red] ',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:tommyred.12','tommyred','tommy red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:topaz.57','topaz','topaz',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:topaz.72','topaz','topaz',[190,138,61],'be8a3d',12487229],
['nbs-iscc-m:topaz.76','topaz','topaz',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:toquet.32','toquet','toquet',[199,173,163],'c7ada3',13086115],
['nbs-iscc-m:toreador.34','toreador','toreador',[226,88,34],'e25822',14833698],
['nbs-iscc-m:torinoblue.186','torinoblue','torino blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:tortoise.55','tortoise','tortoise',[128,70,27],'80461b',8406555],
['nbs-iscc-m:tortoiseshell.58','tortoiseshell','tortoise shell',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:totem.38','totem','totem',[158,71,50],'9e4732',10372914],
['nbs-iscc-m:tourmaline.163','tourmaline','tourmaline',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:tourmalinepink.262','tourmalinepink','tourmaline pink',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:tourterelle.94','tourterelle','tourterelle',[150,113,23],'967117',9859351],
['nbs-iscc-m:transparentchromiumoxide.141','transparentchromiumoxide','transparent chromium oxide',[0,121,89],'007959',31065],
['nbs-iscc-m:transparentchromiumoxide.145','transparentchromiumoxide','transparent chromium oxide',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:transparentgoldochre.54','transparentgoldochre','transparent gold ochre',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:transparentgoldochre.74','transparentgoldochre','transparent gold ochre',[153,101,21],'996515',10052885],
['nbs-iscc-m:traprock.266','traprock','traprock',[85,85,85],'555555',5592405],
['nbs-iscc-m:travertine.94','travertine','travertine ',[150,113,23],'967117',9859351],
['nbs-iscc-m:trentanel.101','trentanel','trentanel',[234,230,121],'eae679',15394425],
['nbs-iscc-m:trentanel.102','trentanel','trentanel',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:trianon.29','trianon','trianon',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:trianon.53','trianon','trianon',[217,144,88],'d99058',14258264],
['nbs-iscc-m:triton.146','triton','triton',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:triumphblue.182','triumphblue','triumph blue',[67,107,149],'436b95',4418453],
['nbs-iscc-m:triumphblue.183','triumphblue','triumph blue',[0,48,78],'00304e',12366],
['nbs-iscc-m:trooper.183','trooper','trooper',[0,48,78],'00304e',12366],
['nbs-iscc-m:trooper.187','trooper','trooper',[54,69,79],'36454f',3556687],
['nbs-iscc-m:trooper.188','trooper','trooper',[32,40,48],'202830',2107440],
['nbs-iscc-m:trotteurtan.58','trotteurtan','trotteur tan',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:troubadorred.12','troubadorred','troubador red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:trublu.186','trublu','trublu',[83,104,120],'536878',5466232],
['nbs-iscc-m:tuileries.182','tuileries','tuileries',[67,107,149],'436b95',4418453],
['nbs-iscc-m:tulipwood.58','tulipwood','tulipwood',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:tunis.187','tunis','tunis',[54,69,79],'36454f',3556687],
['nbs-iscc-m:tunis.188','tunis','tunis',[32,40,48],'202830',2107440],
['nbs-iscc-m:turbithmineral.86','turbithmineral','turbith mineral',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:turkeyblue.204','turkeyblue','turkey blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:turkeyred.15','turkeyred','turkey red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:turkeyumber.77','turkeyumber','turkey umber',[130,102,68],'826644',8545860],
['nbs-iscc-m:turkeyumber.78','turkeyumber','turkey umber',[75,54,33],'4b3621',4929057],
['nbs-iscc-m:turkishblue.204','turkishblue','turkish blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:turkishcrescentred.13','turkishcrescentred','turkish crescent red ',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:turkishcrescentred.16','turkishcrescentred','turkish crescent red ',[114,47,55],'722f37',7483191],
['nbs-iscc-m:turkishred.15','turkishred','turkish red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:turmeric.86','turmeric','turmeric',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:turnersyellow.83','turnersyellow','turner\'s yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:turnersyellow.84','turnersyellow','turner\'s yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:turnersyellow.86','turnersyellow','turner\'s yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:turnersyellow.87','turnersyellow','turner\'s yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:turquoiseblue.163','turquoiseblue','turquoise [blue] ',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:turquoisegreen.163','turquoisegreen','turquoise green',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:turtle.58','turtle','turtle',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:turtledove.23','turtledove','turtledove',[92,80,79],'5c504f',6049871],
['nbs-iscc-m:turtledove.266','turtledove','turtledove',[85,85,85],'555555',5592405],
['nbs-iscc-m:turtlegreen.120','turtlegreen','turtle green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:tuscan.73','tuscan','tuscan',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:tuscanbrown.43','tuscanbrown','tuscan brown',[121,68,59],'79443b',7947323],
['nbs-iscc-m:tuscanbrown.58','tuscanbrown','tuscan brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:tuscanred.43','tuscanred','tuscan red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:tuscantan.57','tuscantan','tuscan tan',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:tuscany.43','tuscany','tuscany',[121,68,59],'79443b',7947323],
['nbs-iscc-m:tussore.28','tussore','tussore',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-m:tussore.29','tussore','tussore',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:twilightblue.186','twilightblue','twilight [blue] ',[83,104,120],'536878',5466232],
['nbs-iscc-m:twine.109','twine','twine',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:twinkleblue.172','twinkleblue','twinkle blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:tyrianblue.186','tyrianblue','tyrian blue',[83,104,120],'536878',5466232],
['nbs-iscc-m:tyrianblue.204','tyrianblue','tyrian blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:tyrianpink.254','tyrianpink','tyrian pink',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:tyrianviolet.258','tyrianviolet','tyrian violet',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:tyrol.77','tyrol','tyrol',[130,102,68],'826644',8545860],
['nbs-iscc-m:tyrolesegreen.136','tyrolesegreen','tyrolese green',[103,146,103],'679267',6787687],
['nbs-iscc-m:tyrolian.152','tyrolian','tyrolian',[26,36,33],'1a2421',1713185],
['nbs-iscc-m:tyrolitegreen.163','tyrolitegreen','tyrolite green',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:tzarine.157','tzarine','tzarine',[30,35,33],'1e2321',1975073],
['nbs-iscc-m:ultramarinegreen.157','ultramarinegreen','ultramarine green',[30,35,33],'1e2321',1975073],
['nbs-iscc-m:ultramarineyellow.84','ultramarineyellow','ultramarine yellow ',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:uraniablue.212','uraniablue','urania blue',[47,33,64],'2f2140',3088704],
['nbs-iscc-m:vagabondgreen.165','vagabondgreen','vagabond green',[0,75,73],'004b49',19273],
['nbs-iscc-m:valencia.248','valencia','valencia',[222,111,161],'de6fa1',14577569],
['nbs-iscc-m:vanda.228','vanda','vanda',[121,104,120],'796878',7956600],
['nbs-iscc-m:vanda.245','vanda','vanda',[131,100,121],'836479',8610937],
['nbs-iscc-m:vandykebrown.58','vandykebrown','vandyke brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:vandykemadder.258','vandykemadder','vandyke madder',[168,81,110],'a8516e',11030894],
['nbs-iscc-m:vandykered.19','vandykered','vandyke red',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:vanilla.89','vanilla','vanilla',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:vanilla.90','vanilla','vanilla',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:vanity.29','vanity','vanity',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-m:variscitegreen.144','variscitegreen','variscite green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:varleysgrey.228','varleysgrey','varley\'s grey ',[121,104,120],'796878',7956600],
['nbs-iscc-m:vassarrose.6','vassarrose','vassar rose',[192,128,129],'c08081',12615809],
['nbs-iscc-m:vassarrose.19','vassarrose','vassar rose',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:vassartan.55','vassartan','vassar tan',[128,70,27],'80461b',8406555],
['nbs-iscc-m:vassartan.58','vassartan','vassar tan',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:vatican.224','vatican','vatican',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:vatican.225','vatican','vatican',[48,25,52],'301934',3152180],
['nbs-iscc-m:veaudor.73','veaudor','veau d\'or',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:veaudor.76','veaudor','veau d\'or',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:vegetablered.11','vegetablered','vegetable red',[190,0,50],'be0032',12451890],
['nbs-iscc-m:vegetablerouge.11','vegetablerouge','vegetable rouge',[190,0,50],'be0032',12451890],
['nbs-iscc-m:velasquez.259','velasquez','velasquez ',[103,49,71],'673147',6762823],
['nbs-iscc-m:velvetbrown.77','velvetbrown','velvet brown',[130,102,68],'826644',8545860],
['nbs-iscc-m:velvetbrown.78','velvetbrown','velvet brown',[75,54,33],'4b3621',4929057],
['nbs-iscc-m:velvetgreen.125','velvetgreen','velvet green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:venet.163','venet','venet',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:venetianblue.169','venetianblue','venetian blue',[0,119,145],'007791',30609],
['nbs-iscc-m:venetianfuchsia.254','venetianfuchsia','venetian fuchsia',[206,70,118],'ce4676',13518454],
['nbs-iscc-m:venetianlake.11','venetianlake','venetian lake',[190,0,50],'be0032',12451890],
['nbs-iscc-m:venetianlake.12','venetianlake','venetian lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:venetianpink.5','venetianpink','venetian pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-m:venetianred.40','venetianred','venetian red',[136,45,23],'882d17',8924439],
['nbs-iscc-m:venetianrose.13','venetianrose','venetian rose',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:venetianrose.15','venetianrose','venetian rose',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:venetianscarlet.11','venetianscarlet','venetian scarlet',[190,0,50],'be0032',12451890],
['nbs-iscc-m:venetianyellow.86','venetianyellow','venetian yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:venetianyellow.87','venetianyellow','venetian yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:venezia.164','venezia','venezia',[49,120,115],'317873',3242099],
['nbs-iscc-m:veniceberries.68','veniceberries','venice berries',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:veniceberries.71','veniceberries','venice berries',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:veniceblue.163','veniceblue','venice [blue] ',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:venicegreen.163','venicegreen','venice green',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:venicered.43','venicered','venice red',[121,68,59],'79443b',7947323],
['nbs-iscc-m:venus.88','venus','venus',[171,145,68],'ab9144',11243844],
['nbs-iscc-m:venus.91','venus','venus',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:verbena.241','verbena','verbena',[145,92,131],'915c83',9526403],
['nbs-iscc-m:verbenaviolet.214','verbenaviolet','verbena [violet] ',[150,144,171],'9690ab',9867435],
['nbs-iscc-m:verbenaviolet.227','verbenaviolet','verbena [violet] ',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:verdantgreen.120','verdantgreen','verdant green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:verdet.136','verdet','verdet',[103,146,103],'679267',6787687],
['nbs-iscc-m:verdevessie.106','verdevessie','verde vessie',[134,126,54],'867e36',8814134],
['nbs-iscc-m:verdgay.120','verdgay','verd gay',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:verdigrisgreen.136','verdigrisgreen','verdigris [green] ',[103,146,103],'679267',6787687],
['nbs-iscc-m:verditergreen.136','verditergreen','verditer green',[103,146,103],'679267',6787687],
['nbs-iscc-m:verdure.137','verdure','verdure',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:veridinegreen.144','veridinegreen','veridine green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:vernoniapurple.262','vernoniapurple','vernonia purple',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:veronabrown.58','veronabrown','verona brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:veronayellow.83','veronayellow','verona yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:veronayellow.84','veronayellow','verona yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:veronayellow.86','veronayellow','verona yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:veronayellow.87','veronayellow','verona yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:veroneseyellow.83','veroneseyellow','veronese yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:veroneseyellow.84','veroneseyellow','veronese yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:veroneseyellow.86','veroneseyellow','veronese yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:veroneseyellow.87','veroneseyellow','veronese yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:veronica.211','veronica','veronica',[96,78,129],'604e81',6311553],
['nbs-iscc-m:versailles.172','versailles','versailles',[102,170,188],'66aabc',6728380],
['nbs-iscc-m:vertrusse.136','vertrusse','vert russe',[103,146,103],'679267',6787687],
['nbs-iscc-m:vertrusse.137','vertrusse','vert russe',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:vertrusse.150','vertrusse','vert russe',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:vervain.214','vervain','vervain',[150,144,171],'9690ab',9867435],
['nbs-iscc-m:vervain.227','vervain','vervain',[170,152,169],'aa98a9',11180201],
['nbs-iscc-m:vestal.228','vestal','vestal',[121,104,120],'796878',7956600],
['nbs-iscc-m:vestal.245','vestal','vestal',[131,100,121],'836479',8610937],
['nbs-iscc-m:veteran.187','veteran','veteran',[54,69,79],'36454f',3556687],
['nbs-iscc-m:vetivergreen.109','vetivergreen','vetiver green',[140,135,103],'8c8767',9209703],
['nbs-iscc-m:victoria.224','victoria','victoria',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:victoriablue.178','victoriablue','victoria blue',[0,103,165],'0067a5',26533],
['nbs-iscc-m:victoriagreen.144','victoriagreen','victoria green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-m:victoriagreen.149','victoriagreen','victoria green',[141,163,153],'8da399',9282457],
['nbs-iscc-m:victoriagreen.163','victoriagreen','victoria green',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:victorialake.16','victorialake','victoria lake',[114,47,55],'722f37',7483191],
['nbs-iscc-m:viennablue.169','viennablue','vienna blue',[0,119,145],'007791',30609],
['nbs-iscc-m:viennabrown.61','viennabrown','vienna brown',[99,81,71],'635147',6508871],
['nbs-iscc-m:viennabrown.80','viennabrown','vienna brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:viennagreen.140','viennagreen','vienna green',[62,180,137],'3eb489',4109449],
['nbs-iscc-m:viennalake.11','viennalake','vienna lake',[190,0,50],'be0032',12451890],
['nbs-iscc-m:viennalake.12','viennalake','vienna lake',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-m:viennasmoke.266','viennasmoke','vienna smoke',[85,85,85],'555555',5592405],
['nbs-iscc-m:vineyard.16','vineyard','vineyard',[114,47,55],'722f37',7483191],
['nbs-iscc-m:vineyard.17','vineyard','vineyard',[63,23,40],'3f1728',4134696],
['nbs-iscc-m:viola.223','viola','viola',[134,96,142],'86608e',8806542],
['nbs-iscc-m:violetcarmine.242','violetcarmine','violet-carmine',[93,57,84],'5d3954',6109524],
['nbs-iscc-m:violine.207','violine','violine',[96,78,151],'604e97',6311575],
['nbs-iscc-m:violine.211','violine','violine',[96,78,129],'604e81',6311553],
['nbs-iscc-m:virgin.182','virgin','virgin',[67,107,149],'436b95',4418453],
['nbs-iscc-m:viridian.141','viridian','viridian',[0,121,89],'007959',31065],
['nbs-iscc-m:viridian.145','viridian','viridian',[59,120,97],'3b7861',3897441],
['nbs-iscc-m:viridinegreen.119','viridinegreen','viridine green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-m:viridineyellow.117','viridineyellow','viridine yellow',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-m:vitellineyellow.84','vitellineyellow','vitelline yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:vitellinous.84','vitellinous','vitellinous',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:vitreous.119','vitreous','vitreous',[201,220,137],'c9dc89',13229193],
['nbs-iscc-m:wad.186','wad','wad',[83,104,120],'536878',5466232],
['nbs-iscc-m:wald.101','wald','wald',[234,230,121],'eae679',15394425],
['nbs-iscc-m:wald.102','wald','wald',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:wallflowerbrown.16','wallflowerbrown','wallflower [brown] ',[114,47,55],'722f37',7483191],
['nbs-iscc-m:wallflowerbrown.43','wallflowerbrown','wallflower [brown] ',[121,68,59],'79443b',7947323],
['nbs-iscc-m:wallgreen.160','wallgreen','wall green',[0,122,116],'007a74',31348],
['nbs-iscc-m:walnutbrown.76','walnutbrown','walnut [brown] ',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:warblergreen.106','warblergreen','warbler green',[134,126,54],'867e36',8814134],
['nbs-iscc-m:warmsepia.61','warmsepia','warm sepia ',[99,81,71],'635147',6508871],
['nbs-iscc-m:waterblue.164','waterblue','water blue',[49,120,115],'317873',3242099],
['nbs-iscc-m:watercolor.163','watercolor','water-color',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:watercress.120','watercress','water cress',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:waterfall.164','waterfall','waterfall',[49,120,115],'317873',3242099],
['nbs-iscc-m:watergreen.105','watergreen','water green',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:watergreen.119','watergreen','water green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-m:watergreen.121','watergreen','water green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-m:watergrey.112','watergrey','water grey ',[138,135,118],'8a8776',9078646],
['nbs-iscc-m:waterloo.204','waterloo','waterloo',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:watermelon.3','watermelon','watermelon',[228,113,122],'e4717a',14971258],
['nbs-iscc-m:watermelon.15','watermelon','watermelon',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:watersprite.135','watersprite','water sprite',[147,197,146],'93c592',9684370],
['nbs-iscc-m:watteau.181','watteau','watteau',[112,163,204],'70a3cc',7381964],
['nbs-iscc-m:wau.101','wau','wau',[234,230,121],'eae679',15394425],
['nbs-iscc-m:wau.102','wau','wau',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:waxbrown.77','waxbrown','wax brown',[130,102,68],'826644',8545860],
['nbs-iscc-m:waxcolor.104','waxcolor','wax color',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-m:waxcolor.105','waxcolor','wax color',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:waxen.104','waxen','waxen',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-m:waxen.105','waxen','waxen',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:waxred.39','waxred','wax red',[180,116,94],'b4745e',11826270],
['nbs-iscc-m:waxwhite.104','waxwhite','wax white',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-m:waxwhite.105','waxwhite','wax white',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:waxyellow.87','waxyellow','wax yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:weatheredoak.59','weatheredoak','weathered oak',[66,37,24],'422518',4334872],
['nbs-iscc-m:weigelia.15','weigelia','weigelia',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:weld.101','weld','weld',[234,230,121],'eae679',15394425],
['nbs-iscc-m:weld.102','weld','weld',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:westminster.194','westminster','westminster',[48,38,122],'30267a',3155578],
['nbs-iscc-m:westminster.205','westminster','westminster',[144,101,202],'9065ca',9463242],
['nbs-iscc-m:westpoint.192','westpoint','west point',[81,88,94],'51585e',5331038],
['nbs-iscc-m:wheat.71','wheat','wheat',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:wheat.73','wheat','wheat',[250,214,165],'fad6a5',16438949],
['nbs-iscc-m:whippet.95','whippet','whippet',[108,84,30],'6c541e',7099422],
['nbs-iscc-m:whirlpool.174','whirlpool','whirlpool',[0,73,88],'004958',18776],
['nbs-iscc-m:whiteblue.190','whiteblue','white blue',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-m:whitejade.31','whitejade','white jade',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-m:whitejade.93','whitejade','white jade',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:wield.101','wield','wield',[234,230,121],'eae679',15394425],
['nbs-iscc-m:wield.102','wield','wield',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:wigwam.58','wigwam','wigwam',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:wildaster.219','wildaster','wild aster',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:wildaster.238','wildaster','wild aster',[112,41,99],'702963',7350627],
['nbs-iscc-m:wildcherry.16','wildcherry','wild cherry',[114,47,55],'722f37',7483191],
['nbs-iscc-m:wildcherry.259','wildcherry','wild cherry',[103,49,71],'673147',6762823],
['nbs-iscc-m:wilddovegrey.64','wilddovegrey','wild dove grey ',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:wildhoney.54','wildhoney','wild honey',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:wildhoney.57','wildhoney','wild honey',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:wildiris.233','wildiris','wild iris',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:wildorchid.245','wildorchid','wild orchid',[131,100,121],'836479',8610937],
['nbs-iscc-m:wildpigeon.64','wildpigeon','wild pigeon',[91,80,79],'5b504f',5984335],
['nbs-iscc-m:wildraspberry.16','wildraspberry','wild raspberry',[114,47,55],'722f37',7483191],
['nbs-iscc-m:wildraspberry.259','wildraspberry','wild raspberry',[103,49,71],'673147',6762823],
['nbs-iscc-m:wildrose.6','wildrose','wild rose',[192,128,129],'c08081',12615809],
['nbs-iscc-m:wildstrawberry.13','wildstrawberry','wild strawberry',[132,27,45],'841b2d',8657709],
['nbs-iscc-m:wildstrawberry.16','wildstrawberry','wild strawberry',[114,47,55],'722f37',7483191],
['nbs-iscc-m:willow.107','willow','willow',[102,93,30],'665d1e',6708510],
['nbs-iscc-m:willowgreen.106','willowgreen','willow green',[134,126,54],'867e36',8814134],
['nbs-iscc-m:windflower.262','windflower','windflower',[145,95,109],'915f6d',9527149],
['nbs-iscc-m:windsor.224','windsor','windsor',[86,60,92],'563c5c',5651548],
['nbs-iscc-m:windsorblue.204','windsorblue','windsor blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-m:windsorgreen.137','windsorgreen','windsor green',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:windsortan.54','windsortan','windsor tan',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:wineberry.259','wineberry','wineberry',[103,49,71],'673147',6762823],
['nbs-iscc-m:winedregs.16','winedregs','wine dregs',[114,47,55],'722f37',7483191],
['nbs-iscc-m:winedregs.41','winedregs','wine dregs',[86,7,12],'56070c',5637900],
['nbs-iscc-m:winedregs.44','winedregs','wine dregs',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-m:winelees.16','winelees','wine lees',[114,47,55],'722f37',7483191],
['nbs-iscc-m:winelees.41','winelees','wine lees',[86,7,12],'56070c',5637900],
['nbs-iscc-m:winelees.44','winelees','wine lees',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-m:wineyellow.89','wineyellow','wine yellow',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-m:wineyellow.90','wineyellow','wine yellow',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:wintergreen.136','wintergreen','winter green',[103,146,103],'679267',6787687],
['nbs-iscc-m:wintergreen.137','wintergreen','wintergreen ',[53,94,59],'355e3b',3497531],
['nbs-iscc-m:winterleaf.61','winterleaf','winter leaf',[99,81,71],'635147',6508871],
['nbs-iscc-m:wireless.182','wireless','wireless',[67,107,149],'436b95',4418453],
['nbs-iscc-m:wistariablue.210','wistariablue','wistaria [blue] ',[140,130,182],'8c82b6',9208502],
['nbs-iscc-m:wistariablue.199','wistariablue','wistaria blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:wistariaviolet.210','wistariaviolet','wistaria [violet] ',[140,130,182],'8c82b6',9208502],
['nbs-iscc-m:witchwood.58','witchwood','witchwood',[111,78,55],'6f4e37',7294519],
['nbs-iscc-m:witheredleaf.51','witheredleaf','withered leaf',[190,101,22],'be6516',12477718],
['nbs-iscc-m:witheredleaf.54','witheredleaf','withered leaf',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:witheredleaf.55','witheredleaf','withered leaf',[128,70,27],'80461b',8406555],
['nbs-iscc-m:witheredrose.19','witheredrose','withered rose',[144,93,93],'905d5d',9461085],
['nbs-iscc-m:witheredrose.43','witheredrose','withered rose',[121,68,59],'79443b',7947323],
['nbs-iscc-m:woad.186','woad','woad',[83,104,120],'536878',5466232],
['nbs-iscc-m:woald.101','woald','woald',[234,230,121],'eae679',15394425],
['nbs-iscc-m:woald.102','woald','woald',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:wod.101','wod','wod',[234,230,121],'eae679',15394425],
['nbs-iscc-m:wod.102','wod','wod',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:wode.186','wode','wode',[83,104,120],'536878',5466232],
['nbs-iscc-m:wold.101','wold','wold',[234,230,121],'eae679',15394425],
['nbs-iscc-m:wold.102','wold','wold',[185,180,89],'b9b459',12170329],
['nbs-iscc-m:woodash.93','woodash','woodash',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-m:woodbark.80','woodbark','woodbark',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-m:woodbinegreen.120','woodbinegreen','woodbine green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-m:woodbinegreen.122','woodbinegreen','woodbine green',[143,151,121],'8f9779',9410425],
['nbs-iscc-m:woodbinegreen.125','woodbinegreen','woodbine green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:woodbinegreen.127','woodbinegreen','woodbine green',[81,87,68],'515744',5330756],
['nbs-iscc-m:woodlandbrown.81','woodlandbrown','woodland brown',[72,60,50],'483c32',4734002],
['nbs-iscc-m:woodlandbrown.267','woodlandbrown','woodland brown',[34,34,34],'222222',2236962],
['nbs-iscc-m:woodlandgreen.125','woodlandgreen','woodland green ',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:woodlandrose.57','woodlandrose','woodland rose',[166,123,91],'a67b5b',10910555],
['nbs-iscc-m:woodrose.45','woodrose','wood rose',[151,127,115],'977f73',9928563],
['nbs-iscc-m:woodrose.60','woodrose','wood rose',[149,128,112],'958070',9797744],
['nbs-iscc-m:woodviolet.219','woodviolet','wood violet',[96,47,107],'602f6b',6303595],
['nbs-iscc-m:wren.111','wren','wren',[54,53,39],'363527',3552551],
['nbs-iscc-m:wren.113','wren','wren',[87,85,76],'57554c',5723468],
['nbs-iscc-m:wroughtiron.266','wroughtiron','wrought iron',[85,85,85],'555555',5592405],
['nbs-iscc-m:yacht.178','yacht','yacht',[0,103,165],'0067a5',26533],
['nbs-iscc-m:yaleblue.179','yaleblue','yale blue',[0,65,106],'00416a',16746],
['nbs-iscc-m:yama.163','yama','yama',[102,173,164],'66ada4',6729124],
['nbs-iscc-m:yellowbeige.76','yellowbeige','yellow beige',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:yellowbeige.77','yellowbeige','yellow beige',[130,102,68],'826644',8545860],
['nbs-iscc-m:yellowberries.68','yellowberries','yellow berries',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:yellowberries.71','yellowberries','yellow berries',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:yellowbrazilwood.87','yellowbrazilwood','yellow brazil wood',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:yellowbrazilwood.90','yellowbrazilwood','yellow brazil wood',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:yellowcarmine.68','yellowcarmine','yellow carmine',[234,162,33],'eaa221',15376929],
['nbs-iscc-m:yellowcarmine.71','yellowcarmine','yellow carmine',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:yellowdaisy.83','yellowdaisy','yellow daisy',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:yellowdaisy.84','yellowdaisy','yellow daisy',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:yellowdaisy.86','yellowdaisy','yellow daisy',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:yellowdaisy.87','yellowdaisy','yellow daisy',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:yellowearth.71','yellowearth','yellow earth',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:yellowmadder.83','yellowmadder','yellow madder',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:yellowmadder.84','yellowmadder','yellow madder',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:yellowmadder.86','yellowmadder','yellow madder',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:yellowmadder.87','yellowmadder','yellow madder',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:yellowochre.71','yellowochre','yellow ochre',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:yellowrealgar.83','yellowrealgar','yellow realgar',[250,218,94],'fada5e',16439902],
['nbs-iscc-m:yellowrealgar.84','yellowrealgar','yellow realgar',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:yellowrealgar.86','yellowrealgar','yellow realgar',[248,222,126],'f8de7e',16309886],
['nbs-iscc-m:yellowrealgar.87','yellowrealgar','yellow realgar',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:yellowsienna.71','yellowsienna','yellow sienna',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:yellowstone.105','yellowstone','yellow stone',[185,181,125],'b9b57d',12170621],
['nbs-iscc-m:yellowstone.91','yellowstone','yellowstone',[161,143,96],'a18f60',10588000],
['nbs-iscc-m:yellowwash.71','yellowwash','yellow wash',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:yellowwood.87','yellowwood','yellow wood',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-m:yellowwood.90','yellowwood','yellow wood',[194,178,128],'c2b280',12759680],
['nbs-iscc-m:yewgreen.125','yewgreen','yew green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-m:yolkyellow.84','yolkyellow','yolk yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:yosemite.76','yosemite','yosemite',[193,154,107],'c19a6b',12687979],
['nbs-iscc-m:yucatan.54','yucatan','yucatan',[174,105,56],'ae6938',11430200],
['nbs-iscc-m:yucatan.74','yucatan','yucatan',[153,101,21],'996515',10052885],
['nbs-iscc-m:yucca.150','yucca','yucca',[94,113,106],'5e716a',6189418],
['nbs-iscc-m:yuchi.146','yuchi','yu chi',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-m:yvetteviolet.228','yvetteviolet','yvette violet',[121,104,120],'796878',7956600],
['nbs-iscc-m:zaffreblue.179','zaffreblue','zaffre blue ',[0,65,106],'00416a',16746],
['nbs-iscc-m:zaffreblue.182','zaffreblue','zaffre blue ',[67,107,149],'436b95',4418453],
['nbs-iscc-m:zanzibar.47','zanzibar','zanzibar',[67,48,46],'43302e',4403246],
['nbs-iscc-m:zedoarywash.71','zedoarywash','zedoary wash',[227,168,87],'e3a857',14919767],
['nbs-iscc-m:zenithblue.199','zenithblue','zenith [blue] ',[135,145,191],'8791bf',8884671],
['nbs-iscc-m:zenithblue.203','zenithblue','zenith [blue] ',[140,146,172],'8c92ac',9212588],
['nbs-iscc-m:zephyr.7','zephyr','zephyr',[234,216,215],'ead8d7',15390935],
['nbs-iscc-m:zephyr.8','zephyr','zephyr',[196,174,173],'c4aead',12889773],
['nbs-iscc-m:zinc.233','zinc','zinc',[139,133,137],'8b8589',9143689],
['nbs-iscc-m:zinc.265','zinc','zinc',[132,132,130],'848482',8684674],
['nbs-iscc-m:zincgreen.136','zincgreen','zinc green',[103,146,103],'679267',6787687],
['nbs-iscc-m:zincorange.50','zincorange','zinc orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-m:zincorange.53','zincorange','zinc orange',[217,144,88],'d99058',14258264],
['nbs-iscc-m:zincyellow.84','zincyellow','zinc yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-m:zinnia.15','zinnia','zinnia',[171,78,82],'ab4e52',11226706],
['nbs-iscc-m:zulu.47','zulu','zulu',[67,48,46],'43302e',4403246],
['nbs-iscc-m:zunibrown.58','zunibrown','zuni brown',[111,78,55],'6f4e37',7294519]
    ];
}

sub _description {
    return {
          'subtitle' => 'Dictionary of Color',
          'title' => 'NBS/ISCC M',
          'description' => '    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


Color Name Dictionary: (M) Dictionary of Color

Maerz and Paul,
Dictionary of Color, 1st ed. 

This idiosyncratic dictionary\'s 3064 names map to 4401 colors, using 223 of the 267 centroids.

[http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#M]
'
        }

}

1;
