#!/usr/bin/env perl

use v5.24;
use strict;
use warnings;
use feature qw( say );
use utf8;
use Encode;

use FindBin;
use lib $FindBin::RealBin;

use Biber::CodePage qw( :DEFAULT );

my $CS_system = get_CS_system();
my $sys_is_utf8 = ($CS_system eq 'UTF-8') || ($CS_system eq 'CP65001');


#----------------------
say "===================\n-----**Tests of non-ASCII names of .bib files** ----\n";
#----------------------

say "=== Cleaning out old files";
system "latexmk", "-C", "-silent";

say "\n===================================================";

my @failures = ();

# Cases that should always succeed, if biber is bug free, given
#  System is macOS, Linux, or Windows with CP1252
my @main = (
    'bib-C1-glob.tex',
    'bib-C1.tex',
    'bib-C2.tex',
    'bib-D1.tex',
    'bib-D1-via-C.tex',
    'bib-Greek.tex',
    'bib-Mix.tex',
    );

# Cases that only work on a UTF-8 system, and fail for Windows with system CP1252. 
my @utf8_only = (
    'bib-D2.tex',
    'bib-D2-via-C.tex',
    'bib-Greek-glob.tex',
    );


#------------------------------

foreach (@main) { one_file($_); }

if ($sys_is_utf8) {
    say "Have UTF-8 system, so I'll do UTF-8 specific test(s).";
    foreach (@utf8_only) { one_file($_); }
}

say "\n===================================================";

if (@failures) {
    say "Failures on the following:";
    foreach (@failures) { say "  $_"; }
}
else {
    say "SUCCESS";
}

exit 1 + $#failures;

#----------------------
sub one_file {
    # Assumes there is a suitable latexmkrc file.
    my ($file) = @_;
    my $ret = system "latexmk", "-interaction=batchmode", "-pdf", "-bibtex",
                     encode_CS_system($file);
    if ($ret) { push @failures, $file; }
    return $ret;
}

#----------------------

