/* Copyright (C) 1997 Bjoern Beutel. */

/* Description. =============================================================*/

/* This module invokes and manages the transmit process from malaga. */

/* Includes. ================================================================*/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <setjmp.h>
#include <glib.h>
#include "basic.h"
#include "pools.h"
#include "values.h"
#include "files.h"
#include "input.h"
#include "symbols.h"
#include "commands.h"
#include "value_parser.h"
#include "scanner.h"
#include "options.h"
#include "rule_type.h"
#include "rules.h"
#include "processes.h"
#include "transmit.h"

/* Variables. ===============================================================*/

static process_t transmit_process;

/* Functions. ===============================================================*/

static void 
transmit_local( void )
/* Stack effects: VALUE -> NEW_VALUE
 * Start the Malaga transmit process by executing TRANSMIT_COMMAND_LINE
 * if it is not already running.
 * Send VALUE, in text format, to the transmit process and receive an answer
 * value in text format, convert it into internal format as NEW_VALUE. */
{
  string_t input_line, value_string;

  /* Start transmit process. */
  transmit_process.use_input = TRUE;
  transmit_process.use_output = TRUE;
  transmit_process.name = "transmit process";
  start_process( &transmit_process );

  /* Send top value to transmit process. */
  value_string = value_to_readable( value_stack[ --top ], TRUE, -1 );
  fprintf( transmit_process.output_stream, "%s\n", value_string );
  fflush( transmit_process.output_stream );
  free_mem( &value_string );

  /* Read result and convert it to value format. */
  input_line = read_line( transmit_process.input_stream );
  if (input_line == NULL) 
    complain( "Transmit process doesn't answer." );
  set_scanner_input( input_line );
  TRY 
  { 
    parse_a_value();
    parse_token( EOF );
  } 
  FINALLY 
  { 
    free_mem( &input_line );
    set_scanner_input( NULL );
  }
  END_TRY;
}

/*---------------------------------------------------------------------------*/

void 
init_transmit( void )
/* Initialise this module. */
{
  transmit = transmit_local;
}

/*---------------------------------------------------------------------------*/

void 
terminate_transmit( void )
/* Terminate this module. */
{
  stop_process( &transmit_process );
  transmit = NULL;
}

/*---------------------------------------------------------------------------*/

static void 
do_transmit_cmd_option( string_t arguments )
/* Set the command line to start the transmit process. */
{
  if (*arguments == EOS) 
  { 
    printf( "transmit-cmd: \"%s\"\n", (transmit_process.command_line == NULL ?
				       "" : transmit_process.command_line) );
  } 
  else 
  { 
    stop_process( &transmit_process );
    transmit_process.command_line = parse_word( &arguments );
  }

  parse_end( &arguments );
}

command_t transmit_cmd_option = 
{ 
  "transmit-cmd transmit-line transmit", do_transmit_cmd_option,
  "Usage: set transmit-cmd \"TRANSMIT_COMMAND_LINE\"\n"
  "Set the command that is used to start the transmit process.\n"
};

/*---------------------------------------------------------------------------*/

static void 
do_transmit( string_t arguments )
/* Communicate with the transmit process. */
{
  string_t result;

  set_scanner_input( arguments );
  TRY 
  {
    parse_a_value(); 
    test_token( EOF );
  }
  FINALLY 
    set_scanner_input( NULL );
  END_TRY;
  transmit_local();
  result = value_to_readable( value_stack[ --top ], FALSE, 0 );
  printf( "%s\n", result );
  free_mem( &result );
}

command_t transmit_command = 
{ 
  "transmit", do_transmit,
  "Usage: transmit VALUE\n"
  "Send VALUE to the transmit process and display the result.\n"
};

/* End of file. =============================================================*/
