'use strict'
const t = require('tap')
const Pax = require('../lib/pax.js')

t.test('create a pax', t => {
  const p = new Pax({
    atime: new Date('1979-07-01T19:10:00.000Z'),
    ctime: new Date('1979-07-01T19:10:00.000Z'),
    mtime: new Date('1979-07-01T19:10:00.000Z'),
    gid: 20,
    gname: 'staff',
    uid: 24561,
    uname: 'isaacs',
    path: 'foo.txt',
    size: 100,
    dev: 123456,
    ino: 7890,
    nlink: 1,
  })

  // console.log(p.encode().toString('hex').split('').reduce((s,c)=>{if(s[s.length-1].length<64)s[s.length-1]+=c;else s.push(c);return s},['']))

  const buf = Buffer.from(
    // pax entry header
    '5061784865616465722f666f6f2e747874000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000030303036343420003035373736312000303030303234200030303030' +
    '3030303330342000323136373231373631302000303136373332200078000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0075737461720030306973616163730000000000000000000000000000000000' +
    '0000000000000000007374616666000000000000000000000000000000000000' +
    '0000000000000000003030303030302000303030303030200000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000032313637' +
    '3231373631302000323136373231373631302000000000000000000000000000' +

    // entry body
    '313620706174683d666f6f2e7478740a3139206374696d653d32393937303432' +
    '30300a3139206174696d653d3239393730343230300a323120534348494c592e' +
    '6465763d3132333435360a313920534348494c592e696e6f3d373839300a3138' +
    '20534348494c592e6e6c696e6b3d310a39206769643d32300a313520676e616d' +
    '653d73746166660a3139206d74696d653d3239393730343230300a3132207369' +
    '7a653d3130300a3133207569643d32343536310a313620756e616d653d697361' +
    '6163730a00000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000',
    'hex')

  const actual = p.encode()
  t.equal(actual.toString('hex'), buf.toString('hex'))
  t.end()
})

t.test('null pax', t => {
  const p = new Pax({})
  t.equal(p.encode(), null)
  t.end()
})

t.test('tiny pax', t => {
  // weird to have a global setting a path.  Maybe this should be
  // an error?
  const p = new Pax({ path: 'ab' }, true)
  const actual = p.encode()
  // console.log(actual.toString('hex').split('').reduce((s,c)=>{if(s[s.length-1].length<64)s[s.length-1]+=c;else s.push(c);return s},['']))
  // return Promise.resolve()

  const buf = Buffer.from(
    // header
    '5061784865616465722f61620000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000030303036343420000000000000000000000000000000000030303030' +
    '3030303031332000000000000000000000000000303037303534200067000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0075737461720030300000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000003030303030302000303030303030200000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +

    // body
    // note that a 2-char path is 11, but a 1 char path is 9, because
    // adding the second char bumps the n to 10, which adds 1, which
    // means it has to be 11.
    // a 1-char path COULD be encoded as EITHER "10 path=x\n", or as
    // "9 path=x\n", and it'd be true either way.
    '313120706174683d61620a000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000' +
    '0000000000000000000000000000000000000000000000000000000000000000',
    'hex')

  t.equal(actual.toString('hex'), buf.toString('hex'))

  t.end()
})

t.test('parse', t => {
  t.same(Pax.parse('11 path=ab\n', { uid: 24561 }, true), {
    atime: null,
    charset: null,
    comment: null,
    ctime: null,
    gid: null,
    gname: null,
    linkpath: null,
    mtime: null,
    path: 'ab',
    size: null,
    uid: 24561,
    uname: null,
    dev: null,
    ino: null,
    nlink: null,
    global: true,
  })

  t.same(Pax.parse('11 path=ab\n', null, false), {
    atime: null,
    charset: null,
    comment: null,
    ctime: null,
    gid: null,
    gname: null,
    linkpath: null,
    mtime: null,
    path: 'ab',
    size: null,
    uid: null,
    uname: null,
    dev: null,
    ino: null,
    nlink: null,
    global: false,
  })

  t.same(Pax.parse('9 gid=20\n9 path=x\n', null, false), {
    atime: null,
    charset: null,
    comment: null,
    ctime: null,
    gid: 20,
    gname: null,
    linkpath: null,
    mtime: null,
    path: 'x',
    size: null,
    uid: null,
    uname: null,
    dev: null,
    ino: null,
    nlink: null,
    global: false,
  })

  t.same(Pax.parse('9 gid=20\n9 path=x\n', null, false), {
    atime: null,
    charset: null,
    comment: null,
    ctime: null,
    gid: 20,
    gname: null,
    linkpath: null,
    mtime: null,
    path: 'x',
    size: null,
    uid: null,
    uname: null,
    dev: null,
    ino: null,
    nlink: null,
    global: false,
  })

  t.same(Pax.parse('20 mtime=1491436800\n', null, false), {
    atime: null,
    charset: null,
    comment: null,
    ctime: null,
    gid: null,
    gname: null,
    linkpath: null,
    mtime: new Date('2017-04-06'),
    path: null,
    size: null,
    uid: null,
    uname: null,
    dev: null,
    ino: null,
    nlink: null,
    global: false,
  })

  const breaky =
    '93 NODETAR.package.readme=karma-moment\n' +
    '=================\n' +
    '\n' +
    'Karma plugin for Moment framework\n' +
    '\n'

  const noKey = '10 =pathx\n'

  t.same(Pax.parse(breaky + '9 gid=20\n10 path=x\n' + noKey, null, false), {
    atime: null,
    charset: null,
    comment: null,
    ctime: null,
    gid: 20,
    gname: null,
    linkpath: null,
    mtime: null,
    path: 'x',
    size: null,
    uid: null,
    uname: null,
    dev: null,
    ino: null,
    nlink: null,
    global: false,
  })

  t.end()
})
