#!/usr/bin/perl

use warnings;
use strict;

use Getopt::Long;
use POSIX;

our $VERSION=1.0;

our $tagname='gt';  # gettext mark tagname

our 
(
  $pot,             # Name for output POT
  $html,            # Name for input html-file
  $printversion,    # Print version only
  $setversion,      # version package
  $package,         # package name
  $copyright,       # copyright string
  $email,           # upstream author email
  $charset,         # charset 
  $langteam,        #
  $desc,            #
);

# show Usage
sub usage()
{
  print <<endusage;
Usage: 
  $0 [ -o output.pot ] [OPTIONS] input.html

OPTIONS:
  -o filename         - put template into filename;
  -v                  - print version;
  -tag tagname        - set mark tagname ( default: 'gt'
                        example: <gt> Selected text </gt> );
  -sv version         - set package version (for POT-headers);
  -package  name      - set package name (for POT-headers);
  -copyright string   - set copyright string (for POT-headers);
  -email mail\@domain  - set upstream author email (for POT-headers);
  -charset name       - set charset (for POT-headers);
  -langteam team      - set language team (for POT-headers);
  -desc string        - descreptive string (for POT-headers);

endusage
  exit -1;
}


GetOptions
(
  "o=s"           => \$pot,
  v               => \$printversion,
  "tag=s"         => \$tagname,
  "sv=s"          => \$setversion,
  "package=s"     => \$package,
  "copyright=s"   => \$copyright,
  "email=s"       => \$email,
  "charset=s"     => \$charset,
  "langteam=s"    => \$langteam,
  "desc=s"        => \$desc,

) or usage;

if ($printversion) { print $VERSION, "\n"; exit 0; }
$html=$ARGV[0];
defined $html or usage;
-r $html or die "Can not found file '$html'";

my $data='';
{
  open my $file, '<', $html or die "Can not open file '$html': $!";
  local $/; $data=<$file>;
}

if (defined $pot)
{
  open my $file_out, ">", $pot or die "Can not open file '$pot': $!";
  POSIX::dup2(fileno($file_out), fileno(STDOUT));
}

my @blocks=split /<\s*$tagname\s*>.*?<\s*\/$tagname\s*>/is, $data;
my @tags=$data=~/<\s*$tagname\s*>\s*(.*?)\s*<\s*\/$tagname\s*>/isg;
s/\s+/ /sg, s/^\s//, s/\s$//, s/\\/\\\\/gs, s/"/\\"/gs for @tags;

my @ct=localtime;
my $current_time=sprintf('%4d-%02d-%02d %02d:%02d',
  $ct[5]+1900, $ct[4]+1, @ct[3, 2, 1]);


defined $setversion or $setversion='PACKAGE VERSION';
defined $package or $package='PACKAGE';
defined $copyright or 
  $copyright="Copyright (C) YEAR THE PACKAGE'S COPYRIGHT HOLDER";
defined $email or $email='FIRST AUTHOR <EMAIL\@ADDRESS>, YEAR.';
defined $charset or $charset='CHARSET';
defined $langteam or $langteam='LANGUAGE <LL@li.org>';
defined $desc or $desc='SOME DESCRIPTIVE TITLE.';
print <<pot;
# $desc
# $copyright
# This file is distributed under the same license as the $package package.
# $email
#
#, fuzzy
msgid ""
msgstr ""
"Project-Id-Version: $setversion\\n"
"Report-Msgid-Bugs-To: \\n"
"POT-Creation-Date: $current_time\\n"
"PO-Revision-Date: YEAR-MO-DA HO:MI+ZONE\\n"
"Last-Translator: FULL NAME <EMAIL\@ADDRESS>\\n"
"Language-Team: $langteam\\n"
"MIME-Version: 1.0\\n"
"Content-Type: text/plain; charset=$charset\\n"
"Content-Transfer-Encoding: 8bit\\n"

pot

my %write_hash;
for (@tags)
{
  exists $write_hash{$_} and next;
  $write_hash{$_}=1;
  print qq(#\nmsgid "$_"\nmsgstr ""\n\n);
}
