#ifndef _RHEOLEF_FLOAT_H
#define _RHEOLEF_FLOAT_H
//
// This file is part of Rheolef.
//
// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
//
// Rheolef is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Rheolef is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Rheolef; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// =========================================================================
// compiler-dependent part for rheolef implementation: the floating point
// author: Pierre.Saramito@imag.fr
// date: 27 january 2000

namespace rheolef {
/**
@classfile Float default floating point

Description
===========
The default floating point in the Rheolef library is the `double`
precision type (64 bits):

    typedef double Float;

This default behavior could be changed at the @ref configuration_page
stage, before the compilation of the library.
For instance, choosing the `float128` quadruple precision (128 bits)
is possible.

Using the `Float` type instead of `double` one
open the possibility to run codes that use Rheolef
with different floating point precision.

Implementation
==============
@showfromfile
*/
} // namespace rheolef

#include "rheolef/config.h"   /* as generated by configure */

#include <cmath>

namespace rheolef {
using std::cos;
using std::sin;
using std::tan;
using std::acos;
using std::asin;
using std::atan;
using std::cosh;
using std::sinh;
using std::tanh;
using std::exp;
using std::log;
using std::log10;
using std::sqrt;
using std::abs;
using std::fabs;
using std::floor;
using std::ceil;
using std::atan2;
using std::pow;
using std::fmod;
} // namespace rheolef


#include <complex>

namespace rheolef {

/// @brief helper for std::complex<T>: get basic T type
template<class T>
struct float_traits {
  typedef T type;
};
template<class T>
struct float_traits<std::complex<T> > 
{
  typedef typename float_traits<T>::type type;
};
} // namespace rheolef

// ==================================================================================================

#include <limits>

#include "rheolef/numeric_flags.h"

// -------------------------------------------------
// Float
// -------------------------------------------------
#if defined(_RHEOLEF_HAVE_FLOAT128)
#include <boost/multiprecision/float128.hpp>
namespace rheolef {
using boost::multiprecision::float128;
using namespace boost::multiprecision;
static inline float128 sqr  (const float128& x) { return (x*x); }
static inline float128 norm (const float128& x) { return fabs(x); } // for completeness with field_expr
//! @brief see the @ref Float_2 page for the full documentation
using Float = float128;
//using boost::math::isnan;
//using boost::math::isinf;
//using boost::math::isfinite;
//using boost::math::isnormal;
} // namespace rheolef

#elif defined(_RHEOLEF_HAVE_CLN)
# include "rheolef/bigfloat.h"
namespace rheolef {
//! @brief see the @ref Float_2 page for the full documentation
typedef bigfloat<_RHEOLEF_DIGITS10> Float;
} // namespace rheolef

#elif defined(_RHEOLEF_HAVE_LONG_DOUBLE)
namespace rheolef {
//! @brief see the @ref Float_2 page for the full documentation
typedef long double Float;
} // namespace rheolef

#else
namespace rheolef {
//! @brief see the @ref Float_2 page for the full documentation
typedef double Float;
} // namespace rheolef
#endif // Float

// is any integer or float type (included extended precision, if used)
namespace rheolef { namespace details {
template <class T, class Sfinae = void>
struct is_rheolef_arithmetic: std::false_type {};

template <class T>
struct is_rheolef_arithmetic<T,
typename std::enable_if<
       std::is_arithmetic<T>::value
    || std::is_same<typename std::decay<T>::type,double>::value
>::type>
: std::true_type {};

#ifdef _RHEOLEF_HAVE_FLOAT128
template <class T>
struct is_rheolef_arithmetic<T,
typename std::enable_if<
    std::is_same<typename std::decay<T>::type,float128>::value
>::type>
: std::true_type {};
#endif // _RHEOLEF_HAVE_FLOAT128

}} // namespace rheolef::details

// -------------------------------------------------

namespace rheolef { namespace details {
template<class T, class Check = T>
struct upgrade_integral_to_float {
  typedef T type;
};
template<class Int>
struct upgrade_integral_to_float<Int, typename std::enable_if<std::is_integral<Int>::value,Int>::type>
{
  // integers upgrated to default Float
  typedef Float type;
};

}} // namespace rheolef::details

#if !defined(_RHEOLEF_HAVE_SQR_DOUBLE)
namespace rheolef {
static inline double sqr (const double& x) { return (x*x); }
} // namespace rheolef
#endif

#include <functional> // greater/less, ect...
#include <numeric>    // inner_product, ect...
#include <algorithm>

namespace rheolef {
using std::min;
using std::max;
#ifdef TO_CLEAN
// TODO: promote<T1,T2>::type max(T1,T2) {...}
// conflict with max(field_nonlinear_expr<E>,int)
template <class T> T max (T x, int y) { return x > y ? x : T(y); }
template <class T> T max (int x, T y) { return x > y ? T(x) : y; }
template <class T> T min (T x, int y) { return x < y ? x : T(y); }
template <class T> T min (int x, T y) { return x < y ? T(x) : y; }
// conflict with ginac::abs
template <class T> T abs (T x)        { return (x > T(0) ? x : -x); }
#endif // TO_CLEAN
} // namespace rheolef

#include <climits>

#endif // _RHEOLEF_FLOAT_H
