use ui;
use progvis:program;
use progvis:data;

/**
 * Restart the program and run it to the first location.
 */
void toStart(Program program) on Render {
	program.reset();
	program.spawn(program.main);

	// First, we need to wait until we find at least one thread. Otherwise 'wait' will not wait properly.
	while (program.threads.empty)
		yield();

	wait(program);
}

/**
 * Find the thread to step from a sequential ID.
 */
ProgThread findSeqThread(Program program, Nat seqId) on Render {
	if (program.sortedThreads.count <= seqId)
		throw CheckError("Can not step thread ${seqId}, it does not exist.");

	var thread = program.sortedThreads[seqId];
	if (thread.sleeping)
		throw CheckError("Can not step the sleeping thread ${seqId}.");

	thread;
}

/**
 * Advance the thread with a particular ID to the next barrier.
 */
void toNext(Program program, Nat seqId) on Render {
	var thread = findSeqThread(program, seqId);

	if (thread.barriersAvailable) {
		thread.resumeBarrier();
	} else {
		thread.resume();
	}

	wait(program);
}

/**
 * Advance to a particular node. Restarts the program and executes it to the location.
 */
void toNode(Program program, Node target) on Render {
	toStart(program);

	Nat[] actions = target.path();

	for (x in actions)
		toNext(program, x);
}

/**
 * Wait for an operation to complete.
 */
void wait(Program program) on Render {
	// Note: We need to wait first so that new threads will have the chance to register themselves
	// properly with the Program instance. If this is slower than one "yield", then we are out of
	// luck. In current language implementations we are, however, more or less guaranteed that this
	// is the case though.
	do {
		yield();
	} while (!program.threadsWaiting());
}
