use core:io;
use core:net;
use lang:bs:macro;
use crypto;

/**
 * Thrown when the user needs to sign in.
 */
class SignInRedirect extends Exception {
	init(Url to) {
		init { to = to; }
	}

	// Redirect here.
	Url to;

	void message(StrBuf to) : override {
		to << "Redirect here to sign in: " << to;
	}
}

/**
 * Thrown on other errors.
 */
class ServerError extends Exception {
	init(Str msg) {
		init { msg = msg; }
	}

	Str msg;

	void message(StrBuf to) : override {
		to << msg;
	}
}

// Helper for 'connect'.
private NetStream connectOrThrow(Str host, Nat port) {
	if (c = connect(host, port))
		return c;
	throw ServerError("Failed to connect to ${host}.");
}

/**
 * Connection to the server.
 */
class Client on Network {
	// Connect and authenticate the user.
	Client connect(Str host, Str identity) : static {
		NetStream socket = connectOrThrow(host, serverPort);

		if (host == "storm-lang.org") {
			var context = ClientContext:pinnedTo(Certificate:loadPEM(serverCertificate));
			context.verifyHostname = false;

			var session = context.connect(socket, host);
			return Client(socket, session.input, session.output, identity);
		} else {
			return Client(socket, socket.input, socket.output, identity);
		}
	}

	// Authenticates the user.
	private init(NetStream socket, IStream input, OStream output, Str identity) {
		init {
			socket = socket;
			input = ObjIStream(input);
			output = ObjOStream(output);
		}

		var response = query(AuthRequest(identity));
		if (response as AuthResponse) {
			uName = response.name;
			if (response.version > named{PROGVIS_VERSION}.version)
				throw ServerError("Please upgrade Progvis. Version ${response.version} is available.");
			print("Username: ${uName}");
		} else if (response as AuthLoginResponse) {
			throw SignInRedirect(response.url);
		}
	}

	private NetStream socket;
	private ObjIStream input;
	private ObjOStream output;

	private Str uName;
	Str username() { uName; }

	// Perform a query to the server.
	Response query(Request request) {
		request.write(output);
		output.flush();

		var reply = Response:read(input);
		if (reply as ErrorResponse)
			throw ServerError(reply.message);
		else if (reply as AuthResponse)
			uName = reply.name;
		reply;
	}

	// Close the connection.
	void close() {
		socket.close();
	}
}
