use ui;
use layout;
use progvis:net;
use core:geometry;
use core:io;

/**
 * Dialog for uploading new problems.
 */
dialog UploadDlg {
	layout Grid {
		expandCol: 1;
		Label nameLbl("Name:", HAlign:right, VAlign:center) {}
		Edit name("Name your problem") { colspan: 2; }
		nextLine;
		Label mainLbl("Test program:", HAlign:right, VAlign:center) {}
		Edit mainName("") {}
		Button mainButton("Select...") {}
		nextLine;
		Label implLbl("Implementation:", HAlign:right, VAlign:center) {}
		Edit implName("") {}
		Button implButton("Select...") {}
		nextLine;
		Label refLbl("Reference implementation:", HAlign:right, VAlign:center) {}
		Edit refName("") {}
		Button refButton("Select...") {}
		nextLine;
		Progress progress { col: 0; colspan: 2; }
		Button uploadButton("Upload") { col: 2; }
	}

	Client client;

	// Upload a new problem.
	init(Client client) {
		init("Submit a problem", Size(400, 10)) {
			client = client;
		}

		progress.visible = false;
		defaultChoice(uploadButton);
		uploadButton.onClick = &this.upload();

		mainButton.onClick = &this.pickMain();
		implButton.onClick = &this.pickImpl();
		refButton.onClick = &this.pickRef();
	}

	void pickMain() {
		unless (f = pickFile())
			return;

		mainName.text = f.format;

		Str title = f.title;
		Str ext = f.ext;

		Url impl = f.parent / (title + "_impl." + ext);
		Url ref = f.parent / (title + "_ref." + ext);
		if (impl.exists())
			implName.text = impl.format;
		if (ref.exists())
			refName.text = ref.format;
	}

	void pickImpl() {
		unless (f = pickFile())
			return;

		implName.text = f.format;
	}

	void pickRef() {
		unless (f = pickFile())
			return;

		refName.text = f.format;
	}

	private Url? pickFile() {
		FileTypes ft("Source code");
		for (k in progvis:program:Program:supportedFiles)
			ft.add("${k}-source", [k]);

		var picker = FilePicker:open(ft);
		if (!picker.show(this))
			return null;

		picker.result;
	}

	void upload() {
		unless (name.text != "") {
			showMessage(this, "No name", "You need to specify a name for the problem.");
			return;
		}

		unless (main = loadCode(mainName.text)) {
			showMessage(this, "File not found", "The file for the test program does not exist!");
			return;
		}

		unless (impl = loadCode(implName.text)) {
			showMessage(this, "File not found", "The file for the implementation does not exist!");
			return;
		}

		unless (ref = loadCode(refName.text)) {
			showMessage(this, "File not found", "The file for the reference implementation does not exist!");
			return;
		}

		progress.visible = true;
		progress.wait();
		for (x in children)
			x.enabled = false;

		(spawn checkCode(name.text, main, impl, ref)).detach();
	}

	private progvis:net:Code? loadCode(Str path) {
		Url url = parsePath(path);
		if (!url.exists)
			return null;

		Str contents = url.readAllText;
		if (contents.empty)
			return null;

		return progvis:net:Code(contents, url.ext);
	}

	void checkCode(Str name, progvis:net:Code main, progvis:net:Code impl, progvis:net:Code ref) {
		MemoryProtocol memory;
		Url m = main.put("main", memory);
		Url i = impl.put("impl", memory);
		Url r = ref.put("ref", memory);

		try {
			if (error = progvis:check:check([m, r])) {
				error("The test program does not follow the specification from the reference solution.");
				return;
			}
		} catch (core:lang:CodeError e) {
			error("Error while compiling the reference solution:\n" + e.message, e.pos);
			return;
		}

		Bool currentError = false;
		try {
			if (error = progvis:check:check([m, i])) {
				currentError = true;
			} else {
				currentError = false;
			}
		} catch (core:lang:CodeError e) {
			error("Error while compiling the implementation:\n" + e.message, e.pos);
			return;
		}

		try {
			client.query(NewProblemRequest(name, main, impl, ref, currentError));
		} catch (Exception e) {
			error(e.message);
			return;
		}

		done();
		close(1);
	}

	void error(Str message, core:lang:SrcPos pos) {
		done();
		progvis:CompileErrorDialog(message, pos).show(this);
	}

	void error(Str message) {
		done();
		showMessage(this, "Error", message);
	}

	void done() {
		progress.visible = false;
		for (x in children)
			x.enabled = true;
	}

}


/**
 * Dialog for uploading solutions to problems.
 */
dialog SolutionUploadDlg {
	layout Grid {
		expandCol: 1;
		Label nameLbl("Name:", HAlign:right, VAlign:center) {}
		Edit name("Name your problem") { colspan: 2; }
		nextLine;
		Label status("Checking your solution...") { colspan: 3; }
		nextLine;
		Progress progress { col: 0; colspan: 2; }
		Button uploadButton("Upload") { col: 2; }
	}

	private Client client;
	private Problem original;
	private progvis:net:Code? newMain;
	private progvis:net:Code? newImpl;
	private ImprovedProblemRequest? request;

	// Cancel auto-grading.
	private Bool cancel;

	init(Client client, Problem original, progvis:net:Code? newMain, progvis:net:Code? newImpl) {
		init("Submit your solution", Size(400, 10)) {
			client = client;
			original = original;
			newMain = newMain;
			newImpl = newImpl;
			cancel = false;
		}

		name.text = original.title;

		progress.wait();
		uploadButton.enabled = false;
		defaultChoice = uploadButton;
		uploadButton.onClick = &this.upload();

		(spawn checkCode()).detach();
	}

	void onDestroy(Int code) : override {
		cancel = true;
	}

	void upload() {
		unless (request)
			return;

		Str n = name.text;
		if (n.empty) {
			showMessage(this, "Error", "You need to specify a name for the problem.");
			return;
		}

		request.title = n;

		try {
			client.query(request);
			close(1);
		} catch (Exception e) {
			showMessage(this, "Error", e.message);
		}
	}

	void checkCode() {
		MemoryProtocol memory;
		Url m = (if (newMain) { newMain; } else { original.main; }).put("main", memory);
		Url i = (if (newImpl) { newImpl; } else { original.impl; }).put("impl", memory);
		Url r = original.ref.put("ref", memory);

		UiProgress progress(this);

		if (newMain.any) {
			try {
				if (error = progvis:check:check([m, r], progress)) {
					error("The test program does not follow the specification of the problem.");
					return;
				}
			} catch (core:lang:CodeError e) {
				if (file = e.pos.file) {
					if (file == m) {
						error("Error while compiling your test program with the reference implementation:\n" + e.message, e.pos);
						return;
					}
				}
				error("Error while compiling your test program with the reference implementation:\n" + e.message);
			}
		}

		if (newImpl) {
			// Check if the new implementation is different from the original one.
			if (newImpl.signature == original.impl.signature) {
				error("You have not modified the implementation. Make some modifications and click \"Reload\" before submitting.");
				return;
			}
		}

		if (cancel)
			return;

		Bool currentError = false;
		try {
			if (error = progvis:check:check([m, i], progress)) {
				currentError = true;
			} else {
				currentError = false;
			}
		} catch (core:lang:CodeError e) {
			error("Error while compiling the implementation:\n" + e.message, e.pos);
		}

		if (newMain.any) {
			if (currentError) {
				done("You found an error, click \"Upload\" to proceed!");
			} else {
				error("Your test program did not find an error in the current implementation, try again!");
			}
			size = minSize();
		} else {
			if (currentError) {
				// We allow uploading the problems again, but with reduced reward. Thus, one student
				// may solve one issue, but another may solve another issue. We check so that the
				// code is not identical as the first part here.
				done("Your solution does not solve all errors with this test program.\nYou may still upload your solution, but you will not receive full points for it.");
			} else {
				done("You fixed the current issues, click \"Upload\" to proceed!");
			}
		}

		if (uploadButton.enabled)
			request = ImprovedProblemRequest(original.id, "", newMain, newImpl, currentError);
	}

	class UiProgress extends progvis:check:Progress {
		init(SolutionUploadDlg owner) {
			init { owner = owner; }
		}

		SolutionUploadDlg owner;

		void progress(Nat depth, Nat states, Nat edges, Nat unexplored) : override {
			if (owner.cancel)
				throw progvis:program:ExitError();
		}
	}

	private void error(Str message, core:lang:SrcPos pos) {
		status.text = "Compilation error, try again.";
		progvis:CompileErrorDialog(message, pos).show(this);
	}

	void error(Str message) {
		status.text = message;
		progress.visible = false;
		size = minSize();
	}

	void done(Str message) {
		status.text = message;
		progress.visible = false;
		uploadButton.enabled = true;
		size = minSize();
	}

}
