use core;
use core:lang;

// Implementation of the string concatenation operator (#).
class ConcatOperator extends OpInfo {
	init(SStr op, Int prio) {
		init(op, prio, false) {}
	}

	Expr meaning(Block block, Expr lhs, Expr rhs) {
		ConcatExpr r(pos, block);
		appendTo(r, lhs);
		appendTo(r, rhs);
		r;
	}

}

// Helper to append parameters.
void appendTo(ConcatExpr expr, Expr e) on Compiler {
	if (e as Operator) {
		OpInfo info = e.op;
		if (info as ConcatOperator) {
			appendTo(expr, e.lhs);
			appendTo(expr, e.rhs);
		} else {
			expr.append(e);
		}
	} else {
		expr.append(e);
	}
}


// The concatenation expression itself. It generates code of
// the structure: { StrBuf b; { buf.add(...); buf.add(...); } buf; }
class ConcatExpr extends Expr {
	// Inner block, so that we can easily add new expressions at the right position.
	ExprBlock outer;
	ExprBlock inner;
	Var strBufVar;

	// Saved state, if any.
	Var? savedState;

	// Resolved string buffer type.
	Type strBufType;

	// Create the root node.
	init(SrcPos pos, Block parent) {
		ExprBlock outer(pos, parent);
		ExprBlock inner(pos, outer);

		Type strBufT = {
			Name t;
			t.add("core");
			t.add("StrBuf");

			if (found = rootScope.find(t) as Type)
				found;
			else
				throw InternalError("Failed to find the StrBuf type.");
		};

		init(pos) {
			outer = outer;
			inner = inner;
			strBufType = strBufT;
			strBufVar(outer, Value(strBufT), SStr("b"), Actuals());
		}

		// Generate the outer parts.
		outer.add(strBufVar);
		outer.add(inner);
		outer.add(namedExpr(outer, SStr("toS"), LocalVarAccess(pos, strBufVar.var), Actuals()));
	}

	// Save the state of the string buffer. Only done the first time called.
	void saveState() {
		if (savedState)
			return;

		Var v(inner, SStr("fmt"), namedExpr(inner, SStr("format"), LocalVarAccess(SrcPos(), strBufVar.var)));
		savedState = v;
		inner.add(v);
	}

	// Restore the state of the string buffer.
	void restoreState() {
		unless (savedState) {
			return;
		}

		LocalVarAccess buf(SrcPos(), strBufVar.var);
		LocalVarAccess fmt(SrcPos(), savedState.var);
		inner.add(namedExpr(inner, SStr("format"), buf, Actuals(fmt)));
	}

	// Append a node to the end of the concatenation.
	void append(Expr e) {
		// First, check if StrBuf has an operator << inside itself. If so, use that.
		// TODO: If we ever move the << operator for values in StrBuf to be generated in StrBuf
		// rather than in core, we should probably revise this logic in order to avoid generating
		// many << overloads.
		if (appendStrBuf(e))
			return;

		// Then, try to use the toS(StrBuf) function in the target type.
		if (appendToSParam(e))
			return;

		// Then, try to use the toS() function in the target type.
		if (appendToSNoParam(e))
			return;

		StrBuf msg;
		msg << "Can not convert " << e.result << " to string by calling 'toS'.";
		throw SyntaxError(pos, msg.toS);
	}

	// Append a node, trying to use a << operator inside the StrBuf directly.
	private Bool appendStrBuf(Expr e) {
		var part = SimplePart("<<", Value:[thisPtr(strBufType), e.result.type]);
		unless (member = strBufType.find(part, outer.scope) as Function)
			return false;

		Actuals params;
		params.add(LocalVarAccess(pos, strBufVar.var));
		params.add(e);
		inner.add(FnCall(pos, outer.scope, member, params));
		return true;
	}

	// Append a node, try to call toS(StrBuf) on the object.
	private Bool appendToSParam(Expr e) {
		Value r = e.result.type;
		unless (type = r.type)
			return false;

		Scope findIn = outer.scope.child(type);
		var part = SimplePart("toS", Value:[r, Value(strBufType)]);
		unless (member = findIn.find(Name(part)) as Function)
			return false;

		Actuals params;
		params.add(e);
		params.add(LocalVarAccess(pos, strBufVar.var));
		inner.add(FnCall(pos, outer.scope, member, params));

		return true;
	}

	// Append a node, try to call toS() on the object.
	private Bool appendToSNoParam(Expr e) {
		Value r = e.result.type;
		unless (type = r.type)
			return false;

		Scope findIn = outer.scope.child(type);
		var part = SimplePart("toS", Value:[r]);
		unless (member = findIn.find(Name(part)) as Function)
			return false;

		FnCall call(pos, outer.scope, member, Actuals(e));
		appendStrBuf(call);

		return true;
	}

	// Relay code generation and type info to the blocks.
	ExprResult result() {
		outer.result();
	}

	void code(CodeGen g, CodeResult r) {
		outer.code(g, r);
	}

	// Nice output.
	Str toS() {
		outer.toS;
	}

}


// Fixup the strings from multiline str{} literals.
Expr rawMultilineStrConstant(SrcPos pos, Str str) {
	rawStrConstant(pos, str.removeIndentation.trimBlankLines);
}

/**
 * Interpolated strings. Uses the # syntax.
 */

/**
 * An interpolated string element.
 */
class InterpolatedItem on Compiler {
	// Modifiers to append before 'insert'.
	Expr[] modifiers;

	// Expression to insert.
	Expr insert;

	// Context.
	Block context;

	// Create.
	init(Block ctx, Expr e) {
		init() { insert = e; context = ctx; }
	}

	// Add a modifier.
	void modify(SrcPos pos, Str name, Actuals params) {
		SrcName n(pos);
		n.add("core");
		n.add(name);

		modifiers << namedExpr(context, n, params);
	}

	// Wrap 'expr'.
	void wrap(SrcPos pos, Str name) {
		SrcName n(pos);
		n.add("core");
		n.add(name);

		insert = namedExpr(context, n, Actuals(insert));
	}
}

// Create string interpolation expressions.
Expr interpolateStr(Block parent, Expr first, InterpolatedItem item, Str second) on Compiler {
	var r = if (first as ConcatExpr) {
		first;
	} else {
		ConcatExpr r(first.pos, parent);
		if (first as StrLiteral) {
			if (first.v.any) {
				r.append(first);
			}
		} else {
			r.append(first);
		}
		r;
	};

	// Append 'item' with any modifiers. If we have any modifiers, save the state first, so that we
	// can restore it later!
	if (item.modifiers.any) {
		r.saveState();

		for (m in item.modifiers)
			r.append(m);

		r.append(item.insert);
		r.restoreState();
	} else {
		r.append(item.insert);
	}

	if (second.any)
		r.append(strConstant(first.pos, second));
	r;
}

// Set width of the output.
void setWidth(SrcPos pos, InterpolatedItem item, Str nr) on Compiler {
	item.modify(pos, "width", Actuals(intConstant(pos, nr)));
}

// Set to left align.
void setLeft(SrcPos pos, InterpolatedItem item) on Compiler {
	item.modify(pos, "left", Actuals());
}

// Set to right align.
void setRight(SrcPos pos, InterpolatedItem item) on Compiler {
	item.modify(pos, "right", Actuals());
}

// Set fill character.
void setFill(SrcPos pos, InterpolatedItem item, Str fill) on Compiler {
	var cp = NumLiteral(pos, fill.begin.v.codepoint.long);
	SrcName charName(pos);
	charName.add("core");
	charName.add("Char");
	var ch = namedExpr(item.context, charName, Actuals(cp));
	item.modify(pos, "fill", Actuals(ch));
}

// Set to hex.
void setHex(SrcPos pos, InterpolatedItem item) on Compiler {
	item.wrap(pos, "hex");
}

// Set significant digits.
void setSignificant(SrcPos pos, InterpolatedItem item, Str digits) on Compiler {
	item.modify(pos, "significant", Actuals(intConstant(pos, digits)));
}

void setDecimals(SrcPos pos, InterpolatedItem item, Str digits) on Compiler {
	item.modify(pos, "fixed", Actuals(intConstant(pos, digits)));
}

void setScientific(SrcPos pos, InterpolatedItem item, Str digits) on Compiler {
	item.modify(pos, "scientific", Actuals(intConstant(pos, digits)));
}

