use core:lang;
use lang:bs:macro;

/**
 * Test the deadlock avoidance of the futures related to compilation.
 *
 * The core issue is that lazy compilation of functions uses os::Future<Semaphore> to wait for
 * compilation to complete without breaking the concurrency model (no UThreads interrupt except on
 * thread calls or IO). This means that if the compilation process requires executing code on a
 * thread that is already waiting for the compilation thread we get into a deadlock. This would not
 * happen with os::Future<os::Sema> since os::Sema still allows other UThreads to run.
 *
 * Below, we trigger this issue as follows:
 * - We run 'compileDeadlock' from the tests on the Compiler thread as usual. This is not a part of
 *   the problem, it only makes what happens later clearer.
 * - We create a DeadlockObj and call run(). This runs on the thread Other.
 * - We call 'increaseCounter' to make sure it is compiled for later, so that compilation does not
 *   interfere with what we are doing later.
 * - We spawn another UThread on other to call increaseCounter. This is so that we can verify
 *   that the call to 'notCompiled' later appears as if it is a normal call.
 * - We call 'notCompiled'. This causes it to be compiled, which causes the compiler thread to
 *   execute code in the generator. Crucially, this involves code on Other, which is blocked.
 *   The issue was discovered in relation to initialization of global variables, but here a more
 *   explicit approach is taken to highlight the complexity of the issue.
 */

class DeadlockObj on Other {
	Int counter;

	Bool run() {
		// Make sure 'increaseCounter' is compiled and the global variable is initialized.
		increaseCounter();

		// Spawn a thread that runs increaseCounter.
		spawn increaseCounter();

		Int before = counter;
		Int callsBefore = otherFnCalls;

		// It should not be possible to observe that other threads run and modify 'counter' here.
		notCompiled();

		Int after = counter;
		Int callsAfter = otherFnCalls;

		Bool ok = true;

		if (callsBefore == callsAfter) {
			print("Error: We failed to trigger compilation. Did an earlier test compile everything?");
			ok = false;
		}

		if (before != after) {
			print("Error: Observed a change from ${before} to ${after}");
			ok = false;
		}

		ok;
	}

	void increaseCounter() {
		for (Nat i = 0; i < 10; i++) {
			counter++;
			yield();
		}
	}

	void notCompiled() {
		var x = DeadlockHelper();
	}
}

Bool compileDeadlock() on Compiler {
	DeadlockObj o;
	o.run();
}


// Generator:

DeadlockHelper : generate(params) {
	if (params.count != 0)
		return null;

	// Run code on Other:
	otherFn();

	// Just return some type.
	Type t("T", TypeFlags:typeClass);
	t.add(TypeDefaultCtor(t));
	t;
}

Int otherFnCalls on Other = 0;

void otherFn() on Other {
	otherFnCalls++;
}
