/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.core;

import java.util.Enumeration;
import java.util.Hashtable;
import lombok.Getter;
import lombok.Setter;
import org.apache.hop.core.row.IRowMeta;

/**
 * This class caches database queries so that the same query doesn't get called twice. Queries are
 * often launched to the databases to get information on tables etc.
 */
public class DbCache {
  private static DbCache dbCache;

  private Hashtable<DbCacheEntry, IRowMeta> cache;

  @Getter @Setter private boolean active;

  public void put(DbCacheEntry entry, IRowMeta fields) {
    if (!active) {
      return;
    }

    IRowMeta copy = fields.clone();
    cache.put(entry, copy);
  }

  /**
   * Get the fields as a row generated by a database cache entry
   *
   * @param entry the entry to look for
   * @return the fields as a row generated by a database cache entry
   */
  public IRowMeta get(DbCacheEntry entry) {
    if (!active) {
      return null;
    }

    IRowMeta fields = cache.get(entry);
    if (fields != null) {
      fields = fields.clone(); // Copy it again!
    }

    return fields;
  }

  public int size() {
    return cache.size();
  }

  /**
   * Clear out all entries of database with a certain name
   *
   * @param dbname The name of the database for which we want to clear the cache or null if we want
   *     to clear it all.
   */
  public void clear(String dbname) {
    if (dbname == null) {
      cache = new Hashtable<>();
    } else {
      Enumeration<DbCacheEntry> keys = cache.keys();
      while (keys.hasMoreElements()) {
        DbCacheEntry entry = keys.nextElement();
        if (entry.sameDB(dbname)) {
          // Same name: remove it!
          cache.remove(entry);
        }
      }
    }
  }

  private DbCache() {
    cache = new Hashtable<>();
    active = true;
  }

  /**
   * Create the database cache instance by loading it from disk
   *
   * @return the database cache instance.
   */
  public static DbCache getInstance() {
    if (dbCache == null) {
      dbCache = new DbCache();
    }
    return dbCache;
  }

  /** Clear the whole cache in one go */
  public static void clearAll() {
    getInstance().clear(null);
  }
}
