/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.database.connector.firebird.metadata.data.loader;

import org.apache.shardingsphere.database.connector.core.metadata.data.loader.DialectMetaDataLoader;
import org.apache.shardingsphere.database.connector.core.metadata.data.loader.MetaDataLoaderMaterial;
import org.apache.shardingsphere.database.connector.core.metadata.data.loader.type.TableMetaDataLoader;
import org.apache.shardingsphere.database.connector.core.metadata.data.model.SchemaMetaData;
import org.apache.shardingsphere.database.connector.core.metadata.data.model.TableMetaData;
import org.apache.shardingsphere.database.connector.firebird.metadata.data.FirebirdBlobInfoRegistry;
import org.apache.shardingsphere.database.connector.firebird.metadata.data.FirebirdNonFixedLengthColumnSizeRegistry;

import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.Map;

/**
 * Meta data loader for Firebird.
 */
public final class FirebirdMetaDataLoader implements DialectMetaDataLoader {
    
    @Override
    public Collection<SchemaMetaData> load(final MetaDataLoaderMaterial material) throws SQLException {
        Collection<TableMetaData> tableMetaData = new LinkedList<>();
        for (String each : material.getActualTableNames()) {
            TableMetaDataLoader.load(material.getDataSource(), each, material.getStorageType()).ifPresent(tableMetaData::add);
        }
        loadBlobColumns(material);
        loadNonFixedLengthColumnSizes(material);
        return Collections.singleton(new SchemaMetaData(material.getDefaultSchemaName(), tableMetaData));
    }
    
    private void loadNonFixedLengthColumnSizes(final MetaDataLoaderMaterial material) throws SQLException {
        Map<String, Map<String, Integer>> nonFixedColumnSizes = new FirebirdNonFixedLengthColumnSizeLoader(material).load();
        for (String each : material.getActualTableNames()) {
            Map<String, Integer> tableSizes = nonFixedColumnSizes.getOrDefault(each, Collections.emptyMap());
            FirebirdNonFixedLengthColumnSizeRegistry.refreshTable(material.getDefaultSchemaName(), each, tableSizes);
        }
    }
    
    private void loadBlobColumns(final MetaDataLoaderMaterial material) throws SQLException {
        Map<String, Map<String, Integer>> blobColumns = new FirebirdBlobColumnLoader(material).load();
        for (String each : material.getActualTableNames()) {
            Map<String, Integer> tableColumns = blobColumns.getOrDefault(each, Collections.emptyMap());
            FirebirdBlobInfoRegistry.refreshTable(material.getDefaultSchemaName(), each, tableColumns);
        }
    }
    
    @Override
    public String getDatabaseType() {
        return "Firebird";
    }
}
