{
    Double Commander
    -------------------------------------------------------------------------
    Fast CRC32 calculation algorithm
    http://create.stephan-brumme.com/crc32

    Copyright (C) 2011-2015 Stephan Brumme. All rights reserved.
    Slicing-by-16 contributed by Bulat Ziganshin
    See http://create.stephan-brumme.com/disclaimer.html

    Pascal tranlastion
    Copyright (C) 2015 Alexander Koblov (alexx2000@mail.ru)

    This software is provided 'as-is', without any express or implied warranty.
    In no event will the author be held liable for any damages arising from the
    use of this software. Permission is granted to anyone to use this software
    for any purpose, including commercial applications, and to alter it and
    redistribute it freely, subject to the following restrictions:

      1. The origin of this software must not be misrepresented; you must not
         claim that you wrote the original software.
      2. If you use this software in a product, an acknowledgment in the product
         documentation would be appreciated but is not required.
      3. Altered source versions must be plainly marked as such, and must not be
         misrepresented as being the original software.
}

unit DCcrc32;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;

function crc32_16bytes(const data: PByte; length: Integer; previousCrc32: Cardinal = 0): Cardinal;

implementation
{$R-}{$Q-}

{
const
  /// zlib's CRC32 polynomial
  Polynomial: Cardinal = $EDB88320;

var
  HaveTable: Boolean = False;
  Crc32Lookup: array[0..15, 0..255] of Cardinal;

procedure crc32_init;
var
  i, j: Integer;
  crc: Cardinal;
  slice: Integer;
begin
  if HaveTable then Exit;

  //// same algorithm as crc32_bitwise
  for i := 0 to $FF do
  begin
    crc := Cardinal(i);

    for j := 0 to 7 do
    begin
      if (crc and 1) <> 0 then
        crc := Polynomial xor (crc shr 1)
      else
        crc := (crc shr 1);
    end;

    Crc32Lookup[0][i] := crc;
  end;
  // ... and the following slicing-by-8 algorithm (from Intel):
  // http://www.intel.com/technology/comms/perfnet/download/CRC_generators.pdf
  // http://sourceforge.net/projects/slicing-by-8/
  for i := 0 to $FF do
  begin
    for slice := 1 to 15 do
      Crc32Lookup[slice][i] := (Crc32Lookup[slice - 1][i] shr 8) xor Crc32Lookup[0][Crc32Lookup[slice - 1][i] and $FF];
  end;
  HaveTable:= True;
end;
}

const
  Crc32Lookup: array[0..15, 0..255] of Cardinal = (
    (
      $00000000,$77073096,$EE0E612C,$990951BA,$076DC419,$706AF48F,$E963A535,$9E6495A3,
      $0EDB8832,$79DCB8A4,$E0D5E91E,$97D2D988,$09B64C2B,$7EB17CBD,$E7B82D07,$90BF1D91,
      $1DB71064,$6AB020F2,$F3B97148,$84BE41DE,$1ADAD47D,$6DDDE4EB,$F4D4B551,$83D385C7,
      $136C9856,$646BA8C0,$FD62F97A,$8A65C9EC,$14015C4F,$63066CD9,$FA0F3D63,$8D080DF5,
      $3B6E20C8,$4C69105E,$D56041E4,$A2677172,$3C03E4D1,$4B04D447,$D20D85FD,$A50AB56B,
      $35B5A8FA,$42B2986C,$DBBBC9D6,$ACBCF940,$32D86CE3,$45DF5C75,$DCD60DCF,$ABD13D59,
      $26D930AC,$51DE003A,$C8D75180,$BFD06116,$21B4F4B5,$56B3C423,$CFBA9599,$B8BDA50F,
      $2802B89E,$5F058808,$C60CD9B2,$B10BE924,$2F6F7C87,$58684C11,$C1611DAB,$B6662D3D,
      $76DC4190,$01DB7106,$98D220BC,$EFD5102A,$71B18589,$06B6B51F,$9FBFE4A5,$E8B8D433,
      $7807C9A2,$0F00F934,$9609A88E,$E10E9818,$7F6A0DBB,$086D3D2D,$91646C97,$E6635C01,
      $6B6B51F4,$1C6C6162,$856530D8,$F262004E,$6C0695ED,$1B01A57B,$8208F4C1,$F50FC457,
      $65B0D9C6,$12B7E950,$8BBEB8EA,$FCB9887C,$62DD1DDF,$15DA2D49,$8CD37CF3,$FBD44C65,
      $4DB26158,$3AB551CE,$A3BC0074,$D4BB30E2,$4ADFA541,$3DD895D7,$A4D1C46D,$D3D6F4FB,
      $4369E96A,$346ED9FC,$AD678846,$DA60B8D0,$44042D73,$33031DE5,$AA0A4C5F,$DD0D7CC9,
      $5005713C,$270241AA,$BE0B1010,$C90C2086,$5768B525,$206F85B3,$B966D409,$CE61E49F,
      $5EDEF90E,$29D9C998,$B0D09822,$C7D7A8B4,$59B33D17,$2EB40D81,$B7BD5C3B,$C0BA6CAD,
      $EDB88320,$9ABFB3B6,$03B6E20C,$74B1D29A,$EAD54739,$9DD277AF,$04DB2615,$73DC1683,
      $E3630B12,$94643B84,$0D6D6A3E,$7A6A5AA8,$E40ECF0B,$9309FF9D,$0A00AE27,$7D079EB1,
      $F00F9344,$8708A3D2,$1E01F268,$6906C2FE,$F762575D,$806567CB,$196C3671,$6E6B06E7,
      $FED41B76,$89D32BE0,$10DA7A5A,$67DD4ACC,$F9B9DF6F,$8EBEEFF9,$17B7BE43,$60B08ED5,
      $D6D6A3E8,$A1D1937E,$38D8C2C4,$4FDFF252,$D1BB67F1,$A6BC5767,$3FB506DD,$48B2364B,
      $D80D2BDA,$AF0A1B4C,$36034AF6,$41047A60,$DF60EFC3,$A867DF55,$316E8EEF,$4669BE79,
      $CB61B38C,$BC66831A,$256FD2A0,$5268E236,$CC0C7795,$BB0B4703,$220216B9,$5505262F,
      $C5BA3BBE,$B2BD0B28,$2BB45A92,$5CB36A04,$C2D7FFA7,$B5D0CF31,$2CD99E8B,$5BDEAE1D,
      $9B64C2B0,$EC63F226,$756AA39C,$026D930A,$9C0906A9,$EB0E363F,$72076785,$05005713,
      $95BF4A82,$E2B87A14,$7BB12BAE,$0CB61B38,$92D28E9B,$E5D5BE0D,$7CDCEFB7,$0BDBDF21,
      $86D3D2D4,$F1D4E242,$68DDB3F8,$1FDA836E,$81BE16CD,$F6B9265B,$6FB077E1,$18B74777,
      $88085AE6,$FF0F6A70,$66063BCA,$11010B5C,$8F659EFF,$F862AE69,$616BFFD3,$166CCF45,
      $A00AE278,$D70DD2EE,$4E048354,$3903B3C2,$A7672661,$D06016F7,$4969474D,$3E6E77DB,
      $AED16A4A,$D9D65ADC,$40DF0B66,$37D83BF0,$A9BCAE53,$DEBB9EC5,$47B2CF7F,$30B5FFE9,
      $BDBDF21C,$CABAC28A,$53B39330,$24B4A3A6,$BAD03605,$CDD70693,$54DE5729,$23D967BF,
      $B3667A2E,$C4614AB8,$5D681B02,$2A6F2B94,$B40BBE37,$C30C8EA1,$5A05DF1B,$2D02EF8D
    )
    ,(
      $00000000,$191B3141,$32366282,$2B2D53C3,$646CC504,$7D77F445,$565AA786,$4F4196C7,
      $C8D98A08,$D1C2BB49,$FAEFE88A,$E3F4D9CB,$ACB54F0C,$B5AE7E4D,$9E832D8E,$87981CCF,
      $4AC21251,$53D92310,$78F470D3,$61EF4192,$2EAED755,$37B5E614,$1C98B5D7,$05838496,
      $821B9859,$9B00A918,$B02DFADB,$A936CB9A,$E6775D5D,$FF6C6C1C,$D4413FDF,$CD5A0E9E,
      $958424A2,$8C9F15E3,$A7B24620,$BEA97761,$F1E8E1A6,$E8F3D0E7,$C3DE8324,$DAC5B265,
      $5D5DAEAA,$44469FEB,$6F6BCC28,$7670FD69,$39316BAE,$202A5AEF,$0B07092C,$121C386D,
      $DF4636F3,$C65D07B2,$ED705471,$F46B6530,$BB2AF3F7,$A231C2B6,$891C9175,$9007A034,
      $179FBCFB,$0E848DBA,$25A9DE79,$3CB2EF38,$73F379FF,$6AE848BE,$41C51B7D,$58DE2A3C,
      $F0794F05,$E9627E44,$C24F2D87,$DB541CC6,$94158A01,$8D0EBB40,$A623E883,$BF38D9C2,
      $38A0C50D,$21BBF44C,$0A96A78F,$138D96CE,$5CCC0009,$45D73148,$6EFA628B,$77E153CA,
      $BABB5D54,$A3A06C15,$888D3FD6,$91960E97,$DED79850,$C7CCA911,$ECE1FAD2,$F5FACB93,
      $7262D75C,$6B79E61D,$4054B5DE,$594F849F,$160E1258,$0F152319,$243870DA,$3D23419B,
      $65FD6BA7,$7CE65AE6,$57CB0925,$4ED03864,$0191AEA3,$188A9FE2,$33A7CC21,$2ABCFD60,
      $AD24E1AF,$B43FD0EE,$9F12832D,$8609B26C,$C94824AB,$D05315EA,$FB7E4629,$E2657768,
      $2F3F79F6,$362448B7,$1D091B74,$04122A35,$4B53BCF2,$52488DB3,$7965DE70,$607EEF31,
      $E7E6F3FE,$FEFDC2BF,$D5D0917C,$CCCBA03D,$838A36FA,$9A9107BB,$B1BC5478,$A8A76539,
      $3B83984B,$2298A90A,$09B5FAC9,$10AECB88,$5FEF5D4F,$46F46C0E,$6DD93FCD,$74C20E8C,
      $F35A1243,$EA412302,$C16C70C1,$D8774180,$9736D747,$8E2DE606,$A500B5C5,$BC1B8484,
      $71418A1A,$685ABB5B,$4377E898,$5A6CD9D9,$152D4F1E,$0C367E5F,$271B2D9C,$3E001CDD,
      $B9980012,$A0833153,$8BAE6290,$92B553D1,$DDF4C516,$C4EFF457,$EFC2A794,$F6D996D5,
      $AE07BCE9,$B71C8DA8,$9C31DE6B,$852AEF2A,$CA6B79ED,$D37048AC,$F85D1B6F,$E1462A2E,
      $66DE36E1,$7FC507A0,$54E85463,$4DF36522,$02B2F3E5,$1BA9C2A4,$30849167,$299FA026,
      $E4C5AEB8,$FDDE9FF9,$D6F3CC3A,$CFE8FD7B,$80A96BBC,$99B25AFD,$B29F093E,$AB84387F,
      $2C1C24B0,$350715F1,$1E2A4632,$07317773,$4870E1B4,$516BD0F5,$7A468336,$635DB277,
      $CBFAD74E,$D2E1E60F,$F9CCB5CC,$E0D7848D,$AF96124A,$B68D230B,$9DA070C8,$84BB4189,
      $03235D46,$1A386C07,$31153FC4,$280E0E85,$674F9842,$7E54A903,$5579FAC0,$4C62CB81,
      $8138C51F,$9823F45E,$B30EA79D,$AA1596DC,$E554001B,$FC4F315A,$D7626299,$CE7953D8,
      $49E14F17,$50FA7E56,$7BD72D95,$62CC1CD4,$2D8D8A13,$3496BB52,$1FBBE891,$06A0D9D0,
      $5E7EF3EC,$4765C2AD,$6C48916E,$7553A02F,$3A1236E8,$230907A9,$0824546A,$113F652B,
      $96A779E4,$8FBC48A5,$A4911B66,$BD8A2A27,$F2CBBCE0,$EBD08DA1,$C0FDDE62,$D9E6EF23,
      $14BCE1BD,$0DA7D0FC,$268A833F,$3F91B27E,$70D024B9,$69CB15F8,$42E6463B,$5BFD777A,
      $DC656BB5,$C57E5AF4,$EE530937,$F7483876,$B809AEB1,$A1129FF0,$8A3FCC33,$9324FD72
    ),
    (
      $00000000,$01C26A37,$0384D46E,$0246BE59,$0709A8DC,$06CBC2EB,$048D7CB2,$054F1685,
      $0E1351B8,$0FD13B8F,$0D9785D6,$0C55EFE1,$091AF964,$08D89353,$0A9E2D0A,$0B5C473D,
      $1C26A370,$1DE4C947,$1FA2771E,$1E601D29,$1B2F0BAC,$1AED619B,$18ABDFC2,$1969B5F5,
      $1235F2C8,$13F798FF,$11B126A6,$10734C91,$153C5A14,$14FE3023,$16B88E7A,$177AE44D,
      $384D46E0,$398F2CD7,$3BC9928E,$3A0BF8B9,$3F44EE3C,$3E86840B,$3CC03A52,$3D025065,
      $365E1758,$379C7D6F,$35DAC336,$3418A901,$3157BF84,$3095D5B3,$32D36BEA,$331101DD,
      $246BE590,$25A98FA7,$27EF31FE,$262D5BC9,$23624D4C,$22A0277B,$20E69922,$2124F315,
      $2A78B428,$2BBADE1F,$29FC6046,$283E0A71,$2D711CF4,$2CB376C3,$2EF5C89A,$2F37A2AD,
      $709A8DC0,$7158E7F7,$731E59AE,$72DC3399,$7793251C,$76514F2B,$7417F172,$75D59B45,
      $7E89DC78,$7F4BB64F,$7D0D0816,$7CCF6221,$798074A4,$78421E93,$7A04A0CA,$7BC6CAFD,
      $6CBC2EB0,$6D7E4487,$6F38FADE,$6EFA90E9,$6BB5866C,$6A77EC5B,$68315202,$69F33835,
      $62AF7F08,$636D153F,$612BAB66,$60E9C151,$65A6D7D4,$6464BDE3,$662203BA,$67E0698D,
      $48D7CB20,$4915A117,$4B531F4E,$4A917579,$4FDE63FC,$4E1C09CB,$4C5AB792,$4D98DDA5,
      $46C49A98,$4706F0AF,$45404EF6,$448224C1,$41CD3244,$400F5873,$4249E62A,$438B8C1D,
      $54F16850,$55330267,$5775BC3E,$56B7D609,$53F8C08C,$523AAABB,$507C14E2,$51BE7ED5,
      $5AE239E8,$5B2053DF,$5966ED86,$58A487B1,$5DEB9134,$5C29FB03,$5E6F455A,$5FAD2F6D,
      $E1351B80,$E0F771B7,$E2B1CFEE,$E373A5D9,$E63CB35C,$E7FED96B,$E5B86732,$E47A0D05,
      $EF264A38,$EEE4200F,$ECA29E56,$ED60F461,$E82FE2E4,$E9ED88D3,$EBAB368A,$EA695CBD,
      $FD13B8F0,$FCD1D2C7,$FE976C9E,$FF5506A9,$FA1A102C,$FBD87A1B,$F99EC442,$F85CAE75,
      $F300E948,$F2C2837F,$F0843D26,$F1465711,$F4094194,$F5CB2BA3,$F78D95FA,$F64FFFCD,
      $D9785D60,$D8BA3757,$DAFC890E,$DB3EE339,$DE71F5BC,$DFB39F8B,$DDF521D2,$DC374BE5,
      $D76B0CD8,$D6A966EF,$D4EFD8B6,$D52DB281,$D062A404,$D1A0CE33,$D3E6706A,$D2241A5D,
      $C55EFE10,$C49C9427,$C6DA2A7E,$C7184049,$C25756CC,$C3953CFB,$C1D382A2,$C011E895,
      $CB4DAFA8,$CA8FC59F,$C8C97BC6,$C90B11F1,$CC440774,$CD866D43,$CFC0D31A,$CE02B92D,
      $91AF9640,$906DFC77,$922B422E,$93E92819,$96A63E9C,$976454AB,$9522EAF2,$94E080C5,
      $9FBCC7F8,$9E7EADCF,$9C381396,$9DFA79A1,$98B56F24,$99770513,$9B31BB4A,$9AF3D17D,
      $8D893530,$8C4B5F07,$8E0DE15E,$8FCF8B69,$8A809DEC,$8B42F7DB,$89044982,$88C623B5,
      $839A6488,$82580EBF,$801EB0E6,$81DCDAD1,$8493CC54,$8551A663,$8717183A,$86D5720D,
      $A9E2D0A0,$A820BA97,$AA6604CE,$ABA46EF9,$AEEB787C,$AF29124B,$AD6FAC12,$ACADC625,
      $A7F18118,$A633EB2F,$A4755576,$A5B73F41,$A0F829C4,$A13A43F3,$A37CFDAA,$A2BE979D,
      $B5C473D0,$B40619E7,$B640A7BE,$B782CD89,$B2CDDB0C,$B30FB13B,$B1490F62,$B08B6555,
      $BBD72268,$BA15485F,$B853F606,$B9919C31,$BCDE8AB4,$BD1CE083,$BF5A5EDA,$BE9834ED
    ),
    (
      $00000000,$B8BC6765,$AA09C88B,$12B5AFEE,$8F629757,$37DEF032,$256B5FDC,$9DD738B9,
      $C5B428EF,$7D084F8A,$6FBDE064,$D7018701,$4AD6BFB8,$F26AD8DD,$E0DF7733,$58631056,
      $5019579F,$E8A530FA,$FA109F14,$42ACF871,$DF7BC0C8,$67C7A7AD,$75720843,$CDCE6F26,
      $95AD7F70,$2D111815,$3FA4B7FB,$8718D09E,$1ACFE827,$A2738F42,$B0C620AC,$087A47C9,
      $A032AF3E,$188EC85B,$0A3B67B5,$B28700D0,$2F503869,$97EC5F0C,$8559F0E2,$3DE59787,
      $658687D1,$DD3AE0B4,$CF8F4F5A,$7733283F,$EAE41086,$525877E3,$40EDD80D,$F851BF68,
      $F02BF8A1,$48979FC4,$5A22302A,$E29E574F,$7F496FF6,$C7F50893,$D540A77D,$6DFCC018,
      $359FD04E,$8D23B72B,$9F9618C5,$272A7FA0,$BAFD4719,$0241207C,$10F48F92,$A848E8F7,
      $9B14583D,$23A83F58,$311D90B6,$89A1F7D3,$1476CF6A,$ACCAA80F,$BE7F07E1,$06C36084,
      $5EA070D2,$E61C17B7,$F4A9B859,$4C15DF3C,$D1C2E785,$697E80E0,$7BCB2F0E,$C377486B,
      $CB0D0FA2,$73B168C7,$6104C729,$D9B8A04C,$446F98F5,$FCD3FF90,$EE66507E,$56DA371B,
      $0EB9274D,$B6054028,$A4B0EFC6,$1C0C88A3,$81DBB01A,$3967D77F,$2BD27891,$936E1FF4,
      $3B26F703,$839A9066,$912F3F88,$299358ED,$B4446054,$0CF80731,$1E4DA8DF,$A6F1CFBA,
      $FE92DFEC,$462EB889,$549B1767,$EC277002,$71F048BB,$C94C2FDE,$DBF98030,$6345E755,
      $6B3FA09C,$D383C7F9,$C1366817,$798A0F72,$E45D37CB,$5CE150AE,$4E54FF40,$F6E89825,
      $AE8B8873,$1637EF16,$048240F8,$BC3E279D,$21E91F24,$99557841,$8BE0D7AF,$335CB0CA,
      $ED59B63B,$55E5D15E,$47507EB0,$FFEC19D5,$623B216C,$DA874609,$C832E9E7,$708E8E82,
      $28ED9ED4,$9051F9B1,$82E4565F,$3A58313A,$A78F0983,$1F336EE6,$0D86C108,$B53AA66D,
      $BD40E1A4,$05FC86C1,$1749292F,$AFF54E4A,$322276F3,$8A9E1196,$982BBE78,$2097D91D,
      $78F4C94B,$C048AE2E,$D2FD01C0,$6A4166A5,$F7965E1C,$4F2A3979,$5D9F9697,$E523F1F2,
      $4D6B1905,$F5D77E60,$E762D18E,$5FDEB6EB,$C2098E52,$7AB5E937,$680046D9,$D0BC21BC,
      $88DF31EA,$3063568F,$22D6F961,$9A6A9E04,$07BDA6BD,$BF01C1D8,$ADB46E36,$15080953,
      $1D724E9A,$A5CE29FF,$B77B8611,$0FC7E174,$9210D9CD,$2AACBEA8,$38191146,$80A57623,
      $D8C66675,$607A0110,$72CFAEFE,$CA73C99B,$57A4F122,$EF189647,$FDAD39A9,$45115ECC,
      $764DEE06,$CEF18963,$DC44268D,$64F841E8,$F92F7951,$41931E34,$5326B1DA,$EB9AD6BF,
      $B3F9C6E9,$0B45A18C,$19F00E62,$A14C6907,$3C9B51BE,$842736DB,$96929935,$2E2EFE50,
      $2654B999,$9EE8DEFC,$8C5D7112,$34E11677,$A9362ECE,$118A49AB,$033FE645,$BB838120,
      $E3E09176,$5B5CF613,$49E959FD,$F1553E98,$6C820621,$D43E6144,$C68BCEAA,$7E37A9CF,
      $D67F4138,$6EC3265D,$7C7689B3,$C4CAEED6,$591DD66F,$E1A1B10A,$F3141EE4,$4BA87981,
      $13CB69D7,$AB770EB2,$B9C2A15C,$017EC639,$9CA9FE80,$241599E5,$36A0360B,$8E1C516E,
      $866616A7,$3EDA71C2,$2C6FDE2C,$94D3B949,$090481F0,$B1B8E695,$A30D497B,$1BB12E1E,
      $43D23E48,$FB6E592D,$E9DBF6C3,$516791A6,$CCB0A91F,$740CCE7A,$66B96194,$DE0506F1
    )
    ,(
      $00000000,$3D6029B0,$7AC05360,$47A07AD0,$F580A6C0,$C8E08F70,$8F40F5A0,$B220DC10,
      $30704BC1,$0D106271,$4AB018A1,$77D03111,$C5F0ED01,$F890C4B1,$BF30BE61,$825097D1,
      $60E09782,$5D80BE32,$1A20C4E2,$2740ED52,$95603142,$A80018F2,$EFA06222,$D2C04B92,
      $5090DC43,$6DF0F5F3,$2A508F23,$1730A693,$A5107A83,$98705333,$DFD029E3,$E2B00053,
      $C1C12F04,$FCA106B4,$BB017C64,$866155D4,$344189C4,$0921A074,$4E81DAA4,$73E1F314,
      $F1B164C5,$CCD14D75,$8B7137A5,$B6111E15,$0431C205,$3951EBB5,$7EF19165,$4391B8D5,
      $A121B886,$9C419136,$DBE1EBE6,$E681C256,$54A11E46,$69C137F6,$2E614D26,$13016496,
      $9151F347,$AC31DAF7,$EB91A027,$D6F18997,$64D15587,$59B17C37,$1E1106E7,$23712F57,
      $58F35849,$659371F9,$22330B29,$1F532299,$AD73FE89,$9013D739,$D7B3ADE9,$EAD38459,
      $68831388,$55E33A38,$124340E8,$2F236958,$9D03B548,$A0639CF8,$E7C3E628,$DAA3CF98,
      $3813CFCB,$0573E67B,$42D39CAB,$7FB3B51B,$CD93690B,$F0F340BB,$B7533A6B,$8A3313DB,
      $0863840A,$3503ADBA,$72A3D76A,$4FC3FEDA,$FDE322CA,$C0830B7A,$872371AA,$BA43581A,
      $9932774D,$A4525EFD,$E3F2242D,$DE920D9D,$6CB2D18D,$51D2F83D,$167282ED,$2B12AB5D,
      $A9423C8C,$9422153C,$D3826FEC,$EEE2465C,$5CC29A4C,$61A2B3FC,$2602C92C,$1B62E09C,
      $F9D2E0CF,$C4B2C97F,$8312B3AF,$BE729A1F,$0C52460F,$31326FBF,$7692156F,$4BF23CDF,
      $C9A2AB0E,$F4C282BE,$B362F86E,$8E02D1DE,$3C220DCE,$0142247E,$46E25EAE,$7B82771E,
      $B1E6B092,$8C869922,$CB26E3F2,$F646CA42,$44661652,$79063FE2,$3EA64532,$03C66C82,
      $8196FB53,$BCF6D2E3,$FB56A833,$C6368183,$74165D93,$49767423,$0ED60EF3,$33B62743,
      $D1062710,$EC660EA0,$ABC67470,$96A65DC0,$248681D0,$19E6A860,$5E46D2B0,$6326FB00,
      $E1766CD1,$DC164561,$9BB63FB1,$A6D61601,$14F6CA11,$2996E3A1,$6E369971,$5356B0C1,
      $70279F96,$4D47B626,$0AE7CCF6,$3787E546,$85A73956,$B8C710E6,$FF676A36,$C2074386,
      $4057D457,$7D37FDE7,$3A978737,$07F7AE87,$B5D77297,$88B75B27,$CF1721F7,$F2770847,
      $10C70814,$2DA721A4,$6A075B74,$576772C4,$E547AED4,$D8278764,$9F87FDB4,$A2E7D404,
      $20B743D5,$1DD76A65,$5A7710B5,$67173905,$D537E515,$E857CCA5,$AFF7B675,$92979FC5,
      $E915E8DB,$D475C16B,$93D5BBBB,$AEB5920B,$1C954E1B,$21F567AB,$66551D7B,$5B3534CB,
      $D965A31A,$E4058AAA,$A3A5F07A,$9EC5D9CA,$2CE505DA,$11852C6A,$562556BA,$6B457F0A,
      $89F57F59,$B49556E9,$F3352C39,$CE550589,$7C75D999,$4115F029,$06B58AF9,$3BD5A349,
      $B9853498,$84E51D28,$C34567F8,$FE254E48,$4C059258,$7165BBE8,$36C5C138,$0BA5E888,
      $28D4C7DF,$15B4EE6F,$521494BF,$6F74BD0F,$DD54611F,$E03448AF,$A794327F,$9AF41BCF,
      $18A48C1E,$25C4A5AE,$6264DF7E,$5F04F6CE,$ED242ADE,$D044036E,$97E479BE,$AA84500E,
      $4834505D,$755479ED,$32F4033D,$0F942A8D,$BDB4F69D,$80D4DF2D,$C774A5FD,$FA148C4D,
      $78441B9C,$4524322C,$028448FC,$3FE4614C,$8DC4BD5C,$B0A494EC,$F704EE3C,$CA64C78C
    ),
    (
      $00000000,$CB5CD3A5,$4DC8A10B,$869472AE,$9B914216,$50CD91B3,$D659E31D,$1D0530B8,
      $EC53826D,$270F51C8,$A19B2366,$6AC7F0C3,$77C2C07B,$BC9E13DE,$3A0A6170,$F156B2D5,
      $03D6029B,$C88AD13E,$4E1EA390,$85427035,$9847408D,$531B9328,$D58FE186,$1ED33223,
      $EF8580F6,$24D95353,$A24D21FD,$6911F258,$7414C2E0,$BF481145,$39DC63EB,$F280B04E,
      $07AC0536,$CCF0D693,$4A64A43D,$81387798,$9C3D4720,$57619485,$D1F5E62B,$1AA9358E,
      $EBFF875B,$20A354FE,$A6372650,$6D6BF5F5,$706EC54D,$BB3216E8,$3DA66446,$F6FAB7E3,
      $047A07AD,$CF26D408,$49B2A6A6,$82EE7503,$9FEB45BB,$54B7961E,$D223E4B0,$197F3715,
      $E82985C0,$23755665,$A5E124CB,$6EBDF76E,$73B8C7D6,$B8E41473,$3E7066DD,$F52CB578,
      $0F580A6C,$C404D9C9,$4290AB67,$89CC78C2,$94C9487A,$5F959BDF,$D901E971,$125D3AD4,
      $E30B8801,$28575BA4,$AEC3290A,$659FFAAF,$789ACA17,$B3C619B2,$35526B1C,$FE0EB8B9,
      $0C8E08F7,$C7D2DB52,$4146A9FC,$8A1A7A59,$971F4AE1,$5C439944,$DAD7EBEA,$118B384F,
      $E0DD8A9A,$2B81593F,$AD152B91,$6649F834,$7B4CC88C,$B0101B29,$36846987,$FDD8BA22,
      $08F40F5A,$C3A8DCFF,$453CAE51,$8E607DF4,$93654D4C,$58399EE9,$DEADEC47,$15F13FE2,
      $E4A78D37,$2FFB5E92,$A96F2C3C,$6233FF99,$7F36CF21,$B46A1C84,$32FE6E2A,$F9A2BD8F,
      $0B220DC1,$C07EDE64,$46EAACCA,$8DB67F6F,$90B34FD7,$5BEF9C72,$DD7BEEDC,$16273D79,
      $E7718FAC,$2C2D5C09,$AAB92EA7,$61E5FD02,$7CE0CDBA,$B7BC1E1F,$31286CB1,$FA74BF14,
      $1EB014D8,$D5ECC77D,$5378B5D3,$98246676,$852156CE,$4E7D856B,$C8E9F7C5,$03B52460,
      $F2E396B5,$39BF4510,$BF2B37BE,$7477E41B,$6972D4A3,$A22E0706,$24BA75A8,$EFE6A60D,
      $1D661643,$D63AC5E6,$50AEB748,$9BF264ED,$86F75455,$4DAB87F0,$CB3FF55E,$006326FB,
      $F135942E,$3A69478B,$BCFD3525,$77A1E680,$6AA4D638,$A1F8059D,$276C7733,$EC30A496,
      $191C11EE,$D240C24B,$54D4B0E5,$9F886340,$828D53F8,$49D1805D,$CF45F2F3,$04192156,
      $F54F9383,$3E134026,$B8873288,$73DBE12D,$6EDED195,$A5820230,$2316709E,$E84AA33B,
      $1ACA1375,$D196C0D0,$5702B27E,$9C5E61DB,$815B5163,$4A0782C6,$CC93F068,$07CF23CD,
      $F6999118,$3DC542BD,$BB513013,$700DE3B6,$6D08D30E,$A65400AB,$20C07205,$EB9CA1A0,
      $11E81EB4,$DAB4CD11,$5C20BFBF,$977C6C1A,$8A795CA2,$41258F07,$C7B1FDA9,$0CED2E0C,
      $FDBB9CD9,$36E74F7C,$B0733DD2,$7B2FEE77,$662ADECF,$AD760D6A,$2BE27FC4,$E0BEAC61,
      $123E1C2F,$D962CF8A,$5FF6BD24,$94AA6E81,$89AF5E39,$42F38D9C,$C467FF32,$0F3B2C97,
      $FE6D9E42,$35314DE7,$B3A53F49,$78F9ECEC,$65FCDC54,$AEA00FF1,$28347D5F,$E368AEFA,
      $16441B82,$DD18C827,$5B8CBA89,$90D0692C,$8DD55994,$46898A31,$C01DF89F,$0B412B3A,
      $FA1799EF,$314B4A4A,$B7DF38E4,$7C83EB41,$6186DBF9,$AADA085C,$2C4E7AF2,$E712A957,
      $15921919,$DECECABC,$585AB812,$93066BB7,$8E035B0F,$455F88AA,$C3CBFA04,$089729A1,
      $F9C19B74,$329D48D1,$B4093A7F,$7F55E9DA,$6250D962,$A90C0AC7,$2F987869,$E4C4ABCC
    ),
    (
      $00000000,$A6770BB4,$979F1129,$31E81A9D,$F44F2413,$52382FA7,$63D0353A,$C5A73E8E,
      $33EF4E67,$959845D3,$A4705F4E,$020754FA,$C7A06A74,$61D761C0,$503F7B5D,$F64870E9,
      $67DE9CCE,$C1A9977A,$F0418DE7,$56368653,$9391B8DD,$35E6B369,$040EA9F4,$A279A240,
      $5431D2A9,$F246D91D,$C3AEC380,$65D9C834,$A07EF6BA,$0609FD0E,$37E1E793,$9196EC27,
      $CFBD399C,$69CA3228,$582228B5,$FE552301,$3BF21D8F,$9D85163B,$AC6D0CA6,$0A1A0712,
      $FC5277FB,$5A257C4F,$6BCD66D2,$CDBA6D66,$081D53E8,$AE6A585C,$9F8242C1,$39F54975,
      $A863A552,$0E14AEE6,$3FFCB47B,$998BBFCF,$5C2C8141,$FA5B8AF5,$CBB39068,$6DC49BDC,
      $9B8CEB35,$3DFBE081,$0C13FA1C,$AA64F1A8,$6FC3CF26,$C9B4C492,$F85CDE0F,$5E2BD5BB,
      $440B7579,$E27C7ECD,$D3946450,$75E36FE4,$B044516A,$16335ADE,$27DB4043,$81AC4BF7,
      $77E43B1E,$D19330AA,$E07B2A37,$460C2183,$83AB1F0D,$25DC14B9,$14340E24,$B2430590,
      $23D5E9B7,$85A2E203,$B44AF89E,$123DF32A,$D79ACDA4,$71EDC610,$4005DC8D,$E672D739,
      $103AA7D0,$B64DAC64,$87A5B6F9,$21D2BD4D,$E47583C3,$42028877,$73EA92EA,$D59D995E,
      $8BB64CE5,$2DC14751,$1C295DCC,$BA5E5678,$7FF968F6,$D98E6342,$E86679DF,$4E11726B,
      $B8590282,$1E2E0936,$2FC613AB,$89B1181F,$4C162691,$EA612D25,$DB8937B8,$7DFE3C0C,
      $EC68D02B,$4A1FDB9F,$7BF7C102,$DD80CAB6,$1827F438,$BE50FF8C,$8FB8E511,$29CFEEA5,
      $DF879E4C,$79F095F8,$48188F65,$EE6F84D1,$2BC8BA5F,$8DBFB1EB,$BC57AB76,$1A20A0C2,
      $8816EAF2,$2E61E146,$1F89FBDB,$B9FEF06F,$7C59CEE1,$DA2EC555,$EBC6DFC8,$4DB1D47C,
      $BBF9A495,$1D8EAF21,$2C66B5BC,$8A11BE08,$4FB68086,$E9C18B32,$D82991AF,$7E5E9A1B,
      $EFC8763C,$49BF7D88,$78576715,$DE206CA1,$1B87522F,$BDF0599B,$8C184306,$2A6F48B2,
      $DC27385B,$7A5033EF,$4BB82972,$EDCF22C6,$28681C48,$8E1F17FC,$BFF70D61,$198006D5,
      $47ABD36E,$E1DCD8DA,$D034C247,$7643C9F3,$B3E4F77D,$1593FCC9,$247BE654,$820CEDE0,
      $74449D09,$D23396BD,$E3DB8C20,$45AC8794,$800BB91A,$267CB2AE,$1794A833,$B1E3A387,
      $20754FA0,$86024414,$B7EA5E89,$119D553D,$D43A6BB3,$724D6007,$43A57A9A,$E5D2712E,
      $139A01C7,$B5ED0A73,$840510EE,$22721B5A,$E7D525D4,$41A22E60,$704A34FD,$D63D3F49,
      $CC1D9F8B,$6A6A943F,$5B828EA2,$FDF58516,$3852BB98,$9E25B02C,$AFCDAAB1,$09BAA105,
      $FFF2D1EC,$5985DA58,$686DC0C5,$CE1ACB71,$0BBDF5FF,$ADCAFE4B,$9C22E4D6,$3A55EF62,
      $ABC30345,$0DB408F1,$3C5C126C,$9A2B19D8,$5F8C2756,$F9FB2CE2,$C813367F,$6E643DCB,
      $982C4D22,$3E5B4696,$0FB35C0B,$A9C457BF,$6C636931,$CA146285,$FBFC7818,$5D8B73AC,
      $03A0A617,$A5D7ADA3,$943FB73E,$3248BC8A,$F7EF8204,$519889B0,$6070932D,$C6079899,
      $304FE870,$9638E3C4,$A7D0F959,$01A7F2ED,$C400CC63,$6277C7D7,$539FDD4A,$F5E8D6FE,
      $647E3AD9,$C209316D,$F3E12BF0,$55962044,$90311ECA,$3646157E,$07AE0FE3,$A1D90457,
      $579174BE,$F1E67F0A,$C00E6597,$66796E23,$A3DE50AD,$05A95B19,$34414184,$92364A30
    ),
    (
      $00000000,$CCAA009E,$4225077D,$8E8F07E3,$844A0EFA,$48E00E64,$C66F0987,$0AC50919,
      $D3E51BB5,$1F4F1B2B,$91C01CC8,$5D6A1C56,$57AF154F,$9B0515D1,$158A1232,$D92012AC,
      $7CBB312B,$B01131B5,$3E9E3656,$F23436C8,$F8F13FD1,$345B3F4F,$BAD438AC,$767E3832,
      $AF5E2A9E,$63F42A00,$ED7B2DE3,$21D12D7D,$2B142464,$E7BE24FA,$69312319,$A59B2387,
      $F9766256,$35DC62C8,$BB53652B,$77F965B5,$7D3C6CAC,$B1966C32,$3F196BD1,$F3B36B4F,
      $2A9379E3,$E639797D,$68B67E9E,$A41C7E00,$AED97719,$62737787,$ECFC7064,$205670FA,
      $85CD537D,$496753E3,$C7E85400,$0B42549E,$01875D87,$CD2D5D19,$43A25AFA,$8F085A64,
      $562848C8,$9A824856,$140D4FB5,$D8A74F2B,$D2624632,$1EC846AC,$9047414F,$5CED41D1,
      $299DC2ED,$E537C273,$6BB8C590,$A712C50E,$ADD7CC17,$617DCC89,$EFF2CB6A,$2358CBF4,
      $FA78D958,$36D2D9C6,$B85DDE25,$74F7DEBB,$7E32D7A2,$B298D73C,$3C17D0DF,$F0BDD041,
      $5526F3C6,$998CF358,$1703F4BB,$DBA9F425,$D16CFD3C,$1DC6FDA2,$9349FA41,$5FE3FADF,
      $86C3E873,$4A69E8ED,$C4E6EF0E,$084CEF90,$0289E689,$CE23E617,$40ACE1F4,$8C06E16A,
      $D0EBA0BB,$1C41A025,$92CEA7C6,$5E64A758,$54A1AE41,$980BAEDF,$1684A93C,$DA2EA9A2,
      $030EBB0E,$CFA4BB90,$412BBC73,$8D81BCED,$8744B5F4,$4BEEB56A,$C561B289,$09CBB217,
      $AC509190,$60FA910E,$EE7596ED,$22DF9673,$281A9F6A,$E4B09FF4,$6A3F9817,$A6959889,
      $7FB58A25,$B31F8ABB,$3D908D58,$F13A8DC6,$FBFF84DF,$37558441,$B9DA83A2,$7570833C,
      $533B85DA,$9F918544,$111E82A7,$DDB48239,$D7718B20,$1BDB8BBE,$95548C5D,$59FE8CC3,
      $80DE9E6F,$4C749EF1,$C2FB9912,$0E51998C,$04949095,$C83E900B,$46B197E8,$8A1B9776,
      $2F80B4F1,$E32AB46F,$6DA5B38C,$A10FB312,$ABCABA0B,$6760BA95,$E9EFBD76,$2545BDE8,
      $FC65AF44,$30CFAFDA,$BE40A839,$72EAA8A7,$782FA1BE,$B485A120,$3A0AA6C3,$F6A0A65D,
      $AA4DE78C,$66E7E712,$E868E0F1,$24C2E06F,$2E07E976,$E2ADE9E8,$6C22EE0B,$A088EE95,
      $79A8FC39,$B502FCA7,$3B8DFB44,$F727FBDA,$FDE2F2C3,$3148F25D,$BFC7F5BE,$736DF520,
      $D6F6D6A7,$1A5CD639,$94D3D1DA,$5879D144,$52BCD85D,$9E16D8C3,$1099DF20,$DC33DFBE,
      $0513CD12,$C9B9CD8C,$4736CA6F,$8B9CCAF1,$8159C3E8,$4DF3C376,$C37CC495,$0FD6C40B,
      $7AA64737,$B60C47A9,$3883404A,$F42940D4,$FEEC49CD,$32464953,$BCC94EB0,$70634E2E,
      $A9435C82,$65E95C1C,$EB665BFF,$27CC5B61,$2D095278,$E1A352E6,$6F2C5505,$A386559B,
      $061D761C,$CAB77682,$44387161,$889271FF,$825778E6,$4EFD7878,$C0727F9B,$0CD87F05,
      $D5F86DA9,$19526D37,$97DD6AD4,$5B776A4A,$51B26353,$9D1863CD,$1397642E,$DF3D64B0,
      $83D02561,$4F7A25FF,$C1F5221C,$0D5F2282,$079A2B9B,$CB302B05,$45BF2CE6,$89152C78,
      $50353ED4,$9C9F3E4A,$121039A9,$DEBA3937,$D47F302E,$18D530B0,$965A3753,$5AF037CD,
      $FF6B144A,$33C114D4,$BD4E1337,$71E413A9,$7B211AB0,$B78B1A2E,$39041DCD,$F5AE1D53,
      $2C8E0FFF,$E0240F61,$6EAB0882,$A201081C,$A8C40105,$646E019B,$EAE10678,$264B06E6
    )
    ,(
      $00000000,$177B1443,$2EF62886,$398D3CC5,$5DEC510C,$4A97454F,$731A798A,$64616DC9,
      $BBD8A218,$ACA3B65B,$952E8A9E,$82559EDD,$E634F314,$F14FE757,$C8C2DB92,$DFB9CFD1,
      $ACC04271,$BBBB5632,$82366AF7,$954D7EB4,$F12C137D,$E657073E,$DFDA3BFB,$C8A12FB8,
      $1718E069,$0063F42A,$39EEC8EF,$2E95DCAC,$4AF4B165,$5D8FA526,$640299E3,$73798DA0,
      $82F182A3,$958A96E0,$AC07AA25,$BB7CBE66,$DF1DD3AF,$C866C7EC,$F1EBFB29,$E690EF6A,
      $392920BB,$2E5234F8,$17DF083D,$00A41C7E,$64C571B7,$73BE65F4,$4A335931,$5D484D72,
      $2E31C0D2,$394AD491,$00C7E854,$17BCFC17,$73DD91DE,$64A6859D,$5D2BB958,$4A50AD1B,
      $95E962CA,$82927689,$BB1F4A4C,$AC645E0F,$C80533C6,$DF7E2785,$E6F31B40,$F1880F03,
      $DE920307,$C9E91744,$F0642B81,$E71F3FC2,$837E520B,$94054648,$AD887A8D,$BAF36ECE,
      $654AA11F,$7231B55C,$4BBC8999,$5CC79DDA,$38A6F013,$2FDDE450,$1650D895,$012BCCD6,
      $72524176,$65295535,$5CA469F0,$4BDF7DB3,$2FBE107A,$38C50439,$014838FC,$16332CBF,
      $C98AE36E,$DEF1F72D,$E77CCBE8,$F007DFAB,$9466B262,$831DA621,$BA909AE4,$ADEB8EA7,
      $5C6381A4,$4B1895E7,$7295A922,$65EEBD61,$018FD0A8,$16F4C4EB,$2F79F82E,$3802EC6D,
      $E7BB23BC,$F0C037FF,$C94D0B3A,$DE361F79,$BA5772B0,$AD2C66F3,$94A15A36,$83DA4E75,
      $F0A3C3D5,$E7D8D796,$DE55EB53,$C92EFF10,$AD4F92D9,$BA34869A,$83B9BA5F,$94C2AE1C,
      $4B7B61CD,$5C00758E,$658D494B,$72F65D08,$169730C1,$01EC2482,$38611847,$2F1A0C04,
      $6655004F,$712E140C,$48A328C9,$5FD83C8A,$3BB95143,$2CC24500,$154F79C5,$02346D86,
      $DD8DA257,$CAF6B614,$F37B8AD1,$E4009E92,$8061F35B,$971AE718,$AE97DBDD,$B9ECCF9E,
      $CA95423E,$DDEE567D,$E4636AB8,$F3187EFB,$97791332,$80020771,$B98F3BB4,$AEF42FF7,
      $714DE026,$6636F465,$5FBBC8A0,$48C0DCE3,$2CA1B12A,$3BDAA569,$025799AC,$152C8DEF,
      $E4A482EC,$F3DF96AF,$CA52AA6A,$DD29BE29,$B948D3E0,$AE33C7A3,$97BEFB66,$80C5EF25,
      $5F7C20F4,$480734B7,$718A0872,$66F11C31,$029071F8,$15EB65BB,$2C66597E,$3B1D4D3D,
      $4864C09D,$5F1FD4DE,$6692E81B,$71E9FC58,$15889191,$02F385D2,$3B7EB917,$2C05AD54,
      $F3BC6285,$E4C776C6,$DD4A4A03,$CA315E40,$AE503389,$B92B27CA,$80A61B0F,$97DD0F4C,
      $B8C70348,$AFBC170B,$96312BCE,$814A3F8D,$E52B5244,$F2504607,$CBDD7AC2,$DCA66E81,
      $031FA150,$1464B513,$2DE989D6,$3A929D95,$5EF3F05C,$4988E41F,$7005D8DA,$677ECC99,
      $14074139,$037C557A,$3AF169BF,$2D8A7DFC,$49EB1035,$5E900476,$671D38B3,$70662CF0,
      $AFDFE321,$B8A4F762,$8129CBA7,$9652DFE4,$F233B22D,$E548A66E,$DCC59AAB,$CBBE8EE8,
      $3A3681EB,$2D4D95A8,$14C0A96D,$03BBBD2E,$67DAD0E7,$70A1C4A4,$492CF861,$5E57EC22,
      $81EE23F3,$969537B0,$AF180B75,$B8631F36,$DC0272FF,$CB7966BC,$F2F45A79,$E58F4E3A,
      $96F6C39A,$818DD7D9,$B800EB1C,$AF7BFF5F,$CB1A9296,$DC6186D5,$E5ECBA10,$F297AE53,
      $2D2E6182,$3A5575C1,$03D84904,$14A35D47,$70C2308E,$67B924CD,$5E341808,$494F0C4B
    ),
    (
      $00000000,$EFC26B3E,$04F5D03D,$EB37BB03,$09EBA07A,$E629CB44,$0D1E7047,$E2DC1B79,
      $13D740F4,$FC152BCA,$172290C9,$F8E0FBF7,$1A3CE08E,$F5FE8BB0,$1EC930B3,$F10B5B8D,
      $27AE81E8,$C86CEAD6,$235B51D5,$CC993AEB,$2E452192,$C1874AAC,$2AB0F1AF,$C5729A91,
      $3479C11C,$DBBBAA22,$308C1121,$DF4E7A1F,$3D926166,$D2500A58,$3967B15B,$D6A5DA65,
      $4F5D03D0,$A09F68EE,$4BA8D3ED,$A46AB8D3,$46B6A3AA,$A974C894,$42437397,$AD8118A9,
      $5C8A4324,$B348281A,$587F9319,$B7BDF827,$5561E35E,$BAA38860,$51943363,$BE56585D,
      $68F38238,$8731E906,$6C065205,$83C4393B,$61182242,$8EDA497C,$65EDF27F,$8A2F9941,
      $7B24C2CC,$94E6A9F2,$7FD112F1,$901379CF,$72CF62B6,$9D0D0988,$763AB28B,$99F8D9B5,
      $9EBA07A0,$71786C9E,$9A4FD79D,$758DBCA3,$9751A7DA,$7893CCE4,$93A477E7,$7C661CD9,
      $8D6D4754,$62AF2C6A,$89989769,$665AFC57,$8486E72E,$6B448C10,$80733713,$6FB15C2D,
      $B9148648,$56D6ED76,$BDE15675,$52233D4B,$B0FF2632,$5F3D4D0C,$B40AF60F,$5BC89D31,
      $AAC3C6BC,$4501AD82,$AE361681,$41F47DBF,$A32866C6,$4CEA0DF8,$A7DDB6FB,$481FDDC5,
      $D1E70470,$3E256F4E,$D512D44D,$3AD0BF73,$D80CA40A,$37CECF34,$DCF97437,$333B1F09,
      $C2304484,$2DF22FBA,$C6C594B9,$2907FF87,$CBDBE4FE,$24198FC0,$CF2E34C3,$20EC5FFD,
      $F6498598,$198BEEA6,$F2BC55A5,$1D7E3E9B,$FFA225E2,$10604EDC,$FB57F5DF,$14959EE1,
      $E59EC56C,$0A5CAE52,$E16B1551,$0EA97E6F,$EC756516,$03B70E28,$E880B52B,$0742DE15,
      $E6050901,$09C7623F,$E2F0D93C,$0D32B202,$EFEEA97B,$002CC245,$EB1B7946,$04D91278,
      $F5D249F5,$1A1022CB,$F12799C8,$1EE5F2F6,$FC39E98F,$13FB82B1,$F8CC39B2,$170E528C,
      $C1AB88E9,$2E69E3D7,$C55E58D4,$2A9C33EA,$C8402893,$278243AD,$CCB5F8AE,$23779390,
      $D27CC81D,$3DBEA323,$D6891820,$394B731E,$DB976867,$34550359,$DF62B85A,$30A0D364,
      $A9580AD1,$469A61EF,$ADADDAEC,$426FB1D2,$A0B3AAAB,$4F71C195,$A4467A96,$4B8411A8,
      $BA8F4A25,$554D211B,$BE7A9A18,$51B8F126,$B364EA5F,$5CA68161,$B7913A62,$5853515C,
      $8EF68B39,$6134E007,$8A035B04,$65C1303A,$871D2B43,$68DF407D,$83E8FB7E,$6C2A9040,
      $9D21CBCD,$72E3A0F3,$99D41BF0,$761670CE,$94CA6BB7,$7B080089,$903FBB8A,$7FFDD0B4,
      $78BF0EA1,$977D659F,$7C4ADE9C,$9388B5A2,$7154AEDB,$9E96C5E5,$75A17EE6,$9A6315D8,
      $6B684E55,$84AA256B,$6F9D9E68,$805FF556,$6283EE2F,$8D418511,$66763E12,$89B4552C,
      $5F118F49,$B0D3E477,$5BE45F74,$B426344A,$56FA2F33,$B938440D,$520FFF0E,$BDCD9430,
      $4CC6CFBD,$A304A483,$48331F80,$A7F174BE,$452D6FC7,$AAEF04F9,$41D8BFFA,$AE1AD4C4,
      $37E20D71,$D820664F,$3317DD4C,$DCD5B672,$3E09AD0B,$D1CBC635,$3AFC7D36,$D53E1608,
      $24354D85,$CBF726BB,$20C09DB8,$CF02F686,$2DDEEDFF,$C21C86C1,$292B3DC2,$C6E956FC,
      $104C8C99,$FF8EE7A7,$14B95CA4,$FB7B379A,$19A72CE3,$F66547DD,$1D52FCDE,$F29097E0,
      $039BCC6D,$EC59A753,$076E1C50,$E8AC776E,$0A706C17,$E5B20729,$0E85BC2A,$E147D714
    ),
    (
      $00000000,$C18EDFC0,$586CB9C1,$99E26601,$B0D97382,$7157AC42,$E8B5CA43,$293B1583,
      $BAC3E145,$7B4D3E85,$E2AF5884,$23218744,$0A1A92C7,$CB944D07,$52762B06,$93F8F4C6,
      $AEF6C4CB,$6F781B0B,$F69A7D0A,$3714A2CA,$1E2FB749,$DFA16889,$46430E88,$87CDD148,
      $1435258E,$D5BBFA4E,$4C599C4F,$8DD7438F,$A4EC560C,$656289CC,$FC80EFCD,$3D0E300D,
      $869C8FD7,$47125017,$DEF03616,$1F7EE9D6,$3645FC55,$F7CB2395,$6E294594,$AFA79A54,
      $3C5F6E92,$FDD1B152,$6433D753,$A5BD0893,$8C861D10,$4D08C2D0,$D4EAA4D1,$15647B11,
      $286A4B1C,$E9E494DC,$7006F2DD,$B1882D1D,$98B3389E,$593DE75E,$C0DF815F,$01515E9F,
      $92A9AA59,$53277599,$CAC51398,$0B4BCC58,$2270D9DB,$E3FE061B,$7A1C601A,$BB92BFDA,
      $D64819EF,$17C6C62F,$8E24A02E,$4FAA7FEE,$66916A6D,$A71FB5AD,$3EFDD3AC,$FF730C6C,
      $6C8BF8AA,$AD05276A,$34E7416B,$F5699EAB,$DC528B28,$1DDC54E8,$843E32E9,$45B0ED29,
      $78BEDD24,$B93002E4,$20D264E5,$E15CBB25,$C867AEA6,$09E97166,$900B1767,$5185C8A7,
      $C27D3C61,$03F3E3A1,$9A1185A0,$5B9F5A60,$72A44FE3,$B32A9023,$2AC8F622,$EB4629E2,
      $50D49638,$915A49F8,$08B82FF9,$C936F039,$E00DE5BA,$21833A7A,$B8615C7B,$79EF83BB,
      $EA17777D,$2B99A8BD,$B27BCEBC,$73F5117C,$5ACE04FF,$9B40DB3F,$02A2BD3E,$C32C62FE,
      $FE2252F3,$3FAC8D33,$A64EEB32,$67C034F2,$4EFB2171,$8F75FEB1,$169798B0,$D7194770,
      $44E1B3B6,$856F6C76,$1C8D0A77,$DD03D5B7,$F438C034,$35B61FF4,$AC5479F5,$6DDAA635,
      $77E1359F,$B66FEA5F,$2F8D8C5E,$EE03539E,$C738461D,$06B699DD,$9F54FFDC,$5EDA201C,
      $CD22D4DA,$0CAC0B1A,$954E6D1B,$54C0B2DB,$7DFBA758,$BC757898,$25971E99,$E419C159,
      $D917F154,$18992E94,$817B4895,$40F59755,$69CE82D6,$A8405D16,$31A23B17,$F02CE4D7,
      $63D41011,$A25ACFD1,$3BB8A9D0,$FA367610,$D30D6393,$1283BC53,$8B61DA52,$4AEF0592,
      $F17DBA48,$30F36588,$A9110389,$689FDC49,$41A4C9CA,$802A160A,$19C8700B,$D846AFCB,
      $4BBE5B0D,$8A3084CD,$13D2E2CC,$D25C3D0C,$FB67288F,$3AE9F74F,$A30B914E,$62854E8E,
      $5F8B7E83,$9E05A143,$07E7C742,$C6691882,$EF520D01,$2EDCD2C1,$B73EB4C0,$76B06B00,
      $E5489FC6,$24C64006,$BD242607,$7CAAF9C7,$5591EC44,$941F3384,$0DFD5585,$CC738A45,
      $A1A92C70,$6027F3B0,$F9C595B1,$384B4A71,$11705FF2,$D0FE8032,$491CE633,$889239F3,
      $1B6ACD35,$DAE412F5,$430674F4,$8288AB34,$ABB3BEB7,$6A3D6177,$F3DF0776,$3251D8B6,
      $0F5FE8BB,$CED1377B,$5733517A,$96BD8EBA,$BF869B39,$7E0844F9,$E7EA22F8,$2664FD38,
      $B59C09FE,$7412D63E,$EDF0B03F,$2C7E6FFF,$05457A7C,$C4CBA5BC,$5D29C3BD,$9CA71C7D,
      $2735A3A7,$E6BB7C67,$7F591A66,$BED7C5A6,$97ECD025,$56620FE5,$CF8069E4,$0E0EB624,
      $9DF642E2,$5C789D22,$C59AFB23,$041424E3,$2D2F3160,$ECA1EEA0,$754388A1,$B4CD5761,
      $89C3676C,$484DB8AC,$D1AFDEAD,$1021016D,$391A14EE,$F894CB2E,$6176AD2F,$A0F872EF,
      $33008629,$F28E59E9,$6B6C3FE8,$AAE2E028,$83D9F5AB,$42572A6B,$DBB54C6A,$1A3B93AA
    ),
    (
      $00000000,$9BA54C6F,$EC3B9E9F,$779ED2F0,$03063B7F,$98A37710,$EF3DA5E0,$7498E98F,
      $060C76FE,$9DA93A91,$EA37E861,$7192A40E,$050A4D81,$9EAF01EE,$E931D31E,$72949F71,
      $0C18EDFC,$97BDA193,$E0237363,$7B863F0C,$0F1ED683,$94BB9AEC,$E325481C,$78800473,
      $0A149B02,$91B1D76D,$E62F059D,$7D8A49F2,$0912A07D,$92B7EC12,$E5293EE2,$7E8C728D,
      $1831DBF8,$83949797,$F40A4567,$6FAF0908,$1B37E087,$8092ACE8,$F70C7E18,$6CA93277,
      $1E3DAD06,$8598E169,$F2063399,$69A37FF6,$1D3B9679,$869EDA16,$F10008E6,$6AA54489,
      $14293604,$8F8C7A6B,$F812A89B,$63B7E4F4,$172F0D7B,$8C8A4114,$FB1493E4,$60B1DF8B,
      $122540FA,$89800C95,$FE1EDE65,$65BB920A,$11237B85,$8A8637EA,$FD18E51A,$66BDA975,
      $3063B7F0,$ABC6FB9F,$DC58296F,$47FD6500,$33658C8F,$A8C0C0E0,$DF5E1210,$44FB5E7F,
      $366FC10E,$ADCA8D61,$DA545F91,$41F113FE,$3569FA71,$AECCB61E,$D95264EE,$42F72881,
      $3C7B5A0C,$A7DE1663,$D040C493,$4BE588FC,$3F7D6173,$A4D82D1C,$D346FFEC,$48E3B383,
      $3A772CF2,$A1D2609D,$D64CB26D,$4DE9FE02,$3971178D,$A2D45BE2,$D54A8912,$4EEFC57D,
      $28526C08,$B3F72067,$C469F297,$5FCCBEF8,$2B545777,$B0F11B18,$C76FC9E8,$5CCA8587,
      $2E5E1AF6,$B5FB5699,$C2658469,$59C0C806,$2D582189,$B6FD6DE6,$C163BF16,$5AC6F379,
      $244A81F4,$BFEFCD9B,$C8711F6B,$53D45304,$274CBA8B,$BCE9F6E4,$CB772414,$50D2687B,
      $2246F70A,$B9E3BB65,$CE7D6995,$55D825FA,$2140CC75,$BAE5801A,$CD7B52EA,$56DE1E85,
      $60C76FE0,$FB62238F,$8CFCF17F,$1759BD10,$63C1549F,$F86418F0,$8FFACA00,$145F866F,
      $66CB191E,$FD6E5571,$8AF08781,$1155CBEE,$65CD2261,$FE686E0E,$89F6BCFE,$1253F091,
      $6CDF821C,$F77ACE73,$80E41C83,$1B4150EC,$6FD9B963,$F47CF50C,$83E227FC,$18476B93,
      $6AD3F4E2,$F176B88D,$86E86A7D,$1D4D2612,$69D5CF9D,$F27083F2,$85EE5102,$1E4B1D6D,
      $78F6B418,$E353F877,$94CD2A87,$0F6866E8,$7BF08F67,$E055C308,$97CB11F8,$0C6E5D97,
      $7EFAC2E6,$E55F8E89,$92C15C79,$09641016,$7DFCF999,$E659B5F6,$91C76706,$0A622B69,
      $74EE59E4,$EF4B158B,$98D5C77B,$03708B14,$77E8629B,$EC4D2EF4,$9BD3FC04,$0076B06B,
      $72E22F1A,$E9476375,$9ED9B185,$057CFDEA,$71E41465,$EA41580A,$9DDF8AFA,$067AC695,
      $50A4D810,$CB01947F,$BC9F468F,$273A0AE0,$53A2E36F,$C807AF00,$BF997DF0,$243C319F,
      $56A8AEEE,$CD0DE281,$BA933071,$21367C1E,$55AE9591,$CE0BD9FE,$B9950B0E,$22304761,
      $5CBC35EC,$C7197983,$B087AB73,$2B22E71C,$5FBA0E93,$C41F42FC,$B381900C,$2824DC63,
      $5AB04312,$C1150F7D,$B68BDD8D,$2D2E91E2,$59B6786D,$C2133402,$B58DE6F2,$2E28AA9D,
      $489503E8,$D3304F87,$A4AE9D77,$3F0BD118,$4B933897,$D03674F8,$A7A8A608,$3C0DEA67,
      $4E997516,$D53C3979,$A2A2EB89,$3907A7E6,$4D9F4E69,$D63A0206,$A1A4D0F6,$3A019C99,
      $448DEE14,$DF28A27B,$A8B6708B,$33133CE4,$478BD56B,$DC2E9904,$ABB04BF4,$3015079B,
      $428198EA,$D924D485,$AEBA0675,$351F4A1A,$4187A395,$DA22EFFA,$ADBC3D0A,$36197165
    ),
    (
      $00000000,$DD96D985,$605CB54B,$BDCA6CCE,$C0B96A96,$1D2FB313,$A0E5DFDD,$7D730658,
      $5A03D36D,$87950AE8,$3A5F6626,$E7C9BFA3,$9ABAB9FB,$472C607E,$FAE60CB0,$2770D535,
      $B407A6DA,$69917F5F,$D45B1391,$09CDCA14,$74BECC4C,$A92815C9,$14E27907,$C974A082,
      $EE0475B7,$3392AC32,$8E58C0FC,$53CE1979,$2EBD1F21,$F32BC6A4,$4EE1AA6A,$937773EF,
      $B37E4BF5,$6EE89270,$D322FEBE,$0EB4273B,$73C72163,$AE51F8E6,$139B9428,$CE0D4DAD,
      $E97D9898,$34EB411D,$89212DD3,$54B7F456,$29C4F20E,$F4522B8B,$49984745,$940E9EC0,
      $0779ED2F,$DAEF34AA,$67255864,$BAB381E1,$C7C087B9,$1A565E3C,$A79C32F2,$7A0AEB77,
      $5D7A3E42,$80ECE7C7,$3D268B09,$E0B0528C,$9DC354D4,$40558D51,$FD9FE19F,$2009381A,
      $BD8D91AB,$601B482E,$DDD124E0,$0047FD65,$7D34FB3D,$A0A222B8,$1D684E76,$C0FE97F3,
      $E78E42C6,$3A189B43,$87D2F78D,$5A442E08,$27372850,$FAA1F1D5,$476B9D1B,$9AFD449E,
      $098A3771,$D41CEEF4,$69D6823A,$B4405BBF,$C9335DE7,$14A58462,$A96FE8AC,$74F93129,
      $5389E41C,$8E1F3D99,$33D55157,$EE4388D2,$93308E8A,$4EA6570F,$F36C3BC1,$2EFAE244,
      $0EF3DA5E,$D36503DB,$6EAF6F15,$B339B690,$CE4AB0C8,$13DC694D,$AE160583,$7380DC06,
      $54F00933,$8966D0B6,$34ACBC78,$E93A65FD,$944963A5,$49DFBA20,$F415D6EE,$29830F6B,
      $BAF47C84,$6762A501,$DAA8C9CF,$073E104A,$7A4D1612,$A7DBCF97,$1A11A359,$C7877ADC,
      $E0F7AFE9,$3D61766C,$80AB1AA2,$5D3DC327,$204EC57F,$FDD81CFA,$40127034,$9D84A9B1,
      $A06A2517,$7DFCFC92,$C036905C,$1DA049D9,$60D34F81,$BD459604,$008FFACA,$DD19234F,
      $FA69F67A,$27FF2FFF,$9A354331,$47A39AB4,$3AD09CEC,$E7464569,$5A8C29A7,$871AF022,
      $146D83CD,$C9FB5A48,$74313686,$A9A7EF03,$D4D4E95B,$094230DE,$B4885C10,$691E8595,
      $4E6E50A0,$93F88925,$2E32E5EB,$F3A43C6E,$8ED73A36,$5341E3B3,$EE8B8F7D,$331D56F8,
      $13146EE2,$CE82B767,$7348DBA9,$AEDE022C,$D3AD0474,$0E3BDDF1,$B3F1B13F,$6E6768BA,
      $4917BD8F,$9481640A,$294B08C4,$F4DDD141,$89AED719,$54380E9C,$E9F26252,$3464BBD7,
      $A713C838,$7A8511BD,$C74F7D73,$1AD9A4F6,$67AAA2AE,$BA3C7B2B,$07F617E5,$DA60CE60,
      $FD101B55,$2086C2D0,$9D4CAE1E,$40DA779B,$3DA971C3,$E03FA846,$5DF5C488,$80631D0D,
      $1DE7B4BC,$C0716D39,$7DBB01F7,$A02DD872,$DD5EDE2A,$00C807AF,$BD026B61,$6094B2E4,
      $47E467D1,$9A72BE54,$27B8D29A,$FA2E0B1F,$875D0D47,$5ACBD4C2,$E701B80C,$3A976189,
      $A9E01266,$7476CBE3,$C9BCA72D,$142A7EA8,$695978F0,$B4CFA175,$0905CDBB,$D493143E,
      $F3E3C10B,$2E75188E,$93BF7440,$4E29ADC5,$335AAB9D,$EECC7218,$53061ED6,$8E90C753,
      $AE99FF49,$730F26CC,$CEC54A02,$13539387,$6E2095DF,$B3B64C5A,$0E7C2094,$D3EAF911,
      $F49A2C24,$290CF5A1,$94C6996F,$495040EA,$342346B2,$E9B59F37,$547FF3F9,$89E92A7C,
      $1A9E5993,$C7088016,$7AC2ECD8,$A754355D,$DA273305,$07B1EA80,$BA7B864E,$67ED5FCB,
      $409D8AFE,$9D0B537B,$20C13FB5,$FD57E630,$8024E068,$5DB239ED,$E0785523,$3DEE8CA6
    ),
    (
      $00000000,$9D0FE176,$E16EC4AD,$7C6125DB,$19AC8F1B,$84A36E6D,$F8C24BB6,$65CDAAC0,
      $33591E36,$AE56FF40,$D237DA9B,$4F383BED,$2AF5912D,$B7FA705B,$CB9B5580,$5694B4F6,
      $66B23C6C,$FBBDDD1A,$87DCF8C1,$1AD319B7,$7F1EB377,$E2115201,$9E7077DA,$037F96AC,
      $55EB225A,$C8E4C32C,$B485E6F7,$298A0781,$4C47AD41,$D1484C37,$AD2969EC,$3026889A,
      $CD6478D8,$506B99AE,$2C0ABC75,$B1055D03,$D4C8F7C3,$49C716B5,$35A6336E,$A8A9D218,
      $FE3D66EE,$63328798,$1F53A243,$825C4335,$E791E9F5,$7A9E0883,$06FF2D58,$9BF0CC2E,
      $ABD644B4,$36D9A5C2,$4AB88019,$D7B7616F,$B27ACBAF,$2F752AD9,$53140F02,$CE1BEE74,
      $988F5A82,$0580BBF4,$79E19E2F,$E4EE7F59,$8123D599,$1C2C34EF,$604D1134,$FD42F042,
      $41B9F7F1,$DCB61687,$A0D7335C,$3DD8D22A,$581578EA,$C51A999C,$B97BBC47,$24745D31,
      $72E0E9C7,$EFEF08B1,$938E2D6A,$0E81CC1C,$6B4C66DC,$F64387AA,$8A22A271,$172D4307,
      $270BCB9D,$BA042AEB,$C6650F30,$5B6AEE46,$3EA74486,$A3A8A5F0,$DFC9802B,$42C6615D,
      $1452D5AB,$895D34DD,$F53C1106,$6833F070,$0DFE5AB0,$90F1BBC6,$EC909E1D,$719F7F6B,
      $8CDD8F29,$11D26E5F,$6DB34B84,$F0BCAAF2,$95710032,$087EE144,$741FC49F,$E91025E9,
      $BF84911F,$228B7069,$5EEA55B2,$C3E5B4C4,$A6281E04,$3B27FF72,$4746DAA9,$DA493BDF,
      $EA6FB345,$77605233,$0B0177E8,$960E969E,$F3C33C5E,$6ECCDD28,$12ADF8F3,$8FA21985,
      $D936AD73,$44394C05,$385869DE,$A55788A8,$C09A2268,$5D95C31E,$21F4E6C5,$BCFB07B3,
      $8373EFE2,$1E7C0E94,$621D2B4F,$FF12CA39,$9ADF60F9,$07D0818F,$7BB1A454,$E6BE4522,
      $B02AF1D4,$2D2510A2,$51443579,$CC4BD40F,$A9867ECF,$34899FB9,$48E8BA62,$D5E75B14,
      $E5C1D38E,$78CE32F8,$04AF1723,$99A0F655,$FC6D5C95,$6162BDE3,$1D039838,$800C794E,
      $D698CDB8,$4B972CCE,$37F60915,$AAF9E863,$CF3442A3,$523BA3D5,$2E5A860E,$B3556778,
      $4E17973A,$D318764C,$AF795397,$3276B2E1,$57BB1821,$CAB4F957,$B6D5DC8C,$2BDA3DFA,
      $7D4E890C,$E041687A,$9C204DA1,$012FACD7,$64E20617,$F9EDE761,$858CC2BA,$188323CC,
      $28A5AB56,$B5AA4A20,$C9CB6FFB,$54C48E8D,$3109244D,$AC06C53B,$D067E0E0,$4D680196,
      $1BFCB560,$86F35416,$FA9271CD,$679D90BB,$02503A7B,$9F5FDB0D,$E33EFED6,$7E311FA0,
      $C2CA1813,$5FC5F965,$23A4DCBE,$BEAB3DC8,$DB669708,$4669767E,$3A0853A5,$A707B2D3,
      $F1930625,$6C9CE753,$10FDC288,$8DF223FE,$E83F893E,$75306848,$09514D93,$945EACE5,
      $A478247F,$3977C509,$4516E0D2,$D81901A4,$BDD4AB64,$20DB4A12,$5CBA6FC9,$C1B58EBF,
      $97213A49,$0A2EDB3F,$764FFEE4,$EB401F92,$8E8DB552,$13825424,$6FE371FF,$F2EC9089,
      $0FAE60CB,$92A181BD,$EEC0A466,$73CF4510,$1602EFD0,$8B0D0EA6,$F76C2B7D,$6A63CA0B,
      $3CF77EFD,$A1F89F8B,$DD99BA50,$40965B26,$255BF1E6,$B8541090,$C435354B,$593AD43D,
      $691C5CA7,$F413BDD1,$8872980A,$157D797C,$70B0D3BC,$EDBF32CA,$91DE1711,$0CD1F667,
      $5A454291,$C74AA3E7,$BB2B863C,$2624674A,$43E9CD8A,$DEE62CFC,$A2870927,$3F88E851
    ),
    (
      $00000000,$B9FBDBE8,$A886B191,$117D6A79,$8A7C6563,$3387BE8B,$22FAD4F2,$9B010F1A,
      $CF89CC87,$7672176F,$670F7D16,$DEF4A6FE,$45F5A9E4,$FC0E720C,$ED731875,$5488C39D,
      $44629F4F,$FD9944A7,$ECE42EDE,$551FF536,$CE1EFA2C,$77E521C4,$66984BBD,$DF639055,
      $8BEB53C8,$32108820,$236DE259,$9A9639B1,$019736AB,$B86CED43,$A911873A,$10EA5CD2,
      $88C53E9E,$313EE576,$20438F0F,$99B854E7,$02B95BFD,$BB428015,$AA3FEA6C,$13C43184,
      $474CF219,$FEB729F1,$EFCA4388,$56319860,$CD30977A,$74CB4C92,$65B626EB,$DC4DFD03,
      $CCA7A1D1,$755C7A39,$64211040,$DDDACBA8,$46DBC4B2,$FF201F5A,$EE5D7523,$57A6AECB,
      $032E6D56,$BAD5B6BE,$ABA8DCC7,$1253072F,$89520835,$30A9D3DD,$21D4B9A4,$982F624C,
      $CAFB7B7D,$7300A095,$627DCAEC,$DB861104,$40871E1E,$F97CC5F6,$E801AF8F,$51FA7467,
      $0572B7FA,$BC896C12,$ADF4066B,$140FDD83,$8F0ED299,$36F50971,$27886308,$9E73B8E0,
      $8E99E432,$37623FDA,$261F55A3,$9FE48E4B,$04E58151,$BD1E5AB9,$AC6330C0,$1598EB28,
      $411028B5,$F8EBF35D,$E9969924,$506D42CC,$CB6C4DD6,$7297963E,$63EAFC47,$DA1127AF,
      $423E45E3,$FBC59E0B,$EAB8F472,$53432F9A,$C8422080,$71B9FB68,$60C49111,$D93F4AF9,
      $8DB78964,$344C528C,$253138F5,$9CCAE31D,$07CBEC07,$BE3037EF,$AF4D5D96,$16B6867E,
      $065CDAAC,$BFA70144,$AEDA6B3D,$1721B0D5,$8C20BFCF,$35DB6427,$24A60E5E,$9D5DD5B6,
      $C9D5162B,$702ECDC3,$6153A7BA,$D8A87C52,$43A97348,$FA52A8A0,$EB2FC2D9,$52D41931,
      $4E87F0BB,$F77C2B53,$E601412A,$5FFA9AC2,$C4FB95D8,$7D004E30,$6C7D2449,$D586FFA1,
      $810E3C3C,$38F5E7D4,$29888DAD,$90735645,$0B72595F,$B28982B7,$A3F4E8CE,$1A0F3326,
      $0AE56FF4,$B31EB41C,$A263DE65,$1B98058D,$80990A97,$3962D17F,$281FBB06,$91E460EE,
      $C56CA373,$7C97789B,$6DEA12E2,$D411C90A,$4F10C610,$F6EB1DF8,$E7967781,$5E6DAC69,
      $C642CE25,$7FB915CD,$6EC47FB4,$D73FA45C,$4C3EAB46,$F5C570AE,$E4B81AD7,$5D43C13F,
      $09CB02A2,$B030D94A,$A14DB333,$18B668DB,$83B767C1,$3A4CBC29,$2B31D650,$92CA0DB8,
      $8220516A,$3BDB8A82,$2AA6E0FB,$935D3B13,$085C3409,$B1A7EFE1,$A0DA8598,$19215E70,
      $4DA99DED,$F4524605,$E52F2C7C,$5CD4F794,$C7D5F88E,$7E2E2366,$6F53491F,$D6A892F7,
      $847C8BC6,$3D87502E,$2CFA3A57,$9501E1BF,$0E00EEA5,$B7FB354D,$A6865F34,$1F7D84DC,
      $4BF54741,$F20E9CA9,$E373F6D0,$5A882D38,$C1892222,$7872F9CA,$690F93B3,$D0F4485B,
      $C01E1489,$79E5CF61,$6898A518,$D1637EF0,$4A6271EA,$F399AA02,$E2E4C07B,$5B1F1B93,
      $0F97D80E,$B66C03E6,$A711699F,$1EEAB277,$85EBBD6D,$3C106685,$2D6D0CFC,$9496D714,
      $0CB9B558,$B5426EB0,$A43F04C9,$1DC4DF21,$86C5D03B,$3F3E0BD3,$2E4361AA,$97B8BA42,
      $C33079DF,$7ACBA237,$6BB6C84E,$D24D13A6,$494C1CBC,$F0B7C754,$E1CAAD2D,$583176C5,
      $48DB2A17,$F120F1FF,$E05D9B86,$59A6406E,$C2A74F74,$7B5C949C,$6A21FEE5,$D3DA250D,
      $8752E690,$3EA93D78,$2FD45701,$962F8CE9,$0D2E83F3,$B4D5581B,$A5A83262,$1C53E98A
    ),
    (
      $00000000,$AE689191,$87A02563,$29C8B4F2,$D4314C87,$7A59DD16,$539169E4,$FDF9F875,
      $73139F4F,$DD7B0EDE,$F4B3BA2C,$5ADB2BBD,$A722D3C8,$094A4259,$2082F6AB,$8EEA673A,
      $E6273E9E,$484FAF0F,$61871BFD,$CFEF8A6C,$32167219,$9C7EE388,$B5B6577A,$1BDEC6EB,
      $9534A1D1,$3B5C3040,$129484B2,$BCFC1523,$4105ED56,$EF6D7CC7,$C6A5C835,$68CD59A4,
      $173F7B7D,$B957EAEC,$909F5E1E,$3EF7CF8F,$C30E37FA,$6D66A66B,$44AE1299,$EAC68308,
      $642CE432,$CA4475A3,$E38CC151,$4DE450C0,$B01DA8B5,$1E753924,$37BD8DD6,$99D51C47,
      $F11845E3,$5F70D472,$76B86080,$D8D0F111,$25290964,$8B4198F5,$A2892C07,$0CE1BD96,
      $820BDAAC,$2C634B3D,$05ABFFCF,$ABC36E5E,$563A962B,$F85207BA,$D19AB348,$7FF222D9,
      $2E7EF6FA,$8016676B,$A9DED399,$07B64208,$FA4FBA7D,$54272BEC,$7DEF9F1E,$D3870E8F,
      $5D6D69B5,$F305F824,$DACD4CD6,$74A5DD47,$895C2532,$2734B4A3,$0EFC0051,$A09491C0,
      $C859C864,$663159F5,$4FF9ED07,$E1917C96,$1C6884E3,$B2001572,$9BC8A180,$35A03011,
      $BB4A572B,$1522C6BA,$3CEA7248,$9282E3D9,$6F7B1BAC,$C1138A3D,$E8DB3ECF,$46B3AF5E,
      $39418D87,$97291C16,$BEE1A8E4,$10893975,$ED70C100,$43185091,$6AD0E463,$C4B875F2,
      $4A5212C8,$E43A8359,$CDF237AB,$639AA63A,$9E635E4F,$300BCFDE,$19C37B2C,$B7ABEABD,
      $DF66B319,$710E2288,$58C6967A,$F6AE07EB,$0B57FF9E,$A53F6E0F,$8CF7DAFD,$229F4B6C,
      $AC752C56,$021DBDC7,$2BD50935,$85BD98A4,$784460D1,$D62CF140,$FFE445B2,$518CD423,
      $5CFDEDF4,$F2957C65,$DB5DC897,$75355906,$88CCA173,$26A430E2,$0F6C8410,$A1041581,
      $2FEE72BB,$8186E32A,$A84E57D8,$0626C649,$FBDF3E3C,$55B7AFAD,$7C7F1B5F,$D2178ACE,
      $BADAD36A,$14B242FB,$3D7AF609,$93126798,$6EEB9FED,$C0830E7C,$E94BBA8E,$47232B1F,
      $C9C94C25,$67A1DDB4,$4E696946,$E001F8D7,$1DF800A2,$B3909133,$9A5825C1,$3430B450,
      $4BC29689,$E5AA0718,$CC62B3EA,$620A227B,$9FF3DA0E,$319B4B9F,$1853FF6D,$B63B6EFC,
      $38D109C6,$96B99857,$BF712CA5,$1119BD34,$ECE04541,$4288D4D0,$6B406022,$C528F1B3,
      $ADE5A817,$038D3986,$2A458D74,$842D1CE5,$79D4E490,$D7BC7501,$FE74C1F3,$501C5062,
      $DEF63758,$709EA6C9,$5956123B,$F73E83AA,$0AC77BDF,$A4AFEA4E,$8D675EBC,$230FCF2D,
      $72831B0E,$DCEB8A9F,$F5233E6D,$5B4BAFFC,$A6B25789,$08DAC618,$211272EA,$8F7AE37B,
      $01908441,$AFF815D0,$8630A122,$285830B3,$D5A1C8C6,$7BC95957,$5201EDA5,$FC697C34,
      $94A42590,$3ACCB401,$130400F3,$BD6C9162,$40956917,$EEFDF886,$C7354C74,$695DDDE5,
      $E7B7BADF,$49DF2B4E,$60179FBC,$CE7F0E2D,$3386F658,$9DEE67C9,$B426D33B,$1A4E42AA,
      $65BC6073,$CBD4F1E2,$E21C4510,$4C74D481,$B18D2CF4,$1FE5BD65,$362D0997,$98459806,
      $16AFFF3C,$B8C76EAD,$910FDA5F,$3F674BCE,$C29EB3BB,$6CF6222A,$453E96D8,$EB560749,
      $839B5EED,$2DF3CF7C,$043B7B8E,$AA53EA1F,$57AA126A,$F9C283FB,$D00A3709,$7E62A698,
      $F088C1A2,$5EE05033,$7728E4C1,$D9407550,$24B98D25,$8AD11CB4,$A319A846,$0D7139D7
    )
  );

/// compute CRC32 (Slicing-by-16 algorithm)
function crc32_16bytes(const data: PByte; length: Integer; previousCrc32: Cardinal = 0): Cardinal;
const
  Unroll = 4;
  BytesAtOnce = 16 * Unroll;
var
  crc: cardinal;
  unrolling: integer;
  current: PLongWord;
  currentChar: PByte;
  one, two, three, four: cardinal;
begin
  crc :=  previousCrc32 xor $FFFFFFFF;
  current := PLongWord(data);

  // enabling optimization (at least -O2) automatically unrolls the inner for-loop
  while (length >= BytesAtOnce) do
  begin
    for unrolling := 0 to Unroll - 1 do
    begin
      one   := current^ xor crc; Inc(current);
      two   := current^; Inc(current);
      three := current^; Inc(current);
      four  := current^; Inc(current);
      crc   := Crc32Lookup[ 0][(four  shr 24) and $FF] xor
               Crc32Lookup[ 1][(four  shr 16) and $FF] xor
               Crc32Lookup[ 2][(four  shr  8) and $FF] xor
               Crc32Lookup[ 3][ four          and $FF] xor
               Crc32Lookup[ 4][(three shr 24) and $FF] xor
               Crc32Lookup[ 5][(three shr 16) and $FF] xor
               Crc32Lookup[ 6][(three shr  8) and $FF] xor
               Crc32Lookup[ 7][ three         and $FF] xor
               Crc32Lookup[ 8][(two   shr 24) and $FF] xor
               Crc32Lookup[ 9][(two   shr 16) and $FF] xor
               Crc32Lookup[10][(two   shr  8) and $FF] xor
               Crc32Lookup[11][ two           and $FF] xor
               Crc32Lookup[12][(one   shr 24) and $FF] xor
               Crc32Lookup[13][(one   shr 16) and $FF] xor
               Crc32Lookup[14][(one   shr  8) and $FF] xor
               Crc32Lookup[15][ one           and $FF];
    end;

    length -= BytesAtOnce;
  end;

  currentChar := PByte(current);
  // remaining 1 to 63 bytes (standard algorithm)
  while (length <> 0) do
  begin
    crc := (crc shr 8) xor Crc32Lookup[0][(crc and $FF) xor currentChar^];
    Inc(currentChar);
    Dec(length);
  end;

  Result:=  crc xor $FFFFFFFF;
end;

end.

