/* $Cambridge: hermes/src/prayer/cmd/cmd_vacation.c,v 1.5 2010/07/02 08:31:08 dpc22 Exp $ */
/************************************************
 *    Prayer - a Webmail Interface              *
 ************************************************/

/* Copyright (c) University of Cambridge 2000 - 2008 */
/* See the file NOTICE for conditions of use and distribution. */

#include "prayer_session.h"

static void generate_form(struct session *session)
{
    struct template_vals *tvals = session->template_vals;
    struct request *request = session->request;
    struct buffer *b = request->write_buffer;
    struct account *account = session->account;
    char *aliases = account->vacation_aliases;
    char *subject = account->vacation_subject;

    if (!(subject && subject[0]))
        subject = "Auto-Response: Vacation Message";

    if (account->vacation_enabled)
        template_vals_ulong(tvals, "use_vacation", 1);
    
    template_vals_string(tvals, "aliases", aliases);
    template_vals_string(tvals, "subject", subject);
    template_vals_ulong(tvals, "days", account->vacation_days);

    if (session->sieved_server)
        template_vals_ulong(tvals, "using_sieve", 1);

    if (account->vacation_msg && account->vacation_msg[0])
        template_vals_string(tvals, "vacation_msg", account->vacation_msg);

    session_seed_template(session, tvals);
    template_expand("vacation", tvals, b);
    response_html(request, 200);        /* Success */
}

static void generate_error(struct session *session)
{
    struct template_vals *tvals = session->template_vals;
    struct account *account = session->account;
    struct request *request = session->request;
    char *msg = account_fetch_message(account);
    struct buffer *b = request->write_buffer;

    if (!(msg && msg[0]))
        msg = "Unable to check mail processing status";

    template_vals_string(tvals, "msg", msg);
    session_seed_template(session, tvals);
    template_expand("vacation_fail", tvals, b);
    response_html(request, 200);
}

void cmd_vacation(struct session *session)
{
    struct request *request = session->request;
    struct assoc *h = NIL;
    struct account *account = session->account;
    ADDRESS *addr = NIL;
    ADDRESS *a = NIL;
    char *s;
    BOOL rc = T;

    if (request->method != POST) {
        if (!account_mail_check(account, request->pool))
            generate_error(session);
        else
            generate_form(session);
        return;
    }

    /* Form post */
    request_decode_form(request);
    h = request->form;

    if (!(s = assoc_lookup(h, "sub_apply"))) {
        session_redirect(session, request, "manage");   /* parent screen */
        return;
    }

    account->vacation_enabled =
        (assoc_lookup(h, "use_vacation")) ? T : NIL;

    if ((s = assoc_lookup(h, "vacation_msg")))
        string_strdup(&account->vacation_msg, s);
    else
        string_strdup(&account->vacation_msg, "");

    if ((s = assoc_lookup(h, "days")) && s[0]) {
        if (!(string_isnumber(s) && (atoi(s) >= 0))) {
            session_message(session,
                            "Days option must be non-negative integer");
            generate_form(session);
            return;

        }
        account->vacation_days = atoi(s);
    }

    if ((s = assoc_lookup(h, "aliases")) && s[0]) {
        string_strdup(&account->vacation_aliases, s);

        if (!(addr=addr_parse_destructive(s, ""))) {
            session_message(session, "Alias list invalid: %s", ml_errmsg());
            rc = NIL;
        } else {
            for (a = addr; a; a = a->next) {
                if (!(a->host && a->host[0])) {
                    session_message(session,
                                    "Unqualified address in alias list");
                    rc = NIL;
                    break;
                }
            }
            mail_free_address(&addr);
        }
    } else
        string_strdup(&account->vacation_aliases, "");

    if ((s = assoc_lookup(h, "subject")))
        string_strdup(&account->vacation_subject, s);
    else
        string_strdup(&account->vacation_subject, "");

    if (rc) {
        if (account_vacation_update(account, request->pool) &&
            account_mail_update(account, request->pool)) {
            session_message(session, "Updated vacation message");
            session_redirect(session, request, "manage");
            return;
        } else {
            char *msg = account_fetch_message(account);

            session_alert(session,
                          "Failed to update vacation message: %s", msg);
            session_log(session,
                        "[cmd_vacation] Failed to update vacation message: %s",
                        msg);
        }
    }

    /* Report some kind of error */
    generate_form(session);
}
