/*
 * Seahorse
 *
 * Copyright (C) 2008 Stefan Walter
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see
 * <http://www.gnu.org/licenses/>.
 */

#include "config.h"

#include "seahorse-gpgme.h"
#include "seahorse-gpgme-subkey.h"
#include "seahorse-gpgme-uid.h"

#include <string.h>

#include <glib/gi18n.h>

enum {
    PROP_0,
    PROP_PUBKEY,
    PROP_SUBKEY
};

struct _SeahorseGpgmeSubkey {
    SeahorsePgpSubkey parent_instance;

    gpgme_key_t pubkey;         /* The public key that this subkey is part of */
    gpgme_subkey_t subkey;      /* The subkey referred to */
};

G_DEFINE_TYPE (SeahorseGpgmeSubkey, seahorse_gpgme_subkey, SEAHORSE_PGP_TYPE_SUBKEY);


gpgme_key_t
seahorse_gpgme_subkey_get_pubkey (SeahorseGpgmeSubkey *self)
{
    g_return_val_if_fail (SEAHORSE_GPGME_IS_SUBKEY (self), NULL);
    g_return_val_if_fail (self->pubkey, NULL);
    return self->pubkey;
}

gpgme_subkey_t
seahorse_gpgme_subkey_get_subkey (SeahorseGpgmeSubkey *self)
{
    g_return_val_if_fail (SEAHORSE_GPGME_IS_SUBKEY (self), NULL);
    g_return_val_if_fail (self->subkey, NULL);
    return self->subkey;
}

void
seahorse_gpgme_subkey_set_subkey (SeahorseGpgmeSubkey *self, gpgme_subkey_t subkey)
{
    g_autofree gchar *description = NULL, *fingerprint = NULL, *name = NULL;
    SeahorsePgpSubkey *base;
    const gchar *algo_type;
    GObject *obj;
    gpgme_subkey_t sub;
    gint i, index;
    guint flags;

    g_return_if_fail (SEAHORSE_GPGME_IS_SUBKEY (self));
    g_return_if_fail (subkey);

    /* Make sure that this subkey is in the pubkey */
    index = -1;
    for (i = 0, sub = self->pubkey->subkeys; sub; ++i, sub = sub->next) {
        if (sub == subkey) {
            index = i;
            break;
        }
    }
    g_return_if_fail (index >= 0);

    /* Calculate the algorithm */
    algo_type = gpgme_pubkey_algo_name (subkey->pubkey_algo);
    if (algo_type == NULL)
        algo_type = C_("Algorithm", "Unknown");
    else if (g_str_equal ("Elg", algo_type) || g_str_equal("ELG-E", algo_type))
        algo_type = _("ElGamal");

    /* Additional properties */
    fingerprint = seahorse_pgp_subkey_calc_fingerprint (subkey->fpr);
    name = seahorse_gpgme_uid_calc_name (self->pubkey->uids);
    description = seahorse_pgp_subkey_calc_description (name, index);

    self->subkey = subkey;

    obj = G_OBJECT (self);
    g_object_freeze_notify (obj);

    base = SEAHORSE_PGP_SUBKEY (self);
    seahorse_pgp_subkey_set_index (base, index);
    seahorse_pgp_subkey_set_keyid (base, subkey->keyid);
    seahorse_pgp_subkey_set_algorithm (base, algo_type);
    seahorse_pgp_subkey_set_length (base, subkey->length);
    seahorse_pgp_subkey_set_description (base, description);
    seahorse_pgp_subkey_set_fingerprint (base, fingerprint);
    seahorse_pgp_subkey_set_created (base, subkey->timestamp);
    seahorse_pgp_subkey_set_expires (base, subkey->expires);

    /* The order below is significant */
    flags = 0;
    if (subkey->revoked)
        flags |= SEAHORSE_FLAG_REVOKED;
    if (subkey->expired)
        flags |= SEAHORSE_FLAG_EXPIRED;
    if (subkey->disabled)
        flags |= SEAHORSE_FLAG_DISABLED;
    if (flags == 0 && !subkey->invalid)
        flags |= SEAHORSE_FLAG_IS_VALID;
    if (subkey->can_encrypt)
        flags |= SEAHORSE_FLAG_CAN_ENCRYPT;
    if (subkey->can_sign)
        flags |= SEAHORSE_FLAG_CAN_SIGN;
    if (subkey->can_certify)
        flags |= SEAHORSE_FLAG_CAN_CERTIFY;
    if (subkey->can_authenticate)
        flags |= SEAHORSE_FLAG_CAN_AUTHENTICATE;

    seahorse_pgp_subkey_set_flags (base, flags);

    g_object_notify (obj, "subkey");
    g_object_thaw_notify (obj);
}

static void
seahorse_gpgme_subkey_init (SeahorseGpgmeSubkey *self)
{
}

static GObject*
seahorse_gpgme_subkey_constructor (GType type, guint n_props, GObjectConstructParam *props)
{
    GObject *obj;
    SeahorseGpgmeSubkey *self = NULL;

    obj = G_OBJECT_CLASS (seahorse_gpgme_subkey_parent_class)->constructor (type, n_props, props);
    if (obj) {
        self = SEAHORSE_GPGME_SUBKEY (obj);
        g_return_val_if_fail (self->pubkey, NULL);
    }

    return obj;
}

static void
seahorse_gpgme_subkey_get_property (GObject *object, guint prop_id,
                                  GValue *value, GParamSpec *pspec)
{
    SeahorseGpgmeSubkey *self = SEAHORSE_GPGME_SUBKEY (object);

    switch (prop_id) {
    case PROP_PUBKEY:
        g_value_set_boxed (value, seahorse_gpgme_subkey_get_pubkey (self));
        break;
    case PROP_SUBKEY:
        g_value_set_pointer (value, seahorse_gpgme_subkey_get_subkey (self));
        break;
    }
}

static void
seahorse_gpgme_subkey_set_property (GObject *object, guint prop_id, const GValue *value,
                                  GParamSpec *pspec)
{
    SeahorseGpgmeSubkey *self = SEAHORSE_GPGME_SUBKEY (object);

    switch (prop_id) {
    case PROP_PUBKEY:
        g_return_if_fail (!self->pubkey);
        self->pubkey = g_value_get_boxed (value);
        if (self->pubkey)
            gpgme_key_ref (self->pubkey);
        break;
    case PROP_SUBKEY:
        seahorse_gpgme_subkey_set_subkey (self, g_value_get_pointer (value));
        break;
    }
}

static void
seahorse_gpgme_subkey_finalize (GObject *gobject)
{
    SeahorseGpgmeSubkey *self = SEAHORSE_GPGME_SUBKEY (gobject);

    /* Unref the key */
    if (self->pubkey)
        gpgme_key_unref (self->pubkey);
    self->pubkey = NULL;
    self->subkey = NULL;

    G_OBJECT_CLASS (seahorse_gpgme_subkey_parent_class)->finalize (gobject);
}

static void
seahorse_gpgme_subkey_class_init (SeahorseGpgmeSubkeyClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

    gobject_class->constructor = seahorse_gpgme_subkey_constructor;
    gobject_class->finalize = seahorse_gpgme_subkey_finalize;
    gobject_class->set_property = seahorse_gpgme_subkey_set_property;
    gobject_class->get_property = seahorse_gpgme_subkey_get_property;

    g_object_class_install_property (gobject_class, PROP_PUBKEY,
        g_param_spec_boxed ("pubkey", "Public Key", "GPGME Public Key that this subkey is on",
                            SEAHORSE_GPGME_BOXED_KEY,
                            G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_CONSTRUCT_ONLY));

    g_object_class_install_property (gobject_class, PROP_SUBKEY,
        g_param_spec_pointer ("subkey", "Subkey", "GPGME Subkey",
                              G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
}

SeahorseGpgmeSubkey*
seahorse_gpgme_subkey_new (gpgme_key_t pubkey, gpgme_subkey_t subkey)
{
    return g_object_new (SEAHORSE_GPGME_TYPE_SUBKEY,
                         "pubkey", pubkey,
                         "subkey", subkey, NULL);
}
