<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Exports the strings needed by the installer
 *
 * Moodle core contains a subset of strings needed for the start of the installation.
 * The list of required strings is maintained in install/stringnames.txt. This
 * script parses that file and exports the translations into a configured destination.
 *
 * @package   local_amos
 * @copyright 2010 David Mudrak <david.mudrak@gmail.com>
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

define('CLI_SCRIPT', true);

require(__DIR__ . '/../../../config.php');
require_once($CFG->dirroot . '/local/amos/cli/config.php');
require_once($CFG->dirroot . '/local/amos/mlanglib.php');

$versions = array_reverse(mlang_version::list_supported(), true);

$git = new \local_amos\local\git(AMOS_REPO_MOODLE);
$git->exec('remote update --prune');

fputs(STDOUT, "*****************************************\n");
fputs(STDOUT, date('Y-m-d H:i', time()));
fputs(STDOUT, " EXPORT INSTALLER JOB STARTED\n");

remove_dir(AMOS_EXPORT_INSTALLER_DIR, true);

$exitstatus = 0;

foreach ($versions as $version) {
    if ($git->has_remote_branch($version->branch)) {
        $gitbranch = 'origin/' . $version->branch;
        $exportdir = $version->code . '_STABLE';

    } else if ($version->code == mlang_version::latest_version()->code) {
        $gitbranch = 'origin/master';
        $exportdir = 'master';

    } else {
        fputs(STDERR, "GIT BRANCH NOT FOUND FOR MOODLE VERSION {$version->label}\n");
        exit(3);
    }

    fputs(STDOUT, "PROCESSING VERSION {$version->code} ({$gitbranch})\n");

    // Read the contents of stringnames.txt on the given branch.
    $gitout = $git->exec('show ' . escapeshellarg($gitbranch . ':install/stringnames.txt'));

    $list = [];

    foreach ($gitout as $string) {
        [$stringid, $component] = array_map('trim', explode(',', $string));
        $list[$component][$stringid] = true;
    }
    unset($gitout);

    $langs = array_keys(mlang_tools::list_languages());

    $phpdoc = <<<EOF
/**
 * Automatically generated strings for Moodle installer
 *
 * Do not edit this file manually! It contains just a subset of strings
 * needed during the very first steps of installation. This file was
 * generated automatically by export-installer.php (which is part of AMOS
 * {@link http://docs.moodle.org/dev/Languages/AMOS}) using the
 * list of strings defined in /install/stringnames.txt.
 *
 * @package   installer
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */


EOF;

    foreach ($list as $componentname => $stringids) {
        foreach ($langs as $lang) {
            if ($lang === 'en_fix') {
                continue;
            }
            $component = mlang_component::from_snapshot($componentname, $lang, $version, null, false, false,
                array_keys($stringids));
            if ($component->has_string()) {
                $file = AMOS_EXPORT_INSTALLER_DIR . '/' . $exportdir . '/install/lang/' . $lang . '/' . $component->name . '.php';
                if (!file_exists(dirname($file))) {
                    mkdir(dirname($file), 0755, true);
                }
                $component->export_phpfile($file, $phpdoc);
            }
            if ($lang === 'en') {
                // Check that all string were exported.
                foreach (array_keys($stringids) as $stringid) {
                    if (!$component->has_string($stringid)) {
                        fputs(STDERR, "ERROR Unknown $stringid,$componentname\n");
                        $exitstatus = 1;
                    }
                }
            }
            $component->clear();
        }
    }
}

fputs(STDOUT, date('Y-m-d H:i', time()));
fputs(STDOUT, " EXPORT INSTALLER JOB DONE\n");

exit($exitstatus);
