#! /usr/bin/env python
# -*- coding: utf-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################

"""Generation des ACL NuFW en fonction des directives depuis un modele Era
usage:

%prog <modele Era>
"""

from sys import exit
from copy import copy, deepcopy
from optparse import OptionParser
from era.tool.IPy import IP

from era.noyau.constants import ACTION_DENY, ACTION_ALLOW, ACTION_FORWARD
from era.noyau.path import NF_ACL_PLAINTEXT, GID_TOUS, GID_GUEST
from era.noyau.initialize import initialize_app
from era.noyau.fwobjects import ServiceGroupe, Directive

from creole.parsedico import parse_dico

dico = parse_dico()

class NotImplementedError(Exception):
    pass

try:
    _
except:
   _ = str

DECISION = {
    ACTION_DENY: 0,
    ACTION_ALLOW: 1,
    ACTION_FORWARD: 1,
}
""" Era action -> NuFW decision : ce sont les codes era """


PROTOCOL = {
    u'icmp': 1,
    u'tcp': 6,
    u'udp': 17,
}
""" protocoles supportés par nufw -> les codes de acl_plaintext """


FULL_IP = IP('0.0.0.0/0.0.0.0')

def _is_full_network(adr):
    """
        une adresse réseau est-elle ou non de type 0/0 ou non
        # utiliser IPy
    """
    if adr == u"":
        raise Exception("une ip ou une adresse reseau n'est pas renseignee dans le modele xml")

    if IP(str(adr)) == FULL_IP:
        return True
    return False

def inst(name):
    """
        instanciation manuelle de variables qui seraient restées templatisées
    """
    if '%' in str(name):
        name = name.replace('%', '')
        return dico[name]
    return name

def parse_fwobject_directive(directive):
    """recupere toutes les infos dispos dans une directive
    afin d'etre utilisées pour le dictionnaire de configuration des acls
    """
    # les zones
    src_zone = directive.src_list[0].zone
    dest_zone = directive.dest_list[0].zone
    # les services
    service = directive.service
    if isinstance(service, ServiceGroupe):
        services = service.services
    else:
        services = [service]
    if directive.src_inv or directive.dest_inv:
        raise NotImplementedError("inversion des extremites non implemente")
    """
     Traitement spécial pour les règles INPUT (bastion comme destination) : il
     faut remplacer l'adresse 127.0.0.1 par l'adresse IP de l'interface réseau
     d'entrée (dépendante de l'extrémité source).
    """
    if src_zone.name == "bastion":
        raise Exception('!!! Pas de regle authentifie depuis bastion !!!')

    if dest_zone.name == "bastion":
        dest_extremites = [inst(str(src_zone.ip))]
    else:
        dest_extremites = []
        for ext in directive.dest_list:
            #ext.ip_list, ext.netmask, ext.subnet, ext.all_zone
            # si l'extremite est la zone entiere, on remplace par 0/0
            if ext.all_zone:
                ext.ip_list = ['0.0.0.0/0']
                ext.netmask = '0.0.0.0'
            dest_extremites.extend(ip_format(ext.ip_list, inst(ext.netmask), inst(ext.subnet)))
    src_extremites = []
    for ext in directive.src_list:
        # si l'extremite est la zone entiere, on remplace par 0/0
        if ext.all_zone:
            ext.ip_list = ['0.0.0.0/0']
            ext.netmask = '0.0.0.0'
        src_extremites.extend(ip_format(ext.ip_list, ext.netmask, ext.subnet))

    if directive.user_group:
        user_group_id = directive.user_group.id
    else:
        user_group_id = GID_TOUS
    if directive.app_group:
        app_group = directive.get_app_group()
        app_group_paths = app_group.get_paths()
    else:
        app_group_paths = None

    acldata = dict( log = directive.is_logged(),
      user_group_id = user_group_id,
      app_group_paths = app_group_paths,
      libelle = directive.libelle,
      action = directive.action,
      # zone source
      src_zone_interface = inst(src_zone.interface),
      # zone destination
      dest_zone_name = inst(dest_zone.name),
      dest_zone_interface = inst(dest_zone.interface),
      # service
      services = [(inst(serv.protocol), serv.port_list) for serv in services],
      # extremites
      src_extremites = src_extremites,
      dest_extremites = dest_extremites,
      )
    for i in _config_dict(acldata, tous=False):
        yield i
    if acldata['app_group_paths'] != None:
        for i in _config_dict(acldata, tous=True, gid=acldata['user_group_id'], deny=True):
            yield i
        for i in _config_dict(acldata, tous=True, gid=GID_GUEST, deny=True):
            yield i
    if directive.action == ACTION_FORWARD:
        for i in _config_dict(acldata, tous=True):
            yield i

def ip_format(ip_list, netmask, subnet):
    """
    Retourne la source et la destination pour une extremite
    verifie que les ips ne sont pas multivaluées
    """
    srcip = []
    if subnet == 1:
        hip = ip_list[0]
        if '/' in hip:
            ip = hip.split('/')[0]
        else:
            ip = hip
        # on regarde si l'ip n'est pas 0/0
        # le xml est instancie, il peut y avoir des variables multiples (listes en unicode)
        if '[' in ip:
            raise Exception('!!! ATTENTION pas de support des variables creole multivaluees dans era+nufw!!!')

        elif not _is_full_network(ip):
            if netmask == u"":
                raise Exception("une adresse reseau n'est pas renseignee pour l'adresse ip %s"% ip)
            ipy_format = IP('%s/%s'%(inst(ip), inst(netmask)))
            srcip.append(ipy_format.strNormal())

    else:
        for ip in ip_list:
            if not _is_full_network(inst(ip)):
                ipy_format = IP(inst(ip))
                srcip.append(ipy_format.strNormal())

    return srcip

def _config_dict(acldata, tous=False, gid=GID_TOUS, deny=False):
    """
    generation du dictionnaire de configuration
    (correspond a une section du fichier de configuration)
    """
    for protocol, port in acldata['services']:
        try:
            protocol = PROTOCOL[protocol.lower()]
            if tous != False and deny == True:
                decision = DECISION[ACTION_DENY]
            else:
                if acldata['app_group_paths']:
                    decision = DECISION[ACTION_ALLOW]
                else:
                    decision = DECISION[acldata['action']]
        except LookupError as err:
            raise Exception("Le type de directive ne permet pas une authentification")

        data = {
            'SrcPort': "1024-65535",
        }
        # vérifions que les variables sont bien renseignées
        if protocol: data['Proto'] = protocol
        # il y aura toujours une décision (obligatoire)
        data['Decision'] = decision

        # journalisation : désactivation du log par défaut
        if not acldata['log']:
            data['flags'] = '2'
        else:
            data['flags'] = '1'

        if tous==True:
            # on met systematiquement le gid de tous (le domain user)
            if acldata['user_group_id']: data['gid'] = gid
            if acldata['libelle']: data['description'] = acldata['libelle'] + u"_gr_tous"
        else:
            if acldata['user_group_id']: data['gid'] = acldata['user_group_id']
            if acldata['libelle']: data['description'] = acldata['libelle']
        if acldata['app_group_paths'] and tous == False:
            data['App'] = ', '.join(acldata['app_group_paths'])

        if acldata['src_zone_interface']: data['indev'] = acldata['src_zone_interface']
        # dans le cas redirect il faut enlever le outdev et le cas ou bastion est en dest
        if not acldata['action'] == 4 and not acldata['dest_zone_name'] == "bastion":
                data['outdev']= acldata['dest_zone_interface']
        # Sources
        if len(acldata['src_extremites']) != 0: data['SrcIP'] = ", ".join(acldata['src_extremites'])
        # Destinations
        if len(acldata['dest_extremites']) != 0: data['DstIP'] = ", ".join(acldata['dest_extremites'])
        # Ports
        data['DstPort'] = "%s-%s" % (port[0], port[-1])
        yield (data, tous, gid)

def _get_matrix_model(model_file):
    """
        recupère la matrice de flux
    """
    try:
        matrix_model = initialize_app(model_file)
    except Exception as e :
        print(e)
        print("Erreur lors de la recuperation du matrix_model du fichier modele %s " % model_file)
        exit(e)
    return matrix_model

def get_auth_directives(model_file):
    """Return a dictionary of authenticating rules."""
    matrix_model = _get_matrix_model(model_file)
    for d in matrix_model.visit_user_group():
        for rule in parse_fwobject_directive(d):
            yield rule

def get_default_policy(model_file):
    """
        genere une action pour chaque flux oriente (accept)
        une regle pour chaque flux
    """
    matrix = _get_matrix_model(model_file)
#    zones = matrix.flux
    # on enlève la zone bastion
    #zones = zones[:-1]

    for flux in matrix.flux:
        zone1 = flux.get_lowlevel_zone()
        zone2 = flux.get_highlevel_zone()

        yield get_default_policy_rule(zone1, zone2, flux.up_directives_model().default_policy)
        yield get_default_policy_rule(zone2, zone1, flux.down_directives_model().default_policy)

def get_default_policy_rule(in_zone, out_zone, default_policy):
    """
        dico des règles de la default_policy_rule
    """

    data = {
            'description': "%s-%s" % (str(in_zone), str(out_zone)),
            'gid': GID_TOUS,
            'indev': inst(in_zone.interface),
            'outdev':  inst(out_zone.interface),
            'flags' : '2'
            }
    if out_zone.name == 'bastion':
        del(data['outdev'])
        data['DstIP'] = inst(in_zone.ip)
    if in_zone.name == 'bastion':
        del(data['indev'])
        data['SrcIP'] = inst(out_zone.ip)

    if default_policy and 'bastion' not in out_zone.name:
        data['Decision'] = '1'
    else:
        data['Decision'] = '0'

    return data


def _make_cfg_parser(fh, rules):
    """
        Ecrit dans le fichier de sortie au format fichier de configuration (.ini)
    """
    for rule in rules:
        if type(rule) == dict:
             fh.write("\n[%s]\n"%rule[u'description'])
             for key, value in rule.items():
                 if key != u'description':
                     fh.write("%s = %s \n"%(key, value))
        else:
            # cas d'un commentaire
            fh.write(rule)

def export_to_ini_file(filename=NF_ACL_PLAINTEXT, rules=None):
    """Save authenticating rules in INI format, suitable for plaintext module for NuFW."""
    fh = open(filename, "w")
    _make_cfg_parser(fh, rules)
    fh.close()

def parse_command_line():
    """
    Construit le parser d'options de la ligne de commande,
    parse la ligne de commande, et retourne
    le couple (options, fichier_modele_era.xml)
    """
    parser = OptionParser(__doc__, version = '%prog v1.0')
    parser.add_option("-o", "--output",
                      dest = "output_filename", default = "/etc/nufw/acls.nufw",
                      help = "write ACLs to this file",
                      metavar = "ACLS_NUFW_FILE")

    options, args = parser.parse_args()
    # On doit avoir 1 et 1 seul argument en plus des options
    # <=> le nom du modèle era à utiliser
    if len(args) != 1:
        parser.error("Need one (and only one) era model file")

    return options, args[0]


def _gen_rules(model_file, acls_file):
    """
        contient l'ordonnancement des piles :
         1. règle normale
         2. règle tous
         3. politique par défaut

    """
    # premiere pile de regles (authentifiées)
    # pile de regles authentifiées
    auth = []
    # pile de règles "tous" correspondante
    tous = []
    guest = []
    grp = []
    for i in get_auth_directives(model_file):
        if i[1] == False:
            auth.append(i[0])
        else:
            # pass dans le cas d'une redirection
            if i[2] == GID_GUEST:
                guest.append(i[0])
            elif i[2] == GID_TOUS:
                tous.append(i[0])
            else:
                grp.append(i[0])

    # regles non authentifiées, policy par defaut
    default = list(get_default_policy(model_file))
    # concatenation des piles de regles
    msg1 = "# regles par groupes d'utilisateurs\n"
    msg2 = "\n# regles par defaut pour le groupe d'utilisateur\n"
    msg3 = "\n# regles par defaut pour les non-authentifies\n"
    msg4 = "\n# regles par defaut pour tous\n"
    msg5 = "\n# regles par defaut\n"
    rules = [msg1] + list(auth) + [msg2] + list(grp) + [msg3] + list(guest) + [msg4] + list(tous) + [msg5] + list(default)
    export_to_ini_file(acls_file, rules)


def run():
    """
        Outil de génération des règles aclplaintext
        (utilisation ligne de commande)

    """
    try:
        options, model_file = parse_command_line()
        acls_file = options.output_filename
        _gen_rules(model_file, acls_file)
    except Exception as e:
        print("Erreur lors de la generation du fichier acl plaintext %s " % acls_file)
        print(e)
        exit(1)

def gen_acls(acls_file, model_file):
    """
        point d'entrée pré-règlé pour les fichier acl et modèle,
        utilisation depuis l'interface era
    """
    try:
        _gen_rules(model_file, acls_file)
    except Exception as e:
        print("Erreur lors de la generation du fichier acl plaintext %s " % acls_file)
        print(e)
        exit(1)

