# -*- mode: python; coding: utf-8 -*-

import re

from twisted.internet.utils import getProcessOutput

from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager.data import HTMLData
from zephir.monitor.agentmanager.data import TableData
from zephir.monitor.agentmanager.util import status_to_img
from zephir.monitor.agentmanager.util import log

SUPPORTED_REGEXP = r'is supported'
INSTALL_REGEXP = r'sudo apt-get'
KERNEL_REGEXP = r'(linux-image-generic\S+)'


class KernelMaintenance(Agent):
    """Check if the current kernel is maintained by Ubuntu

    """

    def __init__(self, name, **params):
        Agent.__init__(self, name, **params)
        self.name = name
        self.status = status.Unknown()
        self.message = 'Aucune mesure'

        titre = HTMLData("<h2>État de la maintenance du noyau</h2>")
        self.table = TableData([
            ('status', 'État', {'align':'center'}, status_to_img),
            ('msg', 'Message', {'align':'left'}, None),
        ])

        self.data = [titre, self.table]

    def measure(self):
        hwe_output = getProcessOutput('hwe-support-status', args=['--verbose'], env={'LC_ALL': 'C'})
        hwe_output.addCallback(self.parse_hwe_output)
        hwe_output.addErrback(self.hwe_error)

        return hwe_output


    def parse_hwe_output(self, output):
        """Parse the hwe-support-status output

        """
        proposed_kernel = ''

        if re.search(SUPPORTED_REGEXP, output.decode()):
            self.status = status.OK()
            self.message = 'le noyau est maintenu par Ubuntu'
        else:
            for line in output.splitlines():
                if not re.search(INSTALL_REGEXP, line):
                    continue

                match = re.search(KERNEL_REGEXP, line)
                if match:
                    proposed_kernel = match.group(1)

            self.status = status.Error()

            if proposed_kernel:
                self.message = 'Vous devez installer le paquet {0}'.format(proposed_kernel)
            else:
                self.message = 'le noyau est installé, vous devez redémarrer'

        return [{'status': self.status, 'msg': self.message}]

    def hwe_error(self, failure):
        """An error occurred during hwe-support-status execution

        """
        self.status = status.Error()
        self.message = 'Erreur: {0}'.format(failure)
        return [{'status': self.status, 'msg': self.message}]


    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value


    def check_status(self):
        """Renvoi le status courant s’il est définie ou status.Unknown()

        """
        if self.last_measure is None:
            log.msg("{0} : pas de dernière mesure disponible.".format(self.name))
        elif self.status != status.OK():
            log.msg("{0} : {1}.".format(self.name, self.message))

        return self.status
