# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""

"""

try: _ # localized string fetch function
except NameError: _ = str

import os

from zephir.monitor.agentmanager import config as cfg
from zephir.monitor.agentmanager.agent import AgentData




class AgentManager:
    """Structure d'accès aux agents d'un poste client donné (classe
    abstraite).

    Se comporte comme un dictionnaire C{{nom: agent}}.

    TODO: utiliser UserDict.DictMixin
    """

    def __init__(self, config, client_name):
        """
        @param config: dictionnaire des options de configuration de
          l'application

        @param client_name: nom du client qui possède les agents
        """
        self.config = config
        self.client_name = client_name
        self.structure = self.load_structure()

    def _sort_struct(self, liste, order_list):
        done = []
        # elements ordonnés
        for item in order_list:
            if item in liste:
                done.append(item)
        # autres éléments (non présents dans site.cfg) ->  tri alphabétique
        liste.sort()
        for item in liste:
            if item not in done:
                done.append(item)
        return done

    def update_structure(self):
        """Recharge le classement en sections des agents.
        utile si un nouvel agent est remonté dans un datamanager"""
        self.structure = self.load_structure()

    def load_structure(self):
        """Charge le classement en sections des agents.

        Charge le fichier C{site.cfg} du répertoire de données
        correspondant au poste client. Ce fichier doit déclarer une
        variable C{SITE}. Cette variable doit contenir une liste de
        couples C{('titre de section': liste d'agents)}, qui définit
        le classement des agents dans la page du client.

        Une structuration par défaut est créée si le C{site.cfg}
        n'existe pas.
        """
        h = { 'SITE': None }
        structure_file = os.path.join(
            cfg.client_data_dir(self.config, self.client_name),
            "site.cfg")
        self.order_section = []
        self.order_agent = {}
        try:
            exec(open(structure_file).read(), globals(), h)
            assert h['SITE'] is not None and type(h['SITE']) is list
            structure = h['SITE']
            for sct, ag in structure:
                self.order_section.append(sct)
                self.order_agent[sct] = ag
        except IOError:
            pass

        unclassified_agents = [] # how does this update when agents get uploaded?
        struct = {}
        for name, agent in list(self.items()):
            if hasattr(agent, 'section'):
                section = agent.section
            else:
                section = None
                # ancienne version, on prend la section de site.cfg
                for sct, agents in list(self.order_agent.items()):
                    if name in agents:
                        section = sct
            if section is None:
                unclassified_agents.append(name)
            elif section in struct:
                struct[section].append(name)
            else:
                struct[section] = [name]

        final_struct = []
        for section in self._sort_struct(list(struct.keys()), self.order_section):
            agents = struct[section]
            final_struct.append((section, self._sort_struct(agents, self.order_agent[section])))

        if len(final_struct) == 0 and len(unclassified_agents) != 0:
            final_struct.append((None, unclassified_agents))

        return final_struct

    def keys(self):
        raise NotImplemented # abstract method

    def get_measure(self):
        """Renvoie les données stockées à la dernière mesure (sentinelle)
        """
        result = {}
        for agent_name,data in list(self.items()):
            if hasattr(data, 'measure_data'):
                result[agent_name] = data.measure_data
            else:
                result[agent_name] = {}
        return result

    def items(self):

        result = []
        for k in list(self.keys()):
            result.append([k, self[k]])
        return result

    def reset_max_status(agent_name):
        raise NotImplemented # abstract method




class AgentDataManager(AgentManager):
    """Implémentation de C{L{AgentManager}} chargeant les données des
    agents depuis leurs archives sur disque exclusivement.

    Le chargement des données se fait à travers un cache.
    """


    def __init__(self, config, client_name):
        self.cache = {} # agent_name => (modif. date, agent)
        AgentManager.__init__(self, config, client_name)

    def __getitem__(self, agent_name):
        """Accès par nom aux agents (opérateur [] d'un dictionnaire).
        """
        # check has_key then empty cache if data removed ?
        if isinstance(agent_name, bytes):
            agent_name = agent_name.decode()
        agent_dir = cfg.agent_data_dir(self.config, self.client_name, agent_name)
        # get a lock on agent_dir ?
        try:
            (mtime, agent) = self.cache[agent_name]
            needs_refresh = False
        except KeyError:
            needs_refresh = True
        filename = os.path.join(agent_dir, 'agent.xml')
        if not os.path.isfile(filename):
            raise Exception("unable to find agent's information file: {}".format(filename))
        disk_mtime = os.path.getmtime(filename)
        if needs_refresh or (disk_mtime > mtime):
            agent = AgentData.from_archive(agent_dir)
            self.cache[agent_name] = (disk_mtime, agent)
        return agent


    def __setitem__(self, agent_name, agent):
        """Affectation par nom aux agents (opérateur []= d'un
        dictionnaire).
        """
        assert false, "Can't assign agent archives"


    def has_key(self, agent_name):
        if isinstance(agent_name, bytes):
            agent_name = agent_name.decode()
        d = cfg.agent_data_dir(self.config, self.client_name, agent_name)
        agent_data_dir_exists = os.path.isdir(d)
        agent_metadata_exists = os.path.isfile(os.path.join(d, 'agent.xml'))
        return agent_data_dir_exists and agent_metadata_exists


    def keys(self):
        d = cfg.client_data_dir(self.config, self.client_name)
        result = []
        for k in os.listdir(d):
            if self.has_key(k):
                result.append(k)
        return result

    def global_status(self):
        """Méthode de compatibilité avec l'ancien système d'agents.
        """
        # liste des agents à prendre en compte
        real_agents = []
        # lecture de la liste des agents à ignorer
        ignore_list = []
        # liste globale
        fic_override = os.path.normpath(os.path.join(cfg.client_data_dir(self.config, self.client_name),'../ignore_list'))
        if os.path.exists(fic_override):
            f = open(fic_override)
            ignore_list.extend(f.read().strip().split('\n'))
            f.close()
        # liste par serveur
        fic_override = os.path.join(cfg.client_data_dir(self.config, self.client_name),'ignore_list')
        if os.path.exists(fic_override):
            f = open(fic_override)
            ignore_list.extend(f.read().strip().split('\n'))
            f.close()
        for fam in self.structure:
            real_agents.extend(fam[1])
        # on les supprime de la liste
        for agent in ignore_list:
            if agent in real_agents:
                real_agents.remove(agent)
        status = 1
        for agent_name in [an for an in self.keys() if an in real_agents]:
            if self[agent_name].last_status.level() in ['Error']:
                status = 0
            elif status == 1 and self[agent_name].last_status.level() in ['Warn']:
                status = 2

        return status

    def agents_status(self):
        """Méthode de compatibilité avec l'ancien système d'agents.

        @return: dictionnaire C{{'nom agent': (description, status, date de dernière mesure)}}.
        """
        result = {}
        for agent_name,data in list(self.items()):
            if data.last_status.level() in ['Error']:
                status = 0
            elif data.last_status.level() in ['Warn']:
                status = 5
            else:
                status = 1
            # on retourne le libellé de l'agent, l'etat et la date de mesure
            libelle = data.description
            if libelle == '':
                libelle=agent_name
            result[agent_name]=[libelle,status,data.last_measure_date]
        return result

    def reset_max_status(self, agent_name):
        # do xmlrpc call to updater on client... impossible since the client may be hidden by a firewall
        pass
        # rpc = xmlrpclib.Server('http://localhost:%d/xmlrpc/' % cfg.DEFAULT_CONFIG['webserver_port'])
        # rpc.reset_max_status_for_agents([agent_name])
        # check if it worked, display errors




class LiveAgentsManager(AgentManager):
    """Implémentation de C{L{AgentManager}} donnant accès direct aux
    agents en mémoire.

    Utile pour les agents chargés sur la même machine. La page web
    sera mise à jour en temps réel plutôt qu'au rythme des envois de
    données au serveur Zephir.
    """

    def __init__(self, config, client_name, agents):
        self.agents = agents
        AgentManager.__init__(self, config, client_name) # needs keys(), thus self.agents defined


    def __getitem__(self, agent_name):
        agent = self.agents[agent_name]
        agent.ensure_data_uptodate()
        return AgentData.from_agent(agent)


    def __setitem__(self, agent_name, agent):
        self.agents[agent_name] = agent

    def has_key(self, agent_name):
        return agent_name in self.agents

    def keys(self):
        return list(self.agents.keys())

    def reset_max_status(self, agent_name):
        self.agents[agent_name].reset_max_status()

