/*
 * Routines to interpret node coordinates and names on a Gemini system
 *
 * Gemini XE6m (XT5m) 2D Torus variants
 *
 * +----------+---------+------------------+
 * | Cabinets | Chassis | Torus Dimensions |
 * +----------+---------+------------------+
 * |        1 |       1 |   1 x  4 x  8    |
 * |        1 |       2 |   1 x  8 x  8    |
 * |        1 |       3 |   1 x 12 x  8    |
 * |        2 |       4 |   1 x  8 x 16    |
 * |        2 |       6 |   1 x 12 x 16    |
 * |        3 |       8 |   1 x 16 x 16    |
 * |        3 |       9 |   1 x 12 x 24    |
 * |        4 |      12 |   1 x 16 x 24    |
 * |        5 |      15 |   1 x 20 x 24    |
 * |        6 |      18 |   1 x 24 x 24    |
 * +----------+---------+------------------+
 *
 * Copyright (c) 2009-2011 Centro Svizzero di Calcolo Scientifico (CSCS)
 * Licensed under the GPLv2.
 */
#include "basil_alps.h"
#include "basil_torus.h"

/** Return the Gemini number of @nc (Gemini 0: nodes 0/1; Gemini 1: nodes 2/3 */
static uint8_t gemini_num(const struct nodecoord *nc)
{
	return nc->node >> 1;
}

/* Class 0 torus */
static struct torus_coord node2coord_class0(const struct nodecoord *nc,
					    const struct torus_info *ti)
{
	struct torus_coord tc;

	/*
	 * FIXME: the algorithm below has so far only been verified on
	 *	  - single-cabinet XE6m (2D torus) system
	 *	  - dual-cabinet XE6m (2D torus) system
	 *        The algorithm for determining X coordinates is PURE GUESSWORK
	 *        and based on observing how SeaStar class 0 systems are cabled.
	 */
	if (ti->dimension == TORUS_2D)
		tc.x = 0;
	else if (nc->cab == 0)
		tc.x = nc->cage;
	else if (nc->cab == 1 && ti->cabs == 3 && nc->cage == 0)
		tc.x = 8;
	else if (nc->cab == 1)
		tc.x = 5 - nc->cage;
	else if (nc->cab == 2)
		tc.x = 7 - nc->cage;
	else
		fatal("Invalid cabinet %d in class0 torus", nc->cab);

	/* Virtual Y coordinate */
	tc.y = 2 * nc->cage + gemini_num(nc);

	/*
	 * FIXME: the Z coordinate assignment below has been tested to hold on
	 *	  - single cabinet XE6m system
	 *	  - double cabinet XE6m system
	 *	  ==> no test so far on systems with X cabling
	 *	      and more than 2 cabinets.
	 */
	if (nc->cab == 0)
		tc.z = nc->slot;
	else
		tc.z = 15 - nc->slot;

	return tc;
}

/** Resolve node coordinates in a Gemini torus */
struct torus_coord nid2gemini_toruscoord(const struct nodecoord *nc,
					 const struct torus_info *ti)
{
	struct torus_coord (*algo[])(const struct nodecoord  *,
				     const struct torus_info *) = {
		[CABLING_CLASS0] = node2coord_class0,
		[CABLING_CLASS1] = NULL,
		[CABLING_CLASS2] = NULL,
		[CABLING_CLASS3] = NULL
	};

	if (algo[ti->cabling] == NULL)
		fatal("unsupported Gemini Cabling Class %d", ti->cabling);

	return (*algo[ti->cabling])(nc, ti);
}
