/* ====================================================================
 * Copyright (c) 2006-2007, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "LogGraphCanvasItems.h"
#include "sublib/Utility.h"
#include "svn/LogEntry.h"
#include "svn/Revision.h"

// qt
#include <QtGui/QPainter>
#include <QtCore/QDateTime>


LogGraphCanvasItem::LogGraphCanvasItem( Q3Canvas* canvas )
: Q3CanvasRectangle(canvas)
{
}

LogGraphCanvasItem::~LogGraphCanvasItem()
{
}


///////////////////////////////////////////////////////////////////////////////

LogGraphCanvasRev::LogGraphCanvasRev( Q3Canvas* canvas, LogGraphNodePtr node )
: super(canvas), _node(node)
{
}

void LogGraphCanvasRev::drawShape( QPainter &p )
{
  if( isSelected() )
  {
    p.setPen( QColor(0,0,0) );
  }

  Q3CanvasRectangle::drawShape(p);

  p.setPen( QColor(0,0,0) );
  QFontMetrics m(p.font());
  p.drawText( (int)(x()+m.height()/3), (int)(y()+m.height()),
    QString("r%1").arg(_node->getLog()->getRevnumber()));
}

LogGraphNodePtr LogGraphCanvasRev::getNode() const
{
  return _node;
}

QString LogGraphCanvasRev::getPath() const
{
  return QString::fromUtf8(_node->getName());
}

svn::Revision* LogGraphCanvasRev::getRev() const
{
  return new svn::RevisionNumber(_node->getLog()->getRevnumber());
}

QString LogGraphCanvasRev::getTooltip() const
{
  QString text = QString(
    "<qt><table cellpadding=0 cellspacing=0>"
    "<tr><td><nobr>%1</nobr></td></tr>")
    .arg(QString::fromUtf8(getNode()->getName()));

  QDateTime date;
  date.setTime_t( (uint)apr_time_sec(getNode()->getLog()->getDate()) );
  QString qdate = date.toString( getDateFormat() );

  text += QString(
    "<tr><td><nobr>r%1, %2, %3</nobr></td></tr>" )
    .arg(getNode()->getLog()->getRevnumber())
    .arg(qdate)
    .arg(QString::fromUtf8(getNode()->getLog()->getAuthor()));

  QString msg = QString::fromUtf8(getNode()->getLog()->getMessage());
  QString cut = msg.section( '\n', 0, 14,
    QString::SectionIncludeLeadingSep | QString::SectionIncludeTrailingSep );
  QString cutmsg;
  if( cut.length() < msg.length() )
  {
    cutmsg = QString("<tr><td>%2<td></tr>").arg( _q(".... (cut)") );
  }

  text += QString(
    "<tr><td>&nbsp;</td></tr>"
    "<tr><td><pre>%1</pre></td></tr>"
    "%2"
    "</table></qt>")
    .arg(cut)
    .arg(cutmsg);

  return text;
}

///////////////////////////////////////////////////////////////////////////////

LogGraphCanvasPath::LogGraphCanvasPath( Q3Canvas* canvas, const QString& initialName )
: super(canvas), _initialName(initialName), _newestName(initialName)
{
}

void LogGraphCanvasPath::drawShape( QPainter &p )
{
  if( isSelected() )
  {
    p.setPen( QColor(0,0,0) );
  }

  Q3CanvasRectangle::drawShape(p);

  p.setPen( QColor(0,0,0) );
  QFontMetrics m(p.font());
  p.drawText( (int)(x()+m.height()/3), (int)(y()+m.height()), getFitName(p) );
}

QString LogGraphCanvasPath::getFitName( QPainter &p )
{
  const int ExtraSpace = 10;

  QString name = getFullName();
  QFontMetrics m(p.font());

  if( m.width(name)+ExtraSpace < width() )
  {
    return name;
  }
  else if( m.width(_newestName)+ExtraSpace < width() )
  {
    return _newestName;
  }
  else
  {
    int oversize = m.width(_newestName)+ ExtraSpace + m.width("..") - width();

    int ccnt = 0;
    while( oversize > 0 )
    {
      oversize -= m.width( _newestName[ccnt] );
      ccnt++;
    }

    return ".." + _newestName.right(_newestName.length()-ccnt);
  }
}

QString LogGraphCanvasPath::getFullName() const
{
  QString name = _newestName;

  if( _initialName != _newestName )
  {
    name += QString(" (%1)").arg(_initialName);
  }

  return name;
}

void LogGraphCanvasPath::addNewName( const QString& name )
{
  _newestName = name;
}

QString LogGraphCanvasPath::getPath() const
{
  return _newestName;
}

svn::Revision* LogGraphCanvasPath::getRev() const
{
  return new svn::Revision(svn::Revision_Head);
}

QString LogGraphCanvasPath::getTooltip() const
{
  return getFullName();
}
