/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.iggy.message;

import org.apache.commons.lang3.ArrayUtils;

import java.nio.ByteBuffer;

public record Partitioning(PartitioningKind kind, byte[] value) {

    public static Partitioning balanced() {
        return new Partitioning(PartitioningKind.Balanced, new byte[] {});
    }

    public static Partitioning partitionId(Long id) {
        ByteBuffer buffer = ByteBuffer.allocate(4);
        buffer.putInt(id.intValue());
        byte[] partitionId = buffer.array();
        ArrayUtils.reverse(partitionId);
        return new Partitioning(PartitioningKind.PartitionId, partitionId);
    }

    public static Partitioning messagesKey(String key) {
        if (key == null || key.isBlank() || key.length() > 255) {
            throw new IllegalArgumentException("Key must be non-empty and less than 255 characters long");
        }
        return new Partitioning(PartitioningKind.MessagesKey, key.getBytes());
    }

    public int getSize() {
        // kind, 1 byte + length, 1 byte + value.length()
        return 2 + value.length;
    }
}
